/*
Copyright (c) 2007, Dennis M. Sosnoski
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.
 * Neither the name of JiBX nor the names of its contributors may be used
   to endorse or promote products derived from this software without specific
   prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

package org.jibx.binding.generator;

import java.net.URL;
import java.util.ArrayList;

import org.jibx.binding.Loader;
import org.jibx.binding.classes.ClassCache;
import org.jibx.binding.classes.ClassFile;
import org.jibx.binding.model.BindingElement;
import org.jibx.binding.model.BindingHolder;
import org.jibx.binding.model.MappingElement;
import org.jibx.schema.codegen.ReferenceCountMap;

/**
 * Test code for binding generation.
 */
public class GeneratorTest extends CustomizationTestBase
{
    public static final String MULTIPLE_PROPERTIES_CLASSES =
        "<custom property-access='true' force-classes='true'>\n" +
        "  <package name='org.jibx.binding'>\n" +
        "    <class name='generator.DataClass2'\n" +
        "        strip-prefixes='s_ m_' excludes='transient static'>\n" +
        "      <collection-field field='m_dataClass1s'\n" +
        "          item-type='org.jibx.binding.generator.DataClass1'/>\n" +
        "    </class>\n" +
        "    <class name='generator.DataClass1'/>\n" +
        "  </package>\n" +
        "</custom>";
    
    /* (non-Javadoc)
     * @see junit.framework.TestCase#setUp()
     */
    protected void setUp() throws Exception {
        
        // set paths to be used for loading referenced classes
        URL[] urls = Loader.getClassPaths();
        String[] paths = new String[urls.length];
        for (int i = 0; i < urls.length; i++) {
            paths[i] = urls[i].getFile();
        }
        ClassCache.setPaths(paths);
        ClassFile.setPaths(paths);
    }

    public void testExpandReferences() {
        GlobalCustom custom = new GlobalCustom();
        BindingGenerator gen = new BindingGenerator(custom);
        ReferenceCountMap refmap = new ReferenceCountMap();
        gen.expandReferences("org.jibx.binding.generator.DataClass2Generic",
            refmap);
        assertEquals("original class references", 0, refmap.getCount("org.jibx.binding.generator.DataClass2Generic"));
        assertEquals("referenced class references", 2, refmap.getCount("org.jibx.binding.generator.DataClass1"));
    }
    
    public void testMultiplePropertiesClasses() throws Exception {
        GlobalCustom custom = readCustom(MULTIPLE_PROPERTIES_CLASSES);
        BindingGenerator gen = new BindingGenerator(custom);
        ReferenceCountMap refmap = new ReferenceCountMap();
        gen.expandReferences("org.jibx.binding.generator.DataClass2Generic",
            refmap);
        assertEquals("original class references", 0, refmap.getCount("org.jibx.binding.generator.DataClass2"));
        assertEquals("referenced class references", 2, refmap.getCount("org.jibx.binding.generator.DataClass1"));
        gen.expandReferences("org.jibx.binding.generator.DataClass2Generic",
            refmap);
        assertEquals("original class references", 0, refmap.getCount("org.jibx.binding.generator.DataClass2Generic"));
        assertEquals("referenced class references", 3, refmap.getCount("org.jibx.binding.generator.DataClass1"));
    }

    public void testSingleClassBinding() {
        GlobalCustom custom = new GlobalCustom();
        custom.initClasses();
        custom.fillClasses();
        BindingGenerator gen = new BindingGenerator(custom);
        ArrayList types = new ArrayList();
        types.add("org.jibx.binding.generator.DataClass1");
        gen.generate(null, types);
        BindingHolder hold = gen.getBinding("http://jibx.org/binding/generator");
        assertNotNull("default namespace binding", hold);
        BindingElement binding = hold.getBinding();
        ArrayList childs = binding.topChildren();
        assertEquals("child count", 2, childs.size());
        Object child = childs.get(1);
        assertTrue("child type", child instanceof MappingElement);
        MappingElement mapping = (MappingElement)child;
        assertEquals("mapped class", mapping.getClassName(), "org.jibx.binding.generator.DataClass1");
        assertEquals("mapped items", 4, mapping.children().size());
    }
}