// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "base/logging.h"
#include "base/memory/scoped_ptr.h"
#include "media/base/media.h"
#include "media/ffmpeg/ffmpeg_common.h"
#include "media/filters/ffmpeg_glue.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace media {

// Verify that the AV_CODEC_ID values match what is specified in histograms.xml
// for Media.DetectedAudioCodec and Media.DetectedVideoCodec
#define STATIC_ASSERT(test) static_assert(test, #test)
STATIC_ASSERT(0 == AV_CODEC_ID_NONE);
STATIC_ASSERT(1 == AV_CODEC_ID_MPEG1VIDEO);
STATIC_ASSERT(2 == AV_CODEC_ID_MPEG2VIDEO);
STATIC_ASSERT(3 == AV_CODEC_ID_MPEG2VIDEO_XVMC);
STATIC_ASSERT(4 == AV_CODEC_ID_H261);
STATIC_ASSERT(5 == AV_CODEC_ID_H263);
STATIC_ASSERT(6 == AV_CODEC_ID_RV10);
STATIC_ASSERT(7 == AV_CODEC_ID_RV20);
STATIC_ASSERT(8 == AV_CODEC_ID_MJPEG);
STATIC_ASSERT(9 == AV_CODEC_ID_MJPEGB);
STATIC_ASSERT(10 == AV_CODEC_ID_LJPEG);
STATIC_ASSERT(11 == AV_CODEC_ID_SP5X);
STATIC_ASSERT(12 == AV_CODEC_ID_JPEGLS);
STATIC_ASSERT(13 == AV_CODEC_ID_MPEG4);
STATIC_ASSERT(14 == AV_CODEC_ID_RAWVIDEO);
STATIC_ASSERT(15 == AV_CODEC_ID_MSMPEG4V1);
STATIC_ASSERT(16 == AV_CODEC_ID_MSMPEG4V2);
STATIC_ASSERT(17 == AV_CODEC_ID_MSMPEG4V3);
STATIC_ASSERT(18 == AV_CODEC_ID_WMV1);
STATIC_ASSERT(19 == AV_CODEC_ID_WMV2);
STATIC_ASSERT(20 == AV_CODEC_ID_H263P);
STATIC_ASSERT(21 == AV_CODEC_ID_H263I);
STATIC_ASSERT(22 == AV_CODEC_ID_FLV1);
STATIC_ASSERT(23 == AV_CODEC_ID_SVQ1);
STATIC_ASSERT(24 == AV_CODEC_ID_SVQ3);
STATIC_ASSERT(25 == AV_CODEC_ID_DVVIDEO);
STATIC_ASSERT(26 == AV_CODEC_ID_HUFFYUV);
STATIC_ASSERT(27 == AV_CODEC_ID_CYUV);
STATIC_ASSERT(28 == AV_CODEC_ID_H264);
STATIC_ASSERT(29 == AV_CODEC_ID_INDEO3);
STATIC_ASSERT(30 == AV_CODEC_ID_VP3);
STATIC_ASSERT(31 == AV_CODEC_ID_THEORA);
STATIC_ASSERT(32 == AV_CODEC_ID_ASV1);
STATIC_ASSERT(33 == AV_CODEC_ID_ASV2);
STATIC_ASSERT(34 == AV_CODEC_ID_FFV1);
STATIC_ASSERT(35 == AV_CODEC_ID_4XM);
STATIC_ASSERT(36 == AV_CODEC_ID_VCR1);
STATIC_ASSERT(37 == AV_CODEC_ID_CLJR);
STATIC_ASSERT(38 == AV_CODEC_ID_MDEC);
STATIC_ASSERT(39 == AV_CODEC_ID_ROQ);
STATIC_ASSERT(40 == AV_CODEC_ID_INTERPLAY_VIDEO);
STATIC_ASSERT(41 == AV_CODEC_ID_XAN_WC3);
STATIC_ASSERT(42 == AV_CODEC_ID_XAN_WC4);
STATIC_ASSERT(43 == AV_CODEC_ID_RPZA);
STATIC_ASSERT(44 == AV_CODEC_ID_CINEPAK);
STATIC_ASSERT(45 == AV_CODEC_ID_WS_VQA);
STATIC_ASSERT(46 == AV_CODEC_ID_MSRLE);
STATIC_ASSERT(47 == AV_CODEC_ID_MSVIDEO1);
STATIC_ASSERT(48 == AV_CODEC_ID_IDCIN);
STATIC_ASSERT(49 == AV_CODEC_ID_8BPS);
STATIC_ASSERT(50 == AV_CODEC_ID_SMC);
STATIC_ASSERT(51 == AV_CODEC_ID_FLIC);
STATIC_ASSERT(52 == AV_CODEC_ID_TRUEMOTION1);
STATIC_ASSERT(53 == AV_CODEC_ID_VMDVIDEO);
STATIC_ASSERT(54 == AV_CODEC_ID_MSZH);
STATIC_ASSERT(55 == AV_CODEC_ID_ZLIB);
STATIC_ASSERT(56 == AV_CODEC_ID_QTRLE);
STATIC_ASSERT(57 == AV_CODEC_ID_SNOW_DEPRECATED);
STATIC_ASSERT(58 == AV_CODEC_ID_TSCC);
STATIC_ASSERT(59 == AV_CODEC_ID_ULTI);
STATIC_ASSERT(60 == AV_CODEC_ID_QDRAW);
STATIC_ASSERT(61 == AV_CODEC_ID_VIXL);
STATIC_ASSERT(62 == AV_CODEC_ID_QPEG);
STATIC_ASSERT(63 == AV_CODEC_ID_PNG);
STATIC_ASSERT(64 == AV_CODEC_ID_PPM);
STATIC_ASSERT(65 == AV_CODEC_ID_PBM);
STATIC_ASSERT(66 == AV_CODEC_ID_PGM);
STATIC_ASSERT(67 == AV_CODEC_ID_PGMYUV);
STATIC_ASSERT(68 == AV_CODEC_ID_PAM);
STATIC_ASSERT(69 == AV_CODEC_ID_FFVHUFF);
STATIC_ASSERT(70 == AV_CODEC_ID_RV30);
STATIC_ASSERT(71 == AV_CODEC_ID_RV40);
STATIC_ASSERT(72 == AV_CODEC_ID_VC1);
STATIC_ASSERT(73 == AV_CODEC_ID_WMV3);
STATIC_ASSERT(74 == AV_CODEC_ID_LOCO);
STATIC_ASSERT(75 == AV_CODEC_ID_WNV1);
STATIC_ASSERT(76 == AV_CODEC_ID_AASC);
STATIC_ASSERT(77 == AV_CODEC_ID_INDEO2);
STATIC_ASSERT(78 == AV_CODEC_ID_FRAPS);
STATIC_ASSERT(79 == AV_CODEC_ID_TRUEMOTION2);
STATIC_ASSERT(80 == AV_CODEC_ID_BMP);
STATIC_ASSERT(81 == AV_CODEC_ID_CSCD);
STATIC_ASSERT(82 == AV_CODEC_ID_MMVIDEO);
STATIC_ASSERT(83 == AV_CODEC_ID_ZMBV);
STATIC_ASSERT(84 == AV_CODEC_ID_AVS);
STATIC_ASSERT(85 == AV_CODEC_ID_SMACKVIDEO);
STATIC_ASSERT(86 == AV_CODEC_ID_NUV);
STATIC_ASSERT(87 == AV_CODEC_ID_KMVC);
STATIC_ASSERT(88 == AV_CODEC_ID_FLASHSV);
STATIC_ASSERT(89 == AV_CODEC_ID_CAVS);
STATIC_ASSERT(90 == AV_CODEC_ID_JPEG2000);
STATIC_ASSERT(91 == AV_CODEC_ID_VMNC);
STATIC_ASSERT(92 == AV_CODEC_ID_VP5);
STATIC_ASSERT(93 == AV_CODEC_ID_VP6);
STATIC_ASSERT(94 == AV_CODEC_ID_VP6F);
STATIC_ASSERT(95 == AV_CODEC_ID_TARGA);
STATIC_ASSERT(96 == AV_CODEC_ID_DSICINVIDEO);
STATIC_ASSERT(97 == AV_CODEC_ID_TIERTEXSEQVIDEO);
STATIC_ASSERT(98 == AV_CODEC_ID_TIFF);
STATIC_ASSERT(99 == AV_CODEC_ID_GIF);
STATIC_ASSERT(100 == AV_CODEC_ID_DXA);
STATIC_ASSERT(101 == AV_CODEC_ID_DNXHD);
STATIC_ASSERT(102 == AV_CODEC_ID_THP);
STATIC_ASSERT(103 == AV_CODEC_ID_SGI);
STATIC_ASSERT(104 == AV_CODEC_ID_C93);
STATIC_ASSERT(105 == AV_CODEC_ID_BETHSOFTVID);
STATIC_ASSERT(106 == AV_CODEC_ID_PTX);
STATIC_ASSERT(107 == AV_CODEC_ID_TXD);
STATIC_ASSERT(108 == AV_CODEC_ID_VP6A);
STATIC_ASSERT(109 == AV_CODEC_ID_AMV);
STATIC_ASSERT(110 == AV_CODEC_ID_VB);
STATIC_ASSERT(111 == AV_CODEC_ID_PCX);
STATIC_ASSERT(112 == AV_CODEC_ID_SUNRAST);
STATIC_ASSERT(113 == AV_CODEC_ID_INDEO4);
STATIC_ASSERT(114 == AV_CODEC_ID_INDEO5);
STATIC_ASSERT(115 == AV_CODEC_ID_MIMIC);
STATIC_ASSERT(116 == AV_CODEC_ID_RL2);
STATIC_ASSERT(117 == AV_CODEC_ID_ESCAPE124);
STATIC_ASSERT(118 == AV_CODEC_ID_DIRAC);
STATIC_ASSERT(119 == AV_CODEC_ID_BFI);
STATIC_ASSERT(120 == AV_CODEC_ID_CMV);
STATIC_ASSERT(121 == AV_CODEC_ID_MOTIONPIXELS);
STATIC_ASSERT(122 == AV_CODEC_ID_TGV);
STATIC_ASSERT(123 == AV_CODEC_ID_TGQ);
STATIC_ASSERT(124 == AV_CODEC_ID_TQI);
STATIC_ASSERT(125 == AV_CODEC_ID_AURA);
STATIC_ASSERT(126 == AV_CODEC_ID_AURA2);
STATIC_ASSERT(127 == AV_CODEC_ID_V210X);
STATIC_ASSERT(128 == AV_CODEC_ID_TMV);
STATIC_ASSERT(129 == AV_CODEC_ID_V210);
STATIC_ASSERT(130 == AV_CODEC_ID_DPX);
STATIC_ASSERT(131 == AV_CODEC_ID_MAD);
STATIC_ASSERT(132 == AV_CODEC_ID_FRWU);
STATIC_ASSERT(133 == AV_CODEC_ID_FLASHSV2);
STATIC_ASSERT(134 == AV_CODEC_ID_CDGRAPHICS);
STATIC_ASSERT(135 == AV_CODEC_ID_R210);
STATIC_ASSERT(136 == AV_CODEC_ID_ANM);
STATIC_ASSERT(137 == AV_CODEC_ID_BINKVIDEO);
STATIC_ASSERT(138 == AV_CODEC_ID_IFF_ILBM);
STATIC_ASSERT(139 == AV_CODEC_ID_IFF_BYTERUN1);
STATIC_ASSERT(140 == AV_CODEC_ID_KGV1);
STATIC_ASSERT(141 == AV_CODEC_ID_YOP);
STATIC_ASSERT(142 == AV_CODEC_ID_VP8);
STATIC_ASSERT(143 == AV_CODEC_ID_PICTOR);
STATIC_ASSERT(144 == AV_CODEC_ID_ANSI);
STATIC_ASSERT(145 == AV_CODEC_ID_A64_MULTI);
STATIC_ASSERT(146 == AV_CODEC_ID_A64_MULTI5);
STATIC_ASSERT(147 == AV_CODEC_ID_R10K);
STATIC_ASSERT(148 == AV_CODEC_ID_MXPEG);
STATIC_ASSERT(149 == AV_CODEC_ID_LAGARITH);
STATIC_ASSERT(150 == AV_CODEC_ID_PRORES);
STATIC_ASSERT(151 == AV_CODEC_ID_JV);
STATIC_ASSERT(152 == AV_CODEC_ID_DFA);
STATIC_ASSERT(153 == AV_CODEC_ID_WMV3IMAGE);
STATIC_ASSERT(154 == AV_CODEC_ID_VC1IMAGE);
STATIC_ASSERT(155 == AV_CODEC_ID_UTVIDEO);
STATIC_ASSERT(156 == AV_CODEC_ID_BMV_VIDEO);
STATIC_ASSERT(157 == AV_CODEC_ID_VBLE);
STATIC_ASSERT(158 == AV_CODEC_ID_DXTORY);
STATIC_ASSERT(159 == AV_CODEC_ID_V410);
STATIC_ASSERT(160 == AV_CODEC_ID_XWD);
STATIC_ASSERT(161 == AV_CODEC_ID_CDXL);
STATIC_ASSERT(162 == AV_CODEC_ID_XBM);
STATIC_ASSERT(163 == AV_CODEC_ID_ZEROCODEC);
STATIC_ASSERT(164 == AV_CODEC_ID_MSS1);
STATIC_ASSERT(165 == AV_CODEC_ID_MSA1);
STATIC_ASSERT(166 == AV_CODEC_ID_TSCC2);
STATIC_ASSERT(167 == AV_CODEC_ID_MTS2);
STATIC_ASSERT(168 == AV_CODEC_ID_CLLC);
STATIC_ASSERT(169 == AV_CODEC_ID_MSS2);
STATIC_ASSERT(170 == AV_CODEC_ID_VP9);
STATIC_ASSERT(65536 == AV_CODEC_ID_PCM_S16LE);
STATIC_ASSERT(65537 == AV_CODEC_ID_PCM_S16BE);
STATIC_ASSERT(65538 == AV_CODEC_ID_PCM_U16LE);
STATIC_ASSERT(65539 == AV_CODEC_ID_PCM_U16BE);
STATIC_ASSERT(65540 == AV_CODEC_ID_PCM_S8);
STATIC_ASSERT(65541 == AV_CODEC_ID_PCM_U8);
STATIC_ASSERT(65542 == AV_CODEC_ID_PCM_MULAW);
STATIC_ASSERT(65543 == AV_CODEC_ID_PCM_ALAW);
STATIC_ASSERT(65544 == AV_CODEC_ID_PCM_S32LE);
STATIC_ASSERT(65545 == AV_CODEC_ID_PCM_S32BE);
STATIC_ASSERT(65546 == AV_CODEC_ID_PCM_U32LE);
STATIC_ASSERT(65547 == AV_CODEC_ID_PCM_U32BE);
STATIC_ASSERT(65548 == AV_CODEC_ID_PCM_S24LE);
STATIC_ASSERT(65549 == AV_CODEC_ID_PCM_S24BE);
STATIC_ASSERT(65550 == AV_CODEC_ID_PCM_U24LE);
STATIC_ASSERT(65551 == AV_CODEC_ID_PCM_U24BE);
STATIC_ASSERT(65552 == AV_CODEC_ID_PCM_S24DAUD);
STATIC_ASSERT(65553 == AV_CODEC_ID_PCM_ZORK);
STATIC_ASSERT(65554 == AV_CODEC_ID_PCM_S16LE_PLANAR);
STATIC_ASSERT(65555 == AV_CODEC_ID_PCM_DVD);
STATIC_ASSERT(65556 == AV_CODEC_ID_PCM_F32BE);
STATIC_ASSERT(65557 == AV_CODEC_ID_PCM_F32LE);
STATIC_ASSERT(65558 == AV_CODEC_ID_PCM_F64BE);
STATIC_ASSERT(65559 == AV_CODEC_ID_PCM_F64LE);
STATIC_ASSERT(65560 == AV_CODEC_ID_PCM_BLURAY);
STATIC_ASSERT(65561 == AV_CODEC_ID_PCM_LXF);
STATIC_ASSERT(65562 == AV_CODEC_ID_S302M);
STATIC_ASSERT(65563 == AV_CODEC_ID_PCM_S8_PLANAR);
STATIC_ASSERT(69632 == AV_CODEC_ID_ADPCM_IMA_QT);
STATIC_ASSERT(69633 == AV_CODEC_ID_ADPCM_IMA_WAV);
STATIC_ASSERT(69634 == AV_CODEC_ID_ADPCM_IMA_DK3);
STATIC_ASSERT(69635 == AV_CODEC_ID_ADPCM_IMA_DK4);
STATIC_ASSERT(69636 == AV_CODEC_ID_ADPCM_IMA_WS);
STATIC_ASSERT(69637 == AV_CODEC_ID_ADPCM_IMA_SMJPEG);
STATIC_ASSERT(69638 == AV_CODEC_ID_ADPCM_MS);
STATIC_ASSERT(69639 == AV_CODEC_ID_ADPCM_4XM);
STATIC_ASSERT(69640 == AV_CODEC_ID_ADPCM_XA);
STATIC_ASSERT(69641 == AV_CODEC_ID_ADPCM_ADX);
STATIC_ASSERT(69642 == AV_CODEC_ID_ADPCM_EA);
STATIC_ASSERT(69643 == AV_CODEC_ID_ADPCM_G726);
STATIC_ASSERT(69644 == AV_CODEC_ID_ADPCM_CT);
STATIC_ASSERT(69645 == AV_CODEC_ID_ADPCM_SWF);
STATIC_ASSERT(69646 == AV_CODEC_ID_ADPCM_YAMAHA);
STATIC_ASSERT(69647 == AV_CODEC_ID_ADPCM_SBPRO_4);
STATIC_ASSERT(69648 == AV_CODEC_ID_ADPCM_SBPRO_3);
STATIC_ASSERT(69649 == AV_CODEC_ID_ADPCM_SBPRO_2);
STATIC_ASSERT(69650 == AV_CODEC_ID_ADPCM_THP);
STATIC_ASSERT(69651 == AV_CODEC_ID_ADPCM_IMA_AMV);
STATIC_ASSERT(69652 == AV_CODEC_ID_ADPCM_EA_R1);
STATIC_ASSERT(69653 == AV_CODEC_ID_ADPCM_EA_R3);
STATIC_ASSERT(69654 == AV_CODEC_ID_ADPCM_EA_R2);
STATIC_ASSERT(69655 == AV_CODEC_ID_ADPCM_IMA_EA_SEAD);
STATIC_ASSERT(69656 == AV_CODEC_ID_ADPCM_IMA_EA_EACS);
STATIC_ASSERT(69657 == AV_CODEC_ID_ADPCM_EA_XAS);
STATIC_ASSERT(69658 == AV_CODEC_ID_ADPCM_EA_MAXIS_XA);
STATIC_ASSERT(69659 == AV_CODEC_ID_ADPCM_IMA_ISS);
STATIC_ASSERT(69660 == AV_CODEC_ID_ADPCM_G722);
STATIC_ASSERT(69661 == AV_CODEC_ID_ADPCM_IMA_APC);
STATIC_ASSERT(73728 == AV_CODEC_ID_AMR_NB);
STATIC_ASSERT(73729 == AV_CODEC_ID_AMR_WB);
STATIC_ASSERT(77824 == AV_CODEC_ID_RA_144);
STATIC_ASSERT(77825 == AV_CODEC_ID_RA_288);
STATIC_ASSERT(81920 == AV_CODEC_ID_ROQ_DPCM);
STATIC_ASSERT(81921 == AV_CODEC_ID_INTERPLAY_DPCM);
STATIC_ASSERT(81922 == AV_CODEC_ID_XAN_DPCM);
STATIC_ASSERT(81923 == AV_CODEC_ID_SOL_DPCM);
STATIC_ASSERT(86016 == AV_CODEC_ID_MP2);
STATIC_ASSERT(86017 == AV_CODEC_ID_MP3);
STATIC_ASSERT(86018 == AV_CODEC_ID_AAC);
STATIC_ASSERT(86019 == AV_CODEC_ID_AC3);
STATIC_ASSERT(86020 == AV_CODEC_ID_DTS);
STATIC_ASSERT(86021 == AV_CODEC_ID_VORBIS);
STATIC_ASSERT(86022 == AV_CODEC_ID_DVAUDIO);
STATIC_ASSERT(86023 == AV_CODEC_ID_WMAV1);
STATIC_ASSERT(86024 == AV_CODEC_ID_WMAV2);
STATIC_ASSERT(86025 == AV_CODEC_ID_MACE3);
STATIC_ASSERT(86026 == AV_CODEC_ID_MACE6);
STATIC_ASSERT(86027 == AV_CODEC_ID_VMDAUDIO);
STATIC_ASSERT(86028 == AV_CODEC_ID_FLAC);
STATIC_ASSERT(86029 == AV_CODEC_ID_MP3ADU);
STATIC_ASSERT(86030 == AV_CODEC_ID_MP3ON4);
STATIC_ASSERT(86031 == AV_CODEC_ID_SHORTEN);
STATIC_ASSERT(86032 == AV_CODEC_ID_ALAC);
STATIC_ASSERT(86033 == AV_CODEC_ID_WESTWOOD_SND1);
STATIC_ASSERT(86034 == AV_CODEC_ID_GSM);
STATIC_ASSERT(86035 == AV_CODEC_ID_QDM2);
STATIC_ASSERT(86036 == AV_CODEC_ID_COOK);
STATIC_ASSERT(86037 == AV_CODEC_ID_TRUESPEECH);
STATIC_ASSERT(86038 == AV_CODEC_ID_TTA);
STATIC_ASSERT(86039 == AV_CODEC_ID_SMACKAUDIO);
STATIC_ASSERT(86040 == AV_CODEC_ID_QCELP);
STATIC_ASSERT(86041 == AV_CODEC_ID_WAVPACK);
STATIC_ASSERT(86042 == AV_CODEC_ID_DSICINAUDIO);
STATIC_ASSERT(86043 == AV_CODEC_ID_IMC);
STATIC_ASSERT(86044 == AV_CODEC_ID_MUSEPACK7);
STATIC_ASSERT(86045 == AV_CODEC_ID_MLP);
STATIC_ASSERT(86046 == AV_CODEC_ID_GSM_MS);
STATIC_ASSERT(86047 == AV_CODEC_ID_ATRAC3);
STATIC_ASSERT(86048 == AV_CODEC_ID_VOXWARE);
STATIC_ASSERT(86049 == AV_CODEC_ID_APE);
STATIC_ASSERT(86050 == AV_CODEC_ID_NELLYMOSER);
STATIC_ASSERT(86051 == AV_CODEC_ID_MUSEPACK8);
STATIC_ASSERT(86052 == AV_CODEC_ID_SPEEX);
STATIC_ASSERT(86053 == AV_CODEC_ID_WMAVOICE);
STATIC_ASSERT(86054 == AV_CODEC_ID_WMAPRO);
STATIC_ASSERT(86055 == AV_CODEC_ID_WMALOSSLESS);
STATIC_ASSERT(86056 == AV_CODEC_ID_ATRAC3P);
STATIC_ASSERT(86057 == AV_CODEC_ID_EAC3);
STATIC_ASSERT(86058 == AV_CODEC_ID_SIPR);
STATIC_ASSERT(86059 == AV_CODEC_ID_MP1);
STATIC_ASSERT(86060 == AV_CODEC_ID_TWINVQ);
STATIC_ASSERT(86061 == AV_CODEC_ID_TRUEHD);
STATIC_ASSERT(86062 == AV_CODEC_ID_MP4ALS);
STATIC_ASSERT(86063 == AV_CODEC_ID_ATRAC1);
STATIC_ASSERT(86064 == AV_CODEC_ID_BINKAUDIO_RDFT);
STATIC_ASSERT(86065 == AV_CODEC_ID_BINKAUDIO_DCT);
STATIC_ASSERT(86066 == AV_CODEC_ID_AAC_LATM);
STATIC_ASSERT(86067 == AV_CODEC_ID_QDMC);
STATIC_ASSERT(86068 == AV_CODEC_ID_CELT);
STATIC_ASSERT(86069 == AV_CODEC_ID_G723_1);
STATIC_ASSERT(86070 == AV_CODEC_ID_G729);
STATIC_ASSERT(86071 == AV_CODEC_ID_8SVX_EXP);
STATIC_ASSERT(86072 == AV_CODEC_ID_8SVX_FIB);
STATIC_ASSERT(86073 == AV_CODEC_ID_BMV_AUDIO);
STATIC_ASSERT(86074 == AV_CODEC_ID_RALF);
STATIC_ASSERT(86075 == AV_CODEC_ID_IAC);
STATIC_ASSERT(86076 == AV_CODEC_ID_ILBC);
STATIC_ASSERT(86077 == AV_CODEC_ID_OPUS_DEPRECATED);
STATIC_ASSERT(86078 == AV_CODEC_ID_COMFORT_NOISE);
STATIC_ASSERT(86079 == AV_CODEC_ID_TAK_DEPRECATED);
STATIC_ASSERT(94208 == AV_CODEC_ID_DVD_SUBTITLE);
STATIC_ASSERT(94209 == AV_CODEC_ID_DVB_SUBTITLE);
STATIC_ASSERT(94210 == AV_CODEC_ID_TEXT);
STATIC_ASSERT(94211 == AV_CODEC_ID_XSUB);
STATIC_ASSERT(94212 == AV_CODEC_ID_SSA);
STATIC_ASSERT(94213 == AV_CODEC_ID_MOV_TEXT);
STATIC_ASSERT(94214 == AV_CODEC_ID_HDMV_PGS_SUBTITLE);
STATIC_ASSERT(94215 == AV_CODEC_ID_DVB_TELETEXT);
STATIC_ASSERT(94216 == AV_CODEC_ID_SRT);
STATIC_ASSERT(98304 == AV_CODEC_ID_TTF);
STATIC_ASSERT(102400 == AV_CODEC_ID_PROBE);
STATIC_ASSERT(131072 == AV_CODEC_ID_MPEG2TS);
STATIC_ASSERT(131073 == AV_CODEC_ID_MPEG4SYSTEMS);
STATIC_ASSERT(135168 == AV_CODEC_ID_FFMETADATA);
STATIC_ASSERT(4665933 == AV_CODEC_ID_G2M);
STATIC_ASSERT(4801606 == AV_CODEC_ID_IDF);
STATIC_ASSERT(5198918 == AV_CODEC_ID_OTF);
STATIC_ASSERT(407917392 == AV_CODEC_ID_PCM_S24LE_PLANAR);
STATIC_ASSERT(542135120 == AV_CODEC_ID_PCM_S32LE_PLANAR);
STATIC_ASSERT(808530518 == AV_CODEC_ID_012V);
STATIC_ASSERT(809850962 == AV_CODEC_ID_EXR);
// AV_CODEC_ID_8SVX_RAW(944985688) is no longer supported by ffmpeg but remains
// in histograms.xml for historical purposes.
STATIC_ASSERT(1095123744 == AV_CODEC_ID_ADPCM_AFC);
STATIC_ASSERT(1096176208 == AV_CODEC_ID_AVRP);
STATIC_ASSERT(1096176238 == AV_CODEC_ID_AVRN);
STATIC_ASSERT(1096176969 == AV_CODEC_ID_AVUI);
STATIC_ASSERT(1096373590 == AV_CODEC_ID_AYUV);
STATIC_ASSERT(1112557912 == AV_CODEC_ID_BRENDER_PIX);
STATIC_ASSERT(1112823892 == AV_CODEC_ID_BINTEXT);
STATIC_ASSERT(1129335105 == AV_CODEC_ID_CPIA);
STATIC_ASSERT(1160852272 == AV_CODEC_ID_ESCAPE130);
STATIC_ASSERT(1179014995 == AV_CODEC_ID_FFWAVESYNTH);
STATIC_ASSERT(1211250229 == AV_CODEC_ID_HEVC);
STATIC_ASSERT(1246975298 == AV_CODEC_ID_JACOSUB);
STATIC_ASSERT(1263294017 == AV_CODEC_ID_SMPTE_KLV);
STATIC_ASSERT(1297108018 == AV_CODEC_ID_MPL2);
STATIC_ASSERT(1297498929 == AV_CODEC_ID_MVC1);
STATIC_ASSERT(1297498930 == AV_CODEC_ID_MVC2);
STATIC_ASSERT(1330333984 == AV_CODEC_ID_ADPCM_IMA_OKI);
STATIC_ASSERT(1330664787 == AV_CODEC_ID_OPUS);
STATIC_ASSERT(1346455105 == AV_CODEC_ID_PAF_AUDIO);
STATIC_ASSERT(1346455126 == AV_CODEC_ID_PAF_VIDEO);
STATIC_ASSERT(1347637264 == AV_CODEC_ID_PCM_S16BE_PLANAR);
STATIC_ASSERT(1349012051 == AV_CODEC_ID_PJS);
STATIC_ASSERT(1381259348 == AV_CODEC_ID_REALTEXT);
STATIC_ASSERT(1396788553 == AV_CODEC_ID_SAMI);
STATIC_ASSERT(1396788813 == AV_CODEC_ID_SANM);
STATIC_ASSERT(1397180754 == AV_CODEC_ID_SGIRLE);
STATIC_ASSERT(1397706307 == AV_CODEC_ID_SONIC);
STATIC_ASSERT(1397706316 == AV_CODEC_ID_SONIC_LS);
STATIC_ASSERT(1397909872 == AV_CODEC_ID_SUBRIP);
STATIC_ASSERT(1398953521 == AV_CODEC_ID_SUBVIEWER1);
STATIC_ASSERT(1400201814 == AV_CODEC_ID_SUBVIEWER);
STATIC_ASSERT(1412575542 == AV_CODEC_ID_TARGA_Y216);
STATIC_ASSERT(1446195256 == AV_CODEC_ID_V308);
STATIC_ASSERT(1446260792 == AV_CODEC_ID_V408);
STATIC_ASSERT(1447644481 == AV_CODEC_ID_VIMA);
STATIC_ASSERT(1448111218 == AV_CODEC_ID_VPLAYER);
STATIC_ASSERT(1465275476 == AV_CODEC_ID_WEBVTT);
STATIC_ASSERT(1480739150 == AV_CODEC_ID_XBIN);
STATIC_ASSERT(1480999235 == AV_CODEC_ID_XFACE);
STATIC_ASSERT(1496592720 == AV_CODEC_ID_Y41P);
STATIC_ASSERT(1498764852 == AV_CODEC_ID_YUV4);
STATIC_ASSERT(1664495672 == AV_CODEC_ID_EIA_608);
STATIC_ASSERT(1833195076 == AV_CODEC_ID_MICRODVD);
STATIC_ASSERT(1936029283 == AV_CODEC_ID_EVRC);
STATIC_ASSERT(1936944502 == AV_CODEC_ID_SMV);
STATIC_ASSERT(1950507339 == AV_CODEC_ID_TAK);

class FFmpegCommonTest : public testing::Test {
 public:
  FFmpegCommonTest() {
    FFmpegGlue::InitializeFFmpeg();
  }
  ~FFmpegCommonTest() override{};
};

TEST_F(FFmpegCommonTest, OpusAudioDecoderConfig) {
  AVCodecContext context = {0};
  context.codec_type = AVMEDIA_TYPE_AUDIO;
  context.codec_id = AV_CODEC_ID_OPUS;
  context.channel_layout = CHANNEL_LAYOUT_STEREO;
  context.channels = 2;
  context.sample_fmt = AV_SAMPLE_FMT_FLT;

  // During conversion this sample rate should be changed to 48kHz.
  context.sample_rate = 44100;

  AudioDecoderConfig decoder_config;
  AVCodecContextToAudioDecoderConfig(&context, false, &decoder_config, false);
  EXPECT_EQ(48000, decoder_config.samples_per_second());
}

TEST_F(FFmpegCommonTest, TimeBaseConversions) {
  const int64 test_data[][5] = {
    {1, 2, 1, 500000, 1 },
    {1, 3, 1, 333333, 1 },
    {1, 3, 2, 666667, 2 },
  };

  for (size_t i = 0; i < arraysize(test_data); ++i) {
    SCOPED_TRACE(i);

    AVRational time_base;
    time_base.num = static_cast<int>(test_data[i][0]);
    time_base.den = static_cast<int>(test_data[i][1]);

    base::TimeDelta time_delta =
        ConvertFromTimeBase(time_base, test_data[i][2]);

    EXPECT_EQ(time_delta.InMicroseconds(), test_data[i][3]);
    EXPECT_EQ(ConvertToTimeBase(time_base, time_delta), test_data[i][4]);
  }
}

TEST_F(FFmpegCommonTest, VerifyFormatSizes) {
  for (AVSampleFormat format = AV_SAMPLE_FMT_NONE;
       format < AV_SAMPLE_FMT_NB;
       format = static_cast<AVSampleFormat>(format + 1)) {
    SampleFormat sample_format = AVSampleFormatToSampleFormat(format);
    if (sample_format == kUnknownSampleFormat) {
      // This format not supported, so skip it.
      continue;
    }

    // Have FFMpeg compute the size of a buffer of 1 channel / 1 frame
    // with 1 byte alignment to make sure the sizes match.
    int single_buffer_size = av_samples_get_buffer_size(NULL, 1, 1, format, 1);
    int bytes_per_channel = SampleFormatToBytesPerChannel(sample_format);
    EXPECT_EQ(bytes_per_channel, single_buffer_size);
  }
}

TEST_F(FFmpegCommonTest, UTCDateToTime_Valid) {
  base::Time result;
  EXPECT_TRUE(FFmpegUTCDateToTime("2012-11-10 12:34:56", &result));

  base::Time::Exploded exploded;
  result.UTCExplode(&exploded);
  EXPECT_TRUE(exploded.HasValidValues());
  EXPECT_EQ(2012, exploded.year);
  EXPECT_EQ(11, exploded.month);
  EXPECT_EQ(6, exploded.day_of_week);
  EXPECT_EQ(10, exploded.day_of_month);
  EXPECT_EQ(12, exploded.hour);
  EXPECT_EQ(34, exploded.minute);
  EXPECT_EQ(56, exploded.second);
  EXPECT_EQ(0, exploded.millisecond);
}

TEST_F(FFmpegCommonTest, UTCDateToTime_Invalid) {
  const char* invalid_date_strings[] = {
    "",
    "2012-11-10",
    "12:34:56",
    "-- ::",
    "2012-11-10 12:34:",
    "2012-11-10 12::56",
    "2012-11-10 :34:56",
    "2012-11- 12:34:56",
    "2012--10 12:34:56",
    "-11-10 12:34:56",
    "2012-11 12:34:56",
    "2012-11-10-12 12:34:56",
    "2012-11-10 12:34",
    "2012-11-10 12:34:56:78",
    "ABCD-11-10 12:34:56",
    "2012-EF-10 12:34:56",
    "2012-11-GH 12:34:56",
    "2012-11-10 IJ:34:56",
    "2012-11-10 12:JL:56",
    "2012-11-10 12:34:MN",
    "2012-11-10 12:34:56.123",
    "2012-11-1012:34:56",
    "2012-11-10 12:34:56 UTC",
  };

  for (size_t i = 0; i < arraysize(invalid_date_strings); ++i) {
    const char* date_string = invalid_date_strings[i];
    base::Time result;
    EXPECT_FALSE(FFmpegUTCDateToTime(date_string, &result))
        << "date_string '" << date_string << "'";
    EXPECT_TRUE(result.is_null());
  }
}

}  // namespace media
