// Copyright (c) 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef MEDIA_VIDEO_CAPTURE_VIDEO_CAPTURER_SOURCE_H_
#define MEDIA_VIDEO_CAPTURE_VIDEO_CAPTURER_SOURCE_H_

#include <string>
#include <vector>

#include "base/basictypes.h"
#include "base/memory/ref_counted.h"
#include "base/single_thread_task_runner.h"
#include "media/base/media_export.h"
#include "media/base/video_capture_types.h"
#include "media/base/video_frame.h"

namespace media {

// VideoCapturerSource is an interface representing the source for
// captured video.  An implementation will periodically call the frame
// callback with new video frames.
class MEDIA_EXPORT VideoCapturerSource {
 public:
  virtual ~VideoCapturerSource();

  // This callback is used to deliver video frames.
  //
  // |estimated_capture_time| - The capture time of the delivered video
  // frame. This field represents the local time at which either: 1) the frame
  // was generated, if it was done so locally; or 2) the targeted play-out time
  // of the frame, if it was generated from a remote source. Either way, an
  // implementation should not present the frame before this point-in-time. This
  // value is NOT a high-resolution timestamp, and so it should not be used as a
  // presentation time; but, instead, it should be used for buffering playback
  // and for A/V synchronization purposes. NOTE: It is possible for this value
  // to be null if the current implementation lacks this timing information.
  //
  // |video_frame->timestamp()| gives the presentation timestamp of the video
  // frame relative to the first frame generated by the corresponding source.
  // Because a source can start generating frames before a subscriber is added,
  // the first video frame delivered may not have timestamp equal to 0.
  typedef base::Callback<
   void(const scoped_refptr<media::VideoFrame>& video_frame,
        const base::TimeTicks& estimated_capture_time)>
      VideoCaptureDeliverFrameCB;

  typedef base::Callback<void(const media::VideoCaptureFormats&)>
      VideoCaptureDeviceFormatsCB;

  typedef base::Callback<void(bool)> RunningCallback;

  // Collects the formats that can currently be used.
  // |max_requested_height|, |max_requested_width|, and
  // |max_requested_frame_rate| is used by Tab and Screen capture to decide what
  // resolution/framerate to generate. |callback| is triggered when the formats
  // have been collected.
  virtual void GetCurrentSupportedFormats(
      int max_requested_width,
      int max_requested_height,
      double max_requested_frame_rate,
      const VideoCaptureDeviceFormatsCB& callback) = 0;

  // Starts capturing frames using the resolution in |params|.
  // |new_frame_callback| is triggered on |frame_callback_task_runner|
  // when a new video frame is available.
  // If capturing is started successfully then |running_callback| will be
  // called with a parameter of true. Note that some implementations may
  // simply reject StartCapture (by calling running_callback with a false
  // argument) if called with the wrong task runner.
  // If capturing fails to start or stopped due to an external event then
  // |running_callback| will be called with a parameter of false.
  // |running_callback| will always be called on the same thread as the
  // StartCapture.
  virtual void StartCapture(
      const media::VideoCaptureParams& params,
      const VideoCaptureDeliverFrameCB& new_frame_callback,
      scoped_refptr<base::SingleThreadTaskRunner> frame_callback_task_runner,
      const RunningCallback& running_callback) = 0;

  // Stops capturing frames and clears all callbacks including the
  // SupportedFormatsCallback callback. Note that queued frame callbacks
  // may still occur after this call, so the caller must take care to
  // use refcounted or weak references in |new_frame_callback|.
  virtual void StopCapture() = 0;
};

}  // namespace media

#endif  // MEDIA_VIDEO_CAPTURE_VIDEO_CAPTURER_SOURCE_H_
