/****************************************************************************
**
** Copyright (C) 2024 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of the release tools of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:GPL-EXCEPT$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3 as published by the Free Software
** Foundation with exceptions as appearing in the file LICENSE.GPL3-EXCEPT
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/

function isToolAvailable(toolName) {
    code = installer.execute("which", new Array(toolName))[1]
    if (code == 0) {
        return true
    } else {
        return false
    }
}

function getXCBInstallInstructions() {
    // Get example install commands based on the package managers are available in PATH
    // to install xcb-cursor library
    var msg = ""
    if (isToolAvailable("apt")) {
        msg += qsTr("Ubuntu/Debian-based:<br>")
        msg += "'apt install libxcb-cursor0 libxcb-cursor-dev'<br><br>"
    } else if (isToolAvailable("apt-get")) {
        msg += qsTr("Ubuntu/Debian-based:<br>")
        msg += "'apt-get install libxcb-cursor0 libxcb-cursor-dev'<br><br>"
    }
    if (isToolAvailable("dnf")) {
        msg += qsTr("Fedora/RHEL-based:<br>")
        msg += "'dnf install xcb-util-cursor xcb-util-cursor-devel'<br><br>"
    } else if (isToolAvailable("yum")) {
        msg += qsTr("Fedora/RHEL-based:<br>")
        msg += "'yum install xcb-util-cursor xcb-util-cursor-devel'<br><br>"
    }
    if (isToolAvailable("pacman")) {
        msg += qsTr("Arch-based:<br>")
        msg += "'pacman -S xcb-util-cursor'<br><br>"
    }
    if (isToolAvailable("zypper")) {
        msg += qsTr("openSUSE-based:<br>")
        msg += "'zypper install xcb-util-cursor xcb-util-cursor-devel'<br><br>"
    }
    if (isToolAvailable("emerge")) {
        msg += qsTr("Gentoo-based:<br>")
        msg += "'emerge x11-libs/xcb-util-cursor'<br><br>"
    }
    if (isToolAvailable("apk")) {
        msg += "Alpine Linux:<br>"
        msg += "'apk add xcb-util-cursor-dev'<br><br>"
    }
    if (isToolAvailable("eopkg")) {
        msg += "Solus:<br>"
        msg += "'eopkg install xcb-util-cursor'<br><br>"
    }
    return msg
}

function getXCBInstalledStatus() {
    // Check whether xcb-cursor is installed with pkg-config or ldconfig if available
    var xcbExists = false
    if (isToolAvailable("pkg-config")) {
        pkgconfigOutput = installer.execute(
            "pkg-config", new Array("--exists", "xcb-cursor")
        )[1]
        if (pkgconfigOutput == 0) {
            xcbExists = true
        }
    }
    if (!xcbExists && isToolAvailable("ldconfig")) {
        ldconfigOutput = installer.execute("ldconfig", new Array("-p"))[0]
        if (ldconfigOutput) {
            if (ldconfigOutput.includes("/sbin/ldconfig")) {
                ldconfigOutput = installer.execute("/sbin/ldconfig", new Array("-p"))[0]
            }
            if (ldconfigOutput.includes("xcb-cursor")) {
                xcbExists = true
            }
        }
    }
    return xcbExists
}

function showXCBMissingMBox() {
    var msg1 = qsTr("The required XCB cursor platform library was not found!")
    var msg2 = msg1 + "<br><br>"
    msg2 += qsTr(
        "Please use your distribution's package manager to install it.<br><br>"
    )
    msg2 += getXCBInstallInstructions()
    QMessageBox["warning"]("XCBError", msg1, msg2)
}

function showXcodePopup() {
    var xcodeFound = installer.execute(
        "/Applications/Xcode.app/Contents/Developer/usr/bin/xcodebuild",
        new Array("-version")
    )[0]
    var msg1 = qsTr("The required 'Xcode' installation")
    if (!xcodeFound) {
        msg1 += qsTr(" was not found!")
        var msg2 = msg1 + "<br><br>"
        msg2 += qsTr(
            "You need to install Xcode and set up Xcode command line tools.<br><br>"
        )
        msg2 += qsTr(
            "Download Xcode from the Mac App Store or https://developer.apple.com<br><br>"
        )
    } else {
        msg1 += qsTr(" is not active!")
        var msg2 = msg1 + "<br><br>"
        msg2 += qsTr(
            "An Xcode installation was found in the /Applications/ directory. <br><br>"
        )
        msg2 += qsTr("However, you need to set up Xcode command line tools.<br><br>")
    }
    msg2 += "To activate Xcode command line tools in your environment, run "
    msg2 += "'xcode-select --switch /path/to/Xcode.app' from the command line.<br>"
    QMessageBox["warning"]("XcodeError", msg1, msg2)
}

// constructor
function Component() {
    var platform = installer.value("os")
    if (platform == "x11") {
        if (!getXCBInstalledStatus()) {
            showXCBMissingMBox()
        }
    }
    if (platform == "mac") {
        if (installer.versionMatches(installer.value("FrameworkVersion"), ">=4.8.1")) {
            var xcodeActive = installer.execute("/usr/bin/xcodebuild", new Array("-version"))[0];
            if (!xcodeActive) {
                showXcodePopup();
            }
        }
    }
}

Component.prototype.createOperations = function() {
    component.createOperations();
}
