/*****************************************************************************
 *
 * Testmanager - Graphical Automation and Visualisation Tool
 *
 * Copyright (C) 2018  Florian Pose <fp@igh.de>
 *
 * This file is part of Testmanager.
 *
 * Testmanager is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * Testmanager is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with Testmanager. If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#include "Property.h"
#include "PropertyGroup.h"
#include "PropertyModel.h"
#include "PropertyFlag.h"
#include "WidgetContainer.h"
#include "Plugin.h"

#include <QDebug>

/****************************************************************************/

Property::Property(PropertyGroup *group, QMetaProperty prop):
    PropertyNode(group, prop.name()),
    metaProperty(prop)
{
    if (metaProperty.isFlagType()) {
        const QMetaEnum &metaEnum(metaProperty.enumerator());
        for (int i = 0; i < metaEnum.keyCount(); i++) {
            new PropertyFlag(this, metaEnum.key(i));
        }
    }
}

/****************************************************************************/

bool Property::isSet(const PropertyModel *model) const
{
    WidgetContainer *container = model->getContainer();
    return container && container->hasProperty(metaProperty.name());
}

/****************************************************************************/

void Property::reset(PropertyModel *model)
{
    foreach (WidgetContainer *container, model->getContainers()) {
        container->resetProperty(metaProperty.name());
    }

    model->notify(this, 1, 1);
}

/****************************************************************************/

QVariant
Property::nodeData(const PropertyModel *model, int role, int section) const
{
    QVariant ret;

    if (role == Qt::BackgroundRole) {
        PropertyGroup *group = dynamic_cast<PropertyGroup *>(getParentNode());
        if (group) {
            return model->getColor(
                    group->getColorIndex(),
                    group->getChildIndex(this));
        }
    }

    if (section == 0) {
        switch (role) {
            case Qt::DisplayRole:
                ret = getNodeName();
                break;
            case Qt::FontRole:
                if (isSet(model)) {
                    QFont font;
                    font.setBold(true);
                    ret = font;
                }
                break;
            default:
                break;
        }
    }
    else if (section == 1) {
        switch (role) {
            case Qt::DisplayRole:
            case Qt::EditRole: {
                QVariant value(getValue(model));
                if (metaProperty.isEnumType() || metaProperty.isFlagType()) {
                    QMetaEnum e(metaProperty.enumerator());
                    QByteArray keys(e.valueToKeys(value.toInt()));
                    if (keys.isEmpty()) {
                        ret = value;
                    }
                    else {
                        ret = keys;
                    }
                }
                else if (
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
                        value.typeId() == QMetaType::QPointF
#else
                        value.type() == QVariant::PointF
#endif
                ) {
                    QPointF p {value.toPointF()};
                    return QString("(%1, %2)").arg(p.x()).arg(p.y());
                }
                else if (
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
                        value.typeId() == QMetaType::QRect
#else
                        value.type() == QVariant::Rect
#endif
                ) {
                    QRect r {value.toRect()};
                    return QString("(%1, %2, %3, %4)")
                            .arg(r.x())
                            .arg(r.y())
                            .arg(r.width())
                            .arg(r.height());
                }
                else {
                    return value;
                }
            } break;

            case Qt::DecorationRole:
                if (
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
                        metaProperty.typeId() == QMetaType::QColor
#else
                        metaProperty.type() == QVariant::Color
#endif
                ) {
                    return getValue(model).value<QColor>();
                }
                break;

            case Qt::CheckStateRole:
                if (
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
                        metaProperty.typeId() == QMetaType::Bool
#else
                        metaProperty.type() == QVariant::Bool
#endif
                ) {
                    QVariant value(getValue(model));
                    return value.toBool() ? Qt::Checked : Qt::Unchecked;
                }
                break;

            default:
                break;
        }
    }

    return ret;
}

/****************************************************************************/

Qt::ItemFlags Property::nodeFlags(
        const PropertyModel *model,
        Qt::ItemFlags flags,
        int section) const
{
    if (section == 1) {
        QVariant value(getValue(model));
        if (value.canConvert<QString>() or
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
            value.typeId() == QMetaType::QPointF
            or value.typeId() == QMetaType::QRect
#else
            value.type() == QVariant::PointF or value.type() == QVariant::Rect
#endif
        ) {
            flags |= Qt::ItemIsEditable;
        }
        if (
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
                metaProperty.typeId() == QMetaType::Bool
#else
                metaProperty.type() == QVariant::Bool
#endif
        ) {
            flags |= Qt::ItemIsUserCheckable;
        }
    }

    return flags;
}

/****************************************************************************/

static bool parsePointF(const QString &text, QPointF &out)
{
    QRegularExpression re(
            R"(\s*\(?\s*(-?\d+(?:\.\d+)?)\s*,\s*(-?\d+(?:\.\d+)?)\s*\)?)");
    QRegularExpressionMatch match = re.match(text);
    if (!match.hasMatch()) {
        return false;
    }

    out.setX(match.captured(1).toDouble());
    out.setY(match.captured(2).toDouble());
    return true;
}

/****************************************************************************/

static bool parseRect(const QString &text, QRect &out)
{
    QRegularExpression re(
            R"(\s*(-?\d+)\s*,\s*(-?\d+)\s*,\s*(-?\d+)\s*,\s*(-?\d+)\s*)");
    QRegularExpressionMatch match = re.match(text);
    if (!match.hasMatch()) {
        return false;
    }

    out.setX(match.captured(1).toInt());
    out.setY(match.captured(2).toInt());
    out.setWidth(match.captured(3).toInt());
    out.setHeight(match.captured(4).toInt());
    return true;
}

/****************************************************************************/

bool Property::nodeSetData(PropertyModel *model, const QVariant &value)
{
    bool ret {false};
    QVariant valueToSet {value};

    if (
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
            metaProperty.typeId() == QMetaType::QPointF
#else
            metaProperty.type() == QVariant::PointF
#endif
    ) {
        QPointF p;
        if (not parsePointF(value.toString(), p)) {
            return ret;
        }
        valueToSet = QVariant(p);
    }
    else if (
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
            metaProperty.typeId() == QMetaType::QRect
#else
            metaProperty.type() == QVariant::Rect
#endif
    ) {
        QRect r;
        if (not parseRect(value.toString(), r)) {
            return ret;
        }
        valueToSet = QVariant(r);
    }

    foreach (WidgetContainer *container, model->getContainers()) {
        if (container->setProperty(
                    metaProperty.name(),
                    valueToSet,
                    WidgetContainer::RequireExisting)) {
            ret = true;
        }
    }

    model->notify(this, 0, 1);  // also first column to indicate set property

    return ret;
}

/****************************************************************************/

QVariant Property::getValue(const PropertyModel *model) const
{
    WidgetContainer *container = model->getContainer();
    if (container) {
        if (QString::fromLocal8Bit(metaProperty.name()) != "geometry") {
            return metaProperty.read(container->getWidget());
        }
        else {
            /* geometry is a special case, since the widget is the child of a
             * WidgetContainer object, that is used to position the widget on
             * the tab. Thus if the geometry is read, return the geometry of
             * the parent container.
             */
            return metaProperty.read(container);
        }
    }
    else {
        return QVariant();
    }
}

/****************************************************************************/

void Property::notifySelfAndChildren(PropertyModel *model)
{
    model->notify(this, 0, 1);

    for (int i = 0; i < getChildCount(); i++) {
        PropertyNode *childNode(getChildNode(i));
        model->notify(childNode, 0, 1);
    }
}

/****************************************************************************/
