/*****************************************************************************
 *
 *  Copyright 2010 Richard Hacker (lerichi at gmx dot net)
 *            2023 Florian Pose <fp@igh.de>
 *
 *  This file is part of the pdserv library.
 *
 *  The pdserv library is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published
 *  by the Free Software Foundation, either version 3 of the License, or (at
 *  your option) any later version.
 *
 *  The pdserv library is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 *  License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with the pdserv library. If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#include "Main.h"

#include "config.h"
#include "pdserv.h"

#include "Event.h"
#include "Parameter.h"
#include "Pointer.h"
#include "ShmemDataStructures.h"
#include "Signal.h"
#include "Task.h"

#include "../Config.h"
#include "../Database.h"
#include "../Debug.h"
#include "../Exceptions.h"
#include "../Session.h"

#include <cerrno>       // EIO / errno
#include <cstdio>       // perror()
#include <fstream>      // ofstream
#include <iostream>
#include <sstream>

#include <pthread.h>
#include <signal.h>     // signal()
#include <sys/mman.h>   // mmap(), munmap()
#include <sys/time.h>   // gettimeofday()
#include <sys/wait.h>
#include <unistd.h>     // exit(), sleep()
#include <yaml.h>

#ifdef _PDSERV_CUSTOM_GCOV_HOOK
extern "C" void __gcov_dump(void);
#endif  // _PDSERV_CUSTOM_GCOV_HOOK

/////////////////////////////////////////////////////////////////////////////
struct SDO {
    enum {ParamChange = 1, PollSignal} type;

    union {
        struct {
            const Parameter *parameter;
            unsigned int offset;
            unsigned int count;
        } paramChangeReq;

        struct {
            int rv;
            struct timespec time;
        } paramChangeAck;

        const Signal* signal;
        struct timespec time;
    };
};

/////////////////////////////////////////////////////////////////////////////
const double Main::bufferTime = 2.0;

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
Main::Main( const char *name, const char *version,
        int (*gettime)(struct timespec*)):
    PdServ::Main(name, version),
    pthread::Thread(std::string("pdserv-main")),
    rttime(gettime ? gettime : &PdServ::Main::localtime)
{}

/////////////////////////////////////////////////////////////////////////////
Main::~Main()
{
    int cancel_state;
    pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, &cancel_state);

    terminate();
    join();


    while (task.size()) {
        delete task.front();
        task.pop_front();
    }

    while (parameters.size()) {
        delete parameters.front();
        parameters.pop_front();
    }

    // close pipes
    ::close(terminatePipe);
    ::close(ipcTx);
    ::close(ipcRx);
    ::close(nrtFeedbackPipe);

    if (pid > 0)
    {
        int stat = 0;
        waitpid(pid, &stat, 0);
    }

    if (shmem)
        ::munmap(shmem, shmem_len);
    pthread_setcancelstate(cancel_state, nullptr);
}

/////////////////////////////////////////////////////////////////////////////
void Main::setConfigFile(const char *file)
{
    configFile = file;
}

/////////////////////////////////////////////////////////////////////////////
void Main::setParameterWriteLock(void (*fn)(int, void*), void* priv_data)
{
    writelock_cb = fn;
    writelock_data = priv_data;
}

/////////////////////////////////////////////////////////////////////////////
int Main::setup()
{
    int rv;
    int ipc_pipe[4][2];
    time_t persistTimeout;
    fd_set fds;
    struct timeval timeout, now, delay;
    struct EventData* eventData = nullptr;

    rv = readConfiguration();
    if (rv) {
        return rv;
    }

    rv = setupLogging(configFile);
    if (rv) {
        return rv;
    }

    persistTimeout = setupPersistent();

    auto exportPath {m_config["eventexport"].toString()};
    if (not exportPath.empty()) {
        exportEvents(exportPath);
    }

    auto importLanguage {m_config["importmessageslanguage"].toString()};
    if (importLanguage.empty()) {
        importLanguage = "en"; // default value
    }
    auto importPath {m_config["importmessages"].toString()};
    if (not importPath.empty()) {
        importMessages(importPath, importLanguage);
    }

    // Initialize library
    rv = prefork_init();
    if (rv)
        return rv;

    // Open a pipe between the two processes. This is used to inform the
    // child that the parent has died
    if (::pipe(ipc_pipe[0]) or ::pipe(ipc_pipe[1]) or ::pipe(ipc_pipe[2])
        or ::pipe(ipc_pipe[3])) {
        rv = errno;
        ::perror("pipe()");
        return rv;
    }

    // Immediately split off a child. The parent returns to the caller so
    // that he can get on with his job.
    //
    // The child continues from here.
    //
    // It is intentional that the child has the same process group as
    // the parent so that it gets all the signals too.
    pid = ::fork();
    if (pid < 0) {
        // Some error occurred
        ::perror("fork()");
        return errno;
    }
    else if (pid) {
        // Parent here. Return to the caller
        ::close(ipc_pipe[0][0]);
        ipcTx = ipc_pipe[0][1];

        ipcRx = ipc_pipe[1][0];
        ::close(ipc_pipe[1][1]);

        ::close(ipc_pipe[2][0]);
        terminatePipe = ipc_pipe[2][1];

        nrtFeedbackPipe = ipc_pipe[3][0];
        ::close(ipc_pipe[3][1]);

        // Send PID to the child, indicating that parent is running
        if (::write(terminatePipe, &pid, sizeof(pid)) != sizeof(pid)) {
            perror("Main::setup(): pid ::write() failed");
            return errno;
        }
        // waiting for child to initialize
        if (::read(nrtFeedbackPipe, &rv, sizeof(rv)) != sizeof(rv)) {
            rv = errno;
            perror("Main::setup(): nRT setup: read() failed");
            ::close(nrtFeedbackPipe);
            nrtFeedbackPipe = -1;
            return rv;
        }
        if (rv != 0) {
            // child init has failed, it will terminate itself
            ::close(nrtFeedbackPipe);
            nrtFeedbackPipe = -1;
            return rv;
        }

        return postfork_rt_setup();
    }

    // Only child runs after this point
    ipcRx = ipc_pipe[0][0];
    ::close(ipc_pipe[0][1]);

    ::close(ipc_pipe[1][0]);
    ipcTx = ipc_pipe[1][1];

    terminatePipe = ipc_pipe[2][0];
    ::close(ipc_pipe[2][1]);

    ::close(ipc_pipe[3][0]);
    nrtFeedbackPipe = ipc_pipe[3][1];

    // Wait till main thread has been initialized
    if (::read(terminatePipe, &pid, sizeof(pid)) != sizeof(pid)
        or pid != getpid()) {
        perror("Main::setup(): pid ::read() failed");
        std::quick_exit(errno);
    }

    // Ignore common terminating signals
    ::signal(SIGINT, SIG_IGN);
    ::signal(SIGTERM, SIG_IGN);

    postfork_nrt_setup();

    rv = 0;
    try {
        startServers();
    }
    catch (PdServ::Errno const &err) {
        log_debug("Caught Pdserv::Errno(%i) while starting servers.",
                err.getErrno());
        rv = -err.getErrno();
    }
    catch (std::exception &e) {
        log_debug("Caught std::exception while starting servers: %s",
                e.what());
        rv = -1;
    }
    catch (...) {
        log_debug("Caught unknown exception while starting servers.");
        rv = -1;
    }

    if (::write(nrtFeedbackPipe, &rv, sizeof(rv)) != sizeof(rv)) {
        perror("Main::setup(): nRT setup: write() failed");
        close(nrtFeedbackPipe);
        std::quick_exit(errno);
    }
    if (rv == -1)
    {
        close(nrtFeedbackPipe);
#ifdef _PDSERV_CUSTOM_GCOV_HOOK
        __gcov_dump();
#endif
        std::quick_exit(-1);
    }

    try {
        postfork_nrt_subscribe_persistent_signals();
    } catch (PdServ::Main::RtProcessExited&) {
        goto exit;
    }

    ::gettimeofday(&timeout, 0);
    timeout.tv_sec += persistTimeout;

    FD_ZERO(&fds);

    // Stay in this loop until real-time thread exits, in which case
    // ipc_pipe[0] becomes readable
    eventData = eventDataStart;
    ipc_error = false;
    do {
        try {
        for (TaskList::iterator it = task.begin();
                it != task.end(); ++it)
            static_cast<Task*>(*it)->nrt_update();
        } catch (RtProcessExited&) {
            rv = 0;
            break;
        }

        if (persistTimeout) {
            if (::gettimeofday(&now, 0)) {
                rv = errno;
                break;
            }

            if ( now.tv_sec >= timeout.tv_sec) {
                timeout.tv_sec += persistTimeout;
                savePersistent();
            }
        }

        while (eventData != *eventDataWp) {
            newEvent(eventData->event, eventData->index,
                    eventData->state, &eventData->time);
            if (++eventData == eventDataEnd)
                eventData = eventDataStart;
        }

        FD_SET(terminatePipe, &fds);
        delay.tv_sec = 0;
        delay.tv_usec = 50000;  // 20Hz
        rv = ::select(terminatePipe + 1, &fds, 0, 0, &delay);
    } while (!(rv or ipc_error));

exit:
    // Ignore rv if ipc_pipe[0] is readable
    if (rv == 1) {
        rv = 0;
    }

    stopServers();

    close(nrtFeedbackPipe);
#ifdef _PDSERV_CUSTOM_GCOV_HOOK
    __gcov_dump();
#endif
    std::quick_exit(rv);
}

/////////////////////////////////////////////////////////////////////////////

void Main::sleep(int msecs) const
{
    fd_set fds;
    FD_ZERO(&fds);
    FD_SET(terminatePipe, &fds);

    struct timeval delay;

    delay.tv_sec = msecs / 1000;
    delay.tv_usec = (msecs - delay.tv_sec * 1000) * 1000;

    const int rv = ::select(terminatePipe + 1, &fds, 0, 0, &delay);
    if (rv == 1 or (rv == -1 && errno != EINTR)) {
        throw PdServ::Main::RtProcessExited{};
    }
}

/////////////////////////////////////////////////////////////////////////////
int Main::readConfiguration()
{
    const char *env;
    const char *err = 0;

    // Load custom configuration file
    if (!configFile.empty()) {
        err = m_config.load(configFile.c_str());
        if (err)
            std::cerr
                << "Error loading configuration file "
                << configFile << " specified on command line: "
                << err << std::endl;
        else {
            log_debug("Loaded specified configuration file %s",
                    configFile.c_str());
        }
    }
    else if ((env = ::getenv("PDSERV_CONFIG")) and ::strlen(env)) {
        // Try to load environment configuration file
        err = m_config.load(env);

        if (err)
            std::cerr << "Error loading configuration file " << env
                << " specified in environment variable PDSERV_CONFIG: "
                << err << std::endl;
        else {
            configFile = env;
            log_debug("Loaded ENV config %s", env);
        }
    }
    else {
        // Try to load default configuration file
        const char *f = QUOTE(SYSCONFDIR) "/pdserv.conf";

        if (::access(f, R_OK) == 0) {
            // file exists
            err = m_config.load(f);
            if (err) {
                std::cerr << "Error loading default configuration file "
                    << f << ": " << err << std::endl;
            }
            else {
                configFile = f;
                log_debug("Loaded default configuration file %s", f);
            }
        }
        else {
            std::cerr << "No configuration file found at " << f
                << ". Using defaults." << std::endl;
        }
    }

    if (!m_config) {
        log_debug("No configuration loaded");
    }

    if (err) {
        return -1;
    }

    return 0;
}

/////////////////////////////////////////////////////////////////////////////
PdServ::Config Main::config(const char* key) const
{
    return m_config[key];
}

/////////////////////////////////////////////////////////////////////////////
Task* Main::addTask(double sampleTime, const char *name)
{
    task.push_back(new Task(this, task.size(), sampleTime, name));
    return task.back();
}

/////////////////////////////////////////////////////////////////////////////
int Main::gettime(struct timespec* t) const
{
    return rttime(t);
}

/////////////////////////////////////////////////////////////////////////////
Event* Main::addEvent(const char *path, size_t nelem)
{
    events.push_back(std::unique_ptr<Event>{new Event(this, path, nelem)});
    return events.back().get();
}

/////////////////////////////////////////////////////////////////////////////
Parameter* Main::addParameter( const char *path,
        unsigned int mode, const PdServ::DataType& datatype,
        void *addr, size_t n, const size_t *dim)
{
    parameters.push_back(
            new Parameter(this, addr, path, mode, datatype, n, dim));

    return parameters.back();
}

/////////////////////////////////////////////////////////////////////////////
void Main::setEvent(const Event* event,
        size_t element, PdServ::Event::Priority prio, const timespec *time)
{
    pthread::MutexLock lock(eventMutex);

    struct EventData *eventData = *eventDataWp;
    eventData->event = event;
    eventData->index = element;
    eventData->state = prio;
    eventData->time  = *time;

    if (++eventData == eventDataEnd)
        eventData = eventDataStart;
    *eventDataWp = eventData;
}

/////////////////////////////////////////////////////////////////////////////
void Main::resetEvent(const Event* event,
        size_t element, const timespec *time)
{
    pthread::MutexLock lock(eventMutex);

    struct EventData *eventData = *eventDataWp;
    eventData->event = event;
    eventData->index = element;
    eventData->state = -1;
    eventData->time  = *time;

    if (++eventData == eventDataEnd)
        eventData = eventDataStart;
    *eventDataWp = eventData;
}

/////////////////////////////////////////////////////////////////////////////
static int yaml_output_handler(
        void *data,
        unsigned char *buffer,
        size_t size)
{
    std::ostream* out = static_cast<std::ostream *>(data);
    out->write(reinterpret_cast<char *>(buffer), size);
    return out->good() ? 1 : 0;
}

/////////////////////////////////////////////////////////////////////////////
static int emit_helper(yaml_emitter_t *emitter, yaml_event_t *event, int line)
{
    int ret = yaml_emitter_emit(emitter, event);
    if (not ret) {
        log_debug("YAML emit error in line %u: %s", line, emitter->problem);
    }
    return ret;
}

#define emit(emitter, event) \
    if (not emit_helper(emitter, event, __LINE__)) goto out_error;

/////////////////////////////////////////////////////////////////////////////
int Main::exportEvents(const std::string &path) const
{
    bool success {false};
    log_debug("Exporting events to %s", path.c_str());

    std::ofstream ofs(path);
    if (not ofs) {
        int ret = errno;
        std::stringstream err;
        err << "Failed to open " << path << ": " << strerror(ret);
        std::cerr << err.str() << std::endl;
        log_debug(err.str().c_str());
        return ret;
    }

    yaml_emitter_t emitter;
    yaml_event_t event;

    if (not yaml_emitter_initialize(&emitter)) {
        std::stringstream err;
        err << "Failed initialize YAML emitter: YAML error " << emitter.error;
        std::cerr << err.str() << std::endl;
        log_debug(err.str().c_str());
        return EINVAL;
    }

    yaml_emitter_set_output(&emitter, yaml_output_handler, &ofs);

    // Start stream
    yaml_stream_start_event_initialize(&event, YAML_UTF8_ENCODING);
    emit(&emitter, &event);

    // Start document
    yaml_document_start_event_initialize(&event, NULL, NULL, NULL, 0);
    emit(&emitter, &event);

    // Start Mapping
    yaml_mapping_start_event_initialize(&event, NULL, NULL, 1,
            YAML_BLOCK_MAPPING_STYLE);
    emit(&emitter, &event);

    // Version
    yaml_scalar_event_initialize(&event, NULL, NULL,
            (yaml_char_t *) "version", -1, 1, 0, YAML_PLAIN_SCALAR_STYLE);
    emit(&emitter, &event);

    yaml_scalar_event_initialize(&event, NULL, NULL,
            (yaml_char_t *) "1",
            -1, 1, 0, YAML_PLAIN_SCALAR_STYLE);
    emit(&emitter, &event);

    // Events
    yaml_scalar_event_initialize(&event, NULL, NULL,
            (yaml_char_t *) "events", -1, 1, 0, YAML_PLAIN_SCALAR_STYLE);
    emit(&emitter, &event);

    // Start sequence with events
    yaml_sequence_start_event_initialize(&event, NULL, NULL, 1,
            YAML_BLOCK_SEQUENCE_STYLE);
    emit(&emitter, &event);

    for (auto pdserv_event : getEvents()) {
        yaml_mapping_start_event_initialize(&event, NULL, NULL, 1,
                YAML_BLOCK_MAPPING_STYLE);
        emit(&emitter, &event);

        yaml_scalar_event_initialize(&event, NULL, NULL,
                (yaml_char_t *) "path", -1, 1, 0, YAML_PLAIN_SCALAR_STYLE);
        emit(&emitter, &event);

        yaml_scalar_event_initialize(&event, NULL, NULL,
                (yaml_char_t *) pdserv_event->path.c_str(),
                -1, 1, 0, YAML_PLAIN_SCALAR_STYLE);
        emit(&emitter, &event);

        yaml_scalar_event_initialize(&event, NULL, NULL,
                (yaml_char_t *) "nelem", -1, 1, 0, YAML_PLAIN_SCALAR_STYLE);
        emit(&emitter, &event);

        std::string nelem { std::to_string(pdserv_event->nelem()) };
        yaml_scalar_event_initialize(&event, NULL, NULL,
                (yaml_char_t *) nelem.c_str(),
                -1, 1, 0, YAML_PLAIN_SCALAR_STYLE);
        emit(&emitter, &event);

        yaml_scalar_event_initialize(&event, NULL, NULL,
                (yaml_char_t *) "messages", -1, 1, 0,
                YAML_PLAIN_SCALAR_STYLE);
        emit(&emitter, &event);

        // Start sequence with messages
        yaml_sequence_start_event_initialize(&event, NULL, NULL, 1,
                YAML_BLOCK_SEQUENCE_STYLE);
        emit(&emitter, &event);

        for (size_t i = 0; i < pdserv_event->nelem(); i++) {
            yaml_scalar_event_initialize(&event, NULL, NULL,
                    (yaml_char_t *) pdserv_event->messages[i].c_str(),
                    -1, 1, 0, YAML_PLAIN_SCALAR_STYLE);
            emit(&emitter, &event);
        }

        // End sequence
        yaml_sequence_end_event_initialize(&event);
        emit(&emitter, &event);

        yaml_mapping_end_event_initialize(&event);
        emit(&emitter, &event);
    }

    // End sequence
    yaml_sequence_end_event_initialize(&event);
    emit(&emitter, &event);

    // End mapping
    yaml_mapping_end_event_initialize(&event);
    emit(&emitter, &event);

    // End document
    yaml_document_end_event_initialize(&event, 1);
    emit(&emitter, &event);

    // End stream
    yaml_stream_end_event_initialize(&event);
    emit(&emitter, &event);

    success = true;

out_error:
    yaml_emitter_delete(&emitter);
    ofs.close();

    if (not ofs or not success) {
        log_debug("Failed to write all YAML data.");
        return EIO;
    }

    return 0;
}

/////////////////////////////////////////////////////////////////////////////
int Main::importMessages(const std::string &path, const std::string &lang)
{
    log_debug("Importing events from %s with language %s.",
            path.c_str(), lang.c_str());

    PdServ::Config yaml;
    const char *err {yaml.load(path.c_str())};
    if (err) {
        log_debug("Error opening %s: %s", path.c_str(), err);
        return -EIO;
    }

    if (yaml["content"].toString() != "EtherLabPlainMessages") {
        log_debug("Found \"%s\" as content. Aborting.",
                yaml["content"].toString().c_str());
        return -EPROTO;
    }

    if (yaml["version"].toInt() != 1) {
        log_debug("Found \"%s\" as content. "
                "Maximum supported version is 1. Aborting.",
                yaml["version"].toString().c_str());
        return -EPROTO;
    }

    auto messages = yaml["messages"];
    int index {0}, loaded {0};
    while (true) {
        auto message = messages[index];
        if (not message or not message.isMapping()) {
            break;
        }
        index++;
        std::string path {message["path"].toString()};
        std::vector<std::string> texts;
        auto elements = message["elements"];
        int elemIndex {0};
        while (true) {
            auto element = elements[elemIndex];
            if (not element or not element.isMapping()) {
                break;
            }
            std::string text {element["text"][lang].toString()};
            texts.push_back(text);
            elemIndex++;
        }

        Event *event {nullptr};
        for (const auto &e : events) {
            if (e->path == path) {
                event = e.get();
                break;
            }
        }

        if (not event) {
            log_debug("Event %s not found during message import.",
                    path.c_str());
            continue;
        }

        if (event->nelem() != texts.size()) {
            log_debug("Size mismatch with event %s (size %zu):"
                    " Imported message has %zu entries.",
                    path.c_str(), event->nelem(), texts.size());
            continue;
        }

        for (size_t i = 0; i < event->nelem(); i++) {
            event->messages[i] = texts[i];
        }

        loaded++;
    }

    log_debug("Loaded %u/%u messages.", loaded, index);
    return 0;
}

/////////////////////////////////////////////////////////////////////////////
int Main::setValue(const PdServ::ProcessParameter* p,
        const char* buf, size_t offset, size_t count,
        const char** value, const struct timespec** time)
{
    pthread::MutexLock lock(sdoMutex);
    const Parameter* param = static_cast<const Parameter*>(p);
    char* shmAddr = param->shmAddr + offset;

    // Backup old values in case of write failure
    char backup[count];
    std::copy(shmAddr, shmAddr + count, backup);

    // Copy new data to shared memory
    std::copy(buf, buf + count, shmAddr);

    // Setup change request
    struct SDO sdo;
    sdo.type = SDO::ParamChange;
    sdo.paramChangeReq.parameter = param;
    sdo.paramChangeReq.offset = offset;
    sdo.paramChangeReq.count = count;

    if (::write(ipcTx, &sdo, sizeof(sdo)) != sizeof(sdo)) {
        log_debug("Main::setValue(): SDO ::write() failed");
        ipc_error = true;
        return -EIO;
    }

    if (::read(ipcRx, &sdo, sizeof(sdo)) != sizeof(sdo)) {
        log_debug("Main::setValue(): SDO ::read() failed");
        ipc_error = true;
        return -EIO;
    }

    if (!sdo.paramChangeAck.rv)
        param->mtime = sdo.paramChangeAck.time; // Save time of update
    else
        // Write failure. Restore data
        std::copy(backup, backup + count, shmAddr);

    *value = param->shmAddr;
    *time = &param->mtime;

    return sdo.paramChangeAck.rv;
}

/////////////////////////////////////////////////////////////////////////////
void Main::initializeParameter(PdServ::Parameter* p,
        const char* data, const struct timespec* mtime,
        const PdServ::Signal* s)
{
    if (data) {
        log_debug("Restoring %s", p->path.c_str());
        const Parameter *parameter = static_cast<const Parameter*>(p);
        std::copy(data, data + parameter->memSize, parameter->addr);
        parameter->mtime = *mtime;
    }

    if (s) {
        const Signal* signal = static_cast<const Signal*>(s);
        signal->task->makePersistent(signal);
    }
}

/////////////////////////////////////////////////////////////////////////////
bool Main::getPersistentSignalValue(const PdServ::Signal *s,
        char* buf, struct timespec* time)
{
    const struct timespec* t;
    bool rv = static_cast<const Signal*>(s)->task->getPersistentValue(
            s, buf, &t);

    if (rv and time)
        *time = *t;

    return rv;
}

/////////////////////////////////////////////////////////////////////////////
int Main::getValue(const Signal *signal, void* dest, struct timespec* time)
{
    pthread::MutexLock lock(sdoMutex);
    struct SDO sdo;

    sdo.type = SDO::PollSignal;
    sdo.signal = signal;

    if (::write(ipcTx, &sdo, sizeof(sdo)) != sizeof(sdo)) {
        log_debug("Main::getValue(): SDO ::write() failed");
        ipc_error = true;
        return -EIO;
    }

    if (::read(ipcRx, &sdo, sizeof(sdo)) != sizeof(sdo)) {
        log_debug("Main::getValue(): SDO ::read() failed");
        ipc_error = true;
        return -EIO;
    }

    std::copy(signalData, signalData + signal->memSize,
            reinterpret_cast<char*>(dest));

    if (time)
        *time = sdo.time;

    return 0;
}

/////////////////////////////////////////////////////////////////////////////
PdServ::Parameter* Main::findParameter(const std::string& path) const
{
    for (ParameterList::const_iterator it = parameters.begin();
            it != parameters.end(); ++it)
        if ((*it)->path == path)
            return *it;
    return 0;
}

/////////////////////////////////////////////////////////////////////////////
// Organization of shared memory:
//      struct SDOStruct        sdo
//      char                    parameterData (binary data of all parameters)
//      char                    pdoData
//      struct EventData        eventDataStart
//
int Main::prefork_init()
{
    size_t numTasks = task.size();
    size_t taskMemSize[numTasks];
    size_t i, eventCount;
    size_t maxSignalSize = 0;

    log_debug("Initializing shared memory before fork()");

    // Find out the largest signal size to reserve space in
    // shared memory for polling
    for (TaskList::const_iterator it = task.begin();
            it != task.end(); ++it) {
        std::list<const PdServ::Signal*> signals =
            static_cast<PdServ::Task*>(*it)->getSignals();

        while (!signals.empty()) {
            const PdServ::Signal* s = signals.front();

            if (s->memSize > maxSignalSize)
                maxSignalSize = s->memSize;

            signals.pop_front();
        }

    }
    shmem_len += maxSignalSize;

    // The following two variables are used to organize parameters according
    // to the size of their elements so that their data type alignment is
    // correct.
    //
    // dataTypeIndex[] maps the data type to the index in parameterDataOffset,
    // e.g. a parameter with data type double (sizeof() = 8) will then go into
    // container parameterDataOffset[dataTypeIndex[8]]
    //
    // parameterDataOffset[] holds the start index of a data types with
    // 8, 4, 2 and 1 bytes alignment
    const size_t dataTypeIndex[PdServ::DataType::maxWidth+1] = {
        3 /*0*/, 3 /*1*/, 2 /*2*/, 3 /*3*/,
        1 /*4*/, 3 /*5*/, 3 /*6*/, 3 /*7*/, 0 /*8*/
    };
    size_t parameterDataOffset[5] = {0, 0, 0, 0, 0};    // need one extra!

    for (ParameterList::iterator it = parameters.begin();
            it != parameters.end(); it++) {
        const Parameter *p = static_cast<const Parameter*>(*it);

        // Push the next smaller data type forward by the parameter's
        // memory requirement
        parameterDataOffset[dataTypeIndex[p->dtype.align()] + 1] +=
            p->memSize;
    }

    // Accumulate the offsets so that they follow each other in the shared
    // data space. This also has the effect, that the value of
    // parameterDataOffset[4] is the total memory requirement of all
    // parameters
    for (i = 1; i < 5; ++i)
        parameterDataOffset[i] += parameterDataOffset[i-1];

    // Extend shared memory size with the parameter memory requirement
    // and as many sdo's for every parameter.
    shmem_len += parameterDataOffset[4];

    // Now check how much memory is required for events
    eventCount = 0;
    for (EventList::iterator it = events.begin(); it != events.end(); ++it)
        eventCount += (*it)->nelem();

    // Increase shared memory by the number of events as well as
    // enough capacity to store eventDataLen event changes
    const size_t eventLen = 10;     // Arbitrary
    shmem_len += sizeof(*eventDataStart) * eventLen * eventCount;

    shmem_len += sizeof(*eventDataWp);  // Memory location for write pointer

    // Find out the memory requirement for the tasks to pipe their variables
    // out of the real time environment
    i = 0;
    size_t bufLimit = m_config["sharedmemlimit"].toUInt();
    for (TaskList::const_iterator it = task.begin();
            it != task.end(); ++it) {
        taskMemSize[i] = ptr_align(
                static_cast<const Task*>(*it)->getShmemSpace(bufferTime,
                    bufLimit));
        shmem_len += taskMemSize[i++];
    }

    // Fudge factor. Every ptr_align can silently increase the pointer in
    // shmem by sizeof(unsigned long).
    // At the moment there are roughly 6 ptr_align's, take 10 to make sure!
    shmem_len += (10 + task.size())*sizeof(unsigned long);

    shmem = ::mmap(0, shmem_len, PROT_READ | PROT_WRITE,
            MAP_SHARED | MAP_ANON, -1, 0);
    if (MAP_FAILED == shmem) {
        // log(LOGCRIT, "could not mmap
        // err << "mmap(): " << strerror(errno);
        ::perror("mmap()");
        return errno;
    }

    // Clear memory; at the same time prefault it, so it does not
    // get swapped out
    ::memset(shmem, 0, shmem_len);

    // Now spread the shared memory for the users thereof

    // 1: Parameter data
    parameterData = ptr_align<char>(shmem);
    for (ParameterList::iterator it = parameters.begin();
            it != parameters.end(); it++) {
        Parameter *p = *it;
        p->shmAddr = parameterData
            + parameterDataOffset[dataTypeIndex[p->dtype.align()]];
        parameterDataOffset[dataTypeIndex[p->dtype.align()]] += p->memSize;

        std::copy(p->addr, p->addr + p->memSize, p->shmAddr);
    }

    // 2: Signal data area for polling
    signalData = ptr_align<char>(parameterData + parameterDataOffset[4]);

    // 3: Streaming data for tasks
    char* buf = ptr_align<char>(signalData + maxSignalSize);
    i = 0;
    for (TaskList::iterator it = task.begin(); it != task.end(); ++it) {
        static_cast<Task*>(*it)->prepare(buf, buf + taskMemSize[i]);
        buf = ptr_align<char>(buf + taskMemSize[i++]);
    }

    // 4: Event data
    struct EventData** ptr = ptr_align<struct EventData*>(buf);
    eventDataWp    = ptr++;
    eventDataStart = ptr_align<struct EventData>(ptr++);
    eventDataEnd   = eventDataStart + eventLen * eventCount;
    *eventDataWp   = eventDataStart;

    log_debug("shmem=%p shmem_end=%p(%zu)\n"
            "param=%p(%zi)\n"
            "stream=%p(%zi)\n"
            "end=%p(%zi)\n",
            shmem, (char*)shmem + shmem_len, shmem_len,
            signalData, (char*)signalData - (char*)shmem,
            eventDataWp, (char*)eventDataWp - (char*)shmem,
            eventDataEnd, (char*)eventDataEnd - (char*)shmem);

    if ((void*)(eventDataEnd + 1) > (void*)((char*)shmem + shmem_len)) {
        log_debug("Not enough memory");
        return ENOMEM;
    }

    log_debug("Finished.");
    return 0;
}

/////////////////////////////////////////////////////////////////////////////
int Main::postfork_rt_setup()
{
    log_debug("Initializing tasks after fork()");

    // Set the priority of the real time helper thread. This priority
    // should be set to the highest real time priority.
    set_priority(sched_get_priority_max(SCHED_FIFO));

    // Initialize the realtime tasks
    for (TaskList::iterator it = task.begin(); it != task.end(); ++it) {
        static_cast<Task*>(*it)->rt_init();
    }

    // Start supervisor thread
    int ret = start();

    log_debug("Finished.");
    return ret;
}

/////////////////////////////////////////////////////////////////////////////
int Main::postfork_nrt_setup()
{
    // Parent here; go back to the caller
    for (TaskList::iterator it = task.begin();
            it != task.end(); ++it)
        static_cast<Task*>(*it)->nrt_init();
    return 0;
}

/////////////////////////////////////////////////////////////////////////////

void Main::postfork_nrt_subscribe_persistent_signals()
{
    for (auto i : task)
        i->nrt_init_persistent();
}

/////////////////////////////////////////////////////////////////////////////
std::list<const PdServ::Parameter*> Main::getParameters() const
{
    return std::list<const PdServ::Parameter*>(
            parameters.begin(), parameters.end());
}

/////////////////////////////////////////////////////////////////////////////
std::list<const PdServ::Event*> Main::getEvents() const
{
    std::list<const PdServ::Event*> ans;
    for (const auto& e : events)
        ans.push_back(e.get());
    return ans;
}

/////////////////////////////////////////////////////////////////////////////
std::list<const PdServ::Task*> Main::getTasks() const
{
    return std::list<const PdServ::Task*>(task.begin(), task.end());
}

/////////////////////////////////////////////////////////////////////////////
void Main::prepare(PdServ::Session* session) const
{
    PdServ::Main::prepare(session);
}

/////////////////////////////////////////////////////////////////////////////
void Main::cleanup(const PdServ::Session* session) const
{
    PdServ::Main::cleanup(session);
}

/////////////////////////////////////////////////////////////////////////////
void Main::run()
{
    struct SDO sdo;

    while (true) {
        if (::read(ipcRx, &sdo, sizeof(sdo)) != sizeof(sdo))
            throw std::runtime_error("Main::run(): SDO ::read() failed");

        switch (sdo.type) {
            case SDO::ParamChange:
                sdo.paramChangeAck.rv = setParameterValue(
                        sdo.paramChangeReq.parameter,
                        sdo.paramChangeReq.offset,
                        sdo.paramChangeReq.count,
                        &sdo.paramChangeAck.time);

                log_debug("Parameter change rv=%i\n", sdo.paramChangeAck.rv);

                break;

            case SDO::PollSignal:
                sdo.signal->task->pollSignalValue(
                        sdo.signal, signalData, &sdo.time);
                break;
        };

        if (::write(ipcTx, &sdo, sizeof(sdo)) != sizeof(sdo))
            throw std::runtime_error("Main::run(): SDO ::write() failed");
    }
}

/////////////////////////////////////////////////////////////////////////////
int Main::setParameterValue(const Parameter* param,
        size_t offset, size_t len, struct timespec* mtime) const
{
    struct LockGuard {
        LockGuard(const Main* m): main(m) {
            if (main->writelock_cb)
                main->writelock_cb(1, main->writelock_data);
        }
        ~LockGuard() {
            if (main->writelock_cb)
                main->writelock_cb(0, main->writelock_data);
        }
        const Main* const main;
    };
    LockGuard guard(this);

    return param->setValue(offset, len, mtime);
}


/////////////////////////////////////////////////////////////////////////////
void Main::final()
{
    ::close(ipcRx);
    ::close(ipcTx);
}
