/***************************************************************************
 *   Copyright (C) 2004-2025 by Pere Constans
 *   constans@molspaces.com
 *   cb2Bib version 2.0.2.90. Licensed under the GNU GPL version 3.
 *   See the LICENSE file that comes with this distribution.
 ***************************************************************************/
#include "arxivXml.h"


arxivXml::arxivXml(const QString& xml) : QXmlStreamReader()
{
    _reference.clearReference();
    addData(xml);
    readReference();
}


void arxivXml::readReference()
{
    if (hasError())
        return;

    QStringList authors;

    while (reading("entry"))
        if (name().toLatin1() == "author")
            appendAuthor(&authors);
        else if (name().toLatin1() == "journal_ref")
            readField("note");
        else if (name().toLatin1() == "title")
            readField("title");
        else if (name().toLatin1() == "summary")
            readField("abstract");
        else if (name().toLatin1() == "doi")
            readField("doi");
        else if (name().toLatin1() == "id")
            processId();
        else if (name().toLatin1() == "published")
            processDate();

    _reference["author"] = authors.join(", ");
}

void arxivXml::readField(const QString& field)
{
    readNext();
    _reference[field] = text().toString();
}

void arxivXml::processId()
{
    readNext();
    QRegExp rxarxiv("abs/(.+)$");
    const int r(rxarxiv.indexIn(text().toString()));
    if (r >= 0)
    {
        QString aid(rxarxiv.cap(1));
        c2bUtils::remove(aid, QRegExp("v\\d{1,2}$"));
        _reference["eprint"] = aid;
        _reference["journal"] = "arXiv:" + aid;
        _reference["url"] = c2bUtils::arxivUrl.arg(aid);
    }
}

void arxivXml::processDate()
{
    readNext();
    _reference["year"] = text().toString().left(4);
}

void arxivXml::appendAuthor(QStringList* authors)
{
    while (reading("name"))
        if (name().toLatin1() == "name")
        {
            readNext();
            authors->append(text().toString());
        }
}

bool arxivXml::reading(const QString& tag)
{
    if (tokenType() == NoToken)
        while (!atEnd())
            if (readNext() == StartElement && name() == tag)
                return true;
    while (!atEnd())
    {
        TokenType t = readNext();
        if (t == StartElement)
            return true;
        else if (t == EndElement && name() == tag)
            return false;
    }
    return false;
}
