/*
 * Simple Virtual Machine - A versatile and robust architecture to
 * easily write applications.
 * Copyright (C) 2021  Julien BRUGUIER
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <src/global/global.h>
#include <src/global/installation.h>
#include <src/outils/outils.h>
#include <src/outils/analyseur2/analyseur.h>

void aide(std::ostream& os, const std::string& binaire)
{
	os	<< binaire << " [options]" << std::endl
		<< std::endl
		<< "This command produces an installable package of SVM content like applications, libraries and plugins." << std::endl 
		<< std::endl
		<< "Options:" << std::endl
		<< std::endl
		<< "\t-i <file>  : use file as input instead of stdin" << std::endl
		<< "\t-o <file>  : use file as output instead of stdout" << std::endl
		<< "\t-O         : use file <name>-<version>-installer.sh as output instead of stdout" << std::endl
		<< "\t-d         : generate package in dev mode (swap values of variables only_in_release and only_in_dev)" << std::endl
		<< "\t-y <flag>  : raise a flag (set the corresponding variable to YES)" << std::endl
		<< "\t-f <flavor>: add a flavor to the package name" << std::endl
		<< "\t-l         : display a list of supported OPTION and SET commands in the package description file" << std::endl
		<< std::endl
		<< std::endl
		<< "\t-h         : display this help" << std::endl
		<< "\t-v         : display svm_config version and license" << std::endl
		<< std::endl;
}

void version(std::ostream& os, const std::string& binaire)
{
	os	<< binaire << " version " << VERSION << "  Copyright (C) 2021  Julien BRUGUIER" << std::endl
		<< std::endl
		<< "This program comes with ABSOLUTELY NO WARRANTY." << std::endl
		<< "This is free software, and you are welcome to redistribute it under certain conditions." << std::endl
		<< "See GNU GPLv3 terms for details." << std::endl
		<< std::endl;
}

int main(int argc, char *argv[])
{
	::opterr = 0;
	int option_trouvee;
	std::string fichier_entree;
	std::string fichier_sortie;
	bool fichier_sortie_par_defaut = false;
	bool debug = false;
	std::set<std::string> drapeaux;
	std::string saveur;
	for( ; ; )
	{
		option_trouvee = ::getopt(argc,argv,"i:o:Ody:f:lhv");
		if(option_trouvee==EOF)
			break;
		switch(option_trouvee)
		{
			case 'i':
				fichier_entree = optarg;
				break;
			case 'o':
				fichier_sortie = optarg;
				break;
			case 'O':
				fichier_sortie_par_defaut = true;
				break;
			case 'd':
				debug = true;
				break;
			case 'y':
				drapeaux.insert(optarg);
				break;
			case 'f':
				saveur = optarg;
				break;
			case 'l':
				std::cout << std::endl
					<< "Supported list of OPTION and SET commands in package description file:" << std::endl
					<< std::endl;
				SVM::Outils::ConfigurationPaquet::details(std::cout);
				return 0;
				break;
			case 'h':
				std::cerr << std::endl;
				::aide(std::cerr,argv[0]);
				return 0;
				break;
			case 'v':
				std::cerr << std::endl;
				::version(std::cerr,argv[0]);
				return 0;
				break;
			case '?':
			default:
				std::cerr << "invalid option -" << std::string(1,optopt) << std::endl << std::endl;
				::aide(std::cerr,argv[0]);
				return 2;
				break;
		}
	}
	try
	{
		SVM::Outils::PaquetSP paquet;	
		if(fichier_entree.empty())
		{
			std::string fichier;
			for(std::string ligne ; std::getline(std::cin, ligne) ; )
			{
				fichier += ligne;
				fichier += '\n';
			}
			paquet = SVM::Outils::Analyseur::analyse_chaine(fichier);	
		}
		else
		{
			paquet = SVM::Outils::Analyseur::analyse_fichier(fichier_entree);
		}
		paquet->verification();
		paquet->construction(saveur,debug,drapeaux);
		if(fichier_sortie_par_defaut)
		{
			fichier_sortie = paquet->fichier_sortie_par_defaut();
		}
		paquet->generation(fichier_sortie);
	}
	catch(SVM::Outils::Erreur& e)
	{
		std::cerr << e << std::endl;
		return 1;
	}
	catch(std::exception& e)
	{
		std::cerr << e.what() << std::endl;
		return 2;
	}
	catch(...)
	{
		std::cerr << "Unexpected error." << std::endl;
		return 2;
	}
	return 0;
}
