"""Perform InnoDB recovery against a MySQL data directory"""

import logging
import os
import signal
import time

from holland.core.backup import BackupError

from ._mysqld import MySQLServer, generate_server_config, locate_mysqld_exe

LOG = logging.getLogger(__name__)


class InnodbRecoveryAction(object):
    """Perform InnoDB recovery against a MySQL data directory """

    def __init__(self, mysqld_config):
        self.mysqld_config = mysqld_config
        if "datadir" not in mysqld_config:
            raise BackupError("datadir must be set for InnodbRecovery")

    def __call__(self, event, snapshot_fsm, snapshot):
        LOG.info("Starting InnoDB recovery")

        mysqld_exe = locate_mysqld_exe(self.mysqld_config)
        LOG.info("Bootstrapping with %s", mysqld_exe)

        mycnf_path = os.path.join(self.mysqld_config["datadir"], "my.innodb_recovery.cnf")
        self.mysqld_config["log-error"] = "innodb_recovery.log"
        my_conf = generate_server_config(self.mysqld_config, mycnf_path)

        mysqld = MySQLServer(mysqld_exe, my_conf)
        mysqld.start(bootstrap=True)

        while mysqld.poll() is None:
            if signal.SIGINT in snapshot_fsm.sigmgr.pending:
                mysqld.kill(signal.SIGKILL)
            time.sleep(0.5)
        LOG.info("%s has stopped", mysqld_exe)

        if mysqld.returncode != 0:
            LOG.error("Configuration file was:")
            LOG.error("#######################")
            for line in open(os.path.join(mycnf_path), "r"):
                LOG.error("%s", line.rstrip())
            LOG.error("#######################")
            datadir = self.mysqld_config["datadir"]
            for line in open(os.path.join(datadir, "innodb_recovery.log"), "r"):
                LOG.error("%s", line.rstrip())
            raise BackupError(
                "%s exited with non-zero status (%s) during "
                "InnoDB recovery" % (mysqld_exe, mysqld.returncode)
            )
        LOG.info("%s ran successfully", mysqld_exe)
