/* sp.c
 *
 * Copyright (C) 2006-2022 wolfSSL Inc.
 *
 * This file is part of wolfSSL.
 *
 * wolfSSL is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * wolfSSL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1335, USA
 */

/* Implementation by Sean Parkinson. */

#ifdef HAVE_CONFIG_H
    #include <config.h>
#endif

#include <wolfssl/wolfcrypt/settings.h>

#if defined(WOLFSSL_HAVE_SP_RSA) || defined(WOLFSSL_HAVE_SP_DH) || \
    defined(WOLFSSL_HAVE_SP_ECC)

#include <wolfssl/wolfcrypt/error-crypt.h>
#include <wolfssl/wolfcrypt/cpuid.h>
#ifdef NO_INLINE
    #include <wolfssl/wolfcrypt/misc.h>
#else
    #define WOLFSSL_MISC_INCLUDED
    #include <wolfcrypt/src/misc.c>
#endif

#ifdef RSA_LOW_MEM
#ifndef WOLFSSL_SP_SMALL
#define WOLFSSL_SP_SMALL
#endif
#endif

#include <wolfssl/wolfcrypt/sp.h>

#ifdef WOLFSSL_SP_ARM64_ASM
#define SP_PRINT_NUM(var, name, total, words, bits)         \
    do {                                                    \
        int ii;                                             \
        fprintf(stderr, name "=0x");                        \
        for (ii = ((bits + 63) / 64) - 1; ii >= 0; ii--)    \
            fprintf(stderr, SP_PRINT_FMT, (var)[ii]);       \
        fprintf(stderr, "\n");                              \
    } while (0)

#define SP_PRINT_VAL(var, name)                             \
    fprintf(stderr, name "=0x" SP_PRINT_FMT "\n", var)

#define SP_PRINT_INT(var, name)                             \
    fprintf(stderr, name "=%d\n", var)

#if defined(WOLFSSL_HAVE_SP_RSA) || defined(WOLFSSL_HAVE_SP_DH)
#ifndef WOLFSSL_SP_NO_2048
/* Read big endian unsigned byte array into r.
 *
 * r  A single precision integer.
 * size  Maximum number of bytes to convert
 * a  Byte array.
 * n  Number of bytes in array to read.
 */
static void sp_2048_from_bin(sp_digit* r, int size, const byte* a, int n)
{
    sp_int64 nl = n;
    sp_int64 bytes = size * 8;

    __asm__ __volatile__ (
        "add	x4, %[a], %[n]\n\t"
        "mov	x5, %[r]\n\t"
        "sub	x4, x4, 8\n\t"
        "subs	x6, %[n], 8\n\t"
        "mov	x7, xzr\n\t"
        "blt	2f\n\t"
        /* Put in mulitples of 8 bytes. */
        "1:\n\t"
        "ldr	x8, [x4], -8\n\t"
        "subs	x6, x6, 8\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "rev	x8, x8\n\t"
    #endif
        "str	x8, [x5], 8\n\t"
        "add	x7, x7, 8\n\t"
        "b.ge	1b\n\t"
        "2:\n\t"
        "cmp	x6, -7\n\t"
        "b.lt	20f\n\t"
        /* Put in less than 8 bytes. */
    #ifdef LITTLE_ENDIAN_ORDER
        "str	xzr, [x5]\n\t"
    #else
        "str	xzr, [x5], 7\n\t"
    #endif
        "add	x7, x7, 8\n\t"
        "add	x4, x4, 7\n\t"
        "b.eq	17f\n\t"
        "cmp	x6, -5\n\t"
        "b.lt	16f\n\t"
        "b.eq	15f\n\t"
        "cmp	x6, -3\n\t"
        "b.lt	14f\n\t"
        "b.eq	13f\n\t"
        "cmp	x6, -2\n\t"
        "b.eq	12f\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "12:\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "13:\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "14:\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "15:\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "16:\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "17:\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "20:\n\t"
        "add	x5, %[r], x7\n\t"
        "subs	x7, %[size], x7\n\t"
        "b.eq	30f\n\t"
        /* Zero out remaining words. */
        "21:\n\t"
        "subs	x7, x7, 8\n\t"
        "str	xzr, [x5], 8\n\t"
        "b.gt	21b\n\t"
        "30:\n\t"
        :
        : [r] "r" (r), [size] "r" (bytes), [a] "r" (a), [n] "r" (nl)
        : "memory", "x4", "x5", "x6", "x7", "x8"
    );
}

/* Convert an mp_int to an array of sp_digit.
 *
 * r  A single precision integer.
 * size  Maximum number of bytes to convert
 * a  A multi-precision integer.
 */
static void sp_2048_from_mp(sp_digit* r, int size, const mp_int* a)
{
#if DIGIT_BIT == 64
    int i;
    int j = 0;

    for (i = 0; i < size; i++) {
        sp_digit mask =
            (((sp_digit)(a->used - i - 1)) >> (SP_WORD_SIZE - 1)) - 1;
        r[i] = a->dp[j] & mask;
        j += (int)(((sp_digit)1) -
                   (((sp_digit)(a->used - i - 2)) >> (SP_WORD_SIZE - 1)));
    }
#elif DIGIT_BIT > 64
    int i;
    int j = 0;
    word32 s = 0;

    r[0] = 0;
    for (i = 0; i < a->used && j < size; i++) {
        r[j] |= ((sp_digit)a->dp[i] << s);
        r[j] &= 0xffffffffffffffffl;
        s = 64U - s;
        if (j + 1 >= size) {
            break;
        }
        /* lint allow cast of mismatch word32 and mp_digit */
        r[++j] = (sp_digit)(a->dp[i] >> s); /*lint !e9033*/
        while ((s + 64U) <= (word32)DIGIT_BIT) {
            s += 64U;
            r[j] &= 0xffffffffffffffffl;
            if (j + 1 >= size) {
                break;
            }
            if (s < (word32)DIGIT_BIT) {
                /* lint allow cast of mismatch word32 and mp_digit */
                r[++j] = (sp_digit)(a->dp[i] >> s); /*lint !e9033*/
            }
            else {
                r[++j] = (sp_digit)0;
            }
        }
        s = (word32)DIGIT_BIT - s;
    }

    for (j++; j < size; j++) {
        r[j] = 0;
    }
#else
    int i;
    int j = 0;
    int s = 0;

    r[0] = 0;
    for (i = 0; i < a->used && j < size; i++) {
        r[j] |= ((sp_digit)a->dp[i]) << s;
        if (s + DIGIT_BIT >= 64) {
            r[j] &= 0xffffffffffffffffl;
            if (j + 1 >= size) {
                break;
            }
            s = 64 - s;
            if (s == DIGIT_BIT) {
                r[++j] = 0;
                s = 0;
            }
            else {
                r[++j] = a->dp[i] >> s;
                s = DIGIT_BIT - s;
            }
        }
        else {
            s += DIGIT_BIT;
        }
    }

    for (j++; j < size; j++) {
        r[j] = 0;
    }
#endif
}

/* Write r as big endian to byte array.
 * Fixed length number of bytes written: 256
 *
 * r  A single precision integer.
 * a  Byte array.
 */
static void sp_2048_to_bin_32(sp_digit* r, byte* a)
{
    int i;
    int j = 0;

    for (i = 31; i >= 0; i--, j += 8) {
        __asm__ __volatile__ (
            "ldr	x4, [%[r]]\n\t"
        #ifdef LITTLE_ENDIAN_ORDER
            "rev	x4, x4\n\t"
        #endif
            "str	x4, [%[a]]\n\t"
            :
            : [r] "r" (r + i), [a] "r" (a + j)
            : "memory", "x4"
        );
    }
}

#if (defined(WOLFSSL_HAVE_SP_RSA) && (!defined(WOLFSSL_RSA_PUBLIC_ONLY) || !defined(WOLFSSL_SP_SMALL))) || defined(WOLFSSL_HAVE_SP_DH)
/* Normalize the values in each word to 64.
 *
 * a  Array of sp_digit to normalize.
 */
#define sp_2048_norm_32(a)

#endif /* (WOLFSSL_HAVE_SP_RSA && (!WOLFSSL_RSA_PUBLIC_ONLY || !WOLFSSL_SP_SMALL)) || WOLFSSL_HAVE_SP_DH */
/* Normalize the values in each word to 64.
 *
 * a  Array of sp_digit to normalize.
 */
#define sp_2048_norm_32(a)

#ifndef WOLFSSL_SP_SMALL
/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_2048_mul_8(sp_digit* r, const sp_digit* a, const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x8, x9, [%[a], 0]\n\t"
        "ldp	x10, x11, [%[a], 16]\n\t"
        "ldp	x12, x13, [%[a], 32]\n\t"
        "ldp	x14, x15, [%[a], 48]\n\t"
        "ldp	x16, x17, [%[b], 0]\n\t"
        "ldp	x19, x20, [%[b], 16]\n\t"
        "ldp	x21, x22, [%[b], 32]\n\t"
        "ldp	x23, x24, [%[b], 48]\n\t"
        "#  A[0] * B[0]\n\t"
        "mul	x3, x8, x16\n\t"
        "umulh	x4, x8, x16\n\t"
        "str	x3, [%[r]]\n\t"
        "#  A[0] * B[1]\n\t"
        "mul	x6, x8, x17\n\t"
        "umulh	x7, x8, x17\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[1] * B[0]\n\t"
        "mul	x6, x9, x16\n\t"
        "adc	x5, xzr, x7\n\t"
        "umulh	x7, x9, x16\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "str	x4, [%[r], 8]\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[0] * B[2]\n\t"
        "mul	x6, x8, x19\n\t"
        "umulh	x7, x8, x19\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[1] * B[1]\n\t"
        "mul	x6, x9, x17\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x9, x17\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[2] * B[0]\n\t"
        "mul	x6, x10, x16\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x10, x16\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "str	x5, [%[r], 16]\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[0] * B[3]\n\t"
        "mul	x6, x8, x20\n\t"
        "umulh	x7, x8, x20\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[1] * B[2]\n\t"
        "mul	x6, x9, x19\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x9, x19\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[2] * B[1]\n\t"
        "mul	x6, x10, x17\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x10, x17\n\t"
        "adc	x5, x5, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[3] * B[0]\n\t"
        "mul	x6, x11, x16\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x11, x16\n\t"
        "adc	x5, x5, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "str	x3, [%[r], 24]\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[0] * B[4]\n\t"
        "mul	x6, x8, x21\n\t"
        "umulh	x7, x8, x21\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[1] * B[3]\n\t"
        "mul	x6, x9, x20\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x9, x20\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[2] * B[2]\n\t"
        "mul	x6, x10, x19\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x10, x19\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[3] * B[1]\n\t"
        "mul	x6, x11, x17\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x11, x17\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[4] * B[0]\n\t"
        "mul	x6, x12, x16\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x12, x16\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "str	x4, [%[r], 32]\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[0] * B[5]\n\t"
        "mul	x6, x8, x22\n\t"
        "umulh	x7, x8, x22\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[1] * B[4]\n\t"
        "mul	x6, x9, x21\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x9, x21\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[2] * B[3]\n\t"
        "mul	x6, x10, x20\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x10, x20\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[3] * B[2]\n\t"
        "mul	x6, x11, x19\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x11, x19\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[4] * B[1]\n\t"
        "mul	x6, x12, x17\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x12, x17\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[5] * B[0]\n\t"
        "mul	x6, x13, x16\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x13, x16\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "str	x5, [%[r], 40]\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[0] * B[6]\n\t"
        "mul	x6, x8, x23\n\t"
        "umulh	x7, x8, x23\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[1] * B[5]\n\t"
        "mul	x6, x9, x22\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x9, x22\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[2] * B[4]\n\t"
        "mul	x6, x10, x21\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x10, x21\n\t"
        "adc	x5, x5, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[3] * B[3]\n\t"
        "mul	x6, x11, x20\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x11, x20\n\t"
        "adc	x5, x5, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[4] * B[2]\n\t"
        "mul	x6, x12, x19\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x12, x19\n\t"
        "adc	x5, x5, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[5] * B[1]\n\t"
        "mul	x6, x13, x17\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x13, x17\n\t"
        "adc	x5, x5, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[6] * B[0]\n\t"
        "mul	x6, x14, x16\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x14, x16\n\t"
        "adc	x5, x5, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "str	x3, [%[r], 48]\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[0] * B[7]\n\t"
        "mul	x6, x8, x24\n\t"
        "umulh	x7, x8, x24\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[1] * B[6]\n\t"
        "mul	x6, x9, x23\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x9, x23\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[2] * B[5]\n\t"
        "mul	x6, x10, x22\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x10, x22\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[3] * B[4]\n\t"
        "mul	x6, x11, x21\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x11, x21\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[4] * B[3]\n\t"
        "mul	x6, x12, x20\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x12, x20\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[5] * B[2]\n\t"
        "mul	x6, x13, x19\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x13, x19\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[6] * B[1]\n\t"
        "mul	x6, x14, x17\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x14, x17\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[7] * B[0]\n\t"
        "mul	x6, x15, x16\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x15, x16\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "str	x4, [%[r], 56]\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[1] * B[7]\n\t"
        "mul	x6, x9, x24\n\t"
        "umulh	x7, x9, x24\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[2] * B[6]\n\t"
        "mul	x6, x10, x23\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x10, x23\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[3] * B[5]\n\t"
        "mul	x6, x11, x22\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x11, x22\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[4] * B[4]\n\t"
        "mul	x6, x12, x21\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x12, x21\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[5] * B[3]\n\t"
        "mul	x6, x13, x20\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x13, x20\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[6] * B[2]\n\t"
        "mul	x6, x14, x19\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x14, x19\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[7] * B[1]\n\t"
        "mul	x6, x15, x17\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x15, x17\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "str	x5, [%[r], 64]\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[2] * B[7]\n\t"
        "mul	x6, x10, x24\n\t"
        "umulh	x7, x10, x24\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[3] * B[6]\n\t"
        "mul	x6, x11, x23\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x11, x23\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[4] * B[5]\n\t"
        "mul	x6, x12, x22\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x12, x22\n\t"
        "adc	x5, x5, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[5] * B[4]\n\t"
        "mul	x6, x13, x21\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x13, x21\n\t"
        "adc	x5, x5, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[6] * B[3]\n\t"
        "mul	x6, x14, x20\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x14, x20\n\t"
        "adc	x5, x5, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[7] * B[2]\n\t"
        "mul	x6, x15, x19\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x15, x19\n\t"
        "adc	x5, x5, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "str	x3, [%[r], 72]\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[3] * B[7]\n\t"
        "mul	x6, x11, x24\n\t"
        "umulh	x7, x11, x24\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[4] * B[6]\n\t"
        "mul	x6, x12, x23\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x12, x23\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[5] * B[5]\n\t"
        "mul	x6, x13, x22\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x13, x22\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[6] * B[4]\n\t"
        "mul	x6, x14, x21\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x14, x21\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[7] * B[3]\n\t"
        "mul	x6, x15, x20\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x15, x20\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "str	x4, [%[r], 80]\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[4] * B[7]\n\t"
        "mul	x6, x12, x24\n\t"
        "umulh	x7, x12, x24\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[5] * B[6]\n\t"
        "mul	x6, x13, x23\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x13, x23\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[6] * B[5]\n\t"
        "mul	x6, x14, x22\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x14, x22\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[7] * B[4]\n\t"
        "mul	x6, x15, x21\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x15, x21\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "str	x5, [%[r], 88]\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[5] * B[7]\n\t"
        "mul	x6, x13, x24\n\t"
        "umulh	x7, x13, x24\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[6] * B[6]\n\t"
        "mul	x6, x14, x23\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x14, x23\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[7] * B[5]\n\t"
        "mul	x6, x15, x22\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x15, x22\n\t"
        "adc	x5, x5, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "str	x3, [%[r], 96]\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[6] * B[7]\n\t"
        "mul	x6, x14, x24\n\t"
        "umulh	x7, x14, x24\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[7] * B[6]\n\t"
        "mul	x6, x15, x23\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x15, x23\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "str	x4, [%[r], 104]\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[7] * B[7]\n\t"
        "mul	x6, x15, x24\n\t"
        "umulh	x7, x15, x24\n\t"
        "adds	x5, x5, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "stp	x5, x3, [%[r], 112]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x19", "x20", "x21", "x22", "x23", "x24"
    );
}

/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_2048_add_8(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "adds	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "cset	%[r], cs\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return (sp_digit)r;
}

/* Add digit to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_2048_add_word_8(sp_digit* r, const sp_digit* a,
        sp_digit b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "adds	x3, x3, %[b]\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "adcs	x3, x3, xzr\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6"
    );
}

/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer and result.
 * b  A single precision integer.
 */
static sp_digit sp_2048_sub_in_place_16(sp_digit* a, const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x2, x3, [%[a], 0]\n\t"
        "ldp	x6, x7, [%[b], 0]\n\t"
        "subs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 16]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 16]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 0]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 16]\n\t"
        "ldp	x2, x3, [%[a], 32]\n\t"
        "ldp	x6, x7, [%[b], 32]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 48]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 48]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 32]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 48]\n\t"
        "ldp	x2, x3, [%[a], 64]\n\t"
        "ldp	x6, x7, [%[b], 64]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 80]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 80]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 64]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 80]\n\t"
        "ldp	x2, x3, [%[a], 96]\n\t"
        "ldp	x6, x7, [%[b], 96]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 112]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 112]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 96]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 112]\n\t"
        "csetm	%[a], cc\n\t"
        : [a] "+r" (a)
        : [b] "r" (b)
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );

    return (sp_digit)a;
}

/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_2048_add_16(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "adds	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x7, x8, [%[b], 64]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 80]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "ldp	x3, x4, [%[a], 96]\n\t"
        "ldp	x7, x8, [%[b], 96]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 112]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 112]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 96]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 112]\n\t"
        "cset	%[r], cs\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return (sp_digit)r;
}

#ifndef WOLFSSL_SP_SMALL
/* Conditionally add a and b using the mask m.
 * m is -1 to add and 0 when not.
 *
 * r  A single precision number representing conditional add result.
 * a  A single precision number to add with.
 * b  A single precision number to add.
 * m  Mask value to apply.
 */
static sp_digit sp_2048_cond_add_8(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
    __asm__ __volatile__ (

        "ldp	x8, x9, [%[b], 0]\n\t"
        "ldp	x10, x11, [%[b], 16]\n\t"
        "ldp	x4, x5, [%[a], 0]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 16]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adds	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 0]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 16]\n\t"
        "ldp	x8, x9, [%[b], 32]\n\t"
        "ldp	x10, x11, [%[b], 48]\n\t"
        "ldp	x4, x5, [%[a], 32]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 48]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adcs	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 32]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 48]\n\t"
        "cset	%[r], cs\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    return (sp_digit)r;
}
#endif /* !WOLFSSL_SP_SMALL */

/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
SP_NOINLINE static void sp_2048_mul_16(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit* z0 = r;
    sp_digit z1[16];
    sp_digit a1[8];
    sp_digit b1[8];
    sp_digit* z2 = r + 16;
    sp_digit u;
    sp_digit ca;
    sp_digit cb;

    ca = sp_2048_add_8(a1, a, &a[8]);
    cb = sp_2048_add_8(b1, b, &b[8]);
    u  = ca & cb;

    sp_2048_mul_8(z2, &a[8], &b[8]);
    sp_2048_mul_8(z0, a, b);
    sp_2048_mul_8(z1, a1, b1);

    u += sp_2048_sub_in_place_16(z1, z0);
    u += sp_2048_sub_in_place_16(z1, z2);
    u += sp_2048_cond_add_8(z1 + 8, z1 + 8, a1, 0 - cb);
    u += sp_2048_cond_add_8(z1 + 8, z1 + 8, b1, 0 - ca);

    u += sp_2048_add_16(r + 8, r + 8, z1);
    (void)sp_2048_add_word_8(r + 24, r + 24, u);
}

/* Add digit to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_2048_add_word_16(sp_digit* r, const sp_digit* a,
        sp_digit b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "adds	x3, x3, %[b]\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "adcs	x3, x3, xzr\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "adcs	x3, x3, xzr\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "ldp	x3, x4, [%[a], 96]\n\t"
        "ldp	x5, x6, [%[a], 112]\n\t"
        "adcs	x3, x3, xzr\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 96]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 112]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6"
    );
}

/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer and result.
 * b  A single precision integer.
 */
static sp_digit sp_2048_sub_in_place_32(sp_digit* a, const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x2, x3, [%[a], 0]\n\t"
        "ldp	x6, x7, [%[b], 0]\n\t"
        "subs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 16]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 16]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 0]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 16]\n\t"
        "ldp	x2, x3, [%[a], 32]\n\t"
        "ldp	x6, x7, [%[b], 32]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 48]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 48]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 32]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 48]\n\t"
        "ldp	x2, x3, [%[a], 64]\n\t"
        "ldp	x6, x7, [%[b], 64]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 80]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 80]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 64]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 80]\n\t"
        "ldp	x2, x3, [%[a], 96]\n\t"
        "ldp	x6, x7, [%[b], 96]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 112]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 112]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 96]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 112]\n\t"
        "ldp	x2, x3, [%[a], 128]\n\t"
        "ldp	x6, x7, [%[b], 128]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 144]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 144]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 128]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 144]\n\t"
        "ldp	x2, x3, [%[a], 160]\n\t"
        "ldp	x6, x7, [%[b], 160]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 176]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 176]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 160]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 176]\n\t"
        "ldp	x2, x3, [%[a], 192]\n\t"
        "ldp	x6, x7, [%[b], 192]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 208]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 208]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 192]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 208]\n\t"
        "ldp	x2, x3, [%[a], 224]\n\t"
        "ldp	x6, x7, [%[b], 224]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 240]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 240]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 224]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 240]\n\t"
        "csetm	%[a], cc\n\t"
        : [a] "+r" (a)
        : [b] "r" (b)
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );

    return (sp_digit)a;
}

/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_2048_add_32(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "adds	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x7, x8, [%[b], 64]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 80]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "ldp	x3, x4, [%[a], 96]\n\t"
        "ldp	x7, x8, [%[b], 96]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 112]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 112]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 96]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 112]\n\t"
        "ldp	x3, x4, [%[a], 128]\n\t"
        "ldp	x7, x8, [%[b], 128]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 144]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 144]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 128]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 144]\n\t"
        "ldp	x3, x4, [%[a], 160]\n\t"
        "ldp	x7, x8, [%[b], 160]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 176]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 176]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 160]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 176]\n\t"
        "ldp	x3, x4, [%[a], 192]\n\t"
        "ldp	x7, x8, [%[b], 192]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 208]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 208]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 192]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 208]\n\t"
        "ldp	x3, x4, [%[a], 224]\n\t"
        "ldp	x7, x8, [%[b], 224]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 240]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 240]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 224]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 240]\n\t"
        "cset	%[r], cs\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return (sp_digit)r;
}

#ifndef WOLFSSL_SP_SMALL
/* Conditionally add a and b using the mask m.
 * m is -1 to add and 0 when not.
 *
 * r  A single precision number representing conditional add result.
 * a  A single precision number to add with.
 * b  A single precision number to add.
 * m  Mask value to apply.
 */
static sp_digit sp_2048_cond_add_16(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
    __asm__ __volatile__ (

        "ldp	x8, x9, [%[b], 0]\n\t"
        "ldp	x10, x11, [%[b], 16]\n\t"
        "ldp	x4, x5, [%[a], 0]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 16]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adds	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 0]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 16]\n\t"
        "ldp	x8, x9, [%[b], 32]\n\t"
        "ldp	x10, x11, [%[b], 48]\n\t"
        "ldp	x4, x5, [%[a], 32]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 48]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adcs	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 32]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 48]\n\t"
        "ldp	x8, x9, [%[b], 64]\n\t"
        "ldp	x10, x11, [%[b], 80]\n\t"
        "ldp	x4, x5, [%[a], 64]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 80]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adcs	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 64]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 80]\n\t"
        "ldp	x8, x9, [%[b], 96]\n\t"
        "ldp	x10, x11, [%[b], 112]\n\t"
        "ldp	x4, x5, [%[a], 96]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 112]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adcs	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 96]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 112]\n\t"
        "cset	%[r], cs\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    return (sp_digit)r;
}
#endif /* !WOLFSSL_SP_SMALL */

/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
SP_NOINLINE static void sp_2048_mul_32(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit* z0 = r;
    sp_digit z1[32];
    sp_digit a1[16];
    sp_digit b1[16];
    sp_digit* z2 = r + 32;
    sp_digit u;
    sp_digit ca;
    sp_digit cb;

    ca = sp_2048_add_16(a1, a, &a[16]);
    cb = sp_2048_add_16(b1, b, &b[16]);
    u  = ca & cb;

    sp_2048_mul_16(z2, &a[16], &b[16]);
    sp_2048_mul_16(z0, a, b);
    sp_2048_mul_16(z1, a1, b1);

    u += sp_2048_sub_in_place_32(z1, z0);
    u += sp_2048_sub_in_place_32(z1, z2);
    u += sp_2048_cond_add_16(z1 + 16, z1 + 16, a1, 0 - cb);
    u += sp_2048_cond_add_16(z1 + 16, z1 + 16, b1, 0 - ca);

    u += sp_2048_add_32(r + 16, r + 16, z1);
    (void)sp_2048_add_word_16(r + 48, r + 48, u);
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
static void sp_2048_sqr_16(sp_digit* r, const sp_digit* a)
{
    __asm__ __volatile__ (
        "ldp	x10, x11, [%[a], 0]\n\t"
        "ldp	x12, x13, [%[a], 16]\n\t"
        "ldp	x14, x15, [%[a], 32]\n\t"
        "ldp	x16, x17, [%[a], 48]\n\t"
        "ldp	x19, x20, [%[a], 64]\n\t"
        "ldp	x21, x22, [%[a], 80]\n\t"
        "ldp	x23, x24, [%[a], 96]\n\t"
        "ldp	x25, x26, [%[a], 112]\n\t"
        "#  A[0] * A[0]\n\t"
        "mul	x2, x10, x10\n\t"
        "umulh	x3, x10, x10\n\t"
        "str	x2, [%[r]]\n\t"
        "mov	x4, xzr\n\t"
        "#  A[0] * A[1]\n\t"
        "mul	x8, x10, x11\n\t"
        "umulh	x9, x10, x11\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, xzr, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "str	x3, [%[r], 8]\n\t"
        "#  A[0] * A[2]\n\t"
        "mul	x8, x10, x12\n\t"
        "adcs	x4, x4, x9\n\t"
        "umulh	x9, x10, x12\n\t"
        "adc	x2, x2, xzr\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x4, x4, x8\n\t"
        "#  A[1] * A[1]\n\t"
        "mul	x8, x11, x11\n\t"
        "adcs	x2, x2, x9\n\t"
        "umulh	x9, x11, x11\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x8\n\t"
        "str	x4, [%[r], 16]\n\t"
        "#  A[0] * A[3]\n\t"
        "mul	x8, x10, x13\n\t"
        "adcs	x2, x2, x9\n\t"
        "umulh	x9, x10, x13\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "#  A[1] * A[2]\n\t"
        "mul	x8, x11, x12\n\t"
        "adcs	x3, x3, x9\n\t"
        "umulh	x9, x11, x12\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "str	x2, [%[r], 24]\n\t"
        "#  A[0] * A[4]\n\t"
        "mul	x8, x10, x14\n\t"
        "adcs	x3, x3, x9\n\t"
        "umulh	x9, x10, x14\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, xzr, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "#  A[1] * A[3]\n\t"
        "mul	x8, x11, x13\n\t"
        "adcs	x4, x4, x9\n\t"
        "umulh	x9, x11, x13\n\t"
        "adc	x2, x2, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "#  A[2] * A[2]\n\t"
        "mul	x8, x12, x12\n\t"
        "adcs	x4, x4, x9\n\t"
        "umulh	x9, x12, x12\n\t"
        "adc	x2, x2, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "str	x3, [%[r], 32]\n\t"
        "#  A[0] * A[5]\n\t"
        "mul	x5, x10, x15\n\t"
        "adcs	x4, x4, x9\n\t"
        "umulh	x6, x10, x15\n\t"
        "adc	x2, x2, xzr\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[4]\n\t"
        "mul	x8, x11, x14\n\t"
        "umulh	x9, x11, x14\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[2] * A[3]\n\t"
        "mul	x8, x12, x13\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x12, x13\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 40]\n\t"
        "#  A[0] * A[6]\n\t"
        "mul	x5, x10, x16\n\t"
        "umulh	x6, x10, x16\n\t"
        "mov	x4, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[5]\n\t"
        "mul	x8, x11, x15\n\t"
        "umulh	x9, x11, x15\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[2] * A[4]\n\t"
        "mul	x8, x12, x14\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x12, x14\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[3] * A[3]\n\t"
        "mul	x8, x13, x13\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x13, x13\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 48]\n\t"
        "#  A[0] * A[7]\n\t"
        "mul	x5, x10, x17\n\t"
        "umulh	x6, x10, x17\n\t"
        "mov	x2, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[6]\n\t"
        "mul	x8, x11, x16\n\t"
        "umulh	x9, x11, x16\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[2] * A[5]\n\t"
        "mul	x8, x12, x15\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x12, x15\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[3] * A[4]\n\t"
        "mul	x8, x13, x14\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x13, x14\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 56]\n\t"
        "#  A[0] * A[8]\n\t"
        "mul	x5, x10, x19\n\t"
        "umulh	x6, x10, x19\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[7]\n\t"
        "mul	x8, x11, x17\n\t"
        "umulh	x9, x11, x17\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[2] * A[6]\n\t"
        "mul	x8, x12, x16\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x12, x16\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[3] * A[5]\n\t"
        "mul	x8, x13, x15\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x13, x15\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[4] * A[4]\n\t"
        "mul	x8, x14, x14\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x14, x14\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 64]\n\t"
        "#  A[0] * A[9]\n\t"
        "mul	x5, x10, x20\n\t"
        "umulh	x6, x10, x20\n\t"
        "mov	x4, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[8]\n\t"
        "mul	x8, x11, x19\n\t"
        "umulh	x9, x11, x19\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[2] * A[7]\n\t"
        "mul	x8, x12, x17\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x12, x17\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[3] * A[6]\n\t"
        "mul	x8, x13, x16\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x13, x16\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[4] * A[5]\n\t"
        "mul	x8, x14, x15\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x14, x15\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 72]\n\t"
        "#  A[0] * A[10]\n\t"
        "mul	x5, x10, x21\n\t"
        "umulh	x6, x10, x21\n\t"
        "mov	x2, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[9]\n\t"
        "mul	x8, x11, x20\n\t"
        "umulh	x9, x11, x20\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[2] * A[8]\n\t"
        "mul	x8, x12, x19\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x12, x19\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[3] * A[7]\n\t"
        "mul	x8, x13, x17\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x13, x17\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[4] * A[6]\n\t"
        "mul	x8, x14, x16\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x14, x16\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[5] * A[5]\n\t"
        "mul	x8, x15, x15\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x15, x15\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 80]\n\t"
        "#  A[0] * A[11]\n\t"
        "mul	x5, x10, x22\n\t"
        "umulh	x6, x10, x22\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[10]\n\t"
        "mul	x8, x11, x21\n\t"
        "umulh	x9, x11, x21\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[2] * A[9]\n\t"
        "mul	x8, x12, x20\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x12, x20\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[3] * A[8]\n\t"
        "mul	x8, x13, x19\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x13, x19\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[4] * A[7]\n\t"
        "mul	x8, x14, x17\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x14, x17\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[5] * A[6]\n\t"
        "mul	x8, x15, x16\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x15, x16\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 88]\n\t"
        "#  A[0] * A[12]\n\t"
        "mul	x5, x10, x23\n\t"
        "umulh	x6, x10, x23\n\t"
        "mov	x4, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[11]\n\t"
        "mul	x8, x11, x22\n\t"
        "umulh	x9, x11, x22\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[2] * A[10]\n\t"
        "mul	x8, x12, x21\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x12, x21\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[3] * A[9]\n\t"
        "mul	x8, x13, x20\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x13, x20\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[4] * A[8]\n\t"
        "mul	x8, x14, x19\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x14, x19\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[5] * A[7]\n\t"
        "mul	x8, x15, x17\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x15, x17\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[6] * A[6]\n\t"
        "mul	x8, x16, x16\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x16, x16\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 96]\n\t"
        "#  A[0] * A[13]\n\t"
        "mul	x5, x10, x24\n\t"
        "umulh	x6, x10, x24\n\t"
        "mov	x2, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[12]\n\t"
        "mul	x8, x11, x23\n\t"
        "umulh	x9, x11, x23\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[2] * A[11]\n\t"
        "mul	x8, x12, x22\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x12, x22\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[3] * A[10]\n\t"
        "mul	x8, x13, x21\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x13, x21\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[4] * A[9]\n\t"
        "mul	x8, x14, x20\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x14, x20\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[5] * A[8]\n\t"
        "mul	x8, x15, x19\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x15, x19\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[6] * A[7]\n\t"
        "mul	x8, x16, x17\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x16, x17\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 104]\n\t"
        "#  A[0] * A[14]\n\t"
        "mul	x5, x10, x25\n\t"
        "umulh	x6, x10, x25\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[13]\n\t"
        "mul	x8, x11, x24\n\t"
        "umulh	x9, x11, x24\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[2] * A[12]\n\t"
        "mul	x8, x12, x23\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x12, x23\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[3] * A[11]\n\t"
        "mul	x8, x13, x22\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x13, x22\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[4] * A[10]\n\t"
        "mul	x8, x14, x21\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x14, x21\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[5] * A[9]\n\t"
        "mul	x8, x15, x20\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x15, x20\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[6] * A[8]\n\t"
        "mul	x8, x16, x19\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x16, x19\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[7] * A[7]\n\t"
        "mul	x8, x17, x17\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x17, x17\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 112]\n\t"
        "#  A[0] * A[15]\n\t"
        "mul	x5, x10, x26\n\t"
        "umulh	x6, x10, x26\n\t"
        "mov	x4, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[14]\n\t"
        "mul	x8, x11, x25\n\t"
        "umulh	x9, x11, x25\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[2] * A[13]\n\t"
        "mul	x8, x12, x24\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x12, x24\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[3] * A[12]\n\t"
        "mul	x8, x13, x23\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x13, x23\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[4] * A[11]\n\t"
        "mul	x8, x14, x22\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x14, x22\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[5] * A[10]\n\t"
        "mul	x8, x15, x21\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x15, x21\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[6] * A[9]\n\t"
        "mul	x8, x16, x20\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x16, x20\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[7] * A[8]\n\t"
        "mul	x8, x17, x19\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x17, x19\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 120]\n\t"
        "#  A[1] * A[15]\n\t"
        "mul	x5, x11, x26\n\t"
        "umulh	x6, x11, x26\n\t"
        "mov	x2, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[2] * A[14]\n\t"
        "mul	x8, x12, x25\n\t"
        "umulh	x9, x12, x25\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[3] * A[13]\n\t"
        "mul	x8, x13, x24\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x13, x24\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[4] * A[12]\n\t"
        "mul	x8, x14, x23\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x14, x23\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[5] * A[11]\n\t"
        "mul	x8, x15, x22\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x15, x22\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[6] * A[10]\n\t"
        "mul	x8, x16, x21\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x16, x21\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[7] * A[9]\n\t"
        "mul	x8, x17, x20\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x17, x20\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[8] * A[8]\n\t"
        "mul	x8, x19, x19\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x19, x19\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 128]\n\t"
        "#  A[2] * A[15]\n\t"
        "mul	x5, x12, x26\n\t"
        "umulh	x6, x12, x26\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[3] * A[14]\n\t"
        "mul	x8, x13, x25\n\t"
        "umulh	x9, x13, x25\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[4] * A[13]\n\t"
        "mul	x8, x14, x24\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x14, x24\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[5] * A[12]\n\t"
        "mul	x8, x15, x23\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x15, x23\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[6] * A[11]\n\t"
        "mul	x8, x16, x22\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x16, x22\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[7] * A[10]\n\t"
        "mul	x8, x17, x21\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x17, x21\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[8] * A[9]\n\t"
        "mul	x8, x19, x20\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x19, x20\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 136]\n\t"
        "#  A[3] * A[15]\n\t"
        "mul	x5, x13, x26\n\t"
        "umulh	x6, x13, x26\n\t"
        "mov	x4, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[4] * A[14]\n\t"
        "mul	x8, x14, x25\n\t"
        "umulh	x9, x14, x25\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[5] * A[13]\n\t"
        "mul	x8, x15, x24\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x15, x24\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[6] * A[12]\n\t"
        "mul	x8, x16, x23\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x16, x23\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[7] * A[11]\n\t"
        "mul	x8, x17, x22\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x17, x22\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[8] * A[10]\n\t"
        "mul	x8, x19, x21\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x19, x21\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[9] * A[9]\n\t"
        "mul	x8, x20, x20\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x20, x20\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 144]\n\t"
        "#  A[4] * A[15]\n\t"
        "mul	x5, x14, x26\n\t"
        "umulh	x6, x14, x26\n\t"
        "mov	x2, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[5] * A[14]\n\t"
        "mul	x8, x15, x25\n\t"
        "umulh	x9, x15, x25\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[6] * A[13]\n\t"
        "mul	x8, x16, x24\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x16, x24\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[7] * A[12]\n\t"
        "mul	x8, x17, x23\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x17, x23\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[8] * A[11]\n\t"
        "mul	x8, x19, x22\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x19, x22\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[9] * A[10]\n\t"
        "mul	x8, x20, x21\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x20, x21\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 152]\n\t"
        "#  A[5] * A[15]\n\t"
        "mul	x5, x15, x26\n\t"
        "umulh	x6, x15, x26\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[6] * A[14]\n\t"
        "mul	x8, x16, x25\n\t"
        "umulh	x9, x16, x25\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[7] * A[13]\n\t"
        "mul	x8, x17, x24\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x17, x24\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[8] * A[12]\n\t"
        "mul	x8, x19, x23\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x19, x23\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[9] * A[11]\n\t"
        "mul	x8, x20, x22\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x20, x22\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[10] * A[10]\n\t"
        "mul	x8, x21, x21\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x21, x21\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 160]\n\t"
        "#  A[6] * A[15]\n\t"
        "mul	x5, x16, x26\n\t"
        "umulh	x6, x16, x26\n\t"
        "mov	x4, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[7] * A[14]\n\t"
        "mul	x8, x17, x25\n\t"
        "umulh	x9, x17, x25\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[8] * A[13]\n\t"
        "mul	x8, x19, x24\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x19, x24\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[9] * A[12]\n\t"
        "mul	x8, x20, x23\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x20, x23\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[10] * A[11]\n\t"
        "mul	x8, x21, x22\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x21, x22\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 168]\n\t"
        "#  A[7] * A[15]\n\t"
        "mul	x5, x17, x26\n\t"
        "umulh	x6, x17, x26\n\t"
        "mov	x2, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[8] * A[14]\n\t"
        "mul	x8, x19, x25\n\t"
        "umulh	x9, x19, x25\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[9] * A[13]\n\t"
        "mul	x8, x20, x24\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x20, x24\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[10] * A[12]\n\t"
        "mul	x8, x21, x23\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x21, x23\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[11] * A[11]\n\t"
        "mul	x8, x22, x22\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x22, x22\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 176]\n\t"
        "#  A[8] * A[15]\n\t"
        "mul	x5, x19, x26\n\t"
        "umulh	x6, x19, x26\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[9] * A[14]\n\t"
        "mul	x8, x20, x25\n\t"
        "umulh	x9, x20, x25\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[10] * A[13]\n\t"
        "mul	x8, x21, x24\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x21, x24\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[11] * A[12]\n\t"
        "mul	x8, x22, x23\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x22, x23\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 184]\n\t"
        "#  A[9] * A[15]\n\t"
        "mul	x5, x20, x26\n\t"
        "umulh	x6, x20, x26\n\t"
        "mov	x4, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[10] * A[14]\n\t"
        "mul	x8, x21, x25\n\t"
        "umulh	x9, x21, x25\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[11] * A[13]\n\t"
        "mul	x8, x22, x24\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x22, x24\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[12] * A[12]\n\t"
        "mul	x8, x23, x23\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x23, x23\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 192]\n\t"
        "#  A[10] * A[15]\n\t"
        "mul	x5, x21, x26\n\t"
        "umulh	x6, x21, x26\n\t"
        "mov	x2, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[11] * A[14]\n\t"
        "mul	x8, x22, x25\n\t"
        "umulh	x9, x22, x25\n\t"
        "adds	x5, x5, x8\n\t"
        "#  A[12] * A[13]\n\t"
        "mul	x8, x23, x24\n\t"
        "adcs	x6, x6, x9\n\t"
        "umulh	x9, x23, x24\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 200]\n\t"
        "#  A[11] * A[15]\n\t"
        "mul	x8, x22, x26\n\t"
        "umulh	x9, x22, x26\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x4, x4, x8\n\t"
        "#  A[12] * A[14]\n\t"
        "mul	x8, x23, x25\n\t"
        "adcs	x2, x2, x9\n\t"
        "umulh	x9, x23, x25\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x8\n\t"
        "#  A[13] * A[13]\n\t"
        "mul	x8, x24, x24\n\t"
        "adcs	x2, x2, x9\n\t"
        "umulh	x9, x24, x24\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x8\n\t"
        "str	x4, [%[r], 208]\n\t"
        "#  A[12] * A[15]\n\t"
        "mul	x8, x23, x26\n\t"
        "adcs	x2, x2, x9\n\t"
        "umulh	x9, x23, x26\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "#  A[13] * A[14]\n\t"
        "mul	x8, x24, x25\n\t"
        "adcs	x3, x3, x9\n\t"
        "umulh	x9, x24, x25\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "str	x2, [%[r], 216]\n\t"
        "#  A[13] * A[15]\n\t"
        "mul	x8, x24, x26\n\t"
        "adcs	x3, x3, x9\n\t"
        "umulh	x9, x24, x26\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, xzr, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "#  A[14] * A[14]\n\t"
        "mul	x8, x25, x25\n\t"
        "adcs	x4, x4, x9\n\t"
        "umulh	x9, x25, x25\n\t"
        "adc	x2, x2, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "str	x3, [%[r], 224]\n\t"
        "#  A[14] * A[15]\n\t"
        "mul	x8, x25, x26\n\t"
        "adcs	x4, x4, x9\n\t"
        "umulh	x9, x25, x26\n\t"
        "adc	x2, x2, xzr\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x4, x4, x8\n\t"
        "str	x4, [%[r], 232]\n\t"
        "#  A[15] * A[15]\n\t"
        "mul	x8, x26, x26\n\t"
        "adcs	x2, x2, x9\n\t"
        "umulh	x9, x26, x26\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adc	x3, x3, x9\n\t"
        "stp	x2, x3, [%[r], 240]\n\t"
        :
        : [r] "r" (r), [a] "r" (a)
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x19", "x20", "x21", "x22", "x23", "x24", "x25", "x26"
    );
}

/* Sub b from a into r. (r = a - b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_2048_sub_16(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "subs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x7, x8, [%[b], 64]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 80]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "ldp	x3, x4, [%[a], 96]\n\t"
        "ldp	x7, x8, [%[b], 96]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 112]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 112]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 96]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 112]\n\t"
        "csetm	%[r], cc\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return (sp_digit)r;
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
SP_NOINLINE static void sp_2048_sqr_32(sp_digit* r, const sp_digit* a)
{
    sp_digit* z0 = r;
    sp_digit* z2 = r + 32;
    sp_digit z1[32];
    sp_digit* a1 = z1;
    sp_digit* zero = z1 + 16;
    sp_digit u;
    sp_digit mask;
    sp_digit* p1;
    sp_digit* p2;

    XMEMSET(zero, 0, sizeof(sp_digit) * 16);

    mask = sp_2048_sub_16(a1, a, &a[16]);
    p1 = (sp_digit*)(((sp_digit)zero &   mask ) | ((sp_digit)a1 & (~mask)));
    p2 = (sp_digit*)(((sp_digit)zero & (~mask)) | ((sp_digit)a1 &   mask ));
    (void)sp_2048_sub_16(a1, p1, p2);

    sp_2048_sqr_16(z2, &a[16]);
    sp_2048_sqr_16(z0, a);
    sp_2048_sqr_16(z1, a1);

    u = 0;
    u -= sp_2048_sub_in_place_32(z1, z2);
    u -= sp_2048_sub_in_place_32(z1, z0);
    u += sp_2048_sub_in_place_32(r + 16, z1);
    sp_2048_add_word_16(r + 48, r + 48, u);
}

#endif /* !WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_2048_add_32(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x11, %[a], 256\n\t"
        "\n1:\n\t"
        "adds	%[c], %[c], #-1\n\t"
        "ldp	x3, x4, [%[a]], #16\n\t"
        "ldp	x5, x6, [%[a]], #16\n\t"
        "ldp	x7, x8, [%[b]], #16\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x9, x10, [%[b]], #16\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r]], #16\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r]], #16\n\t"
        "cset	%[c], cs\n\t"
        "cmp	%[a], x11\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [r] "+r" (r), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_2048_sub_in_place_32(sp_digit* a, const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x10, %[a], 256\n\t"
        "\n1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldp	x2, x3, [%[a]]\n\t"
        "ldp	x4, x5, [%[a], #16]\n\t"
        "ldp	x6, x7, [%[b]], #16\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x8, x9, [%[b]], #16\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a]], #16\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a]], #16\n\t"
        "csetm	%[c], cc\n\t"
        "cmp	%[a], x10\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_2048_mul_32(sp_digit* r, const sp_digit* a, const sp_digit* b)
{
    sp_digit tmp[64];

    __asm__ __volatile__ (
        "mov	x5, xzr\n\t"
        "mov	x6, xzr\n\t"
        "mov	x7, xzr\n\t"
        "mov	x8, xzr\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 248\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[b], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 256\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 496\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
static void sp_2048_sqr_32(sp_digit* r, const sp_digit* a)
{
    sp_digit tmp[64];

    __asm__ __volatile__ (
        "mov	x6, xzr\n\t"
        "mov	x7, xzr\n\t"
        "mov	x8, xzr\n\t"
        "mov	x5, xzr\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 248\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "cmp	x4, x3\n\t"
        "b.eq	4f\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[a], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "b.al	5f\n\t"
        "\n4:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "mul	x9, x10, x10\n\t"
        "umulh	x10, x10, x10\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "\n5:\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 256\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x4\n\t"
        "b.gt	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 496\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

#endif /* WOLFSSL_SP_SMALL */
#if (defined(WOLFSSL_HAVE_SP_RSA) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)) || defined(WOLFSSL_HAVE_SP_DH)
#ifdef WOLFSSL_SP_SMALL
/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_2048_add_16(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x11, %[a], 128\n\t"
        "\n1:\n\t"
        "adds	%[c], %[c], #-1\n\t"
        "ldp	x3, x4, [%[a]], #16\n\t"
        "ldp	x5, x6, [%[a]], #16\n\t"
        "ldp	x7, x8, [%[b]], #16\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x9, x10, [%[b]], #16\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r]], #16\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r]], #16\n\t"
        "cset	%[c], cs\n\t"
        "cmp	%[a], x11\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [r] "+r" (r), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_2048_sub_in_place_16(sp_digit* a, const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x10, %[a], 128\n\t"
        "\n1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldp	x2, x3, [%[a]]\n\t"
        "ldp	x4, x5, [%[a], #16]\n\t"
        "ldp	x6, x7, [%[b]], #16\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x8, x9, [%[b]], #16\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a]], #16\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a]], #16\n\t"
        "csetm	%[c], cc\n\t"
        "cmp	%[a], x10\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_2048_mul_16(sp_digit* r, const sp_digit* a, const sp_digit* b)
{
    sp_digit tmp[32];

    __asm__ __volatile__ (
        "mov	x5, xzr\n\t"
        "mov	x6, xzr\n\t"
        "mov	x7, xzr\n\t"
        "mov	x8, xzr\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 120\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[b], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 128\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 240\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
static void sp_2048_sqr_16(sp_digit* r, const sp_digit* a)
{
    sp_digit tmp[32];

    __asm__ __volatile__ (
        "mov	x6, xzr\n\t"
        "mov	x7, xzr\n\t"
        "mov	x8, xzr\n\t"
        "mov	x5, xzr\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 120\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "cmp	x4, x3\n\t"
        "b.eq	4f\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[a], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "b.al	5f\n\t"
        "\n4:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "mul	x9, x10, x10\n\t"
        "umulh	x10, x10, x10\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "\n5:\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 128\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x4\n\t"
        "b.gt	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 240\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

#endif /* WOLFSSL_SP_SMALL */
#endif /* (WOLFSSL_HAVE_SP_RSA & !WOLFSSL_RSA_PUBLIC_ONLY) | WOLFSSL_HAVE_SP_DH */

/* Caclulate the bottom digit of -1/a mod 2^n.
 *
 * a    A single precision number.
 * rho  Bottom word of inverse.
 */
static void sp_2048_mont_setup(const sp_digit* a, sp_digit* rho)
{
    sp_digit x;
    sp_digit b;

    b = a[0];
    x = (((b + 2) & 4) << 1) + b; /* here x*a==1 mod 2**4 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**8 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**16 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**32 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**64 */

    /* rho = -1/m mod b */
    *rho = (sp_digit)0 - x;
}

/* Mul a by digit b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision digit.
 */
static void sp_2048_mul_d_32(sp_digit* r, const sp_digit* a,
        sp_digit b)
{
#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldr	x8, [%[a]]\n\t"
        "mul	x5, %[b], x8\n\t"
        "umulh	x3, %[b], x8\n\t"
        "mov	x4, xzr\n\t"
        "str	x5, [%[r]]\n\t"
        "mov	x5, xzr\n\t"
        "mov	x9, #8\n\t"
        "1:\n\t"
        "ldr	x8, [%[a], x9]\n\t"
        "mul	x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "str	x3, [%[r], x9]\n\t"
        "mov	x3, x4\n\t"
        "mov	x4, x5\n\t"
        "mov	x5, #0\n\t"
        "add	x9, x9, #8\n\t"
        "cmp	x9, 256\n\t"
        "b.lt	1b\n\t"
        "str	x3, [%[r], 256]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );
#else
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldp	x9, x10, [%[a]]\n\t"
        "mul	x3, %[b], x9\n\t"
        "umulh	x4, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "# A[1] * B\n\t"
        "str	x3, [%[r]]\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[2] * B\n\t"
        "ldp	x9, x10, [%[a], 16]\n\t"
        "str	x4, [%[r], 8]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[3] * B\n\t"
        "str	x5, [%[r], 16]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[4] * B\n\t"
        "ldp	x9, x10, [%[a], 32]\n\t"
        "str	x3, [%[r], 24]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[5] * B\n\t"
        "str	x4, [%[r], 32]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[6] * B\n\t"
        "ldp	x9, x10, [%[a], 48]\n\t"
        "str	x5, [%[r], 40]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[7] * B\n\t"
        "str	x3, [%[r], 48]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[8] * B\n\t"
        "ldp	x9, x10, [%[a], 64]\n\t"
        "str	x4, [%[r], 56]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[9] * B\n\t"
        "str	x5, [%[r], 64]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[10] * B\n\t"
        "ldp	x9, x10, [%[a], 80]\n\t"
        "str	x3, [%[r], 72]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[11] * B\n\t"
        "str	x4, [%[r], 80]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[12] * B\n\t"
        "ldp	x9, x10, [%[a], 96]\n\t"
        "str	x5, [%[r], 88]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[13] * B\n\t"
        "str	x3, [%[r], 96]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[14] * B\n\t"
        "ldp	x9, x10, [%[a], 112]\n\t"
        "str	x4, [%[r], 104]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[15] * B\n\t"
        "str	x5, [%[r], 112]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[16] * B\n\t"
        "ldp	x9, x10, [%[a], 128]\n\t"
        "str	x3, [%[r], 120]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[17] * B\n\t"
        "str	x4, [%[r], 128]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[18] * B\n\t"
        "ldp	x9, x10, [%[a], 144]\n\t"
        "str	x5, [%[r], 136]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[19] * B\n\t"
        "str	x3, [%[r], 144]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[20] * B\n\t"
        "ldp	x9, x10, [%[a], 160]\n\t"
        "str	x4, [%[r], 152]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[21] * B\n\t"
        "str	x5, [%[r], 160]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[22] * B\n\t"
        "ldp	x9, x10, [%[a], 176]\n\t"
        "str	x3, [%[r], 168]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[23] * B\n\t"
        "str	x4, [%[r], 176]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[24] * B\n\t"
        "ldp	x9, x10, [%[a], 192]\n\t"
        "str	x5, [%[r], 184]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[25] * B\n\t"
        "str	x3, [%[r], 192]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[26] * B\n\t"
        "ldp	x9, x10, [%[a], 208]\n\t"
        "str	x4, [%[r], 200]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[27] * B\n\t"
        "str	x5, [%[r], 208]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[28] * B\n\t"
        "ldp	x9, x10, [%[a], 224]\n\t"
        "str	x3, [%[r], 216]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[29] * B\n\t"
        "str	x4, [%[r], 224]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[30] * B\n\t"
        "ldp	x9, x10, [%[a], 240]\n\t"
        "str	x5, [%[r], 232]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[31] * B\n\t"
        "str	x3, [%[r], 240]\n\t"
        "mul	x6, %[b], x10\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "adc	x5, x5, x7\n\t"
        "stp	x4, x5, [%[r], 248]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );
#endif
}

#if (defined(WOLFSSL_HAVE_SP_RSA) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)) || defined(WOLFSSL_HAVE_SP_DH)
/* r = 2^n mod m where n is the number of bits to reduce by.
 * Given m must be 2048 bits, just need to subtract.
 *
 * r  A single precision number.
 * m  A single precision number.
 */
static void sp_2048_mont_norm_16(sp_digit* r, const sp_digit* m)
{
    XMEMSET(r, 0, sizeof(sp_digit) * 16);

    /* r = 2^n mod m */
    sp_2048_sub_in_place_16(r, m);
}

/* Reduce the number back to 2048 bits using Montgomery reduction.
 *
 * a   A single precision number to reduce in place.
 * m   The single precision number representing the modulus.
 * mp  The digit representing the negative inverse of m mod 2^n.
 */
SP_NOINLINE static void sp_2048_mont_reduce_16(sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    __asm__ __volatile__ (
        "ldp	x12, x13, [%[a], 0]\n\t"
        "ldp	x14, x15, [%[a], 16]\n\t"
        "ldp	x16, x17, [%[a], 32]\n\t"
        "ldp	x19, x20, [%[a], 48]\n\t"
        "ldp	x21, x22, [%[a], 64]\n\t"
        "ldp	x23, x24, [%[a], 80]\n\t"
        "ldp	x25, x26, [%[a], 96]\n\t"
        "ldp	x27, x28, [%[a], 112]\n\t"
        "mov	x3, xzr\n\t"
        "# i = 0..15\n\t"
        "mov	x4, 16\n\t"
        "\n1:\n\t"
        "# mu = a[i] * mp\n\t"
        "mul	x9, %[mp], x12\n\t"
        "# a[i+0] += m[0] * mu\n\t"
        "ldp	x10, x11, [%[m], 0]\n\t"
        "mul	x7, x10, x9\n\t"
        "umulh	x8, x10, x9\n\t"
        "adds	x12, x12, x7\n\t"
        "# a[i+1] += m[1] * mu\n\t"
        "adc	x6, x8, xzr\n\t"
        "mul	x7, x11, x9\n\t"
        "umulh	x8, x11, x9\n\t"
        "adds	x12, x13, x7\n\t"
        "# a[i+2] += m[2] * mu\n\t"
        "ldp	x11, x10, [%[m], 16]\n\t"
        "adc	x5, x8, xzr\n\t"
        "adds	x12, x12, x6\n\t"
        "mul	x7, x11, x9\n\t"
        "adc	x5, x5, xzr\n\t"
        "umulh	x8, x11, x9\n\t"
        "adds	x13, x14, x7\n\t"
        "# a[i+3] += m[3] * mu\n\t"
        "adc	x6, x8, xzr\n\t"
        "adds	x13, x13, x5\n\t"
        "mul	x7, x10, x9\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x8, x10, x9\n\t"
        "adds	x14, x15, x7\n\t"
        "# a[i+4] += m[4] * mu\n\t"
        "ldp	x11, x10, [%[m], 32]\n\t"
        "adc	x5, x8, xzr\n\t"
        "adds	x14, x14, x6\n\t"
        "mul	x7, x11, x9\n\t"
        "adc	x5, x5, xzr\n\t"
        "umulh	x8, x11, x9\n\t"
        "adds	x15, x16, x7\n\t"
        "# a[i+5] += m[5] * mu\n\t"
        "adc	x6, x8, xzr\n\t"
        "adds	x15, x15, x5\n\t"
        "mul	x7, x10, x9\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x8, x10, x9\n\t"
        "adds	x16, x17, x7\n\t"
        "# a[i+6] += m[6] * mu\n\t"
        "ldp	x11, x10, [%[m], 48]\n\t"
        "adc	x5, x8, xzr\n\t"
        "adds	x16, x16, x6\n\t"
        "mul	x7, x11, x9\n\t"
        "adc	x5, x5, xzr\n\t"
        "umulh	x8, x11, x9\n\t"
        "adds	x17, x19, x7\n\t"
        "# a[i+7] += m[7] * mu\n\t"
        "adc	x6, x8, xzr\n\t"
        "adds	x17, x17, x5\n\t"
        "mul	x7, x10, x9\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x8, x10, x9\n\t"
        "adds	x19, x20, x7\n\t"
        "# a[i+8] += m[8] * mu\n\t"
        "ldp	x11, x10, [%[m], 64]\n\t"
        "adc	x5, x8, xzr\n\t"
        "adds	x19, x19, x6\n\t"
        "mul	x7, x11, x9\n\t"
        "adc	x5, x5, xzr\n\t"
        "umulh	x8, x11, x9\n\t"
        "adds	x20, x21, x7\n\t"
        "# a[i+9] += m[9] * mu\n\t"
        "adc	x6, x8, xzr\n\t"
        "adds	x20, x20, x5\n\t"
        "mul	x7, x10, x9\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x8, x10, x9\n\t"
        "adds	x21, x22, x7\n\t"
        "# a[i+10] += m[10] * mu\n\t"
        "ldp	x11, x10, [%[m], 80]\n\t"
        "adc	x5, x8, xzr\n\t"
        "adds	x21, x21, x6\n\t"
        "mul	x7, x11, x9\n\t"
        "adc	x5, x5, xzr\n\t"
        "umulh	x8, x11, x9\n\t"
        "adds	x22, x23, x7\n\t"
        "# a[i+11] += m[11] * mu\n\t"
        "adc	x6, x8, xzr\n\t"
        "adds	x22, x22, x5\n\t"
        "mul	x7, x10, x9\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x8, x10, x9\n\t"
        "adds	x23, x24, x7\n\t"
        "# a[i+12] += m[12] * mu\n\t"
        "ldp	x11, x10, [%[m], 96]\n\t"
        "adc	x5, x8, xzr\n\t"
        "adds	x23, x23, x6\n\t"
        "mul	x7, x11, x9\n\t"
        "adc	x5, x5, xzr\n\t"
        "umulh	x8, x11, x9\n\t"
        "adds	x24, x25, x7\n\t"
        "# a[i+13] += m[13] * mu\n\t"
        "adc	x6, x8, xzr\n\t"
        "adds	x24, x24, x5\n\t"
        "mul	x7, x10, x9\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x8, x10, x9\n\t"
        "adds	x25, x26, x7\n\t"
        "# a[i+14] += m[14] * mu\n\t"
        "ldp	x11, x10, [%[m], 112]\n\t"
        "adc	x5, x8, xzr\n\t"
        "adds	x25, x25, x6\n\t"
        "mul	x7, x11, x9\n\t"
        "adc	x5, x5, xzr\n\t"
        "umulh	x8, x11, x9\n\t"
        "adds	x26, x27, x7\n\t"
        "# a[i+15] += m[15] * mu\n\t"
        "ldr	x10, [%[m], 120]\n\t"
        "adc	x6, x8, xzr\n\t"
        "adds	x26, x26, x5\n\t"
        "mul	x7, x10, x9\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x8, x10, x9\n\t"
        "adds	x6, x6, x7\n\t"
        "adcs	x8, x8, x3\n\t"
        "cset	x3, cs\n\t"
        "adds	x27, x28, x6\n\t"
        "ldr	x28, [%[a], 128]\n\t"
        "adcs	x28, x28, x8\n\t"
        "adc	x3, x3, xzr\n\t"
        "subs	x4, x4, 1\n\t"
        "add	%[a], %[a], 8\n\t"
        "bne	1b\n\t"
        "# Create mask\n\t"
        "neg	x3, x3\n\t"
        "mov	x9, %[a]\n\t"
        "sub	%[a], %[a], 128\n\t"
        "# Subtract masked modulus\n\t"
        "ldp	x4, x5, [%[m], 0]\n\t"
        "ldp	x6, x7, [%[m], 16]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "subs	x12, x12, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x13, x13, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x14, x14, x6\n\t"
        "stp	x12, x13, [%[a], 0]\n\t"
        "sbcs	x15, x15, x7\n\t"
        "stp	x14, x15, [%[a], 16]\n\t"
        "ldp	x4, x5, [%[m], 32]\n\t"
        "ldp	x6, x7, [%[m], 48]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x16, x16, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x17, x17, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x19, x19, x6\n\t"
        "stp	x16, x17, [%[a], 32]\n\t"
        "sbcs	x20, x20, x7\n\t"
        "stp	x19, x20, [%[a], 48]\n\t"
        "ldp	x4, x5, [%[m], 64]\n\t"
        "ldp	x6, x7, [%[m], 80]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x21, x21, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x22, x22, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x23, x23, x6\n\t"
        "stp	x21, x22, [%[a], 64]\n\t"
        "sbcs	x24, x24, x7\n\t"
        "stp	x23, x24, [%[a], 80]\n\t"
        "ldp	x4, x5, [%[m], 96]\n\t"
        "ldp	x6, x7, [%[m], 112]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x25, x25, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x26, x26, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x27, x27, x6\n\t"
        "stp	x25, x26, [%[a], 96]\n\t"
        "sbcs	x28, x28, x7\n\t"
        "stp	x27, x28, [%[a], 112]\n\t"
        : [a] "+r" (a), [mp] "+r" (mp)
        : [m] "r" (m)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x19", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );

}

/* Multiply two Montgomery form numbers mod the modulus (prime).
 * (r = a * b mod m)
 *
 * r   Result of multiplication.
 * a   First number to multiply in Montgomery form.
 * b   Second number to multiply in Montgomery form.
 * m   Modulus (prime).
 * mp  Montgomery mulitplier.
 */
SP_NOINLINE static void sp_2048_mont_mul_16(sp_digit* r, const sp_digit* a,
        const sp_digit* b, const sp_digit* m, sp_digit mp)
{
    sp_2048_mul_16(r, a, b);
    sp_2048_mont_reduce_16(r, m, mp);
}

/* Square the Montgomery form number. (r = a * a mod m)
 *
 * r   Result of squaring.
 * a   Number to square in Montgomery form.
 * m   Modulus (prime).
 * mp  Montgomery mulitplier.
 */
SP_NOINLINE static void sp_2048_mont_sqr_16(sp_digit* r, const sp_digit* a,
        const sp_digit* m, sp_digit mp)
{
    sp_2048_sqr_16(r, a);
    sp_2048_mont_reduce_16(r, m, mp);
}

/* Conditionally subtract b from a using the mask m.
 * m is -1 to subtract and 0 when not copying.
 *
 * r  A single precision number representing condition subtract result.
 * a  A single precision number to subtract from.
 * b  A single precision number to subtract.
 * m  Mask value to apply.
 */
static sp_digit sp_2048_cond_sub_16(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
#ifdef WOLFSSL_SP_SMALL
    sp_digit c = 0;

    __asm__ __volatile__ (
        "mov	x8, #0\n\t"
        "1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldr	x4, [%[a], x8]\n\t"
        "ldr	x5, [%[b], x8]\n\t"
        "and	x5, x5, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "csetm	%[c], cc\n\t"
        "str	x4, [%[r], x8]\n\t"
        "add	x8, x8, #8\n\t"
        "cmp	x8, 128\n\t"
        "b.lt	1b\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8", "x9", "x10", "x11", "x12"
    );

    return c;
#else
    __asm__ __volatile__ (

        "ldp	x5, x7, [%[b], 0]\n\t"
        "ldp	x11, x12, [%[b], 16]\n\t"
        "ldp	x4, x6, [%[a], 0]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 16]\n\t"
        "and	x7, x7, %[m]\n\t"
        "subs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 0]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 16]\n\t"
        "ldp	x5, x7, [%[b], 32]\n\t"
        "ldp	x11, x12, [%[b], 48]\n\t"
        "ldp	x4, x6, [%[a], 32]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 48]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 32]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 48]\n\t"
        "ldp	x5, x7, [%[b], 64]\n\t"
        "ldp	x11, x12, [%[b], 80]\n\t"
        "ldp	x4, x6, [%[a], 64]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 80]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 64]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 80]\n\t"
        "ldp	x5, x7, [%[b], 96]\n\t"
        "ldp	x11, x12, [%[b], 112]\n\t"
        "ldp	x4, x6, [%[a], 96]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 112]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 96]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 112]\n\t"
        "csetm	%[r], cc\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8", "x9", "x10", "x11", "x12"
    );

    return (sp_digit)r;
#endif /* WOLFSSL_SP_SMALL */
}

/* Mul a by digit b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision digit.
 */
static void sp_2048_mul_d_16(sp_digit* r, const sp_digit* a,
        sp_digit b)
{
#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldr	x8, [%[a]]\n\t"
        "mul	x5, %[b], x8\n\t"
        "umulh	x3, %[b], x8\n\t"
        "mov	x4, xzr\n\t"
        "str	x5, [%[r]]\n\t"
        "mov	x5, xzr\n\t"
        "mov	x9, #8\n\t"
        "1:\n\t"
        "ldr	x8, [%[a], x9]\n\t"
        "mul	x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "str	x3, [%[r], x9]\n\t"
        "mov	x3, x4\n\t"
        "mov	x4, x5\n\t"
        "mov	x5, #0\n\t"
        "add	x9, x9, #8\n\t"
        "cmp	x9, 128\n\t"
        "b.lt	1b\n\t"
        "str	x3, [%[r], 128]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );
#else
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldp	x9, x10, [%[a]]\n\t"
        "mul	x3, %[b], x9\n\t"
        "umulh	x4, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "# A[1] * B\n\t"
        "str	x3, [%[r]]\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[2] * B\n\t"
        "ldp	x9, x10, [%[a], 16]\n\t"
        "str	x4, [%[r], 8]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[3] * B\n\t"
        "str	x5, [%[r], 16]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[4] * B\n\t"
        "ldp	x9, x10, [%[a], 32]\n\t"
        "str	x3, [%[r], 24]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[5] * B\n\t"
        "str	x4, [%[r], 32]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[6] * B\n\t"
        "ldp	x9, x10, [%[a], 48]\n\t"
        "str	x5, [%[r], 40]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[7] * B\n\t"
        "str	x3, [%[r], 48]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[8] * B\n\t"
        "ldp	x9, x10, [%[a], 64]\n\t"
        "str	x4, [%[r], 56]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[9] * B\n\t"
        "str	x5, [%[r], 64]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[10] * B\n\t"
        "ldp	x9, x10, [%[a], 80]\n\t"
        "str	x3, [%[r], 72]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[11] * B\n\t"
        "str	x4, [%[r], 80]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[12] * B\n\t"
        "ldp	x9, x10, [%[a], 96]\n\t"
        "str	x5, [%[r], 88]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[13] * B\n\t"
        "str	x3, [%[r], 96]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[14] * B\n\t"
        "ldp	x9, x10, [%[a], 112]\n\t"
        "str	x4, [%[r], 104]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[15] * B\n\t"
        "str	x5, [%[r], 112]\n\t"
        "mul	x6, %[b], x10\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "stp	x3, x4, [%[r], 120]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );
#endif
}

/* Divide the double width number (d1|d0) by the divisor. (d1|d0 / div)
 *
 * Assumes divisor has higest bit set.
 *
 * d1   The high order half of the number to divide.
 * d0   The low order half of the number to divide.
 * div  The divisor.
 * returns the result of the division.
 */
static sp_digit div_2048_word_16(sp_digit d1, sp_digit d0, sp_digit div)
{
    __asm__ __volatile__ (
        "lsr	x8, %[div], 32\n\t"
        "add	x5, x8, 1\n\t"

        "udiv	x3, %[d1], x5\n\t"
        "lsl	x7, %[div], 32\n\t"
        "movz	x9, #1, lsl 32\n\t"
        "lsl	x6, x3, 32\n\t"
        "mul	x4, %[div], x6\n\t"
        "umulh	x3, %[div], x6\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "cmp	%[d1], x5\n\t"
        "cset	x9, ge\n\t"
        "csetm	x10, ge\n\t"
        "lsl	x9, x9, #32\n\t"
        "and	x7, x7, x10\n\t"
        "and	x8, x8, x10\n\t"
        "subs	%[d0], %[d0], x7\n\t"
        "add	x6, x6, x9\n\t"
        "sbc	%[d1], %[d1], x8\n\t"

        "extr	x3, %[d1], %[d0], 32\n\t"

        "udiv	x3, x3, x5\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "extr	x3, %[d1], %[d0], 32\n\t"

        "udiv   x3, x3, x5\n\t"
        "add    x6, x6, x3\n\t"
        "mul    x4, %[div], x3\n\t"
        "sub    %[d0], %[d0], x4\n\t"

        "udiv	x3, %[d0], %[div]\n\t"
        "add	%[d1], x6, x3\n\t"

        : [d1] "+r" (d1), [d0] "+r" (d0)
        : [div] "r" (div)
        : "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return d1;
}

/* AND m into each word of a and store in r.
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * m  Mask to AND against each digit.
 */
static void sp_2048_mask_16(sp_digit* r, const sp_digit* a, sp_digit m)
{
#ifdef WOLFSSL_SP_SMALL
    int i;

    for (i=0; i<16; i++) {
        r[i] = a[i] & m;
    }
#else
    int i;

    for (i = 0; i < 16; i += 8) {
        r[i+0] = a[i+0] & m;
        r[i+1] = a[i+1] & m;
        r[i+2] = a[i+2] & m;
        r[i+3] = a[i+3] & m;
        r[i+4] = a[i+4] & m;
        r[i+5] = a[i+5] & m;
        r[i+6] = a[i+6] & m;
        r[i+7] = a[i+7] & m;
    }
#endif
}

/* Compare a with b in constant time.
 *
 * a  A single precision integer.
 * b  A single precision integer.
 * return -ve, 0 or +ve if a is less than, equal to or greater than b
 * respectively.
 */
static sp_int64 sp_2048_cmp_16(const sp_digit* a, const sp_digit* b)
{
#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "mov	x3, #0\n\t"
        "mov	x2, #-1\n\t"
        "mov	x10, #16\n\t"
        "add	%[a], %[a], #112\n\t"
        "add	%[b], %[b], #112\n\t"
        "1:\n\t"
        "ldp	x6, x7, [%[a]], -16\n\t"
        "ldp	x8, x9, [%[b]], -16\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x10, x10, #2\n\t"
        "b.ne	1b\n\t"
        "cmp	x2, #0\n\t"
        "cset	%[a], eq\n\t"
        "orr	%[a], %[a], x3\n\t"
        : [a] "+r" (a), [b] "+r" (b)
        :
        : "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );
#else
    __asm__ __volatile__ (
        "mov	x3, #0\n\t"
        "mov	x2, #-1\n\t"
        "ldp	x6, x7, [%[a], 112]\n\t"
        "ldp	x8, x9, [%[b], 112]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 96]\n\t"
        "ldp	x8, x9, [%[b], 96]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 80]\n\t"
        "ldp	x8, x9, [%[b], 80]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 64]\n\t"
        "ldp	x8, x9, [%[b], 64]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 48]\n\t"
        "ldp	x8, x9, [%[b], 48]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 32]\n\t"
        "ldp	x8, x9, [%[b], 32]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 16]\n\t"
        "ldp	x8, x9, [%[b], 16]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 0]\n\t"
        "ldp	x8, x9, [%[b], 0]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "cmp	x2, #0\n\t"
        "cset	%[a], eq\n\t"
        "orr	%[a], %[a], x3\n\t"
        : [a] "+r" (a)
        : [b] "r" (b)
        : "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );
#endif

    return (sp_int64)a;
}

/* Divide d in a and put remainder into r (m*d + r = a)
 * m is not calculated as it is not needed at this time.
 *
 * a  Number to be divided.
 * d  Number to divide with.
 * m  Multiplier result.
 * r  Remainder from the division.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_2048_div_16(const sp_digit* a, const sp_digit* d, sp_digit* m,
        sp_digit* r)
{
    sp_digit t1[32], t2[17];
    sp_digit div, r1;
    int i;

    (void)m;

    div = d[15];
    XMEMCPY(t1, a, sizeof(*t1) * 2 * 16);
    r1 = sp_2048_cmp_16(&t1[16], d) >= 0;
    sp_2048_cond_sub_16(&t1[16], &t1[16], d, (sp_digit)0 - r1);
    for (i = 15; i >= 0; i--) {
        volatile sp_digit mask = (sp_digit)0 - (t1[16 + i] == div);
        sp_digit hi = t1[16 + i] + mask;
        r1 = div_2048_word_16(hi, t1[16 + i - 1], div);
        r1 |= mask;

        sp_2048_mul_d_16(t2, d, r1);
        t1[16 + i] += sp_2048_sub_in_place_16(&t1[i], t2);
        t1[16 + i] -= t2[16];
        sp_2048_mask_16(t2, d, t1[16 + i]);
        t1[16 + i] += sp_2048_add_16(&t1[i], &t1[i], t2);
        sp_2048_mask_16(t2, d, t1[16 + i]);
        t1[16 + i] += sp_2048_add_16(&t1[i], &t1[i], t2);
    }

    r1 = sp_2048_cmp_16(t1, d) >= 0;
    sp_2048_cond_sub_16(r, t1, d, (sp_digit)0 - r1);

    return MP_OKAY;
}

/* Reduce a modulo m into r. (r = a mod m)
 *
 * r  A single precision number that is the reduced result.
 * a  A single precision number that is to be reduced.
 * m  A single precision number that is the modulus to reduce with.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_2048_mod_16(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    return sp_2048_div_16(a, m, NULL, r);
}

#ifdef WOLFSSL_SP_SMALL
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns  0 on success.
 * returns  MEMORY_E on dynamic memory allocation failure.
 * returns  MP_VAL when base is even or exponent is 0.
 */
static int sp_2048_mod_exp_16(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* td = NULL;
#else
    sp_digit td[16 * 32];
#endif
    sp_digit* t[16];
    sp_digit* norm = NULL;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c;
    byte y;
    int err = MP_OKAY;

    if (bits == 0) {
        err = MP_VAL;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        td = (sp_digit*)XMALLOC(sizeof(sp_digit) * (16 * 32), NULL,
                                DYNAMIC_TYPE_TMP_BUFFER);
        if (td == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        norm = td;
        for (i=0; i<16; i++) {
            t[i] = td + i * 32;
        }

        sp_2048_mont_setup(m, &mp);
        sp_2048_mont_norm_16(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 16U);
        if (reduceA != 0) {
            err = sp_2048_mod_16(t[1] + 16, a, m);
            if (err == MP_OKAY) {
                err = sp_2048_mod_16(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 16, a, sizeof(sp_digit) * 16);
            err = sp_2048_mod_16(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_2048_mont_sqr_16(t[ 2], t[ 1], m, mp);
        sp_2048_mont_mul_16(t[ 3], t[ 2], t[ 1], m, mp);
        sp_2048_mont_sqr_16(t[ 4], t[ 2], m, mp);
        sp_2048_mont_mul_16(t[ 5], t[ 3], t[ 2], m, mp);
        sp_2048_mont_sqr_16(t[ 6], t[ 3], m, mp);
        sp_2048_mont_mul_16(t[ 7], t[ 4], t[ 3], m, mp);
        sp_2048_mont_sqr_16(t[ 8], t[ 4], m, mp);
        sp_2048_mont_mul_16(t[ 9], t[ 5], t[ 4], m, mp);
        sp_2048_mont_sqr_16(t[10], t[ 5], m, mp);
        sp_2048_mont_mul_16(t[11], t[ 6], t[ 5], m, mp);
        sp_2048_mont_sqr_16(t[12], t[ 6], m, mp);
        sp_2048_mont_mul_16(t[13], t[ 7], t[ 6], m, mp);
        sp_2048_mont_sqr_16(t[14], t[ 7], m, mp);
        sp_2048_mont_mul_16(t[15], t[ 8], t[ 7], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 4;
        if (c == 64) {
            c = 60;
        }
        if (c < 0) {
            /* Number of bits in top word is less than number needed. */
            c = -c;
            y = (byte)(n << c);
            n = e[i--];
            y |= (byte)(n >> (64 - c));
            n <<= c;
            c = 64 - c;
        }
        else if (c == 0) {
            /* All bits in top word used. */
            y = (byte)n;
        }
        else {
            y = (byte)(n >> c);
            n <<= 64 - c;
        }
        XMEMCPY(r, t[y], sizeof(sp_digit) * 16);
        for (; i>=0 || c>=4; ) {
            if (c == 0) {
                n = e[i--];
                y = (byte)(n >> 60);
                n <<= 4;
                c = 60;
            }
            else if (c < 4) {
                y = (byte)(n >> 60);
                n = e[i--];
                c = 4 - c;
                y |= (byte)(n >> (64 - c));
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (byte)((n >> 60) & 0xf);
                n <<= 4;
                c -= 4;
            }

            sp_2048_mont_sqr_16(r, r, m, mp);
            sp_2048_mont_sqr_16(r, r, m, mp);
            sp_2048_mont_sqr_16(r, r, m, mp);
            sp_2048_mont_sqr_16(r, r, m, mp);

            sp_2048_mont_mul_16(r, r, t[y], m, mp);
        }

        XMEMSET(&r[16], 0, sizeof(sp_digit) * 16U);
        sp_2048_mont_reduce_16(r, m, mp);

        mask = 0 - (sp_2048_cmp_16(r, m) >= 0);
        sp_2048_cond_sub_16(r, r, m, mask);
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (td != NULL)
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return err;
}
#else
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns  0 on success.
 * returns  MEMORY_E on dynamic memory allocation failure.
 * returns  MP_VAL when base is even or exponent is 0.
 */
static int sp_2048_mod_exp_16(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* td = NULL;
#else
    sp_digit td[32 * 32];
#endif
    sp_digit* t[32];
    sp_digit* norm = NULL;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c;
    byte y;
    int err = MP_OKAY;

    if (bits == 0) {
        err = MP_VAL;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        td = (sp_digit*)XMALLOC(sizeof(sp_digit) * (32 * 32), NULL,
                                DYNAMIC_TYPE_TMP_BUFFER);
        if (td == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        norm = td;
        for (i=0; i<32; i++) {
            t[i] = td + i * 32;
        }

        sp_2048_mont_setup(m, &mp);
        sp_2048_mont_norm_16(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 16U);
        if (reduceA != 0) {
            err = sp_2048_mod_16(t[1] + 16, a, m);
            if (err == MP_OKAY) {
                err = sp_2048_mod_16(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 16, a, sizeof(sp_digit) * 16);
            err = sp_2048_mod_16(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_2048_mont_sqr_16(t[ 2], t[ 1], m, mp);
        sp_2048_mont_mul_16(t[ 3], t[ 2], t[ 1], m, mp);
        sp_2048_mont_sqr_16(t[ 4], t[ 2], m, mp);
        sp_2048_mont_mul_16(t[ 5], t[ 3], t[ 2], m, mp);
        sp_2048_mont_sqr_16(t[ 6], t[ 3], m, mp);
        sp_2048_mont_mul_16(t[ 7], t[ 4], t[ 3], m, mp);
        sp_2048_mont_sqr_16(t[ 8], t[ 4], m, mp);
        sp_2048_mont_mul_16(t[ 9], t[ 5], t[ 4], m, mp);
        sp_2048_mont_sqr_16(t[10], t[ 5], m, mp);
        sp_2048_mont_mul_16(t[11], t[ 6], t[ 5], m, mp);
        sp_2048_mont_sqr_16(t[12], t[ 6], m, mp);
        sp_2048_mont_mul_16(t[13], t[ 7], t[ 6], m, mp);
        sp_2048_mont_sqr_16(t[14], t[ 7], m, mp);
        sp_2048_mont_mul_16(t[15], t[ 8], t[ 7], m, mp);
        sp_2048_mont_sqr_16(t[16], t[ 8], m, mp);
        sp_2048_mont_mul_16(t[17], t[ 9], t[ 8], m, mp);
        sp_2048_mont_sqr_16(t[18], t[ 9], m, mp);
        sp_2048_mont_mul_16(t[19], t[10], t[ 9], m, mp);
        sp_2048_mont_sqr_16(t[20], t[10], m, mp);
        sp_2048_mont_mul_16(t[21], t[11], t[10], m, mp);
        sp_2048_mont_sqr_16(t[22], t[11], m, mp);
        sp_2048_mont_mul_16(t[23], t[12], t[11], m, mp);
        sp_2048_mont_sqr_16(t[24], t[12], m, mp);
        sp_2048_mont_mul_16(t[25], t[13], t[12], m, mp);
        sp_2048_mont_sqr_16(t[26], t[13], m, mp);
        sp_2048_mont_mul_16(t[27], t[14], t[13], m, mp);
        sp_2048_mont_sqr_16(t[28], t[14], m, mp);
        sp_2048_mont_mul_16(t[29], t[15], t[14], m, mp);
        sp_2048_mont_sqr_16(t[30], t[15], m, mp);
        sp_2048_mont_mul_16(t[31], t[16], t[15], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 5;
        if (c == 64) {
            c = 59;
        }
        if (c < 0) {
            /* Number of bits in top word is less than number needed. */
            c = -c;
            y = (byte)(n << c);
            n = e[i--];
            y |= (byte)(n >> (64 - c));
            n <<= c;
            c = 64 - c;
        }
        else if (c == 0) {
            /* All bits in top word used. */
            y = (byte)n;
        }
        else {
            y = (byte)(n >> c);
            n <<= 64 - c;
        }
        XMEMCPY(r, t[y], sizeof(sp_digit) * 16);
        for (; i>=0 || c>=5; ) {
            if (c == 0) {
                n = e[i--];
                y = (byte)(n >> 59);
                n <<= 5;
                c = 59;
            }
            else if (c < 5) {
                y = (byte)(n >> 59);
                n = e[i--];
                c = 5 - c;
                y |= (byte)(n >> (64 - c));
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (byte)((n >> 59) & 0x1f);
                n <<= 5;
                c -= 5;
            }

            sp_2048_mont_sqr_16(r, r, m, mp);
            sp_2048_mont_sqr_16(r, r, m, mp);
            sp_2048_mont_sqr_16(r, r, m, mp);
            sp_2048_mont_sqr_16(r, r, m, mp);
            sp_2048_mont_sqr_16(r, r, m, mp);

            sp_2048_mont_mul_16(r, r, t[y], m, mp);
        }

        XMEMSET(&r[16], 0, sizeof(sp_digit) * 16U);
        sp_2048_mont_reduce_16(r, m, mp);

        mask = 0 - (sp_2048_cmp_16(r, m) >= 0);
        sp_2048_cond_sub_16(r, r, m, mask);
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (td != NULL)
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return err;
}
#endif /* WOLFSSL_SP_SMALL */

#endif /* (WOLFSSL_HAVE_SP_RSA & !WOLFSSL_RSA_PUBLIC_ONLY) | WOLFSSL_HAVE_SP_DH */

#if (defined(WOLFSSL_HAVE_SP_RSA) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)) || defined(WOLFSSL_HAVE_SP_DH)
/* r = 2^n mod m where n is the number of bits to reduce by.
 * Given m must be 2048 bits, just need to subtract.
 *
 * r  A single precision number.
 * m  A single precision number.
 */
static void sp_2048_mont_norm_32(sp_digit* r, const sp_digit* m)
{
    XMEMSET(r, 0, sizeof(sp_digit) * 32);

    /* r = 2^n mod m */
    sp_2048_sub_in_place_32(r, m);
}

#endif /* (WOLFSSL_HAVE_SP_RSA & !WOLFSSL_RSA_PUBLIC_ONLY) | WOLFSSL_HAVE_SP_DH */
/* Reduce the number back to 2048 bits using Montgomery reduction.
 *
 * a   A single precision number to reduce in place.
 * m   The single precision number representing the modulus.
 * mp  The digit representing the negative inverse of m mod 2^n.
 */
SP_NOINLINE static void sp_2048_mont_reduce_32(sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    __asm__ __volatile__ (
        "ldp	x11, x12, [%[a], 0]\n\t"
        "ldp	x13, x14, [%[a], 16]\n\t"
        "ldp	x15, x16, [%[a], 32]\n\t"
        "ldp	x17, x19, [%[a], 48]\n\t"
        "ldp	x20, x21, [%[a], 64]\n\t"
        "ldp	x22, x23, [%[a], 80]\n\t"
        "# No carry yet\n\t"
        "mov	x3, xzr\n\t"
        "# i = 0..31\n\t"
        "mov	x4, 32\n\t"
        "\n1:\n\t"
        "# mu = a[i] * mp\n\t"
        "mul	x10, %[mp], x11\n\t"
        "ldp	x24, x25, [%[m], 0]\n\t"
        "ldp	x26, x27, [%[m], 16]\n\t"
        "# a[i+0] += m[0] * mu\n\t"
        "mul	x5, x24, x10\n\t"
        "umulh	x6, x24, x10\n\t"
        "# a[i+1] += m[1] * mu\n\t"
        "adds	x11, x11, x5\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x11, x12, x5\n\t"
        "# a[i+2] += m[2] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x11, x11, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x12, x13, x5\n\t"
        "# a[i+3] += m[3] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x12, x12, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x13, x14, x5\n\t"
        "ldp	x24, x25, [%[m], 32]\n\t"
        "ldp	x26, x27, [%[m], 48]\n\t"
        "# a[i+4] += m[4] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x13, x13, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x14, x15, x5\n\t"
        "# a[i+5] += m[5] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x14, x14, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x15, x16, x5\n\t"
        "# a[i+6] += m[6] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x15, x15, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x16, x17, x5\n\t"
        "# a[i+7] += m[7] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x16, x16, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x17, x19, x5\n\t"
        "ldp	x24, x25, [%[m], 64]\n\t"
        "ldp	x26, x27, [%[m], 80]\n\t"
        "# a[i+8] += m[8] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x17, x17, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x19, x20, x5\n\t"
        "# a[i+9] += m[9] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x19, x19, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x20, x21, x5\n\t"
        "# a[i+10] += m[10] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x20, x20, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x21, x22, x5\n\t"
        "# a[i+11] += m[11] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x21, x21, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x22, x23, x5\n\t"
        "ldp	x24, x25, [%[m], 96]\n\t"
        "ldp	x26, x27, [%[m], 112]\n\t"
        "# a[i+12] += m[12] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x22, x22, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "ldr	x23, [%[a], 96]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x23, x23, x5\n\t"
        "# a[i+13] += m[13] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x23, x23, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "ldp	x8, x9, [%[a], 104]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+14] += m[14] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 104]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+15] += m[15] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 112]\n\t"
        "ldp	x8, x9, [%[a], 120]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 128]\n\t"
        "ldp	x26, x27, [%[m], 144]\n\t"
        "# a[i+16] += m[16] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 120]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+17] += m[17] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 128]\n\t"
        "ldp	x8, x9, [%[a], 136]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+18] += m[18] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 136]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+19] += m[19] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 144]\n\t"
        "ldp	x8, x9, [%[a], 152]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 160]\n\t"
        "ldp	x26, x27, [%[m], 176]\n\t"
        "# a[i+20] += m[20] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 152]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+21] += m[21] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 160]\n\t"
        "ldp	x8, x9, [%[a], 168]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+22] += m[22] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 168]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+23] += m[23] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 176]\n\t"
        "ldp	x8, x9, [%[a], 184]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 192]\n\t"
        "ldp	x26, x27, [%[m], 208]\n\t"
        "# a[i+24] += m[24] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 184]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+25] += m[25] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 192]\n\t"
        "ldp	x8, x9, [%[a], 200]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+26] += m[26] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 200]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+27] += m[27] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 208]\n\t"
        "ldp	x8, x9, [%[a], 216]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 224]\n\t"
        "ldp	x26, x27, [%[m], 240]\n\t"
        "# a[i+28] += m[28] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 216]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+29] += m[29] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 224]\n\t"
        "ldp	x8, x9, [%[a], 232]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+30] += m[30] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 232]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+31] += m[31] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 240]\n\t"
        "umulh	x7, x27, x10\n\t"
        "ldp	x8, x9, [%[a], 248]\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x7, x7, x3\n\t"
        "cset	x3, cs\n\t"
        "adds	x8, x8, x5\n\t"
        "str	x8, [%[a], 248]\n\t"
        "adcs	x9, x9, x7\n\t"
        "str	x9, [%[a], 256]\n\t"
        "adc	x3, x3, xzr\n\t"
        "subs	x4, x4, 1\n\t"
        "add	%[a], %[a], 8\n\t"
        "b.ne	1b\n\t"
        "# Create mask\n\t"
        "neg	x3, x3\n\t"
        "mov   %[mp], %[a]\n\t"
        "sub	%[a], %[a], 256\n\t"
        "# Subtract masked modulus\n\t"
        "ldp	x4, x5, [%[m], 0]\n\t"
        "ldp	x6, x7, [%[m], 16]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "subs	x11, x11, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x12, x12, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x13, x13, x6\n\t"
        "stp	x11, x12, [%[a], 0]\n\t"
        "sbcs	x14, x14, x7\n\t"
        "stp	x13, x14, [%[a], 16]\n\t"
        "ldp	x4, x5, [%[m], 32]\n\t"
        "ldp	x6, x7, [%[m], 48]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x15, x15, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x16, x16, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x17, x17, x6\n\t"
        "stp	x15, x16, [%[a], 32]\n\t"
        "sbcs	x19, x19, x7\n\t"
        "stp	x17, x19, [%[a], 48]\n\t"
        "ldp	x4, x5, [%[m], 64]\n\t"
        "ldp	x6, x7, [%[m], 80]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x20, x20, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x21, x21, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x22, x22, x6\n\t"
        "stp	x20, x21, [%[a], 64]\n\t"
        "sbcs	x23, x23, x7\n\t"
        "stp	x22, x23, [%[a], 80]\n\t"
        "ldp	x4, x5, [%[m], 96]\n\t"
        "ldp	x6, x7, [%[m], 112]\n\t"
        "ldp	x8, x9, [%[mp], 96]\n\t"
        "ldp	x10, x11, [%[mp], 112]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 96]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 112]\n\t"
        "ldp	x4, x5, [%[m], 128]\n\t"
        "ldp	x6, x7, [%[m], 144]\n\t"
        "ldp	x8, x9, [%[mp], 128]\n\t"
        "ldp	x10, x11, [%[mp], 144]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 128]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 144]\n\t"
        "ldp	x4, x5, [%[m], 160]\n\t"
        "ldp	x6, x7, [%[m], 176]\n\t"
        "ldp	x8, x9, [%[mp], 160]\n\t"
        "ldp	x10, x11, [%[mp], 176]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 160]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 176]\n\t"
        "ldp	x4, x5, [%[m], 192]\n\t"
        "ldp	x6, x7, [%[m], 208]\n\t"
        "ldp	x8, x9, [%[mp], 192]\n\t"
        "ldp	x10, x11, [%[mp], 208]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 192]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 208]\n\t"
        "ldp	x4, x5, [%[m], 224]\n\t"
        "ldp	x6, x7, [%[m], 240]\n\t"
        "ldp	x8, x9, [%[mp], 224]\n\t"
        "ldp	x10, x11, [%[mp], 240]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 224]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 240]\n\t"
        : [a] "+r" (a), [mp] "+r" (mp)
        : [m] "r" (m)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x10", "x8", "x9", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x19", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27"
    );

}

/* Multiply two Montgomery form numbers mod the modulus (prime).
 * (r = a * b mod m)
 *
 * r   Result of multiplication.
 * a   First number to multiply in Montgomery form.
 * b   Second number to multiply in Montgomery form.
 * m   Modulus (prime).
 * mp  Montgomery mulitplier.
 */
SP_NOINLINE static void sp_2048_mont_mul_32(sp_digit* r, const sp_digit* a,
        const sp_digit* b, const sp_digit* m, sp_digit mp)
{
    sp_2048_mul_32(r, a, b);
    sp_2048_mont_reduce_32(r, m, mp);
}

/* Square the Montgomery form number. (r = a * a mod m)
 *
 * r   Result of squaring.
 * a   Number to square in Montgomery form.
 * m   Modulus (prime).
 * mp  Montgomery mulitplier.
 */
SP_NOINLINE static void sp_2048_mont_sqr_32(sp_digit* r, const sp_digit* a,
        const sp_digit* m, sp_digit mp)
{
    sp_2048_sqr_32(r, a);
    sp_2048_mont_reduce_32(r, m, mp);
}

#ifdef WOLFSSL_SP_SMALL
/* Sub b from a into r. (r = a - b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_2048_sub_32(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x11, %[a], 256\n\t"
        "\n1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldp	x3, x4, [%[a]], #16\n\t"
        "ldp	x5, x6, [%[a]], #16\n\t"
        "ldp	x7, x8, [%[b]], #16\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x9, x10, [%[b]], #16\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r]], #16\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r]], #16\n\t"
        "csetm	%[c], cc\n\t"
        "cmp	%[a], x11\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [r] "+r" (r), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    return c;
}

#else
/* Sub b from a into r. (r = a - b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_2048_sub_32(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "subs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x7, x8, [%[b], 64]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 80]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "ldp	x3, x4, [%[a], 96]\n\t"
        "ldp	x7, x8, [%[b], 96]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 112]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 112]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 96]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 112]\n\t"
        "ldp	x3, x4, [%[a], 128]\n\t"
        "ldp	x7, x8, [%[b], 128]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 144]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 144]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 128]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 144]\n\t"
        "ldp	x3, x4, [%[a], 160]\n\t"
        "ldp	x7, x8, [%[b], 160]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 176]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 176]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 160]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 176]\n\t"
        "ldp	x3, x4, [%[a], 192]\n\t"
        "ldp	x7, x8, [%[b], 192]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 208]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 208]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 192]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 208]\n\t"
        "ldp	x3, x4, [%[a], 224]\n\t"
        "ldp	x7, x8, [%[b], 224]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 240]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 240]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 224]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 240]\n\t"
        "csetm	%[r], cc\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return (sp_digit)r;
}

#endif /* WOLFSSL_SP_SMALL */
/* Divide the double width number (d1|d0) by the divisor. (d1|d0 / div)
 *
 * Assumes divisor has higest bit set.
 *
 * d1   The high order half of the number to divide.
 * d0   The low order half of the number to divide.
 * div  The divisor.
 * returns the result of the division.
 */
static sp_digit div_2048_word_32_cond(sp_digit d1, sp_digit d0, sp_digit div)
{
    __asm__ __volatile__ (
        "lsr	x8, %[div], 32\n\t"
        "add	x5, x8, 1\n\t"

        "udiv	x3, %[d1], x5\n\t"
        "lsl	x7, %[div], 32\n\t"
        "movz	x9, #1, lsl 32\n\t"
        "lsl	x6, x3, 32\n\t"
        "mul	x4, %[div], x6\n\t"
        "umulh	x3, %[div], x6\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "cmp	%[d1], x5\n\t"
        "b.lt	1f\n\t"
        "subs	%[d0], %[d0], x7\n\t"
        "add	x6, x6, x9\n\t"
        "sbc	%[d1], %[d1], x8\n\t"
        "1:\n\t"

        "extr	x3, %[d1], %[d0], 32\n\t"

        "udiv	x3, x3, x5\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "extr	x3, %[d1], %[d0], 32\n\t"

        "cmp	x3, x5\n\t"
        "b.lt	2f\n\t"
        "udiv   x3, x3, x5\n\t"
        "add    x6, x6, x3\n\t"
        "mul    x4, %[div], x3\n\t"
        "sub    %[d0], %[d0], x4\n\t"
        "2:\n\t"

        "udiv	x3, %[d0], %[div]\n\t"
        "add	%[d1], x6, x3\n\t"

        : [d1] "+r" (d1), [d0] "+r" (d0)
        : [div] "r" (div)
        : "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );

    return d1;
}

/* Divide d in a and put remainder into r (m*d + r = a)
 * m is not calculated as it is not needed at this time.
 *
 * a  Number to be divided.
 * d  Number to divide with.
 * m  Multiplier result.
 * r  Remainder from the division.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_2048_div_32_cond(const sp_digit* a, const sp_digit* d, sp_digit* m,
        sp_digit* r)
{
    sp_digit t1[64], t2[33];
    sp_digit div, r1;
    int i;

    (void)m;

    div = d[31];
    XMEMCPY(t1, a, sizeof(*t1) * 2 * 32);
    for (i = 31; i > 0; i--) {
        if (t1[i + 32] != d[i])
            break;
    }
    if (t1[i + 32] >= d[i]) {
        sp_2048_sub_in_place_32(&t1[32], d);
    }
    for (i = 31; i >= 0; i--) {
        if (t1[32 + i] == div) {
            r1 = SP_DIGIT_MAX;
        }
        else {
            r1 = div_2048_word_32_cond(t1[32 + i], t1[32 + i - 1], div);
        }

        sp_2048_mul_d_32(t2, d, r1);
        t1[32 + i] += sp_2048_sub_in_place_32(&t1[i], t2);
        t1[32 + i] -= t2[32];
        if (t1[32 + i] != 0) {
            t1[32 + i] += sp_2048_add_32(&t1[i], &t1[i], d);
            if (t1[32 + i] != 0)
                t1[32 + i] += sp_2048_add_32(&t1[i], &t1[i], d);
        }
    }

    for (i = 31; i > 0; i--) {
        if (t1[i] != d[i])
            break;
    }
    if (t1[i] >= d[i]) {
        sp_2048_sub_32(r, t1, d);
    }
    else {
        XMEMCPY(r, t1, sizeof(*t1) * 32);
    }

    return MP_OKAY;
}

/* Reduce a modulo m into r. (r = a mod m)
 *
 * r  A single precision number that is the reduced result.
 * a  A single precision number that is to be reduced.
 * m  A single precision number that is the modulus to reduce with.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_2048_mod_32_cond(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    return sp_2048_div_32_cond(a, m, NULL, r);
}

#if (defined(WOLFSSL_HAVE_SP_RSA) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)) || defined(WOLFSSL_HAVE_SP_DH)
/* Conditionally subtract b from a using the mask m.
 * m is -1 to subtract and 0 when not copying.
 *
 * r  A single precision number representing condition subtract result.
 * a  A single precision number to subtract from.
 * b  A single precision number to subtract.
 * m  Mask value to apply.
 */
static sp_digit sp_2048_cond_sub_32(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
#ifdef WOLFSSL_SP_SMALL
    sp_digit c = 0;

    __asm__ __volatile__ (
        "mov	x8, #0\n\t"
        "1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldr	x4, [%[a], x8]\n\t"
        "ldr	x5, [%[b], x8]\n\t"
        "and	x5, x5, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "csetm	%[c], cc\n\t"
        "str	x4, [%[r], x8]\n\t"
        "add	x8, x8, #8\n\t"
        "cmp	x8, 256\n\t"
        "b.lt	1b\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8", "x9", "x10", "x11", "x12"
    );

    return c;
#else
    __asm__ __volatile__ (

        "ldp	x5, x7, [%[b], 0]\n\t"
        "ldp	x11, x12, [%[b], 16]\n\t"
        "ldp	x4, x6, [%[a], 0]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 16]\n\t"
        "and	x7, x7, %[m]\n\t"
        "subs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 0]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 16]\n\t"
        "ldp	x5, x7, [%[b], 32]\n\t"
        "ldp	x11, x12, [%[b], 48]\n\t"
        "ldp	x4, x6, [%[a], 32]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 48]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 32]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 48]\n\t"
        "ldp	x5, x7, [%[b], 64]\n\t"
        "ldp	x11, x12, [%[b], 80]\n\t"
        "ldp	x4, x6, [%[a], 64]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 80]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 64]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 80]\n\t"
        "ldp	x5, x7, [%[b], 96]\n\t"
        "ldp	x11, x12, [%[b], 112]\n\t"
        "ldp	x4, x6, [%[a], 96]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 112]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 96]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 112]\n\t"
        "ldp	x5, x7, [%[b], 128]\n\t"
        "ldp	x11, x12, [%[b], 144]\n\t"
        "ldp	x4, x6, [%[a], 128]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 144]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 128]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 144]\n\t"
        "ldp	x5, x7, [%[b], 160]\n\t"
        "ldp	x11, x12, [%[b], 176]\n\t"
        "ldp	x4, x6, [%[a], 160]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 176]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 160]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 176]\n\t"
        "ldp	x5, x7, [%[b], 192]\n\t"
        "ldp	x11, x12, [%[b], 208]\n\t"
        "ldp	x4, x6, [%[a], 192]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 208]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 192]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 208]\n\t"
        "ldp	x5, x7, [%[b], 224]\n\t"
        "ldp	x11, x12, [%[b], 240]\n\t"
        "ldp	x4, x6, [%[a], 224]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 240]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 224]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 240]\n\t"
        "csetm	%[r], cc\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8", "x9", "x10", "x11", "x12"
    );

    return (sp_digit)r;
#endif /* WOLFSSL_SP_SMALL */
}

/* Divide the double width number (d1|d0) by the divisor. (d1|d0 / div)
 *
 * Assumes divisor has higest bit set.
 *
 * d1   The high order half of the number to divide.
 * d0   The low order half of the number to divide.
 * div  The divisor.
 * returns the result of the division.
 */
static sp_digit div_2048_word_32(sp_digit d1, sp_digit d0, sp_digit div)
{
    __asm__ __volatile__ (
        "lsr	x8, %[div], 32\n\t"
        "add	x5, x8, 1\n\t"

        "udiv	x3, %[d1], x5\n\t"
        "lsl	x7, %[div], 32\n\t"
        "movz	x9, #1, lsl 32\n\t"
        "lsl	x6, x3, 32\n\t"
        "mul	x4, %[div], x6\n\t"
        "umulh	x3, %[div], x6\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "cmp	%[d1], x5\n\t"
        "cset	x9, ge\n\t"
        "csetm	x10, ge\n\t"
        "lsl	x9, x9, #32\n\t"
        "and	x7, x7, x10\n\t"
        "and	x8, x8, x10\n\t"
        "subs	%[d0], %[d0], x7\n\t"
        "add	x6, x6, x9\n\t"
        "sbc	%[d1], %[d1], x8\n\t"

        "extr	x3, %[d1], %[d0], 32\n\t"

        "udiv	x3, x3, x5\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "extr	x3, %[d1], %[d0], 32\n\t"

        "udiv   x3, x3, x5\n\t"
        "add    x6, x6, x3\n\t"
        "mul    x4, %[div], x3\n\t"
        "sub    %[d0], %[d0], x4\n\t"

        "udiv	x3, %[d0], %[div]\n\t"
        "add	%[d1], x6, x3\n\t"

        : [d1] "+r" (d1), [d0] "+r" (d0)
        : [div] "r" (div)
        : "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return d1;
}

/* AND m into each word of a and store in r.
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * m  Mask to AND against each digit.
 */
static void sp_2048_mask_32(sp_digit* r, const sp_digit* a, sp_digit m)
{
#ifdef WOLFSSL_SP_SMALL
    int i;

    for (i=0; i<32; i++) {
        r[i] = a[i] & m;
    }
#else
    int i;

    for (i = 0; i < 32; i += 8) {
        r[i+0] = a[i+0] & m;
        r[i+1] = a[i+1] & m;
        r[i+2] = a[i+2] & m;
        r[i+3] = a[i+3] & m;
        r[i+4] = a[i+4] & m;
        r[i+5] = a[i+5] & m;
        r[i+6] = a[i+6] & m;
        r[i+7] = a[i+7] & m;
    }
#endif
}

/* Compare a with b in constant time.
 *
 * a  A single precision integer.
 * b  A single precision integer.
 * return -ve, 0 or +ve if a is less than, equal to or greater than b
 * respectively.
 */
static sp_int64 sp_2048_cmp_32(const sp_digit* a, const sp_digit* b)
{
#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "mov	x3, #0\n\t"
        "mov	x2, #-1\n\t"
        "mov	x10, #32\n\t"
        "add	%[a], %[a], #240\n\t"
        "add	%[b], %[b], #240\n\t"
        "1:\n\t"
        "ldp	x6, x7, [%[a]], -16\n\t"
        "ldp	x8, x9, [%[b]], -16\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x10, x10, #2\n\t"
        "b.ne	1b\n\t"
        "cmp	x2, #0\n\t"
        "cset	%[a], eq\n\t"
        "orr	%[a], %[a], x3\n\t"
        : [a] "+r" (a), [b] "+r" (b)
        :
        : "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );
#else
    __asm__ __volatile__ (
        "mov	x3, #0\n\t"
        "mov	x2, #-1\n\t"
        "ldp	x6, x7, [%[a], 240]\n\t"
        "ldp	x8, x9, [%[b], 240]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 224]\n\t"
        "ldp	x8, x9, [%[b], 224]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 208]\n\t"
        "ldp	x8, x9, [%[b], 208]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 192]\n\t"
        "ldp	x8, x9, [%[b], 192]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 176]\n\t"
        "ldp	x8, x9, [%[b], 176]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 160]\n\t"
        "ldp	x8, x9, [%[b], 160]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 144]\n\t"
        "ldp	x8, x9, [%[b], 144]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 128]\n\t"
        "ldp	x8, x9, [%[b], 128]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 112]\n\t"
        "ldp	x8, x9, [%[b], 112]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 96]\n\t"
        "ldp	x8, x9, [%[b], 96]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 80]\n\t"
        "ldp	x8, x9, [%[b], 80]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 64]\n\t"
        "ldp	x8, x9, [%[b], 64]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 48]\n\t"
        "ldp	x8, x9, [%[b], 48]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 32]\n\t"
        "ldp	x8, x9, [%[b], 32]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 16]\n\t"
        "ldp	x8, x9, [%[b], 16]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 0]\n\t"
        "ldp	x8, x9, [%[b], 0]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "cmp	x2, #0\n\t"
        "cset	%[a], eq\n\t"
        "orr	%[a], %[a], x3\n\t"
        : [a] "+r" (a)
        : [b] "r" (b)
        : "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );
#endif

    return (sp_int64)a;
}

/* Divide d in a and put remainder into r (m*d + r = a)
 * m is not calculated as it is not needed at this time.
 *
 * a  Number to be divided.
 * d  Number to divide with.
 * m  Multiplier result.
 * r  Remainder from the division.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_2048_div_32(const sp_digit* a, const sp_digit* d, sp_digit* m,
        sp_digit* r)
{
    sp_digit t1[64], t2[33];
    sp_digit div, r1;
    int i;

    (void)m;

    div = d[31];
    XMEMCPY(t1, a, sizeof(*t1) * 2 * 32);
    r1 = sp_2048_cmp_32(&t1[32], d) >= 0;
    sp_2048_cond_sub_32(&t1[32], &t1[32], d, (sp_digit)0 - r1);
    for (i = 31; i >= 0; i--) {
        volatile sp_digit mask = (sp_digit)0 - (t1[32 + i] == div);
        sp_digit hi = t1[32 + i] + mask;
        r1 = div_2048_word_32(hi, t1[32 + i - 1], div);
        r1 |= mask;

        sp_2048_mul_d_32(t2, d, r1);
        t1[32 + i] += sp_2048_sub_in_place_32(&t1[i], t2);
        t1[32 + i] -= t2[32];
        sp_2048_mask_32(t2, d, t1[32 + i]);
        t1[32 + i] += sp_2048_add_32(&t1[i], &t1[i], t2);
        sp_2048_mask_32(t2, d, t1[32 + i]);
        t1[32 + i] += sp_2048_add_32(&t1[i], &t1[i], t2);
    }

    r1 = sp_2048_cmp_32(t1, d) >= 0;
    sp_2048_cond_sub_32(r, t1, d, (sp_digit)0 - r1);

    return MP_OKAY;
}

/* Reduce a modulo m into r. (r = a mod m)
 *
 * r  A single precision number that is the reduced result.
 * a  A single precision number that is to be reduced.
 * m  A single precision number that is the modulus to reduce with.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_2048_mod_32(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    return sp_2048_div_32(a, m, NULL, r);
}

#if (defined(WOLFSSL_HAVE_SP_RSA) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)) || \
                                                     defined(WOLFSSL_HAVE_SP_DH)
#ifdef WOLFSSL_SP_SMALL
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns  0 on success.
 * returns  MEMORY_E on dynamic memory allocation failure.
 * returns  MP_VAL when base is even or exponent is 0.
 */
static int sp_2048_mod_exp_32(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* td = NULL;
#else
    sp_digit td[32 * 64];
#endif
    sp_digit* t[32];
    sp_digit* norm = NULL;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c;
    byte y;
    int err = MP_OKAY;

    if (bits == 0) {
        err = MP_VAL;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        td = (sp_digit*)XMALLOC(sizeof(sp_digit) * (32 * 64), NULL,
                                DYNAMIC_TYPE_TMP_BUFFER);
        if (td == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        norm = td;
        for (i=0; i<32; i++) {
            t[i] = td + i * 64;
        }

        sp_2048_mont_setup(m, &mp);
        sp_2048_mont_norm_32(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 32U);
        if (reduceA != 0) {
            err = sp_2048_mod_32(t[1] + 32, a, m);
            if (err == MP_OKAY) {
                err = sp_2048_mod_32(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 32, a, sizeof(sp_digit) * 32);
            err = sp_2048_mod_32(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_2048_mont_sqr_32(t[ 2], t[ 1], m, mp);
        sp_2048_mont_mul_32(t[ 3], t[ 2], t[ 1], m, mp);
        sp_2048_mont_sqr_32(t[ 4], t[ 2], m, mp);
        sp_2048_mont_mul_32(t[ 5], t[ 3], t[ 2], m, mp);
        sp_2048_mont_sqr_32(t[ 6], t[ 3], m, mp);
        sp_2048_mont_mul_32(t[ 7], t[ 4], t[ 3], m, mp);
        sp_2048_mont_sqr_32(t[ 8], t[ 4], m, mp);
        sp_2048_mont_mul_32(t[ 9], t[ 5], t[ 4], m, mp);
        sp_2048_mont_sqr_32(t[10], t[ 5], m, mp);
        sp_2048_mont_mul_32(t[11], t[ 6], t[ 5], m, mp);
        sp_2048_mont_sqr_32(t[12], t[ 6], m, mp);
        sp_2048_mont_mul_32(t[13], t[ 7], t[ 6], m, mp);
        sp_2048_mont_sqr_32(t[14], t[ 7], m, mp);
        sp_2048_mont_mul_32(t[15], t[ 8], t[ 7], m, mp);
        sp_2048_mont_sqr_32(t[16], t[ 8], m, mp);
        sp_2048_mont_mul_32(t[17], t[ 9], t[ 8], m, mp);
        sp_2048_mont_sqr_32(t[18], t[ 9], m, mp);
        sp_2048_mont_mul_32(t[19], t[10], t[ 9], m, mp);
        sp_2048_mont_sqr_32(t[20], t[10], m, mp);
        sp_2048_mont_mul_32(t[21], t[11], t[10], m, mp);
        sp_2048_mont_sqr_32(t[22], t[11], m, mp);
        sp_2048_mont_mul_32(t[23], t[12], t[11], m, mp);
        sp_2048_mont_sqr_32(t[24], t[12], m, mp);
        sp_2048_mont_mul_32(t[25], t[13], t[12], m, mp);
        sp_2048_mont_sqr_32(t[26], t[13], m, mp);
        sp_2048_mont_mul_32(t[27], t[14], t[13], m, mp);
        sp_2048_mont_sqr_32(t[28], t[14], m, mp);
        sp_2048_mont_mul_32(t[29], t[15], t[14], m, mp);
        sp_2048_mont_sqr_32(t[30], t[15], m, mp);
        sp_2048_mont_mul_32(t[31], t[16], t[15], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 5;
        if (c == 64) {
            c = 59;
        }
        if (c < 0) {
            /* Number of bits in top word is less than number needed. */
            c = -c;
            y = (byte)(n << c);
            n = e[i--];
            y |= (byte)(n >> (64 - c));
            n <<= c;
            c = 64 - c;
        }
        else if (c == 0) {
            /* All bits in top word used. */
            y = (byte)n;
        }
        else {
            y = (byte)(n >> c);
            n <<= 64 - c;
        }
        XMEMCPY(r, t[y], sizeof(sp_digit) * 32);
        for (; i>=0 || c>=5; ) {
            if (c == 0) {
                n = e[i--];
                y = (byte)(n >> 59);
                n <<= 5;
                c = 59;
            }
            else if (c < 5) {
                y = (byte)(n >> 59);
                n = e[i--];
                c = 5 - c;
                y |= (byte)(n >> (64 - c));
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (byte)((n >> 59) & 0x1f);
                n <<= 5;
                c -= 5;
            }

            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);

            sp_2048_mont_mul_32(r, r, t[y], m, mp);
        }

        XMEMSET(&r[32], 0, sizeof(sp_digit) * 32U);
        sp_2048_mont_reduce_32(r, m, mp);

        mask = 0 - (sp_2048_cmp_32(r, m) >= 0);
        sp_2048_cond_sub_32(r, r, m, mask);
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (td != NULL)
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return err;
}
#else
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns  0 on success.
 * returns  MEMORY_E on dynamic memory allocation failure.
 * returns  MP_VAL when base is even or exponent is 0.
 */
static int sp_2048_mod_exp_32(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* td = NULL;
#else
    sp_digit td[64 * 64];
#endif
    sp_digit* t[64];
    sp_digit* norm = NULL;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c;
    byte y;
    int err = MP_OKAY;

    if (bits == 0) {
        err = MP_VAL;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        td = (sp_digit*)XMALLOC(sizeof(sp_digit) * (64 * 64), NULL,
                                DYNAMIC_TYPE_TMP_BUFFER);
        if (td == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        norm = td;
        for (i=0; i<64; i++) {
            t[i] = td + i * 64;
        }

        sp_2048_mont_setup(m, &mp);
        sp_2048_mont_norm_32(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 32U);
        if (reduceA != 0) {
            err = sp_2048_mod_32(t[1] + 32, a, m);
            if (err == MP_OKAY) {
                err = sp_2048_mod_32(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 32, a, sizeof(sp_digit) * 32);
            err = sp_2048_mod_32(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_2048_mont_sqr_32(t[ 2], t[ 1], m, mp);
        sp_2048_mont_mul_32(t[ 3], t[ 2], t[ 1], m, mp);
        sp_2048_mont_sqr_32(t[ 4], t[ 2], m, mp);
        sp_2048_mont_mul_32(t[ 5], t[ 3], t[ 2], m, mp);
        sp_2048_mont_sqr_32(t[ 6], t[ 3], m, mp);
        sp_2048_mont_mul_32(t[ 7], t[ 4], t[ 3], m, mp);
        sp_2048_mont_sqr_32(t[ 8], t[ 4], m, mp);
        sp_2048_mont_mul_32(t[ 9], t[ 5], t[ 4], m, mp);
        sp_2048_mont_sqr_32(t[10], t[ 5], m, mp);
        sp_2048_mont_mul_32(t[11], t[ 6], t[ 5], m, mp);
        sp_2048_mont_sqr_32(t[12], t[ 6], m, mp);
        sp_2048_mont_mul_32(t[13], t[ 7], t[ 6], m, mp);
        sp_2048_mont_sqr_32(t[14], t[ 7], m, mp);
        sp_2048_mont_mul_32(t[15], t[ 8], t[ 7], m, mp);
        sp_2048_mont_sqr_32(t[16], t[ 8], m, mp);
        sp_2048_mont_mul_32(t[17], t[ 9], t[ 8], m, mp);
        sp_2048_mont_sqr_32(t[18], t[ 9], m, mp);
        sp_2048_mont_mul_32(t[19], t[10], t[ 9], m, mp);
        sp_2048_mont_sqr_32(t[20], t[10], m, mp);
        sp_2048_mont_mul_32(t[21], t[11], t[10], m, mp);
        sp_2048_mont_sqr_32(t[22], t[11], m, mp);
        sp_2048_mont_mul_32(t[23], t[12], t[11], m, mp);
        sp_2048_mont_sqr_32(t[24], t[12], m, mp);
        sp_2048_mont_mul_32(t[25], t[13], t[12], m, mp);
        sp_2048_mont_sqr_32(t[26], t[13], m, mp);
        sp_2048_mont_mul_32(t[27], t[14], t[13], m, mp);
        sp_2048_mont_sqr_32(t[28], t[14], m, mp);
        sp_2048_mont_mul_32(t[29], t[15], t[14], m, mp);
        sp_2048_mont_sqr_32(t[30], t[15], m, mp);
        sp_2048_mont_mul_32(t[31], t[16], t[15], m, mp);
        sp_2048_mont_sqr_32(t[32], t[16], m, mp);
        sp_2048_mont_mul_32(t[33], t[17], t[16], m, mp);
        sp_2048_mont_sqr_32(t[34], t[17], m, mp);
        sp_2048_mont_mul_32(t[35], t[18], t[17], m, mp);
        sp_2048_mont_sqr_32(t[36], t[18], m, mp);
        sp_2048_mont_mul_32(t[37], t[19], t[18], m, mp);
        sp_2048_mont_sqr_32(t[38], t[19], m, mp);
        sp_2048_mont_mul_32(t[39], t[20], t[19], m, mp);
        sp_2048_mont_sqr_32(t[40], t[20], m, mp);
        sp_2048_mont_mul_32(t[41], t[21], t[20], m, mp);
        sp_2048_mont_sqr_32(t[42], t[21], m, mp);
        sp_2048_mont_mul_32(t[43], t[22], t[21], m, mp);
        sp_2048_mont_sqr_32(t[44], t[22], m, mp);
        sp_2048_mont_mul_32(t[45], t[23], t[22], m, mp);
        sp_2048_mont_sqr_32(t[46], t[23], m, mp);
        sp_2048_mont_mul_32(t[47], t[24], t[23], m, mp);
        sp_2048_mont_sqr_32(t[48], t[24], m, mp);
        sp_2048_mont_mul_32(t[49], t[25], t[24], m, mp);
        sp_2048_mont_sqr_32(t[50], t[25], m, mp);
        sp_2048_mont_mul_32(t[51], t[26], t[25], m, mp);
        sp_2048_mont_sqr_32(t[52], t[26], m, mp);
        sp_2048_mont_mul_32(t[53], t[27], t[26], m, mp);
        sp_2048_mont_sqr_32(t[54], t[27], m, mp);
        sp_2048_mont_mul_32(t[55], t[28], t[27], m, mp);
        sp_2048_mont_sqr_32(t[56], t[28], m, mp);
        sp_2048_mont_mul_32(t[57], t[29], t[28], m, mp);
        sp_2048_mont_sqr_32(t[58], t[29], m, mp);
        sp_2048_mont_mul_32(t[59], t[30], t[29], m, mp);
        sp_2048_mont_sqr_32(t[60], t[30], m, mp);
        sp_2048_mont_mul_32(t[61], t[31], t[30], m, mp);
        sp_2048_mont_sqr_32(t[62], t[31], m, mp);
        sp_2048_mont_mul_32(t[63], t[32], t[31], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 6;
        if (c == 64) {
            c = 58;
        }
        if (c < 0) {
            /* Number of bits in top word is less than number needed. */
            c = -c;
            y = (byte)(n << c);
            n = e[i--];
            y |= (byte)(n >> (64 - c));
            n <<= c;
            c = 64 - c;
        }
        else if (c == 0) {
            /* All bits in top word used. */
            y = (byte)n;
        }
        else {
            y = (byte)(n >> c);
            n <<= 64 - c;
        }
        XMEMCPY(r, t[y], sizeof(sp_digit) * 32);
        for (; i>=0 || c>=6; ) {
            if (c == 0) {
                n = e[i--];
                y = (byte)(n >> 58);
                n <<= 6;
                c = 58;
            }
            else if (c < 6) {
                y = (byte)(n >> 58);
                n = e[i--];
                c = 6 - c;
                y |= (byte)(n >> (64 - c));
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (byte)((n >> 58) & 0x3f);
                n <<= 6;
                c -= 6;
            }

            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);

            sp_2048_mont_mul_32(r, r, t[y], m, mp);
        }

        XMEMSET(&r[32], 0, sizeof(sp_digit) * 32U);
        sp_2048_mont_reduce_32(r, m, mp);

        mask = 0 - (sp_2048_cmp_32(r, m) >= 0);
        sp_2048_cond_sub_32(r, r, m, mask);
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (td != NULL)
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return err;
}
#endif /* WOLFSSL_SP_SMALL */
#endif /* (WOLFSSL_HAVE_SP_RSA && !WOLFSSL_RSA_PUBLIC_ONLY) || WOLFSSL_HAVE_SP_DH */

#endif /* (WOLFSSL_HAVE_SP_RSA && !WOLFSSL_RSA_PUBLIC_ONLY) || WOLFSSL_HAVE_SP_DH */
#ifdef WOLFSSL_HAVE_SP_RSA
/* RSA public key operation.
 *
 * in      Array of bytes representing the number to exponentiate, base.
 * inLen   Number of bytes in base.
 * em      Public exponent.
 * mm      Modulus.
 * out     Buffer to hold big-endian bytes of exponentiation result.
 *         Must be at least 256 bytes long.
 * outLen  Number of bytes in result.
 * returns 0 on success, MP_TO_E when the outLen is too small, MP_READ_E when
 * an array is too long and MEMORY_E when dynamic memory allocation fails.
 */
int sp_RsaPublic_2048(const byte* in, word32 inLen, const mp_int* em,
    const mp_int* mm, byte* out, word32* outLen)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* a = NULL;
#else
    sp_digit a[32 * 5];
#endif
    sp_digit* m = NULL;
    sp_digit* r = NULL;
    sp_digit *ah = NULL;
    sp_digit e[1] = {0};
    int err = MP_OKAY;

    if (*outLen < 256) {
        err = MP_TO_E;
    }
    else if (mp_count_bits(em) > 64 || inLen > 256 ||
                                                     mp_count_bits(mm) != 2048) {
        err = MP_READ_E;
    }
    else if (mp_iseven(mm)) {
        err = MP_VAL;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        a = (sp_digit*)XMALLOC(sizeof(sp_digit) * 32 * 5, NULL,
                                                              DYNAMIC_TYPE_RSA);
        if (a == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        ah = a + 32;
        r = a + 32 * 2;
        m = r + 32 * 2;

        sp_2048_from_bin(ah, 32, in, inLen);
#if DIGIT_BIT >= 64
        e[0] = em->dp[0];
#else
        e[0] = em->dp[0];
        if (em->used > 1) {
            e[0] |= ((sp_digit)em->dp[1]) << DIGIT_BIT;
        }
#endif
        if (e[0] == 0) {
            err = MP_EXPTMOD_E;
        }
    }
    if (err == MP_OKAY) {
        sp_2048_from_mp(m, 32, mm);

        if (e[0] == 0x10001) {
            int i;
            sp_digit mp;

            sp_2048_mont_setup(m, &mp);

            /* Convert to Montgomery form. */
            XMEMSET(a, 0, sizeof(sp_digit) * 32);
            err = sp_2048_mod_32_cond(r, a, m);
            /* Montgomery form: r = a.R mod m */

            if (err == MP_OKAY) {
                /* r = a ^ 0x10000 => r = a squared 16 times */
                for (i = 15; i >= 0; i--) {
                    sp_2048_mont_sqr_32(r, r, m, mp);
                }
                /* mont_red(r.R.R) = (r.R.R / R) mod m = r.R mod m
                 * mont_red(r.R * a) = (r.R.a / R) mod m = r.a mod m
                 */
                sp_2048_mont_mul_32(r, r, ah, m, mp);

                for (i = 31; i > 0; i--) {
                    if (r[i] != m[i]) {
                        break;
                    }
                }
                if (r[i] >= m[i]) {
                    sp_2048_sub_in_place_32(r, m);
                }
            }
        }
        else if (e[0] == 0x3) {
            if (err == MP_OKAY) {
                sp_2048_sqr_32(r, ah);
                err = sp_2048_mod_32_cond(r, r, m);
            }
            if (err == MP_OKAY) {
                sp_2048_mul_32(r, ah, r);
                err = sp_2048_mod_32_cond(r, r, m);
            }
        }
        else {
            int i;
            sp_digit mp;

            sp_2048_mont_setup(m, &mp);

            /* Convert to Montgomery form. */
            XMEMSET(a, 0, sizeof(sp_digit) * 32);
            err = sp_2048_mod_32_cond(a, a, m);

            if (err == MP_OKAY) {
                for (i = 63; i >= 0; i--) {
                    if (e[0] >> i) {
                        break;
                    }
                }

                XMEMCPY(r, a, sizeof(sp_digit) * 32);
                for (i--; i >= 0; i--) {
                    sp_2048_mont_sqr_32(r, r, m, mp);
                    if (((e[0] >> i) & 1) == 1) {
                        sp_2048_mont_mul_32(r, r, a, m, mp);
                    }
                }
                XMEMSET(&r[32], 0, sizeof(sp_digit) * 32);
                sp_2048_mont_reduce_32(r, m, mp);

                for (i = 31; i > 0; i--) {
                    if (r[i] != m[i]) {
                        break;
                    }
                }
                if (r[i] >= m[i]) {
                    sp_2048_sub_in_place_32(r, m);
                }
            }
        }
    }

    if (err == MP_OKAY) {
        sp_2048_to_bin_32(r, out);
        *outLen = 256;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (a != NULL)
        XFREE(a, NULL, DYNAMIC_TYPE_RSA);
#endif

    return err;
}

#ifndef WOLFSSL_RSA_PUBLIC_ONLY
#ifdef WOLFSSL_SP_SMALL
/* Conditionally add a and b using the mask m.
 * m is -1 to add and 0 when not.
 *
 * r  A single precision number representing conditional add result.
 * a  A single precision number to add with.
 * b  A single precision number to add.
 * m  Mask value to apply.
 */
static sp_digit sp_2048_cond_add_16(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "mov	x8, #0\n\t"
        "1:\n\t"
        "adds	%[c], %[c], #-1\n\t"
        "ldr	x4, [%[a], x8]\n\t"
        "ldr	x5, [%[b], x8]\n\t"
        "and	x5, x5, %[m]\n\t"
        "adcs	x4, x4, x5\n\t"
        "cset	%[c], cs\n\t"
        "str	x4, [%[r], x8]\n\t"
        "add	x8, x8, #8\n\t"
        "cmp	x8, 128\n\t"
        "b.lt	1b\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x5", "x8", "x9", "x10", "x11", "x12"
    );

    return c;
}
#endif /* WOLFSSL_SP_SMALL */

/* RSA private key operation.
 *
 * in      Array of bytes representing the number to exponentiate, base.
 * inLen   Number of bytes in base.
 * dm      Private exponent.
 * pm      First prime.
 * qm      Second prime.
 * dpm     First prime's CRT exponent.
 * dqm     Second prime's CRT exponent.
 * qim     Inverse of second prime mod p.
 * mm      Modulus.
 * out     Buffer to hold big-endian bytes of exponentiation result.
 *         Must be at least 256 bytes long.
 * outLen  Number of bytes in result.
 * returns 0 on success, MP_TO_E when the outLen is too small, MP_READ_E when
 * an array is too long and MEMORY_E when dynamic memory allocation fails.
 */
int sp_RsaPrivate_2048(const byte* in, word32 inLen, const mp_int* dm,
    const mp_int* pm, const mp_int* qm, const mp_int* dpm, const mp_int* dqm,
    const mp_int* qim, const mp_int* mm, byte* out, word32* outLen)
{
#if defined(SP_RSA_PRIVATE_EXP_D) || defined(RSA_LOW_MEM)
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* d = NULL;
#else
    sp_digit  d[32 * 4];
#endif
    sp_digit* a = NULL;
    sp_digit* m = NULL;
    sp_digit* r = NULL;
    int err = MP_OKAY;

    (void)pm;
    (void)qm;
    (void)dpm;
    (void)dqm;
    (void)qim;

    if (*outLen < 256U) {
        err = MP_TO_E;
    }
    if (err == MP_OKAY) {
        if (mp_count_bits(dm) > 2048) {
           err = MP_READ_E;
        }
        else if (inLen > 256) {
            err = MP_READ_E;
        }
        else if (mp_count_bits(mm) != 2048) {
            err = MP_READ_E;
        }
        else if (mp_iseven(mm)) {
            err = MP_VAL;
        }
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        d = (sp_digit*)XMALLOC(sizeof(sp_digit) * 32 * 4, NULL,
                                                              DYNAMIC_TYPE_RSA);
        if (d == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        a = d + 32;
        m = a + 64;
        r = a;

        sp_2048_from_bin(a, 32, in, inLen);
        sp_2048_from_mp(d, 32, dm);
        sp_2048_from_mp(m, 32, mm);
        err = sp_2048_mod_exp_32(r, a, d, 2048, m, 0);
    }

    if (err == MP_OKAY) {
        sp_2048_to_bin_32(r, out);
        *outLen = 256;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (d != NULL)
#endif
    {
        /* only "a" and "r" are sensitive and need zeroized (same pointer) */
        if (a != NULL)
            ForceZero(a, sizeof(sp_digit) * 32);
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
        XFREE(d, NULL, DYNAMIC_TYPE_RSA);
#endif
    }

    return err;
#else
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* a = NULL;
#else
    sp_digit a[16 * 11];
#endif
    sp_digit* p = NULL;
    sp_digit* q = NULL;
    sp_digit* dp = NULL;
    sp_digit* tmpa = NULL;
    sp_digit* tmpb = NULL;
    sp_digit* r = NULL;
    sp_digit* qi = NULL;
    sp_digit* dq = NULL;
    sp_digit c;
    int err = MP_OKAY;

    (void)dm;
    (void)mm;

    if (*outLen < 256) {
        err = MP_TO_E;
    }
    else if (inLen > 256 || mp_count_bits(mm) != 2048) {
        err = MP_READ_E;
    }
    else if (mp_iseven(mm)) {
        err = MP_VAL;
    }
    else if (mp_iseven(pm)) {
        err = MP_VAL;
    }
    else if (mp_iseven(qm)) {
        err = MP_VAL;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        a = (sp_digit*)XMALLOC(sizeof(sp_digit) * 16 * 11, NULL,
                                                              DYNAMIC_TYPE_RSA);
        if (a == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        p = a + 32 * 2;
        q = p + 16;
        qi = dq = dp = q + 16;
        tmpa = qi + 16;
        tmpb = tmpa + 32;
        r = a;

        sp_2048_from_bin(a, 32, in, inLen);
        sp_2048_from_mp(p, 16, pm);
        sp_2048_from_mp(q, 16, qm);
        sp_2048_from_mp(dp, 16, dpm);

        err = sp_2048_mod_exp_16(tmpa, a, dp, 1024, p, 1);
    }
    if (err == MP_OKAY) {
        sp_2048_from_mp(dq, 16, dqm);
        err = sp_2048_mod_exp_16(tmpb, a, dq, 1024, q, 1);
    }

    if (err == MP_OKAY) {
        c = sp_2048_sub_in_place_16(tmpa, tmpb);
        c += sp_2048_cond_add_16(tmpa, tmpa, p, c);
        sp_2048_cond_add_16(tmpa, tmpa, p, c);

        sp_2048_from_mp(qi, 16, qim);
        sp_2048_mul_16(tmpa, tmpa, qi);
        err = sp_2048_mod_16(tmpa, tmpa, p);
    }

    if (err == MP_OKAY) {
        sp_2048_mul_16(tmpa, q, tmpa);
        XMEMSET(&tmpb[16], 0, sizeof(sp_digit) * 16);
        sp_2048_add_32(r, tmpb, tmpa);

        sp_2048_to_bin_32(r, out);
        *outLen = 256;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (a != NULL)
#endif
    {
        ForceZero(a, sizeof(sp_digit) * 16 * 11);
    #if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
        XFREE(a, NULL, DYNAMIC_TYPE_RSA);
    #endif
    }
#endif /* SP_RSA_PRIVATE_EXP_D || RSA_LOW_MEM */
    return err;
}
#endif /* WOLFSSL_RSA_PUBLIC_ONLY */
#endif /* WOLFSSL_HAVE_SP_RSA */
#if defined(WOLFSSL_HAVE_SP_DH) || (defined(WOLFSSL_HAVE_SP_RSA) && \
                                              !defined(WOLFSSL_RSA_PUBLIC_ONLY))
/* Convert an array of sp_digit to an mp_int.
 *
 * a  A single precision integer.
 * r  A multi-precision integer.
 */
static int sp_2048_to_mp(const sp_digit* a, mp_int* r)
{
    int err;

    err = mp_grow(r, (2048 + DIGIT_BIT - 1) / DIGIT_BIT);
    if (err == MP_OKAY) { /*lint !e774 case where err is always MP_OKAY*/
#if DIGIT_BIT == 64
        XMEMCPY(r->dp, a, sizeof(sp_digit) * 32);
        r->used = 32;
        mp_clamp(r);
#elif DIGIT_BIT < 64
        int i;
        int j = 0;
        int s = 0;

        r->dp[0] = 0;
        for (i = 0; i < 32; i++) {
            r->dp[j] |= (mp_digit)(a[i] << s);
            r->dp[j] &= ((sp_digit)1 << DIGIT_BIT) - 1;
            s = DIGIT_BIT - s;
            r->dp[++j] = (mp_digit)(a[i] >> s);
            while (s + DIGIT_BIT <= 64) {
                s += DIGIT_BIT;
                r->dp[j++] &= ((sp_digit)1 << DIGIT_BIT) - 1;
                if (s == SP_WORD_SIZE) {
                    r->dp[j] = 0;
                }
                else {
                    r->dp[j] = (mp_digit)(a[i] >> s);
                }
            }
            s = 64 - s;
        }
        r->used = (2048 + DIGIT_BIT - 1) / DIGIT_BIT;
        mp_clamp(r);
#else
        int i;
        int j = 0;
        int s = 0;

        r->dp[0] = 0;
        for (i = 0; i < 32; i++) {
            r->dp[j] |= ((mp_digit)a[i]) << s;
            if (s + 64 >= DIGIT_BIT) {
    #if DIGIT_BIT != 32 && DIGIT_BIT != 64
                r->dp[j] &= ((sp_digit)1 << DIGIT_BIT) - 1;
    #endif
                s = DIGIT_BIT - s;
                r->dp[++j] = a[i] >> s;
                s = 64 - s;
            }
            else {
                s += 64;
            }
        }
        r->used = (2048 + DIGIT_BIT - 1) / DIGIT_BIT;
        mp_clamp(r);
#endif
    }

    return err;
}

/* Perform the modular exponentiation for Diffie-Hellman.
 *
 * base  Base. MP integer.
 * exp   Exponent. MP integer.
 * mod   Modulus. MP integer.
 * res   Result. MP integer.
 * returns 0 on success, MP_READ_E if there are too many bytes in an array
 * and MEMORY_E if memory allocation fails.
 */
int sp_ModExp_2048(const mp_int* base, const mp_int* exp, const mp_int* mod,
    mp_int* res)
{
    int err = MP_OKAY;
    sp_digit b[64];
    sp_digit e[32];
    sp_digit m[32];
    sp_digit* r = b;
    int expBits = mp_count_bits(exp);

    if (mp_count_bits(base) > 2048) {
        err = MP_READ_E;
    }
    else if (expBits > 2048) {
        err = MP_READ_E;
    }
    else if (mp_count_bits(mod) != 2048) {
        err = MP_READ_E;
    }
    else if (mp_iseven(mod)) {
        err = MP_VAL;
    }

    if (err == MP_OKAY) {
        sp_2048_from_mp(b, 32, base);
        sp_2048_from_mp(e, 32, exp);
        sp_2048_from_mp(m, 32, mod);

        err = sp_2048_mod_exp_32(r, b, e, expBits, m, 0);
    }

    if (err == MP_OKAY) {
        err = sp_2048_to_mp(r, res);
    }

    XMEMSET(e, 0, sizeof(e));

    return err;
}

#ifdef WOLFSSL_HAVE_SP_DH

#ifdef HAVE_FFDHE_2048
static void sp_2048_lshift_32(sp_digit* r, const sp_digit* a, byte n)
{
    word64 n64 = n;
    __asm__ __volatile__ (
        "mov	x6, 63\n\t"
        "sub	x6, x6, %[n]\n\t"
        "ldr	x3, [%[a], 248]\n\t"
        "lsr	x4, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x4, x4, x6\n\t"
        "ldr	x2, [%[a], 240]\n\t"
        "str	x4, [%[r], 256]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 232]\n\t"
        "str	x3, [%[r], 248]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 224]\n\t"
        "str	x2, [%[r], 240]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 216]\n\t"
        "str	x4, [%[r], 232]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 208]\n\t"
        "str	x3, [%[r], 224]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 200]\n\t"
        "str	x2, [%[r], 216]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 192]\n\t"
        "str	x4, [%[r], 208]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 184]\n\t"
        "str	x3, [%[r], 200]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 176]\n\t"
        "str	x2, [%[r], 192]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 168]\n\t"
        "str	x4, [%[r], 184]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 160]\n\t"
        "str	x3, [%[r], 176]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 152]\n\t"
        "str	x2, [%[r], 168]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 144]\n\t"
        "str	x4, [%[r], 160]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 136]\n\t"
        "str	x3, [%[r], 152]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 128]\n\t"
        "str	x2, [%[r], 144]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 120]\n\t"
        "str	x4, [%[r], 136]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 112]\n\t"
        "str	x3, [%[r], 128]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 104]\n\t"
        "str	x2, [%[r], 120]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 96]\n\t"
        "str	x4, [%[r], 112]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 88]\n\t"
        "str	x3, [%[r], 104]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 80]\n\t"
        "str	x2, [%[r], 96]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 72]\n\t"
        "str	x4, [%[r], 88]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 64]\n\t"
        "str	x3, [%[r], 80]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 56]\n\t"
        "str	x2, [%[r], 72]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 48]\n\t"
        "str	x4, [%[r], 64]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 40]\n\t"
        "str	x3, [%[r], 56]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 32]\n\t"
        "str	x2, [%[r], 48]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 24]\n\t"
        "str	x4, [%[r], 40]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 16]\n\t"
        "str	x3, [%[r], 32]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 8]\n\t"
        "str	x2, [%[r], 24]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 0]\n\t"
        "str	x4, [%[r], 16]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "stp	x2, x3, [%[r]]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [n] "r" (n64)
        : "memory", "x2", "x3", "x4", "x5", "x6"
    );
}

/* Modular exponentiate 2 to the e mod m. (r = 2^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns  0 on success.
 * returns  MEMORY_E on dynamic memory allocation failure.
 * returns  MP_VAL when base is even.
 */
static int sp_2048_mod_exp_2_32(sp_digit* r, const sp_digit* e, int bits,
        const sp_digit* m)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* td = NULL;
#else
    sp_digit td[97];
#endif
    sp_digit* norm = NULL;
    sp_digit* tmp = NULL;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit o;
    sp_digit mask;
    int i;
    int c;
    byte y;
    int err = MP_OKAY;

    if (bits == 0) {
        err = MP_VAL;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        td = (sp_digit*)XMALLOC(sizeof(sp_digit) * 97, NULL,
                                DYNAMIC_TYPE_TMP_BUFFER);
        if (td == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        norm = td;
        tmp = td + 64;

        sp_2048_mont_setup(m, &mp);
        sp_2048_mont_norm_32(norm, m);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 6;
        if (c == 64) {
            c = 58;
        }
        if (c < 0) {
            /* Number of bits in top word is less than number needed. */
            c = -c;
            y = (byte)(n << c);
            n = e[i--];
            y |= (byte)(n >> (64 - c));
            n <<= c;
            c = 64 - c;
        }
        else if (c == 0) {
            /* All bits in top word used. */
            y = (byte)n;
        }
        else {
            y = (byte)(n >> c);
            n <<= 64 - c;
        }
        sp_2048_lshift_32(r, norm, y);
        for (; i>=0 || c>=6; ) {
            if (c == 0) {
                n = e[i--];
                y = (byte)(n >> 58);
                n <<= 6;
                c = 58;
            }
            else if (c < 6) {
                y = (byte)(n >> 58);
                n = e[i--];
                c = 6 - c;
                y |= (byte)(n >> (64 - c));
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (byte)((n >> 58) & 0x3f);
                n <<= 6;
                c -= 6;
            }

            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);
            sp_2048_mont_sqr_32(r, r, m, mp);

            sp_2048_lshift_32(r, r, y);
            sp_2048_mul_d_32(tmp, norm, r[32]);
            r[32] = 0;
            o = sp_2048_add_32(r, r, tmp);
            sp_2048_cond_sub_32(r, r, m, (sp_digit)0 - o);
        }

        XMEMSET(&r[32], 0, sizeof(sp_digit) * 32U);
        sp_2048_mont_reduce_32(r, m, mp);

        mask = 0 - (sp_2048_cmp_32(r, m) >= 0);
        sp_2048_cond_sub_32(r, r, m, mask);
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (td != NULL)
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return err;
}
#endif /* HAVE_FFDHE_2048 */

/* Perform the modular exponentiation for Diffie-Hellman.
 *
 * base     Base.
 * exp      Array of bytes that is the exponent.
 * expLen   Length of data, in bytes, in exponent.
 * mod      Modulus.
 * out      Buffer to hold big-endian bytes of exponentiation result.
 *          Must be at least 256 bytes long.
 * outLen   Length, in bytes, of exponentiation result.
 * returns 0 on success, MP_READ_E if there are too many bytes in an array
 * and MEMORY_E if memory allocation fails.
 */
int sp_DhExp_2048(const mp_int* base, const byte* exp, word32 expLen,
    const mp_int* mod, byte* out, word32* outLen)
{
    int err = MP_OKAY;
    sp_digit b[64];
    sp_digit e[32];
    sp_digit m[32];
    sp_digit* r = b;
    word32 i;

    if (mp_count_bits(base) > 2048) {
        err = MP_READ_E;
    }
    else if (expLen > 256) {
        err = MP_READ_E;
    }
    else if (mp_count_bits(mod) != 2048) {
        err = MP_READ_E;
    }
    else if (mp_iseven(mod)) {
        err = MP_VAL;
    }

    if (err == MP_OKAY) {
        sp_2048_from_mp(b, 32, base);
        sp_2048_from_bin(e, 32, exp, expLen);
        sp_2048_from_mp(m, 32, mod);

    #ifdef HAVE_FFDHE_2048
        if (base->used == 1 && base->dp[0] == 2 && m[31] == (sp_digit)-1)
            err = sp_2048_mod_exp_2_32(r, e, expLen * 8, m);
        else
    #endif
            err = sp_2048_mod_exp_32(r, b, e, expLen * 8, m, 0);

    }

    if (err == MP_OKAY) {
        sp_2048_to_bin_32(r, out);
        *outLen = 256;
        for (i=0; i<256 && out[i] == 0; i++) {
            /* Search for first non-zero. */
        }
        *outLen -= i;
        XMEMMOVE(out, out + i, *outLen);

    }

    XMEMSET(e, 0, sizeof(e));

    return err;
}
#endif /* WOLFSSL_HAVE_SP_DH */

/* Perform the modular exponentiation for Diffie-Hellman.
 *
 * base  Base. MP integer.
 * exp   Exponent. MP integer.
 * mod   Modulus. MP integer.
 * res   Result. MP integer.
 * returns 0 on success, MP_READ_E if there are too many bytes in an array
 * and MEMORY_E if memory allocation fails.
 */
int sp_ModExp_1024(const mp_int* base, const mp_int* exp, const mp_int* mod,
    mp_int* res)
{
    int err = MP_OKAY;
    sp_digit b[32];
    sp_digit e[16];
    sp_digit m[16];
    sp_digit* r = b;
    int expBits = mp_count_bits(exp);

    if (mp_count_bits(base) > 1024) {
        err = MP_READ_E;
    }
    else if (expBits > 1024) {
        err = MP_READ_E;
    }
    else if (mp_count_bits(mod) != 1024) {
        err = MP_READ_E;
    }
    else if (mp_iseven(mod)) {
        err = MP_VAL;
    }

    if (err == MP_OKAY) {
        sp_2048_from_mp(b, 16, base);
        sp_2048_from_mp(e, 16, exp);
        sp_2048_from_mp(m, 16, mod);

        err = sp_2048_mod_exp_16(r, b, e, expBits, m, 0);
    }

    if (err == MP_OKAY) {
        XMEMSET(r + 16, 0, sizeof(*r) * 16U);
        err = sp_2048_to_mp(r, res);
        res->used = mod->used;
        mp_clamp(res);
    }

    XMEMSET(e, 0, sizeof(e));

    return err;
}

#endif /* WOLFSSL_HAVE_SP_DH | (WOLFSSL_HAVE_SP_RSA & !WOLFSSL_RSA_PUBLIC_ONLY) */

#endif /* !WOLFSSL_SP_NO_2048 */

#ifndef WOLFSSL_SP_NO_3072
/* Read big endian unsigned byte array into r.
 *
 * r  A single precision integer.
 * size  Maximum number of bytes to convert
 * a  Byte array.
 * n  Number of bytes in array to read.
 */
static void sp_3072_from_bin(sp_digit* r, int size, const byte* a, int n)
{
    sp_int64 nl = n;
    sp_int64 bytes = size * 8;

    __asm__ __volatile__ (
        "add	x4, %[a], %[n]\n\t"
        "mov	x5, %[r]\n\t"
        "sub	x4, x4, 8\n\t"
        "subs	x6, %[n], 8\n\t"
        "mov	x7, xzr\n\t"
        "blt	2f\n\t"
        /* Put in mulitples of 8 bytes. */
        "1:\n\t"
        "ldr	x8, [x4], -8\n\t"
        "subs	x6, x6, 8\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "rev	x8, x8\n\t"
    #endif
        "str	x8, [x5], 8\n\t"
        "add	x7, x7, 8\n\t"
        "b.ge	1b\n\t"
        "2:\n\t"
        "cmp	x6, -7\n\t"
        "b.lt	20f\n\t"
        /* Put in less than 8 bytes. */
    #ifdef LITTLE_ENDIAN_ORDER
        "str	xzr, [x5]\n\t"
    #else
        "str	xzr, [x5], 7\n\t"
    #endif
        "add	x7, x7, 8\n\t"
        "add	x4, x4, 7\n\t"
        "b.eq	17f\n\t"
        "cmp	x6, -5\n\t"
        "b.lt	16f\n\t"
        "b.eq	15f\n\t"
        "cmp	x6, -3\n\t"
        "b.lt	14f\n\t"
        "b.eq	13f\n\t"
        "cmp	x6, -2\n\t"
        "b.eq	12f\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "12:\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "13:\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "14:\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "15:\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "16:\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "17:\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "20:\n\t"
        "add	x5, %[r], x7\n\t"
        "subs	x7, %[size], x7\n\t"
        "b.eq	30f\n\t"
        /* Zero out remaining words. */
        "21:\n\t"
        "subs	x7, x7, 8\n\t"
        "str	xzr, [x5], 8\n\t"
        "b.gt	21b\n\t"
        "30:\n\t"
        :
        : [r] "r" (r), [size] "r" (bytes), [a] "r" (a), [n] "r" (nl)
        : "memory", "x4", "x5", "x6", "x7", "x8"
    );
}

/* Convert an mp_int to an array of sp_digit.
 *
 * r  A single precision integer.
 * size  Maximum number of bytes to convert
 * a  A multi-precision integer.
 */
static void sp_3072_from_mp(sp_digit* r, int size, const mp_int* a)
{
#if DIGIT_BIT == 64
    int i;
    int j = 0;

    for (i = 0; i < size; i++) {
        sp_digit mask =
            (((sp_digit)(a->used - i - 1)) >> (SP_WORD_SIZE - 1)) - 1;
        r[i] = a->dp[j] & mask;
        j += (int)(((sp_digit)1) -
                   (((sp_digit)(a->used - i - 2)) >> (SP_WORD_SIZE - 1)));
    }
#elif DIGIT_BIT > 64
    int i;
    int j = 0;
    word32 s = 0;

    r[0] = 0;
    for (i = 0; i < a->used && j < size; i++) {
        r[j] |= ((sp_digit)a->dp[i] << s);
        r[j] &= 0xffffffffffffffffl;
        s = 64U - s;
        if (j + 1 >= size) {
            break;
        }
        /* lint allow cast of mismatch word32 and mp_digit */
        r[++j] = (sp_digit)(a->dp[i] >> s); /*lint !e9033*/
        while ((s + 64U) <= (word32)DIGIT_BIT) {
            s += 64U;
            r[j] &= 0xffffffffffffffffl;
            if (j + 1 >= size) {
                break;
            }
            if (s < (word32)DIGIT_BIT) {
                /* lint allow cast of mismatch word32 and mp_digit */
                r[++j] = (sp_digit)(a->dp[i] >> s); /*lint !e9033*/
            }
            else {
                r[++j] = (sp_digit)0;
            }
        }
        s = (word32)DIGIT_BIT - s;
    }

    for (j++; j < size; j++) {
        r[j] = 0;
    }
#else
    int i;
    int j = 0;
    int s = 0;

    r[0] = 0;
    for (i = 0; i < a->used && j < size; i++) {
        r[j] |= ((sp_digit)a->dp[i]) << s;
        if (s + DIGIT_BIT >= 64) {
            r[j] &= 0xffffffffffffffffl;
            if (j + 1 >= size) {
                break;
            }
            s = 64 - s;
            if (s == DIGIT_BIT) {
                r[++j] = 0;
                s = 0;
            }
            else {
                r[++j] = a->dp[i] >> s;
                s = DIGIT_BIT - s;
            }
        }
        else {
            s += DIGIT_BIT;
        }
    }

    for (j++; j < size; j++) {
        r[j] = 0;
    }
#endif
}

/* Write r as big endian to byte array.
 * Fixed length number of bytes written: 384
 *
 * r  A single precision integer.
 * a  Byte array.
 */
static void sp_3072_to_bin_48(sp_digit* r, byte* a)
{
    int i;
    int j = 0;

    for (i = 47; i >= 0; i--, j += 8) {
        __asm__ __volatile__ (
            "ldr	x4, [%[r]]\n\t"
        #ifdef LITTLE_ENDIAN_ORDER
            "rev	x4, x4\n\t"
        #endif
            "str	x4, [%[a]]\n\t"
            :
            : [r] "r" (r + i), [a] "r" (a + j)
            : "memory", "x4"
        );
    }
}

#if (defined(WOLFSSL_HAVE_SP_RSA) && (!defined(WOLFSSL_RSA_PUBLIC_ONLY) || !defined(WOLFSSL_SP_SMALL))) || defined(WOLFSSL_HAVE_SP_DH)
/* Normalize the values in each word to 64.
 *
 * a  Array of sp_digit to normalize.
 */
#define sp_3072_norm_48(a)

#endif /* (WOLFSSL_HAVE_SP_RSA && (!WOLFSSL_RSA_PUBLIC_ONLY || !WOLFSSL_SP_SMALL)) || WOLFSSL_HAVE_SP_DH */
/* Normalize the values in each word to 64.
 *
 * a  Array of sp_digit to normalize.
 */
#define sp_3072_norm_48(a)

#ifndef WOLFSSL_SP_SMALL
/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_3072_mul_6(sp_digit* r, const sp_digit* a, const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x8, x9, [%[a], 0]\n\t"
        "ldp	x10, x11, [%[a], 16]\n\t"
        "ldp	x12, x13, [%[a], 32]\n\t"
        "ldp	x14, x15, [%[b], 0]\n\t"
        "ldp	x16, x17, [%[b], 16]\n\t"
        "ldp	x19, x20, [%[b], 32]\n\t"
        "#  A[0] * B[0]\n\t"
        "mul	x3, x8, x14\n\t"
        "umulh	x4, x8, x14\n\t"
        "str	x3, [%[r]]\n\t"
        "#  A[0] * B[1]\n\t"
        "mul	x6, x8, x15\n\t"
        "umulh	x7, x8, x15\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[1] * B[0]\n\t"
        "mul	x6, x9, x14\n\t"
        "adc	x5, xzr, x7\n\t"
        "umulh	x7, x9, x14\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "str	x4, [%[r], 8]\n\t"
        "adc	x3, xzr, xzr\n\t"
        "#  A[0] * B[2]\n\t"
        "mul	x6, x8, x16\n\t"
        "umulh	x7, x8, x16\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[1] * B[1]\n\t"
        "mul	x6, x9, x15\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x9, x15\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[2] * B[0]\n\t"
        "mul	x6, x10, x14\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x10, x14\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "str	x5, [%[r], 16]\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[0] * B[3]\n\t"
        "mul	x6, x8, x17\n\t"
        "umulh	x7, x8, x17\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[1] * B[2]\n\t"
        "mul	x6, x9, x16\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x9, x16\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[2] * B[1]\n\t"
        "mul	x6, x10, x15\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x10, x15\n\t"
        "adc	x5, x5, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[3] * B[0]\n\t"
        "mul	x6, x11, x14\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x11, x14\n\t"
        "adc	x5, x5, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "str	x3, [%[r], 24]\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[0] * B[4]\n\t"
        "mul	x6, x8, x19\n\t"
        "umulh	x7, x8, x19\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[1] * B[3]\n\t"
        "mul	x6, x9, x17\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x9, x17\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[2] * B[2]\n\t"
        "mul	x6, x10, x16\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x10, x16\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[3] * B[1]\n\t"
        "mul	x6, x11, x15\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x11, x15\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[4] * B[0]\n\t"
        "mul	x6, x12, x14\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x12, x14\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "str	x4, [%[r], 32]\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[0] * B[5]\n\t"
        "mul	x6, x8, x20\n\t"
        "umulh	x7, x8, x20\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[1] * B[4]\n\t"
        "mul	x6, x9, x19\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x9, x19\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[2] * B[3]\n\t"
        "mul	x6, x10, x17\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x10, x17\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[3] * B[2]\n\t"
        "mul	x6, x11, x16\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x11, x16\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[4] * B[1]\n\t"
        "mul	x6, x12, x15\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x12, x15\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[5] * B[0]\n\t"
        "mul	x6, x13, x14\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x13, x14\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "str	x5, [%[r], 40]\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[1] * B[5]\n\t"
        "mul	x6, x9, x20\n\t"
        "umulh	x7, x9, x20\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[2] * B[4]\n\t"
        "mul	x6, x10, x19\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x10, x19\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[3] * B[3]\n\t"
        "mul	x6, x11, x17\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x11, x17\n\t"
        "adc	x5, x5, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[4] * B[2]\n\t"
        "mul	x6, x12, x16\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x12, x16\n\t"
        "adc	x5, x5, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[5] * B[1]\n\t"
        "mul	x6, x13, x15\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x13, x15\n\t"
        "adc	x5, x5, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "str	x3, [%[r], 48]\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[2] * B[5]\n\t"
        "mul	x6, x10, x20\n\t"
        "umulh	x7, x10, x20\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[3] * B[4]\n\t"
        "mul	x6, x11, x19\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x11, x19\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[4] * B[3]\n\t"
        "mul	x6, x12, x17\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x12, x17\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "#  A[5] * B[2]\n\t"
        "mul	x6, x13, x16\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, x13, x16\n\t"
        "adc	x3, x3, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "adcs	x5, x5, x7\n\t"
        "str	x4, [%[r], 56]\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[3] * B[5]\n\t"
        "mul	x6, x11, x20\n\t"
        "umulh	x7, x11, x20\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[4] * B[4]\n\t"
        "mul	x6, x12, x19\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x12, x19\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "#  A[5] * B[3]\n\t"
        "mul	x6, x13, x17\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, x13, x17\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x3, x3, x7\n\t"
        "str	x5, [%[r], 64]\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[4] * B[5]\n\t"
        "mul	x6, x12, x20\n\t"
        "umulh	x7, x12, x20\n\t"
        "adds	x3, x3, x6\n\t"
        "#  A[5] * B[4]\n\t"
        "mul	x6, x13, x19\n\t"
        "adcs	x4, x4, x7\n\t"
        "umulh	x7, x13, x19\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "str	x3, [%[r], 72]\n\t"
        "adc	x5, x5, xzr\n\t"
        "#  A[5] * B[5]\n\t"
        "mul	x6, x13, x20\n\t"
        "umulh	x7, x13, x20\n\t"
        "adds	x4, x4, x6\n\t"
        "adc	x5, x5, x7\n\t"
        "stp	x4, x5, [%[r], 80]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x19", "x20"
    );
}

/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_3072_add_6(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "adds	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldr	x3, [%[a], 32]\n\t"
        "ldr	x4, [%[a], 40]\n\t"
        "ldr	x7, [%[b], 32]\n\t"
        "ldr	x8, [%[b], 40]\n\t"
        "adcs	x3, x3, x7\n\t"
        "adcs	x4, x4, x8\n\t"
        "str	x3, [%[r], 32]\n\t"
        "str	x4, [%[r], 40]\n\t"
        "cset	%[r], cs\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return (sp_digit)r;
}

/* Add digit to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_3072_add_word_6(sp_digit* r, const sp_digit* a,
        sp_digit b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "adds	x3, x3, %[b]\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldr		x3, [%[a], 32]\n\t"
        "ldr		x4, [%[a], 40]\n\t"
        "adcs	x3, x3, xzr\n\t"
        "adcs	x4, x4, xzr\n\t"
        "str		x3, [%[r], 32]\n\t"
        "str		x4, [%[r], 40]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6"
    );
}

/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer and result.
 * b  A single precision integer.
 */
static sp_digit sp_3072_sub_in_place_12(sp_digit* a, const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x2, x3, [%[a], 0]\n\t"
        "ldp	x6, x7, [%[b], 0]\n\t"
        "subs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 16]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 16]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 0]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 16]\n\t"
        "ldp	x2, x3, [%[a], 32]\n\t"
        "ldp	x6, x7, [%[b], 32]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 48]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 48]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 32]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 48]\n\t"
        "ldp	x2, x3, [%[a], 64]\n\t"
        "ldp	x6, x7, [%[b], 64]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 80]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 80]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 64]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 80]\n\t"
        "csetm	%[a], cc\n\t"
        : [a] "+r" (a)
        : [b] "r" (b)
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );

    return (sp_digit)a;
}

/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_3072_add_12(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "adds	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x7, x8, [%[b], 64]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 80]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "cset	%[r], cs\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return (sp_digit)r;
}

#ifndef WOLFSSL_SP_SMALL
/* Conditionally add a and b using the mask m.
 * m is -1 to add and 0 when not.
 *
 * r  A single precision number representing conditional add result.
 * a  A single precision number to add with.
 * b  A single precision number to add.
 * m  Mask value to apply.
 */
static sp_digit sp_3072_cond_add_6(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
    __asm__ __volatile__ (

        "ldp	x8, x9, [%[b], 0]\n\t"
        "ldp	x10, x11, [%[b], 16]\n\t"
        "ldp	x4, x5, [%[a], 0]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 16]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adds	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 0]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 16]\n\t"
        "ldp	x8, x9, [%[b], 32]\n\t"
        "ldp	x4, x5, [%[a], 32]\n\t"
        "and	x8, x8, %[m]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[r], 32]\n\t"
        "cset	%[r], cs\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    return (sp_digit)r;
}
#endif /* !WOLFSSL_SP_SMALL */

/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
SP_NOINLINE static void sp_3072_mul_12(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit* z0 = r;
    sp_digit z1[12];
    sp_digit a1[6];
    sp_digit b1[6];
    sp_digit* z2 = r + 12;
    sp_digit u;
    sp_digit ca;
    sp_digit cb;

    ca = sp_3072_add_6(a1, a, &a[6]);
    cb = sp_3072_add_6(b1, b, &b[6]);
    u  = ca & cb;

    sp_3072_mul_6(z2, &a[6], &b[6]);
    sp_3072_mul_6(z0, a, b);
    sp_3072_mul_6(z1, a1, b1);

    u += sp_3072_sub_in_place_12(z1, z0);
    u += sp_3072_sub_in_place_12(z1, z2);
    u += sp_3072_cond_add_6(z1 + 6, z1 + 6, a1, 0 - cb);
    u += sp_3072_cond_add_6(z1 + 6, z1 + 6, b1, 0 - ca);

    u += sp_3072_add_12(r + 6, r + 6, z1);
    (void)sp_3072_add_word_6(r + 18, r + 18, u);
}

/* Add digit to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_3072_add_word_12(sp_digit* r, const sp_digit* a,
        sp_digit b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "adds	x3, x3, %[b]\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "adcs	x3, x3, xzr\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "adcs	x3, x3, xzr\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6"
    );
}

/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer and result.
 * b  A single precision integer.
 */
static sp_digit sp_3072_sub_in_place_24(sp_digit* a, const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x2, x3, [%[a], 0]\n\t"
        "ldp	x6, x7, [%[b], 0]\n\t"
        "subs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 16]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 16]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 0]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 16]\n\t"
        "ldp	x2, x3, [%[a], 32]\n\t"
        "ldp	x6, x7, [%[b], 32]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 48]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 48]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 32]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 48]\n\t"
        "ldp	x2, x3, [%[a], 64]\n\t"
        "ldp	x6, x7, [%[b], 64]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 80]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 80]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 64]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 80]\n\t"
        "ldp	x2, x3, [%[a], 96]\n\t"
        "ldp	x6, x7, [%[b], 96]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 112]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 112]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 96]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 112]\n\t"
        "ldp	x2, x3, [%[a], 128]\n\t"
        "ldp	x6, x7, [%[b], 128]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 144]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 144]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 128]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 144]\n\t"
        "ldp	x2, x3, [%[a], 160]\n\t"
        "ldp	x6, x7, [%[b], 160]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 176]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 176]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 160]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 176]\n\t"
        "csetm	%[a], cc\n\t"
        : [a] "+r" (a)
        : [b] "r" (b)
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );

    return (sp_digit)a;
}

/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_3072_add_24(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "adds	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x7, x8, [%[b], 64]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 80]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "ldp	x3, x4, [%[a], 96]\n\t"
        "ldp	x7, x8, [%[b], 96]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 112]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 112]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 96]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 112]\n\t"
        "ldp	x3, x4, [%[a], 128]\n\t"
        "ldp	x7, x8, [%[b], 128]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 144]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 144]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 128]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 144]\n\t"
        "ldp	x3, x4, [%[a], 160]\n\t"
        "ldp	x7, x8, [%[b], 160]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 176]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 176]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 160]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 176]\n\t"
        "cset	%[r], cs\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return (sp_digit)r;
}

#ifndef WOLFSSL_SP_SMALL
/* Conditionally add a and b using the mask m.
 * m is -1 to add and 0 when not.
 *
 * r  A single precision number representing conditional add result.
 * a  A single precision number to add with.
 * b  A single precision number to add.
 * m  Mask value to apply.
 */
static sp_digit sp_3072_cond_add_12(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
    __asm__ __volatile__ (

        "ldp	x8, x9, [%[b], 0]\n\t"
        "ldp	x10, x11, [%[b], 16]\n\t"
        "ldp	x4, x5, [%[a], 0]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 16]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adds	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 0]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 16]\n\t"
        "ldp	x8, x9, [%[b], 32]\n\t"
        "ldp	x10, x11, [%[b], 48]\n\t"
        "ldp	x4, x5, [%[a], 32]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 48]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adcs	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 32]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 48]\n\t"
        "ldp	x8, x9, [%[b], 64]\n\t"
        "ldp	x10, x11, [%[b], 80]\n\t"
        "ldp	x4, x5, [%[a], 64]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 80]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adcs	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 64]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 80]\n\t"
        "cset	%[r], cs\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    return (sp_digit)r;
}
#endif /* !WOLFSSL_SP_SMALL */

/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
SP_NOINLINE static void sp_3072_mul_24(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit* z0 = r;
    sp_digit z1[24];
    sp_digit a1[12];
    sp_digit b1[12];
    sp_digit* z2 = r + 24;
    sp_digit u;
    sp_digit ca;
    sp_digit cb;

    ca = sp_3072_add_12(a1, a, &a[12]);
    cb = sp_3072_add_12(b1, b, &b[12]);
    u  = ca & cb;

    sp_3072_mul_12(z2, &a[12], &b[12]);
    sp_3072_mul_12(z0, a, b);
    sp_3072_mul_12(z1, a1, b1);

    u += sp_3072_sub_in_place_24(z1, z0);
    u += sp_3072_sub_in_place_24(z1, z2);
    u += sp_3072_cond_add_12(z1 + 12, z1 + 12, a1, 0 - cb);
    u += sp_3072_cond_add_12(z1 + 12, z1 + 12, b1, 0 - ca);

    u += sp_3072_add_24(r + 12, r + 12, z1);
    (void)sp_3072_add_word_12(r + 36, r + 36, u);
}

/* Add digit to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_3072_add_word_24(sp_digit* r, const sp_digit* a,
        sp_digit b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "adds	x3, x3, %[b]\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "adcs	x3, x3, xzr\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "adcs	x3, x3, xzr\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "ldp	x3, x4, [%[a], 96]\n\t"
        "ldp	x5, x6, [%[a], 112]\n\t"
        "adcs	x3, x3, xzr\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 96]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 112]\n\t"
        "ldp	x3, x4, [%[a], 128]\n\t"
        "ldp	x5, x6, [%[a], 144]\n\t"
        "adcs	x3, x3, xzr\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 128]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 144]\n\t"
        "ldp	x3, x4, [%[a], 160]\n\t"
        "ldp	x5, x6, [%[a], 176]\n\t"
        "adcs	x3, x3, xzr\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 160]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 176]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6"
    );
}

/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer and result.
 * b  A single precision integer.
 */
static sp_digit sp_3072_sub_in_place_48(sp_digit* a, const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x2, x3, [%[a], 0]\n\t"
        "ldp	x6, x7, [%[b], 0]\n\t"
        "subs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 16]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 16]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 0]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 16]\n\t"
        "ldp	x2, x3, [%[a], 32]\n\t"
        "ldp	x6, x7, [%[b], 32]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 48]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 48]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 32]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 48]\n\t"
        "ldp	x2, x3, [%[a], 64]\n\t"
        "ldp	x6, x7, [%[b], 64]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 80]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 80]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 64]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 80]\n\t"
        "ldp	x2, x3, [%[a], 96]\n\t"
        "ldp	x6, x7, [%[b], 96]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 112]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 112]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 96]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 112]\n\t"
        "ldp	x2, x3, [%[a], 128]\n\t"
        "ldp	x6, x7, [%[b], 128]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 144]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 144]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 128]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 144]\n\t"
        "ldp	x2, x3, [%[a], 160]\n\t"
        "ldp	x6, x7, [%[b], 160]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 176]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 176]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 160]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 176]\n\t"
        "ldp	x2, x3, [%[a], 192]\n\t"
        "ldp	x6, x7, [%[b], 192]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 208]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 208]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 192]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 208]\n\t"
        "ldp	x2, x3, [%[a], 224]\n\t"
        "ldp	x6, x7, [%[b], 224]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 240]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 240]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 224]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 240]\n\t"
        "ldp	x2, x3, [%[a], 256]\n\t"
        "ldp	x6, x7, [%[b], 256]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 272]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 272]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 256]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 272]\n\t"
        "ldp	x2, x3, [%[a], 288]\n\t"
        "ldp	x6, x7, [%[b], 288]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 304]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 304]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 288]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 304]\n\t"
        "ldp	x2, x3, [%[a], 320]\n\t"
        "ldp	x6, x7, [%[b], 320]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 336]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 336]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 320]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 336]\n\t"
        "ldp	x2, x3, [%[a], 352]\n\t"
        "ldp	x6, x7, [%[b], 352]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 368]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 368]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 352]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 368]\n\t"
        "csetm	%[a], cc\n\t"
        : [a] "+r" (a)
        : [b] "r" (b)
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );

    return (sp_digit)a;
}

/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_3072_add_48(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "adds	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x7, x8, [%[b], 64]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 80]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "ldp	x3, x4, [%[a], 96]\n\t"
        "ldp	x7, x8, [%[b], 96]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 112]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 112]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 96]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 112]\n\t"
        "ldp	x3, x4, [%[a], 128]\n\t"
        "ldp	x7, x8, [%[b], 128]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 144]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 144]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 128]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 144]\n\t"
        "ldp	x3, x4, [%[a], 160]\n\t"
        "ldp	x7, x8, [%[b], 160]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 176]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 176]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 160]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 176]\n\t"
        "ldp	x3, x4, [%[a], 192]\n\t"
        "ldp	x7, x8, [%[b], 192]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 208]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 208]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 192]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 208]\n\t"
        "ldp	x3, x4, [%[a], 224]\n\t"
        "ldp	x7, x8, [%[b], 224]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 240]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 240]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 224]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 240]\n\t"
        "ldp	x3, x4, [%[a], 256]\n\t"
        "ldp	x7, x8, [%[b], 256]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 272]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 272]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 256]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 272]\n\t"
        "ldp	x3, x4, [%[a], 288]\n\t"
        "ldp	x7, x8, [%[b], 288]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 304]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 304]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 288]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 304]\n\t"
        "ldp	x3, x4, [%[a], 320]\n\t"
        "ldp	x7, x8, [%[b], 320]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 336]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 336]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 320]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 336]\n\t"
        "ldp	x3, x4, [%[a], 352]\n\t"
        "ldp	x7, x8, [%[b], 352]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 368]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 368]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 352]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 368]\n\t"
        "cset	%[r], cs\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return (sp_digit)r;
}

#ifndef WOLFSSL_SP_SMALL
/* Conditionally add a and b using the mask m.
 * m is -1 to add and 0 when not.
 *
 * r  A single precision number representing conditional add result.
 * a  A single precision number to add with.
 * b  A single precision number to add.
 * m  Mask value to apply.
 */
static sp_digit sp_3072_cond_add_24(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
    __asm__ __volatile__ (

        "ldp	x8, x9, [%[b], 0]\n\t"
        "ldp	x10, x11, [%[b], 16]\n\t"
        "ldp	x4, x5, [%[a], 0]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 16]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adds	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 0]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 16]\n\t"
        "ldp	x8, x9, [%[b], 32]\n\t"
        "ldp	x10, x11, [%[b], 48]\n\t"
        "ldp	x4, x5, [%[a], 32]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 48]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adcs	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 32]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 48]\n\t"
        "ldp	x8, x9, [%[b], 64]\n\t"
        "ldp	x10, x11, [%[b], 80]\n\t"
        "ldp	x4, x5, [%[a], 64]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 80]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adcs	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 64]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 80]\n\t"
        "ldp	x8, x9, [%[b], 96]\n\t"
        "ldp	x10, x11, [%[b], 112]\n\t"
        "ldp	x4, x5, [%[a], 96]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 112]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adcs	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 96]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 112]\n\t"
        "ldp	x8, x9, [%[b], 128]\n\t"
        "ldp	x10, x11, [%[b], 144]\n\t"
        "ldp	x4, x5, [%[a], 128]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 144]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adcs	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 128]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 144]\n\t"
        "ldp	x8, x9, [%[b], 160]\n\t"
        "ldp	x10, x11, [%[b], 176]\n\t"
        "ldp	x4, x5, [%[a], 160]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 176]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adcs	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 160]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 176]\n\t"
        "cset	%[r], cs\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    return (sp_digit)r;
}
#endif /* !WOLFSSL_SP_SMALL */

/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
SP_NOINLINE static void sp_3072_mul_48(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit* z0 = r;
    sp_digit z1[48];
    sp_digit a1[24];
    sp_digit b1[24];
    sp_digit* z2 = r + 48;
    sp_digit u;
    sp_digit ca;
    sp_digit cb;

    ca = sp_3072_add_24(a1, a, &a[24]);
    cb = sp_3072_add_24(b1, b, &b[24]);
    u  = ca & cb;

    sp_3072_mul_24(z2, &a[24], &b[24]);
    sp_3072_mul_24(z0, a, b);
    sp_3072_mul_24(z1, a1, b1);

    u += sp_3072_sub_in_place_48(z1, z0);
    u += sp_3072_sub_in_place_48(z1, z2);
    u += sp_3072_cond_add_24(z1 + 24, z1 + 24, a1, 0 - cb);
    u += sp_3072_cond_add_24(z1 + 24, z1 + 24, b1, 0 - ca);

    u += sp_3072_add_48(r + 24, r + 24, z1);
    (void)sp_3072_add_word_24(r + 72, r + 72, u);
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
static void sp_3072_sqr_24(sp_digit* r, const sp_digit* a)
{
    sp_digit tmp[24];

    __asm__ __volatile__ (
        "#  A[0] * A[0]\n\t"
        "ldr	x9, [%[a], 0]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x3, x9, x9\n\t"
        "mov	x4, xzr\n\t"
        "str	x8, [%[tmp]]\n\t"
        "#  A[0] * A[1]\n\t"
        "ldr	x9, [%[a], 8]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, xzr, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "str	x3, [%[tmp], 8]\n\t"
        "#  A[0] * A[2]\n\t"
        "ldr	x9, [%[a], 16]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[1] * A[1]\n\t"
        "ldr	x9, [%[a], 8]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[tmp], 16]\n\t"
        "#  A[0] * A[3]\n\t"
        "ldr	x9, [%[a], 24]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[1] * A[2]\n\t"
        "ldr	x9, [%[a], 16]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x2, [%[tmp], 24]\n\t"
        "#  A[0] * A[4]\n\t"
        "ldr	x9, [%[a], 32]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, xzr, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "#  A[1] * A[3]\n\t"
        "ldr	x9, [%[a], 24]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "#  A[2] * A[2]\n\t"
        "ldr	x9, [%[a], 16]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "str	x3, [%[tmp], 32]\n\t"
        "#  A[0] * A[5]\n\t"
        "ldr	x9, [%[a], 40]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[4]\n\t"
        "ldr	x9, [%[a], 32]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[3]\n\t"
        "ldr	x9, [%[a], 24]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 40]\n\t"
        "#  A[0] * A[6]\n\t"
        "ldr	x9, [%[a], 48]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[5]\n\t"
        "ldr	x9, [%[a], 40]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[4]\n\t"
        "ldr	x9, [%[a], 32]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[3]\n\t"
        "ldr	x9, [%[a], 24]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[tmp], 48]\n\t"
        "#  A[0] * A[7]\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[6]\n\t"
        "ldr	x9, [%[a], 48]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[5]\n\t"
        "ldr	x9, [%[a], 40]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[4]\n\t"
        "ldr	x9, [%[a], 32]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[tmp], 56]\n\t"
        "#  A[0] * A[8]\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[7]\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[6]\n\t"
        "ldr	x9, [%[a], 48]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[5]\n\t"
        "ldr	x9, [%[a], 40]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[4]\n\t"
        "ldr	x9, [%[a], 32]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 64]\n\t"
        "#  A[0] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[8]\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[7]\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[6]\n\t"
        "ldr	x9, [%[a], 48]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[5]\n\t"
        "ldr	x9, [%[a], 40]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[tmp], 72]\n\t"
        "#  A[0] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[8]\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[7]\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[6]\n\t"
        "ldr	x9, [%[a], 48]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[5]\n\t"
        "ldr	x9, [%[a], 40]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[tmp], 80]\n\t"
        "#  A[0] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[8]\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[7]\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[6]\n\t"
        "ldr	x9, [%[a], 48]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 88]\n\t"
        "#  A[0] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[8]\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[7]\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[6]\n\t"
        "ldr	x9, [%[a], 48]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[tmp], 96]\n\t"
        "#  A[0] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[8]\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[7]\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[tmp], 104]\n\t"
        "#  A[0] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[8]\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[7]\n\t"
        "ldr	x9, [%[a], 56]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 112]\n\t"
        "#  A[0] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[8]\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[tmp], 120]\n\t"
        "#  A[0] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[8]\n\t"
        "ldr	x9, [%[a], 64]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[tmp], 128]\n\t"
        "#  A[0] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 136]\n\t"
        "#  A[0] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[9]\n\t"
        "ldr	x9, [%[a], 72]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[tmp], 144]\n\t"
        "#  A[0] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[tmp], 152]\n\t"
        "#  A[0] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[10]\n\t"
        "ldr	x9, [%[a], 80]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 160]\n\t"
        "#  A[0] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[tmp], 168]\n\t"
        "#  A[0] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[11]\n\t"
        "ldr	x9, [%[a], 88]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[tmp], 176]\n\t"
        "#  A[0] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 0]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[1] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[2] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[tmp], 184]\n\t"
        "#  A[1] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 8]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[2] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[3] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[12]\n\t"
        "ldr	x9, [%[a], 96]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 192]\n\t"
        "#  A[2] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 16]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[3] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[4] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 200]\n\t"
        "#  A[3] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 24]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[4] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[5] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[13]\n\t"
        "ldr	x9, [%[a], 104]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 208]\n\t"
        "#  A[4] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 32]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[5] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[6] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 216]\n\t"
        "#  A[5] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 40]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[6] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[7] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[14]\n\t"
        "ldr	x9, [%[a], 112]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 224]\n\t"
        "#  A[6] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 48]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[7] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[8] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 232]\n\t"
        "#  A[7] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 56]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[8] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[9] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[15]\n\t"
        "ldr	x9, [%[a], 120]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 240]\n\t"
        "#  A[8] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 64]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[9] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[10] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 248]\n\t"
        "#  A[9] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 72]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[10] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[11] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[16]\n\t"
        "ldr	x9, [%[a], 128]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 256]\n\t"
        "#  A[10] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 80]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[11] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[12] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 264]\n\t"
        "#  A[11] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 88]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[12] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[13] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[17] * A[17]\n\t"
        "ldr	x9, [%[a], 136]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 272]\n\t"
        "#  A[12] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 96]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[13] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[14] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[17] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 280]\n\t"
        "#  A[13] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 104]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[14] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[15] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[17] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[18] * A[18]\n\t"
        "ldr	x9, [%[a], 144]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 288]\n\t"
        "#  A[14] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 112]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[15] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[16] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[17] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[18] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "ldr	x10, [%[a], 144]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 296]\n\t"
        "#  A[15] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 120]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[16] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[17] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[18] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 144]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[19] * A[19]\n\t"
        "ldr	x9, [%[a], 152]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 304]\n\t"
        "#  A[16] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 128]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x4, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[17] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[18] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 144]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[19] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "ldr	x10, [%[a], 152]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x2, x2, x5\n\t"
        "adcs	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "str	x2, [%[r], 312]\n\t"
        "#  A[17] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 136]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x2, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[18] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 144]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[19] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 152]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[20] * A[20]\n\t"
        "ldr	x9, [%[a], 160]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x3, x3, x5\n\t"
        "adcs	x4, x4, x6\n\t"
        "adc	x2, x2, x7\n\t"
        "str	x3, [%[r], 320]\n\t"
        "#  A[18] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 144]\n\t"
        "mul	x5, x9, x10\n\t"
        "umulh	x6, x9, x10\n\t"
        "mov	x3, xzr\n\t"
        "mov	x7, xzr\n\t"
        "#  A[19] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 152]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "#  A[20] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "ldr	x10, [%[a], 160]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x5, x5, x8\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adc	x7, x7, x7\n\t"
        "adds	x4, x4, x5\n\t"
        "adcs	x2, x2, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x4, [%[r], 328]\n\t"
        "#  A[19] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 152]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[20] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 160]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "#  A[21] * A[21]\n\t"
        "ldr	x9, [%[a], 168]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x2, [%[r], 336]\n\t"
        "#  A[20] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 160]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, xzr, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "#  A[21] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "ldr	x10, [%[a], 168]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "adds	x3, x3, x8\n\t"
        "adcs	x4, x4, x9\n\t"
        "adc	x2, x2, xzr\n\t"
        "str	x3, [%[r], 344]\n\t"
        "#  A[21] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 168]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, x3, xzr\n\t"
        "#  A[22] * A[22]\n\t"
        "ldr	x9, [%[a], 176]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x4, x4, x8\n\t"
        "adcs	x2, x2, x9\n\t"
        "adc	x3, x3, xzr\n\t"
        "str	x4, [%[r], 352]\n\t"
        "#  A[22] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "ldr	x10, [%[a], 176]\n\t"
        "mul	x8, x9, x10\n\t"
        "umulh	x9, x9, x10\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x2, x2, x8\n\t"
        "adcs	x3, x3, x9\n\t"
        "adc	x4, x4, xzr\n\t"
        "str	x2, [%[r], 360]\n\t"
        "#  A[23] * A[23]\n\t"
        "ldr	x9, [%[a], 184]\n\t"
        "mul	x8, x9, x9\n\t"
        "umulh	x9, x9, x9\n\t"
        "adds	x3, x3, x8\n\t"
        "adc	x4, x4, x9\n\t"
        "stp	x3, x4, [%[r], 368]\n\t"
        "ldp	x9, x10, [%[tmp], 0]\n\t"
        "stp	x9, x10, [%[r], 0]\n\t"
        "ldp	x9, x10, [%[tmp], 16]\n\t"
        "stp	x9, x10, [%[r], 16]\n\t"
        "ldp	x9, x10, [%[tmp], 32]\n\t"
        "stp	x9, x10, [%[r], 32]\n\t"
        "ldp	x9, x10, [%[tmp], 48]\n\t"
        "stp	x9, x10, [%[r], 48]\n\t"
        "ldp	x9, x10, [%[tmp], 64]\n\t"
        "stp	x9, x10, [%[r], 64]\n\t"
        "ldp	x9, x10, [%[tmp], 80]\n\t"
        "stp	x9, x10, [%[r], 80]\n\t"
        "ldp	x9, x10, [%[tmp], 96]\n\t"
        "stp	x9, x10, [%[r], 96]\n\t"
        "ldp	x9, x10, [%[tmp], 112]\n\t"
        "stp	x9, x10, [%[r], 112]\n\t"
        "ldp	x9, x10, [%[tmp], 128]\n\t"
        "stp	x9, x10, [%[r], 128]\n\t"
        "ldp	x9, x10, [%[tmp], 144]\n\t"
        "stp	x9, x10, [%[r], 144]\n\t"
        "ldp	x9, x10, [%[tmp], 160]\n\t"
        "stp	x9, x10, [%[r], 160]\n\t"
        "ldp	x9, x10, [%[tmp], 176]\n\t"
        "stp	x9, x10, [%[r], 176]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [tmp] "r" (tmp)
        : "memory", "x2", "x3", "x4", "x8", "x9", "x10", "x5", "x6", "x7"
    );
}

/* Sub b from a into r. (r = a - b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_3072_sub_24(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "subs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x7, x8, [%[b], 64]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 80]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "ldp	x3, x4, [%[a], 96]\n\t"
        "ldp	x7, x8, [%[b], 96]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 112]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 112]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 96]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 112]\n\t"
        "ldp	x3, x4, [%[a], 128]\n\t"
        "ldp	x7, x8, [%[b], 128]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 144]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 144]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 128]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 144]\n\t"
        "ldp	x3, x4, [%[a], 160]\n\t"
        "ldp	x7, x8, [%[b], 160]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 176]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 176]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 160]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 176]\n\t"
        "csetm	%[r], cc\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return (sp_digit)r;
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
SP_NOINLINE static void sp_3072_sqr_48(sp_digit* r, const sp_digit* a)
{
    sp_digit* z0 = r;
    sp_digit* z2 = r + 48;
    sp_digit z1[48];
    sp_digit* a1 = z1;
    sp_digit* zero = z1 + 24;
    sp_digit u;
    sp_digit mask;
    sp_digit* p1;
    sp_digit* p2;

    XMEMSET(zero, 0, sizeof(sp_digit) * 24);

    mask = sp_3072_sub_24(a1, a, &a[24]);
    p1 = (sp_digit*)(((sp_digit)zero &   mask ) | ((sp_digit)a1 & (~mask)));
    p2 = (sp_digit*)(((sp_digit)zero & (~mask)) | ((sp_digit)a1 &   mask ));
    (void)sp_3072_sub_24(a1, p1, p2);

    sp_3072_sqr_24(z2, &a[24]);
    sp_3072_sqr_24(z0, a);
    sp_3072_sqr_24(z1, a1);

    u = 0;
    u -= sp_3072_sub_in_place_48(z1, z2);
    u -= sp_3072_sub_in_place_48(z1, z0);
    u += sp_3072_sub_in_place_48(r + 24, z1);
    sp_3072_add_word_24(r + 72, r + 72, u);
}

#endif /* !WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_3072_add_48(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x11, %[a], 384\n\t"
        "\n1:\n\t"
        "adds	%[c], %[c], #-1\n\t"
        "ldp	x3, x4, [%[a]], #16\n\t"
        "ldp	x5, x6, [%[a]], #16\n\t"
        "ldp	x7, x8, [%[b]], #16\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x9, x10, [%[b]], #16\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r]], #16\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r]], #16\n\t"
        "cset	%[c], cs\n\t"
        "cmp	%[a], x11\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [r] "+r" (r), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_3072_sub_in_place_48(sp_digit* a, const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x10, %[a], 384\n\t"
        "\n1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldp	x2, x3, [%[a]]\n\t"
        "ldp	x4, x5, [%[a], #16]\n\t"
        "ldp	x6, x7, [%[b]], #16\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x8, x9, [%[b]], #16\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a]], #16\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a]], #16\n\t"
        "csetm	%[c], cc\n\t"
        "cmp	%[a], x10\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_3072_mul_48(sp_digit* r, const sp_digit* a, const sp_digit* b)
{
    sp_digit tmp[96];

    __asm__ __volatile__ (
        "mov	x5, xzr\n\t"
        "mov	x6, xzr\n\t"
        "mov	x7, xzr\n\t"
        "mov	x8, xzr\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 376\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[b], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 384\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 752\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
static void sp_3072_sqr_48(sp_digit* r, const sp_digit* a)
{
    sp_digit tmp[96];

    __asm__ __volatile__ (
        "mov	x6, xzr\n\t"
        "mov	x7, xzr\n\t"
        "mov	x8, xzr\n\t"
        "mov	x5, xzr\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 376\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "cmp	x4, x3\n\t"
        "b.eq	4f\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[a], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "b.al	5f\n\t"
        "\n4:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "mul	x9, x10, x10\n\t"
        "umulh	x10, x10, x10\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "\n5:\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 384\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x4\n\t"
        "b.gt	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 752\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

#endif /* WOLFSSL_SP_SMALL */
#if (defined(WOLFSSL_HAVE_SP_RSA) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)) || defined(WOLFSSL_HAVE_SP_DH)
#ifdef WOLFSSL_SP_SMALL
/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_3072_add_24(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x11, %[a], 192\n\t"
        "\n1:\n\t"
        "adds	%[c], %[c], #-1\n\t"
        "ldp	x3, x4, [%[a]], #16\n\t"
        "ldp	x5, x6, [%[a]], #16\n\t"
        "ldp	x7, x8, [%[b]], #16\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x9, x10, [%[b]], #16\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r]], #16\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r]], #16\n\t"
        "cset	%[c], cs\n\t"
        "cmp	%[a], x11\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [r] "+r" (r), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_3072_sub_in_place_24(sp_digit* a, const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x10, %[a], 192\n\t"
        "\n1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldp	x2, x3, [%[a]]\n\t"
        "ldp	x4, x5, [%[a], #16]\n\t"
        "ldp	x6, x7, [%[b]], #16\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x8, x9, [%[b]], #16\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a]], #16\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a]], #16\n\t"
        "csetm	%[c], cc\n\t"
        "cmp	%[a], x10\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_3072_mul_24(sp_digit* r, const sp_digit* a, const sp_digit* b)
{
    sp_digit tmp[48];

    __asm__ __volatile__ (
        "mov	x5, xzr\n\t"
        "mov	x6, xzr\n\t"
        "mov	x7, xzr\n\t"
        "mov	x8, xzr\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 184\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[b], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 192\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 368\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
static void sp_3072_sqr_24(sp_digit* r, const sp_digit* a)
{
    sp_digit tmp[48];

    __asm__ __volatile__ (
        "mov	x6, xzr\n\t"
        "mov	x7, xzr\n\t"
        "mov	x8, xzr\n\t"
        "mov	x5, xzr\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 184\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "cmp	x4, x3\n\t"
        "b.eq	4f\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[a], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "b.al	5f\n\t"
        "\n4:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "mul	x9, x10, x10\n\t"
        "umulh	x10, x10, x10\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "\n5:\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 192\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x4\n\t"
        "b.gt	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 368\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

#endif /* WOLFSSL_SP_SMALL */
#endif /* (WOLFSSL_HAVE_SP_RSA & !WOLFSSL_RSA_PUBLIC_ONLY) | WOLFSSL_HAVE_SP_DH */

/* Caclulate the bottom digit of -1/a mod 2^n.
 *
 * a    A single precision number.
 * rho  Bottom word of inverse.
 */
static void sp_3072_mont_setup(const sp_digit* a, sp_digit* rho)
{
    sp_digit x;
    sp_digit b;

    b = a[0];
    x = (((b + 2) & 4) << 1) + b; /* here x*a==1 mod 2**4 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**8 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**16 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**32 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**64 */

    /* rho = -1/m mod b */
    *rho = (sp_digit)0 - x;
}

/* Mul a by digit b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision digit.
 */
static void sp_3072_mul_d_48(sp_digit* r, const sp_digit* a,
        sp_digit b)
{
#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldr	x8, [%[a]]\n\t"
        "mul	x5, %[b], x8\n\t"
        "umulh	x3, %[b], x8\n\t"
        "mov	x4, xzr\n\t"
        "str	x5, [%[r]]\n\t"
        "mov	x5, xzr\n\t"
        "mov	x9, #8\n\t"
        "1:\n\t"
        "ldr	x8, [%[a], x9]\n\t"
        "mul	x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "str	x3, [%[r], x9]\n\t"
        "mov	x3, x4\n\t"
        "mov	x4, x5\n\t"
        "mov	x5, #0\n\t"
        "add	x9, x9, #8\n\t"
        "cmp	x9, 384\n\t"
        "b.lt	1b\n\t"
        "str	x3, [%[r], 384]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );
#else
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldp	x9, x10, [%[a]]\n\t"
        "mul	x3, %[b], x9\n\t"
        "umulh	x4, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "# A[1] * B\n\t"
        "str	x3, [%[r]]\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[2] * B\n\t"
        "ldp	x9, x10, [%[a], 16]\n\t"
        "str	x4, [%[r], 8]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[3] * B\n\t"
        "str	x5, [%[r], 16]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[4] * B\n\t"
        "ldp	x9, x10, [%[a], 32]\n\t"
        "str	x3, [%[r], 24]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[5] * B\n\t"
        "str	x4, [%[r], 32]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[6] * B\n\t"
        "ldp	x9, x10, [%[a], 48]\n\t"
        "str	x5, [%[r], 40]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[7] * B\n\t"
        "str	x3, [%[r], 48]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[8] * B\n\t"
        "ldp	x9, x10, [%[a], 64]\n\t"
        "str	x4, [%[r], 56]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[9] * B\n\t"
        "str	x5, [%[r], 64]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[10] * B\n\t"
        "ldp	x9, x10, [%[a], 80]\n\t"
        "str	x3, [%[r], 72]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[11] * B\n\t"
        "str	x4, [%[r], 80]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[12] * B\n\t"
        "ldp	x9, x10, [%[a], 96]\n\t"
        "str	x5, [%[r], 88]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[13] * B\n\t"
        "str	x3, [%[r], 96]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[14] * B\n\t"
        "ldp	x9, x10, [%[a], 112]\n\t"
        "str	x4, [%[r], 104]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[15] * B\n\t"
        "str	x5, [%[r], 112]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[16] * B\n\t"
        "ldp	x9, x10, [%[a], 128]\n\t"
        "str	x3, [%[r], 120]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[17] * B\n\t"
        "str	x4, [%[r], 128]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[18] * B\n\t"
        "ldp	x9, x10, [%[a], 144]\n\t"
        "str	x5, [%[r], 136]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[19] * B\n\t"
        "str	x3, [%[r], 144]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[20] * B\n\t"
        "ldp	x9, x10, [%[a], 160]\n\t"
        "str	x4, [%[r], 152]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[21] * B\n\t"
        "str	x5, [%[r], 160]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[22] * B\n\t"
        "ldp	x9, x10, [%[a], 176]\n\t"
        "str	x3, [%[r], 168]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[23] * B\n\t"
        "str	x4, [%[r], 176]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[24] * B\n\t"
        "ldp	x9, x10, [%[a], 192]\n\t"
        "str	x5, [%[r], 184]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[25] * B\n\t"
        "str	x3, [%[r], 192]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[26] * B\n\t"
        "ldp	x9, x10, [%[a], 208]\n\t"
        "str	x4, [%[r], 200]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[27] * B\n\t"
        "str	x5, [%[r], 208]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[28] * B\n\t"
        "ldp	x9, x10, [%[a], 224]\n\t"
        "str	x3, [%[r], 216]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[29] * B\n\t"
        "str	x4, [%[r], 224]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[30] * B\n\t"
        "ldp	x9, x10, [%[a], 240]\n\t"
        "str	x5, [%[r], 232]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[31] * B\n\t"
        "str	x3, [%[r], 240]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[32] * B\n\t"
        "ldp	x9, x10, [%[a], 256]\n\t"
        "str	x4, [%[r], 248]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[33] * B\n\t"
        "str	x5, [%[r], 256]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[34] * B\n\t"
        "ldp	x9, x10, [%[a], 272]\n\t"
        "str	x3, [%[r], 264]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[35] * B\n\t"
        "str	x4, [%[r], 272]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[36] * B\n\t"
        "ldp	x9, x10, [%[a], 288]\n\t"
        "str	x5, [%[r], 280]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[37] * B\n\t"
        "str	x3, [%[r], 288]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[38] * B\n\t"
        "ldp	x9, x10, [%[a], 304]\n\t"
        "str	x4, [%[r], 296]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[39] * B\n\t"
        "str	x5, [%[r], 304]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[40] * B\n\t"
        "ldp	x9, x10, [%[a], 320]\n\t"
        "str	x3, [%[r], 312]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[41] * B\n\t"
        "str	x4, [%[r], 320]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[42] * B\n\t"
        "ldp	x9, x10, [%[a], 336]\n\t"
        "str	x5, [%[r], 328]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[43] * B\n\t"
        "str	x3, [%[r], 336]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[44] * B\n\t"
        "ldp	x9, x10, [%[a], 352]\n\t"
        "str	x4, [%[r], 344]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[45] * B\n\t"
        "str	x5, [%[r], 352]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[46] * B\n\t"
        "ldp	x9, x10, [%[a], 368]\n\t"
        "str	x3, [%[r], 360]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[47] * B\n\t"
        "str	x4, [%[r], 368]\n\t"
        "mul	x6, %[b], x10\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x5, [%[r], 376]\n\t"
        "str	x3, [%[r], 384]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );
#endif
}

#if (defined(WOLFSSL_HAVE_SP_RSA) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)) || defined(WOLFSSL_HAVE_SP_DH)
/* r = 2^n mod m where n is the number of bits to reduce by.
 * Given m must be 3072 bits, just need to subtract.
 *
 * r  A single precision number.
 * m  A single precision number.
 */
static void sp_3072_mont_norm_24(sp_digit* r, const sp_digit* m)
{
    XMEMSET(r, 0, sizeof(sp_digit) * 24);

    /* r = 2^n mod m */
    sp_3072_sub_in_place_24(r, m);
}

/* Reduce the number back to 3072 bits using Montgomery reduction.
 *
 * a   A single precision number to reduce in place.
 * m   The single precision number representing the modulus.
 * mp  The digit representing the negative inverse of m mod 2^n.
 */
SP_NOINLINE static void sp_3072_mont_reduce_24(sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    __asm__ __volatile__ (
        "ldp	x11, x12, [%[a], 0]\n\t"
        "ldp	x13, x14, [%[a], 16]\n\t"
        "ldp	x15, x16, [%[a], 32]\n\t"
        "ldp	x17, x19, [%[a], 48]\n\t"
        "ldp	x20, x21, [%[a], 64]\n\t"
        "ldp	x22, x23, [%[a], 80]\n\t"
        "# No carry yet\n\t"
        "mov	x3, xzr\n\t"
        "# i = 0..23\n\t"
        "mov	x4, 24\n\t"
        "\n1:\n\t"
        "# mu = a[i] * mp\n\t"
        "mul	x10, %[mp], x11\n\t"
        "ldp	x24, x25, [%[m], 0]\n\t"
        "ldp	x26, x27, [%[m], 16]\n\t"
        "# a[i+0] += m[0] * mu\n\t"
        "mul	x5, x24, x10\n\t"
        "umulh	x6, x24, x10\n\t"
        "# a[i+1] += m[1] * mu\n\t"
        "adds	x11, x11, x5\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x11, x12, x5\n\t"
        "# a[i+2] += m[2] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x11, x11, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x12, x13, x5\n\t"
        "# a[i+3] += m[3] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x12, x12, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x13, x14, x5\n\t"
        "ldp	x24, x25, [%[m], 32]\n\t"
        "ldp	x26, x27, [%[m], 48]\n\t"
        "# a[i+4] += m[4] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x13, x13, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x14, x15, x5\n\t"
        "# a[i+5] += m[5] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x14, x14, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x15, x16, x5\n\t"
        "# a[i+6] += m[6] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x15, x15, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x16, x17, x5\n\t"
        "# a[i+7] += m[7] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x16, x16, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x17, x19, x5\n\t"
        "ldp	x24, x25, [%[m], 64]\n\t"
        "ldp	x26, x27, [%[m], 80]\n\t"
        "# a[i+8] += m[8] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x17, x17, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x19, x20, x5\n\t"
        "# a[i+9] += m[9] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x19, x19, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x20, x21, x5\n\t"
        "# a[i+10] += m[10] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x20, x20, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x21, x22, x5\n\t"
        "# a[i+11] += m[11] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x21, x21, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x22, x23, x5\n\t"
        "ldp	x24, x25, [%[m], 96]\n\t"
        "ldp	x26, x27, [%[m], 112]\n\t"
        "# a[i+12] += m[12] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x22, x22, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "ldr	x23, [%[a], 96]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x23, x23, x5\n\t"
        "# a[i+13] += m[13] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x23, x23, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "ldp	x8, x9, [%[a], 104]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+14] += m[14] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 104]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+15] += m[15] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 112]\n\t"
        "ldp	x8, x9, [%[a], 120]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 128]\n\t"
        "ldp	x26, x27, [%[m], 144]\n\t"
        "# a[i+16] += m[16] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 120]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+17] += m[17] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 128]\n\t"
        "ldp	x8, x9, [%[a], 136]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+18] += m[18] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 136]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+19] += m[19] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 144]\n\t"
        "ldp	x8, x9, [%[a], 152]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 160]\n\t"
        "ldp	x26, x27, [%[m], 176]\n\t"
        "# a[i+20] += m[20] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 152]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+21] += m[21] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 160]\n\t"
        "ldp	x8, x9, [%[a], 168]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+22] += m[22] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 168]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+23] += m[23] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 176]\n\t"
        "umulh	x7, x27, x10\n\t"
        "ldp	x8, x9, [%[a], 184]\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x7, x7, x3\n\t"
        "cset	x3, cs\n\t"
        "adds	x8, x8, x5\n\t"
        "str	x8, [%[a], 184]\n\t"
        "adcs	x9, x9, x7\n\t"
        "str	x9, [%[a], 192]\n\t"
        "adc	x3, x3, xzr\n\t"
        "subs	x4, x4, 1\n\t"
        "add	%[a], %[a], 8\n\t"
        "b.ne	1b\n\t"
        "# Create mask\n\t"
        "neg	x3, x3\n\t"
        "mov   %[mp], %[a]\n\t"
        "sub	%[a], %[a], 192\n\t"
        "# Subtract masked modulus\n\t"
        "ldp	x4, x5, [%[m], 0]\n\t"
        "ldp	x6, x7, [%[m], 16]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "subs	x11, x11, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x12, x12, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x13, x13, x6\n\t"
        "stp	x11, x12, [%[a], 0]\n\t"
        "sbcs	x14, x14, x7\n\t"
        "stp	x13, x14, [%[a], 16]\n\t"
        "ldp	x4, x5, [%[m], 32]\n\t"
        "ldp	x6, x7, [%[m], 48]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x15, x15, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x16, x16, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x17, x17, x6\n\t"
        "stp	x15, x16, [%[a], 32]\n\t"
        "sbcs	x19, x19, x7\n\t"
        "stp	x17, x19, [%[a], 48]\n\t"
        "ldp	x4, x5, [%[m], 64]\n\t"
        "ldp	x6, x7, [%[m], 80]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x20, x20, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x21, x21, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x22, x22, x6\n\t"
        "stp	x20, x21, [%[a], 64]\n\t"
        "sbcs	x23, x23, x7\n\t"
        "stp	x22, x23, [%[a], 80]\n\t"
        "ldp	x4, x5, [%[m], 96]\n\t"
        "ldp	x6, x7, [%[m], 112]\n\t"
        "ldp	x8, x9, [%[mp], 96]\n\t"
        "ldp	x10, x11, [%[mp], 112]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 96]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 112]\n\t"
        "ldp	x4, x5, [%[m], 128]\n\t"
        "ldp	x6, x7, [%[m], 144]\n\t"
        "ldp	x8, x9, [%[mp], 128]\n\t"
        "ldp	x10, x11, [%[mp], 144]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 128]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 144]\n\t"
        "ldp	x4, x5, [%[m], 160]\n\t"
        "ldp	x6, x7, [%[m], 176]\n\t"
        "ldp	x8, x9, [%[mp], 160]\n\t"
        "ldp	x10, x11, [%[mp], 176]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 160]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 176]\n\t"
        : [a] "+r" (a), [mp] "+r" (mp)
        : [m] "r" (m)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x10", "x8", "x9", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x19", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27"
    );

}

/* Multiply two Montgomery form numbers mod the modulus (prime).
 * (r = a * b mod m)
 *
 * r   Result of multiplication.
 * a   First number to multiply in Montgomery form.
 * b   Second number to multiply in Montgomery form.
 * m   Modulus (prime).
 * mp  Montgomery mulitplier.
 */
SP_NOINLINE static void sp_3072_mont_mul_24(sp_digit* r, const sp_digit* a,
        const sp_digit* b, const sp_digit* m, sp_digit mp)
{
    sp_3072_mul_24(r, a, b);
    sp_3072_mont_reduce_24(r, m, mp);
}

/* Square the Montgomery form number. (r = a * a mod m)
 *
 * r   Result of squaring.
 * a   Number to square in Montgomery form.
 * m   Modulus (prime).
 * mp  Montgomery mulitplier.
 */
SP_NOINLINE static void sp_3072_mont_sqr_24(sp_digit* r, const sp_digit* a,
        const sp_digit* m, sp_digit mp)
{
    sp_3072_sqr_24(r, a);
    sp_3072_mont_reduce_24(r, m, mp);
}

/* Conditionally subtract b from a using the mask m.
 * m is -1 to subtract and 0 when not copying.
 *
 * r  A single precision number representing condition subtract result.
 * a  A single precision number to subtract from.
 * b  A single precision number to subtract.
 * m  Mask value to apply.
 */
static sp_digit sp_3072_cond_sub_24(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
#ifdef WOLFSSL_SP_SMALL
    sp_digit c = 0;

    __asm__ __volatile__ (
        "mov	x8, #0\n\t"
        "1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldr	x4, [%[a], x8]\n\t"
        "ldr	x5, [%[b], x8]\n\t"
        "and	x5, x5, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "csetm	%[c], cc\n\t"
        "str	x4, [%[r], x8]\n\t"
        "add	x8, x8, #8\n\t"
        "cmp	x8, 192\n\t"
        "b.lt	1b\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8", "x9", "x10", "x11", "x12"
    );

    return c;
#else
    __asm__ __volatile__ (

        "ldp	x5, x7, [%[b], 0]\n\t"
        "ldp	x11, x12, [%[b], 16]\n\t"
        "ldp	x4, x6, [%[a], 0]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 16]\n\t"
        "and	x7, x7, %[m]\n\t"
        "subs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 0]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 16]\n\t"
        "ldp	x5, x7, [%[b], 32]\n\t"
        "ldp	x11, x12, [%[b], 48]\n\t"
        "ldp	x4, x6, [%[a], 32]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 48]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 32]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 48]\n\t"
        "ldp	x5, x7, [%[b], 64]\n\t"
        "ldp	x11, x12, [%[b], 80]\n\t"
        "ldp	x4, x6, [%[a], 64]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 80]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 64]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 80]\n\t"
        "ldp	x5, x7, [%[b], 96]\n\t"
        "ldp	x11, x12, [%[b], 112]\n\t"
        "ldp	x4, x6, [%[a], 96]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 112]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 96]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 112]\n\t"
        "ldp	x5, x7, [%[b], 128]\n\t"
        "ldp	x11, x12, [%[b], 144]\n\t"
        "ldp	x4, x6, [%[a], 128]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 144]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 128]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 144]\n\t"
        "ldp	x5, x7, [%[b], 160]\n\t"
        "ldp	x11, x12, [%[b], 176]\n\t"
        "ldp	x4, x6, [%[a], 160]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 176]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 160]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 176]\n\t"
        "csetm	%[r], cc\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8", "x9", "x10", "x11", "x12"
    );

    return (sp_digit)r;
#endif /* WOLFSSL_SP_SMALL */
}

/* Mul a by digit b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision digit.
 */
static void sp_3072_mul_d_24(sp_digit* r, const sp_digit* a,
        sp_digit b)
{
#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldr	x8, [%[a]]\n\t"
        "mul	x5, %[b], x8\n\t"
        "umulh	x3, %[b], x8\n\t"
        "mov	x4, xzr\n\t"
        "str	x5, [%[r]]\n\t"
        "mov	x5, xzr\n\t"
        "mov	x9, #8\n\t"
        "1:\n\t"
        "ldr	x8, [%[a], x9]\n\t"
        "mul	x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "str	x3, [%[r], x9]\n\t"
        "mov	x3, x4\n\t"
        "mov	x4, x5\n\t"
        "mov	x5, #0\n\t"
        "add	x9, x9, #8\n\t"
        "cmp	x9, 192\n\t"
        "b.lt	1b\n\t"
        "str	x3, [%[r], 192]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );
#else
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldp	x9, x10, [%[a]]\n\t"
        "mul	x3, %[b], x9\n\t"
        "umulh	x4, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "# A[1] * B\n\t"
        "str	x3, [%[r]]\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[2] * B\n\t"
        "ldp	x9, x10, [%[a], 16]\n\t"
        "str	x4, [%[r], 8]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[3] * B\n\t"
        "str	x5, [%[r], 16]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[4] * B\n\t"
        "ldp	x9, x10, [%[a], 32]\n\t"
        "str	x3, [%[r], 24]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[5] * B\n\t"
        "str	x4, [%[r], 32]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[6] * B\n\t"
        "ldp	x9, x10, [%[a], 48]\n\t"
        "str	x5, [%[r], 40]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[7] * B\n\t"
        "str	x3, [%[r], 48]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[8] * B\n\t"
        "ldp	x9, x10, [%[a], 64]\n\t"
        "str	x4, [%[r], 56]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[9] * B\n\t"
        "str	x5, [%[r], 64]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[10] * B\n\t"
        "ldp	x9, x10, [%[a], 80]\n\t"
        "str	x3, [%[r], 72]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[11] * B\n\t"
        "str	x4, [%[r], 80]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[12] * B\n\t"
        "ldp	x9, x10, [%[a], 96]\n\t"
        "str	x5, [%[r], 88]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[13] * B\n\t"
        "str	x3, [%[r], 96]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[14] * B\n\t"
        "ldp	x9, x10, [%[a], 112]\n\t"
        "str	x4, [%[r], 104]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[15] * B\n\t"
        "str	x5, [%[r], 112]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[16] * B\n\t"
        "ldp	x9, x10, [%[a], 128]\n\t"
        "str	x3, [%[r], 120]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[17] * B\n\t"
        "str	x4, [%[r], 128]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[18] * B\n\t"
        "ldp	x9, x10, [%[a], 144]\n\t"
        "str	x5, [%[r], 136]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[19] * B\n\t"
        "str	x3, [%[r], 144]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[20] * B\n\t"
        "ldp	x9, x10, [%[a], 160]\n\t"
        "str	x4, [%[r], 152]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[21] * B\n\t"
        "str	x5, [%[r], 160]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[22] * B\n\t"
        "ldp	x9, x10, [%[a], 176]\n\t"
        "str	x3, [%[r], 168]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[23] * B\n\t"
        "str	x4, [%[r], 176]\n\t"
        "mul	x6, %[b], x10\n\t"
        "adcs	x5, x5, x7\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "adc	x3, x3, x7\n\t"
        "str	x5, [%[r], 184]\n\t"
        "str	x3, [%[r], 192]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );
#endif
}

/* Divide the double width number (d1|d0) by the divisor. (d1|d0 / div)
 *
 * Assumes divisor has higest bit set.
 *
 * d1   The high order half of the number to divide.
 * d0   The low order half of the number to divide.
 * div  The divisor.
 * returns the result of the division.
 */
static sp_digit div_3072_word_24(sp_digit d1, sp_digit d0, sp_digit div)
{
    __asm__ __volatile__ (
        "lsr	x8, %[div], 32\n\t"
        "add	x5, x8, 1\n\t"

        "udiv	x3, %[d1], x5\n\t"
        "lsl	x7, %[div], 32\n\t"
        "movz	x9, #1, lsl 32\n\t"
        "lsl	x6, x3, 32\n\t"
        "mul	x4, %[div], x6\n\t"
        "umulh	x3, %[div], x6\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "cmp	%[d1], x5\n\t"
        "cset	x9, ge\n\t"
        "csetm	x10, ge\n\t"
        "lsl	x9, x9, #32\n\t"
        "and	x7, x7, x10\n\t"
        "and	x8, x8, x10\n\t"
        "subs	%[d0], %[d0], x7\n\t"
        "add	x6, x6, x9\n\t"
        "sbc	%[d1], %[d1], x8\n\t"

        "extr	x3, %[d1], %[d0], 32\n\t"

        "udiv	x3, x3, x5\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "extr	x3, %[d1], %[d0], 32\n\t"

        "udiv   x3, x3, x5\n\t"
        "add    x6, x6, x3\n\t"
        "mul    x4, %[div], x3\n\t"
        "sub    %[d0], %[d0], x4\n\t"

        "udiv	x3, %[d0], %[div]\n\t"
        "add	%[d1], x6, x3\n\t"

        : [d1] "+r" (d1), [d0] "+r" (d0)
        : [div] "r" (div)
        : "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return d1;
}

/* AND m into each word of a and store in r.
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * m  Mask to AND against each digit.
 */
static void sp_3072_mask_24(sp_digit* r, const sp_digit* a, sp_digit m)
{
#ifdef WOLFSSL_SP_SMALL
    int i;

    for (i=0; i<24; i++) {
        r[i] = a[i] & m;
    }
#else
    int i;

    for (i = 0; i < 24; i += 8) {
        r[i+0] = a[i+0] & m;
        r[i+1] = a[i+1] & m;
        r[i+2] = a[i+2] & m;
        r[i+3] = a[i+3] & m;
        r[i+4] = a[i+4] & m;
        r[i+5] = a[i+5] & m;
        r[i+6] = a[i+6] & m;
        r[i+7] = a[i+7] & m;
    }
#endif
}

/* Compare a with b in constant time.
 *
 * a  A single precision integer.
 * b  A single precision integer.
 * return -ve, 0 or +ve if a is less than, equal to or greater than b
 * respectively.
 */
static sp_int64 sp_3072_cmp_24(const sp_digit* a, const sp_digit* b)
{
#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "mov	x3, #0\n\t"
        "mov	x2, #-1\n\t"
        "mov	x10, #24\n\t"
        "add	%[a], %[a], #176\n\t"
        "add	%[b], %[b], #176\n\t"
        "1:\n\t"
        "ldp	x6, x7, [%[a]], -16\n\t"
        "ldp	x8, x9, [%[b]], -16\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x10, x10, #2\n\t"
        "b.ne	1b\n\t"
        "cmp	x2, #0\n\t"
        "cset	%[a], eq\n\t"
        "orr	%[a], %[a], x3\n\t"
        : [a] "+r" (a), [b] "+r" (b)
        :
        : "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );
#else
    __asm__ __volatile__ (
        "mov	x3, #0\n\t"
        "mov	x2, #-1\n\t"
        "ldp	x6, x7, [%[a], 176]\n\t"
        "ldp	x8, x9, [%[b], 176]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 160]\n\t"
        "ldp	x8, x9, [%[b], 160]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 144]\n\t"
        "ldp	x8, x9, [%[b], 144]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 128]\n\t"
        "ldp	x8, x9, [%[b], 128]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 112]\n\t"
        "ldp	x8, x9, [%[b], 112]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 96]\n\t"
        "ldp	x8, x9, [%[b], 96]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 80]\n\t"
        "ldp	x8, x9, [%[b], 80]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 64]\n\t"
        "ldp	x8, x9, [%[b], 64]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 48]\n\t"
        "ldp	x8, x9, [%[b], 48]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 32]\n\t"
        "ldp	x8, x9, [%[b], 32]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 16]\n\t"
        "ldp	x8, x9, [%[b], 16]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 0]\n\t"
        "ldp	x8, x9, [%[b], 0]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "cmp	x2, #0\n\t"
        "cset	%[a], eq\n\t"
        "orr	%[a], %[a], x3\n\t"
        : [a] "+r" (a)
        : [b] "r" (b)
        : "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );
#endif

    return (sp_int64)a;
}

/* Divide d in a and put remainder into r (m*d + r = a)
 * m is not calculated as it is not needed at this time.
 *
 * a  Number to be divided.
 * d  Number to divide with.
 * m  Multiplier result.
 * r  Remainder from the division.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_3072_div_24(const sp_digit* a, const sp_digit* d, sp_digit* m,
        sp_digit* r)
{
    sp_digit t1[48], t2[25];
    sp_digit div, r1;
    int i;

    (void)m;

    div = d[23];
    XMEMCPY(t1, a, sizeof(*t1) * 2 * 24);
    r1 = sp_3072_cmp_24(&t1[24], d) >= 0;
    sp_3072_cond_sub_24(&t1[24], &t1[24], d, (sp_digit)0 - r1);
    for (i = 23; i >= 0; i--) {
        volatile sp_digit mask = (sp_digit)0 - (t1[24 + i] == div);
        sp_digit hi = t1[24 + i] + mask;
        r1 = div_3072_word_24(hi, t1[24 + i - 1], div);
        r1 |= mask;

        sp_3072_mul_d_24(t2, d, r1);
        t1[24 + i] += sp_3072_sub_in_place_24(&t1[i], t2);
        t1[24 + i] -= t2[24];
        sp_3072_mask_24(t2, d, t1[24 + i]);
        t1[24 + i] += sp_3072_add_24(&t1[i], &t1[i], t2);
        sp_3072_mask_24(t2, d, t1[24 + i]);
        t1[24 + i] += sp_3072_add_24(&t1[i], &t1[i], t2);
    }

    r1 = sp_3072_cmp_24(t1, d) >= 0;
    sp_3072_cond_sub_24(r, t1, d, (sp_digit)0 - r1);

    return MP_OKAY;
}

/* Reduce a modulo m into r. (r = a mod m)
 *
 * r  A single precision number that is the reduced result.
 * a  A single precision number that is to be reduced.
 * m  A single precision number that is the modulus to reduce with.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_3072_mod_24(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    return sp_3072_div_24(a, m, NULL, r);
}

#ifdef WOLFSSL_SP_SMALL
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns  0 on success.
 * returns  MEMORY_E on dynamic memory allocation failure.
 * returns  MP_VAL when base is even or exponent is 0.
 */
static int sp_3072_mod_exp_24(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* td = NULL;
#else
    sp_digit td[16 * 48];
#endif
    sp_digit* t[16];
    sp_digit* norm = NULL;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c;
    byte y;
    int err = MP_OKAY;

    if (bits == 0) {
        err = MP_VAL;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        td = (sp_digit*)XMALLOC(sizeof(sp_digit) * (16 * 48), NULL,
                                DYNAMIC_TYPE_TMP_BUFFER);
        if (td == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        norm = td;
        for (i=0; i<16; i++) {
            t[i] = td + i * 48;
        }

        sp_3072_mont_setup(m, &mp);
        sp_3072_mont_norm_24(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 24U);
        if (reduceA != 0) {
            err = sp_3072_mod_24(t[1] + 24, a, m);
            if (err == MP_OKAY) {
                err = sp_3072_mod_24(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 24, a, sizeof(sp_digit) * 24);
            err = sp_3072_mod_24(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_3072_mont_sqr_24(t[ 2], t[ 1], m, mp);
        sp_3072_mont_mul_24(t[ 3], t[ 2], t[ 1], m, mp);
        sp_3072_mont_sqr_24(t[ 4], t[ 2], m, mp);
        sp_3072_mont_mul_24(t[ 5], t[ 3], t[ 2], m, mp);
        sp_3072_mont_sqr_24(t[ 6], t[ 3], m, mp);
        sp_3072_mont_mul_24(t[ 7], t[ 4], t[ 3], m, mp);
        sp_3072_mont_sqr_24(t[ 8], t[ 4], m, mp);
        sp_3072_mont_mul_24(t[ 9], t[ 5], t[ 4], m, mp);
        sp_3072_mont_sqr_24(t[10], t[ 5], m, mp);
        sp_3072_mont_mul_24(t[11], t[ 6], t[ 5], m, mp);
        sp_3072_mont_sqr_24(t[12], t[ 6], m, mp);
        sp_3072_mont_mul_24(t[13], t[ 7], t[ 6], m, mp);
        sp_3072_mont_sqr_24(t[14], t[ 7], m, mp);
        sp_3072_mont_mul_24(t[15], t[ 8], t[ 7], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 4;
        if (c == 64) {
            c = 60;
        }
        if (c < 0) {
            /* Number of bits in top word is less than number needed. */
            c = -c;
            y = (byte)(n << c);
            n = e[i--];
            y |= (byte)(n >> (64 - c));
            n <<= c;
            c = 64 - c;
        }
        else if (c == 0) {
            /* All bits in top word used. */
            y = (byte)n;
        }
        else {
            y = (byte)(n >> c);
            n <<= 64 - c;
        }
        XMEMCPY(r, t[y], sizeof(sp_digit) * 24);
        for (; i>=0 || c>=4; ) {
            if (c == 0) {
                n = e[i--];
                y = (byte)(n >> 60);
                n <<= 4;
                c = 60;
            }
            else if (c < 4) {
                y = (byte)(n >> 60);
                n = e[i--];
                c = 4 - c;
                y |= (byte)(n >> (64 - c));
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (byte)((n >> 60) & 0xf);
                n <<= 4;
                c -= 4;
            }

            sp_3072_mont_sqr_24(r, r, m, mp);
            sp_3072_mont_sqr_24(r, r, m, mp);
            sp_3072_mont_sqr_24(r, r, m, mp);
            sp_3072_mont_sqr_24(r, r, m, mp);

            sp_3072_mont_mul_24(r, r, t[y], m, mp);
        }

        XMEMSET(&r[24], 0, sizeof(sp_digit) * 24U);
        sp_3072_mont_reduce_24(r, m, mp);

        mask = 0 - (sp_3072_cmp_24(r, m) >= 0);
        sp_3072_cond_sub_24(r, r, m, mask);
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (td != NULL)
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return err;
}
#else
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns  0 on success.
 * returns  MEMORY_E on dynamic memory allocation failure.
 * returns  MP_VAL when base is even or exponent is 0.
 */
static int sp_3072_mod_exp_24(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* td = NULL;
#else
    sp_digit td[32 * 48];
#endif
    sp_digit* t[32];
    sp_digit* norm = NULL;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c;
    byte y;
    int err = MP_OKAY;

    if (bits == 0) {
        err = MP_VAL;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        td = (sp_digit*)XMALLOC(sizeof(sp_digit) * (32 * 48), NULL,
                                DYNAMIC_TYPE_TMP_BUFFER);
        if (td == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        norm = td;
        for (i=0; i<32; i++) {
            t[i] = td + i * 48;
        }

        sp_3072_mont_setup(m, &mp);
        sp_3072_mont_norm_24(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 24U);
        if (reduceA != 0) {
            err = sp_3072_mod_24(t[1] + 24, a, m);
            if (err == MP_OKAY) {
                err = sp_3072_mod_24(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 24, a, sizeof(sp_digit) * 24);
            err = sp_3072_mod_24(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_3072_mont_sqr_24(t[ 2], t[ 1], m, mp);
        sp_3072_mont_mul_24(t[ 3], t[ 2], t[ 1], m, mp);
        sp_3072_mont_sqr_24(t[ 4], t[ 2], m, mp);
        sp_3072_mont_mul_24(t[ 5], t[ 3], t[ 2], m, mp);
        sp_3072_mont_sqr_24(t[ 6], t[ 3], m, mp);
        sp_3072_mont_mul_24(t[ 7], t[ 4], t[ 3], m, mp);
        sp_3072_mont_sqr_24(t[ 8], t[ 4], m, mp);
        sp_3072_mont_mul_24(t[ 9], t[ 5], t[ 4], m, mp);
        sp_3072_mont_sqr_24(t[10], t[ 5], m, mp);
        sp_3072_mont_mul_24(t[11], t[ 6], t[ 5], m, mp);
        sp_3072_mont_sqr_24(t[12], t[ 6], m, mp);
        sp_3072_mont_mul_24(t[13], t[ 7], t[ 6], m, mp);
        sp_3072_mont_sqr_24(t[14], t[ 7], m, mp);
        sp_3072_mont_mul_24(t[15], t[ 8], t[ 7], m, mp);
        sp_3072_mont_sqr_24(t[16], t[ 8], m, mp);
        sp_3072_mont_mul_24(t[17], t[ 9], t[ 8], m, mp);
        sp_3072_mont_sqr_24(t[18], t[ 9], m, mp);
        sp_3072_mont_mul_24(t[19], t[10], t[ 9], m, mp);
        sp_3072_mont_sqr_24(t[20], t[10], m, mp);
        sp_3072_mont_mul_24(t[21], t[11], t[10], m, mp);
        sp_3072_mont_sqr_24(t[22], t[11], m, mp);
        sp_3072_mont_mul_24(t[23], t[12], t[11], m, mp);
        sp_3072_mont_sqr_24(t[24], t[12], m, mp);
        sp_3072_mont_mul_24(t[25], t[13], t[12], m, mp);
        sp_3072_mont_sqr_24(t[26], t[13], m, mp);
        sp_3072_mont_mul_24(t[27], t[14], t[13], m, mp);
        sp_3072_mont_sqr_24(t[28], t[14], m, mp);
        sp_3072_mont_mul_24(t[29], t[15], t[14], m, mp);
        sp_3072_mont_sqr_24(t[30], t[15], m, mp);
        sp_3072_mont_mul_24(t[31], t[16], t[15], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 5;
        if (c == 64) {
            c = 59;
        }
        if (c < 0) {
            /* Number of bits in top word is less than number needed. */
            c = -c;
            y = (byte)(n << c);
            n = e[i--];
            y |= (byte)(n >> (64 - c));
            n <<= c;
            c = 64 - c;
        }
        else if (c == 0) {
            /* All bits in top word used. */
            y = (byte)n;
        }
        else {
            y = (byte)(n >> c);
            n <<= 64 - c;
        }
        XMEMCPY(r, t[y], sizeof(sp_digit) * 24);
        for (; i>=0 || c>=5; ) {
            if (c == 0) {
                n = e[i--];
                y = (byte)(n >> 59);
                n <<= 5;
                c = 59;
            }
            else if (c < 5) {
                y = (byte)(n >> 59);
                n = e[i--];
                c = 5 - c;
                y |= (byte)(n >> (64 - c));
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (byte)((n >> 59) & 0x1f);
                n <<= 5;
                c -= 5;
            }

            sp_3072_mont_sqr_24(r, r, m, mp);
            sp_3072_mont_sqr_24(r, r, m, mp);
            sp_3072_mont_sqr_24(r, r, m, mp);
            sp_3072_mont_sqr_24(r, r, m, mp);
            sp_3072_mont_sqr_24(r, r, m, mp);

            sp_3072_mont_mul_24(r, r, t[y], m, mp);
        }

        XMEMSET(&r[24], 0, sizeof(sp_digit) * 24U);
        sp_3072_mont_reduce_24(r, m, mp);

        mask = 0 - (sp_3072_cmp_24(r, m) >= 0);
        sp_3072_cond_sub_24(r, r, m, mask);
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (td != NULL)
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return err;
}
#endif /* WOLFSSL_SP_SMALL */

#endif /* (WOLFSSL_HAVE_SP_RSA & !WOLFSSL_RSA_PUBLIC_ONLY) | WOLFSSL_HAVE_SP_DH */

#if (defined(WOLFSSL_HAVE_SP_RSA) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)) || defined(WOLFSSL_HAVE_SP_DH)
/* r = 2^n mod m where n is the number of bits to reduce by.
 * Given m must be 3072 bits, just need to subtract.
 *
 * r  A single precision number.
 * m  A single precision number.
 */
static void sp_3072_mont_norm_48(sp_digit* r, const sp_digit* m)
{
    XMEMSET(r, 0, sizeof(sp_digit) * 48);

    /* r = 2^n mod m */
    sp_3072_sub_in_place_48(r, m);
}

#endif /* (WOLFSSL_HAVE_SP_RSA & !WOLFSSL_RSA_PUBLIC_ONLY) | WOLFSSL_HAVE_SP_DH */
/* Reduce the number back to 3072 bits using Montgomery reduction.
 *
 * a   A single precision number to reduce in place.
 * m   The single precision number representing the modulus.
 * mp  The digit representing the negative inverse of m mod 2^n.
 */
SP_NOINLINE static void sp_3072_mont_reduce_48(sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    __asm__ __volatile__ (
        "ldp	x11, x12, [%[a], 0]\n\t"
        "ldp	x13, x14, [%[a], 16]\n\t"
        "ldp	x15, x16, [%[a], 32]\n\t"
        "ldp	x17, x19, [%[a], 48]\n\t"
        "ldp	x20, x21, [%[a], 64]\n\t"
        "ldp	x22, x23, [%[a], 80]\n\t"
        "# No carry yet\n\t"
        "mov	x3, xzr\n\t"
        "# i = 0..47\n\t"
        "mov	x4, 48\n\t"
        "\n1:\n\t"
        "# mu = a[i] * mp\n\t"
        "mul	x10, %[mp], x11\n\t"
        "ldp	x24, x25, [%[m], 0]\n\t"
        "ldp	x26, x27, [%[m], 16]\n\t"
        "# a[i+0] += m[0] * mu\n\t"
        "mul	x5, x24, x10\n\t"
        "umulh	x6, x24, x10\n\t"
        "# a[i+1] += m[1] * mu\n\t"
        "adds	x11, x11, x5\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x11, x12, x5\n\t"
        "# a[i+2] += m[2] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x11, x11, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x12, x13, x5\n\t"
        "# a[i+3] += m[3] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x12, x12, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x13, x14, x5\n\t"
        "ldp	x24, x25, [%[m], 32]\n\t"
        "ldp	x26, x27, [%[m], 48]\n\t"
        "# a[i+4] += m[4] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x13, x13, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x14, x15, x5\n\t"
        "# a[i+5] += m[5] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x14, x14, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x15, x16, x5\n\t"
        "# a[i+6] += m[6] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x15, x15, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x16, x17, x5\n\t"
        "# a[i+7] += m[7] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x16, x16, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x17, x19, x5\n\t"
        "ldp	x24, x25, [%[m], 64]\n\t"
        "ldp	x26, x27, [%[m], 80]\n\t"
        "# a[i+8] += m[8] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x17, x17, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x19, x20, x5\n\t"
        "# a[i+9] += m[9] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x19, x19, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x20, x21, x5\n\t"
        "# a[i+10] += m[10] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x20, x20, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x21, x22, x5\n\t"
        "# a[i+11] += m[11] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x21, x21, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x22, x23, x5\n\t"
        "ldp	x24, x25, [%[m], 96]\n\t"
        "ldp	x26, x27, [%[m], 112]\n\t"
        "# a[i+12] += m[12] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x22, x22, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "ldr	x23, [%[a], 96]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x23, x23, x5\n\t"
        "# a[i+13] += m[13] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x23, x23, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "ldp	x8, x9, [%[a], 104]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+14] += m[14] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 104]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+15] += m[15] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 112]\n\t"
        "ldp	x8, x9, [%[a], 120]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 128]\n\t"
        "ldp	x26, x27, [%[m], 144]\n\t"
        "# a[i+16] += m[16] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 120]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+17] += m[17] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 128]\n\t"
        "ldp	x8, x9, [%[a], 136]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+18] += m[18] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 136]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+19] += m[19] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 144]\n\t"
        "ldp	x8, x9, [%[a], 152]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 160]\n\t"
        "ldp	x26, x27, [%[m], 176]\n\t"
        "# a[i+20] += m[20] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 152]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+21] += m[21] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 160]\n\t"
        "ldp	x8, x9, [%[a], 168]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+22] += m[22] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 168]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+23] += m[23] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 176]\n\t"
        "ldp	x8, x9, [%[a], 184]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 192]\n\t"
        "ldp	x26, x27, [%[m], 208]\n\t"
        "# a[i+24] += m[24] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 184]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+25] += m[25] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 192]\n\t"
        "ldp	x8, x9, [%[a], 200]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+26] += m[26] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 200]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+27] += m[27] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 208]\n\t"
        "ldp	x8, x9, [%[a], 216]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 224]\n\t"
        "ldp	x26, x27, [%[m], 240]\n\t"
        "# a[i+28] += m[28] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 216]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+29] += m[29] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 224]\n\t"
        "ldp	x8, x9, [%[a], 232]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+30] += m[30] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 232]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+31] += m[31] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 240]\n\t"
        "ldp	x8, x9, [%[a], 248]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 256]\n\t"
        "ldp	x26, x27, [%[m], 272]\n\t"
        "# a[i+32] += m[32] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 248]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+33] += m[33] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 256]\n\t"
        "ldp	x8, x9, [%[a], 264]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+34] += m[34] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 264]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+35] += m[35] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 272]\n\t"
        "ldp	x8, x9, [%[a], 280]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 288]\n\t"
        "ldp	x26, x27, [%[m], 304]\n\t"
        "# a[i+36] += m[36] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 280]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+37] += m[37] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 288]\n\t"
        "ldp	x8, x9, [%[a], 296]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+38] += m[38] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 296]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+39] += m[39] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 304]\n\t"
        "ldp	x8, x9, [%[a], 312]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 320]\n\t"
        "ldp	x26, x27, [%[m], 336]\n\t"
        "# a[i+40] += m[40] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 312]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+41] += m[41] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 320]\n\t"
        "ldp	x8, x9, [%[a], 328]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+42] += m[42] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 328]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+43] += m[43] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 336]\n\t"
        "ldp	x8, x9, [%[a], 344]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 352]\n\t"
        "ldp	x26, x27, [%[m], 368]\n\t"
        "# a[i+44] += m[44] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 344]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+45] += m[45] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 352]\n\t"
        "ldp	x8, x9, [%[a], 360]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+46] += m[46] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 360]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+47] += m[47] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 368]\n\t"
        "umulh	x7, x27, x10\n\t"
        "ldp	x8, x9, [%[a], 376]\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x7, x7, x3\n\t"
        "cset	x3, cs\n\t"
        "adds	x8, x8, x5\n\t"
        "str	x8, [%[a], 376]\n\t"
        "adcs	x9, x9, x7\n\t"
        "str	x9, [%[a], 384]\n\t"
        "adc	x3, x3, xzr\n\t"
        "subs	x4, x4, 1\n\t"
        "add	%[a], %[a], 8\n\t"
        "b.ne	1b\n\t"
        "# Create mask\n\t"
        "neg	x3, x3\n\t"
        "mov   %[mp], %[a]\n\t"
        "sub	%[a], %[a], 384\n\t"
        "# Subtract masked modulus\n\t"
        "ldp	x4, x5, [%[m], 0]\n\t"
        "ldp	x6, x7, [%[m], 16]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "subs	x11, x11, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x12, x12, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x13, x13, x6\n\t"
        "stp	x11, x12, [%[a], 0]\n\t"
        "sbcs	x14, x14, x7\n\t"
        "stp	x13, x14, [%[a], 16]\n\t"
        "ldp	x4, x5, [%[m], 32]\n\t"
        "ldp	x6, x7, [%[m], 48]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x15, x15, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x16, x16, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x17, x17, x6\n\t"
        "stp	x15, x16, [%[a], 32]\n\t"
        "sbcs	x19, x19, x7\n\t"
        "stp	x17, x19, [%[a], 48]\n\t"
        "ldp	x4, x5, [%[m], 64]\n\t"
        "ldp	x6, x7, [%[m], 80]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x20, x20, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x21, x21, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x22, x22, x6\n\t"
        "stp	x20, x21, [%[a], 64]\n\t"
        "sbcs	x23, x23, x7\n\t"
        "stp	x22, x23, [%[a], 80]\n\t"
        "ldp	x4, x5, [%[m], 96]\n\t"
        "ldp	x6, x7, [%[m], 112]\n\t"
        "ldp	x8, x9, [%[mp], 96]\n\t"
        "ldp	x10, x11, [%[mp], 112]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 96]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 112]\n\t"
        "ldp	x4, x5, [%[m], 128]\n\t"
        "ldp	x6, x7, [%[m], 144]\n\t"
        "ldp	x8, x9, [%[mp], 128]\n\t"
        "ldp	x10, x11, [%[mp], 144]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 128]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 144]\n\t"
        "ldp	x4, x5, [%[m], 160]\n\t"
        "ldp	x6, x7, [%[m], 176]\n\t"
        "ldp	x8, x9, [%[mp], 160]\n\t"
        "ldp	x10, x11, [%[mp], 176]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 160]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 176]\n\t"
        "ldp	x4, x5, [%[m], 192]\n\t"
        "ldp	x6, x7, [%[m], 208]\n\t"
        "ldp	x8, x9, [%[mp], 192]\n\t"
        "ldp	x10, x11, [%[mp], 208]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 192]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 208]\n\t"
        "ldp	x4, x5, [%[m], 224]\n\t"
        "ldp	x6, x7, [%[m], 240]\n\t"
        "ldp	x8, x9, [%[mp], 224]\n\t"
        "ldp	x10, x11, [%[mp], 240]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 224]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 240]\n\t"
        "ldp	x4, x5, [%[m], 256]\n\t"
        "ldp	x6, x7, [%[m], 272]\n\t"
        "ldp	x8, x9, [%[mp], 256]\n\t"
        "ldp	x10, x11, [%[mp], 272]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 256]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 272]\n\t"
        "ldp	x4, x5, [%[m], 288]\n\t"
        "ldp	x6, x7, [%[m], 304]\n\t"
        "ldp	x8, x9, [%[mp], 288]\n\t"
        "ldp	x10, x11, [%[mp], 304]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 288]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 304]\n\t"
        "ldp	x4, x5, [%[m], 320]\n\t"
        "ldp	x6, x7, [%[m], 336]\n\t"
        "ldp	x8, x9, [%[mp], 320]\n\t"
        "ldp	x10, x11, [%[mp], 336]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 320]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 336]\n\t"
        "ldp	x4, x5, [%[m], 352]\n\t"
        "ldp	x6, x7, [%[m], 368]\n\t"
        "ldp	x8, x9, [%[mp], 352]\n\t"
        "ldp	x10, x11, [%[mp], 368]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 352]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 368]\n\t"
        : [a] "+r" (a), [mp] "+r" (mp)
        : [m] "r" (m)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x10", "x8", "x9", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x19", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27"
    );

}

/* Multiply two Montgomery form numbers mod the modulus (prime).
 * (r = a * b mod m)
 *
 * r   Result of multiplication.
 * a   First number to multiply in Montgomery form.
 * b   Second number to multiply in Montgomery form.
 * m   Modulus (prime).
 * mp  Montgomery mulitplier.
 */
SP_NOINLINE static void sp_3072_mont_mul_48(sp_digit* r, const sp_digit* a,
        const sp_digit* b, const sp_digit* m, sp_digit mp)
{
    sp_3072_mul_48(r, a, b);
    sp_3072_mont_reduce_48(r, m, mp);
}

/* Square the Montgomery form number. (r = a * a mod m)
 *
 * r   Result of squaring.
 * a   Number to square in Montgomery form.
 * m   Modulus (prime).
 * mp  Montgomery mulitplier.
 */
SP_NOINLINE static void sp_3072_mont_sqr_48(sp_digit* r, const sp_digit* a,
        const sp_digit* m, sp_digit mp)
{
    sp_3072_sqr_48(r, a);
    sp_3072_mont_reduce_48(r, m, mp);
}

#ifdef WOLFSSL_SP_SMALL
/* Sub b from a into r. (r = a - b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_3072_sub_48(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x11, %[a], 384\n\t"
        "\n1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldp	x3, x4, [%[a]], #16\n\t"
        "ldp	x5, x6, [%[a]], #16\n\t"
        "ldp	x7, x8, [%[b]], #16\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x9, x10, [%[b]], #16\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r]], #16\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r]], #16\n\t"
        "csetm	%[c], cc\n\t"
        "cmp	%[a], x11\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [r] "+r" (r), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    return c;
}

#else
/* Sub b from a into r. (r = a - b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_3072_sub_48(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "subs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x7, x8, [%[b], 64]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 80]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "ldp	x3, x4, [%[a], 96]\n\t"
        "ldp	x7, x8, [%[b], 96]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 112]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 112]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 96]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 112]\n\t"
        "ldp	x3, x4, [%[a], 128]\n\t"
        "ldp	x7, x8, [%[b], 128]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 144]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 144]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 128]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 144]\n\t"
        "ldp	x3, x4, [%[a], 160]\n\t"
        "ldp	x7, x8, [%[b], 160]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 176]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 176]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 160]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 176]\n\t"
        "ldp	x3, x4, [%[a], 192]\n\t"
        "ldp	x7, x8, [%[b], 192]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 208]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 208]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 192]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 208]\n\t"
        "ldp	x3, x4, [%[a], 224]\n\t"
        "ldp	x7, x8, [%[b], 224]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 240]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 240]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 224]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 240]\n\t"
        "ldp	x3, x4, [%[a], 256]\n\t"
        "ldp	x7, x8, [%[b], 256]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 272]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 272]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 256]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 272]\n\t"
        "ldp	x3, x4, [%[a], 288]\n\t"
        "ldp	x7, x8, [%[b], 288]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 304]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 304]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 288]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 304]\n\t"
        "ldp	x3, x4, [%[a], 320]\n\t"
        "ldp	x7, x8, [%[b], 320]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 336]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 336]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 320]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 336]\n\t"
        "ldp	x3, x4, [%[a], 352]\n\t"
        "ldp	x7, x8, [%[b], 352]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 368]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 368]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 352]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 368]\n\t"
        "csetm	%[r], cc\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return (sp_digit)r;
}

#endif /* WOLFSSL_SP_SMALL */
/* Divide the double width number (d1|d0) by the divisor. (d1|d0 / div)
 *
 * Assumes divisor has higest bit set.
 *
 * d1   The high order half of the number to divide.
 * d0   The low order half of the number to divide.
 * div  The divisor.
 * returns the result of the division.
 */
static sp_digit div_3072_word_48_cond(sp_digit d1, sp_digit d0, sp_digit div)
{
    __asm__ __volatile__ (
        "lsr	x8, %[div], 32\n\t"
        "add	x5, x8, 1\n\t"

        "udiv	x3, %[d1], x5\n\t"
        "lsl	x7, %[div], 32\n\t"
        "movz	x9, #1, lsl 32\n\t"
        "lsl	x6, x3, 32\n\t"
        "mul	x4, %[div], x6\n\t"
        "umulh	x3, %[div], x6\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "cmp	%[d1], x5\n\t"
        "b.lt	1f\n\t"
        "subs	%[d0], %[d0], x7\n\t"
        "add	x6, x6, x9\n\t"
        "sbc	%[d1], %[d1], x8\n\t"
        "1:\n\t"

        "extr	x3, %[d1], %[d0], 32\n\t"

        "udiv	x3, x3, x5\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "extr	x3, %[d1], %[d0], 32\n\t"

        "cmp	x3, x5\n\t"
        "b.lt	2f\n\t"
        "udiv   x3, x3, x5\n\t"
        "add    x6, x6, x3\n\t"
        "mul    x4, %[div], x3\n\t"
        "sub    %[d0], %[d0], x4\n\t"
        "2:\n\t"

        "udiv	x3, %[d0], %[div]\n\t"
        "add	%[d1], x6, x3\n\t"

        : [d1] "+r" (d1), [d0] "+r" (d0)
        : [div] "r" (div)
        : "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );

    return d1;
}

/* Divide d in a and put remainder into r (m*d + r = a)
 * m is not calculated as it is not needed at this time.
 *
 * a  Number to be divided.
 * d  Number to divide with.
 * m  Multiplier result.
 * r  Remainder from the division.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_3072_div_48_cond(const sp_digit* a, const sp_digit* d, sp_digit* m,
        sp_digit* r)
{
    sp_digit t1[96], t2[49];
    sp_digit div, r1;
    int i;

    (void)m;

    div = d[47];
    XMEMCPY(t1, a, sizeof(*t1) * 2 * 48);
    for (i = 47; i > 0; i--) {
        if (t1[i + 48] != d[i])
            break;
    }
    if (t1[i + 48] >= d[i]) {
        sp_3072_sub_in_place_48(&t1[48], d);
    }
    for (i = 47; i >= 0; i--) {
        if (t1[48 + i] == div) {
            r1 = SP_DIGIT_MAX;
        }
        else {
            r1 = div_3072_word_48_cond(t1[48 + i], t1[48 + i - 1], div);
        }

        sp_3072_mul_d_48(t2, d, r1);
        t1[48 + i] += sp_3072_sub_in_place_48(&t1[i], t2);
        t1[48 + i] -= t2[48];
        if (t1[48 + i] != 0) {
            t1[48 + i] += sp_3072_add_48(&t1[i], &t1[i], d);
            if (t1[48 + i] != 0)
                t1[48 + i] += sp_3072_add_48(&t1[i], &t1[i], d);
        }
    }

    for (i = 47; i > 0; i--) {
        if (t1[i] != d[i])
            break;
    }
    if (t1[i] >= d[i]) {
        sp_3072_sub_48(r, t1, d);
    }
    else {
        XMEMCPY(r, t1, sizeof(*t1) * 48);
    }

    return MP_OKAY;
}

/* Reduce a modulo m into r. (r = a mod m)
 *
 * r  A single precision number that is the reduced result.
 * a  A single precision number that is to be reduced.
 * m  A single precision number that is the modulus to reduce with.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_3072_mod_48_cond(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    return sp_3072_div_48_cond(a, m, NULL, r);
}

#if (defined(WOLFSSL_HAVE_SP_RSA) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)) || defined(WOLFSSL_HAVE_SP_DH)
/* Conditionally subtract b from a using the mask m.
 * m is -1 to subtract and 0 when not copying.
 *
 * r  A single precision number representing condition subtract result.
 * a  A single precision number to subtract from.
 * b  A single precision number to subtract.
 * m  Mask value to apply.
 */
static sp_digit sp_3072_cond_sub_48(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
#ifdef WOLFSSL_SP_SMALL
    sp_digit c = 0;

    __asm__ __volatile__ (
        "mov	x8, #0\n\t"
        "1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldr	x4, [%[a], x8]\n\t"
        "ldr	x5, [%[b], x8]\n\t"
        "and	x5, x5, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "csetm	%[c], cc\n\t"
        "str	x4, [%[r], x8]\n\t"
        "add	x8, x8, #8\n\t"
        "cmp	x8, 384\n\t"
        "b.lt	1b\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8", "x9", "x10", "x11", "x12"
    );

    return c;
#else
    __asm__ __volatile__ (

        "ldp	x5, x7, [%[b], 0]\n\t"
        "ldp	x11, x12, [%[b], 16]\n\t"
        "ldp	x4, x6, [%[a], 0]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 16]\n\t"
        "and	x7, x7, %[m]\n\t"
        "subs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 0]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 16]\n\t"
        "ldp	x5, x7, [%[b], 32]\n\t"
        "ldp	x11, x12, [%[b], 48]\n\t"
        "ldp	x4, x6, [%[a], 32]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 48]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 32]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 48]\n\t"
        "ldp	x5, x7, [%[b], 64]\n\t"
        "ldp	x11, x12, [%[b], 80]\n\t"
        "ldp	x4, x6, [%[a], 64]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 80]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 64]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 80]\n\t"
        "ldp	x5, x7, [%[b], 96]\n\t"
        "ldp	x11, x12, [%[b], 112]\n\t"
        "ldp	x4, x6, [%[a], 96]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 112]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 96]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 112]\n\t"
        "ldp	x5, x7, [%[b], 128]\n\t"
        "ldp	x11, x12, [%[b], 144]\n\t"
        "ldp	x4, x6, [%[a], 128]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 144]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 128]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 144]\n\t"
        "ldp	x5, x7, [%[b], 160]\n\t"
        "ldp	x11, x12, [%[b], 176]\n\t"
        "ldp	x4, x6, [%[a], 160]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 176]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 160]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 176]\n\t"
        "ldp	x5, x7, [%[b], 192]\n\t"
        "ldp	x11, x12, [%[b], 208]\n\t"
        "ldp	x4, x6, [%[a], 192]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 208]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 192]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 208]\n\t"
        "ldp	x5, x7, [%[b], 224]\n\t"
        "ldp	x11, x12, [%[b], 240]\n\t"
        "ldp	x4, x6, [%[a], 224]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 240]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 224]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 240]\n\t"
        "ldp	x5, x7, [%[b], 256]\n\t"
        "ldp	x11, x12, [%[b], 272]\n\t"
        "ldp	x4, x6, [%[a], 256]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 272]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 256]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 272]\n\t"
        "ldp	x5, x7, [%[b], 288]\n\t"
        "ldp	x11, x12, [%[b], 304]\n\t"
        "ldp	x4, x6, [%[a], 288]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 304]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 288]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 304]\n\t"
        "ldp	x5, x7, [%[b], 320]\n\t"
        "ldp	x11, x12, [%[b], 336]\n\t"
        "ldp	x4, x6, [%[a], 320]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 336]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 320]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 336]\n\t"
        "ldp	x5, x7, [%[b], 352]\n\t"
        "ldp	x11, x12, [%[b], 368]\n\t"
        "ldp	x4, x6, [%[a], 352]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 368]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 352]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 368]\n\t"
        "csetm	%[r], cc\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8", "x9", "x10", "x11", "x12"
    );

    return (sp_digit)r;
#endif /* WOLFSSL_SP_SMALL */
}

/* Divide the double width number (d1|d0) by the divisor. (d1|d0 / div)
 *
 * Assumes divisor has higest bit set.
 *
 * d1   The high order half of the number to divide.
 * d0   The low order half of the number to divide.
 * div  The divisor.
 * returns the result of the division.
 */
static sp_digit div_3072_word_48(sp_digit d1, sp_digit d0, sp_digit div)
{
    __asm__ __volatile__ (
        "lsr	x8, %[div], 32\n\t"
        "add	x5, x8, 1\n\t"

        "udiv	x3, %[d1], x5\n\t"
        "lsl	x7, %[div], 32\n\t"
        "movz	x9, #1, lsl 32\n\t"
        "lsl	x6, x3, 32\n\t"
        "mul	x4, %[div], x6\n\t"
        "umulh	x3, %[div], x6\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "cmp	%[d1], x5\n\t"
        "cset	x9, ge\n\t"
        "csetm	x10, ge\n\t"
        "lsl	x9, x9, #32\n\t"
        "and	x7, x7, x10\n\t"
        "and	x8, x8, x10\n\t"
        "subs	%[d0], %[d0], x7\n\t"
        "add	x6, x6, x9\n\t"
        "sbc	%[d1], %[d1], x8\n\t"

        "extr	x3, %[d1], %[d0], 32\n\t"

        "udiv	x3, x3, x5\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "extr	x3, %[d1], %[d0], 32\n\t"

        "udiv   x3, x3, x5\n\t"
        "add    x6, x6, x3\n\t"
        "mul    x4, %[div], x3\n\t"
        "sub    %[d0], %[d0], x4\n\t"

        "udiv	x3, %[d0], %[div]\n\t"
        "add	%[d1], x6, x3\n\t"

        : [d1] "+r" (d1), [d0] "+r" (d0)
        : [div] "r" (div)
        : "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return d1;
}

/* AND m into each word of a and store in r.
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * m  Mask to AND against each digit.
 */
static void sp_3072_mask_48(sp_digit* r, const sp_digit* a, sp_digit m)
{
#ifdef WOLFSSL_SP_SMALL
    int i;

    for (i=0; i<48; i++) {
        r[i] = a[i] & m;
    }
#else
    int i;

    for (i = 0; i < 48; i += 8) {
        r[i+0] = a[i+0] & m;
        r[i+1] = a[i+1] & m;
        r[i+2] = a[i+2] & m;
        r[i+3] = a[i+3] & m;
        r[i+4] = a[i+4] & m;
        r[i+5] = a[i+5] & m;
        r[i+6] = a[i+6] & m;
        r[i+7] = a[i+7] & m;
    }
#endif
}

/* Compare a with b in constant time.
 *
 * a  A single precision integer.
 * b  A single precision integer.
 * return -ve, 0 or +ve if a is less than, equal to or greater than b
 * respectively.
 */
static sp_int64 sp_3072_cmp_48(const sp_digit* a, const sp_digit* b)
{
#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "mov	x3, #0\n\t"
        "mov	x2, #-1\n\t"
        "mov	x10, #48\n\t"
        "add	%[a], %[a], #368\n\t"
        "add	%[b], %[b], #368\n\t"
        "1:\n\t"
        "ldp	x6, x7, [%[a]], -16\n\t"
        "ldp	x8, x9, [%[b]], -16\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x10, x10, #2\n\t"
        "b.ne	1b\n\t"
        "cmp	x2, #0\n\t"
        "cset	%[a], eq\n\t"
        "orr	%[a], %[a], x3\n\t"
        : [a] "+r" (a), [b] "+r" (b)
        :
        : "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );
#else
    __asm__ __volatile__ (
        "mov	x3, #0\n\t"
        "mov	x2, #-1\n\t"
        "ldp	x6, x7, [%[a], 368]\n\t"
        "ldp	x8, x9, [%[b], 368]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 352]\n\t"
        "ldp	x8, x9, [%[b], 352]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 336]\n\t"
        "ldp	x8, x9, [%[b], 336]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 320]\n\t"
        "ldp	x8, x9, [%[b], 320]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 304]\n\t"
        "ldp	x8, x9, [%[b], 304]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 288]\n\t"
        "ldp	x8, x9, [%[b], 288]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 272]\n\t"
        "ldp	x8, x9, [%[b], 272]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 256]\n\t"
        "ldp	x8, x9, [%[b], 256]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 240]\n\t"
        "ldp	x8, x9, [%[b], 240]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 224]\n\t"
        "ldp	x8, x9, [%[b], 224]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 208]\n\t"
        "ldp	x8, x9, [%[b], 208]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 192]\n\t"
        "ldp	x8, x9, [%[b], 192]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 176]\n\t"
        "ldp	x8, x9, [%[b], 176]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 160]\n\t"
        "ldp	x8, x9, [%[b], 160]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 144]\n\t"
        "ldp	x8, x9, [%[b], 144]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 128]\n\t"
        "ldp	x8, x9, [%[b], 128]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 112]\n\t"
        "ldp	x8, x9, [%[b], 112]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 96]\n\t"
        "ldp	x8, x9, [%[b], 96]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 80]\n\t"
        "ldp	x8, x9, [%[b], 80]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 64]\n\t"
        "ldp	x8, x9, [%[b], 64]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 48]\n\t"
        "ldp	x8, x9, [%[b], 48]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 32]\n\t"
        "ldp	x8, x9, [%[b], 32]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 16]\n\t"
        "ldp	x8, x9, [%[b], 16]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 0]\n\t"
        "ldp	x8, x9, [%[b], 0]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "cmp	x2, #0\n\t"
        "cset	%[a], eq\n\t"
        "orr	%[a], %[a], x3\n\t"
        : [a] "+r" (a)
        : [b] "r" (b)
        : "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );
#endif

    return (sp_int64)a;
}

/* Divide d in a and put remainder into r (m*d + r = a)
 * m is not calculated as it is not needed at this time.
 *
 * a  Number to be divided.
 * d  Number to divide with.
 * m  Multiplier result.
 * r  Remainder from the division.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_3072_div_48(const sp_digit* a, const sp_digit* d, sp_digit* m,
        sp_digit* r)
{
    sp_digit t1[96], t2[49];
    sp_digit div, r1;
    int i;

    (void)m;

    div = d[47];
    XMEMCPY(t1, a, sizeof(*t1) * 2 * 48);
    r1 = sp_3072_cmp_48(&t1[48], d) >= 0;
    sp_3072_cond_sub_48(&t1[48], &t1[48], d, (sp_digit)0 - r1);
    for (i = 47; i >= 0; i--) {
        volatile sp_digit mask = (sp_digit)0 - (t1[48 + i] == div);
        sp_digit hi = t1[48 + i] + mask;
        r1 = div_3072_word_48(hi, t1[48 + i - 1], div);
        r1 |= mask;

        sp_3072_mul_d_48(t2, d, r1);
        t1[48 + i] += sp_3072_sub_in_place_48(&t1[i], t2);
        t1[48 + i] -= t2[48];
        sp_3072_mask_48(t2, d, t1[48 + i]);
        t1[48 + i] += sp_3072_add_48(&t1[i], &t1[i], t2);
        sp_3072_mask_48(t2, d, t1[48 + i]);
        t1[48 + i] += sp_3072_add_48(&t1[i], &t1[i], t2);
    }

    r1 = sp_3072_cmp_48(t1, d) >= 0;
    sp_3072_cond_sub_48(r, t1, d, (sp_digit)0 - r1);

    return MP_OKAY;
}

/* Reduce a modulo m into r. (r = a mod m)
 *
 * r  A single precision number that is the reduced result.
 * a  A single precision number that is to be reduced.
 * m  A single precision number that is the modulus to reduce with.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_3072_mod_48(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    return sp_3072_div_48(a, m, NULL, r);
}

#if (defined(WOLFSSL_HAVE_SP_RSA) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)) || \
                                                     defined(WOLFSSL_HAVE_SP_DH)
#ifdef WOLFSSL_SP_SMALL
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns  0 on success.
 * returns  MEMORY_E on dynamic memory allocation failure.
 * returns  MP_VAL when base is even or exponent is 0.
 */
static int sp_3072_mod_exp_48(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* td = NULL;
#else
    sp_digit td[8 * 96];
#endif
    sp_digit* t[8];
    sp_digit* norm = NULL;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c;
    byte y;
    int err = MP_OKAY;

    if (bits == 0) {
        err = MP_VAL;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        td = (sp_digit*)XMALLOC(sizeof(sp_digit) * (8 * 96), NULL,
                                DYNAMIC_TYPE_TMP_BUFFER);
        if (td == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        norm = td;
        for (i=0; i<8; i++) {
            t[i] = td + i * 96;
        }

        sp_3072_mont_setup(m, &mp);
        sp_3072_mont_norm_48(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 48U);
        if (reduceA != 0) {
            err = sp_3072_mod_48(t[1] + 48, a, m);
            if (err == MP_OKAY) {
                err = sp_3072_mod_48(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 48, a, sizeof(sp_digit) * 48);
            err = sp_3072_mod_48(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_3072_mont_sqr_48(t[ 2], t[ 1], m, mp);
        sp_3072_mont_mul_48(t[ 3], t[ 2], t[ 1], m, mp);
        sp_3072_mont_sqr_48(t[ 4], t[ 2], m, mp);
        sp_3072_mont_mul_48(t[ 5], t[ 3], t[ 2], m, mp);
        sp_3072_mont_sqr_48(t[ 6], t[ 3], m, mp);
        sp_3072_mont_mul_48(t[ 7], t[ 4], t[ 3], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 3;
        if (c == 64) {
            c = 61;
        }
        if (c < 0) {
            /* Number of bits in top word is less than number needed. */
            c = -c;
            y = (byte)(n << c);
            n = e[i--];
            y |= (byte)(n >> (64 - c));
            n <<= c;
            c = 64 - c;
        }
        else if (c == 0) {
            /* All bits in top word used. */
            y = (byte)n;
        }
        else {
            y = (byte)(n >> c);
            n <<= 64 - c;
        }
        XMEMCPY(r, t[y], sizeof(sp_digit) * 48);
        for (; i>=0 || c>=3; ) {
            if (c == 0) {
                n = e[i--];
                y = (byte)(n >> 61);
                n <<= 3;
                c = 61;
            }
            else if (c < 3) {
                y = (byte)(n >> 61);
                n = e[i--];
                c = 3 - c;
                y |= (byte)(n >> (64 - c));
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (byte)((n >> 61) & 0x7);
                n <<= 3;
                c -= 3;
            }

            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);

            sp_3072_mont_mul_48(r, r, t[y], m, mp);
        }

        XMEMSET(&r[48], 0, sizeof(sp_digit) * 48U);
        sp_3072_mont_reduce_48(r, m, mp);

        mask = 0 - (sp_3072_cmp_48(r, m) >= 0);
        sp_3072_cond_sub_48(r, r, m, mask);
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (td != NULL)
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return err;
}
#else
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns  0 on success.
 * returns  MEMORY_E on dynamic memory allocation failure.
 * returns  MP_VAL when base is even or exponent is 0.
 */
static int sp_3072_mod_exp_48(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* td = NULL;
#else
    sp_digit td[16 * 96];
#endif
    sp_digit* t[16];
    sp_digit* norm = NULL;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c;
    byte y;
    int err = MP_OKAY;

    if (bits == 0) {
        err = MP_VAL;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        td = (sp_digit*)XMALLOC(sizeof(sp_digit) * (16 * 96), NULL,
                                DYNAMIC_TYPE_TMP_BUFFER);
        if (td == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        norm = td;
        for (i=0; i<16; i++) {
            t[i] = td + i * 96;
        }

        sp_3072_mont_setup(m, &mp);
        sp_3072_mont_norm_48(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 48U);
        if (reduceA != 0) {
            err = sp_3072_mod_48(t[1] + 48, a, m);
            if (err == MP_OKAY) {
                err = sp_3072_mod_48(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 48, a, sizeof(sp_digit) * 48);
            err = sp_3072_mod_48(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_3072_mont_sqr_48(t[ 2], t[ 1], m, mp);
        sp_3072_mont_mul_48(t[ 3], t[ 2], t[ 1], m, mp);
        sp_3072_mont_sqr_48(t[ 4], t[ 2], m, mp);
        sp_3072_mont_mul_48(t[ 5], t[ 3], t[ 2], m, mp);
        sp_3072_mont_sqr_48(t[ 6], t[ 3], m, mp);
        sp_3072_mont_mul_48(t[ 7], t[ 4], t[ 3], m, mp);
        sp_3072_mont_sqr_48(t[ 8], t[ 4], m, mp);
        sp_3072_mont_mul_48(t[ 9], t[ 5], t[ 4], m, mp);
        sp_3072_mont_sqr_48(t[10], t[ 5], m, mp);
        sp_3072_mont_mul_48(t[11], t[ 6], t[ 5], m, mp);
        sp_3072_mont_sqr_48(t[12], t[ 6], m, mp);
        sp_3072_mont_mul_48(t[13], t[ 7], t[ 6], m, mp);
        sp_3072_mont_sqr_48(t[14], t[ 7], m, mp);
        sp_3072_mont_mul_48(t[15], t[ 8], t[ 7], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 4;
        if (c == 64) {
            c = 60;
        }
        if (c < 0) {
            /* Number of bits in top word is less than number needed. */
            c = -c;
            y = (byte)(n << c);
            n = e[i--];
            y |= (byte)(n >> (64 - c));
            n <<= c;
            c = 64 - c;
        }
        else if (c == 0) {
            /* All bits in top word used. */
            y = (byte)n;
        }
        else {
            y = (byte)(n >> c);
            n <<= 64 - c;
        }
        XMEMCPY(r, t[y], sizeof(sp_digit) * 48);
        for (; i>=0 || c>=4; ) {
            if (c == 0) {
                n = e[i--];
                y = (byte)(n >> 60);
                n <<= 4;
                c = 60;
            }
            else if (c < 4) {
                y = (byte)(n >> 60);
                n = e[i--];
                c = 4 - c;
                y |= (byte)(n >> (64 - c));
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (byte)((n >> 60) & 0xf);
                n <<= 4;
                c -= 4;
            }

            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);

            sp_3072_mont_mul_48(r, r, t[y], m, mp);
        }

        XMEMSET(&r[48], 0, sizeof(sp_digit) * 48U);
        sp_3072_mont_reduce_48(r, m, mp);

        mask = 0 - (sp_3072_cmp_48(r, m) >= 0);
        sp_3072_cond_sub_48(r, r, m, mask);
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (td != NULL)
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return err;
}
#endif /* WOLFSSL_SP_SMALL */
#endif /* (WOLFSSL_HAVE_SP_RSA && !WOLFSSL_RSA_PUBLIC_ONLY) || WOLFSSL_HAVE_SP_DH */

#endif /* (WOLFSSL_HAVE_SP_RSA && !WOLFSSL_RSA_PUBLIC_ONLY) || WOLFSSL_HAVE_SP_DH */
#ifdef WOLFSSL_HAVE_SP_RSA
/* RSA public key operation.
 *
 * in      Array of bytes representing the number to exponentiate, base.
 * inLen   Number of bytes in base.
 * em      Public exponent.
 * mm      Modulus.
 * out     Buffer to hold big-endian bytes of exponentiation result.
 *         Must be at least 384 bytes long.
 * outLen  Number of bytes in result.
 * returns 0 on success, MP_TO_E when the outLen is too small, MP_READ_E when
 * an array is too long and MEMORY_E when dynamic memory allocation fails.
 */
int sp_RsaPublic_3072(const byte* in, word32 inLen, const mp_int* em,
    const mp_int* mm, byte* out, word32* outLen)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* a = NULL;
#else
    sp_digit a[48 * 5];
#endif
    sp_digit* m = NULL;
    sp_digit* r = NULL;
    sp_digit *ah = NULL;
    sp_digit e[1] = {0};
    int err = MP_OKAY;

    if (*outLen < 384) {
        err = MP_TO_E;
    }
    else if (mp_count_bits(em) > 64 || inLen > 384 ||
                                                     mp_count_bits(mm) != 3072) {
        err = MP_READ_E;
    }
    else if (mp_iseven(mm)) {
        err = MP_VAL;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        a = (sp_digit*)XMALLOC(sizeof(sp_digit) * 48 * 5, NULL,
                                                              DYNAMIC_TYPE_RSA);
        if (a == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        ah = a + 48;
        r = a + 48 * 2;
        m = r + 48 * 2;

        sp_3072_from_bin(ah, 48, in, inLen);
#if DIGIT_BIT >= 64
        e[0] = em->dp[0];
#else
        e[0] = em->dp[0];
        if (em->used > 1) {
            e[0] |= ((sp_digit)em->dp[1]) << DIGIT_BIT;
        }
#endif
        if (e[0] == 0) {
            err = MP_EXPTMOD_E;
        }
    }
    if (err == MP_OKAY) {
        sp_3072_from_mp(m, 48, mm);

        if (e[0] == 0x10001) {
            int i;
            sp_digit mp;

            sp_3072_mont_setup(m, &mp);

            /* Convert to Montgomery form. */
            XMEMSET(a, 0, sizeof(sp_digit) * 48);
            err = sp_3072_mod_48_cond(r, a, m);
            /* Montgomery form: r = a.R mod m */

            if (err == MP_OKAY) {
                /* r = a ^ 0x10000 => r = a squared 16 times */
                for (i = 15; i >= 0; i--) {
                    sp_3072_mont_sqr_48(r, r, m, mp);
                }
                /* mont_red(r.R.R) = (r.R.R / R) mod m = r.R mod m
                 * mont_red(r.R * a) = (r.R.a / R) mod m = r.a mod m
                 */
                sp_3072_mont_mul_48(r, r, ah, m, mp);

                for (i = 47; i > 0; i--) {
                    if (r[i] != m[i]) {
                        break;
                    }
                }
                if (r[i] >= m[i]) {
                    sp_3072_sub_in_place_48(r, m);
                }
            }
        }
        else if (e[0] == 0x3) {
            if (err == MP_OKAY) {
                sp_3072_sqr_48(r, ah);
                err = sp_3072_mod_48_cond(r, r, m);
            }
            if (err == MP_OKAY) {
                sp_3072_mul_48(r, ah, r);
                err = sp_3072_mod_48_cond(r, r, m);
            }
        }
        else {
            int i;
            sp_digit mp;

            sp_3072_mont_setup(m, &mp);

            /* Convert to Montgomery form. */
            XMEMSET(a, 0, sizeof(sp_digit) * 48);
            err = sp_3072_mod_48_cond(a, a, m);

            if (err == MP_OKAY) {
                for (i = 63; i >= 0; i--) {
                    if (e[0] >> i) {
                        break;
                    }
                }

                XMEMCPY(r, a, sizeof(sp_digit) * 48);
                for (i--; i >= 0; i--) {
                    sp_3072_mont_sqr_48(r, r, m, mp);
                    if (((e[0] >> i) & 1) == 1) {
                        sp_3072_mont_mul_48(r, r, a, m, mp);
                    }
                }
                XMEMSET(&r[48], 0, sizeof(sp_digit) * 48);
                sp_3072_mont_reduce_48(r, m, mp);

                for (i = 47; i > 0; i--) {
                    if (r[i] != m[i]) {
                        break;
                    }
                }
                if (r[i] >= m[i]) {
                    sp_3072_sub_in_place_48(r, m);
                }
            }
        }
    }

    if (err == MP_OKAY) {
        sp_3072_to_bin_48(r, out);
        *outLen = 384;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (a != NULL)
        XFREE(a, NULL, DYNAMIC_TYPE_RSA);
#endif

    return err;
}

#ifndef WOLFSSL_RSA_PUBLIC_ONLY
#ifdef WOLFSSL_SP_SMALL
/* Conditionally add a and b using the mask m.
 * m is -1 to add and 0 when not.
 *
 * r  A single precision number representing conditional add result.
 * a  A single precision number to add with.
 * b  A single precision number to add.
 * m  Mask value to apply.
 */
static sp_digit sp_3072_cond_add_24(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "mov	x8, #0\n\t"
        "1:\n\t"
        "adds	%[c], %[c], #-1\n\t"
        "ldr	x4, [%[a], x8]\n\t"
        "ldr	x5, [%[b], x8]\n\t"
        "and	x5, x5, %[m]\n\t"
        "adcs	x4, x4, x5\n\t"
        "cset	%[c], cs\n\t"
        "str	x4, [%[r], x8]\n\t"
        "add	x8, x8, #8\n\t"
        "cmp	x8, 192\n\t"
        "b.lt	1b\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x5", "x8", "x9", "x10", "x11", "x12"
    );

    return c;
}
#endif /* WOLFSSL_SP_SMALL */

/* RSA private key operation.
 *
 * in      Array of bytes representing the number to exponentiate, base.
 * inLen   Number of bytes in base.
 * dm      Private exponent.
 * pm      First prime.
 * qm      Second prime.
 * dpm     First prime's CRT exponent.
 * dqm     Second prime's CRT exponent.
 * qim     Inverse of second prime mod p.
 * mm      Modulus.
 * out     Buffer to hold big-endian bytes of exponentiation result.
 *         Must be at least 384 bytes long.
 * outLen  Number of bytes in result.
 * returns 0 on success, MP_TO_E when the outLen is too small, MP_READ_E when
 * an array is too long and MEMORY_E when dynamic memory allocation fails.
 */
int sp_RsaPrivate_3072(const byte* in, word32 inLen, const mp_int* dm,
    const mp_int* pm, const mp_int* qm, const mp_int* dpm, const mp_int* dqm,
    const mp_int* qim, const mp_int* mm, byte* out, word32* outLen)
{
#if defined(SP_RSA_PRIVATE_EXP_D) || defined(RSA_LOW_MEM)
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* d = NULL;
#else
    sp_digit  d[48 * 4];
#endif
    sp_digit* a = NULL;
    sp_digit* m = NULL;
    sp_digit* r = NULL;
    int err = MP_OKAY;

    (void)pm;
    (void)qm;
    (void)dpm;
    (void)dqm;
    (void)qim;

    if (*outLen < 384U) {
        err = MP_TO_E;
    }
    if (err == MP_OKAY) {
        if (mp_count_bits(dm) > 3072) {
           err = MP_READ_E;
        }
        else if (inLen > 384) {
            err = MP_READ_E;
        }
        else if (mp_count_bits(mm) != 3072) {
            err = MP_READ_E;
        }
        else if (mp_iseven(mm)) {
            err = MP_VAL;
        }
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        d = (sp_digit*)XMALLOC(sizeof(sp_digit) * 48 * 4, NULL,
                                                              DYNAMIC_TYPE_RSA);
        if (d == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        a = d + 48;
        m = a + 96;
        r = a;

        sp_3072_from_bin(a, 48, in, inLen);
        sp_3072_from_mp(d, 48, dm);
        sp_3072_from_mp(m, 48, mm);
        err = sp_3072_mod_exp_48(r, a, d, 3072, m, 0);
    }

    if (err == MP_OKAY) {
        sp_3072_to_bin_48(r, out);
        *outLen = 384;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (d != NULL)
#endif
    {
        /* only "a" and "r" are sensitive and need zeroized (same pointer) */
        if (a != NULL)
            ForceZero(a, sizeof(sp_digit) * 48);
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
        XFREE(d, NULL, DYNAMIC_TYPE_RSA);
#endif
    }

    return err;
#else
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* a = NULL;
#else
    sp_digit a[24 * 11];
#endif
    sp_digit* p = NULL;
    sp_digit* q = NULL;
    sp_digit* dp = NULL;
    sp_digit* tmpa = NULL;
    sp_digit* tmpb = NULL;
    sp_digit* r = NULL;
    sp_digit* qi = NULL;
    sp_digit* dq = NULL;
    sp_digit c;
    int err = MP_OKAY;

    (void)dm;
    (void)mm;

    if (*outLen < 384) {
        err = MP_TO_E;
    }
    else if (inLen > 384 || mp_count_bits(mm) != 3072) {
        err = MP_READ_E;
    }
    else if (mp_iseven(mm)) {
        err = MP_VAL;
    }
    else if (mp_iseven(pm)) {
        err = MP_VAL;
    }
    else if (mp_iseven(qm)) {
        err = MP_VAL;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        a = (sp_digit*)XMALLOC(sizeof(sp_digit) * 24 * 11, NULL,
                                                              DYNAMIC_TYPE_RSA);
        if (a == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        p = a + 48 * 2;
        q = p + 24;
        qi = dq = dp = q + 24;
        tmpa = qi + 24;
        tmpb = tmpa + 48;
        r = a;

        sp_3072_from_bin(a, 48, in, inLen);
        sp_3072_from_mp(p, 24, pm);
        sp_3072_from_mp(q, 24, qm);
        sp_3072_from_mp(dp, 24, dpm);

        err = sp_3072_mod_exp_24(tmpa, a, dp, 1536, p, 1);
    }
    if (err == MP_OKAY) {
        sp_3072_from_mp(dq, 24, dqm);
        err = sp_3072_mod_exp_24(tmpb, a, dq, 1536, q, 1);
    }

    if (err == MP_OKAY) {
        c = sp_3072_sub_in_place_24(tmpa, tmpb);
        c += sp_3072_cond_add_24(tmpa, tmpa, p, c);
        sp_3072_cond_add_24(tmpa, tmpa, p, c);

        sp_3072_from_mp(qi, 24, qim);
        sp_3072_mul_24(tmpa, tmpa, qi);
        err = sp_3072_mod_24(tmpa, tmpa, p);
    }

    if (err == MP_OKAY) {
        sp_3072_mul_24(tmpa, q, tmpa);
        XMEMSET(&tmpb[24], 0, sizeof(sp_digit) * 24);
        sp_3072_add_48(r, tmpb, tmpa);

        sp_3072_to_bin_48(r, out);
        *outLen = 384;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (a != NULL)
#endif
    {
        ForceZero(a, sizeof(sp_digit) * 24 * 11);
    #if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
        XFREE(a, NULL, DYNAMIC_TYPE_RSA);
    #endif
    }
#endif /* SP_RSA_PRIVATE_EXP_D || RSA_LOW_MEM */
    return err;
}
#endif /* WOLFSSL_RSA_PUBLIC_ONLY */
#endif /* WOLFSSL_HAVE_SP_RSA */
#if defined(WOLFSSL_HAVE_SP_DH) || (defined(WOLFSSL_HAVE_SP_RSA) && \
                                              !defined(WOLFSSL_RSA_PUBLIC_ONLY))
/* Convert an array of sp_digit to an mp_int.
 *
 * a  A single precision integer.
 * r  A multi-precision integer.
 */
static int sp_3072_to_mp(const sp_digit* a, mp_int* r)
{
    int err;

    err = mp_grow(r, (3072 + DIGIT_BIT - 1) / DIGIT_BIT);
    if (err == MP_OKAY) { /*lint !e774 case where err is always MP_OKAY*/
#if DIGIT_BIT == 64
        XMEMCPY(r->dp, a, sizeof(sp_digit) * 48);
        r->used = 48;
        mp_clamp(r);
#elif DIGIT_BIT < 64
        int i;
        int j = 0;
        int s = 0;

        r->dp[0] = 0;
        for (i = 0; i < 48; i++) {
            r->dp[j] |= (mp_digit)(a[i] << s);
            r->dp[j] &= ((sp_digit)1 << DIGIT_BIT) - 1;
            s = DIGIT_BIT - s;
            r->dp[++j] = (mp_digit)(a[i] >> s);
            while (s + DIGIT_BIT <= 64) {
                s += DIGIT_BIT;
                r->dp[j++] &= ((sp_digit)1 << DIGIT_BIT) - 1;
                if (s == SP_WORD_SIZE) {
                    r->dp[j] = 0;
                }
                else {
                    r->dp[j] = (mp_digit)(a[i] >> s);
                }
            }
            s = 64 - s;
        }
        r->used = (3072 + DIGIT_BIT - 1) / DIGIT_BIT;
        mp_clamp(r);
#else
        int i;
        int j = 0;
        int s = 0;

        r->dp[0] = 0;
        for (i = 0; i < 48; i++) {
            r->dp[j] |= ((mp_digit)a[i]) << s;
            if (s + 64 >= DIGIT_BIT) {
    #if DIGIT_BIT != 32 && DIGIT_BIT != 64
                r->dp[j] &= ((sp_digit)1 << DIGIT_BIT) - 1;
    #endif
                s = DIGIT_BIT - s;
                r->dp[++j] = a[i] >> s;
                s = 64 - s;
            }
            else {
                s += 64;
            }
        }
        r->used = (3072 + DIGIT_BIT - 1) / DIGIT_BIT;
        mp_clamp(r);
#endif
    }

    return err;
}

/* Perform the modular exponentiation for Diffie-Hellman.
 *
 * base  Base. MP integer.
 * exp   Exponent. MP integer.
 * mod   Modulus. MP integer.
 * res   Result. MP integer.
 * returns 0 on success, MP_READ_E if there are too many bytes in an array
 * and MEMORY_E if memory allocation fails.
 */
int sp_ModExp_3072(const mp_int* base, const mp_int* exp, const mp_int* mod,
    mp_int* res)
{
    int err = MP_OKAY;
    sp_digit b[96];
    sp_digit e[48];
    sp_digit m[48];
    sp_digit* r = b;
    int expBits = mp_count_bits(exp);

    if (mp_count_bits(base) > 3072) {
        err = MP_READ_E;
    }
    else if (expBits > 3072) {
        err = MP_READ_E;
    }
    else if (mp_count_bits(mod) != 3072) {
        err = MP_READ_E;
    }
    else if (mp_iseven(mod)) {
        err = MP_VAL;
    }

    if (err == MP_OKAY) {
        sp_3072_from_mp(b, 48, base);
        sp_3072_from_mp(e, 48, exp);
        sp_3072_from_mp(m, 48, mod);

        err = sp_3072_mod_exp_48(r, b, e, expBits, m, 0);
    }

    if (err == MP_OKAY) {
        err = sp_3072_to_mp(r, res);
    }

    XMEMSET(e, 0, sizeof(e));

    return err;
}

#ifdef WOLFSSL_HAVE_SP_DH

#ifdef HAVE_FFDHE_3072
static void sp_3072_lshift_48(sp_digit* r, const sp_digit* a, byte n)
{
    word64 n64 = n;
    __asm__ __volatile__ (
        "mov	x6, 63\n\t"
        "sub	x6, x6, %[n]\n\t"
        "ldr	x3, [%[a], 376]\n\t"
        "lsr	x4, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x4, x4, x6\n\t"
        "ldr	x2, [%[a], 368]\n\t"
        "str	x4, [%[r], 384]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 360]\n\t"
        "str	x3, [%[r], 376]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 352]\n\t"
        "str	x2, [%[r], 368]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 344]\n\t"
        "str	x4, [%[r], 360]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 336]\n\t"
        "str	x3, [%[r], 352]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 328]\n\t"
        "str	x2, [%[r], 344]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 320]\n\t"
        "str	x4, [%[r], 336]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 312]\n\t"
        "str	x3, [%[r], 328]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 304]\n\t"
        "str	x2, [%[r], 320]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 296]\n\t"
        "str	x4, [%[r], 312]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 288]\n\t"
        "str	x3, [%[r], 304]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 280]\n\t"
        "str	x2, [%[r], 296]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 272]\n\t"
        "str	x4, [%[r], 288]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 264]\n\t"
        "str	x3, [%[r], 280]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 256]\n\t"
        "str	x2, [%[r], 272]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 248]\n\t"
        "str	x4, [%[r], 264]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 240]\n\t"
        "str	x3, [%[r], 256]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 232]\n\t"
        "str	x2, [%[r], 248]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 224]\n\t"
        "str	x4, [%[r], 240]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 216]\n\t"
        "str	x3, [%[r], 232]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 208]\n\t"
        "str	x2, [%[r], 224]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 200]\n\t"
        "str	x4, [%[r], 216]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 192]\n\t"
        "str	x3, [%[r], 208]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 184]\n\t"
        "str	x2, [%[r], 200]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 176]\n\t"
        "str	x4, [%[r], 192]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 168]\n\t"
        "str	x3, [%[r], 184]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 160]\n\t"
        "str	x2, [%[r], 176]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 152]\n\t"
        "str	x4, [%[r], 168]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 144]\n\t"
        "str	x3, [%[r], 160]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 136]\n\t"
        "str	x2, [%[r], 152]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 128]\n\t"
        "str	x4, [%[r], 144]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 120]\n\t"
        "str	x3, [%[r], 136]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 112]\n\t"
        "str	x2, [%[r], 128]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 104]\n\t"
        "str	x4, [%[r], 120]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 96]\n\t"
        "str	x3, [%[r], 112]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 88]\n\t"
        "str	x2, [%[r], 104]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 80]\n\t"
        "str	x4, [%[r], 96]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 72]\n\t"
        "str	x3, [%[r], 88]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 64]\n\t"
        "str	x2, [%[r], 80]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 56]\n\t"
        "str	x4, [%[r], 72]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 48]\n\t"
        "str	x3, [%[r], 64]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 40]\n\t"
        "str	x2, [%[r], 56]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 32]\n\t"
        "str	x4, [%[r], 48]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 24]\n\t"
        "str	x3, [%[r], 40]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 16]\n\t"
        "str	x2, [%[r], 32]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 8]\n\t"
        "str	x4, [%[r], 24]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 0]\n\t"
        "str	x3, [%[r], 16]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "stp	x4, x2, [%[r]]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [n] "r" (n64)
        : "memory", "x2", "x3", "x4", "x5", "x6"
    );
}

/* Modular exponentiate 2 to the e mod m. (r = 2^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns  0 on success.
 * returns  MEMORY_E on dynamic memory allocation failure.
 * returns  MP_VAL when base is even.
 */
static int sp_3072_mod_exp_2_48(sp_digit* r, const sp_digit* e, int bits,
        const sp_digit* m)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* td = NULL;
#else
    sp_digit td[145];
#endif
    sp_digit* norm = NULL;
    sp_digit* tmp = NULL;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit o;
    sp_digit mask;
    int i;
    int c;
    byte y;
    int err = MP_OKAY;

    if (bits == 0) {
        err = MP_VAL;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        td = (sp_digit*)XMALLOC(sizeof(sp_digit) * 145, NULL,
                                DYNAMIC_TYPE_TMP_BUFFER);
        if (td == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        norm = td;
        tmp = td + 96;

        sp_3072_mont_setup(m, &mp);
        sp_3072_mont_norm_48(norm, m);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 6;
        if (c == 64) {
            c = 58;
        }
        if (c < 0) {
            /* Number of bits in top word is less than number needed. */
            c = -c;
            y = (byte)(n << c);
            n = e[i--];
            y |= (byte)(n >> (64 - c));
            n <<= c;
            c = 64 - c;
        }
        else if (c == 0) {
            /* All bits in top word used. */
            y = (byte)n;
        }
        else {
            y = (byte)(n >> c);
            n <<= 64 - c;
        }
        sp_3072_lshift_48(r, norm, y);
        for (; i>=0 || c>=6; ) {
            if (c == 0) {
                n = e[i--];
                y = (byte)(n >> 58);
                n <<= 6;
                c = 58;
            }
            else if (c < 6) {
                y = (byte)(n >> 58);
                n = e[i--];
                c = 6 - c;
                y |= (byte)(n >> (64 - c));
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (byte)((n >> 58) & 0x3f);
                n <<= 6;
                c -= 6;
            }

            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);
            sp_3072_mont_sqr_48(r, r, m, mp);

            sp_3072_lshift_48(r, r, y);
            sp_3072_mul_d_48(tmp, norm, r[48]);
            r[48] = 0;
            o = sp_3072_add_48(r, r, tmp);
            sp_3072_cond_sub_48(r, r, m, (sp_digit)0 - o);
        }

        XMEMSET(&r[48], 0, sizeof(sp_digit) * 48U);
        sp_3072_mont_reduce_48(r, m, mp);

        mask = 0 - (sp_3072_cmp_48(r, m) >= 0);
        sp_3072_cond_sub_48(r, r, m, mask);
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (td != NULL)
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return err;
}
#endif /* HAVE_FFDHE_3072 */

/* Perform the modular exponentiation for Diffie-Hellman.
 *
 * base     Base.
 * exp      Array of bytes that is the exponent.
 * expLen   Length of data, in bytes, in exponent.
 * mod      Modulus.
 * out      Buffer to hold big-endian bytes of exponentiation result.
 *          Must be at least 384 bytes long.
 * outLen   Length, in bytes, of exponentiation result.
 * returns 0 on success, MP_READ_E if there are too many bytes in an array
 * and MEMORY_E if memory allocation fails.
 */
int sp_DhExp_3072(const mp_int* base, const byte* exp, word32 expLen,
    const mp_int* mod, byte* out, word32* outLen)
{
    int err = MP_OKAY;
    sp_digit b[96];
    sp_digit e[48];
    sp_digit m[48];
    sp_digit* r = b;
    word32 i;

    if (mp_count_bits(base) > 3072) {
        err = MP_READ_E;
    }
    else if (expLen > 384) {
        err = MP_READ_E;
    }
    else if (mp_count_bits(mod) != 3072) {
        err = MP_READ_E;
    }
    else if (mp_iseven(mod)) {
        err = MP_VAL;
    }

    if (err == MP_OKAY) {
        sp_3072_from_mp(b, 48, base);
        sp_3072_from_bin(e, 48, exp, expLen);
        sp_3072_from_mp(m, 48, mod);

    #ifdef HAVE_FFDHE_3072
        if (base->used == 1 && base->dp[0] == 2 && m[47] == (sp_digit)-1)
            err = sp_3072_mod_exp_2_48(r, e, expLen * 8, m);
        else
    #endif
            err = sp_3072_mod_exp_48(r, b, e, expLen * 8, m, 0);

    }

    if (err == MP_OKAY) {
        sp_3072_to_bin_48(r, out);
        *outLen = 384;
        for (i=0; i<384 && out[i] == 0; i++) {
            /* Search for first non-zero. */
        }
        *outLen -= i;
        XMEMMOVE(out, out + i, *outLen);

    }

    XMEMSET(e, 0, sizeof(e));

    return err;
}
#endif /* WOLFSSL_HAVE_SP_DH */

/* Perform the modular exponentiation for Diffie-Hellman.
 *
 * base  Base. MP integer.
 * exp   Exponent. MP integer.
 * mod   Modulus. MP integer.
 * res   Result. MP integer.
 * returns 0 on success, MP_READ_E if there are too many bytes in an array
 * and MEMORY_E if memory allocation fails.
 */
int sp_ModExp_1536(const mp_int* base, const mp_int* exp, const mp_int* mod,
    mp_int* res)
{
    int err = MP_OKAY;
    sp_digit b[48];
    sp_digit e[24];
    sp_digit m[24];
    sp_digit* r = b;
    int expBits = mp_count_bits(exp);

    if (mp_count_bits(base) > 1536) {
        err = MP_READ_E;
    }
    else if (expBits > 1536) {
        err = MP_READ_E;
    }
    else if (mp_count_bits(mod) != 1536) {
        err = MP_READ_E;
    }
    else if (mp_iseven(mod)) {
        err = MP_VAL;
    }

    if (err == MP_OKAY) {
        sp_3072_from_mp(b, 24, base);
        sp_3072_from_mp(e, 24, exp);
        sp_3072_from_mp(m, 24, mod);

        err = sp_3072_mod_exp_24(r, b, e, expBits, m, 0);
    }

    if (err == MP_OKAY) {
        XMEMSET(r + 24, 0, sizeof(*r) * 24U);
        err = sp_3072_to_mp(r, res);
        res->used = mod->used;
        mp_clamp(res);
    }

    XMEMSET(e, 0, sizeof(e));

    return err;
}

#endif /* WOLFSSL_HAVE_SP_DH | (WOLFSSL_HAVE_SP_RSA & !WOLFSSL_RSA_PUBLIC_ONLY) */

#endif /* !WOLFSSL_SP_NO_3072 */

#ifdef WOLFSSL_SP_4096
/* Read big endian unsigned byte array into r.
 *
 * r  A single precision integer.
 * size  Maximum number of bytes to convert
 * a  Byte array.
 * n  Number of bytes in array to read.
 */
static void sp_4096_from_bin(sp_digit* r, int size, const byte* a, int n)
{
    sp_int64 nl = n;
    sp_int64 bytes = size * 8;

    __asm__ __volatile__ (
        "add	x4, %[a], %[n]\n\t"
        "mov	x5, %[r]\n\t"
        "sub	x4, x4, 8\n\t"
        "subs	x6, %[n], 8\n\t"
        "mov	x7, xzr\n\t"
        "blt	2f\n\t"
        /* Put in mulitples of 8 bytes. */
        "1:\n\t"
        "ldr	x8, [x4], -8\n\t"
        "subs	x6, x6, 8\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "rev	x8, x8\n\t"
    #endif
        "str	x8, [x5], 8\n\t"
        "add	x7, x7, 8\n\t"
        "b.ge	1b\n\t"
        "2:\n\t"
        "cmp	x6, -7\n\t"
        "b.lt	20f\n\t"
        /* Put in less than 8 bytes. */
    #ifdef LITTLE_ENDIAN_ORDER
        "str	xzr, [x5]\n\t"
    #else
        "str	xzr, [x5], 7\n\t"
    #endif
        "add	x7, x7, 8\n\t"
        "add	x4, x4, 7\n\t"
        "b.eq	17f\n\t"
        "cmp	x6, -5\n\t"
        "b.lt	16f\n\t"
        "b.eq	15f\n\t"
        "cmp	x6, -3\n\t"
        "b.lt	14f\n\t"
        "b.eq	13f\n\t"
        "cmp	x6, -2\n\t"
        "b.eq	12f\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "12:\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "13:\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "14:\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "15:\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "16:\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "17:\n\t"
        "ldrb	w8, [x4], -1\n\t"
    #ifdef LITTLE_ENDIAN_ORDER
        "strb	w8, [x5], 1\n\t"
    #else
        "strb	w8, [x5], -1\n\t"
    #endif
        "20:\n\t"
        "add	x5, %[r], x7\n\t"
        "subs	x7, %[size], x7\n\t"
        "b.eq	30f\n\t"
        /* Zero out remaining words. */
        "21:\n\t"
        "subs	x7, x7, 8\n\t"
        "str	xzr, [x5], 8\n\t"
        "b.gt	21b\n\t"
        "30:\n\t"
        :
        : [r] "r" (r), [size] "r" (bytes), [a] "r" (a), [n] "r" (nl)
        : "memory", "x4", "x5", "x6", "x7", "x8"
    );
}

/* Convert an mp_int to an array of sp_digit.
 *
 * r  A single precision integer.
 * size  Maximum number of bytes to convert
 * a  A multi-precision integer.
 */
static void sp_4096_from_mp(sp_digit* r, int size, const mp_int* a)
{
#if DIGIT_BIT == 64
    int i;
    int j = 0;

    for (i = 0; i < size; i++) {
        sp_digit mask =
            (((sp_digit)(a->used - i - 1)) >> (SP_WORD_SIZE - 1)) - 1;
        r[i] = a->dp[j] & mask;
        j += (int)(((sp_digit)1) -
                   (((sp_digit)(a->used - i - 2)) >> (SP_WORD_SIZE - 1)));
    }
#elif DIGIT_BIT > 64
    int i;
    int j = 0;
    word32 s = 0;

    r[0] = 0;
    for (i = 0; i < a->used && j < size; i++) {
        r[j] |= ((sp_digit)a->dp[i] << s);
        r[j] &= 0xffffffffffffffffl;
        s = 64U - s;
        if (j + 1 >= size) {
            break;
        }
        /* lint allow cast of mismatch word32 and mp_digit */
        r[++j] = (sp_digit)(a->dp[i] >> s); /*lint !e9033*/
        while ((s + 64U) <= (word32)DIGIT_BIT) {
            s += 64U;
            r[j] &= 0xffffffffffffffffl;
            if (j + 1 >= size) {
                break;
            }
            if (s < (word32)DIGIT_BIT) {
                /* lint allow cast of mismatch word32 and mp_digit */
                r[++j] = (sp_digit)(a->dp[i] >> s); /*lint !e9033*/
            }
            else {
                r[++j] = (sp_digit)0;
            }
        }
        s = (word32)DIGIT_BIT - s;
    }

    for (j++; j < size; j++) {
        r[j] = 0;
    }
#else
    int i;
    int j = 0;
    int s = 0;

    r[0] = 0;
    for (i = 0; i < a->used && j < size; i++) {
        r[j] |= ((sp_digit)a->dp[i]) << s;
        if (s + DIGIT_BIT >= 64) {
            r[j] &= 0xffffffffffffffffl;
            if (j + 1 >= size) {
                break;
            }
            s = 64 - s;
            if (s == DIGIT_BIT) {
                r[++j] = 0;
                s = 0;
            }
            else {
                r[++j] = a->dp[i] >> s;
                s = DIGIT_BIT - s;
            }
        }
        else {
            s += DIGIT_BIT;
        }
    }

    for (j++; j < size; j++) {
        r[j] = 0;
    }
#endif
}

/* Write r as big endian to byte array.
 * Fixed length number of bytes written: 512
 *
 * r  A single precision integer.
 * a  Byte array.
 */
static void sp_4096_to_bin_64(sp_digit* r, byte* a)
{
    int i;
    int j = 0;

    for (i = 63; i >= 0; i--, j += 8) {
        __asm__ __volatile__ (
            "ldr	x4, [%[r]]\n\t"
        #ifdef LITTLE_ENDIAN_ORDER
            "rev	x4, x4\n\t"
        #endif
            "str	x4, [%[a]]\n\t"
            :
            : [r] "r" (r + i), [a] "r" (a + j)
            : "memory", "x4"
        );
    }
}

#if (defined(WOLFSSL_HAVE_SP_RSA) && (!defined(WOLFSSL_RSA_PUBLIC_ONLY) || !defined(WOLFSSL_SP_SMALL))) || defined(WOLFSSL_HAVE_SP_DH)
/* Normalize the values in each word to 64.
 *
 * a  Array of sp_digit to normalize.
 */
#define sp_4096_norm_64(a)

#endif /* (WOLFSSL_HAVE_SP_RSA && (!WOLFSSL_RSA_PUBLIC_ONLY || !WOLFSSL_SP_SMALL)) || WOLFSSL_HAVE_SP_DH */
/* Normalize the values in each word to 64.
 *
 * a  Array of sp_digit to normalize.
 */
#define sp_4096_norm_64(a)

#ifndef WOLFSSL_SP_SMALL
/* Add digit to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_4096_add_word_32(sp_digit* r, const sp_digit* a,
        sp_digit b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "adds	x3, x3, %[b]\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "adcs	x3, x3, xzr\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "adcs	x3, x3, xzr\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "ldp	x3, x4, [%[a], 96]\n\t"
        "ldp	x5, x6, [%[a], 112]\n\t"
        "adcs	x3, x3, xzr\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 96]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 112]\n\t"
        "ldp	x3, x4, [%[a], 128]\n\t"
        "ldp	x5, x6, [%[a], 144]\n\t"
        "adcs	x3, x3, xzr\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 128]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 144]\n\t"
        "ldp	x3, x4, [%[a], 160]\n\t"
        "ldp	x5, x6, [%[a], 176]\n\t"
        "adcs	x3, x3, xzr\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 160]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 176]\n\t"
        "ldp	x3, x4, [%[a], 192]\n\t"
        "ldp	x5, x6, [%[a], 208]\n\t"
        "adcs	x3, x3, xzr\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 192]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 208]\n\t"
        "ldp	x3, x4, [%[a], 224]\n\t"
        "ldp	x5, x6, [%[a], 240]\n\t"
        "adcs	x3, x3, xzr\n\t"
        "adcs	x4, x4, xzr\n\t"
        "adcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 224]\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x5, x6, [%[r], 240]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6"
    );
}

/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer and result.
 * b  A single precision integer.
 */
static sp_digit sp_4096_sub_in_place_64(sp_digit* a, const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x2, x3, [%[a], 0]\n\t"
        "ldp	x6, x7, [%[b], 0]\n\t"
        "subs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 16]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 16]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 0]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 16]\n\t"
        "ldp	x2, x3, [%[a], 32]\n\t"
        "ldp	x6, x7, [%[b], 32]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 48]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 48]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 32]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 48]\n\t"
        "ldp	x2, x3, [%[a], 64]\n\t"
        "ldp	x6, x7, [%[b], 64]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 80]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 80]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 64]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 80]\n\t"
        "ldp	x2, x3, [%[a], 96]\n\t"
        "ldp	x6, x7, [%[b], 96]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 112]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 112]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 96]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 112]\n\t"
        "ldp	x2, x3, [%[a], 128]\n\t"
        "ldp	x6, x7, [%[b], 128]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 144]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 144]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 128]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 144]\n\t"
        "ldp	x2, x3, [%[a], 160]\n\t"
        "ldp	x6, x7, [%[b], 160]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 176]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 176]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 160]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 176]\n\t"
        "ldp	x2, x3, [%[a], 192]\n\t"
        "ldp	x6, x7, [%[b], 192]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 208]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 208]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 192]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 208]\n\t"
        "ldp	x2, x3, [%[a], 224]\n\t"
        "ldp	x6, x7, [%[b], 224]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 240]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 240]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 224]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 240]\n\t"
        "ldp	x2, x3, [%[a], 256]\n\t"
        "ldp	x6, x7, [%[b], 256]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 272]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 272]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 256]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 272]\n\t"
        "ldp	x2, x3, [%[a], 288]\n\t"
        "ldp	x6, x7, [%[b], 288]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 304]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 304]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 288]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 304]\n\t"
        "ldp	x2, x3, [%[a], 320]\n\t"
        "ldp	x6, x7, [%[b], 320]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 336]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 336]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 320]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 336]\n\t"
        "ldp	x2, x3, [%[a], 352]\n\t"
        "ldp	x6, x7, [%[b], 352]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 368]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 368]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 352]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 368]\n\t"
        "ldp	x2, x3, [%[a], 384]\n\t"
        "ldp	x6, x7, [%[b], 384]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 400]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 400]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 384]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 400]\n\t"
        "ldp	x2, x3, [%[a], 416]\n\t"
        "ldp	x6, x7, [%[b], 416]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 432]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 432]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 416]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 432]\n\t"
        "ldp	x2, x3, [%[a], 448]\n\t"
        "ldp	x6, x7, [%[b], 448]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 464]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 464]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 448]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 464]\n\t"
        "ldp	x2, x3, [%[a], 480]\n\t"
        "ldp	x6, x7, [%[b], 480]\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x4, x5, [%[a], 496]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x8, x9, [%[b], 496]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a], 480]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a], 496]\n\t"
        "csetm	%[a], cc\n\t"
        : [a] "+r" (a)
        : [b] "r" (b)
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );

    return (sp_digit)a;
}

/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_4096_add_64(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "adds	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x7, x8, [%[b], 64]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 80]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "ldp	x3, x4, [%[a], 96]\n\t"
        "ldp	x7, x8, [%[b], 96]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 112]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 112]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 96]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 112]\n\t"
        "ldp	x3, x4, [%[a], 128]\n\t"
        "ldp	x7, x8, [%[b], 128]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 144]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 144]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 128]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 144]\n\t"
        "ldp	x3, x4, [%[a], 160]\n\t"
        "ldp	x7, x8, [%[b], 160]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 176]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 176]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 160]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 176]\n\t"
        "ldp	x3, x4, [%[a], 192]\n\t"
        "ldp	x7, x8, [%[b], 192]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 208]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 208]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 192]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 208]\n\t"
        "ldp	x3, x4, [%[a], 224]\n\t"
        "ldp	x7, x8, [%[b], 224]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 240]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 240]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 224]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 240]\n\t"
        "ldp	x3, x4, [%[a], 256]\n\t"
        "ldp	x7, x8, [%[b], 256]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 272]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 272]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 256]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 272]\n\t"
        "ldp	x3, x4, [%[a], 288]\n\t"
        "ldp	x7, x8, [%[b], 288]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 304]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 304]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 288]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 304]\n\t"
        "ldp	x3, x4, [%[a], 320]\n\t"
        "ldp	x7, x8, [%[b], 320]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 336]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 336]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 320]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 336]\n\t"
        "ldp	x3, x4, [%[a], 352]\n\t"
        "ldp	x7, x8, [%[b], 352]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 368]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 368]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 352]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 368]\n\t"
        "ldp	x3, x4, [%[a], 384]\n\t"
        "ldp	x7, x8, [%[b], 384]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 400]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 400]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 384]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 400]\n\t"
        "ldp	x3, x4, [%[a], 416]\n\t"
        "ldp	x7, x8, [%[b], 416]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 432]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 432]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 416]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 432]\n\t"
        "ldp	x3, x4, [%[a], 448]\n\t"
        "ldp	x7, x8, [%[b], 448]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 464]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 464]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 448]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 464]\n\t"
        "ldp	x3, x4, [%[a], 480]\n\t"
        "ldp	x7, x8, [%[b], 480]\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 496]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 496]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 480]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 496]\n\t"
        "cset	%[r], cs\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return (sp_digit)r;
}

#ifndef WOLFSSL_SP_SMALL
/* Conditionally add a and b using the mask m.
 * m is -1 to add and 0 when not.
 *
 * r  A single precision number representing conditional add result.
 * a  A single precision number to add with.
 * b  A single precision number to add.
 * m  Mask value to apply.
 */
static sp_digit sp_4096_cond_add_32(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
    __asm__ __volatile__ (

        "ldp	x8, x9, [%[b], 0]\n\t"
        "ldp	x10, x11, [%[b], 16]\n\t"
        "ldp	x4, x5, [%[a], 0]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 16]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adds	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 0]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 16]\n\t"
        "ldp	x8, x9, [%[b], 32]\n\t"
        "ldp	x10, x11, [%[b], 48]\n\t"
        "ldp	x4, x5, [%[a], 32]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 48]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adcs	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 32]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 48]\n\t"
        "ldp	x8, x9, [%[b], 64]\n\t"
        "ldp	x10, x11, [%[b], 80]\n\t"
        "ldp	x4, x5, [%[a], 64]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 80]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adcs	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 64]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 80]\n\t"
        "ldp	x8, x9, [%[b], 96]\n\t"
        "ldp	x10, x11, [%[b], 112]\n\t"
        "ldp	x4, x5, [%[a], 96]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 112]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adcs	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 96]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 112]\n\t"
        "ldp	x8, x9, [%[b], 128]\n\t"
        "ldp	x10, x11, [%[b], 144]\n\t"
        "ldp	x4, x5, [%[a], 128]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 144]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adcs	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 128]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 144]\n\t"
        "ldp	x8, x9, [%[b], 160]\n\t"
        "ldp	x10, x11, [%[b], 176]\n\t"
        "ldp	x4, x5, [%[a], 160]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 176]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adcs	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 160]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 176]\n\t"
        "ldp	x8, x9, [%[b], 192]\n\t"
        "ldp	x10, x11, [%[b], 208]\n\t"
        "ldp	x4, x5, [%[a], 192]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 208]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adcs	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 192]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 208]\n\t"
        "ldp	x8, x9, [%[b], 224]\n\t"
        "ldp	x10, x11, [%[b], 240]\n\t"
        "ldp	x4, x5, [%[a], 224]\n\t"
        "and	x8, x8, %[m]\n\t"
        "ldp	x6, x7, [%[a], 240]\n\t"
        "and	x9, x9, %[m]\n\t"
        "adcs	x4, x4, x8\n\t"
        "and	x10, x10, %[m]\n\t"
        "adcs	x5, x5, x9\n\t"
        "and	x11, x11, %[m]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x4, x5, [%[r], 224]\n\t"
        "adcs	x7, x7, x11\n\t"
        "stp	x6, x7, [%[r], 240]\n\t"
        "cset	%[r], cs\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    return (sp_digit)r;
}
#endif /* !WOLFSSL_SP_SMALL */

/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
SP_NOINLINE static void sp_4096_mul_64(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit* z0 = r;
    sp_digit z1[64];
    sp_digit a1[32];
    sp_digit b1[32];
    sp_digit* z2 = r + 64;
    sp_digit u;
    sp_digit ca;
    sp_digit cb;

    ca = sp_2048_add_32(a1, a, &a[32]);
    cb = sp_2048_add_32(b1, b, &b[32]);
    u  = ca & cb;

    sp_2048_mul_32(z2, &a[32], &b[32]);
    sp_2048_mul_32(z0, a, b);
    sp_2048_mul_32(z1, a1, b1);

    u += sp_4096_sub_in_place_64(z1, z0);
    u += sp_4096_sub_in_place_64(z1, z2);
    u += sp_4096_cond_add_32(z1 + 32, z1 + 32, a1, 0 - cb);
    u += sp_4096_cond_add_32(z1 + 32, z1 + 32, b1, 0 - ca);

    u += sp_4096_add_64(r + 32, r + 32, z1);
    (void)sp_4096_add_word_32(r + 96, r + 96, u);
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
SP_NOINLINE static void sp_4096_sqr_64(sp_digit* r, const sp_digit* a)
{
    sp_digit* z0 = r;
    sp_digit* z2 = r + 64;
    sp_digit z1[64];
    sp_digit* a1 = z1;
    sp_digit* zero = z1 + 32;
    sp_digit u;
    sp_digit mask;
    sp_digit* p1;
    sp_digit* p2;

    XMEMSET(zero, 0, sizeof(sp_digit) * 32);

    mask = sp_2048_sub_32(a1, a, &a[32]);
    p1 = (sp_digit*)(((sp_digit)zero &   mask ) | ((sp_digit)a1 & (~mask)));
    p2 = (sp_digit*)(((sp_digit)zero & (~mask)) | ((sp_digit)a1 &   mask ));
    (void)sp_2048_sub_32(a1, p1, p2);

    sp_2048_sqr_32(z2, &a[32]);
    sp_2048_sqr_32(z0, a);
    sp_2048_sqr_32(z1, a1);

    u = 0;
    u -= sp_4096_sub_in_place_64(z1, z2);
    u -= sp_4096_sub_in_place_64(z1, z0);
    u += sp_4096_sub_in_place_64(r + 32, z1);
    sp_4096_add_word_32(r + 96, r + 96, u);
}

#endif /* !WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_4096_add_64(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x11, %[a], 512\n\t"
        "\n1:\n\t"
        "adds	%[c], %[c], #-1\n\t"
        "ldp	x3, x4, [%[a]], #16\n\t"
        "ldp	x5, x6, [%[a]], #16\n\t"
        "ldp	x7, x8, [%[b]], #16\n\t"
        "adcs	x3, x3, x7\n\t"
        "ldp	x9, x10, [%[b]], #16\n\t"
        "adcs	x4, x4, x8\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r]], #16\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r]], #16\n\t"
        "cset	%[c], cs\n\t"
        "cmp	%[a], x11\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [r] "+r" (r), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Sub b from a into a. (a -= b)
 *
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_4096_sub_in_place_64(sp_digit* a, const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x10, %[a], 512\n\t"
        "\n1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldp	x2, x3, [%[a]]\n\t"
        "ldp	x4, x5, [%[a], #16]\n\t"
        "ldp	x6, x7, [%[b]], #16\n\t"
        "sbcs	x2, x2, x6\n\t"
        "ldp	x8, x9, [%[b]], #16\n\t"
        "sbcs	x3, x3, x7\n\t"
        "sbcs	x4, x4, x8\n\t"
        "stp	x2, x3, [%[a]], #16\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x4, x5, [%[a]], #16\n\t"
        "csetm	%[c], cc\n\t"
        "cmp	%[a], x10\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return c;
}

#endif /* WOLFSSL_SP_SMALL */
#ifdef WOLFSSL_SP_SMALL
/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_4096_mul_64(sp_digit* r, const sp_digit* a, const sp_digit* b)
{
    sp_digit tmp[128];

    __asm__ __volatile__ (
        "mov	x5, xzr\n\t"
        "mov	x6, xzr\n\t"
        "mov	x7, xzr\n\t"
        "mov	x8, xzr\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 504\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[b], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 512\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 1008\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
static void sp_4096_sqr_64(sp_digit* r, const sp_digit* a)
{
    sp_digit tmp[128];

    __asm__ __volatile__ (
        "mov	x6, xzr\n\t"
        "mov	x7, xzr\n\t"
        "mov	x8, xzr\n\t"
        "mov	x5, xzr\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 504\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "cmp	x4, x3\n\t"
        "b.eq	4f\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[a], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "b.al	5f\n\t"
        "\n4:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "mul	x9, x10, x10\n\t"
        "umulh	x10, x10, x10\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "\n5:\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 512\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x4\n\t"
        "b.gt	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 1008\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

#endif /* WOLFSSL_SP_SMALL */
/* Caclulate the bottom digit of -1/a mod 2^n.
 *
 * a    A single precision number.
 * rho  Bottom word of inverse.
 */
static void sp_4096_mont_setup(const sp_digit* a, sp_digit* rho)
{
    sp_digit x;
    sp_digit b;

    b = a[0];
    x = (((b + 2) & 4) << 1) + b; /* here x*a==1 mod 2**4 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**8 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**16 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**32 */
    x *= 2 - b * x;               /* here x*a==1 mod 2**64 */

    /* rho = -1/m mod b */
    *rho = (sp_digit)0 - x;
}

/* Mul a by digit b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision digit.
 */
static void sp_4096_mul_d_64(sp_digit* r, const sp_digit* a,
        sp_digit b)
{
#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldr	x8, [%[a]]\n\t"
        "mul	x5, %[b], x8\n\t"
        "umulh	x3, %[b], x8\n\t"
        "mov	x4, xzr\n\t"
        "str	x5, [%[r]]\n\t"
        "mov	x5, xzr\n\t"
        "mov	x9, #8\n\t"
        "1:\n\t"
        "ldr	x8, [%[a], x9]\n\t"
        "mul	x6, %[b], x8\n\t"
        "umulh	x7, %[b], x8\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x7\n\t"
        "adc	x5, xzr, xzr\n\t"
        "str	x3, [%[r], x9]\n\t"
        "mov	x3, x4\n\t"
        "mov	x4, x5\n\t"
        "mov	x5, #0\n\t"
        "add	x9, x9, #8\n\t"
        "cmp	x9, 512\n\t"
        "b.lt	1b\n\t"
        "str	x3, [%[r], 512]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );
#else
    __asm__ __volatile__ (
        "# A[0] * B\n\t"
        "ldp	x9, x10, [%[a]]\n\t"
        "mul	x3, %[b], x9\n\t"
        "umulh	x4, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "# A[1] * B\n\t"
        "str	x3, [%[r]]\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[2] * B\n\t"
        "ldp	x9, x10, [%[a], 16]\n\t"
        "str	x4, [%[r], 8]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[3] * B\n\t"
        "str	x5, [%[r], 16]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[4] * B\n\t"
        "ldp	x9, x10, [%[a], 32]\n\t"
        "str	x3, [%[r], 24]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[5] * B\n\t"
        "str	x4, [%[r], 32]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[6] * B\n\t"
        "ldp	x9, x10, [%[a], 48]\n\t"
        "str	x5, [%[r], 40]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[7] * B\n\t"
        "str	x3, [%[r], 48]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[8] * B\n\t"
        "ldp	x9, x10, [%[a], 64]\n\t"
        "str	x4, [%[r], 56]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[9] * B\n\t"
        "str	x5, [%[r], 64]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[10] * B\n\t"
        "ldp	x9, x10, [%[a], 80]\n\t"
        "str	x3, [%[r], 72]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[11] * B\n\t"
        "str	x4, [%[r], 80]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[12] * B\n\t"
        "ldp	x9, x10, [%[a], 96]\n\t"
        "str	x5, [%[r], 88]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[13] * B\n\t"
        "str	x3, [%[r], 96]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[14] * B\n\t"
        "ldp	x9, x10, [%[a], 112]\n\t"
        "str	x4, [%[r], 104]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[15] * B\n\t"
        "str	x5, [%[r], 112]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[16] * B\n\t"
        "ldp	x9, x10, [%[a], 128]\n\t"
        "str	x3, [%[r], 120]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[17] * B\n\t"
        "str	x4, [%[r], 128]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[18] * B\n\t"
        "ldp	x9, x10, [%[a], 144]\n\t"
        "str	x5, [%[r], 136]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[19] * B\n\t"
        "str	x3, [%[r], 144]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[20] * B\n\t"
        "ldp	x9, x10, [%[a], 160]\n\t"
        "str	x4, [%[r], 152]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[21] * B\n\t"
        "str	x5, [%[r], 160]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[22] * B\n\t"
        "ldp	x9, x10, [%[a], 176]\n\t"
        "str	x3, [%[r], 168]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[23] * B\n\t"
        "str	x4, [%[r], 176]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[24] * B\n\t"
        "ldp	x9, x10, [%[a], 192]\n\t"
        "str	x5, [%[r], 184]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[25] * B\n\t"
        "str	x3, [%[r], 192]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[26] * B\n\t"
        "ldp	x9, x10, [%[a], 208]\n\t"
        "str	x4, [%[r], 200]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[27] * B\n\t"
        "str	x5, [%[r], 208]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[28] * B\n\t"
        "ldp	x9, x10, [%[a], 224]\n\t"
        "str	x3, [%[r], 216]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[29] * B\n\t"
        "str	x4, [%[r], 224]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[30] * B\n\t"
        "ldp	x9, x10, [%[a], 240]\n\t"
        "str	x5, [%[r], 232]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[31] * B\n\t"
        "str	x3, [%[r], 240]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[32] * B\n\t"
        "ldp	x9, x10, [%[a], 256]\n\t"
        "str	x4, [%[r], 248]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[33] * B\n\t"
        "str	x5, [%[r], 256]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[34] * B\n\t"
        "ldp	x9, x10, [%[a], 272]\n\t"
        "str	x3, [%[r], 264]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[35] * B\n\t"
        "str	x4, [%[r], 272]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[36] * B\n\t"
        "ldp	x9, x10, [%[a], 288]\n\t"
        "str	x5, [%[r], 280]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[37] * B\n\t"
        "str	x3, [%[r], 288]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[38] * B\n\t"
        "ldp	x9, x10, [%[a], 304]\n\t"
        "str	x4, [%[r], 296]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[39] * B\n\t"
        "str	x5, [%[r], 304]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[40] * B\n\t"
        "ldp	x9, x10, [%[a], 320]\n\t"
        "str	x3, [%[r], 312]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[41] * B\n\t"
        "str	x4, [%[r], 320]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[42] * B\n\t"
        "ldp	x9, x10, [%[a], 336]\n\t"
        "str	x5, [%[r], 328]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[43] * B\n\t"
        "str	x3, [%[r], 336]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[44] * B\n\t"
        "ldp	x9, x10, [%[a], 352]\n\t"
        "str	x4, [%[r], 344]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[45] * B\n\t"
        "str	x5, [%[r], 352]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[46] * B\n\t"
        "ldp	x9, x10, [%[a], 368]\n\t"
        "str	x3, [%[r], 360]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[47] * B\n\t"
        "str	x4, [%[r], 368]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[48] * B\n\t"
        "ldp	x9, x10, [%[a], 384]\n\t"
        "str	x5, [%[r], 376]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[49] * B\n\t"
        "str	x3, [%[r], 384]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[50] * B\n\t"
        "ldp	x9, x10, [%[a], 400]\n\t"
        "str	x4, [%[r], 392]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[51] * B\n\t"
        "str	x5, [%[r], 400]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[52] * B\n\t"
        "ldp	x9, x10, [%[a], 416]\n\t"
        "str	x3, [%[r], 408]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[53] * B\n\t"
        "str	x4, [%[r], 416]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[54] * B\n\t"
        "ldp	x9, x10, [%[a], 432]\n\t"
        "str	x5, [%[r], 424]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[55] * B\n\t"
        "str	x3, [%[r], 432]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[56] * B\n\t"
        "ldp	x9, x10, [%[a], 448]\n\t"
        "str	x4, [%[r], 440]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[57] * B\n\t"
        "str	x5, [%[r], 448]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[58] * B\n\t"
        "ldp	x9, x10, [%[a], 464]\n\t"
        "str	x3, [%[r], 456]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[59] * B\n\t"
        "str	x4, [%[r], 464]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[60] * B\n\t"
        "ldp	x9, x10, [%[a], 480]\n\t"
        "str	x5, [%[r], 472]\n\t"
        "adcs	x3, x3, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x5, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "# A[61] * B\n\t"
        "str	x3, [%[r], 480]\n\t"
        "adcs	x4, x4, x7\n\t"
        "mul	x6, %[b], x10\n\t"
        "mov	x3, xzr\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x5, xzr, xzr\n\t"
        "adds	x4, x4, x6\n\t"
        "# A[62] * B\n\t"
        "ldp	x9, x10, [%[a], 496]\n\t"
        "str	x4, [%[r], 488]\n\t"
        "adcs	x5, x5, x7\n\t"
        "mul	x6, %[b], x9\n\t"
        "mov	x4, xzr\n\t"
        "umulh	x7, %[b], x9\n\t"
        "adc	x3, xzr, xzr\n\t"
        "adds	x5, x5, x6\n\t"
        "# A[63] * B\n\t"
        "str	x5, [%[r], 496]\n\t"
        "mul	x6, %[b], x10\n\t"
        "adcs	x3, x3, x7\n\t"
        "umulh	x7, %[b], x10\n\t"
        "adc	x4, xzr, xzr\n\t"
        "adds	x3, x3, x6\n\t"
        "adc	x4, x4, x7\n\t"
        "stp	x3, x4, [%[r], 504]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );
#endif
}

#if (defined(WOLFSSL_HAVE_SP_RSA) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)) || defined(WOLFSSL_HAVE_SP_DH)
/* r = 2^n mod m where n is the number of bits to reduce by.
 * Given m must be 4096 bits, just need to subtract.
 *
 * r  A single precision number.
 * m  A single precision number.
 */
static void sp_4096_mont_norm_64(sp_digit* r, const sp_digit* m)
{
    XMEMSET(r, 0, sizeof(sp_digit) * 64);

    /* r = 2^n mod m */
    sp_4096_sub_in_place_64(r, m);
}

#endif /* (WOLFSSL_HAVE_SP_RSA & !WOLFSSL_RSA_PUBLIC_ONLY) | WOLFSSL_HAVE_SP_DH */
/* Reduce the number back to 4096 bits using Montgomery reduction.
 *
 * a   A single precision number to reduce in place.
 * m   The single precision number representing the modulus.
 * mp  The digit representing the negative inverse of m mod 2^n.
 */
SP_NOINLINE static void sp_4096_mont_reduce_64(sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    __asm__ __volatile__ (
        "ldp	x11, x12, [%[a], 0]\n\t"
        "ldp	x13, x14, [%[a], 16]\n\t"
        "ldp	x15, x16, [%[a], 32]\n\t"
        "ldp	x17, x19, [%[a], 48]\n\t"
        "ldp	x20, x21, [%[a], 64]\n\t"
        "ldp	x22, x23, [%[a], 80]\n\t"
        "# No carry yet\n\t"
        "mov	x3, xzr\n\t"
        "# i = 0..63\n\t"
        "mov	x4, 64\n\t"
        "\n1:\n\t"
        "# mu = a[i] * mp\n\t"
        "mul	x10, %[mp], x11\n\t"
        "ldp	x24, x25, [%[m], 0]\n\t"
        "ldp	x26, x27, [%[m], 16]\n\t"
        "# a[i+0] += m[0] * mu\n\t"
        "mul	x5, x24, x10\n\t"
        "umulh	x6, x24, x10\n\t"
        "# a[i+1] += m[1] * mu\n\t"
        "adds	x11, x11, x5\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x11, x12, x5\n\t"
        "# a[i+2] += m[2] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x11, x11, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x12, x13, x5\n\t"
        "# a[i+3] += m[3] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x12, x12, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x13, x14, x5\n\t"
        "ldp	x24, x25, [%[m], 32]\n\t"
        "ldp	x26, x27, [%[m], 48]\n\t"
        "# a[i+4] += m[4] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x13, x13, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x14, x15, x5\n\t"
        "# a[i+5] += m[5] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x14, x14, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x15, x16, x5\n\t"
        "# a[i+6] += m[6] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x15, x15, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x16, x17, x5\n\t"
        "# a[i+7] += m[7] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x16, x16, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x17, x19, x5\n\t"
        "ldp	x24, x25, [%[m], 64]\n\t"
        "ldp	x26, x27, [%[m], 80]\n\t"
        "# a[i+8] += m[8] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x17, x17, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x19, x20, x5\n\t"
        "# a[i+9] += m[9] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x19, x19, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x20, x21, x5\n\t"
        "# a[i+10] += m[10] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x20, x20, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x21, x22, x5\n\t"
        "# a[i+11] += m[11] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x21, x21, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x22, x23, x5\n\t"
        "ldp	x24, x25, [%[m], 96]\n\t"
        "ldp	x26, x27, [%[m], 112]\n\t"
        "# a[i+12] += m[12] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x22, x22, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "ldr	x23, [%[a], 96]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x23, x23, x5\n\t"
        "# a[i+13] += m[13] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x23, x23, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "ldp	x8, x9, [%[a], 104]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+14] += m[14] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 104]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+15] += m[15] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 112]\n\t"
        "ldp	x8, x9, [%[a], 120]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 128]\n\t"
        "ldp	x26, x27, [%[m], 144]\n\t"
        "# a[i+16] += m[16] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 120]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+17] += m[17] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 128]\n\t"
        "ldp	x8, x9, [%[a], 136]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+18] += m[18] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 136]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+19] += m[19] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 144]\n\t"
        "ldp	x8, x9, [%[a], 152]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 160]\n\t"
        "ldp	x26, x27, [%[m], 176]\n\t"
        "# a[i+20] += m[20] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 152]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+21] += m[21] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 160]\n\t"
        "ldp	x8, x9, [%[a], 168]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+22] += m[22] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 168]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+23] += m[23] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 176]\n\t"
        "ldp	x8, x9, [%[a], 184]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 192]\n\t"
        "ldp	x26, x27, [%[m], 208]\n\t"
        "# a[i+24] += m[24] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 184]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+25] += m[25] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 192]\n\t"
        "ldp	x8, x9, [%[a], 200]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+26] += m[26] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 200]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+27] += m[27] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 208]\n\t"
        "ldp	x8, x9, [%[a], 216]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 224]\n\t"
        "ldp	x26, x27, [%[m], 240]\n\t"
        "# a[i+28] += m[28] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 216]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+29] += m[29] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 224]\n\t"
        "ldp	x8, x9, [%[a], 232]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+30] += m[30] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 232]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+31] += m[31] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 240]\n\t"
        "ldp	x8, x9, [%[a], 248]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 256]\n\t"
        "ldp	x26, x27, [%[m], 272]\n\t"
        "# a[i+32] += m[32] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 248]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+33] += m[33] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 256]\n\t"
        "ldp	x8, x9, [%[a], 264]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+34] += m[34] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 264]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+35] += m[35] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 272]\n\t"
        "ldp	x8, x9, [%[a], 280]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 288]\n\t"
        "ldp	x26, x27, [%[m], 304]\n\t"
        "# a[i+36] += m[36] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 280]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+37] += m[37] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 288]\n\t"
        "ldp	x8, x9, [%[a], 296]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+38] += m[38] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 296]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+39] += m[39] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 304]\n\t"
        "ldp	x8, x9, [%[a], 312]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 320]\n\t"
        "ldp	x26, x27, [%[m], 336]\n\t"
        "# a[i+40] += m[40] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 312]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+41] += m[41] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 320]\n\t"
        "ldp	x8, x9, [%[a], 328]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+42] += m[42] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 328]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+43] += m[43] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 336]\n\t"
        "ldp	x8, x9, [%[a], 344]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 352]\n\t"
        "ldp	x26, x27, [%[m], 368]\n\t"
        "# a[i+44] += m[44] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 344]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+45] += m[45] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 352]\n\t"
        "ldp	x8, x9, [%[a], 360]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+46] += m[46] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 360]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+47] += m[47] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 368]\n\t"
        "ldp	x8, x9, [%[a], 376]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 384]\n\t"
        "ldp	x26, x27, [%[m], 400]\n\t"
        "# a[i+48] += m[48] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 376]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+49] += m[49] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 384]\n\t"
        "ldp	x8, x9, [%[a], 392]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+50] += m[50] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 392]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+51] += m[51] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 400]\n\t"
        "ldp	x8, x9, [%[a], 408]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 416]\n\t"
        "ldp	x26, x27, [%[m], 432]\n\t"
        "# a[i+52] += m[52] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 408]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+53] += m[53] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 416]\n\t"
        "ldp	x8, x9, [%[a], 424]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+54] += m[54] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 424]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+55] += m[55] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 432]\n\t"
        "ldp	x8, x9, [%[a], 440]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 448]\n\t"
        "ldp	x26, x27, [%[m], 464]\n\t"
        "# a[i+56] += m[56] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 440]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+57] += m[57] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 448]\n\t"
        "ldp	x8, x9, [%[a], 456]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+58] += m[58] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 456]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+59] += m[59] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 464]\n\t"
        "ldp	x8, x9, [%[a], 472]\n\t"
        "umulh	x7, x27, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "ldp	x24, x25, [%[m], 480]\n\t"
        "ldp	x26, x27, [%[m], 496]\n\t"
        "# a[i+60] += m[60] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x24, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 472]\n\t"
        "umulh	x6, x24, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+61] += m[61] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x25, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 480]\n\t"
        "ldp	x8, x9, [%[a], 488]\n\t"
        "umulh	x7, x25, x10\n\t"
        "adds	x8, x8, x5\n\t"
        "# a[i+62] += m[62] * mu\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x8, x8, x6\n\t"
        "mul	x5, x26, x10\n\t"
        "adc	x7, x7, xzr\n\t"
        "str	x8, [%[a], 488]\n\t"
        "umulh	x6, x26, x10\n\t"
        "adds	x9, x9, x5\n\t"
        "# a[i+63] += m[63] * mu\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x9, x9, x7\n\t"
        "mul	x5, x27, x10\n\t"
        "adc	x6, x6, xzr\n\t"
        "str	x9, [%[a], 496]\n\t"
        "umulh	x7, x27, x10\n\t"
        "ldp	x8, x9, [%[a], 504]\n\t"
        "adds	x5, x5, x6\n\t"
        "adcs	x7, x7, x3\n\t"
        "cset	x3, cs\n\t"
        "adds	x8, x8, x5\n\t"
        "str	x8, [%[a], 504]\n\t"
        "adcs	x9, x9, x7\n\t"
        "str	x9, [%[a], 512]\n\t"
        "adc	x3, x3, xzr\n\t"
        "subs	x4, x4, 1\n\t"
        "add	%[a], %[a], 8\n\t"
        "b.ne	1b\n\t"
        "# Create mask\n\t"
        "neg	x3, x3\n\t"
        "mov   %[mp], %[a]\n\t"
        "sub	%[a], %[a], 512\n\t"
        "# Subtract masked modulus\n\t"
        "ldp	x4, x5, [%[m], 0]\n\t"
        "ldp	x6, x7, [%[m], 16]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "subs	x11, x11, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x12, x12, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x13, x13, x6\n\t"
        "stp	x11, x12, [%[a], 0]\n\t"
        "sbcs	x14, x14, x7\n\t"
        "stp	x13, x14, [%[a], 16]\n\t"
        "ldp	x4, x5, [%[m], 32]\n\t"
        "ldp	x6, x7, [%[m], 48]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x15, x15, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x16, x16, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x17, x17, x6\n\t"
        "stp	x15, x16, [%[a], 32]\n\t"
        "sbcs	x19, x19, x7\n\t"
        "stp	x17, x19, [%[a], 48]\n\t"
        "ldp	x4, x5, [%[m], 64]\n\t"
        "ldp	x6, x7, [%[m], 80]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x20, x20, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x21, x21, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x22, x22, x6\n\t"
        "stp	x20, x21, [%[a], 64]\n\t"
        "sbcs	x23, x23, x7\n\t"
        "stp	x22, x23, [%[a], 80]\n\t"
        "ldp	x4, x5, [%[m], 96]\n\t"
        "ldp	x6, x7, [%[m], 112]\n\t"
        "ldp	x8, x9, [%[mp], 96]\n\t"
        "ldp	x10, x11, [%[mp], 112]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 96]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 112]\n\t"
        "ldp	x4, x5, [%[m], 128]\n\t"
        "ldp	x6, x7, [%[m], 144]\n\t"
        "ldp	x8, x9, [%[mp], 128]\n\t"
        "ldp	x10, x11, [%[mp], 144]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 128]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 144]\n\t"
        "ldp	x4, x5, [%[m], 160]\n\t"
        "ldp	x6, x7, [%[m], 176]\n\t"
        "ldp	x8, x9, [%[mp], 160]\n\t"
        "ldp	x10, x11, [%[mp], 176]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 160]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 176]\n\t"
        "ldp	x4, x5, [%[m], 192]\n\t"
        "ldp	x6, x7, [%[m], 208]\n\t"
        "ldp	x8, x9, [%[mp], 192]\n\t"
        "ldp	x10, x11, [%[mp], 208]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 192]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 208]\n\t"
        "ldp	x4, x5, [%[m], 224]\n\t"
        "ldp	x6, x7, [%[m], 240]\n\t"
        "ldp	x8, x9, [%[mp], 224]\n\t"
        "ldp	x10, x11, [%[mp], 240]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 224]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 240]\n\t"
        "ldp	x4, x5, [%[m], 256]\n\t"
        "ldp	x6, x7, [%[m], 272]\n\t"
        "ldp	x8, x9, [%[mp], 256]\n\t"
        "ldp	x10, x11, [%[mp], 272]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 256]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 272]\n\t"
        "ldp	x4, x5, [%[m], 288]\n\t"
        "ldp	x6, x7, [%[m], 304]\n\t"
        "ldp	x8, x9, [%[mp], 288]\n\t"
        "ldp	x10, x11, [%[mp], 304]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 288]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 304]\n\t"
        "ldp	x4, x5, [%[m], 320]\n\t"
        "ldp	x6, x7, [%[m], 336]\n\t"
        "ldp	x8, x9, [%[mp], 320]\n\t"
        "ldp	x10, x11, [%[mp], 336]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 320]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 336]\n\t"
        "ldp	x4, x5, [%[m], 352]\n\t"
        "ldp	x6, x7, [%[m], 368]\n\t"
        "ldp	x8, x9, [%[mp], 352]\n\t"
        "ldp	x10, x11, [%[mp], 368]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 352]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 368]\n\t"
        "ldp	x4, x5, [%[m], 384]\n\t"
        "ldp	x6, x7, [%[m], 400]\n\t"
        "ldp	x8, x9, [%[mp], 384]\n\t"
        "ldp	x10, x11, [%[mp], 400]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 384]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 400]\n\t"
        "ldp	x4, x5, [%[m], 416]\n\t"
        "ldp	x6, x7, [%[m], 432]\n\t"
        "ldp	x8, x9, [%[mp], 416]\n\t"
        "ldp	x10, x11, [%[mp], 432]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 416]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 432]\n\t"
        "ldp	x4, x5, [%[m], 448]\n\t"
        "ldp	x6, x7, [%[m], 464]\n\t"
        "ldp	x8, x9, [%[mp], 448]\n\t"
        "ldp	x10, x11, [%[mp], 464]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 448]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 464]\n\t"
        "ldp	x4, x5, [%[m], 480]\n\t"
        "ldp	x6, x7, [%[m], 496]\n\t"
        "ldp	x8, x9, [%[mp], 480]\n\t"
        "ldp	x10, x11, [%[mp], 496]\n\t"
        "and	x4, x4, x3\n\t"
        "and	x5, x5, x3\n\t"
        "sbcs	x8, x8, x4\n\t"
        "and	x6, x6, x3\n\t"
        "sbcs	x9, x9, x5\n\t"
        "and	x7, x7, x3\n\t"
        "sbcs	x10, x10, x6\n\t"
        "stp	x8, x9, [%[a], 480]\n\t"
        "sbcs	x11, x11, x7\n\t"
        "stp	x10, x11, [%[a], 496]\n\t"
        : [a] "+r" (a), [mp] "+r" (mp)
        : [m] "r" (m)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x10", "x8", "x9", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x19", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27"
    );

}

/* Multiply two Montgomery form numbers mod the modulus (prime).
 * (r = a * b mod m)
 *
 * r   Result of multiplication.
 * a   First number to multiply in Montgomery form.
 * b   Second number to multiply in Montgomery form.
 * m   Modulus (prime).
 * mp  Montgomery mulitplier.
 */
SP_NOINLINE static void sp_4096_mont_mul_64(sp_digit* r, const sp_digit* a,
        const sp_digit* b, const sp_digit* m, sp_digit mp)
{
    sp_4096_mul_64(r, a, b);
    sp_4096_mont_reduce_64(r, m, mp);
}

/* Square the Montgomery form number. (r = a * a mod m)
 *
 * r   Result of squaring.
 * a   Number to square in Montgomery form.
 * m   Modulus (prime).
 * mp  Montgomery mulitplier.
 */
SP_NOINLINE static void sp_4096_mont_sqr_64(sp_digit* r, const sp_digit* a,
        const sp_digit* m, sp_digit mp)
{
    sp_4096_sqr_64(r, a);
    sp_4096_mont_reduce_64(r, m, mp);
}

#ifdef WOLFSSL_SP_SMALL
/* Sub b from a into r. (r = a - b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_4096_sub_64(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "add	x11, %[a], 512\n\t"
        "\n1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldp	x3, x4, [%[a]], #16\n\t"
        "ldp	x5, x6, [%[a]], #16\n\t"
        "ldp	x7, x8, [%[b]], #16\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x9, x10, [%[b]], #16\n\t"
        "sbcs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r]], #16\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r]], #16\n\t"
        "csetm	%[c], cc\n\t"
        "cmp	%[a], x11\n\t"
        "b.ne	1b\n\t"
        : [c] "+r" (c), [r] "+r" (r), [a] "+r" (a), [b] "+r" (b)
        :
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    return c;
}

#else
/* Sub b from a into r. (r = a - b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_4096_sub_64(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "subs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "ldp	x3, x4, [%[a], 32]\n\t"
        "ldp	x7, x8, [%[b], 32]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 48]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 48]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 32]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 48]\n\t"
        "ldp	x3, x4, [%[a], 64]\n\t"
        "ldp	x7, x8, [%[b], 64]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 80]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 80]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 64]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 80]\n\t"
        "ldp	x3, x4, [%[a], 96]\n\t"
        "ldp	x7, x8, [%[b], 96]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 112]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 112]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 96]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 112]\n\t"
        "ldp	x3, x4, [%[a], 128]\n\t"
        "ldp	x7, x8, [%[b], 128]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 144]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 144]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 128]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 144]\n\t"
        "ldp	x3, x4, [%[a], 160]\n\t"
        "ldp	x7, x8, [%[b], 160]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 176]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 176]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 160]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 176]\n\t"
        "ldp	x3, x4, [%[a], 192]\n\t"
        "ldp	x7, x8, [%[b], 192]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 208]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 208]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 192]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 208]\n\t"
        "ldp	x3, x4, [%[a], 224]\n\t"
        "ldp	x7, x8, [%[b], 224]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 240]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 240]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 224]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 240]\n\t"
        "ldp	x3, x4, [%[a], 256]\n\t"
        "ldp	x7, x8, [%[b], 256]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 272]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 272]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 256]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 272]\n\t"
        "ldp	x3, x4, [%[a], 288]\n\t"
        "ldp	x7, x8, [%[b], 288]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 304]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 304]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 288]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 304]\n\t"
        "ldp	x3, x4, [%[a], 320]\n\t"
        "ldp	x7, x8, [%[b], 320]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 336]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 336]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 320]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 336]\n\t"
        "ldp	x3, x4, [%[a], 352]\n\t"
        "ldp	x7, x8, [%[b], 352]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 368]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 368]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 352]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 368]\n\t"
        "ldp	x3, x4, [%[a], 384]\n\t"
        "ldp	x7, x8, [%[b], 384]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 400]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 400]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 384]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 400]\n\t"
        "ldp	x3, x4, [%[a], 416]\n\t"
        "ldp	x7, x8, [%[b], 416]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 432]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 432]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 416]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 432]\n\t"
        "ldp	x3, x4, [%[a], 448]\n\t"
        "ldp	x7, x8, [%[b], 448]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 464]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 464]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 448]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 464]\n\t"
        "ldp	x3, x4, [%[a], 480]\n\t"
        "ldp	x7, x8, [%[b], 480]\n\t"
        "sbcs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 496]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 496]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 480]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 496]\n\t"
        "csetm	%[r], cc\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return (sp_digit)r;
}

#endif /* WOLFSSL_SP_SMALL */
/* Divide the double width number (d1|d0) by the divisor. (d1|d0 / div)
 *
 * Assumes divisor has higest bit set.
 *
 * d1   The high order half of the number to divide.
 * d0   The low order half of the number to divide.
 * div  The divisor.
 * returns the result of the division.
 */
static sp_digit div_4096_word_64_cond(sp_digit d1, sp_digit d0, sp_digit div)
{
    __asm__ __volatile__ (
        "lsr	x8, %[div], 32\n\t"
        "add	x5, x8, 1\n\t"

        "udiv	x3, %[d1], x5\n\t"
        "lsl	x7, %[div], 32\n\t"
        "movz	x9, #1, lsl 32\n\t"
        "lsl	x6, x3, 32\n\t"
        "mul	x4, %[div], x6\n\t"
        "umulh	x3, %[div], x6\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "cmp	%[d1], x5\n\t"
        "b.lt	1f\n\t"
        "subs	%[d0], %[d0], x7\n\t"
        "add	x6, x6, x9\n\t"
        "sbc	%[d1], %[d1], x8\n\t"
        "1:\n\t"

        "extr	x3, %[d1], %[d0], 32\n\t"

        "udiv	x3, x3, x5\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "extr	x3, %[d1], %[d0], 32\n\t"

        "cmp	x3, x5\n\t"
        "b.lt	2f\n\t"
        "udiv   x3, x3, x5\n\t"
        "add    x6, x6, x3\n\t"
        "mul    x4, %[div], x3\n\t"
        "sub    %[d0], %[d0], x4\n\t"
        "2:\n\t"

        "udiv	x3, %[d0], %[div]\n\t"
        "add	%[d1], x6, x3\n\t"

        : [d1] "+r" (d1), [d0] "+r" (d0)
        : [div] "r" (div)
        : "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );

    return d1;
}

/* Divide d in a and put remainder into r (m*d + r = a)
 * m is not calculated as it is not needed at this time.
 *
 * a  Number to be divided.
 * d  Number to divide with.
 * m  Multiplier result.
 * r  Remainder from the division.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_4096_div_64_cond(const sp_digit* a, const sp_digit* d, sp_digit* m,
        sp_digit* r)
{
    sp_digit t1[128], t2[65];
    sp_digit div, r1;
    int i;

    (void)m;

    div = d[63];
    XMEMCPY(t1, a, sizeof(*t1) * 2 * 64);
    for (i = 63; i > 0; i--) {
        if (t1[i + 64] != d[i])
            break;
    }
    if (t1[i + 64] >= d[i]) {
        sp_4096_sub_in_place_64(&t1[64], d);
    }
    for (i = 63; i >= 0; i--) {
        if (t1[64 + i] == div) {
            r1 = SP_DIGIT_MAX;
        }
        else {
            r1 = div_4096_word_64_cond(t1[64 + i], t1[64 + i - 1], div);
        }

        sp_4096_mul_d_64(t2, d, r1);
        t1[64 + i] += sp_4096_sub_in_place_64(&t1[i], t2);
        t1[64 + i] -= t2[64];
        if (t1[64 + i] != 0) {
            t1[64 + i] += sp_4096_add_64(&t1[i], &t1[i], d);
            if (t1[64 + i] != 0)
                t1[64 + i] += sp_4096_add_64(&t1[i], &t1[i], d);
        }
    }

    for (i = 63; i > 0; i--) {
        if (t1[i] != d[i])
            break;
    }
    if (t1[i] >= d[i]) {
        sp_4096_sub_64(r, t1, d);
    }
    else {
        XMEMCPY(r, t1, sizeof(*t1) * 64);
    }

    return MP_OKAY;
}

/* Reduce a modulo m into r. (r = a mod m)
 *
 * r  A single precision number that is the reduced result.
 * a  A single precision number that is to be reduced.
 * m  A single precision number that is the modulus to reduce with.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_4096_mod_64_cond(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    return sp_4096_div_64_cond(a, m, NULL, r);
}

#if (defined(WOLFSSL_HAVE_SP_RSA) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)) || defined(WOLFSSL_HAVE_SP_DH)
/* Conditionally subtract b from a using the mask m.
 * m is -1 to subtract and 0 when not copying.
 *
 * r  A single precision number representing condition subtract result.
 * a  A single precision number to subtract from.
 * b  A single precision number to subtract.
 * m  Mask value to apply.
 */
static sp_digit sp_4096_cond_sub_64(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
#ifdef WOLFSSL_SP_SMALL
    sp_digit c = 0;

    __asm__ __volatile__ (
        "mov	x8, #0\n\t"
        "1:\n\t"
        "subs	%[c], xzr, %[c]\n\t"
        "ldr	x4, [%[a], x8]\n\t"
        "ldr	x5, [%[b], x8]\n\t"
        "and	x5, x5, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "csetm	%[c], cc\n\t"
        "str	x4, [%[r], x8]\n\t"
        "add	x8, x8, #8\n\t"
        "cmp	x8, 512\n\t"
        "b.lt	1b\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8", "x9", "x10", "x11", "x12"
    );

    return c;
#else
    __asm__ __volatile__ (

        "ldp	x5, x7, [%[b], 0]\n\t"
        "ldp	x11, x12, [%[b], 16]\n\t"
        "ldp	x4, x6, [%[a], 0]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 16]\n\t"
        "and	x7, x7, %[m]\n\t"
        "subs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 0]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 16]\n\t"
        "ldp	x5, x7, [%[b], 32]\n\t"
        "ldp	x11, x12, [%[b], 48]\n\t"
        "ldp	x4, x6, [%[a], 32]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 48]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 32]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 48]\n\t"
        "ldp	x5, x7, [%[b], 64]\n\t"
        "ldp	x11, x12, [%[b], 80]\n\t"
        "ldp	x4, x6, [%[a], 64]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 80]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 64]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 80]\n\t"
        "ldp	x5, x7, [%[b], 96]\n\t"
        "ldp	x11, x12, [%[b], 112]\n\t"
        "ldp	x4, x6, [%[a], 96]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 112]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 96]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 112]\n\t"
        "ldp	x5, x7, [%[b], 128]\n\t"
        "ldp	x11, x12, [%[b], 144]\n\t"
        "ldp	x4, x6, [%[a], 128]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 144]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 128]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 144]\n\t"
        "ldp	x5, x7, [%[b], 160]\n\t"
        "ldp	x11, x12, [%[b], 176]\n\t"
        "ldp	x4, x6, [%[a], 160]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 176]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 160]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 176]\n\t"
        "ldp	x5, x7, [%[b], 192]\n\t"
        "ldp	x11, x12, [%[b], 208]\n\t"
        "ldp	x4, x6, [%[a], 192]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 208]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 192]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 208]\n\t"
        "ldp	x5, x7, [%[b], 224]\n\t"
        "ldp	x11, x12, [%[b], 240]\n\t"
        "ldp	x4, x6, [%[a], 224]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 240]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 224]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 240]\n\t"
        "ldp	x5, x7, [%[b], 256]\n\t"
        "ldp	x11, x12, [%[b], 272]\n\t"
        "ldp	x4, x6, [%[a], 256]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 272]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 256]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 272]\n\t"
        "ldp	x5, x7, [%[b], 288]\n\t"
        "ldp	x11, x12, [%[b], 304]\n\t"
        "ldp	x4, x6, [%[a], 288]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 304]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 288]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 304]\n\t"
        "ldp	x5, x7, [%[b], 320]\n\t"
        "ldp	x11, x12, [%[b], 336]\n\t"
        "ldp	x4, x6, [%[a], 320]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 336]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 320]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 336]\n\t"
        "ldp	x5, x7, [%[b], 352]\n\t"
        "ldp	x11, x12, [%[b], 368]\n\t"
        "ldp	x4, x6, [%[a], 352]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 368]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 352]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 368]\n\t"
        "ldp	x5, x7, [%[b], 384]\n\t"
        "ldp	x11, x12, [%[b], 400]\n\t"
        "ldp	x4, x6, [%[a], 384]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 400]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 384]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 400]\n\t"
        "ldp	x5, x7, [%[b], 416]\n\t"
        "ldp	x11, x12, [%[b], 432]\n\t"
        "ldp	x4, x6, [%[a], 416]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 432]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 416]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 432]\n\t"
        "ldp	x5, x7, [%[b], 448]\n\t"
        "ldp	x11, x12, [%[b], 464]\n\t"
        "ldp	x4, x6, [%[a], 448]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 464]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 448]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 464]\n\t"
        "ldp	x5, x7, [%[b], 480]\n\t"
        "ldp	x11, x12, [%[b], 496]\n\t"
        "ldp	x4, x6, [%[a], 480]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 496]\n\t"
        "and	x7, x7, %[m]\n\t"
        "sbcs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 480]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 496]\n\t"
        "csetm	%[r], cc\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8", "x9", "x10", "x11", "x12"
    );

    return (sp_digit)r;
#endif /* WOLFSSL_SP_SMALL */
}

/* Divide the double width number (d1|d0) by the divisor. (d1|d0 / div)
 *
 * Assumes divisor has higest bit set.
 *
 * d1   The high order half of the number to divide.
 * d0   The low order half of the number to divide.
 * div  The divisor.
 * returns the result of the division.
 */
static sp_digit div_4096_word_64(sp_digit d1, sp_digit d0, sp_digit div)
{
    __asm__ __volatile__ (
        "lsr	x8, %[div], 32\n\t"
        "add	x5, x8, 1\n\t"

        "udiv	x3, %[d1], x5\n\t"
        "lsl	x7, %[div], 32\n\t"
        "movz	x9, #1, lsl 32\n\t"
        "lsl	x6, x3, 32\n\t"
        "mul	x4, %[div], x6\n\t"
        "umulh	x3, %[div], x6\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "cmp	%[d1], x5\n\t"
        "cset	x9, ge\n\t"
        "csetm	x10, ge\n\t"
        "lsl	x9, x9, #32\n\t"
        "and	x7, x7, x10\n\t"
        "and	x8, x8, x10\n\t"
        "subs	%[d0], %[d0], x7\n\t"
        "add	x6, x6, x9\n\t"
        "sbc	%[d1], %[d1], x8\n\t"

        "extr	x3, %[d1], %[d0], 32\n\t"

        "udiv	x3, x3, x5\n\t"
        "add	x6, x6, x3\n\t"
        "mul	x4, %[div], x3\n\t"
        "umulh	x3, %[div], x3\n\t"
        "subs	%[d0], %[d0], x4\n\t"
        "sbc	%[d1], %[d1], x3\n\t"

        "extr	x3, %[d1], %[d0], 32\n\t"

        "udiv   x3, x3, x5\n\t"
        "add    x6, x6, x3\n\t"
        "mul    x4, %[div], x3\n\t"
        "sub    %[d0], %[d0], x4\n\t"

        "udiv	x3, %[d0], %[div]\n\t"
        "add	%[d1], x6, x3\n\t"

        : [d1] "+r" (d1), [d0] "+r" (d0)
        : [div] "r" (div)
        : "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return d1;
}

/* AND m into each word of a and store in r.
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * m  Mask to AND against each digit.
 */
static void sp_4096_mask_64(sp_digit* r, const sp_digit* a, sp_digit m)
{
#ifdef WOLFSSL_SP_SMALL
    int i;

    for (i=0; i<64; i++) {
        r[i] = a[i] & m;
    }
#else
    int i;

    for (i = 0; i < 64; i += 8) {
        r[i+0] = a[i+0] & m;
        r[i+1] = a[i+1] & m;
        r[i+2] = a[i+2] & m;
        r[i+3] = a[i+3] & m;
        r[i+4] = a[i+4] & m;
        r[i+5] = a[i+5] & m;
        r[i+6] = a[i+6] & m;
        r[i+7] = a[i+7] & m;
    }
#endif
}

/* Compare a with b in constant time.
 *
 * a  A single precision integer.
 * b  A single precision integer.
 * return -ve, 0 or +ve if a is less than, equal to or greater than b
 * respectively.
 */
static sp_int64 sp_4096_cmp_64(const sp_digit* a, const sp_digit* b)
{
#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "mov	x3, #0\n\t"
        "mov	x2, #-1\n\t"
        "mov	x10, #64\n\t"
        "add	%[a], %[a], #496\n\t"
        "add	%[b], %[b], #496\n\t"
        "1:\n\t"
        "ldp	x6, x7, [%[a]], -16\n\t"
        "ldp	x8, x9, [%[b]], -16\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x10, x10, #2\n\t"
        "b.ne	1b\n\t"
        "cmp	x2, #0\n\t"
        "cset	%[a], eq\n\t"
        "orr	%[a], %[a], x3\n\t"
        : [a] "+r" (a), [b] "+r" (b)
        :
        : "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );
#else
    __asm__ __volatile__ (
        "mov	x3, #0\n\t"
        "mov	x2, #-1\n\t"
        "ldp	x6, x7, [%[a], 496]\n\t"
        "ldp	x8, x9, [%[b], 496]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 480]\n\t"
        "ldp	x8, x9, [%[b], 480]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 464]\n\t"
        "ldp	x8, x9, [%[b], 464]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 448]\n\t"
        "ldp	x8, x9, [%[b], 448]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 432]\n\t"
        "ldp	x8, x9, [%[b], 432]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 416]\n\t"
        "ldp	x8, x9, [%[b], 416]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 400]\n\t"
        "ldp	x8, x9, [%[b], 400]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 384]\n\t"
        "ldp	x8, x9, [%[b], 384]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 368]\n\t"
        "ldp	x8, x9, [%[b], 368]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 352]\n\t"
        "ldp	x8, x9, [%[b], 352]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 336]\n\t"
        "ldp	x8, x9, [%[b], 336]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 320]\n\t"
        "ldp	x8, x9, [%[b], 320]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 304]\n\t"
        "ldp	x8, x9, [%[b], 304]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 288]\n\t"
        "ldp	x8, x9, [%[b], 288]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 272]\n\t"
        "ldp	x8, x9, [%[b], 272]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 256]\n\t"
        "ldp	x8, x9, [%[b], 256]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 240]\n\t"
        "ldp	x8, x9, [%[b], 240]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 224]\n\t"
        "ldp	x8, x9, [%[b], 224]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 208]\n\t"
        "ldp	x8, x9, [%[b], 208]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 192]\n\t"
        "ldp	x8, x9, [%[b], 192]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 176]\n\t"
        "ldp	x8, x9, [%[b], 176]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 160]\n\t"
        "ldp	x8, x9, [%[b], 160]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 144]\n\t"
        "ldp	x8, x9, [%[b], 144]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 128]\n\t"
        "ldp	x8, x9, [%[b], 128]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 112]\n\t"
        "ldp	x8, x9, [%[b], 112]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 96]\n\t"
        "ldp	x8, x9, [%[b], 96]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 80]\n\t"
        "ldp	x8, x9, [%[b], 80]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 64]\n\t"
        "ldp	x8, x9, [%[b], 64]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 48]\n\t"
        "ldp	x8, x9, [%[b], 48]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 32]\n\t"
        "ldp	x8, x9, [%[b], 32]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 16]\n\t"
        "ldp	x8, x9, [%[b], 16]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 0]\n\t"
        "ldp	x8, x9, [%[b], 0]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "cmp	x2, #0\n\t"
        "cset	%[a], eq\n\t"
        "orr	%[a], %[a], x3\n\t"
        : [a] "+r" (a)
        : [b] "r" (b)
        : "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );
#endif

    return (sp_int64)a;
}

/* Divide d in a and put remainder into r (m*d + r = a)
 * m is not calculated as it is not needed at this time.
 *
 * a  Number to be divided.
 * d  Number to divide with.
 * m  Multiplier result.
 * r  Remainder from the division.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_4096_div_64(const sp_digit* a, const sp_digit* d, sp_digit* m,
        sp_digit* r)
{
    sp_digit t1[128], t2[65];
    sp_digit div, r1;
    int i;

    (void)m;

    div = d[63];
    XMEMCPY(t1, a, sizeof(*t1) * 2 * 64);
    r1 = sp_4096_cmp_64(&t1[64], d) >= 0;
    sp_4096_cond_sub_64(&t1[64], &t1[64], d, (sp_digit)0 - r1);
    for (i = 63; i >= 0; i--) {
        volatile sp_digit mask = (sp_digit)0 - (t1[64 + i] == div);
        sp_digit hi = t1[64 + i] + mask;
        r1 = div_4096_word_64(hi, t1[64 + i - 1], div);
        r1 |= mask;

        sp_4096_mul_d_64(t2, d, r1);
        t1[64 + i] += sp_4096_sub_in_place_64(&t1[i], t2);
        t1[64 + i] -= t2[64];
        sp_4096_mask_64(t2, d, t1[64 + i]);
        t1[64 + i] += sp_4096_add_64(&t1[i], &t1[i], t2);
        sp_4096_mask_64(t2, d, t1[64 + i]);
        t1[64 + i] += sp_4096_add_64(&t1[i], &t1[i], t2);
    }

    r1 = sp_4096_cmp_64(t1, d) >= 0;
    sp_4096_cond_sub_64(r, t1, d, (sp_digit)0 - r1);

    return MP_OKAY;
}

/* Reduce a modulo m into r. (r = a mod m)
 *
 * r  A single precision number that is the reduced result.
 * a  A single precision number that is to be reduced.
 * m  A single precision number that is the modulus to reduce with.
 * returns MP_OKAY indicating success.
 */
static WC_INLINE int sp_4096_mod_64(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    return sp_4096_div_64(a, m, NULL, r);
}

#if (defined(WOLFSSL_HAVE_SP_RSA) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)) || \
                                                     defined(WOLFSSL_HAVE_SP_DH)
#ifdef WOLFSSL_SP_SMALL
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns  0 on success.
 * returns  MEMORY_E on dynamic memory allocation failure.
 * returns  MP_VAL when base is even or exponent is 0.
 */
static int sp_4096_mod_exp_64(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* td = NULL;
#else
    sp_digit td[8 * 128];
#endif
    sp_digit* t[8];
    sp_digit* norm = NULL;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c;
    byte y;
    int err = MP_OKAY;

    if (bits == 0) {
        err = MP_VAL;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        td = (sp_digit*)XMALLOC(sizeof(sp_digit) * (8 * 128), NULL,
                                DYNAMIC_TYPE_TMP_BUFFER);
        if (td == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        norm = td;
        for (i=0; i<8; i++) {
            t[i] = td + i * 128;
        }

        sp_4096_mont_setup(m, &mp);
        sp_4096_mont_norm_64(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 64U);
        if (reduceA != 0) {
            err = sp_4096_mod_64(t[1] + 64, a, m);
            if (err == MP_OKAY) {
                err = sp_4096_mod_64(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 64, a, sizeof(sp_digit) * 64);
            err = sp_4096_mod_64(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_4096_mont_sqr_64(t[ 2], t[ 1], m, mp);
        sp_4096_mont_mul_64(t[ 3], t[ 2], t[ 1], m, mp);
        sp_4096_mont_sqr_64(t[ 4], t[ 2], m, mp);
        sp_4096_mont_mul_64(t[ 5], t[ 3], t[ 2], m, mp);
        sp_4096_mont_sqr_64(t[ 6], t[ 3], m, mp);
        sp_4096_mont_mul_64(t[ 7], t[ 4], t[ 3], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 3;
        if (c == 64) {
            c = 61;
        }
        if (c < 0) {
            /* Number of bits in top word is less than number needed. */
            c = -c;
            y = (byte)(n << c);
            n = e[i--];
            y |= (byte)(n >> (64 - c));
            n <<= c;
            c = 64 - c;
        }
        else if (c == 0) {
            /* All bits in top word used. */
            y = (byte)n;
        }
        else {
            y = (byte)(n >> c);
            n <<= 64 - c;
        }
        XMEMCPY(r, t[y], sizeof(sp_digit) * 64);
        for (; i>=0 || c>=3; ) {
            if (c == 0) {
                n = e[i--];
                y = (byte)(n >> 61);
                n <<= 3;
                c = 61;
            }
            else if (c < 3) {
                y = (byte)(n >> 61);
                n = e[i--];
                c = 3 - c;
                y |= (byte)(n >> (64 - c));
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (byte)((n >> 61) & 0x7);
                n <<= 3;
                c -= 3;
            }

            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);

            sp_4096_mont_mul_64(r, r, t[y], m, mp);
        }

        XMEMSET(&r[64], 0, sizeof(sp_digit) * 64U);
        sp_4096_mont_reduce_64(r, m, mp);

        mask = 0 - (sp_4096_cmp_64(r, m) >= 0);
        sp_4096_cond_sub_64(r, r, m, mask);
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (td != NULL)
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return err;
}
#else
/* Modular exponentiate a to the e mod m. (r = a^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * a     A single precision number being exponentiated.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns  0 on success.
 * returns  MEMORY_E on dynamic memory allocation failure.
 * returns  MP_VAL when base is even or exponent is 0.
 */
static int sp_4096_mod_exp_64(sp_digit* r, const sp_digit* a, const sp_digit* e,
        int bits, const sp_digit* m, int reduceA)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* td = NULL;
#else
    sp_digit td[16 * 128];
#endif
    sp_digit* t[16];
    sp_digit* norm = NULL;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit mask;
    int i;
    int c;
    byte y;
    int err = MP_OKAY;

    if (bits == 0) {
        err = MP_VAL;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        td = (sp_digit*)XMALLOC(sizeof(sp_digit) * (16 * 128), NULL,
                                DYNAMIC_TYPE_TMP_BUFFER);
        if (td == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        norm = td;
        for (i=0; i<16; i++) {
            t[i] = td + i * 128;
        }

        sp_4096_mont_setup(m, &mp);
        sp_4096_mont_norm_64(norm, m);

        XMEMSET(t[1], 0, sizeof(sp_digit) * 64U);
        if (reduceA != 0) {
            err = sp_4096_mod_64(t[1] + 64, a, m);
            if (err == MP_OKAY) {
                err = sp_4096_mod_64(t[1], t[1], m);
            }
        }
        else {
            XMEMCPY(t[1] + 64, a, sizeof(sp_digit) * 64);
            err = sp_4096_mod_64(t[1], t[1], m);
        }
    }

    if (err == MP_OKAY) {
        sp_4096_mont_sqr_64(t[ 2], t[ 1], m, mp);
        sp_4096_mont_mul_64(t[ 3], t[ 2], t[ 1], m, mp);
        sp_4096_mont_sqr_64(t[ 4], t[ 2], m, mp);
        sp_4096_mont_mul_64(t[ 5], t[ 3], t[ 2], m, mp);
        sp_4096_mont_sqr_64(t[ 6], t[ 3], m, mp);
        sp_4096_mont_mul_64(t[ 7], t[ 4], t[ 3], m, mp);
        sp_4096_mont_sqr_64(t[ 8], t[ 4], m, mp);
        sp_4096_mont_mul_64(t[ 9], t[ 5], t[ 4], m, mp);
        sp_4096_mont_sqr_64(t[10], t[ 5], m, mp);
        sp_4096_mont_mul_64(t[11], t[ 6], t[ 5], m, mp);
        sp_4096_mont_sqr_64(t[12], t[ 6], m, mp);
        sp_4096_mont_mul_64(t[13], t[ 7], t[ 6], m, mp);
        sp_4096_mont_sqr_64(t[14], t[ 7], m, mp);
        sp_4096_mont_mul_64(t[15], t[ 8], t[ 7], m, mp);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 4;
        if (c == 64) {
            c = 60;
        }
        if (c < 0) {
            /* Number of bits in top word is less than number needed. */
            c = -c;
            y = (byte)(n << c);
            n = e[i--];
            y |= (byte)(n >> (64 - c));
            n <<= c;
            c = 64 - c;
        }
        else if (c == 0) {
            /* All bits in top word used. */
            y = (byte)n;
        }
        else {
            y = (byte)(n >> c);
            n <<= 64 - c;
        }
        XMEMCPY(r, t[y], sizeof(sp_digit) * 64);
        for (; i>=0 || c>=4; ) {
            if (c == 0) {
                n = e[i--];
                y = (byte)(n >> 60);
                n <<= 4;
                c = 60;
            }
            else if (c < 4) {
                y = (byte)(n >> 60);
                n = e[i--];
                c = 4 - c;
                y |= (byte)(n >> (64 - c));
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (byte)((n >> 60) & 0xf);
                n <<= 4;
                c -= 4;
            }

            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);

            sp_4096_mont_mul_64(r, r, t[y], m, mp);
        }

        XMEMSET(&r[64], 0, sizeof(sp_digit) * 64U);
        sp_4096_mont_reduce_64(r, m, mp);

        mask = 0 - (sp_4096_cmp_64(r, m) >= 0);
        sp_4096_cond_sub_64(r, r, m, mask);
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (td != NULL)
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return err;
}
#endif /* WOLFSSL_SP_SMALL */
#endif /* (WOLFSSL_HAVE_SP_RSA && !WOLFSSL_RSA_PUBLIC_ONLY) || WOLFSSL_HAVE_SP_DH */

#endif /* (WOLFSSL_HAVE_SP_RSA && !WOLFSSL_RSA_PUBLIC_ONLY) || WOLFSSL_HAVE_SP_DH */
#ifdef WOLFSSL_HAVE_SP_RSA
/* RSA public key operation.
 *
 * in      Array of bytes representing the number to exponentiate, base.
 * inLen   Number of bytes in base.
 * em      Public exponent.
 * mm      Modulus.
 * out     Buffer to hold big-endian bytes of exponentiation result.
 *         Must be at least 512 bytes long.
 * outLen  Number of bytes in result.
 * returns 0 on success, MP_TO_E when the outLen is too small, MP_READ_E when
 * an array is too long and MEMORY_E when dynamic memory allocation fails.
 */
int sp_RsaPublic_4096(const byte* in, word32 inLen, const mp_int* em,
    const mp_int* mm, byte* out, word32* outLen)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* a = NULL;
#else
    sp_digit a[64 * 5];
#endif
    sp_digit* m = NULL;
    sp_digit* r = NULL;
    sp_digit *ah = NULL;
    sp_digit e[1] = {0};
    int err = MP_OKAY;

    if (*outLen < 512) {
        err = MP_TO_E;
    }
    else if (mp_count_bits(em) > 64 || inLen > 512 ||
                                                     mp_count_bits(mm) != 4096) {
        err = MP_READ_E;
    }
    else if (mp_iseven(mm)) {
        err = MP_VAL;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        a = (sp_digit*)XMALLOC(sizeof(sp_digit) * 64 * 5, NULL,
                                                              DYNAMIC_TYPE_RSA);
        if (a == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        ah = a + 64;
        r = a + 64 * 2;
        m = r + 64 * 2;

        sp_4096_from_bin(ah, 64, in, inLen);
#if DIGIT_BIT >= 64
        e[0] = em->dp[0];
#else
        e[0] = em->dp[0];
        if (em->used > 1) {
            e[0] |= ((sp_digit)em->dp[1]) << DIGIT_BIT;
        }
#endif
        if (e[0] == 0) {
            err = MP_EXPTMOD_E;
        }
    }
    if (err == MP_OKAY) {
        sp_4096_from_mp(m, 64, mm);

        if (e[0] == 0x10001) {
            int i;
            sp_digit mp;

            sp_4096_mont_setup(m, &mp);

            /* Convert to Montgomery form. */
            XMEMSET(a, 0, sizeof(sp_digit) * 64);
            err = sp_4096_mod_64_cond(r, a, m);
            /* Montgomery form: r = a.R mod m */

            if (err == MP_OKAY) {
                /* r = a ^ 0x10000 => r = a squared 16 times */
                for (i = 15; i >= 0; i--) {
                    sp_4096_mont_sqr_64(r, r, m, mp);
                }
                /* mont_red(r.R.R) = (r.R.R / R) mod m = r.R mod m
                 * mont_red(r.R * a) = (r.R.a / R) mod m = r.a mod m
                 */
                sp_4096_mont_mul_64(r, r, ah, m, mp);

                for (i = 63; i > 0; i--) {
                    if (r[i] != m[i]) {
                        break;
                    }
                }
                if (r[i] >= m[i]) {
                    sp_4096_sub_in_place_64(r, m);
                }
            }
        }
        else if (e[0] == 0x3) {
            if (err == MP_OKAY) {
                sp_4096_sqr_64(r, ah);
                err = sp_4096_mod_64_cond(r, r, m);
            }
            if (err == MP_OKAY) {
                sp_4096_mul_64(r, ah, r);
                err = sp_4096_mod_64_cond(r, r, m);
            }
        }
        else {
            int i;
            sp_digit mp;

            sp_4096_mont_setup(m, &mp);

            /* Convert to Montgomery form. */
            XMEMSET(a, 0, sizeof(sp_digit) * 64);
            err = sp_4096_mod_64_cond(a, a, m);

            if (err == MP_OKAY) {
                for (i = 63; i >= 0; i--) {
                    if (e[0] >> i) {
                        break;
                    }
                }

                XMEMCPY(r, a, sizeof(sp_digit) * 64);
                for (i--; i >= 0; i--) {
                    sp_4096_mont_sqr_64(r, r, m, mp);
                    if (((e[0] >> i) & 1) == 1) {
                        sp_4096_mont_mul_64(r, r, a, m, mp);
                    }
                }
                XMEMSET(&r[64], 0, sizeof(sp_digit) * 64);
                sp_4096_mont_reduce_64(r, m, mp);

                for (i = 63; i > 0; i--) {
                    if (r[i] != m[i]) {
                        break;
                    }
                }
                if (r[i] >= m[i]) {
                    sp_4096_sub_in_place_64(r, m);
                }
            }
        }
    }

    if (err == MP_OKAY) {
        sp_4096_to_bin_64(r, out);
        *outLen = 512;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (a != NULL)
        XFREE(a, NULL, DYNAMIC_TYPE_RSA);
#endif

    return err;
}

#ifndef WOLFSSL_RSA_PUBLIC_ONLY
#ifdef WOLFSSL_SP_SMALL
/* Conditionally add a and b using the mask m.
 * m is -1 to add and 0 when not.
 *
 * r  A single precision number representing conditional add result.
 * a  A single precision number to add with.
 * b  A single precision number to add.
 * m  Mask value to apply.
 */
static sp_digit sp_4096_cond_add_32(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
    sp_digit c = 0;

    __asm__ __volatile__ (
        "mov	x8, #0\n\t"
        "1:\n\t"
        "adds	%[c], %[c], #-1\n\t"
        "ldr	x4, [%[a], x8]\n\t"
        "ldr	x5, [%[b], x8]\n\t"
        "and	x5, x5, %[m]\n\t"
        "adcs	x4, x4, x5\n\t"
        "cset	%[c], cs\n\t"
        "str	x4, [%[r], x8]\n\t"
        "add	x8, x8, #8\n\t"
        "cmp	x8, 256\n\t"
        "b.lt	1b\n\t"
        : [c] "+r" (c)
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x5", "x8", "x9", "x10", "x11", "x12"
    );

    return c;
}
#endif /* WOLFSSL_SP_SMALL */

/* RSA private key operation.
 *
 * in      Array of bytes representing the number to exponentiate, base.
 * inLen   Number of bytes in base.
 * dm      Private exponent.
 * pm      First prime.
 * qm      Second prime.
 * dpm     First prime's CRT exponent.
 * dqm     Second prime's CRT exponent.
 * qim     Inverse of second prime mod p.
 * mm      Modulus.
 * out     Buffer to hold big-endian bytes of exponentiation result.
 *         Must be at least 512 bytes long.
 * outLen  Number of bytes in result.
 * returns 0 on success, MP_TO_E when the outLen is too small, MP_READ_E when
 * an array is too long and MEMORY_E when dynamic memory allocation fails.
 */
int sp_RsaPrivate_4096(const byte* in, word32 inLen, const mp_int* dm,
    const mp_int* pm, const mp_int* qm, const mp_int* dpm, const mp_int* dqm,
    const mp_int* qim, const mp_int* mm, byte* out, word32* outLen)
{
#if defined(SP_RSA_PRIVATE_EXP_D) || defined(RSA_LOW_MEM)
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* d = NULL;
#else
    sp_digit  d[64 * 4];
#endif
    sp_digit* a = NULL;
    sp_digit* m = NULL;
    sp_digit* r = NULL;
    int err = MP_OKAY;

    (void)pm;
    (void)qm;
    (void)dpm;
    (void)dqm;
    (void)qim;

    if (*outLen < 512U) {
        err = MP_TO_E;
    }
    if (err == MP_OKAY) {
        if (mp_count_bits(dm) > 4096) {
           err = MP_READ_E;
        }
        else if (inLen > 512) {
            err = MP_READ_E;
        }
        else if (mp_count_bits(mm) != 4096) {
            err = MP_READ_E;
        }
        else if (mp_iseven(mm)) {
            err = MP_VAL;
        }
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        d = (sp_digit*)XMALLOC(sizeof(sp_digit) * 64 * 4, NULL,
                                                              DYNAMIC_TYPE_RSA);
        if (d == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        a = d + 64;
        m = a + 128;
        r = a;

        sp_4096_from_bin(a, 64, in, inLen);
        sp_4096_from_mp(d, 64, dm);
        sp_4096_from_mp(m, 64, mm);
        err = sp_4096_mod_exp_64(r, a, d, 4096, m, 0);
    }

    if (err == MP_OKAY) {
        sp_4096_to_bin_64(r, out);
        *outLen = 512;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (d != NULL)
#endif
    {
        /* only "a" and "r" are sensitive and need zeroized (same pointer) */
        if (a != NULL)
            ForceZero(a, sizeof(sp_digit) * 64);
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
        XFREE(d, NULL, DYNAMIC_TYPE_RSA);
#endif
    }

    return err;
#else
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* a = NULL;
#else
    sp_digit a[32 * 11];
#endif
    sp_digit* p = NULL;
    sp_digit* q = NULL;
    sp_digit* dp = NULL;
    sp_digit* tmpa = NULL;
    sp_digit* tmpb = NULL;
    sp_digit* r = NULL;
    sp_digit* qi = NULL;
    sp_digit* dq = NULL;
    sp_digit c;
    int err = MP_OKAY;

    (void)dm;
    (void)mm;

    if (*outLen < 512) {
        err = MP_TO_E;
    }
    else if (inLen > 512 || mp_count_bits(mm) != 4096) {
        err = MP_READ_E;
    }
    else if (mp_iseven(mm)) {
        err = MP_VAL;
    }
    else if (mp_iseven(pm)) {
        err = MP_VAL;
    }
    else if (mp_iseven(qm)) {
        err = MP_VAL;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        a = (sp_digit*)XMALLOC(sizeof(sp_digit) * 32 * 11, NULL,
                                                              DYNAMIC_TYPE_RSA);
        if (a == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        p = a + 64 * 2;
        q = p + 32;
        qi = dq = dp = q + 32;
        tmpa = qi + 32;
        tmpb = tmpa + 64;
        r = a;

        sp_4096_from_bin(a, 64, in, inLen);
        sp_4096_from_mp(p, 32, pm);
        sp_4096_from_mp(q, 32, qm);
        sp_4096_from_mp(dp, 32, dpm);

        err = sp_2048_mod_exp_32(tmpa, a, dp, 2048, p, 1);
    }
    if (err == MP_OKAY) {
        sp_4096_from_mp(dq, 32, dqm);
        err = sp_2048_mod_exp_32(tmpb, a, dq, 2048, q, 1);
    }

    if (err == MP_OKAY) {
        c = sp_2048_sub_in_place_32(tmpa, tmpb);
        c += sp_4096_cond_add_32(tmpa, tmpa, p, c);
        sp_4096_cond_add_32(tmpa, tmpa, p, c);

        sp_2048_from_mp(qi, 32, qim);
        sp_2048_mul_32(tmpa, tmpa, qi);
        err = sp_2048_mod_32(tmpa, tmpa, p);
    }

    if (err == MP_OKAY) {
        sp_2048_mul_32(tmpa, q, tmpa);
        XMEMSET(&tmpb[32], 0, sizeof(sp_digit) * 32);
        sp_4096_add_64(r, tmpb, tmpa);

        sp_4096_to_bin_64(r, out);
        *outLen = 512;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (a != NULL)
#endif
    {
        ForceZero(a, sizeof(sp_digit) * 32 * 11);
    #if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
        XFREE(a, NULL, DYNAMIC_TYPE_RSA);
    #endif
    }
#endif /* SP_RSA_PRIVATE_EXP_D || RSA_LOW_MEM */
    return err;
}
#endif /* WOLFSSL_RSA_PUBLIC_ONLY */
#endif /* WOLFSSL_HAVE_SP_RSA */
#if defined(WOLFSSL_HAVE_SP_DH) || (defined(WOLFSSL_HAVE_SP_RSA) && \
                                              !defined(WOLFSSL_RSA_PUBLIC_ONLY))
/* Convert an array of sp_digit to an mp_int.
 *
 * a  A single precision integer.
 * r  A multi-precision integer.
 */
static int sp_4096_to_mp(const sp_digit* a, mp_int* r)
{
    int err;

    err = mp_grow(r, (4096 + DIGIT_BIT - 1) / DIGIT_BIT);
    if (err == MP_OKAY) { /*lint !e774 case where err is always MP_OKAY*/
#if DIGIT_BIT == 64
        XMEMCPY(r->dp, a, sizeof(sp_digit) * 64);
        r->used = 64;
        mp_clamp(r);
#elif DIGIT_BIT < 64
        int i;
        int j = 0;
        int s = 0;

        r->dp[0] = 0;
        for (i = 0; i < 64; i++) {
            r->dp[j] |= (mp_digit)(a[i] << s);
            r->dp[j] &= ((sp_digit)1 << DIGIT_BIT) - 1;
            s = DIGIT_BIT - s;
            r->dp[++j] = (mp_digit)(a[i] >> s);
            while (s + DIGIT_BIT <= 64) {
                s += DIGIT_BIT;
                r->dp[j++] &= ((sp_digit)1 << DIGIT_BIT) - 1;
                if (s == SP_WORD_SIZE) {
                    r->dp[j] = 0;
                }
                else {
                    r->dp[j] = (mp_digit)(a[i] >> s);
                }
            }
            s = 64 - s;
        }
        r->used = (4096 + DIGIT_BIT - 1) / DIGIT_BIT;
        mp_clamp(r);
#else
        int i;
        int j = 0;
        int s = 0;

        r->dp[0] = 0;
        for (i = 0; i < 64; i++) {
            r->dp[j] |= ((mp_digit)a[i]) << s;
            if (s + 64 >= DIGIT_BIT) {
    #if DIGIT_BIT != 32 && DIGIT_BIT != 64
                r->dp[j] &= ((sp_digit)1 << DIGIT_BIT) - 1;
    #endif
                s = DIGIT_BIT - s;
                r->dp[++j] = a[i] >> s;
                s = 64 - s;
            }
            else {
                s += 64;
            }
        }
        r->used = (4096 + DIGIT_BIT - 1) / DIGIT_BIT;
        mp_clamp(r);
#endif
    }

    return err;
}

/* Perform the modular exponentiation for Diffie-Hellman.
 *
 * base  Base. MP integer.
 * exp   Exponent. MP integer.
 * mod   Modulus. MP integer.
 * res   Result. MP integer.
 * returns 0 on success, MP_READ_E if there are too many bytes in an array
 * and MEMORY_E if memory allocation fails.
 */
int sp_ModExp_4096(const mp_int* base, const mp_int* exp, const mp_int* mod,
    mp_int* res)
{
    int err = MP_OKAY;
    sp_digit b[128];
    sp_digit e[64];
    sp_digit m[64];
    sp_digit* r = b;
    int expBits = mp_count_bits(exp);

    if (mp_count_bits(base) > 4096) {
        err = MP_READ_E;
    }
    else if (expBits > 4096) {
        err = MP_READ_E;
    }
    else if (mp_count_bits(mod) != 4096) {
        err = MP_READ_E;
    }
    else if (mp_iseven(mod)) {
        err = MP_VAL;
    }

    if (err == MP_OKAY) {
        sp_4096_from_mp(b, 64, base);
        sp_4096_from_mp(e, 64, exp);
        sp_4096_from_mp(m, 64, mod);

        err = sp_4096_mod_exp_64(r, b, e, expBits, m, 0);
    }

    if (err == MP_OKAY) {
        err = sp_4096_to_mp(r, res);
    }

    XMEMSET(e, 0, sizeof(e));

    return err;
}

#ifdef WOLFSSL_HAVE_SP_DH

#ifdef HAVE_FFDHE_4096
static void sp_4096_lshift_64(sp_digit* r, const sp_digit* a, byte n)
{
    word64 n64 = n;
    __asm__ __volatile__ (
        "mov	x6, 63\n\t"
        "sub	x6, x6, %[n]\n\t"
        "ldr	x3, [%[a], 504]\n\t"
        "lsr	x4, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x4, x4, x6\n\t"
        "ldr	x2, [%[a], 496]\n\t"
        "str	x4, [%[r], 512]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 488]\n\t"
        "str	x3, [%[r], 504]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 480]\n\t"
        "str	x2, [%[r], 496]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 472]\n\t"
        "str	x4, [%[r], 488]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 464]\n\t"
        "str	x3, [%[r], 480]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 456]\n\t"
        "str	x2, [%[r], 472]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 448]\n\t"
        "str	x4, [%[r], 464]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 440]\n\t"
        "str	x3, [%[r], 456]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 432]\n\t"
        "str	x2, [%[r], 448]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 424]\n\t"
        "str	x4, [%[r], 440]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 416]\n\t"
        "str	x3, [%[r], 432]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 408]\n\t"
        "str	x2, [%[r], 424]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 400]\n\t"
        "str	x4, [%[r], 416]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 392]\n\t"
        "str	x3, [%[r], 408]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 384]\n\t"
        "str	x2, [%[r], 400]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 376]\n\t"
        "str	x4, [%[r], 392]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 368]\n\t"
        "str	x3, [%[r], 384]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 360]\n\t"
        "str	x2, [%[r], 376]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 352]\n\t"
        "str	x4, [%[r], 368]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 344]\n\t"
        "str	x3, [%[r], 360]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 336]\n\t"
        "str	x2, [%[r], 352]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 328]\n\t"
        "str	x4, [%[r], 344]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 320]\n\t"
        "str	x3, [%[r], 336]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 312]\n\t"
        "str	x2, [%[r], 328]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 304]\n\t"
        "str	x4, [%[r], 320]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 296]\n\t"
        "str	x3, [%[r], 312]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 288]\n\t"
        "str	x2, [%[r], 304]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 280]\n\t"
        "str	x4, [%[r], 296]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 272]\n\t"
        "str	x3, [%[r], 288]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 264]\n\t"
        "str	x2, [%[r], 280]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 256]\n\t"
        "str	x4, [%[r], 272]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 248]\n\t"
        "str	x3, [%[r], 264]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 240]\n\t"
        "str	x2, [%[r], 256]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 232]\n\t"
        "str	x4, [%[r], 248]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 224]\n\t"
        "str	x3, [%[r], 240]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 216]\n\t"
        "str	x2, [%[r], 232]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 208]\n\t"
        "str	x4, [%[r], 224]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 200]\n\t"
        "str	x3, [%[r], 216]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 192]\n\t"
        "str	x2, [%[r], 208]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 184]\n\t"
        "str	x4, [%[r], 200]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 176]\n\t"
        "str	x3, [%[r], 192]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 168]\n\t"
        "str	x2, [%[r], 184]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 160]\n\t"
        "str	x4, [%[r], 176]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 152]\n\t"
        "str	x3, [%[r], 168]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 144]\n\t"
        "str	x2, [%[r], 160]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 136]\n\t"
        "str	x4, [%[r], 152]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 128]\n\t"
        "str	x3, [%[r], 144]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 120]\n\t"
        "str	x2, [%[r], 136]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 112]\n\t"
        "str	x4, [%[r], 128]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 104]\n\t"
        "str	x3, [%[r], 120]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 96]\n\t"
        "str	x2, [%[r], 112]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 88]\n\t"
        "str	x4, [%[r], 104]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 80]\n\t"
        "str	x3, [%[r], 96]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 72]\n\t"
        "str	x2, [%[r], 88]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 64]\n\t"
        "str	x4, [%[r], 80]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 56]\n\t"
        "str	x3, [%[r], 72]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 48]\n\t"
        "str	x2, [%[r], 64]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 40]\n\t"
        "str	x4, [%[r], 56]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 32]\n\t"
        "str	x3, [%[r], 48]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 24]\n\t"
        "str	x2, [%[r], 40]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "ldr	x2, [%[a], 16]\n\t"
        "str	x4, [%[r], 32]\n\t"
        "lsr	x5, x2, 1\n\t"
        "lsl	x2, x2, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x3, x3, x5\n\t"
        "ldr	x4, [%[a], 8]\n\t"
        "str	x3, [%[r], 24]\n\t"
        "lsr	x5, x4, 1\n\t"
        "lsl	x4, x4, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x2, x2, x5\n\t"
        "ldr	x3, [%[a], 0]\n\t"
        "str	x2, [%[r], 16]\n\t"
        "lsr	x5, x3, 1\n\t"
        "lsl	x3, x3, %[n]\n\t"
        "lsr	x5, x5, x6\n\t"
        "orr	x4, x4, x5\n\t"
        "stp	x3, x4, [%[r]]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [n] "r" (n64)
        : "memory", "x2", "x3", "x4", "x5", "x6"
    );
}

/* Modular exponentiate 2 to the e mod m. (r = 2^e mod m)
 *
 * r     A single precision number that is the result of the operation.
 * e     A single precision number that is the exponent.
 * bits  The number of bits in the exponent.
 * m     A single precision number that is the modulus.
 * returns  0 on success.
 * returns  MEMORY_E on dynamic memory allocation failure.
 * returns  MP_VAL when base is even.
 */
static int sp_4096_mod_exp_2_64(sp_digit* r, const sp_digit* e, int bits,
        const sp_digit* m)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* td = NULL;
#else
    sp_digit td[193];
#endif
    sp_digit* norm = NULL;
    sp_digit* tmp = NULL;
    sp_digit mp = 1;
    sp_digit n;
    sp_digit o;
    sp_digit mask;
    int i;
    int c;
    byte y;
    int err = MP_OKAY;

    if (bits == 0) {
        err = MP_VAL;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (err == MP_OKAY) {
        td = (sp_digit*)XMALLOC(sizeof(sp_digit) * 193, NULL,
                                DYNAMIC_TYPE_TMP_BUFFER);
        if (td == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        norm = td;
        tmp = td + 128;

        sp_4096_mont_setup(m, &mp);
        sp_4096_mont_norm_64(norm, m);

        i = (bits - 1) / 64;
        n = e[i--];
        c = bits & 63;
        if (c == 0) {
            c = 64;
        }
        c -= bits % 6;
        if (c == 64) {
            c = 58;
        }
        if (c < 0) {
            /* Number of bits in top word is less than number needed. */
            c = -c;
            y = (byte)(n << c);
            n = e[i--];
            y |= (byte)(n >> (64 - c));
            n <<= c;
            c = 64 - c;
        }
        else if (c == 0) {
            /* All bits in top word used. */
            y = (byte)n;
        }
        else {
            y = (byte)(n >> c);
            n <<= 64 - c;
        }
        sp_4096_lshift_64(r, norm, y);
        for (; i>=0 || c>=6; ) {
            if (c == 0) {
                n = e[i--];
                y = (byte)(n >> 58);
                n <<= 6;
                c = 58;
            }
            else if (c < 6) {
                y = (byte)(n >> 58);
                n = e[i--];
                c = 6 - c;
                y |= (byte)(n >> (64 - c));
                n <<= c;
                c = 64 - c;
            }
            else {
                y = (byte)((n >> 58) & 0x3f);
                n <<= 6;
                c -= 6;
            }

            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);
            sp_4096_mont_sqr_64(r, r, m, mp);

            sp_4096_lshift_64(r, r, y);
            sp_4096_mul_d_64(tmp, norm, r[64]);
            r[64] = 0;
            o = sp_4096_add_64(r, r, tmp);
            sp_4096_cond_sub_64(r, r, m, (sp_digit)0 - o);
        }

        XMEMSET(&r[64], 0, sizeof(sp_digit) * 64U);
        sp_4096_mont_reduce_64(r, m, mp);

        mask = 0 - (sp_4096_cmp_64(r, m) >= 0);
        sp_4096_cond_sub_64(r, r, m, mask);
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (td != NULL)
        XFREE(td, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return err;
}
#endif /* HAVE_FFDHE_4096 */

/* Perform the modular exponentiation for Diffie-Hellman.
 *
 * base     Base.
 * exp      Array of bytes that is the exponent.
 * expLen   Length of data, in bytes, in exponent.
 * mod      Modulus.
 * out      Buffer to hold big-endian bytes of exponentiation result.
 *          Must be at least 512 bytes long.
 * outLen   Length, in bytes, of exponentiation result.
 * returns 0 on success, MP_READ_E if there are too many bytes in an array
 * and MEMORY_E if memory allocation fails.
 */
int sp_DhExp_4096(const mp_int* base, const byte* exp, word32 expLen,
    const mp_int* mod, byte* out, word32* outLen)
{
    int err = MP_OKAY;
    sp_digit b[128];
    sp_digit e[64];
    sp_digit m[64];
    sp_digit* r = b;
    word32 i;

    if (mp_count_bits(base) > 4096) {
        err = MP_READ_E;
    }
    else if (expLen > 512) {
        err = MP_READ_E;
    }
    else if (mp_count_bits(mod) != 4096) {
        err = MP_READ_E;
    }
    else if (mp_iseven(mod)) {
        err = MP_VAL;
    }

    if (err == MP_OKAY) {
        sp_4096_from_mp(b, 64, base);
        sp_4096_from_bin(e, 64, exp, expLen);
        sp_4096_from_mp(m, 64, mod);

    #ifdef HAVE_FFDHE_4096
        if (base->used == 1 && base->dp[0] == 2 && m[63] == (sp_digit)-1)
            err = sp_4096_mod_exp_2_64(r, e, expLen * 8, m);
        else
    #endif
            err = sp_4096_mod_exp_64(r, b, e, expLen * 8, m, 0);

    }

    if (err == MP_OKAY) {
        sp_4096_to_bin_64(r, out);
        *outLen = 512;
        for (i=0; i<512 && out[i] == 0; i++) {
            /* Search for first non-zero. */
        }
        *outLen -= i;
        XMEMMOVE(out, out + i, *outLen);

    }

    XMEMSET(e, 0, sizeof(e));

    return err;
}
#endif /* WOLFSSL_HAVE_SP_DH */

#endif /* WOLFSSL_HAVE_SP_DH | (WOLFSSL_HAVE_SP_RSA & !WOLFSSL_RSA_PUBLIC_ONLY) */

#endif /* WOLFSSL_SP_4096 */

#endif /* WOLFSSL_HAVE_SP_RSA | WOLFSSL_HAVE_SP_DH */
#ifdef WOLFSSL_HAVE_SP_ECC
#ifndef WOLFSSL_SP_NO_256

/* Point structure to use. */
typedef struct sp_point_256 {
    /* X ordinate of point. */
    sp_digit x[2 * 4];
    /* Y ordinate of point. */
    sp_digit y[2 * 4];
    /* Z ordinate of point. */
    sp_digit z[2 * 4];
    /* Indicates point is at infinity. */
    int infinity;
} sp_point_256;

/* The modulus (prime) of the curve P256. */
static const sp_digit p256_mod[4] = {
    0xffffffffffffffffL,0x00000000ffffffffL,0x0000000000000000L,
    0xffffffff00000001L
};
/* The Montgomery normalizer for modulus of the curve P256. */
static const sp_digit p256_norm_mod[4] = {
    0x0000000000000001L,0xffffffff00000000L,0xffffffffffffffffL,
    0x00000000fffffffeL
};
/* The Montgomery multiplier for modulus of the curve P256. */
static const sp_digit p256_mp_mod = 0x0000000000000001;
#if defined(WOLFSSL_VALIDATE_ECC_KEYGEN) || defined(HAVE_ECC_SIGN) || \
                                            defined(HAVE_ECC_VERIFY)
/* The order of the curve P256. */
static const sp_digit p256_order[4] = {
    0xf3b9cac2fc632551L,0xbce6faada7179e84L,0xffffffffffffffffL,
    0xffffffff00000000L
};
#endif
/* The order of the curve P256 minus 2. */
static const sp_digit p256_order2[4] = {
    0xf3b9cac2fc63254fL,0xbce6faada7179e84L,0xffffffffffffffffL,
    0xffffffff00000000L
};
#if defined(HAVE_ECC_SIGN) || defined(HAVE_ECC_VERIFY)
/* The Montgomery normalizer for order of the curve P256. */
static const sp_digit p256_norm_order[4] = {
    0x0c46353d039cdaafL,0x4319055258e8617bL,0x0000000000000000L,
    0x00000000ffffffffL
};
#endif
#if defined(HAVE_ECC_SIGN) || defined(HAVE_ECC_VERIFY)
/* The Montgomery multiplier for order of the curve P256. */
static const sp_digit p256_mp_order = 0xccd1c8aaee00bc4fL;
#endif
#ifdef WOLFSSL_SP_SMALL
/* The base point of curve P256. */
static const sp_point_256 p256_base = {
    /* X ordinate */
    {
        0xf4a13945d898c296L,0x77037d812deb33a0L,0xf8bce6e563a440f2L,
        0x6b17d1f2e12c4247L,
        (sp_digit)0, (sp_digit)0, (sp_digit)0, (sp_digit)0
    },
    /* Y ordinate */
    {
        0xcbb6406837bf51f5L,0x2bce33576b315eceL,0x8ee7eb4a7c0f9e16L,
        0x4fe342e2fe1a7f9bL,
        (sp_digit)0, (sp_digit)0, (sp_digit)0, (sp_digit)0
    },
    /* Z ordinate */
    {
        0x0000000000000001L,0x0000000000000000L,0x0000000000000000L,
        0x0000000000000000L,
        (sp_digit)0, (sp_digit)0, (sp_digit)0, (sp_digit)0
    },
    /* infinity */
    0
};
#endif /* WOLFSSL_SP_SMALL */
#if defined(HAVE_ECC_CHECK_KEY) || defined(HAVE_COMP_KEY)
static const sp_digit p256_b[4] = {
    0x3bce3c3e27d2604bL,0x651d06b0cc53b0f6L,0xb3ebbd55769886bcL,
    0x5ac635d8aa3a93e7L
};
#endif

#ifdef WOLFSSL_SP_SMALL
/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_256_mul_4(sp_digit* r, const sp_digit* a, const sp_digit* b)
{
    sp_digit tmp[8];

    __asm__ __volatile__ (
        "mov	x5, xzr\n\t"
        "mov	x6, xzr\n\t"
        "mov	x7, xzr\n\t"
        "mov	x8, xzr\n\t"
        "\n1:\n\t"
        "subs	x3, x5, 24\n\t"
        "csel	x3, xzr, x3, cc\n\t"
        "sub	x4, x5, x3\n\t"
        "\n2:\n\t"
        "ldr	x10, [%[a], x3]\n\t"
        "ldr	x11, [%[b], x4]\n\t"
        "mul	x9, x10, x11\n\t"
        "umulh	x10, x10, x11\n\t"
        "adds	x6, x6, x9\n\t"
        "adcs	x7, x7, x10\n\t"
        "adc	x8, x8, xzr\n\t"
        "add	x3, x3, #8\n\t"
        "sub	x4, x4, #8\n\t"
        "cmp	x3, 32\n\t"
        "b.eq	3f\n\t"
        "cmp	x3, x5\n\t"
        "b.le	2b\n\t"
        "\n3:\n\t"
        "str	x6, [%[r], x5]\n\t"
        "mov	x6, x7\n\t"
        "mov	x7, x8\n\t"
        "mov	x8, #0\n\t"
        "add	x5, x5, #8\n\t"
        "cmp	x5, 48\n\t"
        "b.le	1b\n\t"
        "str	x6, [%[r], x5]\n\t"
        :
        : [r] "r" (tmp), [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11"
    );

    XMEMCPY(r, tmp, sizeof(tmp));
}

#else
/* Multiply a and b into r. (r = a * b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static void sp_256_mul_4(sp_digit* r, const sp_digit* a, const sp_digit* b)
{
    sp_digit tmp[4];

    __asm__ __volatile__ (
        "ldp       x16, x17, [%[a], 0]\n\t"
        "ldp       x21, x22, [%[b], 0]\n\t"
        "#  A[0] * B[0]\n\t"
        "mul       x8, x16, x21\n\t"
        "ldr       x19, [%[a], 16]\n\t"
        "umulh     x9, x16, x21\n\t"
        "ldr       x23, [%[b], 16]\n\t"
        "#  A[0] * B[1]\n\t"
        "mul       x4, x16, x22\n\t"
        "ldr       x20, [%[a], 24]\n\t"
        "umulh     x5, x16, x22\n\t"
        "ldr       x24, [%[b], 24]\n\t"
        "adds  x9, x9, x4\n\t"
        "#  A[1] * B[0]\n\t"
        "mul       x4, x17, x21\n\t"
        "adc   x10, xzr, x5\n\t"
        "umulh     x5, x17, x21\n\t"
        "adds  x9, x9, x4\n\t"
        "#  A[0] * B[2]\n\t"
        "mul       x4, x16, x23\n\t"
        "adcs   x10, x10, x5\n\t"
        "umulh     x5, x16, x23\n\t"
        "adc     x11, xzr, xzr\n\t"
        "adds  x10, x10, x4\n\t"
        "#  A[1] * B[1]\n\t"
        "mul       x4, x17, x22\n\t"
        "adc   x11, x11, x5\n\t"
        "umulh     x5, x17, x22\n\t"
        "adds  x10, x10, x4\n\t"
        "#  A[2] * B[0]\n\t"
        "mul       x4, x19, x21\n\t"
        "adcs   x11, x11, x5\n\t"
        "umulh     x5, x19, x21\n\t"
        "adc     x12, xzr, xzr\n\t"
        "adds  x10, x10, x4\n\t"
        "#  A[0] * B[3]\n\t"
        "mul       x4, x16, x24\n\t"
        "adcs   x11, x11, x5\n\t"
        "umulh     x5, x16, x24\n\t"
        "adc     x12, x12, xzr\n\t"
        "adds  x11, x11, x4\n\t"
        "#  A[1] * B[2]\n\t"
        "mul       x4, x17, x23\n\t"
        "adcs   x12, x12, x5\n\t"
        "umulh     x5, x17, x23\n\t"
        "adc     x13, xzr, xzr\n\t"
        "adds  x11, x11, x4\n\t"
        "#  A[2] * B[1]\n\t"
        "mul       x4, x19, x22\n\t"
        "adcs   x12, x12, x5\n\t"
        "umulh     x5, x19, x22\n\t"
        "adc     x13, x13, xzr\n\t"
        "adds  x11, x11, x4\n\t"
        "#  A[3] * B[0]\n\t"
        "mul       x4, x20, x21\n\t"
        "adcs   x12, x12, x5\n\t"
        "umulh     x5, x20, x21\n\t"
        "adc     x13, x13, xzr\n\t"
        "adds  x11, x11, x4\n\t"
        "#  A[1] * B[3]\n\t"
        "mul       x4, x17, x24\n\t"
        "adcs   x12, x12, x5\n\t"
        "umulh     x5, x17, x24\n\t"
        "adc     x13, x13, xzr\n\t"
        "adds  x12, x12, x4\n\t"
        "#  A[2] * B[2]\n\t"
        "mul       x4, x19, x23\n\t"
        "adcs   x13, x13, x5\n\t"
        "umulh     x5, x19, x23\n\t"
        "adc     x14, xzr, xzr\n\t"
        "adds  x12, x12, x4\n\t"
        "#  A[3] * B[1]\n\t"
        "mul       x4, x20, x22\n\t"
        "adcs   x13, x13, x5\n\t"
        "umulh     x5, x20, x22\n\t"
        "adc     x14, x14, xzr\n\t"
        "adds  x12, x12, x4\n\t"
        "#  A[2] * B[3]\n\t"
        "mul       x4, x19, x24\n\t"
        "adcs   x13, x13, x5\n\t"
        "umulh     x5, x19, x24\n\t"
        "adc     x14, x14, xzr\n\t"
        "adds  x13, x13, x4\n\t"
        "#  A[3] * B[2]\n\t"
        "mul       x4, x20, x23\n\t"
        "adcs   x14, x14, x5\n\t"
        "umulh     x5, x20, x23\n\t"
        "adc     x15, xzr, xzr\n\t"
        "adds  x13, x13, x4\n\t"
        "#  A[3] * B[3]\n\t"
        "mul       x4, x20, x24\n\t"
        "adcs   x14, x14, x5\n\t"
        "umulh     x5, x20, x24\n\t"
        "adc     x15, x15, xzr\n\t"
        "adds  x14, x14, x4\n\t"
        "adc   x15, x15, x5\n\t"
        "stp	x8, x9, [%[r], 0]\n\t"
        "stp	x10, x11, [%[r], 16]\n\t"
        "stp	x12, x13, [%[r], 32]\n\t"
        "stp	x14, x15, [%[r], 48]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [tmp] "r" (tmp)
        : "memory", "x4", "x5", "x6", "x7", "x16", "x17", "x19", "x20", "x21", "x22", "x23", "x24", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15"
    );
}

#endif /* WOLFSSL_SP_SMALL */
/* Square a and put result in r. (r = a * a)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 */
static void sp_256_sqr_4(sp_digit* r, const sp_digit* a)
{
    __asm__ __volatile__ (
        "ldp       x16, x17, [%[a], 0]\n\t"
        "#  A[0] * A[1]\n\t"
        "mul	x9, x16, x17\n\t"
        "ldr       x19, [%[a], 16]\n\t"
        "umulh	x10, x16, x17\n\t"
        "ldr       x20, [%[a], 24]\n\t"
        "#  A[0] * A[2]\n\t"
        "mul	x4, x16, x19\n\t"
        "umulh	x5, x16, x19\n\t"
        "adds	x10, x10, x4\n\t"
        "#  A[0] * A[3]\n\t"
        "mul	x4, x16, x20\n\t"
        "adc	x11, xzr, x5\n\t"
        "umulh	x5, x16, x20\n\t"
        "adds	x11, x11, x4\n\t"
        "#  A[1] * A[2]\n\t"
        "mul	x4, x17, x19\n\t"
        "adc	x12, xzr, x5\n\t"
        "umulh	x5, x17, x19\n\t"
        "adds	x11, x11, x4\n\t"
        "#  A[1] * A[3]\n\t"
        "mul	x4, x17, x20\n\t"
        "adcs	x12, x12, x5\n\t"
        "umulh	x5, x17, x20\n\t"
        "adc	x13, xzr, xzr\n\t"
        "adds	x12, x12, x4\n\t"
        "#  A[2] * A[3]\n\t"
        "mul	x4, x19, x20\n\t"
        "adc	x13, x13, x5\n\t"
        "umulh	x5, x19, x20\n\t"
        "adds	x13, x13, x4\n\t"
        "adc	x14, xzr, x5\n\t"
        "# Double\n\t"
        "adds	x9, x9, x9\n\t"
        "adcs	x10, x10, x10\n\t"
        "adcs	x11, x11, x11\n\t"
        "adcs	x12, x12, x12\n\t"
        "adcs	x13, x13, x13\n\t"
        "#  A[0] * A[0]\n\t"
        "mul	x8, x16, x16\n\t"
        "adcs	x14, x14, x14\n\t"
        "umulh	x3, x16, x16\n\t"
        "cset	x15, cs\n\t"
        "#  A[1] * A[1]\n\t"
        "mul	x4, x17, x17\n\t"
        "adds	x9, x9, x3\n\t"
        "umulh	x5, x17, x17\n\t"
        "adcs	x10, x10, x4\n\t"
        "#  A[2] * A[2]\n\t"
        "mul	x6, x19, x19\n\t"
        "adcs	x11, x11, x5\n\t"
        "umulh	x7, x19, x19\n\t"
        "adcs	x12, x12, x6\n\t"
        "#  A[3] * A[3]\n\t"
        "mul	x16, x20, x20\n\t"
        "adcs	x13, x13, x7\n\t"
        "umulh	x17, x20, x20\n\t"
        "adcs	x14, x14, x16\n\t"
        "adc	x15, x15, x17\n\t"
        "stp	x8, x9, [%[r], 0]\n\t"
        "stp	x10, x11, [%[r], 16]\n\t"
        "stp	x12, x13, [%[r], 32]\n\t"
        "stp	x14, x15, [%[r], 48]\n\t"
        :
        : [r] "r" (r), [a] "r" (a)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x19", "x20"
    );
}

/* Add b to a into r. (r = a + b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_256_add_4(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "adds	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "adcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "adcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "adcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "cset	%[r], cs\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return (sp_digit)r;
}

/* Sub b from a into r. (r = a - b)
 *
 * r  A single precision integer.
 * a  A single precision integer.
 * b  A single precision integer.
 */
static sp_digit sp_256_sub_4(sp_digit* r, const sp_digit* a,
        const sp_digit* b)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "ldp	x7, x8, [%[b], 0]\n\t"
        "subs	x3, x3, x7\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "sbcs	x4, x4, x8\n\t"
        "ldp	x9, x10, [%[b], 16]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        "csetm	%[r], cc\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

    return (sp_digit)r;
}

/* Multiply a number by Montgomery normalizer mod modulus (prime).
 *
 * r  The resulting Montgomery form number.
 * a  The number to convert.
 * m  The modulus (prime).
 */
static int sp_256_mod_mul_norm_4(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    int64_t t[8];
    int64_t a32[8];
    int64_t o;

    (void)m;

    a32[0] = a[0] & 0xffffffff;
    a32[1] = a[0] >> 32;
    a32[2] = a[1] & 0xffffffff;
    a32[3] = a[1] >> 32;
    a32[4] = a[2] & 0xffffffff;
    a32[5] = a[2] >> 32;
    a32[6] = a[3] & 0xffffffff;
    a32[7] = a[3] >> 32;

    /*  1  1  0 -1 -1 -1 -1  0 */
    t[0] = 0 + a32[0] + a32[1] - a32[3] - a32[4] - a32[5] - a32[6];
    /*  0  1  1  0 -1 -1 -1 -1 */
    t[1] = 0 + a32[1] + a32[2] - a32[4] - a32[5] - a32[6] - a32[7];
    /*  0  0  1  1  0 -1 -1 -1 */
    t[2] = 0 + a32[2] + a32[3] - a32[5] - a32[6] - a32[7];
    /* -1 -1  0  2  2  1  0 -1 */
    t[3] = 0 - a32[0] - a32[1] + 2 * a32[3] + 2 * a32[4] + a32[5] - a32[7];
    /*  0 -1 -1  0  2  2  1  0 */
    t[4] = 0 - a32[1] - a32[2] + 2 * a32[4] + 2 * a32[5] + a32[6];
    /*  0  0 -1 -1  0  2  2  1 */
    t[5] = 0 - a32[2] - a32[3] + 2 * a32[5] + 2 * a32[6] + a32[7];
    /* -1 -1  0  0  0  1  3  2 */
    t[6] = 0 - a32[0] - a32[1] + a32[5] + 3 * a32[6] + 2 * a32[7];
    /*  1  0 -1 -1 -1 -1  0  3 */
    t[7] = 0 + a32[0] - a32[2] - a32[3] - a32[4] - a32[5] + 3 * a32[7];

    t[1] += t[0] >> 32; t[0] &= 0xffffffff;
    t[2] += t[1] >> 32; t[1] &= 0xffffffff;
    t[3] += t[2] >> 32; t[2] &= 0xffffffff;
    t[4] += t[3] >> 32; t[3] &= 0xffffffff;
    t[5] += t[4] >> 32; t[4] &= 0xffffffff;
    t[6] += t[5] >> 32; t[5] &= 0xffffffff;
    t[7] += t[6] >> 32; t[6] &= 0xffffffff;
    o     = t[7] >> 32; t[7] &= 0xffffffff;
    t[0] += o;
    t[3] -= o;
    t[6] -= o;
    t[7] += o;
    t[1] += t[0] >> 32; t[0] &= 0xffffffff;
    t[2] += t[1] >> 32; t[1] &= 0xffffffff;
    t[3] += t[2] >> 32; t[2] &= 0xffffffff;
    t[4] += t[3] >> 32; t[3] &= 0xffffffff;
    t[5] += t[4] >> 32; t[4] &= 0xffffffff;
    t[6] += t[5] >> 32; t[5] &= 0xffffffff;
    t[7] += t[6] >> 32; t[6] &= 0xffffffff;
    r[0] = (t[1] << 32) | t[0];
    r[1] = (t[3] << 32) | t[2];
    r[2] = (t[5] << 32) | t[4];
    r[3] = (t[7] << 32) | t[6];

    return MP_OKAY;
}

/* Convert an mp_int to an array of sp_digit.
 *
 * r  A single precision integer.
 * size  Maximum number of bytes to convert
 * a  A multi-precision integer.
 */
static void sp_256_from_mp(sp_digit* r, int size, const mp_int* a)
{
#if DIGIT_BIT == 64
    int i;
    int j = 0;

    for (i = 0; i < size; i++) {
        sp_digit mask =
            (((sp_digit)(a->used - i - 1)) >> (SP_WORD_SIZE - 1)) - 1;
        r[i] = a->dp[j] & mask;
        j += (int)(((sp_digit)1) -
                   (((sp_digit)(a->used - i - 2)) >> (SP_WORD_SIZE - 1)));
    }
#elif DIGIT_BIT > 64
    int i;
    int j = 0;
    word32 s = 0;

    r[0] = 0;
    for (i = 0; i < a->used && j < size; i++) {
        r[j] |= ((sp_digit)a->dp[i] << s);
        r[j] &= 0xffffffffffffffffl;
        s = 64U - s;
        if (j + 1 >= size) {
            break;
        }
        /* lint allow cast of mismatch word32 and mp_digit */
        r[++j] = (sp_digit)(a->dp[i] >> s); /*lint !e9033*/
        while ((s + 64U) <= (word32)DIGIT_BIT) {
            s += 64U;
            r[j] &= 0xffffffffffffffffl;
            if (j + 1 >= size) {
                break;
            }
            if (s < (word32)DIGIT_BIT) {
                /* lint allow cast of mismatch word32 and mp_digit */
                r[++j] = (sp_digit)(a->dp[i] >> s); /*lint !e9033*/
            }
            else {
                r[++j] = (sp_digit)0;
            }
        }
        s = (word32)DIGIT_BIT - s;
    }

    for (j++; j < size; j++) {
        r[j] = 0;
    }
#else
    int i;
    int j = 0;
    int s = 0;

    r[0] = 0;
    for (i = 0; i < a->used && j < size; i++) {
        r[j] |= ((sp_digit)a->dp[i]) << s;
        if (s + DIGIT_BIT >= 64) {
            r[j] &= 0xffffffffffffffffl;
            if (j + 1 >= size) {
                break;
            }
            s = 64 - s;
            if (s == DIGIT_BIT) {
                r[++j] = 0;
                s = 0;
            }
            else {
                r[++j] = a->dp[i] >> s;
                s = DIGIT_BIT - s;
            }
        }
        else {
            s += DIGIT_BIT;
        }
    }

    for (j++; j < size; j++) {
        r[j] = 0;
    }
#endif
}

/* Convert a point of type ecc_point to type sp_point_256.
 *
 * p   Point of type sp_point_256 (result).
 * pm  Point of type ecc_point.
 */
static void sp_256_point_from_ecc_point_4(sp_point_256* p,
        const ecc_point* pm)
{
    XMEMSET(p->x, 0, sizeof(p->x));
    XMEMSET(p->y, 0, sizeof(p->y));
    XMEMSET(p->z, 0, sizeof(p->z));
    sp_256_from_mp(p->x, 4, pm->x);
    sp_256_from_mp(p->y, 4, pm->y);
    sp_256_from_mp(p->z, 4, pm->z);
    p->infinity = 0;
}

/* Convert an array of sp_digit to an mp_int.
 *
 * a  A single precision integer.
 * r  A multi-precision integer.
 */
static int sp_256_to_mp(const sp_digit* a, mp_int* r)
{
    int err;

    err = mp_grow(r, (256 + DIGIT_BIT - 1) / DIGIT_BIT);
    if (err == MP_OKAY) { /*lint !e774 case where err is always MP_OKAY*/
#if DIGIT_BIT == 64
        XMEMCPY(r->dp, a, sizeof(sp_digit) * 4);
        r->used = 4;
        mp_clamp(r);
#elif DIGIT_BIT < 64
        int i;
        int j = 0;
        int s = 0;

        r->dp[0] = 0;
        for (i = 0; i < 4; i++) {
            r->dp[j] |= (mp_digit)(a[i] << s);
            r->dp[j] &= ((sp_digit)1 << DIGIT_BIT) - 1;
            s = DIGIT_BIT - s;
            r->dp[++j] = (mp_digit)(a[i] >> s);
            while (s + DIGIT_BIT <= 64) {
                s += DIGIT_BIT;
                r->dp[j++] &= ((sp_digit)1 << DIGIT_BIT) - 1;
                if (s == SP_WORD_SIZE) {
                    r->dp[j] = 0;
                }
                else {
                    r->dp[j] = (mp_digit)(a[i] >> s);
                }
            }
            s = 64 - s;
        }
        r->used = (256 + DIGIT_BIT - 1) / DIGIT_BIT;
        mp_clamp(r);
#else
        int i;
        int j = 0;
        int s = 0;

        r->dp[0] = 0;
        for (i = 0; i < 4; i++) {
            r->dp[j] |= ((mp_digit)a[i]) << s;
            if (s + 64 >= DIGIT_BIT) {
    #if DIGIT_BIT != 32 && DIGIT_BIT != 64
                r->dp[j] &= ((sp_digit)1 << DIGIT_BIT) - 1;
    #endif
                s = DIGIT_BIT - s;
                r->dp[++j] = a[i] >> s;
                s = 64 - s;
            }
            else {
                s += 64;
            }
        }
        r->used = (256 + DIGIT_BIT - 1) / DIGIT_BIT;
        mp_clamp(r);
#endif
    }

    return err;
}

/* Convert a point of type sp_point_256 to type ecc_point.
 *
 * p   Point of type sp_point_256.
 * pm  Point of type ecc_point (result).
 * returns MEMORY_E when allocation of memory in ecc_point fails otherwise
 * MP_OKAY.
 */
static int sp_256_point_to_ecc_point_4(const sp_point_256* p, ecc_point* pm)
{
    int err;

    err = sp_256_to_mp(p->x, pm->x);
    if (err == MP_OKAY) {
        err = sp_256_to_mp(p->y, pm->y);
    }
    if (err == MP_OKAY) {
        err = sp_256_to_mp(p->z, pm->z);
    }

    return err;
}

/* Conditionally copy a into r using the mask m.
 * m is -1 to copy and 0 when not.
 *
 * r  A single precision number to copy over.
 * a  A single precision number to copy.
 * m  Mask value to apply.
 */
static void sp_256_cond_copy_4(sp_digit* r, const sp_digit* a, sp_digit m)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[r], 0]\n\t"
        "ldp	x7, x8, [%[a], 0]\n\t"
        "eor	x7, x7, x3\n\t"
        "ldp	x5, x6, [%[r], 16]\n\t"
        "eor	x8, x8, x4\n\t"
        "ldp	x9, x10, [%[a], 16]\n\t"
        "eor	x9, x9, x5\n\t"
        "eor	x10, x10, x6\n\t"
        "and	x7, x7, %[m]\n\t"
        "and	x8, x8, %[m]\n\t"
        "and	x9, x9, %[m]\n\t"
        "and	x10, x10, %[m]\n\t"
        "eor	x3, x3, x7\n\t"
        "eor	x4, x4, x8\n\t"
        "eor	x5, x5, x9\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "eor	x6, x6, x10\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [m] "r" (m)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );
}

/* Multiply two Montgomery form numbers mod the modulus (prime).
 * (r = a * b mod m)
 *
 * r   Result of multiplication.
 * a   First number to multiply in Montgomery form.
 * b   Second number to multiply in Montgomery form.
 * m   Modulus (prime).
 * mp  Montgomery mulitplier.
 */
SP_NOINLINE static void sp_256_mont_mul_4(sp_digit* r, const sp_digit* a, const sp_digit* b,
        const sp_digit* m, sp_digit mp)
{
    (void)m;
    (void)mp;

    __asm__ __volatile__ (
        "ldp       x16, x17, [%[a], 0]\n\t"
        "ldp       x21, x22, [%[b], 0]\n\t"
        "#  A[0] * B[0]\n\t"
        "mul       x8, x16, x21\n\t"
        "ldr       x19, [%[a], 16]\n\t"
        "umulh     x9, x16, x21\n\t"
        "ldr       x23, [%[b], 16]\n\t"
        "#  A[0] * B[1]\n\t"
        "mul       x4, x16, x22\n\t"
        "ldr       x20, [%[a], 24]\n\t"
        "umulh     x5, x16, x22\n\t"
        "ldr       x24, [%[b], 24]\n\t"
        "adds  x9, x9, x4\n\t"
        "#  A[1] * B[0]\n\t"
        "mul       x4, x17, x21\n\t"
        "adc   x10, xzr, x5\n\t"
        "umulh     x5, x17, x21\n\t"
        "adds  x9, x9, x4\n\t"
        "#  A[0] * B[2]\n\t"
        "mul       x4, x16, x23\n\t"
        "adcs   x10, x10, x5\n\t"
        "umulh     x5, x16, x23\n\t"
        "adc     x11, xzr, xzr\n\t"
        "adds  x10, x10, x4\n\t"
        "#  A[1] * B[1]\n\t"
        "mul       x4, x17, x22\n\t"
        "adc   x11, x11, x5\n\t"
        "umulh     x5, x17, x22\n\t"
        "adds  x10, x10, x4\n\t"
        "#  A[2] * B[0]\n\t"
        "mul       x4, x19, x21\n\t"
        "adcs   x11, x11, x5\n\t"
        "umulh     x5, x19, x21\n\t"
        "adc     x12, xzr, xzr\n\t"
        "adds  x10, x10, x4\n\t"
        "#  A[0] * B[3]\n\t"
        "mul       x4, x16, x24\n\t"
        "adcs   x11, x11, x5\n\t"
        "umulh     x5, x16, x24\n\t"
        "adc     x12, x12, xzr\n\t"
        "adds  x11, x11, x4\n\t"
        "#  A[1] * B[2]\n\t"
        "mul       x4, x17, x23\n\t"
        "adcs   x12, x12, x5\n\t"
        "umulh     x5, x17, x23\n\t"
        "adc     x13, xzr, xzr\n\t"
        "adds  x11, x11, x4\n\t"
        "#  A[2] * B[1]\n\t"
        "mul       x4, x19, x22\n\t"
        "adcs   x12, x12, x5\n\t"
        "umulh     x5, x19, x22\n\t"
        "adc     x13, x13, xzr\n\t"
        "adds  x11, x11, x4\n\t"
        "#  A[3] * B[0]\n\t"
        "mul       x4, x20, x21\n\t"
        "adcs   x12, x12, x5\n\t"
        "umulh     x5, x20, x21\n\t"
        "adc     x13, x13, xzr\n\t"
        "adds  x11, x11, x4\n\t"
        "#  A[1] * B[3]\n\t"
        "mul       x4, x17, x24\n\t"
        "adcs   x12, x12, x5\n\t"
        "umulh     x5, x17, x24\n\t"
        "adc     x13, x13, xzr\n\t"
        "adds  x12, x12, x4\n\t"
        "#  A[2] * B[2]\n\t"
        "mul       x4, x19, x23\n\t"
        "adcs   x13, x13, x5\n\t"
        "umulh     x5, x19, x23\n\t"
        "adc     x14, xzr, xzr\n\t"
        "adds  x12, x12, x4\n\t"
        "#  A[3] * B[1]\n\t"
        "mul       x4, x20, x22\n\t"
        "adcs   x13, x13, x5\n\t"
        "umulh     x5, x20, x22\n\t"
        "adc     x14, x14, xzr\n\t"
        "adds  x12, x12, x4\n\t"
        "#  A[2] * B[3]\n\t"
        "mul       x4, x19, x24\n\t"
        "adcs   x13, x13, x5\n\t"
        "umulh     x5, x19, x24\n\t"
        "adc     x14, x14, xzr\n\t"
        "adds  x13, x13, x4\n\t"
        "#  A[3] * B[2]\n\t"
        "mul       x4, x20, x23\n\t"
        "adcs   x14, x14, x5\n\t"
        "umulh     x5, x20, x23\n\t"
        "adc     x15, xzr, xzr\n\t"
        "adds  x13, x13, x4\n\t"
        "#  A[3] * B[3]\n\t"
        "mul       x4, x20, x24\n\t"
        "adcs   x14, x14, x5\n\t"
        "umulh     x5, x20, x24\n\t"
        "adc     x15, x15, xzr\n\t"
        "adds  x14, x14, x4\n\t"
        "mov	x4, x8\n\t"
        "adc   x15, x15, x5\n\t"
        "# Start Reduction\n\t"
        "mov	x5, x9\n\t"
        "mov	x6, x10\n\t"
        "# mu = a[0]-a[3] + a[0]-a[2] << 32 << 64 + (a[0] * 2) << 192\n\t"
        "#    - a[0] << 32 << 192\n\t"
        "#   + (a[0] * 2) << 192\n\t"
        "#   a[0]-a[2] << 32\n\t"
        "extr	x10, x10, x9, 32\n\t"
        "add	x7, x11, x8\n\t"
        "extr	x9, x9, x8, 32\n\t"
        "add	x7, x7, x8\n\t"
        "#   + a[0]-a[2] << 32 << 64\n\t"
        "#   - a[0] << 32 << 192\n\t"
        "adds	x5, x5, x8, lsl #32\n\t"
        "sub	x7, x7, x8, lsl #32\n\t"
        "adcs	x6, x6, x9\n\t"
        "adc	x7, x7, x10\n\t"
        "# a += (mu << 256) - (mu << 224) + (mu << 192) + (mu << 96) - mu\n\t"
        "#   a += mu << 256\n\t"
        "adds	x12, x12, x4\n\t"
        "adcs	x13, x13, x5\n\t"
        "adcs	x14, x14, x6\n\t"
        "adcs	x15, x15, x7\n\t"
        "cset	x8, cs\n\t"
        "#   a += mu << 192\n\t"
        "# mu <<= 32\n\t"
        "#   a += (mu << 32) << 64\n\t"
        "adds	x11, x11, x4\n\t"
        "adcs	x12, x12, x5\n\t"
        "adcs	x13, x13, x6\n\t"
        "lsr	x16, x7, 32\n\t"
        "adcs	x14, x14, x7\n\t"
        "extr	x7, x7, x6, 32\n\t"
        "adcs	x15, x15, xzr\n\t"
        "extr	x6, x6, x5, 32\n\t"
        "adc	x8, x8, xzr\n\t"
        "extr	x5, x5, x4, 32\n\t"
        "lsl	x4, x4, 32\n\t"
        "adds	x9, x9, x4\n\t"
        "adcs	x10, x10, x5\n\t"
        "adcs	x11, x11, x6\n\t"
        "adcs	x12, x12, x7\n\t"
        "adcs	x13, x13, x16\n\t"
        "adcs	x14, x14, xzr\n\t"
        "adcs	x15, x15, xzr\n\t"
        "adc	x8, x8, xzr\n\t"
        "#   a -= (mu << 32) << 192\n\t"
        "subs	x11, x11, x4\n\t"
        "sbcs	x12, x12, x5\n\t"
        "sbcs	x13, x13, x6\n\t"
        "sbcs	x14, x14, x7\n\t"
        "sbcs	x15, x15, x16\n\t"
        "mov	x19, 0xffffffff00000001\n\t"
        "sbc	x8, x8, xzr\n\t"
        "neg	x8, x8\n\t"
        "# mask m and sub from result if overflow\n\t"
        "#  m[0] = -1 & mask = mask\n\t"
        "subs	x12, x12, x8\n\t"
        "#  m[1] = 0xffffffff & mask = mask >> 32 as mask is all 1s or 0s\n\t"
        "lsr	x17, x8, 32\n\t"
        "sbcs	x13, x13, x17\n\t"
        "and	x19, x19, x8\n\t"
        "#  m[2] =  0 & mask = 0\n\t"
        "sbcs	x14, x14, xzr\n\t"
        "stp	x12, x13, [%[r], 0]\n\t"
        "#  m[3] =  0xffffffff00000001 & mask\n\t"
        "sbc	x15, x15, x19\n\t"
        "stp	x14, x15, [%[r], 16]\n\t"
        : [a] "+r" (a), [b] "+r" (b)
        : [r] "r" (r)
        : "memory", "x4", "x5", "x6", "x7", "x16", "x17", "x19", "x20", "x21", "x22", "x23", "x24", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15"
    );
}

/* Square the Montgomery form number mod the modulus (prime). (r = a * a mod m)
 *
 * r   Result of squaring.
 * a   Number to square in Montgomery form.
 * m   Modulus (prime).
 * mp  Montgomery mulitplier.
 */
SP_NOINLINE static void sp_256_mont_sqr_4(sp_digit* r, const sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    (void)m;
    (void)mp;

    __asm__ __volatile__ (
        "ldp       x16, x17, [%[a], 0]\n\t"
        "#  A[0] * A[1]\n\t"
        "mul	x9, x16, x17\n\t"
        "ldr       x19, [%[a], 16]\n\t"
        "umulh	x10, x16, x17\n\t"
        "ldr       x20, [%[a], 24]\n\t"
        "#  A[0] * A[2]\n\t"
        "mul	x4, x16, x19\n\t"
        "umulh	x5, x16, x19\n\t"
        "adds	x10, x10, x4\n\t"
        "#  A[0] * A[3]\n\t"
        "mul	x4, x16, x20\n\t"
        "adc	x11, xzr, x5\n\t"
        "umulh	x5, x16, x20\n\t"
        "adds	x11, x11, x4\n\t"
        "#  A[1] * A[2]\n\t"
        "mul	x4, x17, x19\n\t"
        "adc	x12, xzr, x5\n\t"
        "umulh	x5, x17, x19\n\t"
        "adds	x11, x11, x4\n\t"
        "#  A[1] * A[3]\n\t"
        "mul	x4, x17, x20\n\t"
        "adcs	x12, x12, x5\n\t"
        "umulh	x5, x17, x20\n\t"
        "adc	x13, xzr, xzr\n\t"
        "adds	x12, x12, x4\n\t"
        "#  A[2] * A[3]\n\t"
        "mul	x4, x19, x20\n\t"
        "adc	x13, x13, x5\n\t"
        "umulh	x5, x19, x20\n\t"
        "adds	x13, x13, x4\n\t"
        "adc	x14, xzr, x5\n\t"
        "# Double\n\t"
        "adds	x9, x9, x9\n\t"
        "adcs	x10, x10, x10\n\t"
        "adcs	x11, x11, x11\n\t"
        "adcs	x12, x12, x12\n\t"
        "adcs	x13, x13, x13\n\t"
        "#  A[0] * A[0]\n\t"
        "mul	x8, x16, x16\n\t"
        "adcs	x14, x14, x14\n\t"
        "umulh	x3, x16, x16\n\t"
        "cset	x15, cs\n\t"
        "#  A[1] * A[1]\n\t"
        "mul	x4, x17, x17\n\t"
        "adds	x9, x9, x3\n\t"
        "umulh	x5, x17, x17\n\t"
        "adcs	x10, x10, x4\n\t"
        "#  A[2] * A[2]\n\t"
        "mul	x6, x19, x19\n\t"
        "adcs	x11, x11, x5\n\t"
        "umulh	x7, x19, x19\n\t"
        "adcs	x12, x12, x6\n\t"
        "#  A[3] * A[3]\n\t"
        "mul	x16, x20, x20\n\t"
        "adcs	x13, x13, x7\n\t"
        "umulh	x17, x20, x20\n\t"
        "adcs	x14, x14, x16\n\t"
        "mov	x3, x8\n\t"
        "adc	x15, x15, x17\n\t"
        "# Start Reduction\n\t"
        "mov	x4, x9\n\t"
        "mov	x5, x10\n\t"
        "# mu = a[0]-a[3] + a[0]-a[2] << 32 << 64 + (a[0] * 2) << 192\n\t"
        "#    - a[0] << 32 << 192\n\t"
        "#   + (a[0] * 2) << 192\n\t"
        "#   a[0]-a[2] << 32\n\t"
        "extr	x10, x10, x9, 32\n\t"
        "add	x6, x11, x8\n\t"
        "extr	x9, x9, x8, 32\n\t"
        "add	x6, x6, x8\n\t"
        "#   + a[0]-a[2] << 32 << 64\n\t"
        "#   - a[0] << 32 << 192\n\t"
        "adds	x4, x4, x8, lsl #32\n\t"
        "sub	x6, x6, x8, lsl #32\n\t"
        "adcs	x5, x5, x9\n\t"
        "adc	x6, x6, x10\n\t"
        "# a += (mu << 256) - (mu << 224) + (mu << 192) + (mu << 96) - mu\n\t"
        "#   a += mu << 256\n\t"
        "adds	x12, x12, x3\n\t"
        "adcs	x13, x13, x4\n\t"
        "adcs	x14, x14, x5\n\t"
        "adcs	x15, x15, x6\n\t"
        "cset	x8, cs\n\t"
        "#   a += mu << 192\n\t"
        "# mu <<= 32\n\t"
        "#   a += (mu << 32) << 64\n\t"
        "adds	x11, x11, x3\n\t"
        "adcs	x12, x12, x4\n\t"
        "adcs	x13, x13, x5\n\t"
        "lsr	x7, x6, 32\n\t"
        "adcs	x14, x14, x6\n\t"
        "extr	x6, x6, x5, 32\n\t"
        "adcs	x15, x15, xzr\n\t"
        "extr	x5, x5, x4, 32\n\t"
        "adc	x8, x8, xzr\n\t"
        "extr	x4, x4, x3, 32\n\t"
        "lsl	x3, x3, 32\n\t"
        "adds	x9, x9, x3\n\t"
        "adcs	x10, x10, x4\n\t"
        "adcs	x11, x11, x5\n\t"
        "adcs	x12, x12, x6\n\t"
        "adcs	x13, x13, x7\n\t"
        "adcs	x14, x14, xzr\n\t"
        "adcs	x15, x15, xzr\n\t"
        "adc	x8, x8, xzr\n\t"
        "#   a -= (mu << 32) << 192\n\t"
        "subs	x11, x11, x3\n\t"
        "sbcs	x12, x12, x4\n\t"
        "sbcs	x13, x13, x5\n\t"
        "sbcs	x14, x14, x6\n\t"
        "sbcs	x15, x15, x7\n\t"
        "mov	x17, 0xffffffff00000001\n\t"
        "sbc	x8, x8, xzr\n\t"
        "neg	x8, x8\n\t"
        "# mask m and sub from result if overflow\n\t"
        "#  m[0] = -1 & mask = mask\n\t"
        "subs	x12, x12, x8\n\t"
        "#  m[1] = 0xffffffff & mask = mask >> 32 as mask is all 1s or 0s\n\t"
        "lsr	x16, x8, 32\n\t"
        "sbcs	x13, x13, x16\n\t"
        "and	x17, x17, x8\n\t"
        "#  m[2] =  0 & mask = 0\n\t"
        "sbcs	x14, x14, xzr\n\t"
        "stp	x12, x13, [%[r], 0]\n\t"
        "#  m[3] =  0xffffffff00000001 & mask\n\t"
        "sbc	x15, x15, x17\n\t"
        "stp	x14, x15, [%[r], 16]\n\t"
        :
        : [r] "r" (r), [a] "r" (a)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x19", "x20"
    );
}

#if !defined(WOLFSSL_SP_SMALL) || defined(HAVE_COMP_KEY)
/* Square the Montgomery form number a number of times. (r = a ^ n mod m)
 *
 * r   Result of squaring.
 * a   Number to square in Montgomery form.
 * n   Number of times to square.
 * m   Modulus (prime).
 * mp  Montgomery mulitplier.
 */
static void sp_256_mont_sqr_n_4(sp_digit* r, const sp_digit* a, int n,
        const sp_digit* m, sp_digit mp)
{
    sp_256_mont_sqr_4(r, a, m, mp);
    for (; n > 1; n--) {
        sp_256_mont_sqr_4(r, r, m, mp);
    }
}

#endif /* !WOLFSSL_SP_SMALL || HAVE_COMP_KEY */
#ifdef WOLFSSL_SP_SMALL
/* Mod-2 for the P256 curve. */
static const uint64_t p256_mod_minus_2[4] = {
    0xfffffffffffffffdU,0x00000000ffffffffU,0x0000000000000000U,
    0xffffffff00000001U
};
#endif /* !WOLFSSL_SP_SMALL */

/* Invert the number, in Montgomery form, modulo the modulus (prime) of the
 * P256 curve. (r = 1 / a mod m)
 *
 * r   Inverse result.
 * a   Number to invert.
 * td  Temporary data.
 */
static void sp_256_mont_inv_4(sp_digit* r, const sp_digit* a, sp_digit* td)
{
#ifdef WOLFSSL_SP_SMALL
    sp_digit* t = td;
    int i;

    XMEMCPY(t, a, sizeof(sp_digit) * 4);
    for (i=254; i>=0; i--) {
        sp_256_mont_sqr_4(t, t, p256_mod, p256_mp_mod);
        if (p256_mod_minus_2[i / 64] & ((sp_digit)1 << (i % 64)))
            sp_256_mont_mul_4(t, t, a, p256_mod, p256_mp_mod);
    }
    XMEMCPY(r, t, sizeof(sp_digit) * 4);
#else
    sp_digit* t1 = td;
    sp_digit* t2 = td + 2 * 4;
    sp_digit* t3 = td + 4 * 4;
    /* 0x2 */
    sp_256_mont_sqr_4(t1, a, p256_mod, p256_mp_mod);
    /* 0x3 */
    sp_256_mont_mul_4(t2, t1, a, p256_mod, p256_mp_mod);
    /* 0xc */
    sp_256_mont_sqr_n_4(t1, t2, 2, p256_mod, p256_mp_mod);
    /* 0xd */
    sp_256_mont_mul_4(t3, t1, a, p256_mod, p256_mp_mod);
    /* 0xf */
    sp_256_mont_mul_4(t2, t2, t1, p256_mod, p256_mp_mod);
    /* 0xf0 */
    sp_256_mont_sqr_n_4(t1, t2, 4, p256_mod, p256_mp_mod);
    /* 0xfd */
    sp_256_mont_mul_4(t3, t3, t1, p256_mod, p256_mp_mod);
    /* 0xff */
    sp_256_mont_mul_4(t2, t2, t1, p256_mod, p256_mp_mod);
    /* 0xff00 */
    sp_256_mont_sqr_n_4(t1, t2, 8, p256_mod, p256_mp_mod);
    /* 0xfffd */
    sp_256_mont_mul_4(t3, t3, t1, p256_mod, p256_mp_mod);
    /* 0xffff */
    sp_256_mont_mul_4(t2, t2, t1, p256_mod, p256_mp_mod);
    /* 0xffff0000 */
    sp_256_mont_sqr_n_4(t1, t2, 16, p256_mod, p256_mp_mod);
    /* 0xfffffffd */
    sp_256_mont_mul_4(t3, t3, t1, p256_mod, p256_mp_mod);
    /* 0xffffffff */
    sp_256_mont_mul_4(t2, t2, t1, p256_mod, p256_mp_mod);
    /* 0xffffffff00000000 */
    sp_256_mont_sqr_n_4(t1, t2, 32, p256_mod, p256_mp_mod);
    /* 0xffffffffffffffff */
    sp_256_mont_mul_4(t2, t2, t1, p256_mod, p256_mp_mod);
    /* 0xffffffff00000001 */
    sp_256_mont_mul_4(r, t1, a, p256_mod, p256_mp_mod);
    /* 0xffffffff000000010000000000000000000000000000000000000000 */
    sp_256_mont_sqr_n_4(r, r, 160, p256_mod, p256_mp_mod);
    /* 0xffffffff00000001000000000000000000000000ffffffffffffffff */
    sp_256_mont_mul_4(r, r, t2, p256_mod, p256_mp_mod);
    /* 0xffffffff00000001000000000000000000000000ffffffffffffffff00000000 */
    sp_256_mont_sqr_n_4(r, r, 32, p256_mod, p256_mp_mod);
    /* 0xffffffff00000001000000000000000000000000fffffffffffffffffffffffd */
    sp_256_mont_mul_4(r, r, t3, p256_mod, p256_mp_mod);
#endif /* WOLFSSL_SP_SMALL */
}

/* Compare a with b in constant time.
 *
 * a  A single precision integer.
 * b  A single precision integer.
 * return -ve, 0 or +ve if a is less than, equal to or greater than b
 * respectively.
 */
static sp_int64 sp_256_cmp_4(const sp_digit* a, const sp_digit* b)
{
#ifdef WOLFSSL_SP_SMALL
    __asm__ __volatile__ (
        "mov	x3, #0\n\t"
        "mov	x2, #-1\n\t"
        "mov	x10, #4\n\t"
        "add	%[a], %[a], #16\n\t"
        "add	%[b], %[b], #16\n\t"
        "1:\n\t"
        "ldp	x6, x7, [%[a]], -16\n\t"
        "ldp	x8, x9, [%[b]], -16\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x10, x10, #2\n\t"
        "b.ne	1b\n\t"
        "cmp	x2, #0\n\t"
        "cset	%[a], eq\n\t"
        "orr	%[a], %[a], x3\n\t"
        : [a] "+r" (a), [b] "+r" (b)
        :
        : "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );
#else
    __asm__ __volatile__ (
        "mov	x3, #0\n\t"
        "mov	x2, #-1\n\t"
        "ldp	x6, x7, [%[a], 16]\n\t"
        "ldp	x8, x9, [%[b], 16]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "ldp	x6, x7, [%[a], 0]\n\t"
        "ldp	x8, x9, [%[b], 0]\n\t"
        "subs	x7, x7, x9\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "subs	x6, x6, x8\n\t"
        "csel	x4, x2, xzr, lo\n\t"
        "csetm	x5, eq\n\t"
        "orr	x3, x3, x4\n\t"
        "and	x2, x2, x5\n\t"
        "cmp	x2, #0\n\t"
        "cset	%[a], eq\n\t"
        "orr	%[a], %[a], x3\n\t"
        : [a] "+r" (a)
        : [b] "r" (b)
        : "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );
#endif

    return (sp_int64)a;
}

/* Normalize the values in each word to 64.
 *
 * a  Array of sp_digit to normalize.
 */
#define sp_256_norm_4(a)

/* Conditionally subtract b from a using the mask m.
 * m is -1 to subtract and 0 when not copying.
 *
 * r  A single precision number representing condition subtract result.
 * a  A single precision number to subtract from.
 * b  A single precision number to subtract.
 * m  Mask value to apply.
 */
static sp_digit sp_256_cond_sub_4(sp_digit* r, const sp_digit* a, const sp_digit* b,
        sp_digit m)
{
    __asm__ __volatile__ (

        "ldp	x5, x7, [%[b], 0]\n\t"
        "ldp	x11, x12, [%[b], 16]\n\t"
        "ldp	x4, x6, [%[a], 0]\n\t"
        "and	x5, x5, %[m]\n\t"
        "ldp	x9, x10, [%[a], 16]\n\t"
        "and	x7, x7, %[m]\n\t"
        "subs	x4, x4, x5\n\t"
        "and	x11, x11, %[m]\n\t"
        "sbcs	x6, x6, x7\n\t"
        "and	x12, x12, %[m]\n\t"
        "sbcs	x9, x9, x11\n\t"
        "stp	x4, x6, [%[r], 0]\n\t"
        "sbcs	x10, x10, x12\n\t"
        "stp	x9, x10, [%[r], 16]\n\t"
        "csetm	%[r], cc\n\t"
        : [r] "+r" (r)
        : [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x6", "x5", "x7", "x8", "x9", "x10", "x11", "x12"
    );

    return (sp_digit)r;
}

/* Reduce the number back to 256 bits using Montgomery reduction.
 *
 * a   A single precision number to reduce in place.
 * m   The single precision number representing the modulus.
 * mp  The digit representing the negative inverse of m mod 2^n.
 */
SP_NOINLINE static void sp_256_mont_reduce_4(sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    __asm__ __volatile__ (
        "ldp	x10, x11, [%[a], 0]\n\t"
        "ldp	x12, x13, [%[a], 16]\n\t"
        "ldp	x14, x15, [%[a], 32]\n\t"
        "ldp	x16, x17, [%[a], 48]\n\t"
        "mov	x3, x10\n\t"
        "# Start Reduction\n\t"
        "mov	x4, x11\n\t"
        "mov	x5, x12\n\t"
        "# mu = a[0]-a[3] + a[0]-a[2] << 32 << 64 + (a[0] * 2) << 192\n\t"
        "#    - a[0] << 32 << 192\n\t"
        "#   + (a[0] * 2) << 192\n\t"
        "#   a[0]-a[2] << 32\n\t"
        "extr	x12, x12, x11, 32\n\t"
        "add	x6, x13, x10\n\t"
        "extr	x11, x11, x10, 32\n\t"
        "add	x6, x6, x10\n\t"
        "#   + a[0]-a[2] << 32 << 64\n\t"
        "#   - a[0] << 32 << 192\n\t"
        "adds	x4, x4, x10, lsl #32\n\t"
        "sub	x6, x6, x10, lsl #32\n\t"
        "adcs	x5, x5, x11\n\t"
        "adc	x6, x6, x12\n\t"
        "# a += (mu << 256) - (mu << 224) + (mu << 192) + (mu << 96) - mu\n\t"
        "#   a += mu << 256\n\t"
        "adds	x14, x14, x3\n\t"
        "adcs	x15, x15, x4\n\t"
        "adcs	x16, x16, x5\n\t"
        "adcs	x17, x17, x6\n\t"
        "cset	x10, cs\n\t"
        "#   a += mu << 192\n\t"
        "# mu <<= 32\n\t"
        "#   a += (mu << 32) << 64\n\t"
        "adds	x13, x13, x3\n\t"
        "adcs	x14, x14, x4\n\t"
        "adcs	x15, x15, x5\n\t"
        "lsr	x7, x6, 32\n\t"
        "adcs	x16, x16, x6\n\t"
        "extr	x6, x6, x5, 32\n\t"
        "adcs	x17, x17, xzr\n\t"
        "extr	x5, x5, x4, 32\n\t"
        "adc	x10, x10, xzr\n\t"
        "extr	x4, x4, x3, 32\n\t"
        "lsl	x3, x3, 32\n\t"
        "adds	x11, x11, x3\n\t"
        "adcs	x12, x12, x4\n\t"
        "adcs	x13, x13, x5\n\t"
        "adcs	x14, x14, x6\n\t"
        "adcs	x15, x15, x7\n\t"
        "adcs	x16, x16, xzr\n\t"
        "adcs	x17, x17, xzr\n\t"
        "adc	x10, x10, xzr\n\t"
        "#   a -= (mu << 32) << 192\n\t"
        "subs	x13, x13, x3\n\t"
        "sbcs	x14, x14, x4\n\t"
        "sbcs	x15, x15, x5\n\t"
        "sbcs	x16, x16, x6\n\t"
        "sbcs	x17, x17, x7\n\t"
        "mov	x9, 0xffffffff00000001\n\t"
        "sbc	x10, x10, xzr\n\t"
        "neg	x10, x10\n\t"
        "# mask m and sub from result if overflow\n\t"
        "#  m[0] = -1 & mask = mask\n\t"
        "subs	x14, x14, x10\n\t"
        "#  m[1] = 0xffffffff & mask = mask >> 32 as mask is all 1s or 0s\n\t"
        "lsr	x8, x10, 32\n\t"
        "sbcs	x15, x15, x8\n\t"
        "and	x9, x9, x10\n\t"
        "#  m[2] =  0 & mask = 0\n\t"
        "sbcs	x16, x16, xzr\n\t"
        "stp	x14, x15, [%[a], 0]\n\t"
        "#  m[3] =  0xffffffff00000001 & mask\n\t"
        "sbc	x17, x17, x9\n\t"
        "stp	x16, x17, [%[a], 16]\n\t"
        :
        : [a] "r" (a), [m] "r" (m), [mp] "r" (mp)
        : "memory", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );
}
/* Reduce the number back to 256 bits using Montgomery reduction.
 *
 * a   A single precision number to reduce in place.
 * m   The single precision number representing the modulus.
 * mp  The digit representing the negative inverse of m mod 2^n.
 */
SP_NOINLINE static void sp_256_mont_reduce_order_4(sp_digit* a, const sp_digit* m,
        sp_digit mp)
{
    __asm__ __volatile__ (
        "ldp	x9, x10, [%[a], 0]\n\t"
        "ldp	x11, x12, [%[a], 16]\n\t"
        "ldp	x17, x19, [%[m], 0]\n\t"
        "ldp	x20, x21, [%[m], 16]\n\t"
        "mov	x8, xzr\n\t"
        "# mu = a[0] * mp\n\t"
        "mul	x5, %[mp], x9\n\t"
        "ldr	x13, [%[a], 32]\n\t"
        "# a[0+0] += m[0] * mu\n\t"
        "mul	x3, x17, x5\n\t"
        "ldr	x14, [%[a], 40]\n\t"
        "umulh	x6, x17, x5\n\t"
        "ldr	x15, [%[a], 48]\n\t"
        "adds	x9, x9, x3\n\t"
        "ldr	x16, [%[a], 56]\n\t"
        "adc	x6, x6, xzr\n\t"
        "# a[0+1] += m[1] * mu\n\t"
        "mul	x3, x19, x5\n\t"
        "umulh	x7, x19, x5\n\t"
        "adds	x3, x3, x6\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x10, x10, x3\n\t"
        "adc	x7, x7, xzr\n\t"
        "# a[0+2] += m[2] * mu\n\t"
        "mul	x3, x20, x5\n\t"
        "umulh	x6, x20, x5\n\t"
        "adds	x3, x3, x7\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x11, x11, x3\n\t"
        "adc	x6, x6, xzr\n\t"
        "# a[0+3] += m[3] * mu\n\t"
        "mul	x3, x21, x5\n\t"
        "umulh	x4, x21, x5\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x8\n\t"
        "cset	x8, cs\n\t"
        "adds	x12, x12, x3\n\t"
        "adcs	x13, x13, x4\n\t"
        "adc	x8, x8, xzr\n\t"
        "# mu = a[1] * mp\n\t"
        "mul	x5, %[mp], x10\n\t"
        "# a[1+0] += m[0] * mu\n\t"
        "mul	x3, x17, x5\n\t"
        "umulh	x6, x17, x5\n\t"
        "adds	x10, x10, x3\n\t"
        "adc	x6, x6, xzr\n\t"
        "# a[1+1] += m[1] * mu\n\t"
        "mul	x3, x19, x5\n\t"
        "umulh	x7, x19, x5\n\t"
        "adds	x3, x3, x6\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x11, x11, x3\n\t"
        "adc	x7, x7, xzr\n\t"
        "# a[1+2] += m[2] * mu\n\t"
        "mul	x3, x20, x5\n\t"
        "umulh	x6, x20, x5\n\t"
        "adds	x3, x3, x7\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x12, x12, x3\n\t"
        "adc	x6, x6, xzr\n\t"
        "# a[1+3] += m[3] * mu\n\t"
        "mul	x3, x21, x5\n\t"
        "umulh	x4, x21, x5\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x8\n\t"
        "cset	x8, cs\n\t"
        "adds	x13, x13, x3\n\t"
        "adcs	x14, x14, x4\n\t"
        "adc	x8, x8, xzr\n\t"
        "# mu = a[2] * mp\n\t"
        "mul	x5, %[mp], x11\n\t"
        "# a[2+0] += m[0] * mu\n\t"
        "mul	x3, x17, x5\n\t"
        "umulh	x6, x17, x5\n\t"
        "adds	x11, x11, x3\n\t"
        "adc	x6, x6, xzr\n\t"
        "# a[2+1] += m[1] * mu\n\t"
        "mul	x3, x19, x5\n\t"
        "umulh	x7, x19, x5\n\t"
        "adds	x3, x3, x6\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x12, x12, x3\n\t"
        "adc	x7, x7, xzr\n\t"
        "# a[2+2] += m[2] * mu\n\t"
        "mul	x3, x20, x5\n\t"
        "umulh	x6, x20, x5\n\t"
        "adds	x3, x3, x7\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x13, x13, x3\n\t"
        "adc	x6, x6, xzr\n\t"
        "# a[2+3] += m[3] * mu\n\t"
        "mul	x3, x21, x5\n\t"
        "umulh	x4, x21, x5\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x8\n\t"
        "cset	x8, cs\n\t"
        "adds	x14, x14, x3\n\t"
        "adcs	x15, x15, x4\n\t"
        "adc	x8, x8, xzr\n\t"
        "# mu = a[3] * mp\n\t"
        "mul	x5, %[mp], x12\n\t"
        "# a[3+0] += m[0] * mu\n\t"
        "mul	x3, x17, x5\n\t"
        "umulh	x6, x17, x5\n\t"
        "adds	x12, x12, x3\n\t"
        "adc	x6, x6, xzr\n\t"
        "# a[3+1] += m[1] * mu\n\t"
        "mul	x3, x19, x5\n\t"
        "umulh	x7, x19, x5\n\t"
        "adds	x3, x3, x6\n\t"
        "adc	x7, x7, xzr\n\t"
        "adds	x13, x13, x3\n\t"
        "adc	x7, x7, xzr\n\t"
        "# a[3+2] += m[2] * mu\n\t"
        "mul	x3, x20, x5\n\t"
        "umulh	x6, x20, x5\n\t"
        "adds	x3, x3, x7\n\t"
        "adc	x6, x6, xzr\n\t"
        "adds	x14, x14, x3\n\t"
        "adc	x6, x6, xzr\n\t"
        "# a[3+3] += m[3] * mu\n\t"
        "mul	x3, x21, x5\n\t"
        "umulh	x4, x21, x5\n\t"
        "adds	x3, x3, x6\n\t"
        "adcs	x4, x4, x8\n\t"
        "cset	x8, cs\n\t"
        "adds	x15, x15, x3\n\t"
        "adcs	x16, x16, x4\n\t"
        "adc	x8, x8, xzr\n\t"
        "sub	x3, xzr, x8\n\t"
        "and	x17, x17, x3\n\t"
        "and	x19, x19, x3\n\t"
        "and	x20, x20, x3\n\t"
        "and	x21, x21, x3\n\t"
        "subs	x13, x13, x17\n\t"
        "sbcs	x14, x14, x19\n\t"
        "sbcs	x15, x15, x20\n\t"
        "stp	x13, x14, [%[a], 0]\n\t"
        "sbc	x16, x16, x21\n\t"
        "stp	x15, x16, [%[a], 16]\n\t"
        :
        : [a] "r" (a), [m] "r" (m), [mp] "r" (mp)
        : "memory", "x3", "x4", "x5", "x8", "x6", "x7", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x19", "x20", "x21"
    );
}

/* Map the Montgomery form projective coordinate point to an affine point.
 *
 * r  Resulting affine coordinate point.
 * p  Montgomery form projective coordinate point.
 * t  Temporary ordinate data.
 */
static void sp_256_map_4(sp_point_256* r, const sp_point_256* p,
    sp_digit* t)
{
    sp_digit* t1 = t;
    sp_digit* t2 = t + 2*4;
    sp_int64 n;

    sp_256_mont_inv_4(t1, p->z, t + 2*4);

    sp_256_mont_sqr_4(t2, t1, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(t1, t2, t1, p256_mod, p256_mp_mod);

    /* x /= z^2 */
    sp_256_mont_mul_4(r->x, p->x, t2, p256_mod, p256_mp_mod);
    XMEMSET(r->x + 4, 0, sizeof(r->x) / 2U);
    sp_256_mont_reduce_4(r->x, p256_mod, p256_mp_mod);
    /* Reduce x to less than modulus */
    n = sp_256_cmp_4(r->x, p256_mod);
    sp_256_cond_sub_4(r->x, r->x, p256_mod, ~(n >> 63));
    sp_256_norm_4(r->x);

    /* y /= z^3 */
    sp_256_mont_mul_4(r->y, p->y, t1, p256_mod, p256_mp_mod);
    XMEMSET(r->y + 4, 0, sizeof(r->y) / 2U);
    sp_256_mont_reduce_4(r->y, p256_mod, p256_mp_mod);
    /* Reduce y to less than modulus */
    n = sp_256_cmp_4(r->y, p256_mod);
    sp_256_cond_sub_4(r->y, r->y, p256_mod, ~(n >> 63));
    sp_256_norm_4(r->y);

    XMEMSET(r->z, 0, sizeof(r->z) / 2);
    r->z[0] = 1;

}

/* Add two Montgomery form numbers (r = a + b % m).
 *
 * r   Result of addition.
 * a   First number to add in Montgomery form.
 * b   Second number to add in Montgomery form.
 * m   Modulus (prime).
 */
static void sp_256_mont_add_4(sp_digit* r, const sp_digit* a, const sp_digit* b,
        const sp_digit* m)
{
    __asm__ __volatile__ (
        "ldp	x4, x5, [%[a], 0]\n\t"
        "ldp	x8, x9, [%[b], 0]\n\t"
        "adds	x4, x4, x8\n\t"
        "ldp	x6, x7, [%[a], 16]\n\t"
        "adcs	x5, x5, x9\n\t"
        "ldp	x10, x11, [%[b], 16]\n\t"
        "adcs	x6, x6, x10\n\t"
        "adcs	x7, x7, x11\n\t"
        "mov	x13, 0xffffffff00000001\n\t"
        "csetm	x14, cs\n\t"
        "subs	x4, x4, x14\n\t"
        "lsr	x12, x14, 32\n\t"
        "sbcs	x5, x5, x12\n\t"
        "and	x13, x13, x14\n\t"
        "sbcs	x6, x6, xzr\n\t"
        "stp	x4, x5, [%[r],0]\n\t"
        "sbc	x7, x7, x13\n\t"
        "stp	x6, x7, [%[r],16]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14"
    );
}

/* Double a Montgomery form number (r = a + a % m).
 *
 * r   Result of doubling.
 * a   Number to double in Montgomery form.
 * m   Modulus (prime).
 */
static void sp_256_mont_dbl_4(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a]]\n\t"
        "ldp	x5, x6, [%[a],16]\n\t"
        "adds	x3, x3, x3\n\t"
        "adcs	x4, x4, x4\n\t"
        "adcs	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "mov	x8, 0xffffffff00000001\n\t"
        "csetm	x9, cs\n\t"
        "subs	x3, x3, x9\n\t"
        "lsr	x7, x9, 32\n\t"
        "sbcs	x4, x4, x7\n\t"
        "and	x8, x8, x9\n\t"
        "sbcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r],0]\n\t"
        "sbc	x6, x6, x8\n\t"
        "stp	x5, x6, [%[r],16]\n\t"
        :
        : [r] "r" (r), [a] "r" (a)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );

    (void)m;
}

/* Triple a Montgomery form number (r = a + a + a % m).
 *
 * r   Result of Tripling.
 * a   Number to triple in Montgomery form.
 * m   Modulus (prime).
 */
static void sp_256_mont_tpl_4(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    __asm__ __volatile__ (
        "ldp	x10, x11, [%[a]]\n\t"
        "adds	x3, x10, x10\n\t"
        "ldr	x12, [%[a], 16]\n\t"
        "adcs	x4, x11, x11\n\t"
        "ldr	x13, [%[a], 24]\n\t"
        "adcs	x5, x12, x12\n\t"
        "adcs	x6, x13, x13\n\t"
        "mov	x8, 0xffffffff00000001\n\t"
        "csetm	x9, cs\n\t"
        "subs	x3, x3, x9\n\t"
        "lsr	x7, x9, 32\n\t"
        "sbcs	x4, x4, x7\n\t"
        "and	x8, x8, x9\n\t"
        "sbcs	x5, x5, xzr\n\t"
        "sbc	x6, x6, x8\n\t"
        "adds	x3, x3, x10\n\t"
        "adcs	x4, x4, x11\n\t"
        "adcs	x5, x5, x12\n\t"
        "adcs	x6, x6, x13\n\t"
        "mov	x8, 0xffffffff00000001\n\t"
        "csetm	x9, cs\n\t"
        "subs	x3, x3, x9\n\t"
        "lsr	x7, x9, 32\n\t"
        "sbcs	x4, x4, x7\n\t"
        "and	x8, x8, x9\n\t"
        "sbcs	x5, x5, xzr\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "sbc	x6, x6, x8\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        :
        : [r] "r" (r), [a] "r" (a)
        : "memory", "x10", "x11", "x12", "x13", "x3", "x4", "x5", "x6", "x7", "x8", "x9"
    );

    (void)m;
}

/* Subtract two Montgomery form numbers (r = a - b % m).
 *
 * r   Result of subtration.
 * a   Number to subtract from in Montgomery form.
 * b   Number to subtract with in Montgomery form.
 * m   Modulus (prime).
 */
static void sp_256_mont_sub_4(sp_digit* r, const sp_digit* a, const sp_digit* b,
        const sp_digit* m)
{
    __asm__ __volatile__ (
        "ldp	x4, x5, [%[a], 0]\n\t"
        "ldp	x8, x9, [%[b], 0]\n\t"
        "subs	x4, x4, x8\n\t"
        "ldp	x6, x7, [%[a], 16]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "ldp	x10, x11, [%[b], 16]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "sbcs	x7, x7, x11\n\t"
        "mov	x13, 0xffffffff00000001\n\t"
        "csetm	x14, cc\n\t"
        "adds	x4, x4, x14\n\t"
        "lsr	x12, x14, 32\n\t"
        "adcs	x5, x5, x12\n\t"
        "and	x13, x13, x14\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x4, x5, [%[r],0]\n\t"
        "adc	x7, x7, x13\n\t"
        "stp	x6, x7, [%[r],16]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14"
    );
}

#define sp_256_mont_sub_lower_4 sp_256_mont_sub_4
/* Divide the number by 2 mod the modulus (prime). (r = a / 2 % m)
 *
 * r  Result of division by 2.
 * a  Number to divide.
 * m  Modulus (prime).
 */
static void sp_256_div2_4(sp_digit* r, const sp_digit* a, const sp_digit* m)
{
    __asm__ __volatile__ (
        "ldp	x3, x4, [%[a], 0]\n\t"
        "and	x9, x3, 1\n\t"
        "ldp	x5, x6, [%[a], 16]\n\t"
        "sub	x10, xzr, x9\n\t"
        "lsr	x7, x10, 32\n\t"
        "adds	x3, x3, x10\n\t"
        "and	x8, x10, 0xffffffff00000001\n\t"
        "adcs	x4, x4, x7\n\t"
        "adcs	x5, x5, xzr\n\t"
        "extr	x3, x4, x3, 1\n\t"
        "adcs	x6, x6, x8\n\t"
        "extr	x4, x5, x4, 1\n\t"
        "cset	x9, cs\n\t"
        "extr	x5, x6, x5, 1\n\t"
        "extr	x6, x9, x6, 1\n\t"
        "stp	x3, x4, [%[r], 0]\n\t"
        "stp	x5, x6, [%[r], 16]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [m] "r" (m)
        : "memory", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10"
    );

}

/* Double the Montgomery form projective point p.
 *
 * r  Result of doubling point.
 * p  Point to double.
 * t  Temporary ordinate data.
 */
#ifdef WOLFSSL_SP_NONBLOCK
typedef struct sp_256_proj_point_dbl_4_ctx {
    int state;
    sp_digit* t1;
    sp_digit* t2;
    sp_digit* x;
    sp_digit* y;
    sp_digit* z;
} sp_256_proj_point_dbl_4_ctx;

static int sp_256_proj_point_dbl_4_nb(sp_ecc_ctx_t* sp_ctx, sp_point_256* r, const sp_point_256* p, sp_digit* t)
{
    int err = FP_WOULDBLOCK;
    sp_256_proj_point_dbl_4_ctx* ctx = (sp_256_proj_point_dbl_4_ctx*)sp_ctx->data;

    typedef char ctx_size_test[sizeof(sp_256_proj_point_dbl_4_ctx) >= sizeof(*sp_ctx) ? -1 : 1];
    (void)sizeof(ctx_size_test);

    switch (ctx->state) {
    case 0:
        ctx->t1 = t;
        ctx->t2 = t + 2*4;
        ctx->x = r->x;
        ctx->y = r->y;
        ctx->z = r->z;

        /* Put infinity into result. */
        if (r != p) {
            r->infinity = p->infinity;
        }
        ctx->state = 1;
        break;
    case 1:
        /* T1 = Z * Z */
        sp_256_mont_sqr_4(ctx->t1, p->z, p256_mod, p256_mp_mod);
        ctx->state = 2;
        break;
    case 2:
        /* Z = Y * Z */
        sp_256_mont_mul_4(ctx->z, p->y, p->z, p256_mod, p256_mp_mod);
        ctx->state = 3;
        break;
    case 3:
        /* Z = 2Z */
        sp_256_mont_dbl_4(ctx->z, ctx->z, p256_mod);
        ctx->state = 4;
        break;
    case 4:
        /* T2 = X - T1 */
        sp_256_mont_sub_4(ctx->t2, p->x, ctx->t1, p256_mod);
        ctx->state = 5;
        break;
    case 5:
        /* T1 = X + T1 */
        sp_256_mont_add_4(ctx->t1, p->x, ctx->t1, p256_mod);
        ctx->state = 6;
        break;
    case 6:
        /* T2 = T1 * T2 */
        sp_256_mont_mul_4(ctx->t2, ctx->t1, ctx->t2, p256_mod, p256_mp_mod);
        ctx->state = 7;
        break;
    case 7:
        /* T1 = 3T2 */
        sp_256_mont_tpl_4(ctx->t1, ctx->t2, p256_mod);
        ctx->state = 8;
        break;
    case 8:
        /* Y = 2Y */
        sp_256_mont_dbl_4(ctx->y, p->y, p256_mod);
        ctx->state = 9;
        break;
    case 9:
        /* Y = Y * Y */
        sp_256_mont_sqr_4(ctx->y, ctx->y, p256_mod, p256_mp_mod);
        ctx->state = 10;
        break;
    case 10:
        /* T2 = Y * Y */
        sp_256_mont_sqr_4(ctx->t2, ctx->y, p256_mod, p256_mp_mod);
        ctx->state = 11;
        break;
    case 11:
        /* T2 = T2/2 */
        sp_256_div2_4(ctx->t2, ctx->t2, p256_mod);
        ctx->state = 12;
        break;
    case 12:
        /* Y = Y * X */
        sp_256_mont_mul_4(ctx->y, ctx->y, p->x, p256_mod, p256_mp_mod);
        ctx->state = 13;
        break;
    case 13:
        /* X = T1 * T1 */
        sp_256_mont_sqr_4(ctx->x, ctx->t1, p256_mod, p256_mp_mod);
        ctx->state = 14;
        break;
    case 14:
        /* X = X - Y */
        sp_256_mont_sub_4(ctx->x, ctx->x, ctx->y, p256_mod);
        ctx->state = 15;
        break;
    case 15:
        /* X = X - Y */
        sp_256_mont_sub_4(ctx->x, ctx->x, ctx->y, p256_mod);
        ctx->state = 16;
        break;
    case 16:
        /* Y = Y - X */
        sp_256_mont_sub_lower_4(ctx->y, ctx->y, ctx->x, p256_mod);
        ctx->state = 17;
        break;
    case 17:
        /* Y = Y * T1 */
        sp_256_mont_mul_4(ctx->y, ctx->y, ctx->t1, p256_mod, p256_mp_mod);
        ctx->state = 18;
        break;
    case 18:
        /* Y = Y - T2 */
        sp_256_mont_sub_4(ctx->y, ctx->y, ctx->t2, p256_mod);
        ctx->state = 19;
        /* fall-through */
    case 19:
        err = MP_OKAY;
        break;
    }

    if (err == MP_OKAY && ctx->state != 19) {
        err = FP_WOULDBLOCK;
    }

    return err;
}
#endif /* WOLFSSL_SP_NONBLOCK */

static void sp_256_proj_point_dbl_4(sp_point_256* r, const sp_point_256* p,
    sp_digit* t)
{
    sp_digit* t1 = t;
    sp_digit* t2 = t + 2*4;
    sp_digit* x;
    sp_digit* y;
    sp_digit* z;

    x = r->x;
    y = r->y;
    z = r->z;
    /* Put infinity into result. */
    if (r != p) {
        r->infinity = p->infinity;
    }

    /* T1 = Z * Z */
    sp_256_mont_sqr_4(t1, p->z, p256_mod, p256_mp_mod);
    /* Z = Y * Z */
    sp_256_mont_mul_4(z, p->y, p->z, p256_mod, p256_mp_mod);
    /* Z = 2Z */
    sp_256_mont_dbl_4(z, z, p256_mod);
    /* T2 = X - T1 */
    sp_256_mont_sub_4(t2, p->x, t1, p256_mod);
    /* T1 = X + T1 */
    sp_256_mont_add_4(t1, p->x, t1, p256_mod);
    /* T2 = T1 * T2 */
    sp_256_mont_mul_4(t2, t1, t2, p256_mod, p256_mp_mod);
    /* T1 = 3T2 */
    sp_256_mont_tpl_4(t1, t2, p256_mod);
    /* Y = 2Y */
    sp_256_mont_dbl_4(y, p->y, p256_mod);
    /* Y = Y * Y */
    sp_256_mont_sqr_4(y, y, p256_mod, p256_mp_mod);
    /* T2 = Y * Y */
    sp_256_mont_sqr_4(t2, y, p256_mod, p256_mp_mod);
    /* T2 = T2/2 */
    sp_256_div2_4(t2, t2, p256_mod);
    /* Y = Y * X */
    sp_256_mont_mul_4(y, y, p->x, p256_mod, p256_mp_mod);
    /* X = T1 * T1 */
    sp_256_mont_sqr_4(x, t1, p256_mod, p256_mp_mod);
    /* X = X - Y */
    sp_256_mont_sub_4(x, x, y, p256_mod);
    /* X = X - Y */
    sp_256_mont_sub_4(x, x, y, p256_mod);
    /* Y = Y - X */
    sp_256_mont_sub_lower_4(y, y, x, p256_mod);
    /* Y = Y * T1 */
    sp_256_mont_mul_4(y, y, t1, p256_mod, p256_mp_mod);
    /* Y = Y - T2 */
    sp_256_mont_sub_4(y, y, t2, p256_mod);
}

#define sp_256_mont_tpl_lower_4 sp_256_mont_tpl_4
/* Subtract two Montgomery form numbers (r = a - b % m).
 *
 * r   Result of subtration.
 * a   Number to subtract from in Montgomery form.
 * b   Number to subtract with in Montgomery form.
 * m   Modulus (prime).
 */
static void sp_256_mont_sub_dbl_4(sp_digit* r, const sp_digit* a, const sp_digit* b,
        const sp_digit* m)
{
    __asm__ __volatile__ (
        "ldp	x8, x9, [%[b]]\n\t"
        "ldp	x10, x11, [%[b],16]\n\t"
        "adds	x8, x8, x8\n\t"
        "ldp	x4, x5, [%[a]]\n\t"
        "adcs	x9, x9, x9\n\t"
        "ldp	x6, x7, [%[a],16]\n\t"
        "adcs	x10, x10, x10\n\t"
        "adcs	x11, x11, x11\n\t"
        "mov	x13, 0xffffffff00000001\n\t"
        "csetm	x14, cs\n\t"
        "subs	x8, x8, x14\n\t"
        "lsr	x12, x14, 32\n\t"
        "sbcs	x9, x9, x12\n\t"
        "and	x13, x13, x14\n\t"
        "sbcs	x10, x10, xzr\n\t"
        "sbc	x11, x11, x13\n\t"
        "subs	x4, x4, x8\n\t"
        "sbcs	x5, x5, x9\n\t"
        "sbcs	x6, x6, x10\n\t"
        "sbcs	x7, x7, x11\n\t"
        "mov	x13, 0xffffffff00000001\n\t"
        "csetm	x14, cc\n\t"
        "adds	x4, x4, x14\n\t"
        "lsr	x12, x14, 32\n\t"
        "adcs	x5, x5, x12\n\t"
        "and	x13, x13, x14\n\t"
        "adcs	x6, x6, xzr\n\t"
        "stp	x4, x5, [%[r],0]\n\t"
        "adc	x7, x7, x13\n\t"
        "stp	x6, x7, [%[r],16]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14"
    );
}

/* Subtract two Montgomery form numbers (r = a - b % m).
 *
 * r   Result of subtration.
 * a   Number to subtract from in Montgomery form.
 * b   Number to subtract with in Montgomery form.
 * m   Modulus (prime).
 */
static void sp_256_mont_dbl_sub_4(sp_digit* r, const sp_digit* a, const sp_digit* b,
        const sp_digit* m)
{
    __asm__ __volatile__ (
        "ldp	x4, x5, [%[a], 0]\n\t"
        "ldp	x8, x9, [%[b], 0]\n\t"
        "subs	x4, x4, x8\n\t"
        "ldp	x6, x7, [%[a], 16]\n\t"
        "sbcs	x5, x5, x9\n\t"
        "ldp	x10, x11, [%[b], 16]\n\t"
        "sbcs	x6, x6, x10\n\t"
        "sbcs	x7, x7, x11\n\t"
        "mov	x13, 0xffffffff00000001\n\t"
        "csetm	x14, cc\n\t"
        "adds	x4, x4, x14\n\t"
        "lsr	x12, x14, 32\n\t"
        "adcs	x5, x5, x12\n\t"
        "and	x13, x13, x14\n\t"
        "adcs	x6, x6, xzr\n\t"
        "adc	x7, x7, x13\n\t"
        "adds	x4, x4, x4\n\t"
        "adcs	x5, x5, x5\n\t"
        "adcs	x6, x6, x6\n\t"
        "adcs	x7, x7, x7\n\t"
        "mov	x13, 0xffffffff00000001\n\t"
        "csetm	x14, cs\n\t"
        "subs	x4, x4, x14\n\t"
        "lsr	x12, x14, 32\n\t"
        "sbcs	x5, x5, x12\n\t"
        "and	x13, x13, x14\n\t"
        "sbcs	x6, x6, xzr\n\t"
        "stp	x4, x5, [%[r],0]\n\t"
        "sbc	x7, x7, x13\n\t"
        "stp	x6, x7, [%[r],16]\n\t"
        :
        : [r] "r" (r), [a] "r" (a), [b] "r" (b), [m] "r" (m)
        : "memory", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14"
    );
}

/* Double the Montgomery form projective point p a number of times.
 *
 * r  Result of repeated doubling of point.
 * p  Point to double.
 * n  Number of times to double
 * t  Temporary ordinate data.
 */
static void sp_256_proj_point_dbl_n_4(sp_point_256* p, int i,
    sp_digit* t)
{
    sp_digit* w = t;
    sp_digit* a = t + 2*4;
    sp_digit* b = t + 4*4;
    sp_digit* t1 = t + 6*4;
    sp_digit* x;
    sp_digit* y;
    sp_digit* z;
    volatile int n = i;

    x = p->x;
    y = p->y;
    z = p->z;

    /* Y = 2*Y */
    sp_256_mont_dbl_4(y, y, p256_mod);
    /* W = Z^4 */
    sp_256_mont_sqr_4(w, z, p256_mod, p256_mp_mod);
    sp_256_mont_sqr_4(w, w, p256_mod, p256_mp_mod);

#ifndef WOLFSSL_SP_SMALL
    while (--n > 0)
#else
    while (--n >= 0)
#endif
    {
        /* A = 3*(X^2 - W) */
        sp_256_mont_sqr_4(t1, x, p256_mod, p256_mp_mod);
        sp_256_mont_sub_4(t1, t1, w, p256_mod);
        sp_256_mont_tpl_lower_4(a, t1, p256_mod);
        /* B = X*Y^2 */
        sp_256_mont_sqr_4(t1, y, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(b, t1, x, p256_mod, p256_mp_mod);
        /* X = A^2 - 2B */
        sp_256_mont_sqr_4(x, a, p256_mod, p256_mp_mod);
        sp_256_mont_sub_dbl_4(x, x, b, p256_mod);
        /*   b = 2.(B - X) */
        sp_256_mont_dbl_sub_4(b, b, x, p256_mod);
        /* Z = Z*Y */
        sp_256_mont_mul_4(z, z, y, p256_mod, p256_mp_mod);
        /* t1 = Y^4 */
        sp_256_mont_sqr_4(t1, t1, p256_mod, p256_mp_mod);
#ifdef WOLFSSL_SP_SMALL
        if (n != 0)
#endif
        {
            /* W = W*Y^4 */
            sp_256_mont_mul_4(w, w, t1, p256_mod, p256_mp_mod);
        }
        /* y = 2*A*(B - X) - Y^4 */
        sp_256_mont_mul_4(y, b, a, p256_mod, p256_mp_mod);
        sp_256_mont_sub_4(y, y, t1, p256_mod);
    }
#ifndef WOLFSSL_SP_SMALL
    /* A = 3*(X^2 - W) */
    sp_256_mont_sqr_4(t1, x, p256_mod, p256_mp_mod);
    sp_256_mont_sub_4(t1, t1, w, p256_mod);
    sp_256_mont_tpl_lower_4(a, t1, p256_mod);
    /* B = X*Y^2 */
    sp_256_mont_sqr_4(t1, y, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(b, t1, x, p256_mod, p256_mp_mod);
    /* X = A^2 - 2B */
    sp_256_mont_sqr_4(x, a, p256_mod, p256_mp_mod);
    sp_256_mont_sub_dbl_4(x, x, b, p256_mod);
    /*   b = 2.(B - X) */
    sp_256_mont_dbl_sub_4(b, b, x, p256_mod);
    /* Z = Z*Y */
    sp_256_mont_mul_4(z, z, y, p256_mod, p256_mp_mod);
    /* t1 = Y^4 */
    sp_256_mont_sqr_4(t1, t1, p256_mod, p256_mp_mod);
    /* y = 2*A*(B - X) - Y^4 */
    sp_256_mont_mul_4(y, b, a, p256_mod, p256_mp_mod);
    sp_256_mont_sub_4(y, y, t1, p256_mod);
#endif
    /* Y = Y/2 */
    sp_256_div2_4(y, y, p256_mod);
}

/* Compare two numbers to determine if they are equal.
 * Constant time implementation.
 *
 * a  First number to compare.
 * b  Second number to compare.
 * returns 1 when equal and 0 otherwise.
 */
static int sp_256_cmp_equal_4(const sp_digit* a, const sp_digit* b)
{
    return ((a[0] ^ b[0]) | (a[1] ^ b[1]) | (a[2] ^ b[2]) |
            (a[3] ^ b[3])) == 0;
}

/* Returns 1 if the number of zero.
 * Implementation is constant time.
 *
 * a  Number to check.
 * returns 1 if the number is zero and 0 otherwise.
 */
static int sp_256_iszero_4(const sp_digit* a)
{
    return (a[0] | a[1] | a[2] | a[3]) == 0;
}

/* Add two Montgomery form projective points.
 *
 * r  Result of addition.
 * p  First point to add.
 * q  Second point to add.
 * t  Temporary ordinate data.
 */

#ifdef WOLFSSL_SP_NONBLOCK
typedef struct sp_256_proj_point_add_4_ctx {
    int state;
    sp_256_proj_point_dbl_4_ctx dbl_ctx;
    const sp_point_256* ap[2];
    sp_point_256* rp[2];
    sp_digit* t1;
    sp_digit* t2;
    sp_digit* t3;
    sp_digit* t4;
    sp_digit* t5;
    sp_digit* t6;
    sp_digit* x;
    sp_digit* y;
    sp_digit* z;
} sp_256_proj_point_add_4_ctx;

static int sp_256_proj_point_add_4_nb(sp_ecc_ctx_t* sp_ctx, sp_point_256* r,
    const sp_point_256* p, const sp_point_256* q, sp_digit* t)
{
    int err = FP_WOULDBLOCK;
    sp_256_proj_point_add_4_ctx* ctx = (sp_256_proj_point_add_4_ctx*)sp_ctx->data;

    /* Ensure only the first point is the same as the result. */
    if (q == r) {
        const sp_point_256* a = p;
        p = q;
        q = a;
    }

    typedef char ctx_size_test[sizeof(sp_256_proj_point_add_4_ctx) >= sizeof(*sp_ctx) ? -1 : 1];
    (void)sizeof(ctx_size_test);

    switch (ctx->state) {
    case 0: /* INIT */
        ctx->t1 = t;
        ctx->t2 = t + 2*4;
        ctx->t3 = t + 4*4;
        ctx->t4 = t + 6*4;
        ctx->t5 = t + 8*4;
        ctx->t6 = t + 10*4;
        ctx->x = ctx->t6;
        ctx->y = ctx->t1;
        ctx->z = ctx->t2;

        ctx->state = 1;
        break;
    case 1:
        /* Check double */
        (void)sp_256_sub_4(ctx->t1, p256_mod, q->y);
        sp_256_norm_4(ctx->t1);
        if ((~p->infinity & ~q->infinity &
            sp_256_cmp_equal_4(p->x, q->x) & sp_256_cmp_equal_4(p->z, q->z) &
            (sp_256_cmp_equal_4(p->y, q->y) | sp_256_cmp_equal_4(p->y, ctx->t1))) != 0)
        {
            XMEMSET(&ctx->dbl_ctx, 0, sizeof(ctx->dbl_ctx));
            ctx->state = 2;
        }
        else {
            ctx->state = 3;
        }
        break;
    case 2:
        err = sp_256_proj_point_dbl_4_nb((sp_ecc_ctx_t*)&ctx->dbl_ctx, r, p, t);
        if (err == MP_OKAY)
            ctx->state = 27; /* done */
        break;
    case 3:
    {
        ctx->state = 4;
        break;
    }
    case 4:
        /* U1 = X1*Z2^2 */
        sp_256_mont_sqr_4(ctx->t1, q->z, p256_mod, p256_mp_mod);
        ctx->state = 5;
        break;
    case 5:
        sp_256_mont_mul_4(ctx->t3, ctx->t1, q->z, p256_mod, p256_mp_mod);
        ctx->state = 6;
        break;
    case 6:
        sp_256_mont_mul_4(ctx->t1, ctx->t1, p->x, p256_mod, p256_mp_mod);
        ctx->state = 7;
        break;
    case 7:
        /* U2 = X2*Z1^2 */
        sp_256_mont_sqr_4(ctx->t2, p->z, p256_mod, p256_mp_mod);
        ctx->state = 8;
        break;
    case 8:
        sp_256_mont_mul_4(ctx->t4, ctx->t2, p->z, p256_mod, p256_mp_mod);
        ctx->state = 9;
        break;
    case 9:
        sp_256_mont_mul_4(ctx->t2, ctx->t2, q->x, p256_mod, p256_mp_mod);
        ctx->state = 10;
        break;
    case 10:
        /* S1 = Y1*Z2^3 */
        sp_256_mont_mul_4(ctx->t3, ctx->t3, p->y, p256_mod, p256_mp_mod);
        ctx->state = 11;
        break;
    case 11:
        /* S2 = Y2*Z1^3 */
        sp_256_mont_mul_4(ctx->t4, ctx->t4, q->y, p256_mod, p256_mp_mod);
        ctx->state = 12;
        break;
    case 12:
        /* H = U2 - U1 */
        sp_256_mont_sub_4(ctx->t2, ctx->t2, ctx->t1, p256_mod);
        ctx->state = 13;
        break;
    case 13:
        /* R = S2 - S1 */
        sp_256_mont_sub_4(ctx->t4, ctx->t4, ctx->t3, p256_mod);
        ctx->state = 14;
        break;
    case 14:
        /* X3 = R^2 - H^3 - 2*U1*H^2 */
        sp_256_mont_sqr_4(ctx->t5, ctx->t2, p256_mod, p256_mp_mod);
        ctx->state = 15;
        break;
    case 15:
        sp_256_mont_mul_4(ctx->y, ctx->t1, ctx->t5, p256_mod, p256_mp_mod);
        ctx->state = 16;
        break;
    case 16:
        sp_256_mont_mul_4(ctx->t5, ctx->t5, ctx->t2, p256_mod, p256_mp_mod);
        ctx->state = 17;
        break;
    case 17:
        /* Z3 = H*Z1*Z2 */
        sp_256_mont_mul_4(ctx->z, p->z, ctx->t2, p256_mod, p256_mp_mod);
        ctx->state = 18;
        break;
    case 18:
        sp_256_mont_mul_4(ctx->z, ctx->z, q->z, p256_mod, p256_mp_mod);
        ctx->state = 19;
        break;
    case 19:
        sp_256_mont_sqr_4(ctx->x, ctx->t4, p256_mod, p256_mp_mod);
        ctx->state = 20;
        break;
    case 20:
        sp_256_mont_sub_4(ctx->x, ctx->x, ctx->t5, p256_mod);
        ctx->state = 21;
        break;
    case 21:
        sp_256_mont_mul_4(ctx->t5, ctx->t5, ctx->t3, p256_mod, p256_mp_mod);
        ctx->state = 22;
        break;
    case 22:
        sp_256_mont_dbl_4(ctx->t3, ctx->y, p256_mod);
        ctx->state = 23;
        break;
    case 23:
        sp_256_mont_sub_4(ctx->x, ctx->x, ctx->t3, p256_mod);
        ctx->state = 24;
        break;
    case 24:
        /* Y3 = R*(U1*H^2 - X3) - S1*H^3 */
        sp_256_mont_sub_lower_4(ctx->y, ctx->y, ctx->x, p256_mod);
        ctx->state = 25;
        break;
    case 25:
        sp_256_mont_mul_4(ctx->y, ctx->y, ctx->t4, p256_mod, p256_mp_mod);
        ctx->state = 26;
        break;
    case 26:
        sp_256_mont_sub_4(ctx->y, ctx->y, ctx->t5, p256_mod);
        ctx->state = 27;
        /* fall-through */
    case 27:
    {
        int i;
        sp_digit maskp = 0 - (q->infinity & (!p->infinity));
        sp_digit maskq = 0 - (p->infinity & (!q->infinity));
        sp_digit maskt = ~(maskp | maskq);
        for (i = 0; i < 4; i++) {
            r->x[i] = (p->x[i] & maskp) | (q->x[i] & maskq) |
                      (ctx->x[i] & maskt);
        }
        for (i = 0; i < 4; i++) {
            r->y[i] = (p->y[i] & maskp) | (q->y[i] & maskq) |
                      (ctx->y[i] & maskt);
        }
        for (i = 0; i < 4; i++) {
            r->z[i] = (p->z[i] & maskp) | (q->z[i] & maskq) |
                      (ctx->z[i] & maskt);
        }
        r->z[0] |= p->infinity & q->infinity;
        r->infinity = p->infinity & q->infinity;

        err = MP_OKAY;
        break;
    }
    }

    if (err == MP_OKAY && ctx->state != 27) {
        err = FP_WOULDBLOCK;
    }
    return err;
}
#endif /* WOLFSSL_SP_NONBLOCK */

static void sp_256_proj_point_add_4(sp_point_256* r,
        const sp_point_256* p, const sp_point_256* q, sp_digit* t)
{
    sp_digit* t1 = t;
    sp_digit* t2 = t + 2*4;
    sp_digit* t3 = t + 4*4;
    sp_digit* t4 = t + 6*4;
    sp_digit* t5 = t + 8*4;
    sp_digit* t6 = t + 10*4;


    /* Check double */
    (void)sp_256_sub_4(t1, p256_mod, q->y);
    sp_256_norm_4(t1);
    if ((~p->infinity & ~q->infinity &
        sp_256_cmp_equal_4(p->x, q->x) & sp_256_cmp_equal_4(p->z, q->z) &
        (sp_256_cmp_equal_4(p->y, q->y) | sp_256_cmp_equal_4(p->y, t1))) != 0) {
        sp_256_proj_point_dbl_4(r, p, t);
    }
    else {
        sp_digit maskp;
        sp_digit maskq;
        sp_digit maskt;
        sp_digit* x = t6;
        sp_digit* y = t1;
        sp_digit* z = t2;
        int i;

        maskp = 0 - (q->infinity & (!p->infinity));
        maskq = 0 - (p->infinity & (!q->infinity));
        maskt = ~(maskp | maskq);

        /* U1 = X1*Z2^2 */
        sp_256_mont_sqr_4(t1, q->z, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t3, t1, q->z, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t1, t1, p->x, p256_mod, p256_mp_mod);
        /* U2 = X2*Z1^2 */
        sp_256_mont_sqr_4(t2, p->z, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t4, t2, p->z, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t2, t2, q->x, p256_mod, p256_mp_mod);
        /* S1 = Y1*Z2^3 */
        sp_256_mont_mul_4(t3, t3, p->y, p256_mod, p256_mp_mod);
        /* S2 = Y2*Z1^3 */
        sp_256_mont_mul_4(t4, t4, q->y, p256_mod, p256_mp_mod);
        /* H = U2 - U1 */
        sp_256_mont_sub_4(t2, t2, t1, p256_mod);
        /* R = S2 - S1 */
        sp_256_mont_sub_4(t4, t4, t3, p256_mod);
        if (~p->infinity & ~q->infinity &
            sp_256_iszero_4(t2) & sp_256_iszero_4(t4) & maskt) {
            sp_256_proj_point_dbl_4(r, p, t);
        }
        else {
            /* X3 = R^2 - H^3 - 2*U1*H^2 */
            sp_256_mont_sqr_4(t5, t2, p256_mod, p256_mp_mod);
            sp_256_mont_mul_4(y, t1, t5, p256_mod, p256_mp_mod);
            sp_256_mont_mul_4(t5, t5, t2, p256_mod, p256_mp_mod);
            /* Z3 = H*Z1*Z2 */
            sp_256_mont_mul_4(z, p->z, t2, p256_mod, p256_mp_mod);
            sp_256_mont_mul_4(z, z, q->z, p256_mod, p256_mp_mod);
            sp_256_mont_sqr_4(x, t4, p256_mod, p256_mp_mod);
            sp_256_mont_sub_4(x, x, t5, p256_mod);
            sp_256_mont_mul_4(t5, t5, t3, p256_mod, p256_mp_mod);
            sp_256_mont_sub_dbl_4(x, x, y, p256_mod);
            /* Y3 = R*(U1*H^2 - X3) - S1*H^3 */
            sp_256_mont_sub_lower_4(y, y, x, p256_mod);
            sp_256_mont_mul_4(y, y, t4, p256_mod, p256_mp_mod);
            sp_256_mont_sub_4(y, y, t5, p256_mod);

            for (i = 0; i < 4; i++) {
                r->x[i] = (p->x[i] & maskp) | (q->x[i] & maskq) |
                          (x[i] & maskt);
            }
            for (i = 0; i < 4; i++) {
                r->y[i] = (p->y[i] & maskp) | (q->y[i] & maskq) |
                          (y[i] & maskt);
            }
            for (i = 0; i < 4; i++) {
                r->z[i] = (p->z[i] & maskp) | (q->z[i] & maskq) |
                          (z[i] & maskt);
            }
            r->z[0] |= p->infinity & q->infinity;
            r->infinity = p->infinity & q->infinity;
        }
    }
}

/* Double the Montgomery form projective point p a number of times.
 *
 * r  Result of repeated doubling of point.
 * p  Point to double.
 * n  Number of times to double
 * t  Temporary ordinate data.
 */
static void sp_256_proj_point_dbl_n_store_4(sp_point_256* r,
        const sp_point_256* p, int n, int m, sp_digit* t)
{
    sp_digit* w = t;
    sp_digit* a = t + 2*4;
    sp_digit* b = t + 4*4;
    sp_digit* t1 = t + 6*4;
    sp_digit* x = r[2*m].x;
    sp_digit* y = r[(1<<n)*m].y;
    sp_digit* z = r[2*m].z;
    int i;
    int j;

    for (i=0; i<4; i++) {
        x[i] = p->x[i];
    }
    for (i=0; i<4; i++) {
        y[i] = p->y[i];
    }
    for (i=0; i<4; i++) {
        z[i] = p->z[i];
    }

    /* Y = 2*Y */
    sp_256_mont_dbl_4(y, y, p256_mod);
    /* W = Z^4 */
    sp_256_mont_sqr_4(w, z, p256_mod, p256_mp_mod);
    sp_256_mont_sqr_4(w, w, p256_mod, p256_mp_mod);
    j = m;
    for (i=1; i<=n; i++) {
        j *= 2;

        /* A = 3*(X^2 - W) */
        sp_256_mont_sqr_4(t1, x, p256_mod, p256_mp_mod);
        sp_256_mont_sub_4(t1, t1, w, p256_mod);
        sp_256_mont_tpl_lower_4(a, t1, p256_mod);
        /* B = X*Y^2 */
        sp_256_mont_sqr_4(t1, y, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(b, t1, x, p256_mod, p256_mp_mod);
        x = r[j].x;
        /* X = A^2 - 2B */
        sp_256_mont_sqr_4(x, a, p256_mod, p256_mp_mod);
        sp_256_mont_sub_dbl_4(x, x, b, p256_mod);
        /*   b = 2.(B - X) */
        sp_256_mont_dbl_sub_4(b, b, x, p256_mod);
        /* Z = Z*Y */
        sp_256_mont_mul_4(r[j].z, z, y, p256_mod, p256_mp_mod);
        z = r[j].z;
        /* t1 = Y^4 */
        sp_256_mont_sqr_4(t1, t1, p256_mod, p256_mp_mod);
        if (i != n) {
            /* W = W*Y^4 */
            sp_256_mont_mul_4(w, w, t1, p256_mod, p256_mp_mod);
        }
        /* y = 2*A*(B - X) - Y^4 */
        sp_256_mont_mul_4(y, b, a, p256_mod, p256_mp_mod);
        sp_256_mont_sub_4(y, y, t1, p256_mod);

        /* Y = Y/2 */
        sp_256_div2_4(r[j].y, y, p256_mod);
        r[j].infinity = 0;
    }
}

/* Add two Montgomery form projective points.
 *
 * ra  Result of addition.
 * rs  Result of subtraction.
 * p   First point to add.
 * q   Second point to add.
 * t   Temporary ordinate data.
 */
static void sp_256_proj_point_add_sub_4(sp_point_256* ra,
        sp_point_256* rs, const sp_point_256* p, const sp_point_256* q,
        sp_digit* t)
{
    sp_digit* t1 = t;
    sp_digit* t2 = t + 2*4;
    sp_digit* t3 = t + 4*4;
    sp_digit* t4 = t + 6*4;
    sp_digit* t5 = t + 8*4;
    sp_digit* t6 = t + 10*4;
    sp_digit* xa = ra->x;
    sp_digit* ya = ra->y;
    sp_digit* za = ra->z;
    sp_digit* xs = rs->x;
    sp_digit* ys = rs->y;
    sp_digit* zs = rs->z;


    XMEMCPY(xa, p->x, sizeof(p->x) / 2);
    XMEMCPY(ya, p->y, sizeof(p->y) / 2);
    XMEMCPY(za, p->z, sizeof(p->z) / 2);
    ra->infinity = 0;
    rs->infinity = 0;

    /* U1 = X1*Z2^2 */
    sp_256_mont_sqr_4(t1, q->z, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(t3, t1, q->z, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(t1, t1, xa, p256_mod, p256_mp_mod);
    /* U2 = X2*Z1^2 */
    sp_256_mont_sqr_4(t2, za, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(t4, t2, za, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(t2, t2, q->x, p256_mod, p256_mp_mod);
    /* S1 = Y1*Z2^3 */
    sp_256_mont_mul_4(t3, t3, ya, p256_mod, p256_mp_mod);
    /* S2 = Y2*Z1^3 */
    sp_256_mont_mul_4(t4, t4, q->y, p256_mod, p256_mp_mod);
    /* H = U2 - U1 */
    sp_256_mont_sub_4(t2, t2, t1, p256_mod);
    /* RS = S2 + S1 */
    sp_256_mont_add_4(t6, t4, t3, p256_mod);
    /* R = S2 - S1 */
    sp_256_mont_sub_4(t4, t4, t3, p256_mod);
    /* Z3 = H*Z1*Z2 */
    /* ZS = H*Z1*Z2 */
    sp_256_mont_mul_4(za, za, q->z, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(za, za, t2, p256_mod, p256_mp_mod);
    XMEMCPY(zs, za, sizeof(p->z)/2);
    /* X3 = R^2 - H^3 - 2*U1*H^2 */
    /* XS = RS^2 - H^3 - 2*U1*H^2 */
    sp_256_mont_sqr_4(xa, t4, p256_mod, p256_mp_mod);
    sp_256_mont_sqr_4(xs, t6, p256_mod, p256_mp_mod);
    sp_256_mont_sqr_4(t5, t2, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(ya, t1, t5, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(t5, t5, t2, p256_mod, p256_mp_mod);
    sp_256_mont_sub_4(xa, xa, t5, p256_mod);
    sp_256_mont_sub_4(xs, xs, t5, p256_mod);
    sp_256_mont_dbl_4(t1, ya, p256_mod);
    sp_256_mont_sub_4(xa, xa, t1, p256_mod);
    sp_256_mont_sub_4(xs, xs, t1, p256_mod);
    /* Y3 = R*(U1*H^2 - X3) - S1*H^3 */
    /* YS = -RS*(U1*H^2 - XS) - S1*H^3 */
    sp_256_mont_sub_lower_4(ys, ya, xs, p256_mod);
    sp_256_mont_sub_lower_4(ya, ya, xa, p256_mod);
    sp_256_mont_mul_4(ya, ya, t4, p256_mod, p256_mp_mod);
    sp_256_sub_4(t6, p256_mod, t6);
    sp_256_mont_mul_4(ys, ys, t6, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(t5, t5, t3, p256_mod, p256_mp_mod);
    sp_256_mont_sub_4(ya, ya, t5, p256_mod);
    sp_256_mont_sub_4(ys, ys, t5, p256_mod);
}

/* Structure used to describe recoding of scalar multiplication. */
typedef struct ecc_recode_256 {
    /* Index into pre-computation table. */
    uint8_t i;
    /* Use the negative of the point. */
    uint8_t neg;
} ecc_recode_256;

/* The index into pre-computation table to use. */
static const uint8_t recode_index_4_6[66] = {
     0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
    16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
    32, 31, 30, 29, 28, 27, 26, 25, 24, 23, 22, 21, 20, 19, 18, 17,
    16, 15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,
     0,  1,
};

/* Whether to negate y-ordinate. */
static const uint8_t recode_neg_4_6[66] = {
     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
     0,  0,
};

/* Recode the scalar for multiplication using pre-computed values and
 * subtraction.
 *
 * k  Scalar to multiply by.
 * v  Vector of operations to perform.
 */
static void sp_256_ecc_recode_6_4(const sp_digit* k, ecc_recode_256* v)
{
    int i;
    int j;
    uint8_t y;
    int carry = 0;
    int o;
    sp_digit n;

    j = 0;
    n = k[j];
    o = 0;
    for (i=0; i<43; i++) {
        y = (int8_t)n;
        if (o + 6 < 64) {
            y &= 0x3f;
            n >>= 6;
            o += 6;
        }
        else if (o + 6 == 64) {
            n >>= 6;
            if (++j < 4)
                n = k[j];
            o = 0;
        }
        else if (++j < 4) {
            n = k[j];
            y |= (uint8_t)((n << (64 - o)) & 0x3f);
            o -= 58;
            n >>= o;
        }

        y += (uint8_t)carry;
        v[i].i = recode_index_4_6[y];
        v[i].neg = recode_neg_4_6[y];
        carry = (y >> 6) + v[i].neg;
    }
}

#ifndef WC_NO_CACHE_RESISTANT
/* Touch each possible point that could be being copied.
 *
 * r      Point to copy into.
 * table  Table - start of the entires to access
 * idx    Index of entry to retrieve.
 */
static void sp_256_get_point_33_4(sp_point_256* r, const sp_point_256* table,
    int idx)
{
    int i;
    sp_digit mask;

    r->x[0] = 0;
    r->x[1] = 0;
    r->x[2] = 0;
    r->x[3] = 0;
    r->y[0] = 0;
    r->y[1] = 0;
    r->y[2] = 0;
    r->y[3] = 0;
    r->z[0] = 0;
    r->z[1] = 0;
    r->z[2] = 0;
    r->z[3] = 0;
    for (i = 1; i < 33; i++) {
        mask = 0 - (i == idx);
        r->x[0] |= mask & table[i].x[0];
        r->x[1] |= mask & table[i].x[1];
        r->x[2] |= mask & table[i].x[2];
        r->x[3] |= mask & table[i].x[3];
        r->y[0] |= mask & table[i].y[0];
        r->y[1] |= mask & table[i].y[1];
        r->y[2] |= mask & table[i].y[2];
        r->y[3] |= mask & table[i].y[3];
        r->z[0] |= mask & table[i].z[0];
        r->z[1] |= mask & table[i].z[1];
        r->z[2] |= mask & table[i].z[2];
        r->z[3] |= mask & table[i].z[3];
    }
}
#endif /* !WC_NO_CACHE_RESISTANT */
/* Multiply the point by the scalar and return the result.
 * If map is true then convert result to affine coordinates.
 *
 * Window technique of 6 bits. (Add-Sub variation.)
 * Calculate 0..32 times the point. Use function that adds and
 * subtracts the same two points.
 * Recode to add or subtract one of the computed points.
 * Double to push up.
 * NOT a sliding window.
 *
 * r     Resulting point.
 * g     Point to multiply.
 * k     Scalar to multiply by.
 * map   Indicates whether to convert result to affine.
 * ct    Constant time required.
 * heap  Heap to use for allocation.
 * returns MEMORY_E when memory allocation fails and MP_OKAY on success.
 */
static int sp_256_ecc_mulmod_win_add_sub_4(sp_point_256* r, const sp_point_256* g,
        const sp_digit* k, int map, int ct, void* heap)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_point_256* t = NULL;
    sp_digit* tmp = NULL;
#else
    sp_point_256 t[33+2];
    sp_digit tmp[2 * 4 * 6];
#endif
    sp_point_256* rt = NULL;
    sp_point_256* p = NULL;
    sp_digit* negy;
    int i;
    ecc_recode_256 v[43];
    int err = MP_OKAY;

    /* Constant time used for cache attack resistance implementation. */
    (void)ct;
    (void)heap;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    t = (sp_point_256*)XMALLOC(sizeof(sp_point_256) * 
        (33+2), heap, DYNAMIC_TYPE_ECC);
    if (t == NULL)
        err = MEMORY_E;
    if (err == MP_OKAY) {
        tmp = (sp_digit*)XMALLOC(sizeof(sp_digit) * 2 * 4 * 6,
                                 heap, DYNAMIC_TYPE_ECC);
        if (tmp == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        rt = t + 33;
        p  = t + 33+1;

        /* t[0] = {0, 0, 1} * norm */
        XMEMSET(&t[0], 0, sizeof(t[0]));
        t[0].infinity = 1;
        /* t[1] = {g->x, g->y, g->z} * norm */
        err = sp_256_mod_mul_norm_4(t[1].x, g->x, p256_mod);
    }
    if (err == MP_OKAY) {
        err = sp_256_mod_mul_norm_4(t[1].y, g->y, p256_mod);
    }
    if (err == MP_OKAY) {
        err = sp_256_mod_mul_norm_4(t[1].z, g->z, p256_mod);
    }

    if (err == MP_OKAY) {
        t[1].infinity = 0;
        /* t[2] ... t[32]  */
        sp_256_proj_point_dbl_n_store_4(t, &t[ 1], 5, 1, tmp);
        sp_256_proj_point_add_4(&t[ 3], &t[ 2], &t[ 1], tmp);
        sp_256_proj_point_dbl_4(&t[ 6], &t[ 3], tmp);
        sp_256_proj_point_add_sub_4(&t[ 7], &t[ 5], &t[ 6], &t[ 1], tmp);
        sp_256_proj_point_dbl_4(&t[10], &t[ 5], tmp);
        sp_256_proj_point_add_sub_4(&t[11], &t[ 9], &t[10], &t[ 1], tmp);
        sp_256_proj_point_dbl_4(&t[12], &t[ 6], tmp);
        sp_256_proj_point_dbl_4(&t[14], &t[ 7], tmp);
        sp_256_proj_point_add_sub_4(&t[15], &t[13], &t[14], &t[ 1], tmp);
        sp_256_proj_point_dbl_4(&t[18], &t[ 9], tmp);
        sp_256_proj_point_add_sub_4(&t[19], &t[17], &t[18], &t[ 1], tmp);
        sp_256_proj_point_dbl_4(&t[20], &t[10], tmp);
        sp_256_proj_point_dbl_4(&t[22], &t[11], tmp);
        sp_256_proj_point_add_sub_4(&t[23], &t[21], &t[22], &t[ 1], tmp);
        sp_256_proj_point_dbl_4(&t[24], &t[12], tmp);
        sp_256_proj_point_dbl_4(&t[26], &t[13], tmp);
        sp_256_proj_point_add_sub_4(&t[27], &t[25], &t[26], &t[ 1], tmp);
        sp_256_proj_point_dbl_4(&t[28], &t[14], tmp);
        sp_256_proj_point_dbl_4(&t[30], &t[15], tmp);
        sp_256_proj_point_add_sub_4(&t[31], &t[29], &t[30], &t[ 1], tmp);

        negy = t[0].y;

        sp_256_ecc_recode_6_4(k, v);

        i = 42;
    #ifndef WC_NO_CACHE_RESISTANT
        if (ct) {
            sp_256_get_point_33_4(rt, t, v[i].i);
            rt->infinity = !v[i].i;
        }
        else
    #endif
        {
            XMEMCPY(rt, &t[v[i].i], sizeof(sp_point_256));
        }
        for (--i; i>=0; i--) {
            sp_256_proj_point_dbl_n_4(rt, 6, tmp);

        #ifndef WC_NO_CACHE_RESISTANT
            if (ct) {
                sp_256_get_point_33_4(p, t, v[i].i);
                p->infinity = !v[i].i;
            }
            else
        #endif
            {
                XMEMCPY(p, &t[v[i].i], sizeof(sp_point_256));
            }
            sp_256_sub_4(negy, p256_mod, p->y);
            sp_256_norm_4(negy);
            sp_256_cond_copy_4(p->y, negy, (sp_digit)0 - v[i].neg);
            sp_256_proj_point_add_4(rt, rt, p, tmp);
        }

        if (map != 0) {
            sp_256_map_4(r, rt, tmp);
        }
        else {
            XMEMCPY(r, rt, sizeof(sp_point_256));
        }
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (t != NULL)
        XFREE(t, heap, DYNAMIC_TYPE_ECC);
    if (tmp != NULL)
        XFREE(tmp, heap, DYNAMIC_TYPE_ECC);
#endif

    return err;
}

#ifndef WC_NO_CACHE_RESISTANT
/* A table entry for pre-computed points. */
typedef struct sp_table_entry_256 {
    sp_digit x[4];
    sp_digit y[4];
} sp_table_entry_256;

#if defined(FP_ECC) || defined(WOLFSSL_SP_SMALL)
#endif /* FP_ECC | WOLFSSL_SP_SMALL */
/* Add two Montgomery form projective points. The second point has a q value of
 * one.
 * Only the first point can be the same pointer as the result point.
 *
 * r  Result of addition.
 * p  First point to add.
 * q  Second point to add.
 * t  Temporary ordinate data.
 */
static void sp_256_proj_point_add_qz1_4(sp_point_256* r, const sp_point_256* p,
        const sp_point_256* q, sp_digit* t)
{
    sp_digit* t1 = t;
    sp_digit* t2 = t + 2*4;
    sp_digit* t3 = t + 4*4;
    sp_digit* t4 = t + 6*4;
    sp_digit* t5 = t + 8*4;
    sp_digit* t6 = t + 10*4;

    /* Check double */
    (void)sp_256_sub_4(t1, p256_mod, q->y);
    sp_256_norm_4(t1);
    if ((~p->infinity & ~q->infinity &
        sp_256_cmp_equal_4(p->x, q->x) & sp_256_cmp_equal_4(p->z, q->z) &
        (sp_256_cmp_equal_4(p->y, q->y) | sp_256_cmp_equal_4(p->y, t1))) != 0) {
        sp_256_proj_point_dbl_4(r, p, t);
    }
    else {
        sp_digit maskp;
        sp_digit maskq;
        sp_digit maskt;
        sp_digit* x = t2;
        sp_digit* y = t5;
        sp_digit* z = t6;
        int i;

        /* U2 = X2*Z1^2 */
        sp_256_mont_sqr_4(t2, p->z, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t4, t2, p->z, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t2, t2, q->x, p256_mod, p256_mp_mod);
        /* S2 = Y2*Z1^3 */
        sp_256_mont_mul_4(t4, t4, q->y, p256_mod, p256_mp_mod);
        /* H = U2 - X1 */
        sp_256_mont_sub_4(t2, t2, p->x, p256_mod);
        /* R = S2 - Y1 */
        sp_256_mont_sub_4(t4, t4, p->y, p256_mod);
        /* Z3 = H*Z1 */
        sp_256_mont_mul_4(z, p->z, t2, p256_mod, p256_mp_mod);
        /* X3 = R^2 - H^3 - 2*X1*H^2 */
        sp_256_mont_sqr_4(t1, t4, p256_mod, p256_mp_mod);
        sp_256_mont_sqr_4(t5, t2, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t3, p->x, t5, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t5, t5, t2, p256_mod, p256_mp_mod);
        sp_256_mont_sub_4(x, t1, t5, p256_mod);
        sp_256_mont_sub_dbl_4(x, x, t3, p256_mod);
        /* Y3 = R*(X1*H^2 - X3) - Y1*H^3 */
        sp_256_mont_sub_lower_4(t3, t3, x, p256_mod);
        sp_256_mont_mul_4(t3, t3, t4, p256_mod, p256_mp_mod);
        sp_256_mont_mul_4(t5, t5, p->y, p256_mod, p256_mp_mod);
        sp_256_mont_sub_4(y, t3, t5, p256_mod);

        maskp = 0 - (q->infinity & (!p->infinity));
        maskq = 0 - (p->infinity & (!q->infinity));
        maskt = ~(maskp | maskq);
        for (i = 0; i < 4; i++) {
            r->x[i] = (p->x[i] & maskp) | (q->x[i] & maskq) | (x[i] & maskt);
        }
        for (i = 0; i < 4; i++) {
            r->y[i] = (p->y[i] & maskp) | (q->y[i] & maskq) | (y[i] & maskt);
        }
        for (i = 0; i < 4; i++) {
            r->z[i] = (p->z[i] & maskp) | (q->z[i] & maskq) | (z[i] & maskt);
        }
        r->z[0] |= p->infinity & q->infinity;
        r->infinity = p->infinity & q->infinity;
    }
}

#ifdef FP_ECC
/* Convert the projective point to affine.
 * Ordinates are in Montgomery form.
 *
 * a  Point to convert.
 * t  Temporary data.
 */
static void sp_256_proj_to_affine_4(sp_point_256* a, sp_digit* t)
{
    sp_digit* t1 = t;
    sp_digit* t2 = t + 2 * 4;
    sp_digit* tmp = t + 4 * 4;

    sp_256_mont_inv_4(t1, a->z, tmp);

    sp_256_mont_sqr_4(t2, t1, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(t1, t2, t1, p256_mod, p256_mp_mod);

    sp_256_mont_mul_4(a->x, a->x, t2, p256_mod, p256_mp_mod);
    sp_256_mont_mul_4(a->y, a->y, t1, p256_mod, p256_mp_mod);
    XMEMCPY(a->z, p256_norm_mod, sizeof(p256_norm_mod));
}

/* Generate the pre-computed table of points for the base point.
 *
 * width = 6
 * 64 entries
 * 42 bits between
 *
 * a      The base point.
 * table  Place to store generated point data.
 * tmp    Temporary data.
 * heap  Heap to use for allocation.
 */
static int sp_256_gen_stripe_table_4(const sp_point_256* a,
        sp_table_entry_256* table, sp_digit* tmp, void* heap)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_point_256* t = NULL;
#else
    sp_point_256 t[3];
#endif
    sp_point_256* s1 = NULL;
    sp_point_256* s2 = NULL;
    int i;
    int j;
    int err = MP_OKAY;

    (void)heap;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    t = (sp_point_256*)XMALLOC(sizeof(sp_point_256) * 3, heap,
                                     DYNAMIC_TYPE_ECC);
    if (t == NULL)
        err = MEMORY_E;
#endif

    if (err == MP_OKAY) {
        s1 = t + 1;
        s2 = t + 2;

        err = sp_256_mod_mul_norm_4(t->x, a->x, p256_mod);
    }
    if (err == MP_OKAY) {
        err = sp_256_mod_mul_norm_4(t->y, a->y, p256_mod);
    }
    if (err == MP_OKAY) {
        err = sp_256_mod_mul_norm_4(t->z, a->z, p256_mod);
    }
    if (err == MP_OKAY) {
        t->infinity = 0;
        sp_256_proj_to_affine_4(t, tmp);

        XMEMCPY(s1->z, p256_norm_mod, sizeof(p256_norm_mod));
        s1->infinity = 0;
        XMEMCPY(s2->z, p256_norm_mod, sizeof(p256_norm_mod));
        s2->infinity = 0;

        /* table[0] = {0, 0, infinity} */
        XMEMSET(&table[0], 0, sizeof(sp_table_entry_256));
        /* table[1] = Affine version of 'a' in Montgomery form */
        XMEMCPY(table[1].x, t->x, sizeof(table->x));
        XMEMCPY(table[1].y, t->y, sizeof(table->y));

        for (i=1; i<6; i++) {
            sp_256_proj_point_dbl_n_4(t, 43, tmp);
            sp_256_proj_to_affine_4(t, tmp);
            XMEMCPY(table[1<<i].x, t->x, sizeof(table->x));
            XMEMCPY(table[1<<i].y, t->y, sizeof(table->y));
        }

        for (i=1; i<6; i++) {
            XMEMCPY(s1->x, table[1<<i].x, sizeof(table->x));
            XMEMCPY(s1->y, table[1<<i].y, sizeof(table->y));
            for (j=(1<<i)+1; j<(1<<(i+1)); j++) {
                XMEMCPY(s2->x, table[j-(1<<i)].x, sizeof(table->x));
                XMEMCPY(s2->y, table[j-(1<<i)].y, sizeof(table->y));
                sp_256_proj_point_add_qz1_4(t, s1, s2, tmp);
                sp_256_proj_to_affine_4(t, tmp);
                XMEMCPY(table[j].x, t->x, sizeof(table->x));
                XMEMCPY(table[j].y, t->y, sizeof(table->y));
            }
        }
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (t != NULL)
        XFREE(t, heap, DYNAMIC_TYPE_ECC);
#endif

    return err;
}

#endif /* FP_ECC */
#if defined(FP_ECC) || defined(WOLFSSL_SP_SMALL)
#ifndef WC_NO_CACHE_RESISTANT
/* Touch each possible entry that could be being copied.
 *
 * r      Point to copy into.
 * table  Table - start of the entires to access
 * idx    Index of entry to retrieve.
 */
static void sp_256_get_entry_64_4(sp_point_256* r,
    const sp_table_entry_256* table, int idx)
{
    int i;
    sp_digit mask;

    r->x[0] = 0;
    r->x[1] = 0;
    r->x[2] = 0;
    r->x[3] = 0;
    r->y[0] = 0;
    r->y[1] = 0;
    r->y[2] = 0;
    r->y[3] = 0;
    for (i = 1; i < 64; i++) {
        mask = 0 - (i == idx);
        r->x[0] |= mask & table[i].x[0];
        r->x[1] |= mask & table[i].x[1];
        r->x[2] |= mask & table[i].x[2];
        r->x[3] |= mask & table[i].x[3];
        r->y[0] |= mask & table[i].y[0];
        r->y[1] |= mask & table[i].y[1];
        r->y[2] |= mask & table[i].y[2];
        r->y[3] |= mask & table[i].y[3];
    }
}
#endif /* !WC_NO_CACHE_RESISTANT */
/* Multiply the point by the scalar and return the result.
 * If map is true then convert result to affine coordinates.
 *
 * Stripe implementation.
 * Pre-generated: 2^0, 2^42, ...
 * Pre-generated: products of all combinations of above.
 * 6 doubles and adds (with qz=1)
 *
 * r      Resulting point.
 * k      Scalar to multiply by.
 * table  Pre-computed table.
 * map    Indicates whether to convert result to affine.
 * ct     Constant time required.
 * heap   Heap to use for allocation.
 * returns MEMORY_E when memory allocation fails and MP_OKAY on success.
 */
static int sp_256_ecc_mulmod_stripe_4(sp_point_256* r, const sp_point_256* g,
        const sp_table_entry_256* table, const sp_digit* k, int map,
        int ct, void* heap)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_point_256* rt = NULL;
    sp_digit* t = NULL;
#else
    sp_point_256 rt[2];
    sp_digit t[2 * 4 * 6];
#endif
    sp_point_256* p = NULL;
    int i;
    int j;
    int y;
    int x;
    int err = MP_OKAY;

    (void)g;
    /* Constant time used for cache attack resistance implementation. */
    (void)ct;
    (void)heap;


#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    rt = (sp_point_256*)XMALLOC(sizeof(sp_point_256) * 2, heap,
                                      DYNAMIC_TYPE_ECC);
    if (rt == NULL)
        err = MEMORY_E;
    if (err == MP_OKAY) {
        t = (sp_digit*)XMALLOC(sizeof(sp_digit) * 2 * 4 * 6, heap,
                               DYNAMIC_TYPE_ECC);
        if (t == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        p = rt + 1;

        XMEMCPY(p->z, p256_norm_mod, sizeof(p256_norm_mod));
        XMEMCPY(rt->z, p256_norm_mod, sizeof(p256_norm_mod));

        y = 0;
        x = 42;
        for (j=0; j<6 && x<256; j++) {
            y |= (int)(((k[x / 64] >> (x % 64)) & 1) << j);
            x += 43;
        }
    #ifndef WC_NO_CACHE_RESISTANT
        if (ct) {
            sp_256_get_entry_64_4(rt, table, y);
        } else
    #endif
        {
            XMEMCPY(rt->x, table[y].x, sizeof(table[y].x));
            XMEMCPY(rt->y, table[y].y, sizeof(table[y].y));
        }
        rt->infinity = !y;
        for (i=41; i>=0; i--) {
            y = 0;
            x = i;
            for (j=0; j<6 && x<256; j++) {
                y |= (int)(((k[x / 64] >> (x % 64)) & 1) << j);
                x += 43;
            }

            sp_256_proj_point_dbl_4(rt, rt, t);
        #ifndef WC_NO_CACHE_RESISTANT
            if (ct) {
                sp_256_get_entry_64_4(p, table, y);
            }
            else
        #endif
            {
                XMEMCPY(p->x, table[y].x, sizeof(table[y].x));
                XMEMCPY(p->y, table[y].y, sizeof(table[y].y));
            }
            p->infinity = !y;
            sp_256_proj_point_add_qz1_4(rt, rt, p, t);
        }

        if (map != 0) {
            sp_256_map_4(r, rt, t);
        }
        else {
            XMEMCPY(r, rt, sizeof(sp_point_256));
        }
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (t != NULL)
        XFREE(t, heap, DYNAMIC_TYPE_ECC);
    if (rt != NULL)
        XFREE(rt, heap, DYNAMIC_TYPE_ECC);
#endif

    return err;
}

#endif /* FP_ECC | WOLFSSL_SP_SMALL */
#ifdef FP_ECC
#ifndef FP_ENTRIES
    #define FP_ENTRIES 16
#endif

/* Cache entry - holds precomputation tables for a point. */
typedef struct sp_cache_256_t {
    /* X ordinate of point that table was generated from. */
    sp_digit x[4];
    /* Y ordinate of point that table was generated from. */
    sp_digit y[4];
    /* Precomputation table for point. */
    sp_table_entry_256 table[64];
    /* Count of entries in table. */
    uint32_t cnt;
    /* Point and table set in entry. */
    int set;
} sp_cache_256_t;

/* Cache of tables. */
static THREAD_LS_T sp_cache_256_t sp_cache_256[FP_ENTRIES];
/* Index of last entry in cache. */
static THREAD_LS_T int sp_cache_256_last = -1;
/* Cache has been initialized. */
static THREAD_LS_T int sp_cache_256_inited = 0;

#ifndef HAVE_THREAD_LS
    static volatile int initCacheMutex_256 = 0;
    static wolfSSL_Mutex sp_cache_256_lock;
#endif

/* Get the cache entry for the point.
 *
 * g      [in]   Point scalar multipling.
 * cache  [out]  Cache table to use.
 */
static void sp_ecc_get_cache_256(const sp_point_256* g, sp_cache_256_t** cache)
{
    int i;
    int j;
    uint32_t least;

    if (sp_cache_256_inited == 0) {
        for (i=0; i<FP_ENTRIES; i++) {
            sp_cache_256[i].set = 0;
        }
        sp_cache_256_inited = 1;
    }

    /* Compare point with those in cache. */
    for (i=0; i<FP_ENTRIES; i++) {
        if (!sp_cache_256[i].set)
            continue;

        if (sp_256_cmp_equal_4(g->x, sp_cache_256[i].x) &
                           sp_256_cmp_equal_4(g->y, sp_cache_256[i].y)) {
            sp_cache_256[i].cnt++;
            break;
        }
    }

    /* No match. */
    if (i == FP_ENTRIES) {
        /* Find empty entry. */
        i = (sp_cache_256_last + 1) % FP_ENTRIES;
        for (; i != sp_cache_256_last; i=(i+1)%FP_ENTRIES) {
            if (!sp_cache_256[i].set) {
                break;
            }
        }

        /* Evict least used. */
        if (i == sp_cache_256_last) {
            least = sp_cache_256[0].cnt;
            for (j=1; j<FP_ENTRIES; j++) {
                if (sp_cache_256[j].cnt < least) {
                    i = j;
                    least = sp_cache_256[i].cnt;
                }
            }
        }

        XMEMCPY(sp_cache_256[i].x, g->x, sizeof(sp_cache_256[i].x));
        XMEMCPY(sp_cache_256[i].y, g->y, sizeof(sp_cache_256[i].y));
        sp_cache_256[i].set = 1;
        sp_cache_256[i].cnt = 1;
    }

    *cache = &sp_cache_256[i];
    sp_cache_256_last = i;
}
#endif /* FP_ECC */

/* Multiply the base point of P256 by the scalar and return the result.
 * If map is true then convert result to affine coordinates.
 *
 * r     Resulting point.
 * g     Point to multiply.
 * k     Scalar to multiply by.
 * map   Indicates whether to convert result to affine.
 * ct    Constant time required.
 * heap  Heap to use for allocation.
 * returns MEMORY_E when memory allocation fails and MP_OKAY on success.
 */
static int sp_256_ecc_mulmod_4(sp_point_256* r, const sp_point_256* g, const sp_digit* k,
        int map, int ct, void* heap)
{
#ifndef FP_ECC
    return sp_256_ecc_mulmod_win_add_sub_4(r, g, k, map, ct, heap);
#else
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* tmp;
#else
    sp_digit tmp[2 * 4 * 6];
#endif
    sp_cache_256_t* cache;
    int err = MP_OKAY;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    tmp = (sp_digit*)XMALLOC(sizeof(sp_digit) * 2 * 4 * 6, heap, DYNAMIC_TYPE_ECC);
    if (tmp == NULL) {
        err = MEMORY_E;
    }
#endif
#ifndef HAVE_THREAD_LS
    if (err == MP_OKAY) {
        if (initCacheMutex_256 == 0) {
            wc_InitMutex(&sp_cache_256_lock);
            initCacheMutex_256 = 1;
        }
        if (wc_LockMutex(&sp_cache_256_lock) != 0) {
            err = BAD_MUTEX_E;
        }
    }
#endif /* HAVE_THREAD_LS */

    if (err == MP_OKAY) {
        sp_ecc_get_cache_256(g, &cache);
        if (cache->cnt == 2)
            sp_256_gen_stripe_table_4(g, cache->table, tmp, heap);

#ifndef HAVE_THREAD_LS
        wc_UnLockMutex(&sp_cache_256_lock);
#endif /* HAVE_THREAD_LS */

        if (cache->cnt < 2) {
            err = sp_256_ecc_mulmod_win_add_sub_4(r, g, k, map, ct, heap);
        }
        else {
            err = sp_256_ecc_mulmod_stripe_4(r, g, cache->table, k,
                    map, ct, heap);
        }
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    XFREE(tmp, heap, DYNAMIC_TYPE_ECC);
#endif
    return err;
#endif
}

#else
#if defined(FP_ECC) || defined(WOLFSSL_SP_SMALL)
#endif /* FP_ECC | WOLFSSL_SP_SMALL */
#ifdef FP_ECC
/* Generate the pre-computed table of points for the base point.
 *
 * width = 8
 * 256 entries
 * 32 bits between
 *
 * a      The base point.
 * table  Place to store generated point data.
 * tmp    Temporary data.
 * heap  Heap to use for allocation.
 */
static int sp_256_gen_stripe_table_4(const sp_point_256* a,
        sp_table_entry_256* table, sp_digit* tmp, void* heap)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_point_256* t = NULL;
#else
    sp_point_256 t[3];
#endif
    sp_point_256* s1 = NULL;
    sp_point_256* s2 = NULL;
    int i;
    int j;
    int err = MP_OKAY;

    (void)heap;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    t = (sp_point_256*)XMALLOC(sizeof(sp_point_256) * 3, heap,
                                     DYNAMIC_TYPE_ECC);
    if (t == NULL)
        err = MEMORY_E;
#endif

    if (err == MP_OKAY) {
        s1 = t + 1;
        s2 = t + 2;

        err = sp_256_mod_mul_norm_4(t->x, a->x, p256_mod);
    }
    if (err == MP_OKAY) {
        err = sp_256_mod_mul_norm_4(t->y, a->y, p256_mod);
    }
    if (err == MP_OKAY) {
        err = sp_256_mod_mul_norm_4(t->z, a->z, p256_mod);
    }
    if (err == MP_OKAY) {
        t->infinity = 0;
        sp_256_proj_to_affine_4(t, tmp);

        XMEMCPY(s1->z, p256_norm_mod, sizeof(p256_norm_mod));
        s1->infinity = 0;
        XMEMCPY(s2->z, p256_norm_mod, sizeof(p256_norm_mod));
        s2->infinity = 0;

        /* table[0] = {0, 0, infinity} */
        XMEMSET(&table[0], 0, sizeof(sp_table_entry_256));
        /* table[1] = Affine version of 'a' in Montgomery form */
        XMEMCPY(table[1].x, t->x, sizeof(table->x));
        XMEMCPY(table[1].y, t->y, sizeof(table->y));

        for (i=1; i<8; i++) {
            sp_256_proj_point_dbl_n_4(t, 32, tmp);
            sp_256_proj_to_affine_4(t, tmp);
            XMEMCPY(table[1<<i].x, t->x, sizeof(table->x));
            XMEMCPY(table[1<<i].y, t->y, sizeof(table->y));
        }

        for (i=1; i<8; i++) {
            XMEMCPY(s1->x, table[1<<i].x, sizeof(table->x));
            XMEMCPY(s1->y, table[1<<i].y, sizeof(table->y));
            for (j=(1<<i)+1; j<(1<<(i+1)); j++) {
                XMEMCPY(s2->x, table[j-(1<<i)].x, sizeof(table->x));
                XMEMCPY(s2->y, table[j-(1<<i)].y, sizeof(table->y));
                sp_256_proj_point_add_qz1_4(t, s1, s2, tmp);
                sp_256_proj_to_affine_4(t, tmp);
                XMEMCPY(table[j].x, t->x, sizeof(table->x));
                XMEMCPY(table[j].y, t->y, sizeof(table->y));
            }
        }
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (t != NULL)
        XFREE(t, heap, DYNAMIC_TYPE_ECC);
#endif

    return err;
}

#endif /* FP_ECC */
#if defined(FP_ECC) || defined(WOLFSSL_SP_SMALL)
#ifndef WC_NO_CACHE_RESISTANT
/* Touch each possible entry that could be being copied.
 *
 * r      Point to copy into.
 * table  Table - start of the entires to access
 * idx    Index of entry to retrieve.
 */
static void sp_256_get_entry_256_4(sp_point_256* r,
    const sp_table_entry_256* table, int idx)
{
    int i;
    sp_digit mask;

    r->x[0] = 0;
    r->x[1] = 0;
    r->x[2] = 0;
    r->x[3] = 0;
    r->y[0] = 0;
    r->y[1] = 0;
    r->y[2] = 0;
    r->y[3] = 0;
    for (i = 1; i < 256; i++) {
        mask = 0 - (i == idx);
        r->x[0] |= mask & table[i].x[0];
        r->x[1] |= mask & table[i].x[1];
        r->x[2] |= mask & table[i].x[2];
        r->x[3] |= mask & table[i].x[3];
        r->y[0] |= mask & table[i].y[0];
        r->y[1] |= mask & table[i].y[1];
        r->y[2] |= mask & table[i].y[2];
        r->y[3] |= mask & table[i].y[3];
    }
}
#endif /* !WC_NO_CACHE_RESISTANT */
/* Multiply the point by the scalar and return the result.
 * If map is true then convert result to affine coordinates.
 *
 * Stripe implementation.
 * Pre-generated: 2^0, 2^32, ...
 * Pre-generated: products of all combinations of above.
 * 8 doubles and adds (with qz=1)
 *
 * r      Resulting point.
 * k      Scalar to multiply by.
 * table  Pre-computed table.
 * map    Indicates whether to convert result to affine.
 * ct     Constant time required.
 * heap   Heap to use for allocation.
 * returns MEMORY_E when memory allocation fails and MP_OKAY on success.
 */
static int sp_256_ecc_mulmod_stripe_4(sp_point_256* r, const sp_point_256* g,
        const sp_table_entry_256* table, const sp_digit* k, int map,
        int ct, void* heap)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_point_256* rt = NULL;
    sp_digit* t = NULL;
#else
    sp_point_256 rt[2];
    sp_digit t[2 * 4 * 6];
#endif
    sp_point_256* p = NULL;
    int i;
    int j;
    int y;
    int x;
    int err = MP_OKAY;

    (void)g;
    /* Constant time used for cache attack resistance implementation. */
    (void)ct;
    (void)heap;


#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    rt = (sp_point_256*)XMALLOC(sizeof(sp_point_256) * 2, heap,
                                      DYNAMIC_TYPE_ECC);
    if (rt == NULL)
        err = MEMORY_E;
    if (err == MP_OKAY) {
        t = (sp_digit*)XMALLOC(sizeof(sp_digit) * 2 * 4 * 6, heap,
                               DYNAMIC_TYPE_ECC);
        if (t == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        p = rt + 1;

        XMEMCPY(p->z, p256_norm_mod, sizeof(p256_norm_mod));
        XMEMCPY(rt->z, p256_norm_mod, sizeof(p256_norm_mod));

        y = 0;
        x = 31;
        for (j=0; j<8; j++) {
            y |= (int)(((k[x / 64] >> (x % 64)) & 1) << j);
            x += 32;
        }
    #ifndef WC_NO_CACHE_RESISTANT
        if (ct) {
            sp_256_get_entry_256_4(rt, table, y);
        } else
    #endif
        {
            XMEMCPY(rt->x, table[y].x, sizeof(table[y].x));
            XMEMCPY(rt->y, table[y].y, sizeof(table[y].y));
        }
        rt->infinity = !y;
        for (i=30; i>=0; i--) {
            y = 0;
            x = i;
            for (j=0; j<8; j++) {
                y |= (int)(((k[x / 64] >> (x % 64)) & 1) << j);
                x += 32;
            }

            sp_256_proj_point_dbl_4(rt, rt, t);
        #ifndef WC_NO_CACHE_RESISTANT
            if (ct) {
                sp_256_get_entry_256_4(p, table, y);
            }
            else
        #endif
            {
                XMEMCPY(p->x, table[y].x, sizeof(table[y].x));
                XMEMCPY(p->y, table[y].y, sizeof(table[y].y));
            }
            p->infinity = !y;
            sp_256_proj_point_add_qz1_4(rt, rt, p, t);
        }

        if (map != 0) {
            sp_256_map_4(r, rt, t);
        }
        else {
            XMEMCPY(r, rt, sizeof(sp_point_256));
        }
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (t != NULL)
        XFREE(t, heap, DYNAMIC_TYPE_ECC);
    if (rt != NULL)
        XFREE(rt, heap, DYNAMIC_TYPE_ECC);
#endif

    return err;
}

#endif /* FP_ECC | WOLFSSL_SP_SMALL */
#ifdef FP_ECC
#ifndef FP_ENTRIES
    #define FP_ENTRIES 16
#endif

/* Cache entry - holds precomputation tables for a point. */
typedef struct sp_cache_256_t {
    /* X ordinate of point that table was generated from. */
    sp_digit x[4];
    /* Y ordinate of point that table was generated from. */
    sp_digit y[4];
    /* Precomputation table for point. */
    sp_table_entry_256 table[256];
    /* Count of entries in table. */
    uint32_t cnt;
    /* Point and table set in entry. */
    int set;
} sp_cache_256_t;

/* Cache of tables. */
static THREAD_LS_T sp_cache_256_t sp_cache_256[FP_ENTRIES];
/* Index of last entry in cache. */
static THREAD_LS_T int sp_cache_256_last = -1;
/* Cache has been initialized. */
static THREAD_LS_T int sp_cache_256_inited = 0;

#ifndef HAVE_THREAD_LS
    static volatile int initCacheMutex_256 = 0;
    static wolfSSL_Mutex sp_cache_256_lock;
#endif

/* Get the cache entry for the point.
 *
 * g      [in]   Point scalar multipling.
 * cache  [out]  Cache table to use.
 */
static void sp_ecc_get_cache_256(const sp_point_256* g, sp_cache_256_t** cache)
{
    int i;
    int j;
    uint32_t least;

    if (sp_cache_256_inited == 0) {
        for (i=0; i<FP_ENTRIES; i++) {
            sp_cache_256[i].set = 0;
        }
        sp_cache_256_inited = 1;
    }

    /* Compare point with those in cache. */
    for (i=0; i<FP_ENTRIES; i++) {
        if (!sp_cache_256[i].set)
            continue;

        if (sp_256_cmp_equal_4(g->x, sp_cache_256[i].x) &
                           sp_256_cmp_equal_4(g->y, sp_cache_256[i].y)) {
            sp_cache_256[i].cnt++;
            break;
        }
    }

    /* No match. */
    if (i == FP_ENTRIES) {
        /* Find empty entry. */
        i = (sp_cache_256_last + 1) % FP_ENTRIES;
        for (; i != sp_cache_256_last; i=(i+1)%FP_ENTRIES) {
            if (!sp_cache_256[i].set) {
                break;
            }
        }

        /* Evict least used. */
        if (i == sp_cache_256_last) {
            least = sp_cache_256[0].cnt;
            for (j=1; j<FP_ENTRIES; j++) {
                if (sp_cache_256[j].cnt < least) {
                    i = j;
                    least = sp_cache_256[i].cnt;
                }
            }
        }

        XMEMCPY(sp_cache_256[i].x, g->x, sizeof(sp_cache_256[i].x));
        XMEMCPY(sp_cache_256[i].y, g->y, sizeof(sp_cache_256[i].y));
        sp_cache_256[i].set = 1;
        sp_cache_256[i].cnt = 1;
    }

    *cache = &sp_cache_256[i];
    sp_cache_256_last = i;
}
#endif /* FP_ECC */

/* Multiply the base point of P256 by the scalar and return the result.
 * If map is true then convert result to affine coordinates.
 *
 * r     Resulting point.
 * g     Point to multiply.
 * k     Scalar to multiply by.
 * map   Indicates whether to convert result to affine.
 * ct    Constant time required.
 * heap  Heap to use for allocation.
 * returns MEMORY_E when memory allocation fails and MP_OKAY on success.
 */
static int sp_256_ecc_mulmod_4(sp_point_256* r, const sp_point_256* g, const sp_digit* k,
        int map, int ct, void* heap)
{
#ifndef FP_ECC
    return sp_256_ecc_mulmod_win_add_sub_4(r, g, k, map, ct, heap);
#else
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_digit* tmp;
#else
    sp_digit tmp[2 * 4 * 6];
#endif
    sp_cache_256_t* cache;
    int err = MP_OKAY;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    tmp = (sp_digit*)XMALLOC(sizeof(sp_digit) * 2 * 4 * 6, heap, DYNAMIC_TYPE_ECC);
    if (tmp == NULL) {
        err = MEMORY_E;
    }
#endif
#ifndef HAVE_THREAD_LS
    if (err == MP_OKAY) {
        if (initCacheMutex_256 == 0) {
            wc_InitMutex(&sp_cache_256_lock);
            initCacheMutex_256 = 1;
        }
        if (wc_LockMutex(&sp_cache_256_lock) != 0) {
            err = BAD_MUTEX_E;
        }
    }
#endif /* HAVE_THREAD_LS */

    if (err == MP_OKAY) {
        sp_ecc_get_cache_256(g, &cache);
        if (cache->cnt == 2)
            sp_256_gen_stripe_table_4(g, cache->table, tmp, heap);

#ifndef HAVE_THREAD_LS
        wc_UnLockMutex(&sp_cache_256_lock);
#endif /* HAVE_THREAD_LS */

        if (cache->cnt < 2) {
            err = sp_256_ecc_mulmod_win_add_sub_4(r, g, k, map, ct, heap);
        }
        else {
            err = sp_256_ecc_mulmod_stripe_4(r, g, cache->table, k,
                    map, ct, heap);
        }
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    XFREE(tmp, heap, DYNAMIC_TYPE_ECC);
#endif
    return err;
#endif
}

#endif /* !WC_NO_CACHE_RESISTANT */
/* Multiply the point by the scalar and return the result.
 * If map is true then convert result to affine coordinates.
 *
 * km    Scalar to multiply by.
 * p     Point to multiply.
 * r     Resulting point.
 * map   Indicates whether to convert result to affine.
 * heap  Heap to use for allocation.
 * returns MEMORY_E when memory allocation fails and MP_OKAY on success.
 */
int sp_ecc_mulmod_256(const mp_int* km, const ecc_point* gm, ecc_point* r,
        int map, void* heap)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_point_256* point = NULL;
    sp_digit* k = NULL;
#else
    sp_point_256 point[1];
    sp_digit k[4];
#endif
    int err = MP_OKAY;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    point = (sp_point_256*)XMALLOC(sizeof(sp_point_256), heap,
                                         DYNAMIC_TYPE_ECC);
    if (point == NULL)
        err = MEMORY_E;
    if (err == MP_OKAY) {
        k = (sp_digit*)XMALLOC(sizeof(sp_digit) * 4, heap,
                               DYNAMIC_TYPE_ECC);
        if (k == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        sp_256_from_mp(k, 4, km);
        sp_256_point_from_ecc_point_4(point, gm);

            err = sp_256_ecc_mulmod_4(point, point, k, map, 1, heap);
    }
    if (err == MP_OKAY) {
        err = sp_256_point_to_ecc_point_4(point, r);
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (k != NULL)
        XFREE(k, heap, DYNAMIC_TYPE_ECC);
    if (point != NULL)
        XFREE(point, heap, DYNAMIC_TYPE_ECC);
#endif

    return err;
}

/* Multiply the point by the scalar, add point a and return the result.
 * If map is true then convert result to affine coordinates.
 *
 * km      Scalar to multiply by.
 * p       Point to multiply.
 * am      Point to add to scalar mulitply result.
 * inMont  Point to add is in montgomery form.
 * r       Resulting point.
 * map     Indicates whether to convert result to affine.
 * heap    Heap to use for allocation.
 * returns MEMORY_E when memory allocation fails and MP_OKAY on success.
 */
int sp_ecc_mulmod_add_256(const mp_int* km, const ecc_point* gm,
    const ecc_point* am, int inMont, ecc_point* r, int map, void* heap)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    sp_point_256* point = NULL;    
    sp_digit* k = NULL;
#else
    sp_point_256 point[2];
    sp_digit k[4 + 4 * 2 * 6];
#endif
    sp_point_256* addP = NULL;
    sp_digit* tmp = NULL;
    int err = MP_OKAY;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    point = (sp_point_256*)XMALLOC(sizeof(sp_point_256) * 2, heap,
                                         DYNAMIC_TYPE_ECC);
    if (point == NULL)
        err = MEMORY_E;
    if (err == MP_OKAY) {
        k = (sp_digit*)XMALLOC(
            sizeof(sp_digit) * (4 + 4 * 2 * 6), heap,
            DYNAMIC_TYPE_ECC);
        if (k == NULL)
            err = MEMORY_E;
    }
#endif

    if (err == MP_OKAY) {
        addP = point + 1;
        tmp = k + 4;

        sp_256_from_mp(k, 4, km);
        sp_256_point_from_ecc_point_4(point, gm);
        sp_256_point_from_ecc_point_4(addP, am);
    }
    if ((err == MP_OKAY) && (!inMont)) {
        err = sp_256_mod_mul_norm_4(addP->x, addP->x, p256_mod);
    }
    if ((err == MP_OKAY) && (!inMont)) {
        err = sp_256_mod_mul_norm_4(addP->y, addP->y, p256_mod);
    }
    if ((err == MP_OKAY) && (!inMont)) {
        err = sp_256_mod_mul_norm_4(addP->z, addP->z, p256_mod);
    }
    if (err == MP_OKAY) {
            err = sp_256_ecc_mulmod_4(point, point, k, 0, 0, heap);
    }
    if (err == MP_OKAY) {
            sp_256_proj_point_add_4(point, point, addP, tmp);

        if (map) {
                sp_256_map_4(point, point, tmp);
        }

        err = sp_256_point_to_ecc_point_4(point, r);
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_SP_NO_MALLOC)
    if (k != NULL)
        XFREE(k, heap, DYNAMIC_TYPE_ECC);
    if (point != NULL)
        XFREE(point, heap, DYNAMIC_TYPE_ECC);
#endif

    return err;
}

#ifdef WOLFSSL_SP_SMALL
#ifndef WC_NO_CACHE_RESISTANT
/* Striping precomputation table.
 * 6 points combined into a table of 64 points.
 * Distance of 43 between points.
 */
static const sp_table_entry_256 p256_table[64] = {
    /* 0 */
    { { 0x00, 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x00, 0x00 } },
    /* 1 */
    { { 0x79e730d418a9143cL,0x75ba95fc5fedb601L,0x79fb732b77622510L,
        0x18905f76a53755c6L },
      { 0xddf25357ce95560aL,0x8b4ab8e4ba19e45cL,0xd2e88688dd21f325L,
        0x8571ff1825885d85L } },
    /* 2 */
    { { 0x8910507903605c39L,0xf0843d9ea142c96cL,0xf374493416923684L,
        0x732caa2ffa0a2893L },
      { 0xb2e8c27061160170L,0xc32788cc437fbaa3L,0x39cd818ea6eda3acL,
        0xe2e942399e2b2e07L } },
    /* 3 */
    { { 0xb9c0d276abc3e190L,0x610e3d4dcb55b9caL,0xd16dbd025720f50aL,
        0xd0ed73dca607de84L },
      { 0x3bbde5bf49219fb5L,0x698e12c057771843L,0xdb606a9763470a5eL,
        0x61c71975853635d5L } },
    /* 4 */
    { { 0xeb5ddcb6ec7fae9fL,0x995f2714efb66e5aL,0xdee95d8e69445d52L,
        0x1b6c2d4609e27620L },
      { 0x32621c318129d716L,0xb03909f10958c1aaL,0x8c468ef91af4af63L,
        0x162c429ffba5cdf6L } },
    /* 5 */
    { { 0x4615d912c1d85f12L,0x1f0880b0e1f4e302L,0x336bcc896f1fca13L,
        0xda59ad0dc70dedbcL },
      { 0x3897efaeb0f62eceL,0xbaed81cdf4990cfdL,0xa3b1c2f260321bbbL,
        0x2aefd95addc84f79L } },
    /* 6 */
    { { 0x2d427e3cee9e92e6L,0x43d40da0437fe629L,0x0006e4e06ab72b31L,
        0x21ccfbb46f5c8e02L },
      { 0x53a2f1a753e821ecL,0x5d72d201e209d591L,0xfd84a26445e8ad41L,
        0x86ee0e684059cc6eL } },
    /* 7 */
    { { 0x3d8242d09248fce2L,0x32d4bf827f49f33dL,0x78807beb29d41fd1L,
        0xfce48b99f8f562cbL },
      { 0x72a7d4849f38f097L,0x1b482c10a37059adL,0xc1aa8284472e5ed3L,
        0xc5d6f3bbef23e9c9L } },
    /* 8 */
    { { 0x23f949feb8a24a20L,0x17ebfed1f52ca53fL,0x9b691bbebcfb4853L,
        0x5617ff6b6278a05dL },
      { 0x241b34c5e3c99ebdL,0xfc64242e1784156aL,0x4206482f695d67dfL,
        0xb967ce0eee27c011L } },
    /* 9 */
    { { 0x569aacdf9fc3df19L,0x0c6782c7c34c6fb2L,0xbb5f98b2c4ec873dL,
        0x5578433b9fe9e475L },
      { 0xfa14f3869ca84821L,0xb8ef658d39589501L,0x4022c48e07127b8eL,
        0xcbc4dfe35402ea12L } },
    /* 10 */
    { { 0x092ef96a2ad408a3L,0xf1e1a4c4cfbc45a3L,0x966b2676efeecdeeL,
        0xa0e2c6713a6216c5L },
      { 0xcd6e22a292c4bf61L,0x56d99a11d830dfc7L,0xb8c612bd259de547L,
        0x3d8e9a72e91f8ff7L } },
    /* 11 */
    { { 0x0b885e962352b4ffL,0x6be320d2a6545766L,0xbd22a444b9a59e72L,
        0x2f2d32d6ccc55d7dL },
      { 0xd86e4c4cddcec70bL,0x19cdb0e97a25c934L,0x542ade069ca97e28L,
        0x58c5927c746517f7L } },
    /* 12 */
    { { 0x24abb0f08d087091L,0x6aa2c2ef51add8deL,0xc3e1cb4ccc2a2134L,
        0x3563112895589212L },
      { 0x3bf17d2a7984344bL,0xbcb6f7b2f8a142ccL,0xd6057d8a08ec9266L,
        0x75c150d22852405aL } },
    /* 13 */
    { { 0xa8f88eb5a9fee73eL,0x72a84174576ea39bL,0x671fa0ade2692e7dL,
        0x2556288596769f9eL },
      { 0x254323bce850a6b0L,0x74b61c18fff6c89aL,0x2e7c563fcfae2690L,
        0x2cf454b7164afb0fL } },
    /* 14 */
    { { 0xe312a5618f10f423L,0x59a1f1fff2b85df4L,0x56c5991941c48122L,
        0x74953c1eae3d175fL },
      { 0x4d767fc78859244cL,0xc486bc00719a4cc1L,0xdd282985df1c1787L,
        0x1143301aae93c719L } },
    /* 15 */
    { { 0x7201a1d61fab7d71L,0x65931f5432cbbee8L,0x202955d3dcb387eeL,
        0xa5045ba5c4678432L },
      { 0xcfb5ee87dca85ff6L,0xdd25a7c6dfec0f67L,0xfee47169356a87c6L,
        0x20a8f159c3d7ece9L } },
    /* 16 */
    { { 0xe4ac8b33070d3aabL,0x2643672b9a2cd5e5L,0x52eff79b1cfc9173L,
        0x665ca49b90a7c13fL },
      { 0x5a8dda59b3efb998L,0x8a5b922d052f1341L,0xae9ebbab3cf9a530L,
        0x35986e7bf56da4d7L } },
    /* 17 */
    { { 0x21e07f9abc0a70c0L,0xecfdb3a2989a0182L,0x360682c0e40e8125L,
        0x73a637952f837f32L },
      { 0xf4eb8cef9c0d326bL,0xefb97fecebf4c7a5L,0xf9352123af3d5d7eL,
        0xb71ef4ef34e22ab1L } },
    /* 18 */
    { { 0xd6bd0d810d488032L,0x1676df9971f0b92eL,0xa7acdcfcb6d215acL,
        0x82461a26cd0ff939L },
      { 0x827189c0b635d2e5L,0x18f3b6dda92f1622L,0x10d738aa05cef325L,
        0x12c2a13f39bb0aa6L } },
    /* 19 */
    { { 0x5f94d8deb50b4e82L,0xbcd9144e34bd93e9L,0x61c3392107c08623L,
        0xedec947e7e3de8eeL },
      { 0x9d2da51d2f21b202L,0xc0c885cd96692a89L,0x4a613462a5e7309cL,
        0x227788550f28dee6L } },
    /* 20 */
    { { 0x1ff0bd527695447aL,0x63534a4a42ae2627L,0xd96af0dad0cc09f2L,
        0xb59ea545412d3e1aL },
      { 0xd10518cf6a759072L,0xffeec37c10475dfdL,0xacbc29ccb25089c4L,
        0xbf3dfc8521b6d4eeL } },
    /* 21 */
    { { 0x8f2eacfe49388995L,0x000fc8d4841be9edL,0x2ed8085a6955c290L,
        0x1929cf606d8e176fL },
      { 0x2efd26a5fd1a09dbL,0x58d767ad6cb626cdL,0x13a81b95b26c6e05L,
        0x68fe61078f61832bL } },
    /* 22 */
    { { 0x4ad7de2e2d85c2f6L,0xcd552fcb510101a1L,0x638d122b02acdabfL,
        0x117221e850bfd921L },
      { 0x08571ee199a99129L,0xebd046d1ba2f03a9L,0x035ed7baa6f8a181L,
        0x8aabf98d3187c6f3L } },
    /* 23 */
    { { 0xaf8e65cae3ab5f4eL,0x8b0b8b897561a69cL,0x37e83aa0b17c1e66L,
        0xe894d84cf8d80edcL },
      { 0xf1e465e7ce514e22L,0xc7fa324ca72340efL,0x08297fcae7370673L,
        0x4f799682b119ae5eL } },
    /* 24 */
    { { 0x014d6bd8f180f206L,0x56640c8b7ab44f55L,0x9a39660d93f9a5b8L,
        0xcac069e9959b68f1L },
      { 0x2bf6b65e208d9918L,0xb7e45dfb3f943291L,0xad5770f0d439c712L,
        0xfec635e17654d805L } },
    /* 25 */
    { { 0x37221cd13f031a88L,0xe4d53d2f0b5558d4L,0x2ede8e8fdafc51cdL,
        0xb587284ca8a883eaL },
      { 0xfa37674044fa5251L,0x5e5e18f95c5e3528L,0x8af51fac6e10b958L,
        0x09be79032c429b30L } },
    /* 26 */
    { { 0x7a468ba47f29936dL,0xacbbe3657cfb8176L,0xe892c10a4db9cd5dL,
        0xcb2f29d7a1aade8bL },
      { 0x3087eef4efffcb14L,0x92a7f3ec2afe8f2eL,0x199d89b8136f29d2L,
        0x3131604eb4836623L } },
    /* 27 */
    { { 0xf5cca5da31b5df76L,0x9431318676a4abc0L,0x5db8e6f71877c7c7L,
        0x3ce3f5f96031ac99L },
      { 0x585961d07e7cef80L,0x5ed6e841d424f16aL,0x18289cd056b16a49L,
        0x8008d03b2e5770faL } },
    /* 28 */
    { { 0xc8c2af64254e39deL,0x783cea738582571cL,0x2f2f55f1a6edd971L,
        0x7e00cc92c86bf30aL },
      { 0xa0db735447d7491fL,0xb3eb751ca5b12260L,0x3bc39a23297fb234L,
        0xd1330c20b8b4bfe4L } },
    /* 29 */
    { { 0xfb776af07824d53aL,0x04709096422dea35L,0x6f480b6b5fec3ac7L,
        0xdb2b1b62e27edda4L },
      { 0x0bba904cda78b494L,0x37ef59b691a147f7L,0xf880517726a4730aL,
        0xecc9d79aa8ab368eL } },
    /* 30 */
    { { 0x628e05c185a4bd0eL,0xebf7b67800e244e8L,0xf645947b8b176eebL,
        0xc92bf8301641ab35L },
      { 0x7a039c1a21be7a6fL,0x11e4354d2fd4bd92L,0x42552422886fd224L,
        0xdbf3194cc44ced37L } },
    /* 31 */
    { { 0x832da983c56f6b04L,0x7aaa84eb8ef098aeL,0x602e3eefa6a616a2L,
        0xc2824ddcb7b717a3L },
      { 0x19f50324ddb0a2e9L,0x04553a285bedfbbdL,0x37ea8b12aa1aee0aL,
        0xc1844e79945959a1L } },
    /* 32 */
    { { 0x5043dea7e0f222c2L,0x309d42ac72e65142L,0x94fe9ddd9216cd30L,
        0xd6539c7d0f87feecL },
      { 0x03c5a57c432ac7d7L,0x72692cf0327fda10L,0xec28c85f280698deL,
        0x2331fb467ec283b1L } },
    /* 33 */
    { { 0x651cfdeb43248e67L,0x2c3d72ceee561de8L,0xa48b8f33443dac8bL,
        0xe6b042fe7991f986L },
      { 0xd091636de810bcd2L,0xfc1e96aea97416d7L,0x2b6087cb2892694dL,
        0x0f8ac2459985a628L } },
    /* 34 */
    { { 0x54e908747f2326a2L,0xce43dd44fa9e1131L,0x4b2c740cd3d2d948L,
        0x9b0b126aa86e8b07L },
      { 0x228ef320b77f5af2L,0x14fc8a01ca07661cL,0x1d72509ed34f1a3aL,
        0xd169031729d9086eL } },
    /* 35 */
    { { 0x13e44acc03c5fe33L,0x13f4374e0105bbc6L,0x0cba5018cb4451b8L,
        0xa1a38e4afa29a4e1L },
      { 0x063fb9a8f4403917L,0x7afe108f996ea7f2L,0xec252363f93a1f87L,
        0xc029c8117e432609L } },
    /* 36 */
    { { 0x25080c29486e548eL,0xdaa411327868ab32L,0x46891511d61d1a3aL,
        0xc87f3f533efc8facL },
      { 0x984f613ff3e31393L,0x10bb15f67648f5d2L,0xe4990f2bdefaa440L,
        0xce647f03dd51c31dL } },
    /* 37 */
    { { 0x3161ebdd9c2c0abfL,0x48b7ee7bf497cf35L,0x9233e31d94dd9c97L,
        0x4aef9a62c5d2988fL },
      { 0x89a54161a03e6456L,0x9d25e003c1f02b47L,0x8784cdbfc1857782L,
        0x7928cafd0222b49cL } },
    /* 38 */
    { { 0x5a591abdecf4ea23L,0xb2725e8a80bd9b8aL,0xf569679f29ff348bL,
        0xa28163d36f22536aL },
      { 0x89e7a8f621c43971L,0x60cbe4a1c4a09567L,0x41046c8f5928b03dL,
        0x646feda7ef74a95aL } },
    /* 39 */
    { { 0x3aef6bc05d75d310L,0xf3e7f03c82476e5cL,0x9dcf3d508419b8a0L,
        0x221a3885eaf07f07L },
      { 0x16d533f337bdcb7dL,0xd778066bbb49550dL,0xf6f4540936c2600cL,
        0x7544396fc1c61709L } },
    /* 40 */
    { { 0xf79f556fde08cd42L,0x7d0aba1ee13cadc8L,0x841d9df6d4d81fefL,
        0x8f7ae1f2602d2043L },
      { 0x950c4de4b57ee181L,0xfe51e045c55cf490L,0xdb60b56a1efdd0a8L,
        0x276bccb3bf0fa497L } },
    /* 41 */
    { { 0x7926625b19e5a603L,0xf1b98e93e1bf712bL,0x933ecb52e33abeccL,
        0x9ebfc506f826619bL },
      { 0xd2965f67a1692c52L,0x8ac4012dfc4f9564L,0xa8af57036739f003L,
        0x7dd2282dbc715e13L } },
    /* 42 */
    { { 0x3ec01587cf2bb490L,0x5346082c3f1ea428L,0xf2c679e26739e506L,
        0xeab710d6930c28e4L },
      { 0xe9947ff8e043249aL,0x63640678ad54b0e6L,0x8cde42591854eaafL,
        0xf1feeaec6b25bdceL } },
    /* 43 */
    { { 0x49f7e8991bdd2aa2L,0x88fd273534e3cae9L,0x5ac0510182cbfea2L,
        0x324c9d414cf84578L },
      { 0xa242311719f13061L,0x69d67cf15f3b9932L,0x32ecdb3cdde2dfadL,
        0x2f74d995b916f7a6L } },
    /* 44 */
    { { 0x35f7ed423d14bc68L,0x32f63a0445574f91L,0xd04108335e8801e7L,
        0x63b6f13c1c9c1462L },
      { 0x180dcbcd9dc7201fL,0xa07b5b2c360350dfL,0x2582b2774236f5ccL,
        0x90163924a7ab06b9L } },
    /* 45 */
    { { 0x35e751b50767cdf2L,0x808372e69d8e2838L,0xcbad6b30646914d7L,
        0x4eeeb1de6c7b3cabL },
      { 0x3ef3af968c965004L,0xd162290fd281920bL,0x4626c313181f811bL,
        0x5fa42f4fbe61dd14L } },
    /* 46 */
    { { 0x1f5a9c53a185e98eL,0x13c28277ea9e83c3L,0xb566e4c0b693a226L,
        0x2ea3f1c001533e9eL },
      { 0xb4dbcc336215a21fL,0x7df608c3cb4e98f0L,0x677df928b4dd95ddL,
        0x4c1d7142eeed2934L } },
    /* 47 */
    { { 0x30bf236c86a2ee12L,0x74d5a12705ecb4c0L,0x9ef43b0f1601cca9L,
        0xbe1b1bf9ac4dd202L },
      { 0x84943e4717b6f93bL,0x6f789757cd5214b3L,0x5e0db1a97f313dfaL,
        0x0515efacece0b72bL } },
    /* 48 */
    { { 0x433a677ca78c3f8bL,0x204a9feaf376a9c1L,0xb6bfbea444baeadfL,
        0x5a43cafd2b48a3f4L },
      { 0xe25a7d0b67d1d226L,0xb2115844f6837985L,0x8c9cca3ed87c2b88L,
        0xecd4bc73894772e1L } },
    /* 49 */
    { { 0x368abec6783490e7L,0xf26da8bdd925c359L,0xf9b643e5e8fb0679L,
        0x7ab803d9b555d175L },
      { 0x1b4059994ebae595L,0x07fbbf25ba417a49L,0x02d7cf1cc617957aL,
        0x79070ea5565c1fbbL } },
    /* 50 */
    { { 0x70194602d9b028faL,0x9c49969d9ff06760L,0xbf4add816ad27b42L,
        0x7d1f226d8651524eL },
      { 0xb0779b40eecd7724L,0xd356077265938707L,0xe3a61fe5d054b903L,
        0xd6f5a3433365136bL } },
    /* 51 */
    { { 0x25c87c76d2970fcfL,0x7c9f60a04d5546a8L,0x7dab072f8dd8bf8cL,
        0x3d10907ce8ff9f28L },
      { 0xb08d6d0e34bb2a29L,0x5dfd4907c3fcfdafL,0xe4a2d4b147123ba6L,
        0x6e9eef0b42de6d8dL } },
    /* 52 */
    { { 0x81255af5cbb55f9dL,0x579f27055328d39eL,0xa7bfc9173e5ae663L,
        0xe9b55d57a1246e42L },
      { 0x240ecd9475629188L,0x8748d297457bd3c0L,0x50e215ef373c361cL,
        0xaf9d8a8618c967b9L } },
    /* 53 */
    { { 0x79a041040a04143fL,0x03f7410fc700c616L,0xe8f2a3f291108ca6L,
        0xa26d67e8f5ac679aL },
      { 0xa15dbfebb83fbd9aL,0xf1aaebd23a0b5587L,0x639a97ddce0ead44L,
        0xf253b00c71d12ee0L } },
    /* 54 */
    { { 0x7baecf4c9e35e57cL,0x522e26a16786e3a5L,0x600b538b8af829a2L,
        0x19fa80b72c6de44aL },
      { 0xb52364f0aaf0ff52L,0x2e4bc21a6714587fL,0x401377a3c245967dL,
        0x65178766a23cf3ebL } },
    /* 55 */
    { { 0xc1c81838923ac000L,0x42021f02c4abc0eeL,0xcde3bc9a47132a20L,
        0x6f52a864c69f55fbL },
      { 0x0bdfd3e4df89ff6aL,0x244c943bc88bd74eL,0x649e0b532612998bL,
        0xce61ebc3d3413d4aL } },
    /* 56 */
    { { 0xe31629042cba5a90L,0xa72710aedb6c224eL,0x51831390d87e44dbL,
        0xa687dc9848fe2ef3L },
      { 0x857e985516a21ca9L,0xe3428d8ec9a7bc12L,0x16d3bcd012b044a2L,
        0xe6fa0c69e85f6704L } },
    /* 57 */
    { { 0xe4cca34b8fd42692L,0xc86d49a6e15f3acfL,0xbfe1f263a6b18392L,
        0x0664c933dcd266f6L },
      { 0x86738cf519399d88L,0x1cbcc8c3749ce6bcL,0x28171f7bc773b884L,
        0x306fc95701acf19eL } },
    /* 58 */
    { { 0x0da7a737afb6a419L,0x637fc26a195fbc40L,0x0fc8f8769c64e8e7L,
        0x2a68579b208c0626L },
      { 0x82e823108628abc3L,0xe4e09313ab23ae94L,0x66bf9adbe5155cf1L,
        0x17909f6ce8a2dd0cL } },
    /* 59 */
    { { 0x767c359643d7ad31L,0x7ba3a1aa49ccef62L,0x5261c3160242bf5aL,
        0x85f452199eb82dfbL },
      { 0x554cb38237b42e47L,0xc9771ec14cf66133L,0xde70617a153905a3L,
        0x2cab26fcbc61316dL } },
    /* 60 */
    { { 0x7dababbd75c10315L,0x9a8fbe88a48df64eL,0x2b076fe5e1b8f912L,
        0x1a530ce9ccbd50dcL },
      { 0x47361ab76647d225L,0xf84e73be4d636a15L,0xd58fcaaf5904a2faL,
        0x73747d4b38523a19L } },
    /* 61 */
    { { 0x6e6b0fb8b6864cc0L,0x5d8a0027ab3b623cL,0x5e6665389a1cfc9cL,
        0x816b19de521e4ff3L },
      { 0x56709ad00bc447f8L,0x1d46cb1c8f1464d7L,0x49cef820a949873dL,
        0x02804692d9d3e65fL } },
    /* 62 */
    { { 0x1ae0ea28ad8b5976L,0x4e9ad48e869458fbL,0xe9437ec996cfedf8L,
        0xa4f924a22afa74d9L },
      { 0xcb5b1845aaf797c0L,0xe5d6dd0eba6f557fL,0xa1496fe691dc2e7cL,
        0xad31edac8c179fc7L } },
    /* 63 */
    { { 0xf9c5e9de44b06ed7L,0x6ce7c4f74a597159L,0xd02ec441833accb5L,
        0xf30205996296e8fcL },
      { 0x7df6c5c6c2afbe06L,0xff429dda9c849b09L,0x42170166f5dd78d6L,
        0x2403ea21830c388bL } },
};

/* Multiply the base point of P256 by the scalar and return the result.
 * If map is true then convert result to affine coordinates.
 *
 * Stripe implementation.
 * Pre-generated: 2^0, 2^42, ...
 * Pre-generated: products of all combinations of above.
 * 6 doubles and adds (with qz=1)
 *
 * r     Resulting point.
 * k     Scalar to multiply by.
 * map   Indicates whether to convert result to affine.
 * ct    Constant time required.
 * heap  Heap to use for allocation.
 * returns MEMORY_E when memory allocation fails and MP_OKAY on success.
 */
static int sp_256_ecc_mulmod_base_4(sp_point_256* r, const sp_digit* k,
        int map, int ct, void* heap)
{
    return sp_256_ecc_mulmod_stripe_4(r, &p256_base, p256_table,
                                      k, map, ct, heap);
}

#else
/* Striping precomputation table.
 * 8 points combined into a table of 256 points.
 * Distance of 32 between points.
 */
static const sp_table_entry_256 p256_table[256] = {
    /* 0 */
    { { 0x00, 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x00, 0x00 } },
    /* 1 */
    { { 0x79e730d418a9143cL,0x75ba95fc5fedb601L,0x79fb732b77622510L,
        0x18905f76a53755c6L },
      { 0xddf25357ce95560aL,0x8b4ab8e4ba19e45cL,0xd2e88688dd21f325L,
        0x8571ff1825885d85L } },
    /* 2 */
    { { 0x202886024147519aL,0xd0981eac26b372f0L,0xa9d4a7caa785ebc8L,
        0xd953c50ddbdf58e9L },
      { 0x9d6361ccfd590f8fL,0x72e9626b44e6c917L,0x7fd9611022eb64cfL,
        0x863ebb7e9eb288f3L } },
    /* 3 */
    { { 0x7856b6235cdb6485L,0x808f0ea22f0a2f97L,0x3e68d9544f7e300bL,
        0x00076055b5ff80a0L },
      { 0x7634eb9b838d2010L,0x54014fbb3243708aL,0xe0e47d39842a6606L,
        0x8308776134373ee0L } },
    /* 4 */
    { { 0x4f922fc516a0d2bbL,0x0d5cc16c1a623499L,0x9241cf3a57c62c8bL,
        0x2f5e6961fd1b667fL },
      { 0x5c15c70bf5a01797L,0x3d20b44d60956192L,0x04911b37071fdb52L,
        0xf648f9168d6f0f7bL } },
    /* 5 */
    { { 0x9e566847e137bbbcL,0xe434469e8a6a0becL,0xb1c4276179d73463L,
        0x5abe0285133d0015L },
      { 0x92aa837cc04c7dabL,0x573d9f4c43260c07L,0x0c93156278e6cc37L,
        0x94bb725b6b6f7383L } },
    /* 6 */
    { { 0xbbf9b48f720f141cL,0x6199b3cd2df5bc74L,0xdc3f6129411045c4L,
        0xcdd6bbcb2f7dc4efL },
      { 0xcca6700beaf436fdL,0x6f647f6db99326beL,0x0c0fa792014f2522L,
        0xa361bebd4bdae5f6L } },
    /* 7 */
    { { 0x28aa2558597c13c7L,0xc38d635f50b7c3e1L,0x07039aecf3c09d1dL,
        0xba12ca09c4b5292cL },
      { 0x9e408fa459f91dfdL,0x3af43b66ceea07fbL,0x1eceb0899d780b29L,
        0x53ebb99d701fef4bL } },
    /* 8 */
    { { 0x4fe7ee31b0e63d34L,0xf4600572a9e54fabL,0xc0493334d5e7b5a4L,
        0x8589fb9206d54831L },
      { 0xaa70f5cc6583553aL,0x0879094ae25649e5L,0xcc90450710044652L,
        0xebb0696d02541c4fL } },
    /* 9 */
    { { 0x4616ca15ac1647c5L,0xb8127d47c4cf5799L,0xdc666aa3764dfbacL,
        0xeb2820cbd1b27da3L },
      { 0x9406f8d86a87e008L,0xd87dfa9d922378f3L,0x56ed2e4280ccecb2L,
        0x1f28289b55a7da1dL } },
    /* 10 */
    { { 0xabbaa0c03b89da99L,0xa6f2d79eb8284022L,0x27847862b81c05e8L,
        0x337a4b5905e54d63L },
      { 0x3c67500d21f7794aL,0x207005b77d6d7f61L,0x0a5a378104cfd6e8L,
        0x0d65e0d5f4c2fbd6L } },
    /* 11 */
    { { 0xd9d09bbeb5275d38L,0x4268a7450be0a358L,0xf0762ff4973eb265L,
        0xc23da24252f4a232L },
      { 0x5da1b84f0b94520cL,0x09666763b05bd78eL,0x3a4dcb8694d29ea1L,
        0x19de3b8cc790cff1L } },
    /* 12 */
    { { 0x183a716c26c5fe04L,0x3b28de0b3bba1bdbL,0x7432c586a4cb712cL,
        0xe34dcbd491fccbfdL },
      { 0xb408d46baaa58403L,0x9a69748682e97a53L,0x9e39012736aaa8afL,
        0xe7641f447b4e0f7fL } },
    /* 13 */
    { { 0x7d753941df64ba59L,0xd33f10ec0b0242fcL,0x4f06dfc6a1581859L,
        0x4a12df57052a57bfL },
      { 0xbfa6338f9439dbd0L,0xd3c24bd4bde53e1fL,0xfd5e4ffa21f1b314L,
        0x6af5aa93bb5bea46L } },
    /* 14 */
    { { 0xda10b69910c91999L,0x0a24b4402a580491L,0x3e0094b4b8cc2090L,
        0x5fe3475a66a44013L },
      { 0xb0f8cabdf93e7b4bL,0x292b501a7c23f91aL,0x42e889aecd1e6263L,
        0xb544e308ecfea916L } },
    /* 15 */
    { { 0x6478c6e916ddfdceL,0x2c329166f89179e6L,0x4e8d6e764d4e67e1L,
        0xe0b6b2bda6b0c20bL },
      { 0x0d312df2bb7efb57L,0x1aac0dde790c4007L,0xf90336ad679bc944L,
        0x71c023de25a63774L } },
    /* 16 */
    { { 0x62a8c244bfe20925L,0x91c19ac38fdce867L,0x5a96a5d5dd387063L,
        0x61d587d421d324f6L },
      { 0xe87673a2a37173eaL,0x2384800853778b65L,0x10f8441e05bab43eL,
        0xfa11fe124621efbeL } },
    /* 17 */
    { { 0x1c891f2b2cb19ffdL,0x01ba8d5bb1923c23L,0xb6d03d678ac5ca8eL,
        0x586eb04c1f13bedcL },
      { 0x0c35c6e527e8ed09L,0x1e81a33c1819ede2L,0x278fd6c056c652faL,
        0x19d5ac0870864f11L } },
    /* 18 */
    { { 0x1e99f581309a4e1fL,0xab7de71be9270074L,0x26a5ef0befd28d20L,
        0xe7c0073f7f9c563fL },
      { 0x1f6d663a0ef59f76L,0x669b3b5420fcb050L,0xc08c1f7a7a6602d4L,
        0xe08504fec65b3c0aL } },
    /* 19 */
    { { 0xf098f68da031b3caL,0x6d1cab9ee6da6d66L,0x5bfd81fa94f246e8L,
        0x78f018825b0996b4L },
      { 0xb7eefde43a25787fL,0x8016f80d1dccac9bL,0x0cea4877b35bfc36L,
        0x43a773b87e94747aL } },
    /* 20 */
    { { 0x62577734d2b533d5L,0x673b8af6a1bdddc0L,0x577e7c9aa79ec293L,
        0xbb6de651c3b266b1L },
      { 0xe7e9303ab65259b3L,0xd6a0afd3d03a7480L,0xc5ac83d19b3cfc27L,
        0x60b4619a5d18b99bL } },
    /* 21 */
    { { 0xbd6a38e11ae5aa1cL,0xb8b7652b49e73658L,0x0b130014ee5f87edL,
        0x9d0f27b2aeebffcdL },
      { 0xca9246317a730a55L,0x9c955b2fddbbc83aL,0x07c1dfe0ac019a71L,
        0x244a566d356ec48dL } },
    /* 22 */
    { { 0x6db0394aeacf1f96L,0x9f2122a9024c271cL,0x2626ac1b82cbd3b9L,
        0x45e58c873581ef69L },
      { 0xd3ff479da38f9dbcL,0xa8aaf146e888a040L,0x945adfb246e0bed7L,
        0xc040e21cc1e4b7a4L } },
    /* 23 */
    { { 0x847af0006f8117b6L,0x651969ff73a35433L,0x482b35761d9475ebL,
        0x1cdf5c97682c6ec7L },
      { 0x7db775b411f04839L,0x7dbeacf448de1698L,0xb2921dd1b70b3219L,
        0x046755f8a92dff3dL } },
    /* 24 */
    { { 0xcc8ac5d2bce8ffcdL,0x0d53c48b2fe61a82L,0xf6f161727202d6c7L,
        0x046e5e113b83a5f3L },
      { 0xe7b8ff64d8007f01L,0x7fb1ef125af43183L,0x045c5ea635e1a03cL,
        0x6e0106c3303d005bL } },
    /* 25 */
    { { 0x48c7358488dd73b1L,0x7670708f995ed0d9L,0x38385ea8c56a2ab7L,
        0x442594ede901cf1fL },
      { 0xf8faa2c912d4b65bL,0x94c2343b96c90c37L,0xd326e4a15e978d1fL,
        0xa796fa514c2ee68eL } },
    /* 26 */
    { { 0x359fb604823addd7L,0x9e2a6183e56693b3L,0xf885b78e3cbf3c80L,
        0xe4ad2da9c69766e9L },
      { 0x357f7f428e048a61L,0x082d198cc092d9a0L,0xfc3a1af4c03ed8efL,
        0xc5e94046c37b5143L } },
    /* 27 */
    { { 0x476a538c2be75f9eL,0x6fd1a9e8cb123a78L,0xd85e4df0b109c04bL,
        0x63283dafdb464747L },
      { 0xce728cf7baf2df15L,0xe592c4550ad9a7f4L,0xfab226ade834bcc3L,
        0x68bd19ab1981a938L } },
    /* 28 */
    { { 0xc08ead511887d659L,0x3374d5f4b359305aL,0x96986981cfe74fe3L,
        0x495292f53c6fdfd6L },
      { 0x4a878c9e1acec896L,0xd964b210ec5b4484L,0x6696f7e2664d60a7L,
        0x0ec7530d26036837L } },
    /* 29 */
    { { 0x2da13a05ad2687bbL,0xa1f83b6af32e21faL,0x390f5ef51dd4607bL,
        0x0f6207a664863f0bL },
      { 0xbd67e3bb0f138233L,0xdd66b96c272aa718L,0x8ed0040726ec88aeL,
        0xff0db07208ed6dcfL } },
    /* 30 */
    { { 0x749fa1014c95d553L,0xa44052fd5d680a8aL,0x183b4317ff3b566fL,
        0x313b513c88740ea3L },
      { 0xb402e2ac08d11549L,0x071ee10bb4dee21cL,0x26b987dd47f2320eL,
        0x2d3abcf986f19f81L } },
    /* 31 */
    { { 0x4c288501815581a2L,0x9a0a6d56632211afL,0x19ba7a0f0cab2e99L,
        0xc036fa10ded98cdfL },
      { 0x29ae08bac1fbd009L,0x0b68b19006d15816L,0xc2eb32779b9e0d8fL,
        0xa6b2a2c4b6d40194L } },
    /* 32 */
    { { 0xd433e50f6d3549cfL,0x6f33696ffacd665eL,0x695bfdacce11fcb4L,
        0x810ee252af7c9860L },
      { 0x65450fe17159bb2cL,0xf7dfbebe758b357bL,0x2b057e74d69fea72L,
        0xd485717a92731745L } },
    /* 33 */
    { { 0x11741a8af0cb5a98L,0xd3da8f931f3110bfL,0x1994e2cbab382adfL,
        0x6a6045a72f9a604eL },
      { 0x170c0d3fa2b2411dL,0xbe0eb83e510e96e0L,0x3bcc9f738865b3ccL,
        0xd3e45cfaf9e15790L } },
    /* 34 */
    { { 0xce1f69bbe83f7669L,0x09f8ae8272877d6bL,0x9548ae543244278dL,
        0x207755dee3c2c19cL },
      { 0x87bd61d96fef1945L,0x18813cefb12d28c3L,0x9fbcd1d672df64aaL,
        0x48dc5ee57154b00dL } },
    /* 35 */
    { { 0x123790bff7e5a199L,0xe0efb8cf989ccbb7L,0xc27a2bfe0a519c79L,
        0xf2fb0aeddff6f445L },
      { 0x41c09575f0b5025fL,0x550543d740fa9f22L,0x8fa3c8ad380bfbd0L,
        0xa13e9015db28d525L } },
    /* 36 */
    { { 0xf9f7a350a2b65cbcL,0x0b04b9722a464226L,0x265ce241e23f07a1L,
        0x2bf0d6b01497526fL },
      { 0xd3d4dd3f4b216fb7L,0xf7d7b867fbdda26aL,0xaeb7b83f6708505cL,
        0x42a94a5a162fe89fL } },
    /* 37 */
    { { 0x5846ad0beaadf191L,0x0f8a489025a268d7L,0xe8603050494dc1f6L,
        0x2c2dd969c65ede3dL },
      { 0x6d02171d93849c17L,0x460488ba1da250ddL,0x4810c7063c3a5485L,
        0xf437fa1f42c56dbcL } },
    /* 38 */
    { { 0x6aa0d7144a0f7dabL,0x0f0497931776e9acL,0x52c0a050f5f39786L,
        0xaaf45b3354707aa8L },
      { 0x85e37c33c18d364aL,0xd40b9b063e497165L,0xf417168115ec5444L,
        0xcdf6310df4f272bcL } },
    /* 39 */
    { { 0x7473c6238ea8b7efL,0x08e9351885bc2287L,0x419567722bda8e34L,
        0xf0d008bada9e2ff2L },
      { 0x2912671d2414d3b1L,0xb3754985b019ea76L,0x5c61b96d453bcbdbL,
        0x5bd5c2f5ca887b8bL } },
    /* 40 */
    { { 0xef0f469ef49a3154L,0x3e85a5956e2b2e9aL,0x45aaec1eaa924a9cL,
        0xaa12dfc8a09e4719L },
      { 0x26f272274df69f1dL,0xe0e4c82ca2ff5e73L,0xb9d8ce73b7a9dd44L,
        0x6c036e73e48ca901L } },
    /* 41 */
    { { 0x5cfae12a0f6e3138L,0x6966ef0025ad345aL,0x8993c64b45672bc5L,
        0x292ff65896afbe24L },
      { 0xd5250d445e213402L,0xf6580e274392c9feL,0x097b397fda1c72e8L,
        0x644e0c90311b7276L } },
    /* 42 */
    { { 0xe1e421e1a47153f0L,0xb86c3b79920418c9L,0x93bdce87705d7672L,
        0xf25ae793cab79a77L },
      { 0x1f3194a36d869d0cL,0x9d55c8824986c264L,0x49fb5ea3096e945eL,
        0x39b8e65313db0a3eL } },
    /* 43 */
    { { 0x37754200b6fd2e59L,0x35e2c0669255c98fL,0xd9dab21a0e2a5739L,
        0x39122f2f0f19db06L },
      { 0xcfbce1e003cad53cL,0x225b2c0fe65c17e3L,0x72baf1d29aa13877L,
        0x8de80af8ce80ff8dL } },
    /* 44 */
    { { 0xafbea8d9207bbb76L,0x921c7e7c21782758L,0xdfa2b74b1c0436b1L,
        0x871949062e368c04L },
      { 0xb5f928bba3993df5L,0x639d75b5f3b3d26aL,0x011aa78a85b55050L,
        0xfc315e6a5b74fde1L } },
    /* 45 */
    { { 0x561fd41ae8d6ecfaL,0x5f8c44f61aec7f86L,0x98452a7b4924741dL,
        0xe6d4a7adee389088L },
      { 0x60552ed14593c75dL,0x70a70da4dd271162L,0xd2aede937ba2c7dbL,
        0x35dfaf9a9be2ae57L } },
    /* 46 */
    { { 0x6b956fcdaa736636L,0x09f51d97ae2cab7eL,0xfb10bf410f349966L,
        0x1da5c7d71c830d2bL },
      { 0x5c41e4833cce6825L,0x15ad118ff9573c3bL,0xa28552c7f23036b8L,
        0x7077c0fddbf4b9d6L } },
    /* 47 */
    { { 0xbf63ff8d46b9661cL,0xa1dfd36b0d2cfd71L,0x0373e140a847f8f7L,
        0x53a8632ee50efe44L },
      { 0x0976ff68696d8051L,0xdaec0c95c74f468aL,0x62994dc35e4e26bdL,
        0x028ca76d34e1fcc1L } },
    /* 48 */
    { { 0xd11d47dcfc9877eeL,0xc8b36210801d0002L,0xd002c11754c260b6L,
        0x04c17cd86962f046L },
      { 0x6d9bd094b0daddf5L,0xbea2357524ce55c0L,0x663356e672da03b5L,
        0xf7ba4de9fed97474L } },
    /* 49 */
    { { 0xd0dbfa34ebe1263fL,0x5576373571ae7ce6L,0xd244055382a6f523L,
        0xe31f960052131c41L },
      { 0xd1bb9216ea6b6ec6L,0x37a1d12e73c2fc44L,0xc10e7eac89d0a294L,
        0xaa3a6259ce34d47bL } },
    /* 50 */
    { { 0xfbcf9df536f3dcd3L,0x6ceded50d2bf7360L,0x491710fadf504f5bL,
        0x2398dd627e79daeeL },
      { 0xcf4705a36d09569eL,0xea0619bb5149f769L,0xff9c037735f6034cL,
        0x5717f5b21c046210L } },
    /* 51 */
    { { 0x9fe229c921dd895eL,0x8e51850040c28451L,0xfa13d2391d637ecdL,
        0x660a2c560e3c28deL },
      { 0x9cca88aed67fcbd0L,0xc84724780ea9f096L,0x32b2f48172e92b4dL,
        0x624ee54c4f522453L } },
    /* 52 */
    { { 0x09549ce4d897ecccL,0x4d49d1d93f9880aaL,0x723c2423043a7c20L,
        0x4f392afb92bdfbc0L },
      { 0x6969f8fa7de44fd9L,0xb66cfbe457b32156L,0xdb2fa803368ebc3cL,
        0x8a3e7977ccdb399cL } },
    /* 53 */
    { { 0xdde1881f06c4b125L,0xae34e300f6e3ca8cL,0xef6999de5c7a13e9L,
        0x3888d02370c24404L },
      { 0x7628035644f91081L,0x3d9fcf615f015504L,0x1827edc8632cd36eL,
        0xa5e62e4718102336L } },
    /* 54 */
    { { 0x1a825ee32facd6c8L,0x699c635454bcbc66L,0x0ce3edf798df9931L,
        0x2c4768e6466a5adcL },
      { 0xb346ff8c90a64bc9L,0x630a6020e4779f5cL,0xd949d064bc05e884L,
        0x7b5e6441f9e652a0L } },
    /* 55 */
    { { 0x2169422c1d28444aL,0xe996c5d8be136a39L,0x2387afe5fb0c7fceL,
        0xb8af73cb0c8d744aL },
      { 0x5fde83aa338b86fdL,0xfee3f158a58a5cffL,0xc9ee8f6f20ac9433L,
        0xa036395f7f3f0895L } },
    /* 56 */
    { { 0x8c73c6bba10f7770L,0xa6f16d81a12a0e24L,0x100df68251bc2b9fL,
        0x4be36b01875fb533L },
      { 0x9226086e9fb56dbbL,0x306fef8b07e7a4f8L,0xeeaccc0566d52f20L,
        0x8cbc9a871bdc00c0L } },
    /* 57 */
    { { 0xe131895cc0dac4abL,0xa874a440712ff112L,0x6332ae7c6a1cee57L,
        0x44e7553e0c0835f8L },
      { 0x6d503fff7734002dL,0x9d35cb8b0b34425cL,0x95f702760e8738b5L,
        0x470a683a5eb8fc18L } },
    /* 58 */
    { { 0x81b761dc90513482L,0x0287202a01e9276aL,0xcda441ee0ce73083L,
        0x16410690c63dc6efL },
      { 0xf5034a066d06a2edL,0xdd4d7745189b100bL,0xd914ae72ab8218c9L,
        0xd73479fd7abcbb4fL } },
    /* 59 */
    { { 0x7edefb165ad4c6e5L,0x262cf08f5b06d04dL,0x12ed5bb18575cb14L,
        0x816469e30771666bL },
      { 0xd7ab9d79561e291eL,0xeb9daf22c1de1661L,0xf49827eb135e0513L,
        0x0a36dd23f0dd3f9cL } },
    /* 60 */
    { { 0x098d32c741d5533cL,0x7c5f5a9e8684628fL,0x39a228ade349bd11L,
        0xe331dfd6fdbab118L },
      { 0x5100ab686bcc6ed8L,0x7160c3bdef7a260eL,0x9063d9a7bce850d7L,
        0xd3b4782a492e3389L } },
    /* 61 */
    { { 0xa149b6e8f3821f90L,0x92edd9ed66eb7aadL,0x0bb669531a013116L,
        0x7281275a4c86a5bdL },
      { 0x503858f7d3ff47e5L,0x5e1616bc61016441L,0x62b0f11a7dfd9bb1L,
        0x2c062e7ece145059L } },
    /* 62 */
    { { 0xa76f996f0159ac2eL,0x281e7736cbdb2713L,0x2ad6d28808e46047L,
        0x282a35f92c4e7ef1L },
      { 0x9c354b1ec0ce5cd2L,0xcf99efc91379c229L,0x992caf383e82c11eL,
        0xc71cd513554d2abdL } },
    /* 63 */
    { { 0x4885de9c09b578f4L,0x1884e258e3affa7aL,0x8f76b1b759182f1fL,
        0xc50f6740cf47f3a3L },
      { 0xa9c4adf3374b68eaL,0xa406f32369965fe2L,0x2f86a22285a53050L,
        0xb9ecb3a7212958dcL } },
    /* 64 */
    { { 0x56f8410ef4f8b16aL,0x97241afec47b266aL,0x0a406b8e6d9c87c1L,
        0x803f3e02cd42ab1bL },
      { 0x7f0309a804dbec69L,0xa83b85f73bbad05fL,0xc6097273ad8e197fL,
        0xc097440e5067adc1L } },
    /* 65 */
    { { 0x846a56f2c379ab34L,0xa8ee068b841df8d1L,0x20314459176c68efL,
        0xf1af32d5915f1f30L },
      { 0x99c375315d75bd50L,0x837cffbaf72f67bcL,0x0613a41848d7723fL,
        0x23d0f130e2d41c8bL } },
    /* 66 */
    { { 0x857ab6edf41500d9L,0x0d890ae5fcbeada8L,0x52fe864889725951L,
        0xb0288dd6c0a3faddL },
      { 0x85320f30650bcb08L,0x71af6313695d6e16L,0x31f520a7b989aa76L,
        0xffd3724ff408c8d2L } },
    /* 67 */
    { { 0x53968e64b458e6cbL,0x992dad20317a5d28L,0x3814ae0b7aa75f56L,
        0xf5590f4ad78c26dfL },
      { 0x0fc24bd3cf0ba55aL,0x0fc4724a0c778baeL,0x1ce9864f683b674aL,
        0x18d6da54f6f74a20L } },
    /* 68 */
    { { 0xed93e225d5be5a2bL,0x6fe799835934f3c6L,0x4314092622626ffcL,
        0x50bbb4d97990216aL },
      { 0x378191c6e57ec63eL,0x65422c40181dcdb2L,0x41a8099b0236e0f6L,
        0x2b10011801fe49c3L } },
    /* 69 */
    { { 0xfc68b5c59b391593L,0xc385f5a2598270fcL,0x7144f3aad19adcbbL,
        0xdd55899983fbae0cL },
      { 0x93b88b8e74b82ff4L,0xd2e03c4071e734c9L,0x9a7a9eaf43c0322aL,
        0xe6e4c551149d6041L } },
    /* 70 */
    { { 0x55f655bb1e9af288L,0x647e1a64f7ada931L,0x43697e4bcb2820e5L,
        0x51e00db107ed56ffL },
      { 0x43d169b8771c327eL,0x29cdb20b4a96c2adL,0xc07d51f53deb4779L,
        0xe22f424149829177L } },
    /* 71 */
    { { 0xcd45e8f4635f1abbL,0x7edc0cb568538874L,0xc9472c1fb5a8034dL,
        0xf709373d52dc48c9L },
      { 0x401966bba8af30d6L,0x95bf5f4af137b69cL,0x3966162a9361c47eL,
        0xbd52d288e7275b11L } },
    /* 72 */
    { { 0xab155c7a9c5fa877L,0x17dad6727d3a3d48L,0x43f43f9e73d189d8L,
        0xa0d0f8e4c8aa77a6L },
      { 0x0bbeafd8cc94f92dL,0xd818c8be0c4ddb3aL,0x22cc65f8b82eba14L,
        0xa56c78c7946d6a00L } },
    /* 73 */
    { { 0x2962391b0dd09529L,0x803e0ea63daddfcfL,0x2c77351f5b5bf481L,
        0xd8befdf8731a367aL },
      { 0xab919d42fc0157f4L,0xf51caed7fec8e650L,0xcdf9cb4002d48b0aL,
        0x854a68a5ce9f6478L } },
    /* 74 */
    { { 0xdc35f67b63506ea5L,0x9286c489a4fe0d66L,0x3f101d3bfe95cd4dL,
        0x5cacea0b98846a95L },
      { 0xa90df60c9ceac44dL,0x3db29af4354d1c3aL,0x08dd3de8ad5dbabeL,
        0xe4982d1235e4efa9L } },
    /* 75 */
    { { 0x23104a22c34cd55eL,0x58695bb32680d132L,0xfb345afa1fa1d943L,
        0x8046b7f616b20499L },
      { 0xb533581e38e7d098L,0xd7f61e8df46f0b70L,0x30dea9ea44cb78c4L,
        0xeb17ca7b9082af55L } },
    /* 76 */
    { { 0x1751b59876a145b9L,0xa5cf6b0fc1bc71ecL,0xd3e03565392715bbL,
        0x097b00bafab5e131L },
      { 0xaa66c8e9565f69e1L,0x77e8f75ab5be5199L,0x6033ba11da4fd984L,
        0xf95c747bafdbcc9eL } },
    /* 77 */
    { { 0x558f01d3bebae45eL,0xa8ebe9f0c4bc6955L,0xaeb705b1dbc64fc6L,
        0x3512601e566ed837L },
      { 0x9336f1e1fa1161cdL,0x328ab8d54c65ef87L,0x4757eee2724f21e5L,
        0x0ef971236068ab6bL } },
    /* 78 */
    { { 0x02598cf754ca4226L,0x5eede138f8642c8eL,0x48963f74468e1790L,
        0xfc16d9333b4fbc95L },
      { 0xbe96fb31e7c800caL,0x138063312678adaaL,0x3d6244976ff3e8b5L,
        0x14ca4af1b95d7a17L } },
    /* 79 */
    { { 0x7a4771babd2f81d5L,0x1a5f9d6901f7d196L,0xd898bef7cad9c907L,
        0x4057b063f59c231dL },
      { 0xbffd82fe89c05c0aL,0xe4911c6f1dc0df85L,0x3befccaea35a16dbL,
        0x1c3b5d64f1330b13L } },
    /* 80 */
    { { 0x5fe14bfe80ec21feL,0xf6ce116ac255be82L,0x98bc5a072f4a5d67L,
        0xfad27148db7e63afL },
      { 0x90c0b6ac29ab05b3L,0x37a9a83c4e251ae6L,0x0a7dc875c2aade7dL,
        0x77387de39f0e1a84L } },
    /* 81 */
    { { 0x1e9ecc49a56c0dd7L,0xa5cffcd846086c74L,0x8f7a1408f505aeceL,
        0xb37b85c0bef0c47eL },
      { 0x3596b6e4cc0e6a8fL,0xfd6d4bbf6b388f23L,0xaba453fac39cef4eL,
        0x9c135ac8f9f628d5L } },
    /* 82 */
    { { 0x32aa320284e35743L,0x320d6ab185a3cdefL,0xb821b1761df19819L,
        0x5721361fc433851fL },
      { 0x1f0db36a71fc9168L,0x5f98ba735e5c403cL,0xf64ca87e37bcd8f5L,
        0xdcbac3c9e6bb11bdL } },
    /* 83 */
    { { 0xf01d99684518cbe2L,0xd242fc189c9eb04eL,0x727663c7e47feebfL,
        0xb8c1c89e2d626862L },
      { 0x51a58bddc8e1d569L,0x563809c8b7d88cd0L,0x26c27fd9f11f31ebL,
        0x5d23bbda2f9422d4L } },
    /* 84 */
    { { 0x0a1c729495c8f8beL,0x2961c4803bf362bfL,0x9e418403df63d4acL,
        0xc109f9cb91ece900L },
      { 0xc2d095d058945705L,0xb9083d96ddeb85c0L,0x84692b8d7a40449bL,
        0x9bc3344f2eee1ee1L } },
    /* 85 */
    { { 0x0d5ae35642913074L,0x55491b2748a542b1L,0x469ca665b310732aL,
        0x29591d525f1a4cc1L },
      { 0xe76f5b6bb84f983fL,0xbe7eef419f5f84e1L,0x1200d49680baa189L,
        0x6376551f18ef332cL } },
    /* 86 */
    { { 0xbda5f14e562976ccL,0x22bca3e60ef12c38L,0xbbfa30646cca9852L,
        0xbdb79dc808e2987aL },
      { 0xfd2cb5c9cb06a772L,0x38f475aafe536dceL,0xc2a3e0227c2b5db8L,
        0x8ee86001add3c14aL } },
    /* 87 */
    { { 0xcbe96981a4ade873L,0x7ee9aa4dc4fba48cL,0x2cee28995a054ba5L,
        0x92e51d7a6f77aa4bL },
      { 0x948bafa87190a34dL,0xd698f75bf6bd1ed1L,0xd00ee6e30caf1144L,
        0x5182f86f0a56aaaaL } },
    /* 88 */
    { { 0xfba6212c7a4cc99cL,0xff609b683e6d9ca1L,0x5dbb27cb5ac98c5aL,
        0x91dcab5d4073a6f2L },
      { 0x01b6cc3d5f575a70L,0x0cb361396f8d87faL,0x165d4e8c89981736L,
        0x17a0cedb97974f2bL } },
    /* 89 */
    { { 0x38861e2a076c8d3aL,0x701aad39210f924bL,0x94d0eae413a835d9L,
        0x2e8ce36c7f4cdf41L },
      { 0x91273dab037a862bL,0x01ba9bb760e4c8faL,0xf964538833baf2ddL,
        0xf4ccc6cb34f668f3L } },
    /* 90 */
    { { 0x44ef525cf1f79687L,0x7c59549592efa815L,0xe1231741a5c78d29L,
        0xac0db4889a0df3c9L },
      { 0x86bfc711df01747fL,0x592b9358ef17df13L,0xe5880e4f5ccb6bb5L,
        0x95a64a6194c974a2L } },
    /* 91 */
    { { 0x72c1efdac15a4c93L,0x40269b7382585141L,0x6a8dfb1c16cb0badL,
        0x231e54ba29210677L },
      { 0xa70df9178ae6d2dcL,0x4d6aa63f39112918L,0xf627726b5e5b7223L,
        0xab0be032d8a731e1L } },
    /* 92 */
    { { 0x097ad0e98d131f2dL,0x637f09e33b04f101L,0x1ac86196d5e9a748L,
        0xf1bcc8802cf6a679L },
      { 0x25c69140e8daacb4L,0x3c4e405560f65009L,0x591cc8fc477937a6L,
        0x851694695aebb271L } },
    /* 93 */
    { { 0xde35c143f1dcf593L,0x78202b29b018be3bL,0xe9cdadc29bdd9d3dL,
        0x8f67d9d2daad55d8L },
      { 0x841116567481ea5fL,0xe7d2dde9e34c590cL,0xffdd43f405053fa8L,
        0xf84572b9c0728b5dL } },
    /* 94 */
    { { 0x5e1a7a7197af71c9L,0xa14494447a736565L,0xa1b4ae070e1d5063L,
        0xedee2710616b2c19L },
      { 0xb2f034f511734121L,0x1cac6e554a25e9f0L,0x8dc148f3a40c2ecfL,
        0x9fd27e9b44ebd7f4L } },
    /* 95 */
    { { 0x3cc7658af6e2cb16L,0xe3eb7d2cfe5919b6L,0x5a8c5816168d5583L,
        0xa40c2fb6958ff387L },
      { 0x8c9ec560fedcc158L,0x7ad804c655f23056L,0xd93967049a307e12L,
        0x99bc9bb87dc6decfL } },
    /* 96 */
    { { 0x84a9521d927dafc6L,0x52c1fb695c09cd19L,0x9d9581a0f9366ddeL,
        0x9abe210ba16d7e64L },
      { 0x480af84a48915220L,0xfa73176a4dd816c6L,0xc7d539871681ca5aL,
        0x7881c25787f344b0L } },
    /* 97 */
    { { 0x93399b51e0bcf3ffL,0x0d02cbc5127f74f6L,0x8fb465a2dd01d968L,
        0x15e6e319a30e8940L },
      { 0x646d6e0d3e0e05f4L,0xfad7bddc43588404L,0xbe61c7d1c4f850d3L,
        0x0e55facf191172ceL } },
    /* 98 */
    { { 0x7e9d9806f8787564L,0x1a33172131e85ce6L,0x6b0158cab819e8d6L,
        0xd73d09766fe96577L },
      { 0x424834251eb7206eL,0xa519290fc618bb42L,0x5dcbb8595e30a520L,
        0x9250a3748f15a50bL } },
    /* 99 */
    { { 0xcaff08f8be577410L,0xfd408a035077a8c6L,0xf1f63289ec0a63a4L,
        0x77414082c1cc8c0bL },
      { 0x05a40fa6eb0991cdL,0xc1ca086649fdc296L,0x3a68a3c7b324fd40L,
        0x8cb04f4d12eb20b9L } },
    /* 100 */
    { { 0xb1c2d0556906171cL,0x9073e9cdb0240c3fL,0xdb8e6b4fd8906841L,
        0xe4e429ef47123b51L },
      { 0x0b8dd53c38ec36f4L,0xf9d2dc01ff4b6a27L,0x5d066e07879a9a48L,
        0x37bca2ff3c6e6552L } },
    /* 101 */
    { { 0x4cd2e3c7df562470L,0x44f272a2c0964ac9L,0x7c6d5df980c793beL,
        0x59913edc3002b22aL },
      { 0x7a139a835750592aL,0x99e01d80e783de02L,0xcf8c0375ea05d64fL,
        0x43786e4ab013e226L } },
    /* 102 */
    { { 0xff32b0ed9e56b5a6L,0x0750d9a6d9fc68f9L,0xec15e845597846a7L,
        0x8638ca98b7e79e7aL },
      { 0x2f5ae0960afc24b2L,0x05398eaf4dace8f2L,0x3b765dd0aecba78fL,
        0x1ecdd36a7b3aa6f0L } },
    /* 103 */
    { { 0x5d3acd626c5ff2f3L,0xa2d516c02873a978L,0xad94c9fad2110d54L,
        0xd85d0f85d459f32dL },
      { 0x9f700b8d10b11da3L,0xd2c22c30a78318c4L,0x556988f49208decdL,
        0xa04f19c3b4ed3c62L } },
    /* 104 */
    { { 0x087924c8ed7f93bdL,0xcb64ac5d392f51f6L,0x7cae330a821b71afL,
        0x92b2eeea5c0950b0L },
      { 0x85ac4c9485b6e235L,0xab2ca4a92936c0f0L,0x80faa6b3e0508891L,
        0x1ee782215834276cL } },
    /* 105 */
    { { 0xa60a2e00e63e79f7L,0xf590e7b2f399d906L,0x9021054a6607c09dL,
        0xf3f2ced857a6e150L },
      { 0x200510f3f10d9b55L,0x9d2fcfacd8642648L,0xe5631aa7e8bd0e7cL,
        0x0f56a4543da3e210L } },
    /* 106 */
    { { 0x5b21bffa1043e0dfL,0x6c74b6cc9c007e6dL,0x1a656ec0d4a8517aL,
        0xbd8f17411969e263L },
      { 0x8a9bbb86beb7494aL,0x1567d46f45f3b838L,0xdf7a12a7a4e5a79aL,
        0x2d1a1c3530ccfa09L } },
    /* 107 */
    { { 0x192e3813506508daL,0x336180c4a1d795a7L,0xcddb59497a9944b3L,
        0xa107a65eb91fba46L },
      { 0xe6d1d1c50f94d639L,0x8b4af3758a58b7d7L,0x1a7c5584bd37ca1cL,
        0x183d760af87a9af2L } },
    /* 108 */
    { { 0x29d697110dde59a4L,0xf1ad8d070e8bef87L,0x229b49634f2ebe78L,
        0x1d44179dc269d754L },
      { 0xb32dc0cf8390d30eL,0x0a3b27530de8110cL,0x31af1dc52bc0339aL,
        0x771f9cc29606d262L } },
    /* 109 */
    { { 0x99993e7785040739L,0x44539db98026a939L,0xcf40f6f2f5f8fc26L,
        0x64427a310362718eL },
      { 0x4f4f2d8785428aa8L,0x7b7adc3febfb49a8L,0x201b2c6df23d01acL,
        0x49d9b7496ae90d6dL } },
    /* 110 */
    { { 0xcc78d8bc435d1099L,0x2adbcd4e8e8d1a08L,0x02c2e2a02cb68a41L,
        0x9037d81b3f605445L },
      { 0x7cdbac27074c7b61L,0xfe2031ab57bfd72eL,0x61ccec96596d5352L,
        0x08c3de6a7cc0639cL } },
    /* 111 */
    { { 0x20fdd020f6d552abL,0x56baff9805cd81f1L,0x06fb7c3e91351291L,
        0xc690944245796b2fL },
      { 0x17b3ae9c41231bd1L,0x1eac6e875cc58205L,0x208837abf9d6a122L,
        0x3fa3db02cafe3ac0L } },
    /* 112 */
    { { 0xd75a3e6505058880L,0x7da365ef643943f2L,0x4147861cfab24925L,
        0xc5c4bdb0fdb808ffL },
      { 0x73513e34b272b56bL,0xc8327e9511b9043aL,0xfd8ce37df8844969L,
        0x2d56db9446c2b6b5L } },
    /* 113 */
    { { 0x2461782fff46ac6bL,0xd19f792607a2e425L,0xfafea3c409a48de1L,
        0x0f56bd9de503ba42L },
      { 0x137d4ed1345cda49L,0x821158fc816f299dL,0xe7c6a54aaeb43402L,
        0x4003bb9d1173b5f1L } },
    /* 114 */
    { { 0x3b8e8189a0803387L,0xece115f539cbd404L,0x4297208dd2877f21L,
        0x53765522a07f2f9eL },
      { 0xa4980a21a8a4182dL,0xa2bbd07a3219df79L,0x674d0a2e1a19a2d4L,
        0x7a056f586c5d4549L } },
    /* 115 */
    { { 0x646b25589d8a2a47L,0x5b582948c3df2773L,0x51ec000eabf0d539L,
        0x77d482f17a1a2675L },
      { 0xb8a1bd9587853948L,0xa6f817bd6cfbffeeL,0xab6ec05780681e47L,
        0x4115012b2b38b0e4L } },
    /* 116 */
    { { 0x3c73f0f46de28cedL,0x1d5da7609b13ec47L,0x61b8ce9e6e5c6392L,
        0xcdf04572fbea0946L },
      { 0x1cb3c58b6c53c3b0L,0x97fe3c10447b843cL,0xfb2b8ae12cb9780eL,
        0xee703dda97383109L } },
    /* 117 */
    { { 0x34515140ff57e43aL,0xd44660d3b1b811b8L,0x2b3b5dff8f42b986L,
        0x2a0ad89da162ce21L },
      { 0x64e4a6946bc277baL,0xc788c954c141c276L,0x141aa64ccabf6274L,
        0xd62d0b67ac2b4659L } },
    /* 118 */
    { { 0x39c5d87b2c054ac4L,0x57005859f27df788L,0xedf7cbf3b18128d6L,
        0xb39a23f2991c2426L },
      { 0x95284a15f0b16ae5L,0x0c6a05b1a136f51bL,0x1d63c137f2700783L,
        0x04ed0092c0674cc5L } },
    /* 119 */
    { { 0x1f4185d19ae90393L,0x3047b4294a3d64e6L,0xae0001a69854fc14L,
        0xa0a91fc10177c387L },
      { 0xff0a3f01ae2c831eL,0xbb76ae822b727e16L,0x8f12c8a15a3075b4L,
        0x084cf9889ed20c41L } },
    /* 120 */
    { { 0xd98509defca6becfL,0x2fceae807dffb328L,0x5d8a15c44778e8b9L,
        0xd57955b273abf77eL },
      { 0x210da79e31b5d4f1L,0xaa52f04b3cfa7a1cL,0xd4d12089dc27c20bL,
        0x8e14ea4202d141f1L } },
    /* 121 */
    { { 0xeed50345f2897042L,0x8d05331f43402c4aL,0xc8d9c194c8bdfb21L,
        0x597e1a372aa4d158L },
      { 0x0327ec1acf0bd68cL,0x6d4be0dcab024945L,0x5b9c8d7ac9fe3e84L,
        0xca3f0236199b4deaL } },
    /* 122 */
    { { 0x592a10b56170bd20L,0x0ea897f16d3f5de7L,0xa3363ff144b2ade2L,
        0xbde7fd7e309c07e4L },
      { 0x516bb6d2b8f5432cL,0x210dc1cbe043444bL,0x3db01e6ff8f95b5aL,
        0xb623ad0e0a7dd198L } },
    /* 123 */
    { { 0xa75bd67560c7b65bL,0xab8c559023a4a289L,0xf8220fd0d7b26795L,
        0xd6aa2e4658ec137bL },
      { 0x10abc00b5138bb85L,0x8c31d121d833a95cL,0xb24ff00b1702a32eL,
        0x111662e02dcc513aL } },
    /* 124 */
    { { 0x78114015efb42b87L,0xbd9f5d701b6c4dffL,0x66ecccd7a7d7c129L,
        0xdb3ee1cb94b750f8L },
      { 0xb26f3db0f34837cfL,0xe7eed18bb9578d4fL,0x5d2cdf937c56657dL,
        0x886a644252206a59L } },
    /* 125 */
    { { 0x3c234cfb65b569eaL,0x20011141f72119c1L,0x8badc85da15a619eL,
        0xa70cf4eb018a17bcL },
      { 0x224f97ae8c4a6a65L,0x36e5cf270134378fL,0xbe3a609e4f7e0960L,
        0xaa4772abd1747b77L } },
    /* 126 */
    { { 0x676761317aa60cc0L,0xc79163610368115fL,0xded98bb4bbc1bb5aL,
        0x611a6ddc30faf974L },
      { 0x30e78cbcc15ee47aL,0x2e8962824e0d96a5L,0x36f35adf3dd9ed88L,
        0x5cfffaf816429c88L } },
    /* 127 */
    { { 0xc0d54cff9b7a99cdL,0x7bf3b99d843c45a1L,0x038a908f62c739e1L,
        0x6e5a6b237dc1994cL },
      { 0xef8b454e0ba5db77L,0xb7b8807facf60d63L,0xe591c0c676608378L,
        0x481a238d242dabccL } },
    /* 128 */
    { { 0xe3417bc035d0b34aL,0x440b386b8327c0a7L,0x8fb7262dac0362d1L,
        0x2c41114ce0cdf943L },
      { 0x2ba5cef1ad95a0b1L,0xc09b37a867d54362L,0x26d6cdd201e486c9L,
        0x20477abf42ff9297L } },
    /* 129 */
    { { 0x2f75173c18d65dbfL,0x77bf940e339edad8L,0x7022d26bdcf1001cL,
        0xac66409ac77396b6L },
      { 0x8b0bb36fc6261cc3L,0x213f7bc9190e7e90L,0x6541cebaa45e6c10L,
        0xce8e6975cc122f85L } },
    /* 130 */
    { { 0x0f121b41bc0a67d2L,0x62d4760a444d248aL,0x0e044f1d659b4737L,
        0x08fde365250bb4a8L },
      { 0xaceec3da848bf287L,0xc2a62182d3369d6eL,0x3582dfdc92449482L,
        0x2f7e2fd2565d6cd7L } },
    /* 131 */
    { { 0xae4b92dbc3770fa7L,0x095e8d5c379043f9L,0x54f34e9d17761171L,
        0xc65be92e907702aeL },
      { 0x2758a303f6fd0a40L,0xe7d822e3bcce784bL,0x7ae4f5854f9767bfL,
        0x4bff8e47d1193b3aL } },
    /* 132 */
    { { 0xcd41d21f00ff1480L,0x2ab8fb7d0754db16L,0xac81d2efbbe0f3eaL,
        0x3e4e4ae65772967dL },
      { 0x7e18f36d3c5303e6L,0x3bd9994b92262397L,0x9ed70e261324c3c0L,
        0x5388aefd58ec6028L } },
    /* 133 */
    { { 0xad1317eb5e5d7713L,0x09b985ee75de49daL,0x32f5bc4fc74fb261L,
        0x5cf908d14f75be0eL },
      { 0x760435108e657b12L,0xbfd421a5b96ed9e6L,0x0e29f51f8970ccc2L,
        0xa698ba4060f00ce2L } },
    /* 134 */
    { { 0x73db1686ef748fecL,0xe6e755a27e9d2cf9L,0x630b6544ce265effL,
        0xb142ef8a7aebad8dL },
      { 0xad31af9f17d5770aL,0x66af3b672cb3412fL,0x6bd60d1bdf3359deL,
        0xd1896a9658515075L } },
    /* 135 */
    { { 0xec5957ab33c41c08L,0x87de94ac5468e2e1L,0x18816b73ac472f6cL,
        0x267b0e0b7981da39L },
      { 0x6e554e5d8e62b988L,0xd8ddc755116d21e7L,0x4610faf03d2a6f99L,
        0xb54e287aa1119393L } },
    /* 136 */
    { { 0x0a0122b5178a876bL,0x51ff96ff085104b4L,0x050b31ab14f29f76L,
        0x84abb28b5f87d4e6L },
      { 0xd5ed439f8270790aL,0x2d6cb59d85e3f46bL,0x75f55c1b6c1e2212L,
        0xe5436f6717655640L } },
    /* 137 */
    { { 0x53f9025e2286e8d5L,0x353c95b4864453beL,0xd832f5bde408e3a0L,
        0x0404f68b5b9ce99eL },
      { 0xcad33bdea781e8e5L,0x3cdf5018163c2f5bL,0x575769600119caa3L,
        0x3a4263df0ac1c701L } },
    /* 138 */
    { { 0xc2965ecc9aeb596dL,0x01ea03e7023c92b4L,0x4704b4b62e013961L,
        0x0ca8fd3f905ea367L },
      { 0x92523a42551b2b61L,0x1eb7a89c390fcd06L,0xe7f1d2be0392a63eL,
        0x96dca2644ddb0c33L } },
    /* 139 */
    { { 0x203bb43a387510afL,0x846feaa8a9a36a01L,0xd23a57702f950378L,
        0x4363e2123aad59dcL },
      { 0xca43a1c740246a47L,0xb362b8d2e55dd24dL,0xf9b086045d8faf96L,
        0x840e115cd8bb98c4L } },
    /* 140 */
    { { 0xf12205e21023e8a7L,0xc808a8cdd8dc7a0bL,0xe292a272163a5ddfL,
        0x5e0d6abd30ded6d4L },
      { 0x07a721c27cfc0f64L,0x42eec01d0e55ed88L,0x26a7bef91d1f9db2L,
        0x7dea48f42945a25aL } },
    /* 141 */
    { { 0xabdf6f1ce5060a81L,0xe79f9c72f8f95615L,0xcfd36c5406ac268bL,
        0xabc2a2beebfd16d1L },
      { 0x8ac66f91d3e2eac7L,0x6f10ba63d2dd0466L,0x6790e3770282d31bL,
        0x4ea353946c7eefc1L } },
    /* 142 */
    { { 0xed8a2f8d5266309dL,0x0a51c6c081945a3eL,0xcecaf45a578c5dc1L,
        0x3a76e6891c94ffc3L },
      { 0x9aace8a47d7b0d0fL,0x963ace968f584a5fL,0x51a30c724e697fbeL,
        0x8212a10a465e6464L } },
    /* 143 */
    { { 0xef7c61c3cfab8caaL,0x18eb8e840e142390L,0xcd1dff677e9733caL,
        0xaa7cab71599cb164L },
      { 0x02fc9273bc837bd1L,0xc06407d0c36af5d7L,0x17621292f423da49L,
        0x40e38073fe0617c3L } },
    /* 144 */
    { { 0xf4f80824a7bf9b7cL,0x365d23203fbe30d0L,0xbfbe532097cf9ce3L,
        0xe3604700b3055526L },
      { 0x4dcb99116cc6c2c7L,0x72683708ba4cbee6L,0xdcded434637ad9ecL,
        0x6542d677a3dee15fL } },
    /* 145 */
    { { 0x3f32b6d07b6c377aL,0x6cb03847903448beL,0xd6fdd3a820da8af7L,
        0xa6534aee09bb6f21L },
      { 0x30a1780d1035facfL,0x35e55a339dcb47e6L,0x6ea50fe1c447f393L,
        0xf3cb672fdc9aef22L } },
    /* 146 */
    { { 0xeb3719fe3b55fd83L,0xe0d7a46c875ddd10L,0x33ac9fa905cea784L,
        0x7cafaa2eaae870e7L },
      { 0x9b814d041d53b338L,0xe0acc0a0ef87e6c6L,0xfb93d10811672b0fL,
        0x0aab13c1b9bd522eL } },
    /* 147 */
    { { 0xddcce278d2681297L,0xcb350eb1b509546aL,0x2dc431737661aaf2L,
        0x4b91a602847012e9L },
      { 0xdcff109572f8ddcfL,0x08ebf61e9a911af4L,0x48f4360ac372430eL,
        0x49534c5372321cabL } },
    /* 148 */
    { { 0x83df7d71f07b7e9dL,0xa478efa313cd516fL,0x78ef264b6c047ee3L,
        0xcaf46c4fd65ac5eeL },
      { 0xa04d0c7792aa8266L,0xedf45466913684bbL,0x56e65168ae4b16b0L,
        0x14ce9e5704c6770fL } },
    /* 149 */
    { { 0x99445e3e965e8f91L,0xd3aca1bacb0f2492L,0xd31cc70f90c8a0a0L,
        0x1bb708a53e4c9a71L },
      { 0xd5ca9e69558bdd7aL,0x734a0508018a26b1L,0xb093aa714c9cf1ecL,
        0xf9d126f2da300102L } },
    /* 150 */
    { { 0x749bca7aaff9563eL,0xdd077afeb49914a0L,0xe27a0311bf5f1671L,
        0x807afcb9729ecc69L },
      { 0x7f8a9337c9b08b77L,0x86c3a785443c7e38L,0x85fafa59476fd8baL,
        0x751adcd16568cd8cL } },
    /* 151 */
    { { 0x8aea38b410715c0dL,0xd113ea718f7697f7L,0x665eab1493fbf06dL,
        0x29ec44682537743fL },
      { 0x3d94719cb50bebbcL,0x399ee5bfe4505422L,0x90cd5b3a8d2dedb1L,
        0xff9370e392a4077dL } },
    /* 152 */
    { { 0x59a2d69bc6b75b65L,0x4188f8d5266651c5L,0x28a9f33e3de9d7d2L,
        0x9776478ba2a9d01aL },
      { 0x8852622d929af2c7L,0x334f5d6d4e690923L,0xce6cc7e5a89a51e9L,
        0x74a6313fac2f82faL } },
    /* 153 */
    { { 0xb2f4dfddb75f079cL,0x85b07c9518e36fbbL,0x1b6cfcf0e7cd36ddL,
        0xab75be150ff4863dL },
      { 0x81b367c0173fc9b7L,0xb90a7420d2594fd0L,0x15fdbf03c4091236L,
        0x4ebeac2e0b4459f6L } },
    /* 154 */
    { { 0xeb6c5fe75c9f2c53L,0xd25220118eae9411L,0xc8887633f95ac5d8L,
        0xdf99887b2c1baffcL },
      { 0xbb78eed2850aaecbL,0x9d49181b01d6a272L,0x978dd511b1cdbcacL,
        0x27b040a7779f4058L } },
    /* 155 */
    { { 0x90405db7f73b2eb2L,0xe0df85088e1b2118L,0x501b71525962327eL,
        0xb393dd37e4cfa3f5L },
      { 0xa1230e7b3fd75165L,0xd66344c2bcd33554L,0x6c36f1be0f7b5022L,
        0x09588c12d0463419L } },
    /* 156 */
    { { 0xe086093f02601c3bL,0xfb0252f8cf5c335fL,0x955cf280894aff28L,
        0x81c879a9db9f648bL },
      { 0x040e687cc6f56c51L,0xfed471693f17618cL,0x44f88a419059353bL,
        0xfa0d48f55fc11bc4L } },
    /* 157 */
    { { 0xbc6e1c9de1608e4dL,0x010dda113582822cL,0xf6b7ddc1157ec2d7L,
        0x8ea0e156b6a367d6L },
      { 0xa354e02f2383b3b4L,0x69966b943f01f53cL,0x4ff6632b2de03ca5L,
        0x3f5ab924fa00b5acL } },
    /* 158 */
    { { 0x337bb0d959739efbL,0xc751b0f4e7ebec0dL,0x2da52dd6411a67d1L,
        0x8bc768872b74256eL },
      { 0xa5be3b7282d3d253L,0xa9f679a1f58d779fL,0xa1cac168e16767bbL,
        0xb386f19060fcf34fL } },
    /* 159 */
    { { 0x31f3c1352fedcfc2L,0x5396bf6262f8af0dL,0x9a02b4eae57288c2L,
        0x4cb460f71b069c4dL },
      { 0xae67b4d35b8095eaL,0x92bbf8596fc07603L,0xe1475f66b614a165L,
        0x52c0d50895ef5223L } },
    /* 160 */
    { { 0x231c210e15339848L,0xe87a28e870778c8dL,0x9d1de6616956e170L,
        0x4ac3c9382bb09c0bL },
      { 0x19be05516998987dL,0x8b2376c4ae09f4d6L,0x1de0b7651a3f933dL,
        0x380d94c7e39705f4L } },
    /* 161 */
    { { 0x01a355aa81542e75L,0x96c724a1ee01b9b7L,0x6b3a2977624d7087L,
        0x2ce3e171de2637afL },
      { 0xcfefeb49f5d5bc1aL,0xa655607e2777e2b5L,0x4feaac2f9513756cL,
        0x2e6cd8520b624e4dL } },
    /* 162 */
    { { 0x3685954b8c31c31dL,0x68533d005bf21a0cL,0x0bd7626e75c79ec9L,
        0xca17754742c69d54L },
      { 0xcc6edafff6d2dbb2L,0xfd0d8cbd174a9d18L,0x875e8793aa4578e8L,
        0xa976a7139cab2ce6L } },
    /* 163 */
    { { 0x0a651f1b93fb353dL,0xd75cab8b57fcfa72L,0xaa88cfa731b15281L,
        0x8720a7170a1f4999L },
      { 0x8c3e8d37693e1b90L,0xd345dc0b16f6dfc3L,0x8ea8d00ab52a8742L,
        0x9719ef29c769893cL } },
    /* 164 */
    { { 0x820eed8d58e35909L,0x9366d8dc33ddc116L,0xd7f999d06e205026L,
        0xa5072976e15704c1L },
      { 0x002a37eac4e70b2eL,0x84dcf6576890aa8aL,0xcd71bf18645b2a5cL,
        0x99389c9df7b77725L } },
    /* 165 */
    { { 0x238c08f27ada7a4bL,0x3abe9d03fd389366L,0x6b672e89766f512cL,
        0xa88806aa202c82e4L },
      { 0x6602044ad380184eL,0xa8cb78c4126a8b85L,0x79d670c0ad844f17L,
        0x0043bffb4738dcfeL } },
    /* 166 */
    { { 0x8d59b5dc36d5192eL,0xacf885d34590b2afL,0x83566d0a11601781L,
        0x52f3ef01ba6c4866L },
      { 0x3986732a0edcb64dL,0x0a482c238068379fL,0x16cbe5fa7040f309L,
        0x3296bd899ef27e75L } },
    /* 167 */
    { { 0x476aba89454d81d7L,0x9eade7ef51eb9b3cL,0x619a21cd81c57986L,
        0x3b90febfaee571e9L },
      { 0x9393023e5496f7cbL,0x55be41d87fb51bc4L,0x03f1dd4899beb5ceL,
        0x6e88069d9f810b18L } },
    /* 168 */
    { { 0xce37ab11b43ea1dbL,0x0a7ff1a95259d292L,0x851b02218f84f186L,
        0xa7222beadefaad13L },
      { 0xa2ac78ec2b0a9144L,0x5a024051f2fa59c5L,0x91d1eca56147ce38L,
        0xbe94d523bc2ac690L } },
    /* 169 */
    { { 0x72f4945e0b226ce7L,0xb8afd747967e8b70L,0xedea46f185a6c63eL,
        0x7782defe9be8c766L },
      { 0x760d2aa43db38626L,0x460ae78776f67ad1L,0x341b86fc54499cdbL,
        0x03838567a2892e4bL } },
    /* 170 */
    { { 0x2d8daefd79ec1a0fL,0x3bbcd6fdceb39c97L,0xf5575ffc58f61a95L,
        0xdbd986c4adf7b420L },
      { 0x81aa881415f39eb7L,0x6ee2fcf5b98d976cL,0x5465475dcf2f717dL,
        0x8e24d3c46860bbd0L } },
    /* 171 */
    { { 0x749d8e549a587390L,0x12bb194f0cbec588L,0x46e07da4b25983c6L,
        0x541a99c4407bafc8L },
      { 0xdb241692624c8842L,0x6044c12ad86c05ffL,0xc59d14b44f7fcf62L,
        0xc0092c49f57d35d1L } },
    /* 172 */
    { { 0xd3cc75c3df2e61efL,0x7e8841c82e1b35caL,0xc62d30d1909f29f4L,
        0x75e406347286944dL },
      { 0xe7d41fc5bbc237d0L,0xc9537bf0ec4f01c9L,0x91c51a16282bd534L,
        0x5b7cb658c7848586L } },
    /* 173 */
    { { 0x964a70848a28ead1L,0x802dc508fd3b47f6L,0x9ae4bfd1767e5b39L,
        0x7ae13eba8df097a1L },
      { 0xfd216ef8eadd384eL,0x0361a2d9b6b2ff06L,0x204b98784bcdb5f3L,
        0x787d8074e2a8e3fdL } },
    /* 174 */
    { { 0xc5e25d6b757fbb1cL,0xe47bddb2ca201debL,0x4a55e9a36d2233ffL,
        0x5c2228199ef28484L },
      { 0x773d4a8588315250L,0x21b21a2b827097c1L,0xab7c4ea1def5d33fL,
        0xe45d37abbaf0f2b0L } },
    /* 175 */
    { { 0xd2df1e3428511c8aL,0xebb229c8bdca6cd3L,0x578a71a7627c39a7L,
        0xed7bc12284dfb9d3L },
      { 0xcf22a6df93dea561L,0x5443f18dd48f0ed1L,0xd8b861405bad23e8L,
        0xaac97cc945ca6d27L } },
    /* 176 */
    { { 0xeb54ea74a16bd00aL,0xd839e9adf5c0bcc1L,0x092bb7f11f9bfc06L,
        0x318f97b31163dc4eL },
      { 0xecc0c5bec30d7138L,0x44e8df23abc30220L,0x2bb7972fb0223606L,
        0xfa41faa19a84ff4dL } },
    /* 177 */
    { { 0x4402d974a6642269L,0xc81814ce9bb783bdL,0x398d38e47941e60bL,
        0x38bb6b2c1d26e9e2L },
      { 0xc64e4a256a577f87L,0x8b52d253dc11fe1cL,0xff336abf62280728L,
        0x94dd0905ce7601a5L } },
    /* 178 */
    { { 0x156cf7dcde93f92aL,0xa01333cb89b5f315L,0x02404df9c995e750L,
        0x92077867d25c2ae9L },
      { 0xe2471e010bf39d44L,0x5f2c902096bb53d7L,0x4c44b7b35c9c3d8fL,
        0x81e8428bd29beb51L } },
    /* 179 */
    { { 0x6dd9c2bac477199fL,0x8cb8eeee6b5ecdd9L,0x8af7db3fee40fd0eL,
        0x1b94ab62dbbfa4b1L },
      { 0x44f0d8b3ce47f143L,0x51e623fc63f46163L,0xf18f270fcc599383L,
        0x06a38e28055590eeL } },
    /* 180 */
    { { 0x2e5b0139b3355b49L,0x20e26560b4ebf99bL,0xc08ffa6bd269f3dcL,
        0xa7b36c2083d9d4f8L },
      { 0x64d15c3a1b3e8830L,0xd5fceae1a89f9c0bL,0xcfeee4a2e2d16930L,
        0xbe54c6b4a2822a20L } },
    /* 181 */
    { { 0xd6cdb3df8d91167cL,0x517c3f79e7a6625eL,0x7105648f346ac7f4L,
        0xbf30a5abeae022bbL },
      { 0x8e7785be93828a68L,0x5161c3327f3ef036L,0xe11b5feb592146b2L,
        0xd1c820de2732d13aL } },
    /* 182 */
    { { 0x043e13479038b363L,0x58c11f546b05e519L,0x4fe57abe6026cad1L,
        0xb7d17bed68a18da3L },
      { 0x44ca5891e29c2559L,0x4f7a03765bfffd84L,0x498de4af74e46948L,
        0x3997fd5e6412cc64L } },
    /* 183 */
    { { 0xf20746828bd61507L,0x29e132d534a64d2aL,0xffeddfb08a8a15e3L,
        0x0eeb89293c6c13e8L },
      { 0xe9b69a3ea7e259f8L,0xce1db7e6d13e7e67L,0x277318f6ad1fa685L,
        0x228916f8c922b6efL } },
    /* 184 */
    { { 0x959ae25b0a12ab5bL,0xcc11171f957bc136L,0x8058429ed16e2b0cL,
        0xec05ad1d6e93097eL },
      { 0x157ba5beac3f3708L,0x31baf93530b59d77L,0x47b55237118234e5L,
        0x7d3141567ff11b37L } },
    /* 185 */
    { { 0x7bd9c05cf6dfefabL,0xbe2f2268dcb37707L,0xe53ead973a38bb95L,
        0xe9ce66fc9bc1d7a3L },
      { 0x75aa15766f6a02a1L,0x38c087df60e600edL,0xf8947f3468cdc1b9L,
        0xd9650b0172280651L } },
    /* 186 */
    { { 0x504b4c4a5a057e60L,0xcbccc3be8def25e4L,0xa635320817c1ccbdL,
        0x14d6699a804eb7a2L },
      { 0x2c8a8415db1f411aL,0x09fbaf0bf80d769cL,0xb4deef901c2f77adL,
        0x6f4c68410d43598aL } },
    /* 187 */
    { { 0x8726df4e96c24a96L,0x534dbc85fcbd99a3L,0x3c466ef28b2ae30aL,
        0x4c4350fd61189abbL },
      { 0x2967f716f855b8daL,0x41a42394463c38a1L,0xc37e1413eae93343L,
        0xa726d2425a3118b5L } },
    /* 188 */
    { { 0xdae6b3ee948c1086L,0xf1de503dcbd3a2e1L,0x3f35ed3f03d022f3L,
        0x13639e82cc6cf392L },
      { 0x9ac938fbcdafaa86L,0xf45bc5fb2654a258L,0x1963b26e45051329L,
        0xca9365e1c1a335a3L } },
    /* 189 */
    { { 0x3615ac754c3b2d20L,0x742a5417904e241bL,0xb08521c4cc9d071dL,
        0x9ce29c34970b72a5L },
      { 0x8cc81f736d3e0ad6L,0x8060da9ef2f8434cL,0x35ed1d1a6ce862d9L,
        0x48c4abd7ab42af98L } },
    /* 190 */
    { { 0xd221b0cc40c7485aL,0xead455bbe5274dbfL,0x493c76989263d2e8L,
        0x78017c32f67b33cbL },
      { 0xb9d35769930cb5eeL,0xc0d14e940c408ed2L,0xf8b7bf55272f1a4dL,
        0x53cd0454de5c1c04L } },
    /* 191 */
    { { 0xbcd585fa5d28ccacL,0x5f823e56005b746eL,0x7c79f0a1cd0123aaL,
        0xeea465c1d3d7fa8fL },
      { 0x7810659f0551803bL,0x6c0b599f7ce6af70L,0x4195a77029288e70L,
        0x1b6e42a47ae69193L } },
    /* 192 */
    { { 0x2e80937cf67d04c3L,0x1e312be289eeb811L,0x56b5d88792594d60L,
        0x0224da14187fbd3dL },
      { 0x87abb8630c5fe36fL,0x580f3c604ef51f5fL,0x964fb1bfb3b429ecL,
        0x60838ef042bfff33L } },
    /* 193 */
    { { 0x432cb2f27e0bbe99L,0x7bda44f304aa39eeL,0x5f497c7a9fa93903L,
        0x636eb2022d331643L },
      { 0xfcfd0e6193ae00aaL,0x875a00fe31ae6d2fL,0xf43658a29f93901cL,
        0x8844eeb639218bacL } },
    /* 194 */
    { { 0x114171d26b3bae58L,0x7db3df7117e39f3eL,0xcd37bc7f81a8eadaL,
        0x27ba83dc51fb789eL },
      { 0xa7df439ffbf54de5L,0x7277030bb5fe1a71L,0x42ee8e35db297a48L,
        0xadb62d3487f3a4abL } },
    /* 195 */
    { { 0x9b1168a2a175df2aL,0x082aa04f618c32e9L,0xc9e4f2e7146b0916L,
        0xb990fd7675e7c8b2L },
      { 0x0829d96b4df37313L,0x1c205579d0b40789L,0x66c9ae4a78087711L,
        0x81707ef94d10d18dL } },
    /* 196 */
    { { 0x97d7cab203d6ff96L,0x5b851bfc0d843360L,0x268823c4d042db4bL,
        0x3792daead5a8aa5cL },
      { 0x52818865941afa0bL,0xf3e9e74142d83671L,0x17c825275be4e0a7L,
        0x5abd635e94b001baL } },
    /* 197 */
    { { 0x727fa84e0ac4927cL,0xe3886035a7c8cf23L,0xa4bcd5ea4adca0dfL,
        0x5995bf21846ab610L },
      { 0xe90f860b829dfa33L,0xcaafe2ae958fc18bL,0x9b3baf4478630366L,
        0x44c32ca2d483411eL } },
    /* 198 */
    { { 0xa74a97f1e40ed80cL,0x5f938cb131d2ca82L,0x53f2124b7c2d6ad9L,
        0x1f2162fb8082a54cL },
      { 0x7e467cc5720b173eL,0x40e8a666085f12f9L,0x8cebc20e4c9d65dcL,
        0x8f1d402bc3e907c9L } },
    /* 199 */
    { { 0x4f592f9cfbc4058aL,0xb15e14b6292f5670L,0xc55cfe37bc1d8c57L,
        0xb1980f43926edbf9L },
      { 0x98c33e0932c76b09L,0x1df5279d33b07f78L,0x6f08ead4863bb461L,
        0x2828ad9b37448e45L } },
    /* 200 */
    { { 0x696722c4c4cf4ac5L,0xf5ac1a3fdde64afbL,0x0551baa2e0890832L,
        0x4973f1275a14b390L },
      { 0xe59d8335322eac5dL,0x5e07eef50bd9b568L,0xab36720fa2588393L,
        0x6dac8ed0db168ac7L } },
    /* 201 */
    { { 0xf7b545aeeda835efL,0x4aa113d21d10ed51L,0x035a65e013741b09L,
        0x4b23ef5920b9de4cL },
      { 0xe82bb6803c4c7341L,0xd457706d3f58bc37L,0x73527863a51e3ee8L,
        0x4dd71534ddf49a4eL } },
    /* 202 */
    { { 0xbf94467295476cd9L,0x648d072fe31a725bL,0x1441c8b8fc4b67e0L,
        0xfd3170002f4a4dbbL },
      { 0x1cb43ff48995d0e1L,0x76e695d10ef729aaL,0xe0d5f97641798982L,
        0x14fac58c9569f365L } },
    /* 203 */
    { { 0xad9a0065f312ae18L,0x51958dc0fcc93fc9L,0xd9a142408a7d2846L,
        0xed7c765136abda50L },
      { 0x46270f1a25d4abbcL,0x9b5dd8f3f1a113eaL,0xc609b0755b51952fL,
        0xfefcb7f74d2e9f53L } },
    /* 204 */
    { { 0xbd09497aba119185L,0xd54e8c30aac45ba4L,0x492479deaa521179L,
        0x1801a57e87e0d80bL },
      { 0x073d3f8dfcafffb0L,0x6cf33c0bae255240L,0x781d763b5b5fdfbcL,
        0x9f8fc11e1ead1064L } },
    /* 205 */
    { { 0x1583a1715e69544cL,0x0eaf8567f04b7813L,0x1e22a8fd278a4c32L,
        0xa9d3809d3d3a69a9L },
      { 0x936c2c2c59a2da3bL,0x38ccbcf61895c847L,0x5e65244e63d50869L,
        0x3006b9aee1178ef7L } },
    /* 206 */
    { { 0x0bb1f2b0c9eead28L,0x7eef635d89f4dfbcL,0x074757fdb2ce8939L,
        0x0ab85fd745f8f761L },
      { 0xecda7c933e5b4549L,0x4be2bb5c97922f21L,0x261a1274b43b8040L,
        0xb122d67511e942c2L } },
    /* 207 */
    { { 0x3be607be66a5ae7aL,0x01e703fa76adcbe3L,0xaf9043014eb6e5c5L,
        0x9f599dc1097dbaecL },
      { 0x6d75b7180ff250edL,0x8eb91574349a20dcL,0x425605a410b227a3L,
        0x7d5528e08a294b78L } },
    /* 208 */
    { { 0xf0f58f6620c26defL,0x025585ea582b2d1eL,0xfbe7d79b01ce3881L,
        0x28ccea01303f1730L },
      { 0xd1dabcd179644ba5L,0x1fc643e806fff0b8L,0xa60a76fc66b3e17bL,
        0xc18baf48a1d013bfL } },
    /* 209 */
    { { 0x34e638c85dc4216dL,0x00c01067206142acL,0xd453a17195f5064aL,
        0x9def809db7a9596bL },
      { 0x41e8642e67ab8d2cL,0xb42404336237a2b6L,0x7d506a6d64c4218bL,
        0x0357f8b068808ce5L } },
    /* 210 */
    { { 0x8e9dbe644cd2cc88L,0xcc61c28df0b8f39dL,0x4a309874cd30a0c8L,
        0xe4a01add1b489887L },
      { 0x2ed1eeacf57cd8f9L,0x1b767d3ebd594c48L,0xa7295c717bd2f787L,
        0x466d7d79ce10cc30L } },
    /* 211 */
    { { 0x47d318929dada2c7L,0x4fa0a6c38f9aa27dL,0x90e4fd28820a59e1L,
        0xc672a522451ead1aL },
      { 0x30607cc85d86b655L,0xf0235d3bf9ad4af1L,0x99a08680571172a6L,
        0x5e3d64faf2a67513L } },
    /* 212 */
    { { 0xaa6410c79b3b4416L,0xcd8fcf85eab26d99L,0x5ebff74adb656a74L,
        0x6c8a7a95eb8e42fcL },
      { 0x10c60ba7b02a63bdL,0x6b2f23038b8f0047L,0x8c6c3738312d90b0L,
        0x348ae422ad82ca91L } },
    /* 213 */
    { { 0x7f4746635ccda2fbL,0x22accaa18e0726d2L,0x85adf782492b1f20L,
        0xc1074de0d9ef2d2eL },
      { 0xfcf3ce44ae9a65b3L,0xfd71e4ac05d7151bL,0xd4711f50ce6a9788L,
        0xfbadfbdbc9e54ffcL } },
    /* 214 */
    { { 0x1713f1cd20a99363L,0xb915658f6cf22775L,0x968175cd24d359b2L,
        0xb7f976b483716fcdL },
      { 0x5758e24d5d6dbf74L,0x8d23bafd71c3af36L,0x48f477600243dfe3L,
        0xf4d41b2ecafcc805L } },
    /* 215 */
    { { 0x51f1cf28fdabd48dL,0xce81be3632c078a4L,0x6ace2974117146e9L,
        0x180824eae0160f10L },
      { 0x0387698b66e58358L,0x63568752ce6ca358L,0x82380e345e41e6c5L,
        0x67e5f63983cf6d25L } },
    /* 216 */
    { { 0xf89ccb8dcf4899efL,0x949015f09ebb44c0L,0x546f9276b2598ec9L,
        0x9fef789a04c11fc6L },
      { 0x6d367ecf53d2a071L,0xb10e1a7fa4519b09L,0xca6b3fb0611e2eefL,
        0xbc80c181a99c4e20L } },
    /* 217 */
    { { 0x972536f8e5eb82e6L,0x1a484fc7f56cb920L,0xc78e217150b5da5eL,
        0x49270e629f8cdf10L },
      { 0x1a39b7bbea6b50adL,0x9a0284c1a2388ffcL,0x5403eb178107197bL,
        0xd2ee52f961372f7fL } },
    /* 218 */
    { { 0xd37cd28588e0362aL,0x442fa8a78fa5d94dL,0xaff836e5a434a526L,
        0xdfb478bee5abb733L },
      { 0xa91f1ce7673eede6L,0xa5390ad42b5b2f04L,0x5e66f7bf5530da2fL,
        0xd9a140b408df473aL } },
    /* 219 */
    { { 0x0e0221b56e8ea498L,0x623478293563ee09L,0xe06b8391335d2adeL,
        0x760c058d623f4b1aL },
      { 0x0b89b58cc198aa79L,0xf74890d2f07aba7fL,0x4e204110fde2556aL,
        0x7141982d8f190409L } },
    /* 220 */
    { { 0x6f0a0e334d4b0f45L,0xd9280b38392a94e1L,0x3af324c6b3c61d5eL,
        0x3af9d1ce89d54e47L },
      { 0xfd8f798120930371L,0xeda2664c21c17097L,0x0e9545dcdc42309bL,
        0xb1f815c373957dd6L } },
    /* 221 */
    { { 0x84faa78e89fec44aL,0xc8c2ae473caa4cafL,0x691c807dc1b6a624L,
        0xa41aed141543f052L },
      { 0x424353997d5ffe04L,0x8bacb2df625b6e20L,0x85d660be87817775L,
        0xd6e9c1dd86fb60efL } },
    /* 222 */
    { { 0x3aa2e97ec6853264L,0x771533b7e2304a0bL,0x1b912bb7b8eae9beL,
        0x9c9c6e10ae9bf8c2L },
      { 0xa2309a59e030b74cL,0x4ed7494d6a631e90L,0x89f44b23a49b79f2L,
        0x566bd59640fa61b6L } },
    /* 223 */
    { { 0x066c0118c18061f3L,0x190b25d37c83fc70L,0xf05fc8e027273245L,
        0xcf2c7390f525345eL },
      { 0xa09bceb410eb30cfL,0xcfd2ebba0d77703aL,0xe842c43a150ff255L,
        0x02f517558aa20979L } },
    /* 224 */
    { { 0x396ef794addb7d07L,0x0b4fc74224455500L,0xfaff8eacc78aa3ceL,
        0x14e9ada5e8d4d97dL },
      { 0xdaa480a12f7079e2L,0x45baa3cde4b0800eL,0x01765e2d7838157dL,
        0xa0ad4fab8e9d9ae8L } },
    /* 225 */
    { { 0x0bfb76214a653618L,0x1872813c31eaaa5fL,0x1553e73744949d5eL,
        0xbcd530b86e56ed1eL },
      { 0x169be85332e9c47bL,0xdc2776feb50059abL,0xcdba9761192bfbb4L,
        0x909283cf6979341dL } },
    /* 226 */
    { { 0x67b0032476e81a13L,0x9bee1a9962171239L,0x08ed361bd32e19d6L,
        0x35eeb7c9ace1549aL },
      { 0x1280ae5a7e4e5bdcL,0x2dcd2cd3b6ceec6eL,0x52e4224c6e266bc1L,
        0x9a8b2cf4448ae864L } },
    /* 227 */
    { { 0xf6471bf209d03b59L,0xc90e62a3b65af2abL,0xff7ff168ebd5eec9L,
        0x6bdb60f4d4491379L },
      { 0xdadafebc8a55bc30L,0xc79ead1610097fe0L,0x42e197414c1e3bddL,
        0x01ec3cfd94ba08a9L } },
    /* 228 */
    { { 0xba6277ebdc9485c2L,0x48cc9a7922fb10c7L,0x4f61d60f70a28d8aL,
        0xd1acb1c0475464f6L },
      { 0xd26902b126f36612L,0x59c3a44ee0618d8bL,0x4df8a813308357eeL,
        0x7dcd079d405626c2L } },
    /* 229 */
    { { 0x5ce7d4d3f05a4b48L,0xadcd295237230772L,0xd18f7971812a915aL,
        0x0bf53589377d19b8L },
      { 0x35ecd95a6c68ea73L,0xc7f3bbca823a584dL,0x9fb674c6f473a723L,
        0xd28be4d9e16686fcL } },
    /* 230 */
    { { 0x5d2b990638fa8e4bL,0x559f186e893fd8fcL,0x3a6de2aa436fb6fcL,
        0xd76007aa510f88ceL },
      { 0x2d10aab6523a4988L,0xb455cf4474dd0273L,0x7f467082a3407278L,
        0xf2b52f68b303bb01L } },
    /* 231 */
    { { 0x0d57eafa9835b4caL,0x2d2232fcbb669cbcL,0x8eeeb680c6643198L,
        0xd8dbe98ecc5aed3aL },
      { 0xcba9be3fc5a02709L,0x30be68e5f5ba1fa8L,0xfebd43cdf10ea852L,
        0xe01593a3ee559705L } },
    /* 232 */
    { { 0xd3e5af50ea75a0a6L,0x512226ac57858033L,0x6fe6d50fd0176406L,
        0xafec07b1aeb8ef06L },
      { 0x7fb9956780bb0a31L,0x6f1af3cc37309aaeL,0x9153a15a01abf389L,
        0xa71b93546e2dbfddL } },
    /* 233 */
    { { 0xbf8e12e018f593d2L,0xd1a90428a078122bL,0x150505db0ba4f2adL,
        0x53a2005c628523d9L },
      { 0x07c8b639e7f2b935L,0x2bff975ac182961aL,0x86bceea77518ca2cL,
        0xbf47d19b3d588e3dL } },
    /* 234 */
    { { 0x672967a7dd7665d5L,0x4e3030572f2f4de5L,0x144005ae80d4903fL,
        0x001c2c7f39c9a1b6L },
      { 0x143a801469efc6d6L,0xc810bdaa7bc7a724L,0x5f65670ba78150a4L,
        0xfdadf8e786ffb99bL } },
    /* 235 */
    { { 0xfd38cb88ffc00785L,0x77fa75913b48eb67L,0x0454d055bf368fbcL,
        0x3a838e4d5aa43c94L },
      { 0x561663293e97bb9aL,0x9eb93363441d94d9L,0x515591a60adb2a83L,
        0x3cdb8257873e1da3L } },
    /* 236 */
    { { 0x137140a97de77eabL,0xf7e1c50d41648109L,0x762dcad2ceb1d0dfL,
        0x5a60cc89f1f57fbaL },
      { 0x80b3638240d45673L,0x1b82be195913c655L,0x057284b8dd64b741L,
        0x922ff56fdbfd8fc0L } },
    /* 237 */
    { { 0x1b265deec9a129a1L,0xa5b1ce57cc284e04L,0x04380c46cebfbe3cL,
        0x72919a7df6c5cd62L },
      { 0x298f453a8fb90f9aL,0xd719c00b88e4031bL,0xe32c0e77796f1856L,
        0x5e7917803624089aL } },
    /* 238 */
    { { 0x5c16ec557f63cdfbL,0x8e6a3571f1cae4fdL,0xfce26bea560597caL,
        0x4e0a5371e24c2fabL },
      { 0x276a40d3a5765357L,0x3c89af440d73a2b4L,0xb8f370ae41d11a32L,
        0xf5ff7818d56604eeL } },
    /* 239 */
    { { 0xfbf3e3fe1a09df21L,0x26d5d28ee66e8e47L,0x2096bd0a29c89015L,
        0xe41df0e9533f5e64L },
      { 0x305fda40b3ba9e3fL,0xf2340ceb2604d895L,0x0866e1927f0367c7L,
        0x8edd7d6eac4f155fL } },
    /* 240 */
    { { 0xc9a1dc0e0bfc8ff3L,0x14efd82be936f42fL,0x67016f7ccca381efL,
        0x1432c1caed8aee96L },
      { 0xec68482970b23c26L,0xa64fe8730735b273L,0xe389f6e5eaef0f5aL,
        0xcaef480b5ac8d2c6L } },
    /* 241 */
    { { 0x5245c97875315922L,0xd82951713063cca5L,0xf3ce60d0b64ef2cbL,
        0xd0ba177e8efae236L },
      { 0x53a9ae8fb1b3af60L,0x1a796ae53d2da20eL,0x01d63605df9eef28L,
        0xf31c957c1c54ae16L } },
    /* 242 */
    { { 0xc0f58d5249cc4597L,0xdc5015b0bae0a028L,0xefc5fc55734a814aL,
        0x013404cb96e17c3aL },
      { 0xb29e2585c9a824bfL,0xd593185e001eaed7L,0x8d6ee68261ef68acL,
        0x6f377c4b91933e6cL } },
    /* 243 */
    { { 0x9f93bad1a8333fd2L,0xa89302025a2a95b8L,0x211e5037eaf75aceL,
        0x6dba3e4ed2d09506L },
      { 0xa48ef98cd04399cdL,0x1811c66ee6b73adeL,0x72f60752c17ecaf3L,
        0xf13cf3423becf4a7L } },
    /* 244 */
    { { 0xceeb9ec0a919e2ebL,0x83a9a195f62c0f68L,0xcfba3bb67aba2299L,
        0xc83fa9a9274bbad3L },
      { 0x0d7d1b0b62fa1ce0L,0xe58b60f53418efbfL,0xbfa8ef9e52706f04L,
        0xb49d70f45d702683L } },
    /* 245 */
    { { 0x914c7510fad5513bL,0x05f32eecb1751e2dL,0x6d850418d9fb9d59L,
        0x59cfadbb0c30f1cfL },
      { 0xe167ac2355cb7fd6L,0x249367b8820426a3L,0xeaeec58c90a78864L,
        0x5babf362354a4b67L } },
    /* 246 */
    { { 0x37c981d1ee424865L,0x8b002878f2e5577fL,0x702970f1b9e0c058L,
        0x6188c6a79026c8f0L },
      { 0x06f9a19bd0f244daL,0x1ecced5cfb080873L,0x35470f9b9f213637L,
        0x993fe475df50b9d9L } },
    /* 247 */
    { { 0x68e31cdf9b2c3609L,0x84eb19c02c46d4eaL,0x7ac9ec1a9a775101L,
        0x81f764664c80616bL },
      { 0x1d7c2a5a75fbe978L,0x6743fed3f183b356L,0x838d1f04501dd2bfL,
        0x564a812a5fe9060dL } },
    /* 248 */
    { { 0x7a5a64f4fa817d1dL,0x55f96844bea82e0fL,0xb5ff5a0fcd57f9aaL,
        0x226bf3cf00e51d6cL },
      { 0xd6d1a9f92f2833cfL,0x20a0a35a4f4f89a8L,0x11536c498f3f7f77L,
        0x68779f47ff257836L } },
    /* 249 */
    { { 0x79b0c1c173043d08L,0xa54467741fc020faL,0xd3767e289a6d26d0L,
        0x97bcb0d1eb092e0bL },
      { 0x2ab6eaa8f32ed3c3L,0xc8a4f151b281bc48L,0x4d1bf4f3bfa178f3L,
        0xa872ffe80a784655L } },
    /* 250 */
    { { 0xb1ab7935a32b2086L,0xe1eb710e8160f486L,0x9bd0cd913b6ae6beL,
        0x02812bfcb732a36aL },
      { 0xa63fd7cacf605318L,0x646e5d50fdfd6d1dL,0xa1d683982102d619L,
        0x07391cc9fe5396afL } },
    /* 251 */
    { { 0xc50157f08b80d02bL,0x6b8333d162877f7fL,0x7aca1af878d542aeL,
        0x355d2adc7e6d2a08L },
      { 0xb41f335a287386e1L,0xfd272a94f8e43275L,0x286ca2cde79989eaL,
        0x3dc2b1e37c2a3a79L } },
    /* 252 */
    { { 0xd689d21c04581352L,0x0a00c825376782beL,0x203bd5909fed701fL,
        0xc47869103ccd846bL },
      { 0x5dba770824c768edL,0x72feea026841f657L,0x73313ed56accce0eL,
        0xccc42968d5bb4d32L } },
    /* 253 */
    { { 0x94e50de13d7620b9L,0xd89a5c8a5992a56aL,0xdc007640675487c9L,
        0xe147eb42aa4871cfL },
      { 0x274ab4eeacf3ae46L,0xfd4936fb50350fbeL,0xdf2afe4748c840eaL,
        0x239ac047080e96e3L } },
    /* 254 */
    { { 0x481d1f352bfee8d4L,0xce80b5cffa7b0fecL,0x105c4c9e2ce9af3cL,
        0xc55fa1a3f5f7e59dL },
      { 0x3186f14e8257c227L,0xc5b1653f342be00bL,0x09afc998aa904fb2L,
        0x094cd99cd4f4b699L } },
    /* 255 */
    { { 0x8a981c84d703bebaL,0x8631d15032ceb291L,0xa445f2c9e3bd49ecL,
        0xb90a30b642abad33L },
      { 0xb465404fb4a5abf9L,0x004750c375db7603L,0x6f9a42ccca35d89fL,
        0x019f8b9a1b7924f7L } },
};

/* Multiply the base point of P256 by the scalar and return the result.
 * If map is true then convert result to affine coordinates.
 *
 * Stripe implementation.
 * Pre-generated: 2^0, 2^32, ...
 * Pre-generated: products of all combinations of above.
 * 8 doubles and adds (with qz=1)
 *
 * r     Resulting point.
 * k     Scalar to multiply by.
 * map   Indicates whether to convert result to affine.
 * ct    Constant time required.
 * heap  Heap to use for allocation.
 * returns MEMORY_E when memory allocation fails and MP_OKAY on success.
 */
static int sp_256_ecc_mulmod_base_4(sp_point_256* r, const sp_digit* k,
        int map, int ct, void* heap)
{
    return sp_256_ecc_mulmod_stripe_4(r, &p256_base, p256_table,
                                      k, map, ct, heap);
}

#endif /* WC_NO_CACHE_RESISTANT */
#else
/* The index into pre-computation table to use. */
static const uint8_t recode_index_4_7[130] = {
     0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
    16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
    32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
    48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
    64, 63, 62, 61, 60, 59, 58, 57, 56, 55, 54, 53, 52, 51, 50, 49,
    48, 47, 46, 45, 44, 43, 42, 41, 40, 39, 38, 37, 36, 35, 34, 33,
    32, 31, 30, 29, 28, 27, 26, 25, 24, 23, 22, 21, 20, 19, 18, 17,
    16, 15, 14, 13, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,
     0,  1,
};

/* Whether to negate y-ordinate. */
static const uint8_t recode_neg_4_7[130] = {
     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
     0,  0,
};

/* Recode the scalar for multiplication using pre-computed values and
 * subtraction.
 *
 * k  Scalar to multiply by.
 * v  Vector of operations to perform.
 */
static void sp_256_ecc_recode_7_4(const sp_digit* k, ecc_recode_256* v)
{
    int i;
    int j;
    uint8_t y;
    int carry = 0;
    int o;
    sp_digit n;

    j = 0;
    n = k[j];
    o = 0;
    for (i=0; i<37; i++) {
        y = (int8_t)n;
        if (o + 7 < 64) {
            y &= 0x7f;
            n >>= 7;
            o += 7;
        }
        else if (o + 7 == 64) {
            n >>= 7;
            if (++j < 4)
                n = k[j];
            o = 0;
        }
        else if (++j < 4) {
            n = k[j];
            y |= (uint8_t)((n << (64 - o)) & 0x7f);
            o -= 57;
            n >>= o;
        }

        y += (uint8_t)carry;
        v[i].i = recode_index_4_7[y];
        v[i].neg = recode_neg_4_7[y];
        carry = (y >> 7) + v[i].neg;
    }
}

#ifndef WC_NO_CACHE_RESISTANT
/* Touch each possible entry that could be being copied.
 *
 * r      Point to copy into.
 * table  Table - start of the entires to access
 * idx    Index of entry to retrieve.
 */
static void sp_256_get_entry_65_4(sp_point_256* r,
    const sp_table_entry_256* table, int idx)
{
    int i;
    sp_digit mask;

    r->x[0] = 0;
    r->x[1] = 0;
    r->x[2] = 0;
    r->x[3] = 0;
    r->y[0] = 0;
    r->y[1] = 0;
    r->y[2] = 0;
    r->y[3] = 0;
    for (i = 1; i < 65; i++) {
        mask = 0 - (i == idx);
        r->x[0] |= mask & table[i].x[0];
        r->x[1] |= mask & table[i].x[1];
        r->x[2] |= mask & table[i].x[2];
        r->x[3] |= mask & table[i].x[3];
        r->y[0] |= mask & table[i].y[0];
        r->y[1] |= mask & table[i].y[1];
        r->y[2] |= mask & table[i].y[2];
        r->y[3] |= mask & table[i].y[3];
    }
}
#endif /* !WC_NO_CACHE_RESISTANT */
static const sp_table_entry_256 p256_table[2405] = {
    /* 0 << 0 */
    { { 0x00, 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x00, 0x00 } },
    /* 1 << 0 */
    { { 0x79e730d418a9143cL,0x75ba95fc5fedb601L,0x79fb732b77622510L,
        0x18905f76a53755c6L },
      { 0xddf25357ce95560aL,0x8b4ab8e4ba19e45cL,0xd2e88688dd21f325L,
        0x8571ff1825885d85L } },
    /* 2 << 0 */
    { { 0x850046d410ddd64dL,0xaa6ae3c1a433827dL,0x732205038d1490d9L,
        0xf6bb32e43dcf3a3bL },
      { 0x2f3648d361bee1a5L,0x152cd7cbeb236ff8L,0x19a8fb0e92042dbeL,
        0x78c577510a5b8a3bL } },
    /* 3 << 0 */
    { { 0xffac3f904eebc127L,0xb027f84a087d81fbL,0x66ad77dd87cbbc98L,
        0x26936a3fb6ff747eL },
      { 0xb04c5c1fc983a7ebL,0x583e47ad0861fe1aL,0x788208311a2ee98eL,
        0xd5f06a29e587cc07L } },
    /* 4 << 0 */
    { { 0x74b0b50d46918dccL,0x4650a6edc623c173L,0x0cdaacace8100af2L,
        0x577362f541b0176bL },
      { 0x2d96f24ce4cbaba6L,0x17628471fad6f447L,0x6b6c36dee5ddd22eL,
        0x84b14c394c5ab863L } },
    /* 5 << 0 */
    { { 0xbe1b8aaec45c61f5L,0x90ec649a94b9537dL,0x941cb5aad076c20cL,
        0xc9079605890523c8L },
      { 0xeb309b4ae7ba4f10L,0x73c568efe5eb882bL,0x3540a9877e7a1f68L,
        0x73a076bb2dd1e916L } },
    /* 6 << 0 */
    { { 0x403947373e77664aL,0x55ae744f346cee3eL,0xd50a961a5b17a3adL,
        0x13074b5954213673L },
      { 0x93d36220d377e44bL,0x299c2b53adff14b5L,0xf424d44cef639f11L,
        0xa4c9916d4a07f75fL } },
    /* 7 << 0 */
    { { 0x0746354ea0173b4fL,0x2bd20213d23c00f7L,0xf43eaab50c23bb08L,
        0x13ba5119c3123e03L },
      { 0x2847d0303f5b9d4dL,0x6742f2f25da67bddL,0xef933bdc77c94195L,
        0xeaedd9156e240867L } },
    /* 8 << 0 */
    { { 0x27f14cd19499a78fL,0x462ab5c56f9b3455L,0x8f90f02af02cfc6bL,
        0xb763891eb265230dL },
      { 0xf59da3a9532d4977L,0x21e3327dcf9eba15L,0x123c7b84be60bbf0L,
        0x56ec12f27706df76L } },
    /* 9 << 0 */
    { { 0x75c96e8f264e20e8L,0xabe6bfed59a7a841L,0x2cc09c0444c8eb00L,
        0xe05b3080f0c4e16bL },
      { 0x1eb7777aa45f3314L,0x56af7bedce5d45e3L,0x2b6e019a88b12f1aL,
        0x086659cdfd835f9bL } },
    /* 10 << 0 */
    { { 0x2c18dbd19dc21ec8L,0x98f9868a0fcf8139L,0x737d2cd648250b49L,
        0xcc61c94724b3428fL },
      { 0x0c2b407880dd9e76L,0xc43a8991383fbe08L,0x5f7d2d65779be5d2L,
        0x78719a54eb3b4ab5L } },
    /* 11 << 0 */
    { { 0xea7d260a6245e404L,0x9de407956e7fdfe0L,0x1ff3a4158dac1ab5L,
        0x3e7090f1649c9073L },
      { 0x1a7685612b944e88L,0x250f939ee57f61c8L,0x0c0daa891ead643dL,
        0x68930023e125b88eL } },
    /* 12 << 0 */
    { { 0x04b71aa7d2697768L,0xabdedef5ca345a33L,0x2409d29dee37385eL,
        0x4ee1df77cb83e156L },
      { 0x0cac12d91cbb5b43L,0x170ed2f6ca895637L,0x28228cfa8ade6d66L,
        0x7ff57c9553238acaL } },
    /* 13 << 0 */
    { { 0xccc425634b2ed709L,0x0e356769856fd30dL,0xbcbcd43f559e9811L,
        0x738477ac5395b759L },
      { 0x35752b90c00ee17fL,0x68748390742ed2e3L,0x7cd06422bd1f5bc1L,
        0xfbc08769c9e7b797L } },
    /* 14 << 0 */
    { { 0xa242a35bb0cf664aL,0x126e48f77f9707e3L,0x1717bf54c6832660L,
        0xfaae7332fd12c72eL },
      { 0x27b52db7995d586bL,0xbe29569e832237c2L,0xe8e4193e2a65e7dbL,
        0x152706dc2eaa1bbbL } },
    /* 15 << 0 */
    { { 0x72bcd8b7bc60055bL,0x03cc23ee56e27e4bL,0xee337424e4819370L,
        0xe2aa0e430ad3da09L },
      { 0x40b8524f6383c45dL,0xd766355442a41b25L,0x64efa6de778a4797L,
        0x2042170a7079adf4L } },
    /* 16 << 0 */
    { { 0x808b0b650bc6fb80L,0x5882e0753ffe2e6bL,0xd5ef2f7c2c83f549L,
        0x54d63c809103b723L },
      { 0xf2f11bd652a23f9bL,0x3670c3194b0b6587L,0x55c4623bb1580e9eL,
        0x64edf7b201efe220L } },
    /* 17 << 0 */
    { { 0x97091dcbd53c5c9dL,0xf17624b6ac0a177bL,0xb0f139752cfe2dffL,
        0xc1a35c0a6c7a574eL },
      { 0x227d314693e79987L,0x0575bf30e89cb80eL,0x2f4e247f0d1883bbL,
        0xebd512263274c3d0L } },
    /* 18 << 0 */
    { { 0x5f3e51c856ada97aL,0x4afc964d8f8b403eL,0xa6f247ab412e2979L,
        0x675abd1b6f80ebdaL },
      { 0x66a2bd725e485a1dL,0x4b2a5caf8f4f0b3cL,0x2626927f1b847bbaL,
        0x6c6fc7d90502394dL } },
    /* 19 << 0 */
    { { 0xfea912baa5659ae8L,0x68363aba25e1a16eL,0xb8842277752c41acL,
        0xfe545c282897c3fcL },
      { 0x2d36e9e7dc4c696bL,0x5806244afba977c5L,0x85665e9be39508c1L,
        0xf720ee256d12597bL } },
    /* 20 << 0 */
    { { 0x8a979129d2337a31L,0x5916868f0f862bdcL,0x048099d95dd283baL,
        0xe2d1eeb6fe5bfb4eL },
      { 0x82ef1c417884005dL,0xa2d4ec17ffffcbaeL,0x9161c53f8aa95e66L,
        0x5ee104e1c5fee0d0L } },
    /* 21 << 0 */
    { { 0x562e4cecc135b208L,0x74e1b2654783f47dL,0x6d2a506c5a3f3b30L,
        0xecead9f4c16762fcL },
      { 0xf29dd4b2e286e5b9L,0x1b0fadc083bb3c61L,0x7a75023e7fac29a4L,
        0xc086d5f1c9477fa3L } },
    /* 22 << 0 */
    { { 0x0fc611352f6f3076L,0xc99ffa23e3912a9aL,0x6a0b0685d2f8ba3dL,
        0xfdc777e8e93358a4L },
      { 0x94a787bb35415f04L,0x640c2d6a4d23fea4L,0x9de917da153a35b5L,
        0x793e8d075d5cd074L } },
    /* 23 << 0 */
    { { 0xf4f876532de45068L,0x37c7a7e89e2e1f6eL,0xd0825fa2a3584069L,
        0xaf2cea7c1727bf42L },
      { 0x0360a4fb9e4785a9L,0xe5fda49c27299f4aL,0x48068e1371ac2f71L,
        0x83d0687b9077666fL } },
    /* 24 << 0 */
    { { 0x6d3883b215d02819L,0x6d0d755040dd9a35L,0x61d7cbf91d2b469fL,
        0xf97b232f2efc3115L },
      { 0xa551d750b24bcbc7L,0x11ea494988a1e356L,0x7669f03193cb7501L,
        0x595dc55eca737b8aL } },
    /* 25 << 0 */
    { { 0xa4a319acd837879fL,0x6fc1b49eed6b67b0L,0xe395993332f1f3afL,
        0x966742eb65432a2eL },
      { 0x4b8dc9feb4966228L,0x96cc631243f43950L,0x12068859c9b731eeL,
        0x7b948dc356f79968L } },
    /* 26 << 0 */
    { { 0x61e4ad32ed1f8008L,0xe6c9267ad8b17538L,0x1ac7c5eb857ff6fbL,
        0x994baaa855f2fb10L },
      { 0x84cf14e11d248018L,0x5a39898b628ac508L,0x14fde97b5fa944f5L,
        0xed178030d12e5ac7L } },
    /* 27 << 0 */
    { { 0x042c2af497e2feb4L,0xd36a42d7aebf7313L,0x49d2c9eb084ffdd7L,
        0x9f8aa54b2ef7c76aL },
      { 0x9200b7ba09895e70L,0x3bd0c66fddb7fb58L,0x2d97d10878eb4cbbL,
        0x2d431068d84bde31L } },
    /* 28 << 0 */
    { { 0x4b523eb7172ccd1fL,0x7323cb2830a6a892L,0x97082ec0cfe153ebL,
        0xe97f6b6af2aadb97L },
      { 0x1d3d393ed1a83da1L,0xa6a7f9c7804b2a68L,0x4a688b482d0cb71eL,
        0xa9b4cc5f40585278L } },
    /* 29 << 0 */
    { { 0x5e5db46acb66e132L,0xf1be963a0d925880L,0x944a70270317b9e2L,
        0xe266f95948603d48L },
      { 0x98db66735c208899L,0x90472447a2fb18a3L,0x8a966939777c619fL,
        0x3798142a2a3be21bL } },
    /* 30 << 0 */
    { { 0xb4241cb13298b343L,0xa3a14e49b44f65a1L,0xc5f4d6cd3ac77acdL,
        0xd0288cb552b6fc3cL },
      { 0xd5cc8c2f1c040abcL,0xb675511e06bf9b4aL,0xd667da379b3aa441L,
        0x460d45ce51601f72L } },
    /* 31 << 0 */
    { { 0xe2f73c696755ff89L,0xdd3cf7e7473017e6L,0x8ef5689d3cf7600dL,
        0x948dc4f8b1fc87b4L },
      { 0xd9e9fe814ea53299L,0x2d921ca298eb6028L,0xfaecedfd0c9803fcL,
        0xf38ae8914d7b4745L } },
    /* 32 << 0 */
    { { 0xd8c5fccfc5e3a3d8L,0xbefd904c4079dfbfL,0xbc6d6a58fead0197L,
        0x39227077695532a4L },
      { 0x09e23e6ddbef42f5L,0x7e449b64480a9908L,0x7b969c1aad9a2e40L,
        0x6231d7929591c2a4L } },
    /* 33 << 0 */
    { { 0x871514560f664534L,0x85ceae7c4b68f103L,0xac09c4ae65578ab9L,
        0x33ec6868f044b10cL },
      { 0x6ac4832b3a8ec1f1L,0x5509d1285847d5efL,0xf909604f763f1574L,
        0xb16c4303c32f63c4L } },
    /* 34 << 0 */
    { { 0xb6ab20147ca23cd3L,0xcaa7a5c6a391849dL,0x5b0673a375678d94L,
        0xc982ddd4dd303e64L },
      { 0xfd7b000b5db6f971L,0xbba2cb1f6f876f92L,0xc77332a33c569426L,
        0xa159100c570d74f8L } },
    /* 35 << 0 */
    { { 0xfd16847fdec67ef5L,0x742ee464233e76b7L,0x0b8e4134efc2b4c8L,
        0xca640b8642a3e521L },
      { 0x653a01908ceb6aa9L,0x313c300c547852d5L,0x24e4ab126b237af7L,
        0x2ba901628bb47af8L } },
    /* 36 << 0 */
    { { 0x3d5e58d6a8219bb7L,0xc691d0bd1b06c57fL,0x0ae4cb10d257576eL,
        0x3569656cd54a3dc3L },
      { 0xe5ebaebd94cda03aL,0x934e82d3162bfe13L,0x450ac0bae251a0c6L,
        0x480b9e11dd6da526L } },
    /* 37 << 0 */
    { { 0x00467bc58cce08b5L,0xb636458c7f178d55L,0xc5748baea677d806L,
        0x2763a387dfa394ebL },
      { 0xa12b448a7d3cebb6L,0xe7adda3e6f20d850L,0xf63ebce51558462cL,
        0x58b36143620088a8L } },
    /* 38 << 0 */
    { { 0x8a2cc3ca4d63c0eeL,0x512331170fe948ceL,0x7463fd85222ef33bL,
        0xadf0c7dc7c603d6cL },
      { 0x0ec32d3bfe7765e5L,0xccaab359bf380409L,0xbdaa84d68e59319cL,
        0xd9a4c2809c80c34dL } },
    /* 39 << 0 */
    { { 0xa9d89488a059c142L,0x6f5ae714ff0b9346L,0x068f237d16fb3664L,
        0x5853e4c4363186acL },
      { 0xe2d87d2363c52f98L,0x2ec4a76681828876L,0x47b864fae14e7b1cL,
        0x0c0bc0e569192408L } },
    /* 40 << 0 */
    { { 0xe4d7681db82e9f3eL,0x83200f0bdf25e13cL,0x8909984c66f27280L,
        0x462d7b0075f73227L },
      { 0xd90ba188f2651798L,0x74c6e18c36ab1c34L,0xab256ea35ef54359L,
        0x03466612d1aa702fL } },
    /* 41 << 0 */
    { { 0x624d60492ed22e91L,0x6fdfe0b56f072822L,0xeeca111539ce2271L,
        0x98100a4fdb01614fL },
      { 0xb6b0daa2a35c628fL,0xb6f94d2ec87e9a47L,0xc67732591d57d9ceL,
        0xf70bfeec03884a7bL } },
    /* 42 << 0 */
    { { 0x5fb35ccfed2bad01L,0xa155cbe31da6a5c7L,0xc2e2594c30a92f8fL,
        0x649c89ce5bfafe43L },
      { 0xd158667de9ff257aL,0x9b359611f32c50aeL,0x4b00b20b906014cfL,
        0xf3a8cfe389bc7d3dL } },
    /* 43 << 0 */
    { { 0x4ff23ffd248a7d06L,0x80c5bfb4878873faL,0xb7d9ad9005745981L,
        0x179c85db3db01994L },
      { 0xba41b06261a6966cL,0x4d82d052eadce5a8L,0x9e91cd3ba5e6a318L,
        0x47795f4f95b2dda0L } },
    /* 44 << 0 */
    { { 0xecfd7c1fd55a897cL,0x009194abb29110fbL,0x5f0e2046e381d3b0L,
        0x5f3425f6a98dd291L },
      { 0xbfa06687730d50daL,0x0423446c4b083b7fL,0x397a247dd69d3417L,
        0xeb629f90387ba42aL } },
    /* 45 << 0 */
    { { 0x1ee426ccd5cd79bfL,0x0032940b946c6e18L,0x1b1e8ae057477f58L,
        0xe94f7d346d823278L },
      { 0xc747cb96782ba21aL,0xc5254469f72b33a5L,0x772ef6dec7f80c81L,
        0xd73acbfe2cd9e6b5L } },
    /* 46 << 0 */
    { { 0x4075b5b149ee90d9L,0x785c339aa06e9ebaL,0xa1030d5babf825e0L,
        0xcec684c3a42931dcL },
      { 0x42ab62c9c1586e63L,0x45431d665ab43f2bL,0x57c8b2c055f7835dL,
        0x033da338c1b7f865L } },
    /* 47 << 0 */
    { { 0x283c7513caa76097L,0x0a624fa936c83906L,0x6b20afec715af2c7L,
        0x4b969974eba78bfdL },
      { 0x220755ccd921d60eL,0x9b944e107baeca13L,0x04819d515ded93d4L,
        0x9bbff86e6dddfd27L } },
    /* 48 << 0 */
    { { 0x6b34413077adc612L,0xa7496529bbd803a0L,0x1a1baaa76d8805bdL,
        0xc8403902470343adL },
      { 0x39f59f66175adff1L,0x0b26d7fbb7d8c5b7L,0xa875f5ce529d75e3L,
        0x85efc7e941325cc2L } },
    /* 49 << 0 */
    { { 0x21950b421ff6acd3L,0xffe7048453dc6909L,0xff4cd0b228766127L,
        0xabdbe6084fb7db2bL },
      { 0x837c92285e1109e8L,0x26147d27f4645b5aL,0x4d78f592f7818ed8L,
        0xd394077ef247fa36L } },
    /* 50 << 0 */
    { { 0x0fb9c2d0488c171aL,0xa78bfbaa13685278L,0xedfbe268d5b1fa6aL,
        0x0dceb8db2b7eaba7L },
      { 0xbf9e80899ae2b710L,0xefde7ae6a4449c96L,0x43b7716bcc143a46L,
        0xd7d34194c3628c13L } },
    /* 51 << 0 */
    { { 0x508cec1c3b3f64c9L,0xe20bc0ba1e5edf3fL,0xda1deb852f4318d4L,
        0xd20ebe0d5c3fa443L },
      { 0x370b4ea773241ea3L,0x61f1511c5e1a5f65L,0x99a5e23d82681c62L,
        0xd731e383a2f54c2dL } },
    /* 52 << 0 */
    { { 0x2692f36e83445904L,0x2e0ec469af45f9c0L,0x905a3201c67528b7L,
        0x88f77f34d0e5e542L },
      { 0xf67a8d295864687cL,0x23b92eae22df3562L,0x5c27014b9bbec39eL,
        0x7ef2f2269c0f0f8dL } },
    /* 53 << 0 */
    { { 0x97359638546c4d8dL,0x5f9c3fc492f24679L,0x912e8beda8c8acd9L,
        0xec3a318d306634b0L },
      { 0x80167f41c31cb264L,0x3db82f6f522113f2L,0xb155bcd2dcafe197L,
        0xfba1da5943465283L } },
    /* 54 << 0 */
    { { 0xa0425b8eb212cf53L,0x4f2e512ef8557c5fL,0xc1286ff925c4d56cL,
        0xbb8a0feaee26c851L },
      { 0xc28f70d2e7d6107eL,0x7ee0c444e76265aaL,0x3df277a41d1936b1L,
        0x1a556e3fea9595ebL } },
    /* 55 << 0 */
    { { 0x258bbbf9e7305683L,0x31eea5bf07ef5be6L,0x0deb0e4a46c814c1L,
        0x5cee8449a7b730ddL },
      { 0xeab495c5a0182bdeL,0xee759f879e27a6b4L,0xc2cf6a6880e518caL,
        0x25e8013ff14cf3f4L } },
    /* 56 << 0 */
    { { 0x8fc441407e8d7a14L,0xbb1ff3ca9556f36aL,0x6a84438514600044L,
        0xba3f0c4a7451ae63L },
      { 0xdfcac25b1f9af32aL,0x01e0db86b1f2214bL,0x4e9a5bc2a4b596acL,
        0x83927681026c2c08L } },
    /* 57 << 0 */
    { { 0x3ec832e77acaca28L,0x1bfeea57c7385b29L,0x068212e3fd1eaf38L,
        0xc13298306acf8cccL },
      { 0xb909f2db2aac9e59L,0x5748060db661782aL,0xc5ab2632c79b7a01L,
        0xda44c6c600017626L } },
    /* 58 << 0 */
    { { 0xf26c00e8a7ea82f0L,0x99cac80de4299aafL,0xd66fe3b67ed78be1L,
        0x305f725f648d02cdL },
      { 0x33ed1bc4623fb21bL,0xfa70533e7a6319adL,0x17ab562dbe5ffb3eL,
        0x0637499456674741L } },
    /* 59 << 0 */
    { { 0x69d44ed65c46aa8eL,0x2100d5d3a8d063d1L,0xcb9727eaa2d17c36L,
        0x4c2bab1b8add53b7L },
      { 0xa084e90c15426704L,0x778afcd3a837ebeaL,0x6651f7017ce477f8L,
        0xa062499846fb7a8bL } },
    /* 60 << 0 */
    { { 0xdc1e6828ed8a6e19L,0x33fc23364189d9c7L,0x026f8fe2671c39bcL,
        0xd40c4ccdbc6f9915L },
      { 0xafa135bbf80e75caL,0x12c651a022adff2cL,0xc40a04bd4f51ad96L,
        0x04820109bbe4e832L } },
    /* 61 << 0 */
    { { 0x3667eb1a7f4c04ccL,0x59556621a9404f84L,0x71cdf6537eceb50aL,
        0x994a44a69b8335faL },
      { 0xd7faf819dbeb9b69L,0x473c5680eed4350dL,0xb6658466da44bba2L,
        0x0d1bc780872bdbf3L } },
    /* 62 << 0 */
    { { 0xe535f175a1962f91L,0x6ed7e061ed58f5a7L,0x177aa4c02089a233L,
        0x0dbcb03ae539b413L },
      { 0xe3dc424ebb32e38eL,0x6472e5ef6806701eL,0xdd47ff98814be9eeL,
        0x6b60cfff35ace009L } },
    /* 63 << 0 */
    { { 0xb8d3d9319ff91fe5L,0x039c4800f0518eedL,0x95c376329182cb26L,
        0x0763a43482fc568dL },
      { 0x707c04d5383e76baL,0xac98b930824e8197L,0x92bf7c8f91230de0L,
        0x90876a0140959b70L } },
    /* 64 << 0 */
    { { 0xdb6d96f305968b80L,0x380a0913089f73b9L,0x7da70b83c2c61e01L,
        0x95fb8394569b38c7L },
      { 0x9a3c651280edfe2fL,0x8f726bb98faeaf82L,0x8010a4a078424bf8L,
        0x296720440e844970L } },
    /* 0 << 7 */
    { { 0x00, 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x00, 0x00 } },
    /* 1 << 7 */
    { { 0x63c5cb817a2ad62aL,0x7ef2b6b9ac62ff54L,0x3749bba4b3ad9db5L,
        0xad311f2c46d5a617L },
      { 0xb77a8087c2ff3b6dL,0xb46feaf3367834ffL,0xf8aa266d75d6b138L,
        0xfa38d320ec008188L } },
    /* 2 << 7 */
    { { 0x486d8ffa696946fcL,0x50fbc6d8b9cba56dL,0x7e3d423e90f35a15L,
        0x7c3da195c0dd962cL },
      { 0xe673fdb03cfd5d8bL,0x0704b7c2889dfca5L,0xf6ce581ff52305aaL,
        0x399d49eb914d5e53L } },
    /* 3 << 7 */
    { { 0x380a496d6ec293cdL,0x733dbda78e7051f5L,0x037e388db849140aL,
        0xee4b32b05946dbf6L },
      { 0xb1c4fda9cae368d1L,0x5001a7b0fdb0b2f3L,0x6df593742e3ac46eL,
        0x4af675f239b3e656L } },
    /* 4 << 7 */
    { { 0x44e3811039949296L,0x5b63827b361db1b5L,0x3e5323ed206eaff5L,
        0x942370d2c21f4290L },
      { 0xf2caaf2ee0d985a1L,0x192cc64b7239846dL,0x7c0b8f47ae6312f8L,
        0x7dc61f9196620108L } },
    /* 5 << 7 */
    { { 0xb830fb5bc2da7de9L,0xd0e643df0ff8d3beL,0x31ee77ba188a9641L,
        0x4e8aa3aabcf6d502L },
      { 0xf9fb65329a49110fL,0xd18317f62dd6b220L,0x7e3ced4152c3ea5aL,
        0x0d296a147d579c4aL } },
    /* 6 << 7 */
    { { 0x35d6a53eed4c3717L,0x9f8240cf3d0ed2a3L,0x8c0d4d05e5543aa5L,
        0x45d5bbfbdd33b4b4L },
      { 0xfa04cc73137fd28eL,0x862ac6efc73b3ffdL,0x403ff9f531f51ef2L,
        0x34d5e0fcbc73f5a2L } },
    /* 7 << 7 */
    { { 0xf252682008913f4fL,0xea20ed61eac93d95L,0x51ed38b46ca6b26cL,
        0x8662dcbcea4327b0L },
      { 0x6daf295c725d2aaaL,0xbad2752f8e52dcdaL,0x2210e7210b17daccL,
        0xa37f7912d51e8232L } },
    /* 8 << 7 */
    { { 0x4f7081e144cc3addL,0xd5ffa1d687be82cfL,0x89890b6c0edd6472L,
        0xada26e1a3ed17863L },
      { 0x276f271563483caaL,0xe6924cd92f6077fdL,0x05a7fe980a466e3cL,
        0xf1c794b0b1902d1fL } },
    /* 9 << 7 */
    { { 0xe521368882a8042cL,0xd931cfafcd278298L,0x069a0ae0f597a740L,
        0x0adbb3f3eb59107cL },
      { 0x983e951e5eaa8eb8L,0xe663a8b511b48e78L,0x1631cc0d8a03f2c5L,
        0x7577c11e11e271e2L } },
    /* 10 << 7 */
    { { 0x33b2385c08369a90L,0x2990c59b190eb4f8L,0x819a6145c68eac80L,
        0x7a786d622ec4a014L },
      { 0x33faadbe20ac3a8dL,0x31a217815aba2d30L,0x209d2742dba4f565L,
        0xdb2ce9e355aa0fbbL } },
    /* 11 << 7 */
    { { 0x8cef334b168984dfL,0xe81dce1733879638L,0xf6e6949c263720f0L,
        0x5c56feaff593cbecL },
      { 0x8bff5601fde58c84L,0x74e241172eccb314L,0xbcf01b614c9a8a78L,
        0xa233e35e544c9868L } },
    /* 12 << 7 */
    { { 0xb3156bf38bd7aff1L,0x1b5ee4cb1d81b146L,0x7ba1ac41d628a915L,
        0x8f3a8f9cfd89699eL },
      { 0x7329b9c9a0748be7L,0x1d391c95a92e621fL,0xe51e6b214d10a837L,
        0xd255f53a4947b435L } },
    /* 13 << 7 */
    { { 0x07669e04f1788ee3L,0xc14f27afa86938a2L,0x8b47a334e93a01c0L,
        0xff627438d9366808L },
      { 0x7a0985d8ca2a5965L,0x3d9a5542d6e9b9b3L,0xc23eb80b4cf972e8L,
        0x5c1c33bb4fdf72fdL } },
    /* 14 << 7 */
    { { 0x0c4a58d474a86108L,0xf8048a8fee4c5d90L,0xe3c7c924e86d4c80L,
        0x28c889de056a1e60L },
      { 0x57e2662eb214a040L,0xe8c48e9837e10347L,0x8774286280ac748aL,
        0xf1c24022186b06f2L } },
    /* 15 << 7 */
    { { 0xac2dd4c35f74040aL,0x409aeb71fceac957L,0x4fbad78255c4ec23L,
        0xb359ed618a7b76ecL },
      { 0x12744926ed6f4a60L,0xe21e8d7f4b912de3L,0xe2575a59fc705a59L,
        0x72f1d4deed2dbc0eL } },
    /* 16 << 7 */
    { { 0x3d2b24b9eb7926b8L,0xbff88cb3cdbe5509L,0xd0f399afe4dd640bL,
        0x3c5fe1302f76ed45L },
      { 0x6f3562f43764fb3dL,0x7b5af3183151b62dL,0xd5bd0bc7d79ce5f3L,
        0xfdaf6b20ec66890fL } },
    /* 17 << 7 */
    { { 0x735c67ec6063540cL,0x50b259c2e5f9cb8fL,0xb8734f9a3f99c6abL,
        0xf8cc13d5a3a7bc85L },
      { 0x80c1b305c5217659L,0xfe5364d44ec12a54L,0xbd87045e681345feL,
        0x7f8efeb1582f897fL } },
    /* 18 << 7 */
    { { 0xe8cbf1e5d5923359L,0xdb0cea9d539b9fb0L,0x0c5b34cf49859b98L,
        0x5e583c56a4403cc6L },
      { 0x11fc1a2dd48185b7L,0xc93fbc7e6e521787L,0x47e7a05805105b8bL,
        0x7b4d4d58db8260c8L } },
    /* 19 << 7 */
    { { 0xe33930b046eb842aL,0x8e844a9a7bdae56dL,0x34ef3a9e13f7fdfcL,
        0xb3768f82636ca176L },
      { 0x2821f4e04e09e61cL,0x414dc3a1a0c7cddcL,0xd537943754945fcdL,
        0x151b6eefb3555ff1L } },
    /* 20 << 7 */
    { { 0xb31bd6136339c083L,0x39ff8155dfb64701L,0x7c3388d2e29604abL,
        0x1e19084ba6b10442L },
      { 0x17cf54c0eccd47efL,0x896933854a5dfb30L,0x69d023fb47daf9f6L,
        0x9222840b7d91d959L } },
    /* 21 << 7 */
    { { 0x439108f5803bac62L,0x0b7dd91d379bd45fL,0xd651e827ca63c581L,
        0x5c5d75f6509c104fL },
      { 0x7d5fc7381f2dc308L,0x20faa7bfd98454beL,0x95374beea517b031L,
        0xf036b9b1642692acL } },
    /* 22 << 7 */
    { { 0xc510610939842194L,0xb7e2353e49d05295L,0xfc8c1d5cefb42ee0L,
        0xe04884eb08ce811cL },
      { 0xf1f75d817419f40eL,0x5b0ac162a995c241L,0x120921bbc4c55646L,
        0x713520c28d33cf97L } },
    /* 23 << 7 */
    { { 0xb4a65a5ce98c5100L,0x6cec871d2ddd0f5aL,0x251f0b7f9ba2e78bL,
        0x224a8434ce3a2a5fL },
      { 0x26827f6125f5c46fL,0x6a22bedc48545ec0L,0x25ae5fa0b1bb5cdcL,
        0xd693682ffcb9b98fL } },
    /* 24 << 7 */
    { { 0x32027fe891e5d7d3L,0xf14b7d1773a07678L,0xf88497b3c0dfdd61L,
        0xf7c2eec02a8c4f48L },
      { 0xaa5573f43756e621L,0xc013a2401825b948L,0x1c03b34563878572L,
        0xa0472bea653a4184L } },
    /* 25 << 7 */
    { { 0xf4222e270ac69a80L,0x34096d25f51e54f6L,0x00a648cb8fffa591L,
        0x4e87acdc69b6527fL },
      { 0x0575e037e285ccb4L,0x188089e450ddcf52L,0xaa96c9a8870ff719L,
        0x74a56cd81fc7e369L } },
    /* 26 << 7 */
    { { 0x41d04ee21726931aL,0x0bbbb2c83660ecfdL,0xa6ef6de524818e18L,
        0xe421cc51e7d57887L },
      { 0xf127d208bea87be6L,0x16a475d3b1cdd682L,0x9db1b684439b63f7L,
        0x5359b3dbf0f113b6L } },
    /* 27 << 7 */
    { { 0xdfccf1de8bf06e31L,0x1fdf8f44dd383901L,0x10775cad5017e7d2L,
        0xdfc3a59758d11eefL },
      { 0x6ec9c8a0b1ecff10L,0xee6ed6cc28400549L,0xb5ad7bae1b4f8d73L,
        0x61b4f11de00aaab9L } },
    /* 28 << 7 */
    { { 0x7b32d69bd4eff2d7L,0x88ae67714288b60fL,0x159461b437a1e723L,
        0x1f3d4789570aae8cL },
      { 0x869118c07f9871daL,0x35fbda78f635e278L,0x738f3641e1541dacL,
        0x6794b13ac0dae45fL } },
    /* 29 << 7 */
    { { 0x065064ac09cc0917L,0x27c53729c68540fdL,0x0d2d4c8eef227671L,
        0xd23a9f80a1785a04L },
      { 0x98c5952852650359L,0xfa09ad0174a1acadL,0x082d5a290b55bf5cL,
        0xa40f1c67419b8084L } },
    /* 30 << 7 */
    { { 0x3a5c752edcc18770L,0x4baf1f2f8825c3a5L,0xebd63f7421b153edL,
        0xa2383e47b2f64723L },
      { 0xe7bf620a2646d19aL,0x56cb44ec03c83ffdL,0xaf7267c94f6be9f1L,
        0x8b2dfd7bc06bb5e9L } },
    /* 31 << 7 */
    { { 0xb87072f2a672c5c7L,0xeacb11c80d53c5e2L,0x22dac29dff435932L,
        0x37bdb99d4408693cL },
      { 0xf6e62fb62899c20fL,0x3535d512447ece24L,0xfbdc6b88ff577ce3L,
        0x726693bd190575f2L } },
    /* 32 << 7 */
    { { 0x6772b0e5ab4b35a2L,0x1d8b6001f5eeaacfL,0x728f7ce4795b9580L,
        0x4a20ed2a41fb81daL },
      { 0x9f685cd44fec01e6L,0x3ed7ddcca7ff50adL,0x460fd2640c2d97fdL,
        0x3a241426eb82f4f9L } },
    /* 33 << 7 */
    { { 0x17d1df2c6a8ea820L,0xb2b50d3bf22cc254L,0x03856cbab7291426L,
        0x87fd26ae04f5ee39L },
      { 0x9cb696cc02bee4baL,0x5312180406820fd6L,0xa5dfc2690212e985L,
        0x666f7ffa160f9a09L } },
    /* 34 << 7 */
    { { 0xc503cd33bccd9617L,0x365dede4ba7730a3L,0x798c63555ddb0786L,
        0xa6c3200efc9cd3bcL },
      { 0x060ffb2ce5e35efdL,0x99a4e25b5555a1c1L,0x11d95375f70b3751L,
        0x0a57354a160e1bf6L } },
    /* 35 << 7 */
    { { 0xecb3ae4bf8e4b065L,0x07a834c42e53022bL,0x1cd300b38692ed96L,
        0x16a6f79261ee14ecL },
      { 0x8f1063c66a8649edL,0xfbcdfcfe869f3e14L,0x2cfb97c100a7b3ecL,
        0xcea49b3c7130c2f1L } },
    /* 36 << 7 */
    { { 0x462d044fe9d96488L,0x4b53d52e8182a0c1L,0x84b6ddd30391e9e9L,
        0x80ab7b48b1741a09L },
      { 0xec0e15d427d3317fL,0x8dfc1ddb1a64671eL,0x93cc5d5fd49c5b92L,
        0xc995d53d3674a331L } },
    /* 37 << 7 */
    { { 0x302e41ec090090aeL,0x2278a0ccedb06830L,0x1d025932fbc99690L,
        0x0c32fbd2b80d68daL },
      { 0xd79146daf341a6c1L,0xae0ba1391bef68a0L,0xc6b8a5638d774b3aL,
        0x1cf307bd880ba4d7L } },
    /* 38 << 7 */
    { { 0xc033bdc719803511L,0xa9f97b3b8888c3beL,0x3d68aebc85c6d05eL,
        0xc3b88a9d193919ebL },
      { 0x2d300748c48b0ee3L,0x7506bc7c07a746c1L,0xfc48437c6e6d57f3L,
        0x5bd71587cfeaa91aL } },
    /* 39 << 7 */
    { { 0xa4ed0408c1bc5225L,0xd0b946db2719226dL,0x109ecd62758d2d43L,
        0x75c8485a2751759bL },
      { 0xb0b75f499ce4177aL,0x4fa61a1e79c10c3dL,0xc062d300a167fcd7L,
        0x4df3874c750f0fa8L } },
    /* 40 << 7 */
    { { 0x29ae2cf983dfedc9L,0xf84371348d87631aL,0xaf5717117429c8d2L,
        0x18d15867146d9272L },
      { 0x83053ecf69769bb7L,0xc55eb856c479ab82L,0x5ef7791c21b0f4b2L,
        0xaa5956ba3d491525L } },
    /* 41 << 7 */
    { { 0x407a96c29fe20ebaL,0xf27168bbe52a5ad3L,0x43b60ab3bf1d9d89L,
        0xe45c51ef710e727aL },
      { 0xdfca5276099b4221L,0x8dc6407c2557a159L,0x0ead833591035895L,
        0x0a9db9579c55dc32L } },
    /* 42 << 7 */
    { { 0xe40736d3df61bc76L,0x13a619c03f778cdbL,0x6dd921a4c56ea28fL,
        0x76a524332fa647b4L },
      { 0x23591891ac5bdc5dL,0xff4a1a72bac7dc01L,0x9905e26162df8453L,
        0x3ac045dfe63b265fL } },
    /* 43 << 7 */
    { { 0x8a3f341bad53dba7L,0x8ec269cc837b625aL,0xd71a27823ae31189L,
        0x8fb4f9a355e96120L },
      { 0x804af823ff9875cfL,0x23224f575d442a9bL,0x1c4d3b9eecc62679L,
        0x91da22fba0e7ddb1L } },
    /* 44 << 7 */
    { { 0xa370324d6c04a661L,0x9710d3b65e376d17L,0xed8c98f03044e357L,
        0xc364ebbe6422701cL },
      { 0x347f5d517733d61cL,0xd55644b9cea826c3L,0x80c6e0ad55a25548L,
        0x0aa7641d844220a7L } },
    /* 45 << 7 */
    { { 0x1438ec8131810660L,0x9dfa6507de4b4043L,0x10b515d8cc3e0273L,
        0x1b6066dd28d8cfb2L },
      { 0xd3b045919c9efebdL,0x425d4bdfa21c1ff4L,0x5fe5af19d57607d3L,
        0xbbf773f754481084L } },
    /* 46 << 7 */
    { { 0x8435bd6994b03ed1L,0xd9ad1de3634cc546L,0x2cf423fc00e420caL,
        0xeed26d80a03096ddL },
      { 0xd7f60be7a4db09d2L,0xf47f569d960622f7L,0xe5925fd77296c729L,
        0xeff2db2626ca2715L } },
    /* 47 << 7 */
    { { 0xa6fcd014b913e759L,0x53da47868ff4de93L,0x14616d79c32068e1L,
        0xb187d664ccdf352eL },
      { 0xf7afb6501dc90b59L,0x8170e9437daa1b26L,0xc8e3bdd8700c0a84L,
        0x6e8d345f6482bdfaL } },
    /* 48 << 7 */
    { { 0x84cfbfa1c5c5ea50L,0xd3baf14c67960681L,0x263984030dd50942L,
        0xe4b7839c4716a663L },
      { 0xd5f1f794e7de6dc0L,0x5cd0f4d4622aa7ceL,0x5295f3f159acfeecL,
        0x8d933552953e0607L } },
    /* 49 << 7 */
    { { 0xc7db8ec5776c5722L,0xdc467e622b5f290cL,0xd4297e704ff425a9L,
        0x4be924c10cf7bb72L },
      { 0x0d5dc5aea1892131L,0x8bf8a8e3a705c992L,0x73a0b0647a305ac5L,
        0x00c9ca4e9a8c77a8L } },
    /* 50 << 7 */
    { { 0x5dfee80f83774bddL,0x6313160285734485L,0xa1b524ae914a69a9L,
        0xebc2ffafd4e300d7L },
      { 0x52c93db77cfa46a5L,0x71e6161f21653b50L,0x3574fc57a4bc580aL,
        0xc09015dde1bc1253L } },
    /* 51 << 7 */
    { { 0x4b7b47b2d174d7aaL,0x4072d8e8f3a15d04L,0xeeb7d47fd6fa07edL,
        0x6f2b9ff9edbdafb1L },
      { 0x18c516153760fe8aL,0x7a96e6bff06c6c13L,0x4d7a04100ea2d071L,
        0xa1914e9b0be2a5ceL } },
    /* 52 << 7 */
    { { 0x5726e357d8a3c5cfL,0x1197ecc32abb2b13L,0x6c0d7f7f31ae88ddL,
        0x15b20d1afdbb3efeL },
      { 0xcd06aa2670584039L,0x2277c969a7dc9747L,0xbca695877855d815L,
        0x899ea2385188b32aL } },
    /* 53 << 7 */
    { { 0x37d9228b760c1c9dL,0xc7efbb119b5c18daL,0x7f0d1bc819f6dbc5L,
        0x4875384b07e6905bL },
      { 0xc7c50baa3ba8cd86L,0xb0ce40fbc2905de0L,0x708406737a231952L,
        0xa912a262cf43de26L } },
    /* 54 << 7 */
    { { 0x9c38ddcceb5b76c1L,0x746f528526fc0ab4L,0x52a63a50d62c269fL,
        0x60049c5599458621L },
      { 0xe7f48f823c2f7c9eL,0x6bd99043917d5cf3L,0xeb1317a88701f469L,
        0xbd3fe2ed9a449fe0L } },
    /* 55 << 7 */
    { { 0x421e79ca12ef3d36L,0x9ee3c36c3e7ea5deL,0xe48198b5cdff36f7L,
        0xaff4f967c6b82228L },
      { 0x15e19dd0c47adb7eL,0x45699b23032e7dfaL,0x40680c8b1fae026aL,
        0x5a347a48550dbf4dL } },
    /* 56 << 7 */
    { { 0xe652533b3cef0d7dL,0xd94f7b182bbb4381L,0x838752be0e80f500L,
        0x8e6e24889e9c9bfbL },
      { 0xc975169716caca6aL,0x866c49d838531ad9L,0xc917e2397151ade1L,
        0x2d016ec16037c407L } },
    /* 57 << 7 */
    { { 0xa407ccc900eac3f9L,0x835f6280e2ed4748L,0xcc54c3471cc98e0dL,
        0x0e969937dcb572ebL },
      { 0x1b16c8e88f30c9cbL,0xa606ae75373c4661L,0x47aa689b35502cabL,
        0xf89014ae4d9bb64fL } },
    /* 58 << 7 */
    { { 0x202f6a9c31c71f7bL,0x01f95aa3296ffe5cL,0x5fc0601453cec3a3L,
        0xeb9912375f498a45L },
      { 0xae9a935e5d91ba87L,0xc6ac62810b564a19L,0x8a8fe81c3bd44e69L,
        0x7c8b467f9dd11d45L } },
    /* 59 << 7 */
    { { 0xf772251fea5b8e69L,0xaeecb3bdc5b75fbcL,0x1aca3331887ff0e5L,
        0xbe5d49ff19f0a131L },
      { 0x582c13aae5c8646fL,0xdbaa12e820e19980L,0x8f40f31af7abbd94L,
        0x1f13f5a81dfc7663L } },
    /* 60 << 7 */
    { { 0x5d81f1eeaceb4fc0L,0x362560025e6f0f42L,0x4b67d6d7751370c8L,
        0x2608b69803e80589L },
      { 0xcfc0d2fc05268301L,0xa6943d3940309212L,0x192a90c21fd0e1c2L,
        0xb209f11337f1dc76L } },
    /* 61 << 7 */
    { { 0xefcc5e0697bf1298L,0xcbdb6730219d639eL,0xd009c116b81e8c6fL,
        0xa3ffdde31a7ce2e5L },
      { 0xc53fbaaaa914d3baL,0x836d500f88df85eeL,0xd98dc71b66ee0751L,
        0x5a3d7005714516fdL } },
    /* 62 << 7 */
    { { 0x21d3634d39eedbbaL,0x35cd2e680455a46dL,0xc8cafe65f9d7eb0cL,
        0xbda3ce9e00cefb3eL },
      { 0xddc17a602c9cf7a4L,0x01572ee47bcb8773L,0xa92b2b018c7548dfL,
        0x732fd309a84600e3L } },
    /* 63 << 7 */
    { { 0xe22109c716543a40L,0x9acafd36fede3c6cL,0xfb2068526824e614L,
        0x2a4544a9da25dca0L },
      { 0x2598526291d60b06L,0x281b7be928753545L,0xec667b1a90f13b27L,
        0x33a83aff940e2eb4L } },
    /* 64 << 7 */
    { { 0x80009862d5d721d5L,0x0c3357a35bd3a182L,0x27f3a83b7aa2cda4L,
        0xb58ae74ef6f83085L },
      { 0x2a911a812e6dad6bL,0xde286051f43d6c5bL,0x4bdccc41f996c4d8L,
        0xe7312ec00ae1e24eL } },
    /* 0 << 14 */
    { { 0x00, 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x00, 0x00 } },
    /* 1 << 14 */
    { { 0xf8d112e76e6485b3L,0x4d3e24db771c52f8L,0x48e3ee41684a2f6dL,
        0x7161957d21d95551L },
      { 0x19631283cdb12a6cL,0xbf3fa8822e50e164L,0xf6254b633166cc73L,
        0x3aefa7aeaee8cc38L } },
    /* 2 << 14 */
    { { 0x79b0fe623b36f9fdL,0x26543b23fde19fc0L,0x136e64a0958482efL,
        0x23f637719b095825L },
      { 0x14cfd596b6a1142eL,0x5ea6aac6335aac0bL,0x86a0e8bdf3081dd5L,
        0x5fb89d79003dc12aL } },
    /* 3 << 14 */
    { { 0xf615c33af72e34d4L,0x0bd9ea40110eec35L,0x1c12bc5bc1dea34eL,
        0x686584c949ae4699L },
      { 0x13ad95d38c97b942L,0x4609561a4e5c7562L,0x9e94a4aef2737f89L,
        0xf57594c6371c78b6L } },
    /* 4 << 14 */
    { { 0x0f0165fce3779ee3L,0xe00e7f9dbd495d9eL,0x1fa4efa220284e7aL,
        0x4564bade47ac6219L },
      { 0x90e6312ac4708e8eL,0x4f5725fba71e9adfL,0xe95f55ae3d684b9fL,
        0x47f7ccb11e94b415L } },
    /* 5 << 14 */
    { { 0x7322851b8d946581L,0xf0d13133bdf4a012L,0xa3510f696584dae0L,
        0x03a7c1713c9f6c6dL },
      { 0x5be97f38e475381aL,0xca1ba42285823334L,0xf83cc5c70be17ddaL,
        0x158b14940b918c0fL } },
    /* 6 << 14 */
    { { 0xda3a77e5522e6b69L,0x69c908c3bbcd6c18L,0x1f1b9e48d924fd56L,
        0x37c64e36aa4bb3f7L },
      { 0x5a4fdbdfee478d7dL,0xba75c8bc0193f7a0L,0x84bc1e8456cd16dfL,
        0x1fb08f0846fad151L } },
    /* 7 << 14 */
    { { 0x8a7cabf9842e9f30L,0xa331d4bf5eab83afL,0xd272cfba017f2a6aL,
        0x27560abc83aba0e3L },
      { 0x94b833870e3a6b75L,0x25c6aea26b9f50f5L,0x803d691db5fdf6d0L,
        0x03b77509e6333514L } },
    /* 8 << 14 */
    { { 0x3617890361a341c1L,0x3604dc600cfd6142L,0x022295eb8533316cL,
        0x3dbde4ac44af2922L },
      { 0x898afc5d1c7eef69L,0x58896805d14f4fa1L,0x05002160203c21caL,
        0x6f0d1f3040ef730bL } },
    /* 9 << 14 */
    { { 0x8e8c44d4196224f8L,0x75a4ab95374d079dL,0x79085ecc7d48f123L,
        0x56f04d311bf65ad8L },
      { 0xe220bf1cbda602b2L,0x73ee1742f9612c69L,0x76008fc8084fd06bL,
        0x4000ef9ff11380d1L } },
    /* 10 << 14 */
    { { 0x48201b4b12cfe297L,0x3eee129c292f74e5L,0xe1fe114ec9e874e8L,
        0x899b055c92c5fc41L },
      { 0x4e477a643a39c8cfL,0x82f09efe78963cc9L,0x6fd3fd8fd333f863L,
        0x85132b2adc949c63L } },
    /* 11 << 14 */
    { { 0x7e06a3ab516eb17bL,0x73bec06fd2c7372bL,0xe4f74f55ba896da6L,
        0xbb4afef88e9eb40fL },
      { 0x2d75bec8e61d66b0L,0x02bda4b4ef29300bL,0x8bbaa8de026baa5aL,
        0xff54befda07f4440L } },
    /* 12 << 14 */
    { { 0xbd9b8b1dbe7a2af3L,0xec51caa94fb74a72L,0xb9937a4b63879697L,
        0x7c9a9d20ec2687d5L },
      { 0x1773e44f6ef5f014L,0x8abcf412e90c6900L,0x387bd0228142161eL,
        0x50393755fcb6ff2aL } },
    /* 13 << 14 */
    { { 0x9813fd56ed6def63L,0x53cf64827d53106cL,0x991a35bd431f7ac1L,
        0xf1e274dd63e65fafL },
      { 0xf63ffa3c44cc7880L,0x411a426b7c256981L,0xb698b9fd93a420e0L,
        0x89fdddc0ae53f8feL } },
    /* 14 << 14 */
    { { 0x766e072232398baaL,0x205fee425cfca031L,0xa49f53417a029cf2L,
        0xa88c68b84023890dL },
      { 0xbc2750417337aaa8L,0x9ed364ad0eb384f4L,0xe0816f8529aba92fL,
        0x2e9e194104e38a88L } },
    /* 15 << 14 */
    { { 0x57eef44a3dafd2d5L,0x35d1fae597ed98d8L,0x50628c092307f9b1L,
        0x09d84aaed6cba5c6L },
      { 0x67071bc788aaa691L,0x2dea57a9afe6cb03L,0xdfe11bb43d78ac01L,
        0x7286418c7fd7aa51L } },
    /* 16 << 14 */
    { { 0xfabf770977f7195aL,0x8ec86167adeb838fL,0xea1285a8bb4f012dL,
        0xd68835039a3eab3fL },
      { 0xee5d24f8309004c2L,0xa96e4b7613ffe95eL,0x0cdffe12bd223ea4L,
        0x8f5c2ee5b6739a53L } },
    /* 17 << 14 */
    { { 0x5cb4aaa5dd968198L,0xfa131c5272413a6cL,0x53d46a909536d903L,
        0xb270f0d348606d8eL },
      { 0x518c7564a053a3bcL,0x088254b71a86caefL,0xb3ba8cb40ab5efd0L,
        0x5c59900e4605945dL } },
    /* 18 << 14 */
    { { 0xecace1dda1887395L,0x40960f36932a65deL,0x9611ff5c3aa95529L,
        0xc58215b07c1e5a36L },
      { 0xd48c9b58f0e1a524L,0xb406856bf590dfb8L,0xc7605e049cd95662L,
        0x0dd036eea33ecf82L } },
    /* 19 << 14 */
    { { 0xa50171acc33156b3L,0xf09d24ea4a80172eL,0x4e1f72c676dc8eefL,
        0xe60caadc5e3d44eeL },
      { 0x006ef8a6979b1d8fL,0x60908a1c97788d26L,0x6e08f95b266feec0L,
        0x618427c222e8c94eL } },
    /* 20 << 14 */
    { { 0x3d61333959145a65L,0xcd9bc368fa406337L,0x82d11be32d8a52a0L,
        0xf6877b2797a1c590L },
      { 0x837a819bf5cbdb25L,0x2a4fd1d8de090249L,0x622a7de774990e5fL,
        0x840fa5a07945511bL } },
    /* 21 << 14 */
    { { 0x30b974be6558842dL,0x70df8c6417f3d0a6L,0x7c8035207542e46dL,
        0x7251fe7fe4ecc823L },
      { 0xe59134cb5e9aac9aL,0x11bb0934f0045d71L,0x53e5d9b5dbcb1d4eL,
        0x8d97a90592defc91L } },
    /* 22 << 14 */
    { { 0xfe2893277946d3f9L,0xe132bd2407472273L,0xeeeb510c1eb6ae86L,
        0x777708c5f0595067L },
      { 0x18e2c8cd1297029eL,0x2c61095cbbf9305eL,0xe466c2586b85d6d9L,
        0x8ac06c36da1ea530L } },
    /* 23 << 14 */
    { { 0xa365dc39a1304668L,0xe4a9c88507f89606L,0x65a4898facc7228dL,
        0x3e2347ff84ca8303L },
      { 0xa5f6fb77ea7d23a3L,0x2fac257d672a71cdL,0x6908bef87e6a44d3L,
        0x8ff87566891d3d7aL } },
    /* 24 << 14 */
    { { 0xe58e90b36b0cf82eL,0x6438d2462615b5e7L,0x07b1f8fc669c145aL,
        0xb0d8b2da36f1e1cbL },
      { 0x54d5dadbd9184c4dL,0x3dbb18d5f93d9976L,0x0a3e0f56d1147d47L,
        0x2afa8c8da0a48609L } },
    /* 25 << 14 */
    { { 0x275353e8bc36742cL,0x898f427eeea0ed90L,0x26f4947e3e477b00L,
        0x8ad8848a308741e3L },
      { 0x6c703c38d74a2a46L,0x5e3e05a99ba17ba2L,0xc1fa6f664ab9a9e4L,
        0x474a2d9a3841d6ecL } },
    /* 26 << 14 */
    { { 0x871239ad653ae326L,0x14bcf72aa74cbb43L,0x8737650e20d4c083L,
        0x3df86536110ed4afL },
      { 0xd2d86fe7b53ca555L,0x688cb00dabd5d538L,0xcf81bda31ad38468L,
        0x7ccfe3ccf01167b6L } },
    /* 27 << 14 */
    { { 0xcf4f47e06c4c1fe6L,0x557e1f1a298bbb79L,0xf93b974f30d45a14L,
        0x174a1d2d0baf97c4L },
      { 0x7a003b30c51fbf53L,0xd8940991ee68b225L,0x5b0aa7b71c0f4173L,
        0x975797c9a20a7153L } },
    /* 28 << 14 */
    { { 0x26e08c07e3533d77L,0xd7222e6a2e341c99L,0x9d60ec3d8d2dc4edL,
        0xbdfe0d8f7c476cf8L },
      { 0x1fe59ab61d056605L,0xa9ea9df686a8551fL,0x8489941e47fb8d8cL,
        0xfeb874eb4a7f1b10L } },
    /* 29 << 14 */
    { { 0xfe5fea867ee0d98fL,0x201ad34bdbf61864L,0x45d8fe4737c031d4L,
        0xd5f49fae795f0822L },
      { 0xdb0fb291c7f4a40cL,0x2e69d9c1730ddd92L,0x754e105449d76987L,
        0x8a24911d7662db87L } },
    /* 30 << 14 */
    { { 0x61fc181060a71676L,0xe852d1a8f66a8ad1L,0x172bbd656417231eL,
        0x0d6de7bd3babb11fL },
      { 0x6fde6f88c8e347f8L,0x1c5875479bd99cc3L,0x78e54ed034076950L,
        0x97f0f334796e83baL } },
    /* 31 << 14 */
    { { 0xe4dbe1ce4924867aL,0xbd5f51b060b84917L,0x375300403cb09a79L,
        0xdb3fe0f8ff1743d8L },
      { 0xed7894d8556fa9dbL,0xfa26216923412fbfL,0x563be0dbba7b9291L,
        0x6ca8b8c00c9fb234L } },
    /* 32 << 14 */
    { { 0xed406aa9bd763802L,0xc21486a065303da1L,0x61ae291ec7e62ec4L,
        0x622a0492df99333eL },
      { 0x7fd80c9dbb7a8ee0L,0xdc2ed3bc6c01aedbL,0x35c35a1208be74ecL,
        0xd540cb1a469f671fL } },
    /* 33 << 14 */
    { { 0xd16ced4ecf84f6c7L,0x8561fb9c2d090f43L,0x7e693d796f239db4L,
        0xa736f92877bd0d94L },
      { 0x07b4d9292c1950eeL,0xda17754356dc11b3L,0xa5dfbbaa7a6a878eL,
        0x1c70cb294decb08aL } },
    /* 34 << 14 */
    { { 0xfba28c8b6f0f7c50L,0xa8eba2b8854dcc6dL,0x5ff8e89a36b78642L,
        0x070c1c8ef6873adfL },
      { 0xbbd3c3716484d2e4L,0xfb78318f0d414129L,0x2621a39c6ad93b0bL,
        0x979d74c2a9e917f7L } },
    /* 35 << 14 */
    { { 0xfc19564761fb0428L,0x4d78954abee624d4L,0xb94896e0b8ae86fdL,
        0x6667ac0cc91c8b13L },
      { 0x9f18051243bcf832L,0xfbadf8b7a0010137L,0xc69b4089b3ba8aa7L,
        0xfac4bacde687ce85L } },
    /* 36 << 14 */
    { { 0x9164088d977eab40L,0x51f4c5b62760b390L,0xd238238f340dd553L,
        0x358566c3db1d31c9L },
      { 0x3a5ad69e5068f5ffL,0xf31435fcdaff6b06L,0xae549a5bd6debff0L,
        0x59e5f0b775e01331L } },
    /* 37 << 14 */
    { { 0x5d492fb898559acfL,0x96018c2e4db79b50L,0x55f4a48f609f66aaL,
        0x1943b3af4900a14fL },
      { 0xc22496df15a40d39L,0xb2a446844c20f7c5L,0x76a35afa3b98404cL,
        0xbec75725ff5d1b77L } },
    /* 38 << 14 */
    { { 0xb67aa163bea06444L,0x27e95bb2f724b6f2L,0x3c20e3e9d238c8abL,
        0x1213754eddd6ae17L },
      { 0x8c431020716e0f74L,0x6679c82effc095c2L,0x2eb3adf4d0ac2932L,
        0x2cc970d301bb7a76L } },
    /* 39 << 14 */
    { { 0x70c71f2f740f0e66L,0x545c616b2b6b23ccL,0x4528cfcbb40a8bd7L,
        0xff8396332ab27722L },
      { 0x049127d9025ac99aL,0xd314d4a02b63e33bL,0xc8c310e728d84519L,
        0x0fcb8983b3bc84baL } },
    /* 40 << 14 */
    { { 0x2cc5226138634818L,0x501814f4b44c2e0bL,0xf7e181aa54dfdba3L,
        0xcfd58ff0e759718cL },
      { 0xf90cdb14d3b507a8L,0x57bd478ec50bdad8L,0x29c197e250e5f9aaL,
        0x4db6eef8e40bc855L } },
    /* 41 << 14 */
    { { 0x2cc8f21ad1fc0654L,0xc71cc96381269d73L,0xecfbb204077f49f9L,
        0xdde92571ca56b793L },
      { 0x9abed6a3f97ad8f7L,0xe6c19d3f924de3bdL,0x8dce92f4a140a800L,
        0x85f44d1e1337af07L } },
    /* 42 << 14 */
    { { 0x5953c08b09d64c52L,0xa1b5e49ff5df9749L,0x336a8fb852735f7dL,
        0xb332b6db9add676bL },
      { 0x558b88a0b4511aa4L,0x09788752dbd5cc55L,0x16b43b9cd8cd52bdL,
        0x7f0bc5a0c2a2696bL } },
    /* 43 << 14 */
    { { 0x146e12d4c11f61efL,0x9ce107543a83e79eL,0x08ec73d96cbfca15L,
        0x09ff29ad5b49653fL },
      { 0xe31b72bde7da946eL,0xebf9eb3bee80a4f2L,0xd1aabd0817598ce4L,
        0x18b5fef453f37e80L } },
    /* 44 << 14 */
    { { 0xd5d5cdd35958cd79L,0x3580a1b51d373114L,0xa36e4c91fa935726L,
        0xa38c534def20d760L },
      { 0x7088e40a2ff5845bL,0xe5bb40bdbd78177fL,0x4f06a7a8857f9920L,
        0xe3cc3e50e968f05dL } },
    /* 45 << 14 */
    { { 0x1d68b7fee5682d26L,0x5206f76faec7f87cL,0x41110530041951abL,
        0x58ec52c1d4b5a71aL },
      { 0xf3488f990f75cf9aL,0xf411951fba82d0d5L,0x27ee75be618895abL,
        0xeae060d46d8aab14L } },
    /* 46 << 14 */
    { { 0x9ae1df737fb54dc2L,0x1f3e391b25963649L,0x242ec32afe055081L,
        0x5bd450ef8491c9bdL },
      { 0x367efc67981eb389L,0xed7e19283a0550d5L,0x362e776bab3ce75cL,
        0xe890e3081f24c523L } },
    /* 47 << 14 */
    { { 0xb961b682feccef76L,0x8b8e11f58bba6d92L,0x8f2ccc4c2b2375c4L,
        0x0d7f7a52e2f86cfaL },
      { 0xfd94d30a9efe5633L,0x2d8d246b5451f934L,0x2234c6e3244e6a00L,
        0xde2b5b0dddec8c50L } },
    /* 48 << 14 */
    { { 0x2ce53c5abf776f5bL,0x6f72407160357b05L,0xb259371771bf3f7aL,
        0x87d2501c440c4a9fL },
      { 0x440552e187b05340L,0xb7bf7cc821624c32L,0x4155a6ce22facddbL,
        0x5a4228cb889837efL } },
    /* 49 << 14 */
    { { 0xef87d6d6fd4fd671L,0xa233687ec2daa10eL,0x7562224403c0eb96L,
        0x7632d1848bf19be6L },
      { 0x05d0f8e940735ff4L,0x3a3e6e13c00931f1L,0x31ccde6adafe3f18L,
        0xf381366acfe51207L } },
    /* 50 << 14 */
    { { 0x24c222a960167d92L,0x62f9d6f87529f18cL,0x412397c00353b114L,
        0x334d89dcef808043L },
      { 0xd9ec63ba2a4383ceL,0xcec8e9375cf92ba0L,0xfb8b4288c8be74c0L,
        0x67d6912f105d4391L } },
    /* 51 << 14 */
    { { 0x7b996c461b913149L,0x36aae2ef3a4e02daL,0xb68aa003972de594L,
        0x284ec70d4ec6d545L },
      { 0xf3d2b2d061391d54L,0x69c5d5d6fe114e92L,0xbe0f00b5b4482dffL,
        0xe1596fa5f5bf33c5L } },
    /* 52 << 14 */
    { { 0x10595b5696a71cbaL,0x944938b2fdcadeb7L,0xa282da4cfccd8471L,
        0x98ec05f30d37bfe1L },
      { 0xe171ce1b0698304aL,0x2d69144421bdf79bL,0xd0cd3b741b21dec1L,
        0x712ecd8b16a15f71L } },
    /* 53 << 14 */
    { { 0x8d4c00a700fd56e1L,0x02ec9692f9527c18L,0x21c449374a3e42e1L,
        0x9176fbab1392ae0aL },
      { 0x8726f1ba44b7b618L,0xb4d7aae9f1de491cL,0xf91df7b907b582c0L,
        0x7e116c30ef60aa3aL } },
    /* 54 << 14 */
    { { 0x99270f81466265d7L,0xb15b6fe24df7adf0L,0xfe33b2d3f9738f7fL,
        0x48553ab9d6d70f95L },
      { 0x2cc72ac8c21e94dbL,0x795ac38dbdc0bbeeL,0x0a1be4492e40478fL,
        0x81bd3394052bde55L } },
    /* 55 << 14 */
    { { 0x63c8dbe956b3c4f2L,0x017a99cf904177ccL,0x947bbddb4d010fc1L,
        0xacf9b00bbb2c9b21L },
      { 0x2970bc8d47173611L,0x1a4cbe08ac7d756fL,0x06d9f4aa67d541a2L,
        0xa3e8b68959c2cf44L } },
    /* 56 << 14 */
    { { 0xaad066da4d88f1ddL,0xc604f1657ad35deaL,0x7edc07204478ca67L,
        0xa10dfae0ba02ce06L },
      { 0xeceb1c76af36f4e4L,0x994b2292af3f8f48L,0xbf9ed77b77c8a68cL,
        0x74f544ea51744c9dL } },
    /* 57 << 14 */
    { { 0x82d05bb98113a757L,0x4ef2d2b48a9885e4L,0x1e332be51aa7865fL,
        0x22b76b18290d1a52L },
      { 0x308a231044351683L,0x9d861896a3f22840L,0x5959ddcd841ed947L,
        0x0def0c94154b73bfL } },
    /* 58 << 14 */
    { { 0xf01054174c7c15e0L,0x539bfb023a277c32L,0xe699268ef9dccf5fL,
        0x9f5796a50247a3bdL },
      { 0x8b839de84f157269L,0xc825c1e57a30196bL,0x6ef0aabcdc8a5a91L,
        0xf4a8ce6c498b7fe6L } },
    /* 59 << 14 */
    { { 0x1cce35a770cbac78L,0x83488e9bf6b23958L,0x0341a070d76cb011L,
        0xda6c9d06ae1b2658L },
      { 0xb701fb30dd648c52L,0x994ca02c52fb9fd1L,0x069331176f563086L,
        0x3d2b810017856babL } },
    /* 60 << 14 */
    { { 0xe89f48c85963a46eL,0x658ab875a99e61c7L,0x6e296f874b8517b4L,
        0x36c4fcdcfc1bc656L },
      { 0xde5227a1a3906defL,0x9fe95f5762418945L,0x20c91e81fdd96cdeL,
        0x5adbe47eda4480deL } },
    /* 61 << 14 */
    { { 0xa009370f396de2b6L,0x98583d4bf0ecc7bdL,0xf44f6b57e51d0672L,
        0x03d6b078556b1984L },
      { 0x27dbdd93b0b64912L,0x9b3a343415687b09L,0x0dba646151ec20a9L,
        0xec93db7fff28187cL } },
    /* 62 << 14 */
    { { 0x00ff8c2466e48bddL,0x2514f2f911ccd78eL,0xeba11f4fe1250603L,
        0x8a22cd41243fa156L },
      { 0xa4e58df4b283e4c6L,0x78c298598b39783fL,0x5235aee2a5259809L,
        0xc16284b50e0227ddL } },
    /* 63 << 14 */
    { { 0xa5f579161338830dL,0x6d4b8a6bd2123fcaL,0x236ea68af9c546f8L,
        0xc1d36873fa608d36L },
      { 0xcd76e4958d436d13L,0xd4d9c2218fb080afL,0x665c1728e8ad3fb5L,
        0xcf1ebe4db3d572e0L } },
    /* 64 << 14 */
    { { 0xa7a8746a584c5e20L,0x267e4ea1b9dc7035L,0x593a15cfb9548c9bL,
        0x5e6e21354bd012f3L },
      { 0xdf31cc6a8c8f936eL,0x8af84d04b5c241dcL,0x63990a6f345efb86L,
        0x6fef4e61b9b962cbL } },
    /* 0 << 21 */
    { { 0x00, 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x00, 0x00 } },
    /* 1 << 21 */
    { { 0xf6368f0925722608L,0x131260db131cf5c6L,0x40eb353bfab4f7acL,
        0x85c7888037eee829L },
      { 0x4c1581ffc3bdf24eL,0x5bff75cbf5c3c5a8L,0x35e8c83fa14e6f40L,
        0xb81d1c0f0295e0caL } },
    /* 2 << 21 */
    { { 0xfcde7cc8f43a730fL,0xe89b6f3c33ab590eL,0xc823f529ad03240bL,
        0x82b79afe98bea5dbL },
      { 0x568f2856962fe5deL,0x0c590adb60c591f3L,0x1fc74a144a28a858L,
        0x3b662498b3203f4cL } },
    /* 3 << 21 */
    { { 0x91e3cf0d6c39765aL,0xa2db3acdac3cca0bL,0x288f2f08cb953b50L,
        0x2414582ccf43cf1aL },
      { 0x8dec8bbc60eee9a8L,0x54c79f02729aa042L,0xd81cd5ec6532f5d5L,
        0xa672303acf82e15fL } },
    /* 4 << 21 */
    { { 0x376aafa8719c0563L,0xcd8ad2dcbc5fc79fL,0x303fdb9fcb750cd3L,
        0x14ff052f4418b08eL },
      { 0xf75084cf3e2d6520L,0x7ebdf0f8144ed509L,0xf43bf0f2d3f25b98L,
        0x86ad71cfa354d837L } },
    /* 5 << 21 */
    { { 0xb827fe9226f43572L,0xdfd3ab5b5d824758L,0x315dd23a539094c1L,
        0x85c0e37a66623d68L },
      { 0x575c79727be19ae0L,0x616a3396df0d36b5L,0xa1ebb3c826b1ff7eL,
        0x635b9485140ad453L } },
    /* 6 << 21 */
    { { 0x92bf3cdada430c0bL,0x4702850e3a96dac6L,0xc91cf0a515ac326aL,
        0x95de4f49ab8c25e4L },
      { 0xb01bad09e265c17cL,0x24e45464087b3881L,0xd43e583ce1fac5caL,
        0xe17cb3186ead97a6L } },
    /* 7 << 21 */
    { { 0x6cc3924374dcec46L,0x33cfc02d54c2b73fL,0x82917844f26cd99cL,
        0x8819dd95d1773f89L },
      { 0x09572aa60871f427L,0x8e0cf365f6f01c34L,0x7fa52988bff1f5afL,
        0x4eb357eae75e8e50L } },
    /* 8 << 21 */
    { { 0xd9d0c8c4868af75dL,0xd7325cff45c8c7eaL,0xab471996cc81ecb0L,
        0xff5d55f3611824edL },
      { 0xbe3145411977a0eeL,0x5085c4c5722038c6L,0x2d5335bff94bb495L,
        0x894ad8a6c8e2a082L } },
    /* 9 << 21 */
    { { 0x5c3e2341ada35438L,0xf4a9fc89049b8c4eL,0xbeeb355a9f17cf34L,
        0x3f311e0e6c91fe10L },
      { 0xc2d2003892ab9891L,0x257bdcc13e8ce9a9L,0x1b2d978988c53beeL,
        0x927ce89acdba143aL } },
    /* 10 << 21 */
    { { 0xb0a32cca523db280L,0x5c889f8a50d43783L,0x503e04b34897d16fL,
        0x8cdb6e7808f5f2e8L },
      { 0x6ab91cf0179c8e74L,0xd8874e5248211d60L,0xf948d4d5ea851200L,
        0x4076d41ee6f9840aL } },
    /* 11 << 21 */
    { { 0xc20e263c47b517eaL,0x79a448fd30685e5eL,0xe55f6f78f90631a0L,
        0x88a790b1a79e6346L },
      { 0x62160c7d80969fe8L,0x54f92fd441491bb9L,0xa6645c235c957526L,
        0xf44cc5aebea3ce7bL } },
    /* 12 << 21 */
    { { 0xf76283278b1e68b7L,0xc731ad7a303f29d3L,0xfe5a9ca957d03ecbL,
        0x96c0d50c41bc97a7L },
      { 0xc4669fe79b4f7f24L,0xfdd781d83d9967efL,0x7892c7c35d2c208dL,
        0x8bf64f7cae545cb3L } },
    /* 13 << 21 */
    { { 0xc01f862c467be912L,0xf4c85ee9c73d30ccL,0x1fa6f4be6ab83ec7L,
        0xa07a3c1c4e3e3cf9L },
      { 0x87f8ef450c00beb3L,0x30e2c2b3000d4c3eL,0x1aa00b94fe08bf5bL,
        0x32c133aa9224ef52L } },
    /* 14 << 21 */
    { { 0x38df16bb32e5685dL,0x68a9e06958e6f544L,0x495aaff7cdc5ebc6L,
        0xf894a645378b135fL },
      { 0xf316350a09e27ecfL,0xeced201e58f7179dL,0x2eec273ce97861baL,
        0x47ec2caed693be2eL } },
    /* 15 << 21 */
    { { 0xfa4c97c4f68367ceL,0xe4f47d0bbe5a5755L,0x17de815db298a979L,
        0xd7eca659c177dc7dL },
      { 0x20fdbb7149ded0a3L,0x4cb2aad4fb34d3c5L,0x2cf31d2860858a33L,
        0x3b6873efa24aa40fL } },
    /* 16 << 21 */
    { { 0x540234b22c11bb37L,0x2d0366dded4c74a3L,0xf9a968daeec5f25dL,
        0x3660106867b63142L },
      { 0x07cd6d2c68d7b6d4L,0xa8f74f090c842942L,0xe27514047768b1eeL,
        0x4b5f7e89fe62aee4L } },
    /* 17 << 21 */
    { { 0xc6a7717789070d26L,0xa1f28e4edd1c8bc7L,0xea5f4f06469e1f17L,
        0x78fc242afbdb78e0L },
      { 0xc9c7c5928b0588f1L,0xb6b7a0fd1535921eL,0xcc5bdb91bde5ae35L,
        0xb42c485e12ff1864L } },
    /* 18 << 21 */
    { { 0xa1113e13dbab98aaL,0xde9d469ba17b1024L,0x23f48b37c0462d3aL,
        0x3752e5377c5c078dL },
      { 0xe3a86add15544eb9L,0xf013aea780fba279L,0x8b5bb76cf22001b5L,
        0xe617ba14f02891abL } },
    /* 19 << 21 */
    { { 0xd39182a6936219d3L,0x5ce1f194ae51cb19L,0xc78f8598bf07a74cL,
        0x6d7158f222cbf1bcL },
      { 0x3b846b21e300ce18L,0x35fba6302d11275dL,0x5fe25c36a0239b9bL,
        0xd8beb35ddf05d940L } },
    /* 20 << 21 */
    { { 0x4db02bb01f7e320dL,0x0641c3646da320eaL,0x6d95fa5d821389a3L,
        0x926997488fcd8e3dL },
      { 0x316fef17ceb6c143L,0x67fcb841d933762bL,0xbb837e35118b17f8L,
        0x4b92552f9fd24821L } },
    /* 21 << 21 */
    { { 0xae6bc70e46aca793L,0x1cf0b0e4e579311bL,0x8dc631be5802f716L,
        0x099bdc6fbddbee4dL },
      { 0xcc352bb20caf8b05L,0xf74d505a72d63df2L,0xb9876d4b91c4f408L,
        0x1ce184739e229b2dL } },
    /* 22 << 21 */
    { { 0x4950759783abdb4aL,0x850fbcb6dee84b18L,0x6325236e609e67dcL,
        0x04d831d99336c6d8L },
      { 0x8deaae3bfa12d45dL,0xe425f8ce4746e246L,0x8004c17524f5f31eL,
        0xaca16d8fad62c3b7L } },
    /* 23 << 21 */
    { { 0x0dc15a6a9152f934L,0xf1235e5ded0e12c1L,0xc33c06ecda477dacL,
        0x76be8732b2ea0006L },
      { 0xcf3f78310c0cd313L,0x3c524553a614260dL,0x31a756f8cab22d15L,
        0x03ee10d177827a20L } },
    /* 24 << 21 */
    { { 0xd1e059b21994ef20L,0x2a653b69638ae318L,0x70d5eb582f699010L,
        0x279739f709f5f84aL },
      { 0x5da4663c8b799336L,0xfdfdf14d203c37ebL,0x32d8a9dca1dbfb2dL,
        0xab40cff077d48f9bL } },
    /* 25 << 21 */
    { { 0xc018b383d20b42d5L,0xf9a810ef9f78845fL,0x40af3753bdba9df0L,
        0xb90bdcfc131dfdf9L },
      { 0x18720591f01ab782L,0xc823f2116af12a88L,0xa51b80f30dc14401L,
        0xde248f77fb2dfbe3L } },
    /* 26 << 21 */
    { { 0xef5a44e50cafe751L,0x73997c9cd4dcd221L,0x32fd86d1de854024L,
        0xd5b53adca09b84bbL },
      { 0x008d7a11dcedd8d1L,0x406bd1c874b32c84L,0x5d4472ff05dde8b1L,
        0x2e25f2cdfce2b32fL } },
    /* 27 << 21 */
    { { 0xbec0dd5e29dfc254L,0x4455fcf62b98b267L,0x0b4d43a5c72df2adL,
        0xea70e6be48a75397L },
      { 0x2aad61695820f3bfL,0xf410d2dd9e37f68fL,0x70fb7dba7be5ac83L,
        0x636bb64536ec3eecL } },
    /* 28 << 21 */
    { { 0x27104ea39754e21cL,0xbc87a3e68d63c373L,0x483351d74109db9aL,
        0x0fa724e360134da7L },
      { 0x9ff44c29b0720b16L,0x2dd0cf1306aceeadL,0x5942758ce26929a6L,
        0x96c5db92b766a92bL } },
    /* 29 << 21 */
    { { 0xcec7d4c05f18395eL,0xd3f227441f80d032L,0x7a68b37acb86075bL,
        0x074764ddafef92dbL },
      { 0xded1e9507bc7f389L,0xc580c850b9756460L,0xaeeec2a47da48157L,
        0x3f0b4e7f82c587b3L } },
    /* 30 << 21 */
    { { 0x231c6de8a9f19c53L,0x5717bd736974e34eL,0xd9e1d216f1508fa9L,
        0x9f112361dadaa124L },
      { 0x80145e31823b7348L,0x4dd8f0d5ac634069L,0xe3d82fc72297c258L,
        0x276fcfee9cee7431L } },
    /* 31 << 21 */
    { { 0x8eb61b5e2bc0aea9L,0x4f668fd5de329431L,0x03a32ab138e4b87eL,
        0xe137451773d0ef0bL },
      { 0x1a46f7e6853ac983L,0xc3bdf42e68e78a57L,0xacf207852ea96dd1L,
        0xa10649b9f1638460L } },
    /* 32 << 21 */
    { { 0xf2369f0b879fbbedL,0x0ff0ae86da9d1869L,0x5251d75956766f45L,
        0x4984d8c02be8d0fcL },
      { 0x7ecc95a6d21008f0L,0x29bd54a03a1a1c49L,0xab9828c5d26c50f3L,
        0x32c0087c51d0d251L } },
    /* 33 << 21 */
    { { 0x9bac3ce60c1cdb26L,0xcd94d947557ca205L,0x1b1bd5989db1fdcdL,
        0x0eda0108a3d8b149L },
      { 0x9506661056152fccL,0xc2f037e6e7192b33L,0xdeffb41ac92e05a4L,
        0x1105f6c2c2f6c62eL } },
    /* 34 << 21 */
    { { 0x68e735008733913cL,0xcce861633f3adc40L,0xf407a94238a278e9L,
        0xd13c1b9d2ab21292L },
      { 0x93ed7ec71c74cf5cL,0x8887dc48f1a4c1b4L,0x3830ff304b3a11f1L,
        0x358c5a3c58937cb6L } },
    /* 35 << 21 */
    { { 0x027dc40489022829L,0x40e939773b798f79L,0x90ad333738be6eadL,
        0x9c23f6bcf34c0a5dL },
      { 0xd1711a35fbffd8bbL,0x60fcfb491949d3ddL,0x09c8ef4b7825d93aL,
        0x24233cffa0a8c968L } },
    /* 36 << 21 */
    { { 0x67ade46ce6d982afL,0xebb6bf3ee7544d7cL,0xd6b9ba763d8bd087L,
        0x46fe382d4dc61280L },
      { 0xbd39a7e8b5bdbd75L,0xab381331b8f228feL,0x0709a77cce1c4300L,
        0x6a247e56f337ceacL } },
    /* 37 << 21 */
    { { 0x8f34f21b636288beL,0x9dfdca74c8a7c305L,0x6decfd1bea919e04L,
        0xcdf2688d8e1991f8L },
      { 0xe607df44d0f8a67eL,0xd985df4b0b58d010L,0x57f834c50c24f8f4L,
        0xe976ef56a0bf01aeL } },
    /* 38 << 21 */
    { { 0x536395aca1c32373L,0x351027aa734c0a13L,0xd2f1b5d65e6bd5bcL,
        0x2b539e24223debedL },
      { 0xd4994cec0eaa1d71L,0x2a83381d661dcf65L,0x5f1aed2f7b54c740L,
        0x0bea3fa5d6dda5eeL } },
    /* 39 << 21 */
    { { 0x9d4fb68436cc6134L,0x8eb9bbf3c0a443ddL,0xfc500e2e383b7d2aL,
        0x7aad621c5b775257L },
      { 0x69284d740a8f7cc0L,0xe820c2ce07562d65L,0xbf9531b9499758eeL,
        0x73e95ca56ee0cc2dL } },
    /* 40 << 21 */
    { { 0xf61790abfbaf50a5L,0xdf55e76b684e0750L,0xec516da7f176b005L,
        0x575553bb7a2dddc7L },
      { 0x37c87ca3553afa73L,0x315f3ffc4d55c251L,0xe846442aaf3e5d35L,
        0x61b911496495ff28L } },
    /* 41 << 21 */
    { { 0x23cc95d3fa326dc3L,0x1df4da1f18fc2ceaL,0x24bf9adcd0a37d59L,
        0xb6710053320d6e1eL },
      { 0x96f9667e618344d1L,0xcc7ce042a06445afL,0xa02d8514d68dbc3aL,
        0x4ea109e4280b5a5bL } },
    /* 42 << 21 */
    { { 0x5741a7acb40961bfL,0x4ada59376aa56bfaL,0x7feb914502b765d1L,
        0x561e97bee6ad1582L },
      { 0xbbc4a5b6da3982f5L,0x0c2659edb546f468L,0xb8e7e6aa59612d20L,
        0xd83dfe20ac19e8e0L } },
    /* 43 << 21 */
    { { 0x8530c45fb835398cL,0x6106a8bfb38a41c2L,0x21e8f9a635f5dcdbL,
        0x39707137cae498edL },
      { 0x70c23834d8249f00L,0x9f14b58fab2537a0L,0xd043c3655f61c0c2L,
        0xdc5926d609a194a7L } },
    /* 44 << 21 */
    { { 0xddec03398e77738aL,0xd07a63effba46426L,0x2e58e79cee7f6e86L,
        0xe59b0459ff32d241L },
      { 0xc5ec84e520fa0338L,0x97939ac8eaff5aceL,0x0310a4e3b4a38313L,
        0x9115fba28f9d9885L } },
    /* 45 << 21 */
    { { 0x8dd710c25fadf8c3L,0x66be38a2ce19c0e2L,0xd42a279c4cfe5022L,
        0x597bb5300e24e1b8L },
      { 0x3cde86b7c153ca7fL,0xa8d30fb3707d63bdL,0xac905f92bd60d21eL,
        0x98e7ffb67b9a54abL } },
    /* 46 << 21 */
    { { 0xd7147df8e9726a30L,0xb5e216ffafce3533L,0xb550b7992ff1ec40L,
        0x6b613b87a1e953fdL },
      { 0x87b88dba792d5610L,0x2ee1270aa190fbe1L,0x02f4e2dc2ef581daL,
        0x016530e4eff82a95L } },
    /* 47 << 21 */
    { { 0xcbb93dfd8fd6ee89L,0x16d3d98646848fffL,0x600eff241da47adfL,
        0x1b9754a00ad47a71L },
      { 0x8f9266df70c33b98L,0xaadc87aedf34186eL,0x0d2ce8e14ad24132L,
        0x8a47cbfc19946ebaL } },
    /* 48 << 21 */
    { { 0x47feeb6662b5f3afL,0xcefab5610abb3734L,0x449de60e19f35cb1L,
        0x39f8db14157f0eb9L },
      { 0xffaecc5b3c61bfd6L,0xa5a4d41d41216703L,0x7f8fabed224e1cc2L,
        0x0d5a8186871ad953L } },
    /* 49 << 21 */
    { { 0xf10774f7d22da9a9L,0x45b8a678cc8a9b0dL,0xd9c2e722bdc32cffL,
        0xbf71b5f5337202a5L },
      { 0x95c57f2f69fc4db9L,0xb6dad34c765d01e1L,0x7e0bd13fcb904635L,
        0x61751253763a588cL } },
    /* 50 << 21 */
    { { 0xd85c299781af2c2dL,0xc0f7d9c481b9d7daL,0x838a34ae08533e8dL,
        0x15c4cb08311d8311L },
      { 0x97f832858e121e14L,0xeea7dc1e85000a5fL,0x0c6059b65d256274L,
        0xec9beaceb95075c0L } },
    /* 51 << 21 */
    { { 0x173daad71df97828L,0xbf851cb5a8937877L,0xb083c59401646f3cL,
        0x3bad30cf50c6d352L },
      { 0xfeb2b202496bbceaL,0x3cf9fd4f18a1e8baL,0xd26de7ff1c066029L,
        0x39c81e9e4e9ed4f8L } },
    /* 52 << 21 */
    { { 0xd8be0cb97b390d35L,0x01df2bbd964aab27L,0x3e8c1a65c3ef64f8L,
        0x567291d1716ed1ddL },
      { 0x95499c6c5f5406d3L,0x71fdda395ba8e23fL,0xcfeb320ed5096eceL,
        0xbe7ba92bca66dd16L } },
    /* 53 << 21 */
    { { 0x4608d36bc6fb5a7dL,0xe3eea15a6d2dd0e0L,0x75b0a3eb8f97a36aL,
        0xf59814cc1c83de1eL },
      { 0x56c9c5b01c33c23fL,0xa96c1da46faa4136L,0x46bf2074de316551L,
        0x3b866e7b1f756c8fL } },
    /* 54 << 21 */
    { { 0x727727d81495ed6bL,0xb2394243b682dce7L,0x8ab8454e758610f3L,
        0xc243ce84857d72a4L },
      { 0x7b320d71dbbf370fL,0xff9afa3778e0f7caL,0x0119d1e0ea7b523fL,
        0xb997f8cb058c7d42L } },
    /* 55 << 21 */
    { { 0x285bcd2a37bbb184L,0x51dcec49a45d1fa6L,0x6ade3b64e29634cbL,
        0x080c94a726b86ef1L },
      { 0xba583db12283fbe3L,0x902bddc85a9315edL,0x07c1ccb386964becL,
        0x78f4eacfb6258301L } },
    /* 56 << 21 */
    { { 0x4bdf3a4956f90823L,0xba0f5080741d777bL,0x091d71c3f38bf760L,
        0x9633d50f9b625b02L },
      { 0x03ecb743b8c9de61L,0xb47512545de74720L,0x9f9defc974ce1cb2L,
        0x774a4f6a00bd32efL } },
    /* 57 << 21 */
    { { 0xaca385f773848f22L,0x53dad716f3f8558eL,0xab7b34b093c471f9L,
        0xf530e06919644bc7L },
      { 0x3d9fb1ffdd59d31aL,0x4382e0df08daa795L,0x165c6f4bd5cc88d7L,
        0xeaa392d54a18c900L } },
    /* 58 << 21 */
    { { 0x94203c67648024eeL,0x188763f28c2fabcdL,0xa80f87acbbaec835L,
        0x632c96e0f29d8d54L },
      { 0x29b0a60e4c00a95eL,0x2ef17f40e011e9faL,0xf6c0e1d115b77223L,
        0xaaec2c6214b04e32L } },
    /* 59 << 21 */
    { { 0xd35688d83d84e58cL,0x2af5094c958571dbL,0x4fff7e19760682a6L,
        0x4cb27077e39a407cL },
      { 0x0f59c5474ff0e321L,0x169f34a61b34c8ffL,0x2bff109652bc1ba7L,
        0xa25423b783583544L } },
    /* 60 << 21 */
    { { 0x5d55d5d50ac8b782L,0xff6622ec2db3c892L,0x48fce7416b8bb642L,
        0x31d6998c69d7e3dcL },
      { 0xdbaf8004cadcaed0L,0x801b0142d81d053cL,0x94b189fc59630ec6L,
        0x120e9934af762c8eL } },
    /* 61 << 21 */
    { { 0x53a29aa4fdc6a404L,0x19d8e01ea1909948L,0x3cfcabf1d7e89681L,
        0x3321a50d4e132d37L },
      { 0xd0496863e9a86111L,0x8c0cde6106a3bc65L,0xaf866c49fc9f8eefL,
        0x2066350eff7f5141L } },
    /* 62 << 21 */
    { { 0x4f8a4689e56ddfbdL,0xea1b0c07fe32983aL,0x2b317462873cb8cbL,
        0x658deddc2d93229fL },
      { 0x65efaf4d0f64ef58L,0xfe43287d730cc7a8L,0xaebc0c723d047d70L,
        0x92efa539d92d26c9L } },
    /* 63 << 21 */
    { { 0x06e7845794b56526L,0x415cb80f0961002dL,0x89e5c56576dcb10fL,
        0x8bbb6982ff9259feL },
      { 0x4fe8795b9abc2668L,0xb5d4f5341e678fb1L,0x6601f3be7b7da2b9L,
        0x98da59e2a13d6805L } },
    /* 64 << 21 */
    { { 0x190d8ea601799a52L,0xa20cec41b86d2952L,0x3062ffb27fff2a7cL,
        0x741b32e579f19d37L },
      { 0xf80d81814eb57d47L,0x7a2d0ed416aef06bL,0x09735fb01cecb588L,
        0x1641caaac6061f5bL } },
    /* 0 << 28 */
    { { 0x00, 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x00, 0x00 } },
    /* 1 << 28 */
    { { 0x7f99824f20151427L,0x206828b692430206L,0xaa9097d7e1112357L,
        0xacf9a2f209e414ecL },
      { 0xdbdac9da27915356L,0x7e0734b7001efee3L,0x54fab5bbd2b288e2L,
        0x4c630fc4f62dd09cL } },
    /* 2 << 28 */
    { { 0x8537107a1ac2703bL,0xb49258d86bc857b5L,0x57df14debcdaccd1L,
        0x24ab68d7c4ae8529L },
      { 0x7ed8b5d4734e59d0L,0x5f8740c8c495cc80L,0x84aedd5a291db9b3L,
        0x80b360f84fb995beL } },
    /* 3 << 28 */
    { { 0xae915f5d5fa067d1L,0x4134b57f9668960cL,0xbd3656d6a48edaacL,
        0xdac1e3e4fc1d7436L },
      { 0x674ff869d81fbb26L,0x449ed3ecb26c33d4L,0x85138705d94203e8L,
        0xccde538bbeeb6f4aL } },
    /* 4 << 28 */
    { { 0x55d5c68da61a76faL,0x598b441dca1554dcL,0xd39923b9773b279cL,
        0x33331d3c36bf9efcL },
      { 0x2d4c848e298de399L,0xcfdb8e77a1a27f56L,0x94c855ea57b8ab70L,
        0xdcdb9dae6f7879baL } },
    /* 5 << 28 */
    { { 0x7bdff8c2019f2a59L,0xb3ce5bb3cb4fbc74L,0xea907f688a9173ddL,
        0x6cd3d0d395a75439L },
      { 0x92ecc4d6efed021cL,0x09a9f9b06a77339aL,0x87ca6b157188c64aL,
        0x10c2996844899158L } },
    /* 6 << 28 */
    { { 0x5859a229ed6e82efL,0x16f338e365ebaf4eL,0x0cd313875ead67aeL,
        0x1c73d22854ef0bb4L },
      { 0x4cb5513174a5c8c7L,0x01cd29707f69ad6aL,0xa04d00dde966f87eL,
        0xd96fe4470b7b0321L } },
    /* 7 << 28 */
    { { 0x342ac06e88fbd381L,0x02cd4a845c35a493L,0xe8fa89de54f1bbcdL,
        0x341d63672575ed4cL },
      { 0xebe357fbd238202bL,0x600b4d1aa984ead9L,0xc35c9f4452436ea0L,
        0x96fe0a39a370751bL } },
    /* 8 << 28 */
    { { 0x4c4f07367f636a38L,0x9f943fb70e76d5cbL,0xb03510baa8b68b8bL,
        0xc246780a9ed07a1fL },
      { 0x3c0514156d549fc2L,0xc2953f31607781caL,0x955e2c69d8d95413L,
        0xb300fadc7bd282e3L } },
    /* 9 << 28 */
    { { 0x81fe7b5087e9189fL,0xdb17375cf42dda27L,0x22f7d896cf0a5904L,
        0xa0e57c5aebe348e6L },
      { 0xa61011d3f40e3c80L,0xb11893218db705c5L,0x4ed9309e50fedec3L,
        0xdcf14a104d6d5c1dL } },
    /* 10 << 28 */
    { { 0x056c265b55691342L,0xe8e0850491049dc7L,0x131329f5c9bae20aL,
        0x96c8b3e8d9dccdb4L },
      { 0x8c5ff838fb4ee6b4L,0xfc5a9aeb41e8ccf0L,0x7417b764fae050c6L,
        0x0953c3d700452080L } },
    /* 11 << 28 */
    { { 0x2137268238dfe7e8L,0xea417e152bb79d4bL,0x59641f1c76e7cf2dL,
        0x271e3059ea0bcfccL },
      { 0x624c7dfd7253ecbdL,0x2f552e254fca6186L,0xcbf84ecd4d866e9cL,
        0x73967709f68d4610L } },
    /* 12 << 28 */
    { { 0xa14b1163c27901b4L,0xfd9236e0899b8bf3L,0x42b091eccbc6da0aL,
        0xbb1dac6f5ad1d297L },
      { 0x80e61d53a91cf76eL,0x4110a412d31f1ee7L,0x2d87c3ba13efcf77L,
        0x1f374bb4df450d76L } },
    /* 13 << 28 */
    { { 0x5e78e2f20d188dabL,0xe3968ed0f4b885efL,0x46c0568e7314570fL,
        0x3161633801170521L },
      { 0x18e1e7e24f0c8afeL,0x4caa75ffdeea78daL,0x82db67f27c5d8a51L,
        0x36a44d866f505370L } },
    /* 14 << 28 */
    { { 0xd72c5bda0333974fL,0x5db516ae27a70146L,0x34705281210ef921L,
        0xbff17a8f0c9c38e5L },
      { 0x78f4814e12476da1L,0xc1e1661333c16980L,0x9e5b386f424d4bcaL,
        0x4c274e87c85740deL } },
    /* 15 << 28 */
    { { 0xb6a9b88d6c2f5226L,0x14d1b944550d7ca8L,0x580c85fc1fc41709L,
        0xc1da368b54c6d519L },
      { 0x2b0785ced5113cf7L,0x0670f6335a34708fL,0x46e2376715cc3f88L,
        0x1b480cfa50c72c8fL } },
    /* 16 << 28 */
    { { 0x202886024147519aL,0xd0981eac26b372f0L,0xa9d4a7caa785ebc8L,
        0xd953c50ddbdf58e9L },
      { 0x9d6361ccfd590f8fL,0x72e9626b44e6c917L,0x7fd9611022eb64cfL,
        0x863ebb7e9eb288f3L } },
    /* 17 << 28 */
    { { 0x6e6ab7616aca8ee7L,0x97d10b39d7b40358L,0x1687d3771e5feb0dL,
        0xc83e50e48265a27aL },
      { 0x8f75a9fec954b313L,0xcc2e8f47310d1f61L,0xf5ba81c56557d0e0L,
        0x25f9680c3eaf6207L } },
    /* 18 << 28 */
    { { 0xf95c66094354080bL,0x5225bfa57bf2fe1cL,0xc5c004e25c7d98faL,
        0x3561bf1c019aaf60L },
      { 0x5e6f9f17ba151474L,0xdec2f934b04f6ecaL,0x64e368a1269acb1eL,
        0x1332d9e40cdda493L } },
    /* 19 << 28 */
    { { 0x60d6cf69df23de05L,0x66d17da2009339a0L,0x9fcac9850a693923L,
        0xbcf057fced7c6a6dL },
      { 0xc3c5c8c5f0b5662cL,0x25318dd8dcba4f24L,0x60e8cb75082b69ffL,
        0x7c23b3ee1e728c01L } },
    /* 20 << 28 */
    { { 0x15e10a0a097e4403L,0xcb3d0a8619854665L,0x88d8e211d67d4826L,
        0xb39af66e0b9d2839L },
      { 0xa5f94588bd475ca8L,0xe06b7966c077b80bL,0xfedb1485da27c26cL,
        0xd290d33afe0fd5e0L } },
    /* 21 << 28 */
    { { 0xa40bcc47f34fb0faL,0xb4760cc81fb1ab09L,0x8fca0993a273bfe3L,
        0x13e4fe07f70b213cL },
      { 0x3bcdb992fdb05163L,0x8c484b110c2b19b6L,0x1acb815faaf2e3e2L,
        0xc6905935b89ff1b4L } },
    /* 22 << 28 */
    { { 0xb2ad6f9d586e74e1L,0x488883ad67b80484L,0x758aa2c7369c3ddbL,
        0x8ab74e699f9afd31L },
      { 0x10fc2d285e21beb1L,0x3484518a318c42f9L,0x377427dc53cf40c3L,
        0x9de0781a391bc1d9L } },
    /* 23 << 28 */
    { { 0x8faee858693807e1L,0xa38653274e81ccc7L,0x02c30ff26f835b84L,
        0xb604437b0d3d38d4L },
      { 0xb3fc8a985ca1823dL,0xb82f7ec903be0324L,0xee36d761cf684a33L,
        0x5a01df0e9f29bf7dL } },
    /* 24 << 28 */
    { { 0x686202f31306583dL,0x05b10da0437c622eL,0xbf9aaa0f076a7bc8L,
        0x25e94efb8f8f4e43L },
      { 0x8a35c9b7fa3dc26dL,0xe0e5fb9396ff03c5L,0xa77e3843ebc394ceL,
        0xcede65958361de60L } },
    /* 25 << 28 */
    { { 0xd27c22f6a1993545L,0xab01cc3624d671baL,0x63fa2877a169c28eL,
        0x925ef9042eb08376L },
      { 0x3b2fa3cf53aa0b32L,0xb27beb5b71c49d7aL,0xb60e1834d105e27fL,
        0xd60897884f68570dL } },
    /* 26 << 28 */
    { { 0x23094ce0d6fbc2acL,0x738037a1815ff551L,0xda73b1bb6bef119cL,
        0xdcf6c430eef506baL },
      { 0x00e4fe7be3ef104aL,0xebdd9a2c0a065628L,0x853a81c38792043eL,
        0x22ad6eceb3b59108L } },
    /* 27 << 28 */
    { { 0x9fb813c039cd297dL,0x8ec7e16e05bda5d9L,0x2834797c0d104b96L,
        0xcc11a2e77c511510L },
      { 0x96ca5a5396ee6380L,0x054c8655cea38742L,0xb5946852d54dfa7dL,
        0x97c422e71f4ab207L } },
    /* 28 << 28 */
    { { 0xbf9075090c22b540L,0x2cde42aab7c267d4L,0xba18f9ed5ab0d693L,
        0x3ba62aa66e4660d9L },
      { 0xb24bf97bab9ea96aL,0x5d039642e3b60e32L,0x4e6a45067c4d9bd5L,
        0x666c5b9e7ed4a6a4L } },
    /* 29 << 28 */
    { { 0xfa3fdcd98edbd7ccL,0x4660bb87c6ccd753L,0x9ae9082021e6b64fL,
        0x8a56a713b36bfb3fL },
      { 0xabfce0965726d47fL,0x9eed01b20b1a9a7fL,0x30e9cad44eb74a37L,
        0x7b2524cc53e9666dL } },
    /* 30 << 28 */
    { { 0x6a29683b8f4b002fL,0xc2200d7a41f4fc20L,0xcf3af47a3a338accL,
        0x6539a4fbe7128975L },
      { 0xcec31c14c33c7fcfL,0x7eb6799bc7be322bL,0x119ef4e96646f623L,
        0x7b7a26a554d7299bL } },
    /* 31 << 28 */
    { { 0xcb37f08d403f46f2L,0x94b8fc431a0ec0c7L,0xbb8514e3c332142fL,
        0xf3ed2c33e80d2a7aL },
      { 0x8d2080afb639126cL,0xf7b6be60e3553adeL,0x3950aa9f1c7e2b09L,
        0x847ff9586410f02bL } },
    /* 32 << 28 */
    { { 0x877b7cf5678a31b0L,0xd50301ae3998b620L,0x734257c5c00fb396L,
        0xf9fb18a004e672a6L },
      { 0xff8bd8ebe8758851L,0x1e64e4c65d99ba44L,0x4b8eaedf7dfd93b7L,
        0xba2f2a9804e76b8cL } },
    /* 33 << 28 */
    { { 0x7d790cbae8053433L,0xc8e725a03d2c9585L,0x58c5c476cdd8f5edL,
        0xd106b952efa9fe1dL },
      { 0x3c5c775b0eff13a9L,0x242442bae057b930L,0xe9f458d4c9b70cbdL,
        0x69b71448a3cdb89aL } },
    /* 34 << 28 */
    { { 0x41ee46f60e2ed742L,0x573f104540067493L,0xb1e154ff9d54c304L,
        0x2ad0436a8d3a7502L },
      { 0xee4aaa2d431a8121L,0xcd38b3ab886f11edL,0x57d49ea6034a0eb7L,
        0xd2b773bdf7e85e58L } },
    /* 35 << 28 */
    { { 0x4a559ac49b5c1f14L,0xc444be1a3e54df2bL,0x13aad704eda41891L,
        0xcd927bec5eb5c788L },
      { 0xeb3c8516e48c8a34L,0x1b7ac8124b546669L,0x1815f896594df8ecL,
        0x87c6a79c79227865L } },
    /* 36 << 28 */
    { { 0xae02a2f09b56ddbdL,0x1339b5ac8a2f1cf3L,0xf2b569c7839dff0dL,
        0xb0b9e864fee9a43dL },
      { 0x4ff8ca4177bb064eL,0x145a2812fd249f63L,0x3ab7beacf86f689aL,
        0x9bafec2701d35f5eL } },
    /* 37 << 28 */
    { { 0x28054c654265aa91L,0xa4b18304035efe42L,0x6887b0e69639dec7L,
        0xf4b8f6ad3d52aea5L },
      { 0xfb9293cc971a8a13L,0x3f159e5d4c934d07L,0x2c50e9b109acbc29L,
        0x08eb65e67154d129L } },
    /* 38 << 28 */
    { { 0x4feff58930b75c3eL,0x0bb82fe294491c93L,0xd8ac377a89af62bbL,
        0xd7b514909685e49fL },
      { 0xabca9a7b04497f19L,0x1b35ed0a1a7ad13fL,0x6b601e213ec86ed6L,
        0xda91fcb9ce0c76f1L } },
    /* 39 << 28 */
    { { 0x9e28507bd7ab27e1L,0x7c19a55563945b7bL,0x6b43f0a1aafc9827L,
        0x443b4fbd3aa55b91L },
      { 0x962b2e656962c88fL,0x139da8d4ce0db0caL,0xb93f05dd1b8d6c4fL,
        0x779cdff7180b9824L } },
    /* 40 << 28 */
    { { 0xbba23fddae57c7b7L,0x345342f21b932522L,0xfd9c80fe556d4aa3L,
        0xa03907ba6525bb61L },
      { 0x38b010e1ff218933L,0xc066b654aa52117bL,0x8e14192094f2e6eaL,
        0x66a27dca0d32f2b2L } },
    /* 41 << 28 */
    { { 0x69c7f993048b3717L,0xbf5a989ab178ae1cL,0x49fa9058564f1d6bL,
        0x27ec6e15d31fde4eL },
      { 0x4cce03737276e7fcL,0x64086d7989d6bf02L,0x5a72f0464ccdd979L,
        0x909c356647775631L } },
    /* 42 << 28 */
    { { 0x1c07bc6b75dd7125L,0xb4c6bc9787a0428dL,0x507ece52fdeb6b9dL,
        0xfca56512b2c95432L },
      { 0x15d97181d0e8bd06L,0x384dd317c6bb46eaL,0x5441ea203952b624L,
        0xbcf70dee4e7dc2fbL } },
    /* 43 << 28 */
    { { 0x372b016e6628e8c3L,0x07a0d667b60a7522L,0xcf05751b0a344ee2L,
        0x0ec09a48118bdeecL },
      { 0x6e4b3d4ed83dce46L,0x43a6316d99d2fc6eL,0xa99d898956cf044cL,
        0x7c7f4454ae3e5fb7L } },
    /* 44 << 28 */
    { { 0xb2e6b121fbabbe92L,0x281850fbe1330076L,0x093581ec97890015L,
        0x69b1dded75ff77f5L },
      { 0x7cf0b18fab105105L,0x953ced31a89ccfefL,0x3151f85feb914009L,
        0x3c9f1b8788ed48adL } },
    /* 45 << 28 */
    { { 0xc9aba1a14a7eadcbL,0x928e7501522e71cfL,0xeaede7273a2e4f83L,
        0x467e10d11ce3bbd3L },
      { 0xf3442ac3b955dcf0L,0xba96307dd3d5e527L,0xf763a10efd77f474L,
        0x5d744bd06a6e1ff0L } },
    /* 46 << 28 */
    { { 0xd287282aa777899eL,0xe20eda8fd03f3cdeL,0x6a7e75bb50b07d31L,
        0x0b7e2a946f379de4L },
      { 0x31cb64ad19f593cfL,0x7b1a9e4f1e76ef1dL,0xe18c9c9db62d609cL,
        0x439bad6de779a650L } },
    /* 47 << 28 */
    { { 0x219d9066e032f144L,0x1db632b8e8b2ec6aL,0xff0d0fd4fda12f78L,
        0x56fb4c2d2a25d265L },
      { 0x5f4e2ee1255a03f1L,0x61cd6af2e96af176L,0xe0317ba8d068bc97L,
        0x927d6bab264b988eL } },
    /* 48 << 28 */
    { { 0xa18f07e0e90fb21eL,0x00fd2b80bba7fca1L,0x20387f2795cd67b5L,
        0x5b89a4e7d39707f7L },
      { 0x8f83ad3f894407ceL,0xa0025b946c226132L,0xc79563c7f906c13bL,
        0x5f548f314e7bb025L } },
    /* 49 << 28 */
    { { 0x2b4c6b8feac6d113L,0xa67e3f9c0e813c76L,0x3982717c3fe1f4b9L,
        0x5886581926d8050eL },
      { 0x99f3640cf7f06f20L,0xdc6102162a66ebc2L,0x52f2c175767a1e08L,
        0x05660e1a5999871bL } },
    /* 50 << 28 */
    { { 0x6b0f17626d3c4693L,0xf0e7d62737ed7beaL,0xc51758c7b75b226dL,
        0x40a886281f91613bL },
      { 0x889dbaa7bbb38ce0L,0xe0404b65bddcad81L,0xfebccd3a8bc9671fL,
        0xfbf9a357ee1f5375L } },
    /* 51 << 28 */
    { { 0x5dc169b028f33398L,0xb07ec11d72e90f65L,0xae7f3b4afaab1eb1L,
        0xd970195e5f17538aL },
      { 0x52b05cbe0181e640L,0xf5debd622643313dL,0x761481545df31f82L,
        0x23e03b333a9e13c5L } },
    /* 52 << 28 */
    { { 0xff7589494fde0c1fL,0xbf8a1abee5b6ec20L,0x702278fb87e1db6cL,
        0xc447ad7a35ed658fL },
      { 0x48d4aa3803d0ccf2L,0x80acb338819a7c03L,0x9bc7c89e6e17ceccL,
        0x46736b8b03be1d82L } },
    /* 53 << 28 */
    { { 0xd65d7b60c0432f96L,0xddebe7a3deb5442fL,0x79a253077dff69a2L,
        0x37a56d9402cf3122L },
      { 0x8bab8aedf2350d0aL,0x13c3f276037b0d9aL,0xc664957c44c65caeL,
        0x88b44089c2e71a88L } },
    /* 54 << 28 */
    { { 0xdb88e5a35cb02664L,0x5d4c0bf18686c72eL,0xea3d9b62a682d53eL,
        0x9b605ef40b2ad431L },
      { 0x71bac202c69645d0L,0xa115f03a6a1b66e7L,0xfe2c563a158f4dc4L,
        0xf715b3a04d12a78cL } },
    /* 55 << 28 */
    { { 0x8f7f0a48d413213aL,0x2035806dc04becdbL,0xecd34a995d8587f5L,
        0x4d8c30799f6d3a71L },
      { 0x1b2a2a678d95a8f6L,0xc58c9d7df2110d0dL,0xdeee81d5cf8fba3fL,
        0xa42be3c00c7cdf68L } },
    /* 56 << 28 */
    { { 0x2126f742d43b5eaaL,0x054a0766dfa59b85L,0x9d0d5e36126bfd45L,
        0xa1f8fbd7384f8a8fL },
      { 0x317680f5d563fcccL,0x48ca5055f280a928L,0xe00b81b227b578cfL,
        0x10aad9182994a514L } },
    /* 57 << 28 */
    { { 0xd9e07b62b7bdc953L,0x9f0f6ff25bc086ddL,0x09d1ccff655eee77L,
        0x45475f795bef7df1L },
      { 0x3faa28fa86f702ccL,0x92e609050f021f07L,0xe9e629687f8fa8c6L,
        0xbd71419af036ea2cL } },
    /* 58 << 28 */
    { { 0x171ee1cc6028da9aL,0x5352fe1ac251f573L,0xf8ff236e3fa997f4L,
        0xd831b6c9a5749d5fL },
      { 0x7c872e1de350e2c2L,0xc56240d91e0ce403L,0xf9deb0776974f5cbL,
        0x7d50ba87961c3728L } },
    /* 59 << 28 */
    { { 0xd6f894265a3a2518L,0xcf817799c6303d43L,0x510a0471619e5696L,
        0xab049ff63a5e307bL },
      { 0xe4cdf9b0feb13ec7L,0xd5e971179d8ff90cL,0xf6f64d069afa96afL,
        0x00d0bf5e9d2012a2L } },
    /* 60 << 28 */
    { { 0xe63f301f358bcdc0L,0x07689e990a9d47f8L,0x1f689e2f4f43d43aL,
        0x4d542a1690920904L },
      { 0xaea293d59ca0a707L,0xd061fe458ac68065L,0x1033bf1b0090008cL,
        0x29749558c08a6db6L } },
    /* 61 << 28 */
    { { 0x74b5fc59c1d5d034L,0xf712e9f667e215e0L,0xfd520cbd860200e6L,
        0x0229acb43ea22588L },
      { 0x9cd1e14cfff0c82eL,0x87684b6259c69e73L,0xda85e61c96ccb989L,
        0x2d5dbb02a3d06493L } },
    /* 62 << 28 */
    { { 0xf22ad33ae86b173cL,0xe8e41ea5a79ff0e3L,0x01d2d725dd0d0c10L,
        0x31f39088032d28f9L },
      { 0x7b3f71e17829839eL,0x0cf691b44502ae58L,0xef658dbdbefc6115L,
        0xa5cd6ee5b3ab5314L } },
    /* 63 << 28 */
    { { 0x206c8d7b5f1d2347L,0x794645ba4cc2253aL,0xd517d8ff58389e08L,
        0x4fa20dee9f847288L },
      { 0xeba072d8d797770aL,0x7360c91dbf429e26L,0x7200a3b380af8279L,
        0x6a1c915082dadce3L } },
    /* 64 << 28 */
    { { 0x0ee6d3a7c35d8794L,0x042e65580356bae5L,0x9f59698d643322fdL,
        0x9379ae1550a61967L },
      { 0x64b9ae62fcc9981eL,0xaed3d6316d2934c6L,0x2454b3025e4e65ebL,
        0xab09f647f9950428L } },
    /* 0 << 35 */
    { { 0x00, 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x00, 0x00 } },
    /* 1 << 35 */
    { { 0xb2083a1222248accL,0x1f6ec0ef3264e366L,0x5659b7045afdee28L,
        0x7a823a40e6430bb5L },
      { 0x24592a04e1900a79L,0xcde09d4ac9ee6576L,0x52b6463f4b5ea54aL,
        0x1efe9ed3d3ca65a7L } },
    /* 2 << 35 */
    { { 0xe27a6dbe305406ddL,0x8eb7dc7fdd5d1957L,0xf54a6876387d4d8fL,
        0x9c479409c7762de4L },
      { 0xbe4d5b5d99b30778L,0x25380c566e793682L,0x602d37f3dac740e3L,
        0x140deabe1566e4aeL } },
    /* 3 << 35 */
    { { 0x4481d067afd32acfL,0xd8f0fccae1f71ccfL,0xd208dd0cb596f2daL,
        0xd049d7309aad93f9L },
      { 0xc79f263d42ab580eL,0x09411bb123f707b4L,0x8cfde1ff835e0edaL,
        0x7270749090f03402L } },
    /* 4 << 35 */
    { { 0xeaee6126c49a861eL,0x024f3b65e14f0d06L,0x51a3f1e8c69bfc17L,
        0xc3c3a8e9a7686381L },
      { 0x3400752cb103d4c8L,0x02bc46139218b36bL,0xc67f75eb7651504aL,
        0xd6848b56d02aebfaL } },
    /* 5 << 35 */
    { { 0xbd9802e6c30fa92bL,0x5a70d96d9a552784L,0x9085c4ea3f83169bL,
        0xfa9423bb06908228L },
      { 0x2ffebe12fe97a5b9L,0x85da604971b99118L,0x9cbc2f7f63178846L,
        0xfd96bc709153218eL } },
    /* 6 << 35 */
    { { 0x958381db1782269bL,0xae34bf792597e550L,0xbb5c60645f385153L,
        0x6f0e96afe3088048L },
      { 0xbf6a021577884456L,0xb3b5688c69310ea7L,0x17c9429504fad2deL,
        0xe020f0e517896d4dL } },
    /* 7 << 35 */
    { { 0x730ba0ab0976505fL,0x567f6813095e2ec5L,0x470620106331ab71L,
        0x72cfa97741d22b9fL },
      { 0x33e55ead8a2373daL,0xa8d0d5f47ba45a68L,0xba1d8f9c03029d15L,
        0x8f34f1ccfc55b9f3L } },
    /* 8 << 35 */
    { { 0xcca4428dbbe5a1a9L,0x8187fd5f3126bd67L,0x0036973a48105826L,
        0xa39b6663b8bd61a0L },
      { 0x6d42deef2d65a808L,0x4969044f94636b19L,0xf611ee47dd5d564cL,
        0x7b2f3a49d2873077L } },
    /* 9 << 35 */
    { { 0x94157d45300eb294L,0x2b2a656e169c1494L,0xc000dd76d3a47aa9L,
        0xa2864e4fa6243ea4L },
      { 0x82716c47db89842eL,0x12dfd7d761479fb7L,0x3b9a2c56e0b2f6dcL,
        0x46be862ad7f85d67L } },
    /* 10 << 35 */
    { { 0x03b0d8dd0f82b214L,0x460c34f9f103cbc6L,0xf32e5c0318d79e19L,
        0x8b8888baa84117f8L },
      { 0x8f3c37dcc0722677L,0x10d21be91c1c0f27L,0xd47c8468e0f7a0c6L,
        0x9bf02213adecc0e0L } },
    /* 11 << 35 */
    { { 0x0baa7d1242b48b99L,0x1bcb665d48424096L,0x8b847cd6ebfb5cfbL,
        0x87c2ae569ad4d10dL },
      { 0xf1cbb1220de36726L,0xe7043c683fdfbd21L,0x4bd0826a4e79d460L,
        0x11f5e5984bd1a2cbL } },
    /* 12 << 35 */
    { { 0x97554160b7fe7b6eL,0x7d16189a400a3fb2L,0xd73e9beae328ca1eL,
        0x0dd04b97e793d8ccL },
      { 0xa9c83c9b506db8ccL,0x5cd47aaecf38814cL,0x26fc430db64b45e6L,
        0x079b5499d818ea84L } },
    /* 13 << 35 */
    { { 0xebb01102c1c24a3bL,0xca24e5681c161c1aL,0x103eea6936f00a4aL,
        0x9ad76ee876176c7bL },
      { 0x97451fc2538e0ff7L,0x94f898096604b3b0L,0x6311436e3249cfd7L,
        0x27b4a7bd41224f69L } },
    /* 14 << 35 */
    { { 0x03b5d21ae0ac2941L,0x279b0254c2d31937L,0x3307c052cac992d0L,
        0x6aa7cb92efa8b1f3L },
      { 0x5a1825800d37c7a5L,0x13380c37342d5422L,0x92ac2d66d5d2ef92L,
        0x035a70c9030c63c6L } },
    /* 15 << 35 */
    { { 0xc16025dd4ce4f152L,0x1f419a71f9df7c06L,0x6d5b221491e4bb14L,
        0xfc43c6cc839fb4ceL },
      { 0x49f06591925d6b2dL,0x4b37d9d362186598L,0x8c54a971d01b1629L,
        0xe1a9c29f51d50e05L } },
    /* 16 << 35 */
    { { 0x5109b78571ba1861L,0x48b22d5cd0c8f93dL,0xe8fa84a78633bb93L,
        0x53fba6ba5aebbd08L },
      { 0x7ff27df3e5eea7d8L,0x521c879668ca7158L,0xb9d5133bce6f1a05L,
        0x2d50cd53fd0ebee4L } },
    /* 17 << 35 */
    { { 0xc82115d6c5a3ef16L,0x993eff9dba079221L,0xe4da2c5e4b5da81cL,
        0x9a89dbdb8033fd85L },
      { 0x60819ebf2b892891L,0x53902b215d14a4d5L,0x6ac35051d7fda421L,
        0xcc6ab88561c83284L } },
    /* 18 << 35 */
    { { 0x14eba133f74cff17L,0x240aaa03ecb813f2L,0xcfbb65406f665beeL,
        0x084b1fe4a425ad73L },
      { 0x009d5d16d081f6a6L,0x35304fe8eef82c90L,0xf20346d5aa9eaa22L,
        0x0ada9f07ac1c91e3L } },
    /* 19 << 35 */
    { { 0xa6e21678968a6144L,0x54c1f77c07b31a1eL,0xd6bb787e5781fbe1L,
        0x61bd2ee0e31f1c4aL },
      { 0xf25aa1e9781105fcL,0x9cf2971f7b2f8e80L,0x26d15412cdff919bL,
        0x01db4ebe34bc896eL } },
    /* 20 << 35 */
    { { 0x7d9b3e23b40df1cfL,0x5933737394e971b4L,0xbf57bd14669cf921L,
        0x865daedf0c1a1064L },
      { 0x3eb70bd383279125L,0xbc3d5b9f34ecdaabL,0x91e3ed7e5f755cafL,
        0x49699f54d41e6f02L } },
    /* 21 << 35 */
    { { 0x185770e1d4a7a15bL,0x08f3587aeaac87e7L,0x352018db473133eaL,
        0x674ce71904fd30fcL },
      { 0x7b8d9835088b3e0eL,0x7a0356a95d0d47a1L,0x9d9e76596474a3c4L,
        0x61ea48a7ff66966cL } },
    /* 22 << 35 */
    { { 0x304177580f3e4834L,0xfdbb21c217a9afcbL,0x756fa17f2f9a67b3L,
        0x2a6b2421a245c1a8L },
      { 0x64be27944af02291L,0xade465c62a5804feL,0x8dffbd39a6f08fd7L,
        0xc4efa84caa14403bL } },
    /* 23 << 35 */
    { { 0xa1b91b2a442b0f5cL,0xb748e317cf997736L,0x8d1b62bfcee90e16L,
        0x907ae2710b2078c0L },
      { 0xdf31534b0c9bcdddL,0x043fb05439adce83L,0x99031043d826846aL,
        0x61a9c0d6b144f393L } },
    /* 24 << 35 */
    { { 0xdab4804647718427L,0xdf17ff9b6e830f8bL,0x408d7ee8e49a1347L,
        0x6ac71e2391c1d4aeL },
      { 0xc8cbb9fd1defd73cL,0x19840657bbbbfec5L,0x39db1cb59e7ef8eaL,
        0x78aa829664105f30L } },
    /* 25 << 35 */
    { { 0xa3d9b7f0a3738c29L,0x0a2f235abc3250a3L,0x55e506f6445e4cafL,
        0x0974f73d33475f7aL },
      { 0xd37dbba35ba2f5a8L,0x542c6e636af40066L,0x26d99b53c5d73e2cL,
        0x06060d7d6c3ca33eL } },
    /* 26 << 35 */
    { { 0xcdbef1c2065fef4aL,0x77e60f7dfd5b92e3L,0xd7c549f026708350L,
        0x201b3ad034f121bfL },
      { 0x5fcac2a10334fc14L,0x8a9a9e09344552f6L,0x7dd8a1d397653082L,
        0x5fc0738f79d4f289L } },
    /* 27 << 35 */
    { { 0x787d244d17d2d8c3L,0xeffc634570830684L,0x5ddb96dde4f73ae5L,
        0x8efb14b1172549a5L },
      { 0x6eb73eee2245ae7aL,0xbca4061eea11f13eL,0xb577421d30b01f5dL,
        0xaa688b24782e152cL } },
    /* 28 << 35 */
    { { 0x67608e71bd3502baL,0x4ef41f24b4de75a0L,0xb08dde5efd6125e5L,
        0xde484825a409543fL },
      { 0x1f198d9865cc2295L,0x428a37716e0edfa2L,0x4f9697a2adf35fc7L,
        0x01a43c79f7cac3c7L } },
    /* 29 << 35 */
    { { 0xb05d70590fd3659aL,0x8927f30cbb7f2d9aL,0x4023d1ac8cf984d3L,
        0x32125ed302897a45L },
      { 0xfb572dad3d414205L,0x73000ef2e3fa82a9L,0x4c0868e9f10a5581L,
        0x5b61fc676b0b3ca5L } },
    /* 30 << 35 */
    { { 0xc1258d5b7cae440cL,0x21c08b41402b7531L,0xf61a8955de932321L,
        0x3568faf82d1408afL },
      { 0x71b15e999ecf965bL,0xf14ed248e917276fL,0xc6f4caa1820cf9e2L,
        0x681b20b218d83c7eL } },
    /* 31 << 35 */
    { { 0x6cde738dc6c01120L,0x71db0813ae70e0dbL,0x95fc064474afe18cL,
        0x34619053129e2be7L },
      { 0x80615ceadb2a3b15L,0x0a49a19edb4c7073L,0x0e1b84c88fd2d367L,
        0xd74bf462033fb8aaL } },
    /* 32 << 35 */
    { { 0x889f6d65533ef217L,0x7158c7e4c3ca2e87L,0xfb670dfbdc2b4167L,
        0x75910a01844c257fL },
      { 0xf336bf07cf88577dL,0x22245250e45e2aceL,0x2ed92e8d7ca23d85L,
        0x29f8be4c2b812f58L } },
    /* 33 << 35 */
    { { 0xdd9ebaa7076fe12bL,0x3f2400cbae1537f9L,0x1aa9352817bdfb46L,
        0xc0f9843067883b41L },
      { 0x5590ede10170911dL,0x7562f5bb34d4b17fL,0xe1fa1df21826b8d2L,
        0xb40b796a6bd80d59L } },
    /* 34 << 35 */
    { { 0xd65bf1973467ba92L,0x8c9b46dbf70954b0L,0x97c8a0f30e78f15dL,
        0xa8f3a69a85a4c961L },
      { 0x4242660f61e4ce9bL,0xbf06aab36ea6790cL,0xc6706f8eec986416L,
        0x9e56dec19a9fc225L } },
    /* 35 << 35 */
    { { 0x527c46f49a9898d9L,0xd799e77b5633cdefL,0x24eacc167d9e4297L,
        0xabb61cea6b1cb734L },
      { 0xbee2e8a7f778443cL,0x3bb42bf129de2fe6L,0xcbed86a13003bb6fL,
        0xd3918e6cd781cdf6L } },
    /* 36 << 35 */
    { { 0x4bee32719a5103f1L,0x5243efc6f50eac06L,0xb8e122cb6adcc119L,
        0x1b7faa84c0b80a08L },
      { 0x32c3d1bd6dfcd08cL,0x129dec4e0be427deL,0x98ab679c1d263c83L,
        0xafc83cb7cef64effL } },
    /* 37 << 35 */
    { { 0x85eb60882fa6be76L,0x892585fb1328cbfeL,0xc154d3edcf618ddaL,
        0xc44f601b3abaf26eL },
      { 0x7bf57d0b2be1fdfdL,0xa833bd2d21137feeL,0x9353af362db591a8L,
        0xc76f26dc5562a056L } },
    /* 38 << 35 */
    { { 0x1d87e47d3fdf5a51L,0x7afb5f9355c9cab0L,0x91bbf58f89e0586eL,
        0x7c72c0180d843709L },
      { 0xa9a5aafb99b5c3dcL,0xa48a0f1d3844aeb0L,0x7178b7ddb667e482L,
        0x453985e96e23a59aL } },
    /* 39 << 35 */
    { { 0x4a54c86001b25dd8L,0x0dd37f48fb897c8aL,0x5f8aa6100ea90cd9L,
        0xc8892c6816d5830dL },
      { 0xeb4befc0ef514ca5L,0x478eb679e72c9ee6L,0x9bca20dadbc40d5fL,
        0xf015de21dde4f64aL } },
    /* 40 << 35 */
    { { 0xaa6a4de0eaf4b8a5L,0x68cfd9ca4bc60e32L,0x668a4b017fd15e70L,
        0xd9f0694af27dc09dL },
      { 0xf6c3cad5ba708bcdL,0x5cd2ba695bb95c2aL,0xaa28c1d333c0a58fL,
        0x23e274e3abc77870L } },
    /* 41 << 35 */
    { { 0x44c3692ddfd20a4aL,0x091c5fd381a66653L,0x6c0bb69109a0757dL,
        0x9072e8b9667343eaL },
      { 0x31d40eb080848becL,0x95bd480a79fd36ccL,0x01a77c6165ed43f5L,
        0xafccd1272e0d40bfL } },
    /* 42 << 35 */
    { { 0xeccfc82d1cc1884bL,0xc85ac2015d4753b4L,0xc7a6caac658e099fL,
        0xcf46369e04b27390L },
      { 0xe2e7d049506467eaL,0x481b63a237cdecccL,0x4029abd8ed80143aL,
        0x28bfe3c7bcb00b88L } },
    /* 43 << 35 */
    { { 0x3bec10090643d84aL,0x885f3668abd11041L,0xdb02432cf83a34d6L,
        0x32f7b360719ceebeL },
      { 0xf06c7837dad1fe7aL,0x60a157a95441a0b0L,0x704970e9e2d47550L,
        0xcd2bd553271b9020L } },
    /* 44 << 35 */
    { { 0xff57f82f33e24a0bL,0x9cbee23ff2565079L,0x16353427eb5f5825L,
        0x276feec4e948d662L },
      { 0xd1b62bc6da10032bL,0x718351ddf0e72a53L,0x934520762420e7baL,
        0x96368fff3a00118dL } },
    /* 45 << 35 */
    { { 0x00ce2d26150a49e4L,0x0c28b6363f04706bL,0xbad65a4658b196d0L,
        0x6c8455fcec9f8b7cL },
      { 0xe90c895f2d71867eL,0x5c0be31bedf9f38cL,0x2a37a15ed8f6ec04L,
        0x239639e78cd85251L } },
    /* 46 << 35 */
    { { 0xd89753159c7c4c6bL,0x603aa3c0d7409af7L,0xb8d53d0c007132fbL,
        0x68d12af7a6849238L },
      { 0xbe0607e7bf5d9279L,0x9aa50055aada74ceL,0xe81079cbba7e8ccbL,
        0x610c71d1a5f4ff5eL } },
    /* 47 << 35 */
    { { 0x9e2ee1a75aa07093L,0xca84004ba75da47cL,0x074d39513de75401L,
        0xf938f756bb311592L },
      { 0x9619761800a43421L,0x39a2536207bc78c8L,0x278f710a0a171276L,
        0xb28446ea8d1a8f08L } },
    /* 48 << 35 */
    { { 0x184781bfe3b6a661L,0x7751cb1de6d279f7L,0xf8ff95d6c59eb662L,
        0x186d90b758d3dea7L },
      { 0x0e4bb6c1dfb4f754L,0x5c5cf56b2b2801dcL,0xc561e4521f54564dL,
        0xb4fb8c60f0dd7f13L } },
    /* 49 << 35 */
    { { 0xf884963033ff98c7L,0x9619fffacf17769cL,0xf8090bf61bfdd80aL,
        0x14d9a149422cfe63L },
      { 0xb354c3606f6df9eaL,0xdbcf770d218f17eaL,0x207db7c879eb3480L,
        0x213dbda8559b6a26L } },
    /* 50 << 35 */
    { { 0xac4c200b29fc81b3L,0xebc3e09f171d87c1L,0x917995301481aa9eL,
        0x051b92e192e114faL },
      { 0xdf8f92e9ecb5537fL,0x44b1b2cc290c7483L,0xa711455a2adeb016L,
        0x964b685681a10c2cL } },
    /* 51 << 35 */
    { { 0x4f159d99cec03623L,0x05532225ef3271eaL,0xb231bea3c5ee4849L,
        0x57a54f507094f103L },
      { 0x3e2d421d9598b352L,0xe865a49c67412ab4L,0xd2998a251cc3a912L,
        0x5d0928080c74d65dL } },
    /* 52 << 35 */
    { { 0x73f459084088567aL,0xeb6b280e1f214a61L,0x8c9adc34caf0c13dL,
        0x39d12938f561fb80L },
      { 0xb2dc3a5ebc6edfb4L,0x7485b1b1fe4d210eL,0x062e0400e186ae72L,
        0x91e32d5c6eeb3b88L } },
    /* 53 << 35 */
    { { 0x6df574d74be59224L,0xebc88ccc716d55f3L,0x26c2e6d0cad6ed33L,
        0xc6e21e7d0d3e8b10L },
      { 0x2cc5840e5bcc36bbL,0x9292445e7da74f69L,0x8be8d3214e5193a8L,
        0x3ec236298df06413L } },
    /* 54 << 35 */
    { { 0xc7e9ae85b134defaL,0x6073b1d01bb2d475L,0xb9ad615e2863c00dL,
        0x9e29493d525f4ac4L },
      { 0xc32b1dea4e9acf4fL,0x3e1f01c8a50db88dL,0xb05d70ea04da916cL,
        0x714b0d0ad865803eL } },
    /* 55 << 35 */
    { { 0x4bd493fc9920cb5eL,0x5b44b1f792c7a3acL,0xa2a77293bcec9235L,
        0x5ee06e87cd378553L },
      { 0xceff8173da621607L,0x2bb03e4c99f5d290L,0x2945106aa6f734acL,
        0xb5056604d25c4732L } },
    /* 56 << 35 */
    { { 0x5945920ce079afeeL,0x686e17a06789831fL,0x5966bee8b74a5ae5L,
        0x38a673a21e258d46L },
      { 0xbd1cc1f283141c95L,0x3b2ecf4f0e96e486L,0xcd3aa89674e5fc78L,
        0x415ec10c2482fa7aL } },
    /* 57 << 35 */
    { { 0x1523441980503380L,0x513d917ad314b392L,0xb0b52f4e63caecaeL,
        0x07bf22ad2dc7780bL },
      { 0xe761e8a1e4306839L,0x1b3be9625dd7feaaL,0x4fe728de74c778f1L,
        0xf1fa0bda5e0070f6L } },
    /* 58 << 35 */
    { { 0x85205a316ec3f510L,0x2c7e4a14d2980475L,0xde3c19c06f30ebfdL,
        0xdb1c1f38d4b7e644L },
      { 0xfe291a755dce364aL,0xb7b22a3c058f5be3L,0x2cd2c30237fea38cL,
        0x2930967a2e17be17L } },
    /* 59 << 35 */
    { { 0x87f009de0c061c65L,0xcb014aacedc6ed44L,0x49bd1cb43bafb1ebL,
        0x81bd8b5c282d3688L },
      { 0x1cdab87ef01a17afL,0x21f37ac4e710063bL,0x5a6c567642fc8193L,
        0xf4753e7056a6015cL } },
    /* 60 << 35 */
    { { 0x020f795ea15b0a44L,0x8f37c8d78958a958L,0x63b7e89ba4b675b5L,
        0xb4fb0c0c0fc31aeaL },
      { 0xed95e639a7ff1f2eL,0x9880f5a3619614fbL,0xdeb6ff02947151abL,
        0x5bc5118ca868dcdbL } },
    /* 61 << 35 */
    { { 0xd8da20554c20cea5L,0xcac2776e14c4d69aL,0xcccb22c1622d599bL,
        0xa4ddb65368a9bb50L },
      { 0x2c4ff1511b4941b4L,0xe1ff19b46efba588L,0x35034363c48345e0L,
        0x45542e3d1e29dfc4L } },
    /* 62 << 35 */
    { { 0xf197cb91349f7aedL,0x3b2b5a008fca8420L,0x7c175ee823aaf6d8L,
        0x54dcf42135af32b6L },
      { 0x0ba1430727d6561eL,0x879d5ee4d175b1e2L,0xc7c4367399807db5L,
        0x77a544559cd55bcdL } },
    /* 63 << 35 */
    { { 0xe6c2ff130105c072L,0x18f7a99f8dda7da4L,0x4c3018200e2d35c1L,
        0x06a53ca0d9cc6c82L },
      { 0xaa21cc1ef1aa1d9eL,0x324143344a75b1e8L,0x2a6d13280ebe9fdcL,
        0x16bd173f98a4755aL } },
    /* 64 << 35 */
    { { 0xfbb9b2452133ffd9L,0x39a8b2f1830f1a20L,0x484bc97dd5a1f52aL,
        0xd6aebf56a40eddf8L },
      { 0x32257acb76ccdac6L,0xaf4d36ec1586ff27L,0x8eaa8863f8de7dd1L,
        0x0045d5cf88647c16L } },
    /* 0 << 42 */
    { { 0x00, 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x00, 0x00 } },
    /* 1 << 42 */
    { { 0xa6f3d574c005979dL,0xc2072b426a40e350L,0xfca5c1568de2ecf9L,
        0xa8c8bf5ba515344eL },
      { 0x97aee555114df14aL,0xd4374a4dfdc5ec6bL,0x754cc28f2ca85418L,
        0x71cb9e27d3c41f78L } },
    /* 2 << 42 */
    { { 0x8910507903605c39L,0xf0843d9ea142c96cL,0xf374493416923684L,
        0x732caa2ffa0a2893L },
      { 0xb2e8c27061160170L,0xc32788cc437fbaa3L,0x39cd818ea6eda3acL,
        0xe2e942399e2b2e07L } },
    /* 3 << 42 */
    { { 0x6967d39b0260e52aL,0xd42585cc90653325L,0x0d9bd60521ca7954L,
        0x4fa2087781ed57b3L },
      { 0x60c1eff8e34a0bbeL,0x56b0040c84f6ef64L,0x28be2b24b1af8483L,
        0xb2278163f5531614L } },
    /* 4 << 42 */
    { { 0x8df275455922ac1cL,0xa7b3ef5ca52b3f63L,0x8e77b21471de57c4L,
        0x31682c10834c008bL },
      { 0xc76824f04bd55d31L,0xb6d1c08617b61c71L,0x31db0903c2a5089dL,
        0x9c092172184e5d3fL } },
    /* 5 << 42 */
    { { 0xdd7ced5bc00cc638L,0x1a2015eb61278fc2L,0x2e8e52886a37f8d6L,
        0xc457786fe79933adL },
      { 0xb3fe4cce2c51211aL,0xad9b10b224c20498L,0x90d87a4fd28db5e5L,
        0x698cd1053aca2fc3L } },
    /* 6 << 42 */
    { { 0x4f112d07e91b536dL,0xceb982f29eba09d6L,0x3c157b2c197c396fL,
        0xe23c2d417b66eb24L },
      { 0x480c57d93f330d37L,0xb3a4c8a179108debL,0x702388decb199ce5L,
        0x0b019211b944a8d4L } },
    /* 7 << 42 */
    { { 0x24f2a692840bb336L,0x7c353bdca669fa7bL,0xda20d6fcdec9c300L,
        0x625fbe2fa13a4f17L },
      { 0xa2b1b61adbc17328L,0x008965bfa9515621L,0x49690939c620ff46L,
        0x182dd27d8717e91cL } },
    /* 8 << 42 */
    { { 0x5ace5035ea6c3997L,0x54259aaac2610befL,0xef18bb3f3c80dd39L,
        0x6910b95b5fc3fa39L },
      { 0xfce2f51043e09aeeL,0xced56c9fa7675665L,0x10e265acd872db61L,
        0x6982812eae9fce69L } },
    /* 9 << 42 */
    { { 0x29be11c6ce800998L,0x72bb1752b90360d9L,0x2c1931975a4ad590L,
        0x2ba2f5489fc1dbc0L },
      { 0x7fe4eebbe490ebe0L,0x12a0a4cd7fae11c0L,0x7197cf81e903ba37L,
        0xcf7d4aa8de1c6dd8L } },
    /* 10 << 42 */
    { { 0x92af6bf43fd5684cL,0x2b26eecf80360aa1L,0xbd960f3000546a82L,
        0x407b3c43f59ad8feL },
      { 0x86cae5fe249c82baL,0x9e0faec72463744cL,0x87f551e894916272L,
        0x033f93446ceb0615L } },
    /* 11 << 42 */
    { { 0x1e5eb0d18be82e84L,0x89967f0e7a582fefL,0xbcf687d5a6e921faL,
        0xdfee4cf3d37a09baL },
      { 0x94f06965b493c465L,0x638b9a1c7635c030L,0x7666786466f05e9fL,
        0xccaf6808c04da725L } },
    /* 12 << 42 */
    { { 0xca2eb690768fccfcL,0xf402d37db835b362L,0x0efac0d0e2fdfcceL,
        0xefc9cdefb638d990L },
      { 0x2af12b72d1669a8bL,0x33c536bc5774ccbdL,0x30b21909fb34870eL,
        0xc38fa2f77df25acaL } },
    /* 13 << 42 */
    { { 0x74c5f02bbf81f3f5L,0x0525a5aeaf7e4581L,0x88d2aaba433c54aeL,
        0xed9775db806a56c5L },
      { 0xd320738ac0edb37dL,0x25fdb6ee66cc1f51L,0xac661d1710600d76L,
        0x931ec1f3bdd1ed76L } },
    /* 14 << 42 */
    { { 0x65c11d6219ee43f1L,0x5cd57c3e60829d97L,0xd26c91a3984be6e8L,
        0xf08d93098b0c53bdL },
      { 0x94bc9e5bc016e4eaL,0xd391683911d43d2bL,0x886c5ad773701155L,
        0xe037762620b00715L } },
    /* 15 << 42 */
    { { 0x7f01c9ecaa80ba59L,0x3083411a68538e51L,0x970370f1e88128afL,
        0x625cc3db91dec14bL },
      { 0xfef9666c01ac3107L,0xb2a8d577d5057ac3L,0xb0f2629992be5df7L,
        0xf579c8e500353924L } },
    /* 16 << 42 */
    { { 0xb8fa3d931341ed7aL,0x4223272ca7b59d49L,0x3dcb194783b8c4a4L,
        0x4e413c01ed1302e4L },
      { 0x6d999127e17e44ceL,0xee86bf7533b3adfbL,0xf6902fe625aa96caL,
        0xb73540e4e5aae47dL } },
    /* 17 << 42 */
    { { 0x32801d7b1b4a158cL,0xe571c99e27e2a369L,0x40cb76c010d9f197L,
        0xc308c2893167c0aeL },
      { 0xa6ef9dd3eb7958f2L,0xa7226dfc300879b1L,0x6cd0b3627edf0636L,
        0x4efbce6c7bc37eedL } },
    /* 18 << 42 */
    { { 0x75f92a058d699021L,0x586d4c79772566e3L,0x378ca5f1761ad23aL,
        0x650d86fc1465a8acL },
      { 0x7a4ed457842ba251L,0x6b65e3e642234933L,0xaf1543b731aad657L,
        0xa4cefe98cbfec369L } },
    /* 19 << 42 */
    { { 0xb587da909f47befbL,0x6562e9fb41312d13L,0xa691ea59eff1cefeL,
        0xcc30477a05fc4cf6L },
      { 0xa16324610b0ffd3dL,0xa1f16f3b5b355956L,0x5b148d534224ec24L,
        0xdc834e7bf977012aL } },
    /* 20 << 42 */
    { { 0x7bfc5e75b2c69dbcL,0x3aa77a2903c3da6cL,0xde0df03cca910271L,
        0xcbd5ca4a7806dc55L },
      { 0xe1ca58076db476cbL,0xfde15d625f37a31eL,0xf49af520f41af416L,
        0x96c5c5b17d342db5L } },
    /* 21 << 42 */
    { { 0x155c43b7eb4ceb9bL,0x2e9930104e77371aL,0x1d2987da675d43afL,
        0xef2bc1c08599fd72L },
      { 0x96894b7b9342f6b2L,0x201eadf27c8e71f0L,0xf3479d9f4a1f3efcL,
        0xe0f8a742702a9704L } },
    /* 22 << 42 */
    { { 0xeafd44b6b3eba40cL,0xf9739f29c1c1e0d0L,0x0091471a619d505eL,
        0xc15f9c969d7c263eL },
      { 0x5be4728583afbe33L,0xa3b6d6af04f1e092L,0xe76526b9751a9d11L,
        0x2ec5b26d9a4ae4d2L } },
    /* 23 << 42 */
    { { 0xeb66f4d902f6fb8dL,0x4063c56196912164L,0xeb7050c180ef3000L,
        0x288d1c33eaa5b3f0L },
      { 0xe87c68d607806fd8L,0xb2f7f9d54bbbf50fL,0x25972f3aac8d6627L,
        0xf854777410e8c13bL } },
    /* 24 << 42 */
    { { 0xcc50ef6c872b4a60L,0xab2a34a44613521bL,0x39c5c190983e15d1L,
        0x61dde5df59905512L },
      { 0xe417f6219f2275f3L,0x0750c8b6451d894bL,0x75b04ab978b0bdaaL,
        0x3bfd9fd4458589bdL } },
    /* 25 << 42 */
    { { 0xf1013e30ee9120b6L,0x2b51af9323a4743eL,0xea96ffae48d14d9eL,
        0x71dc0dbe698a1d32L },
      { 0x914962d20180cca4L,0x1ae60677c3568963L,0x8cf227b1437bc444L,
        0xc650c83bc9962c7aL } },
    /* 26 << 42 */
    { { 0x23c2c7ddfe7ccfc4L,0xf925c89d1b929d48L,0x4460f74b06783c33L,
        0xac2c8d49a590475aL },
      { 0xfb40b407b807bba0L,0x9d1e362d69ff8f3aL,0xa33e9681cbef64a4L,
        0x67ece5fa332fb4b2L } },
    /* 27 << 42 */
    { { 0x6900a99b739f10e3L,0xc3341ca9ff525925L,0xee18a626a9e2d041L,
        0xa5a8368529580dddL },
      { 0xf3470c819d7de3cdL,0xedf025862062cf9cL,0xf43522fac010edb0L,
        0x3031413513a4b1aeL } },
    /* 28 << 42 */
    { { 0xc792e02adb22b94bL,0x993d8ae9a1eaa45bL,0x8aad6cd3cd1e1c63L,
        0x89529ca7c5ce688aL },
      { 0x2ccee3aae572a253L,0xe02b643802a21efbL,0xa7091b6ec9430358L,
        0x06d1b1fa9d7db504L } },
    /* 29 << 42 */
    { { 0x58846d32c4744733L,0x40517c71379f9e34L,0x2f65655f130ef6caL,
        0x526e4488f1f3503fL },
      { 0x8467bd177ee4a976L,0x1d9dc913921363d1L,0xd8d24c33b069e041L,
        0x5eb5da0a2cdf7f51L } },
    /* 30 << 42 */
    { { 0x1c0f3cb1197b994fL,0x3c95a6c52843eae9L,0x7766ffc9a6097ea5L,
        0x7bea4093d723b867L },
      { 0xb48e1f734db378f9L,0x70025b00e37b77acL,0x943dc8e7af24ad46L,
        0xb98a15ac16d00a85L } },
    /* 31 << 42 */
    { { 0x3adc38ba2743b004L,0xb1c7f4f7334415eeL,0xea43df8f1e62d05aL,
        0x326189059d76a3b6L },
      { 0x2fbd0bb5a23a0f46L,0x5bc971db6a01918cL,0x7801d94ab4743f94L,
        0xb94df65e676ae22bL } },
    /* 32 << 42 */
    { { 0xaafcbfabaf95894cL,0x7b9bdc07276b2241L,0xeaf983625bdda48bL,
        0x5977faf2a3fcb4dfL },
      { 0xbed042ef052c4b5bL,0x9fe87f71067591f0L,0xc89c73ca22f24ec7L,
        0x7d37fa9ee64a9f1bL } },
    /* 33 << 42 */
    { { 0x2710841a15562627L,0x2c01a613c243b034L,0x1d135c562bc68609L,
        0xc2ca17158b03f1f6L },
      { 0xc9966c2d3eb81d82L,0xc02abf4a8f6df13eL,0x77b34bd78f72b43bL,
        0xaff6218f360c82b0L } },
    /* 34 << 42 */
    { { 0x0aa5726c8d55b9d2L,0xdc0adbe999e9bffbL,0x9097549cefb9e72aL,
        0x167557129dfb3111L },
      { 0xdd8bf984f26847f9L,0xbcb8e387dfb30cb7L,0xc1fd32a75171ef9cL,
        0x977f3fc7389b363fL } },
    /* 35 << 42 */
    { { 0x116eaf2bf4babda0L,0xfeab68bdf7113c8eL,0xd1e3f064b7def526L,
        0x1ac30885e0b3fa02L },
      { 0x1c5a6e7b40142d9dL,0x839b560330921c0bL,0x48f301fa36a116a3L,
        0x380e1107cfd9ee6dL } },
    /* 36 << 42 */
    { { 0x7945ead858854be1L,0x4111c12ecbd4d49dL,0xece3b1ec3a29c2efL,
        0x6356d4048d3616f5L },
      { 0x9f0d6a8f594d320eL,0x0989316df651ccd2L,0x6c32117a0f8fdde4L,
        0x9abe5cc5a26a9bbcL } },
    /* 37 << 42 */
    { { 0xcff560fb9723f671L,0x21b2a12d7f3d593cL,0xe4cb18da24ba0696L,
        0x186e2220c3543384L },
      { 0x722f64e088312c29L,0x94282a9917dc7752L,0x62467bbf5a85ee89L,
        0xf435c650f10076a0L } },
    /* 38 << 42 */
    { { 0xc9ff153943b3a50bL,0x7132130c1a53efbcL,0x31bfe063f7b0c5b7L,
        0xb0179a7d4ea994ccL },
      { 0x12d064b3c85f455bL,0x472593288f6e0062L,0xf64e590bb875d6d9L,
        0x22dd6225ad92bcc7L } },
    /* 39 << 42 */
    { { 0xb658038eb9c3bd6dL,0x00cdb0d6fbba27c8L,0x0c6813371062c45dL,
        0xd8515b8c2d33407dL },
      { 0xcb8f699e8cbb5ecfL,0x8c4347f8c608d7d8L,0x2c11850abb3e00dbL,
        0x20a8dafdecb49d19L } },
    /* 40 << 42 */
    { { 0xbd78148045ee2f40L,0x75e354af416b60cfL,0xde0b58a18d49a8c4L,
        0xe40e94e2fa359536L },
      { 0xbd4fa59f62accd76L,0x05cf466a8c762837L,0xb5abda99448c277bL,
        0x5a9e01bf48b13740L } },
    /* 41 << 42 */
    { { 0x9d457798326aad8dL,0xbdef4954c396f7e7L,0x6fb274a2c253e292L,
        0x2800bf0a1cfe53e7L },
      { 0x22426d3144438fd4L,0xef2339235e259f9aL,0x4188503c03f66264L,
        0x9e5e7f137f9fdfabL } },
    /* 42 << 42 */
    { { 0x565eb76c5fcc1abaL,0xea63254859b5bff8L,0x5587c087aab6d3faL,
        0x92b639ea6ce39c1bL },
      { 0x0706e782953b135cL,0x7308912e425268efL,0x599e92c7090e7469L,
        0x83b90f529bc35e75L } },
    /* 43 << 42 */
    { { 0x4750b3d0244975b3L,0xf3a4435811965d72L,0x179c67749c8dc751L,
        0xff18cdfed23d9ff0L },
      { 0xc40138332028e247L,0x96e280e2f3bfbc79L,0xf60417bdd0880a84L,
        0x263c9f3d2a568151L } },
    /* 44 << 42 */
    { { 0x36be15b32d2ce811L,0x846dc0c2f8291d21L,0x5cfa0ecb789fcfdbL,
        0x45a0beedd7535b9aL },
      { 0xec8e9f0796d69af1L,0x31a7c5b8599ab6dcL,0xd36d45eff9e2e09fL,
        0x3cf49ef1dcee954bL } },
    /* 45 << 42 */
    { { 0x6be34cf3086cff9bL,0x88dbd49139a3360fL,0x1e96b8cc0dbfbd1dL,
        0xc1e5f7bfcb7e2552L },
      { 0x0547b21428819d98L,0xc770dd9c7aea9dcbL,0xaef0d4c7041d68c8L,
        0xcc2b981813cb9ba8L } },
    /* 46 << 42 */
    { { 0x7fc7bc76fe86c607L,0x6b7b9337502a9a95L,0x1948dc27d14dab63L,
        0x249dd198dae047beL },
      { 0xe8356584a981a202L,0x3531dd183a893387L,0x1be11f90c85c7209L,
        0x93d2fe1ee2a52b5aL } },
    /* 47 << 42 */
    { { 0x8225bfe2ec6d6b97L,0x9cf6d6f4bd0aa5deL,0x911459cb54779f5fL,
        0x5649cddb86aeb1f3L },
      { 0x321335793f26ce5aL,0xc289a102550f431eL,0x559dcfda73b84c6fL,
        0x84973819ee3ac4d7L } },
    /* 48 << 42 */
    { { 0xb51e55e6f2606a82L,0xe25f706190f2fb57L,0xacef6c2ab1a4e37cL,
        0x864e359d5dcf2706L },
      { 0x479e6b187ce57316L,0x2cab25003a96b23dL,0xed4898628ef16df7L,
        0x2056538cef3758b5L } },
    /* 49 << 42 */
    { { 0xa7df865ef15d3101L,0x80c5533a61b553d7L,0x366e19974ed14294L,
        0x6620741fb3c0bcd6L },
      { 0x21d1d9c4edc45418L,0x005b859ec1cc4a9dL,0xdf01f630a1c462f0L,
        0x15d06cf3f26820c7L } },
    /* 50 << 42 */
    { { 0x9f7f24ee3484be47L,0x2ff33e964a0c902fL,0x00bdf4575a0bc453L,
        0x2378dfaf1aa238dbL },
      { 0x272420ec856720f2L,0x2ad9d95b96797291L,0xd1242cc6768a1558L,
        0x2e287f8b5cc86aa8L } },
    /* 51 << 42 */
    { { 0x796873d0990cecaaL,0xade55f81675d4080L,0x2645eea321f0cd84L,
        0x7a1efa0fb4e17d02L },
      { 0xf6858420037cc061L,0x682e05f0d5d43e12L,0x59c3699427218710L,
        0x85cbba4d3f7cd2fcL } },
    /* 52 << 42 */
    { { 0x726f97297a3cd22aL,0x9f8cd5dc4a628397L,0x17b93ab9c23165edL,
        0xff5f5dbf122823d4L },
      { 0xc1e4e4b5654a446dL,0xd1a9496f677257baL,0x6387ba94de766a56L,
        0x23608bc8521ec74aL } },
    /* 53 << 42 */
    { { 0x16a522d76688c4d4L,0x9d6b428207373abdL,0xa62f07acb42efaa3L,
        0xf73e00f7e3b90180L },
      { 0x36175fec49421c3eL,0xc4e44f9b3dcf2678L,0x76df436b7220f09fL,
        0x172755fb3aa8b6cfL } },
    /* 54 << 42 */
    { { 0xbab89d57446139ccL,0x0a0a6e025fe0208fL,0xcdbb63e211e5d399L,
        0x33ecaa12a8977f0bL },
      { 0x59598b21f7c42664L,0xb3e91b32ab65d08aL,0x035822eef4502526L,
        0x1dcf0176720a82a9L } },
    /* 55 << 42 */
    { { 0x50f8598f3d589e02L,0xdf0478ffb1d63d2cL,0x8b8068bd1571cd07L,
        0x30c3aa4fd79670cdL },
      { 0x25e8fd4b941ade7fL,0x3d1debdc32790011L,0x65b6dcbd3a3f9ff0L,
        0x282736a4793de69cL } },
    /* 56 << 42 */
    { { 0xef69a0c3d41d3bd3L,0xb533b8c907a26bdeL,0xe2801d97db2edf9fL,
        0xdc4a8269e1877af0L },
      { 0x6c1c58513d590dbeL,0x84632f6bee4e9357L,0xd36d36b779b33374L,
        0xb46833e39bbca2e6L } },
    /* 57 << 42 */
    { { 0x37893913f7fc0586L,0x385315f766bf4719L,0x72c56293b31855dcL,
        0xd1416d4e849061feL },
      { 0xbeb3ab7851047213L,0x447f6e61f040c996L,0xd06d310d638b1d0cL,
        0xe28a413fbad1522eL } },
    /* 58 << 42 */
    { { 0x685a76cb82003f86L,0x610d07f70bcdbca3L,0x6ff660219ca4c455L,
        0x7df39b87cea10eecL },
      { 0xb9255f96e22db218L,0x8cc6d9eb08a34c44L,0xcd4ffb86859f9276L,
        0x8fa15eb250d07335L } },
    /* 59 << 42 */
    { { 0xdf553845cf2c24b5L,0x89f66a9f52f9c3baL,0x8f22b5b9e4a7ceb3L,
        0xaffef8090e134686L },
      { 0x3e53e1c68eb8fac2L,0x93c1e4eb28aec98eL,0xb6b91ec532a43bcbL,
        0x2dbfa947b2d74a51L } },
    /* 60 << 42 */
    { { 0xe065d190ca84bad7L,0xfb13919fad58e65cL,0x3c41718bf1cb6e31L,
        0x688969f006d05c3fL },
      { 0xd4f94ce721264d45L,0xfdfb65e97367532bL,0x5b1be8b10945a39dL,
        0x229f789c2b8baf3bL } },
    /* 61 << 42 */
    { { 0xd8f41f3e6f49f15dL,0x678ce828907f0792L,0xc69ace82fca6e867L,
        0x106451aed01dcc89L },
      { 0x1bb4f7f019fc32d2L,0x64633dfcb00c52d2L,0x8f13549aad9ea445L,
        0x99a3bf50fb323705L } },
    /* 62 << 42 */
    { { 0x0c9625a2534d4dbcL,0x45b8f1d1c2a2fea3L,0x76ec21a1a530fc1aL,
        0x4bac9c2a9e5bd734L },
      { 0x5996d76a7b4e3587L,0x0045cdee1182d9e3L,0x1aee24b91207f13dL,
        0x66452e9797345a41L } },
    /* 63 << 42 */
    { { 0x16e5b0549f950cd0L,0x9cc72fb1d7fdd075L,0x6edd61e766249663L,
        0xde4caa4df043cccbL },
      { 0x11b1f57a55c7ac17L,0x779cbd441a85e24dL,0x78030f86e46081e7L,
        0xfd4a60328e20f643L } },
    /* 64 << 42 */
    { { 0xcc7a64880a750c0fL,0x39bacfe34e548e83L,0x3d418c760c110f05L,
        0x3e4daa4cb1f11588L },
      { 0x2733e7b55ffc69ffL,0x46f147bc92053127L,0x885b2434d722df94L,
        0x6a444f65e6fc6b7cL } },
    /* 0 << 49 */
    { { 0x00, 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x00, 0x00 } },
    /* 1 << 49 */
    { { 0x7a1a465ac3f16ea8L,0x115a461db2f1d11cL,0x4767dd956c68a172L,
        0x3392f2ebd13a4698L },
      { 0xc7a99ccde526cdc7L,0x8e537fdc22292b81L,0x76d8cf69a6d39198L,
        0xffc5ff432446852dL } },
    /* 2 << 49 */
    { { 0x97b14f7ea90567e6L,0x513257b7b6ae5cb7L,0x85454a3c9f10903dL,
        0xd8d2c9ad69bc3724L },
      { 0x38da93246b29cb44L,0xb540a21d77c8cbacL,0x9bbfe43501918e42L,
        0xfffa707a56c3614eL } },
    /* 3 << 49 */
    { { 0x0ce4e3f1d4e353b7L,0x062d8a14ef46b0a0L,0x6408d5ab574b73fdL,
        0xbc41d1c9d3273ffdL },
      { 0x3538e1e76be77800L,0x71fe8b37c5655031L,0x1cd916216b9b331aL,
        0xad825d0bbb388f73L } },
    /* 4 << 49 */
    { { 0x56c2e05b1cb76219L,0x0ec0bf9171567e7eL,0xe7076f8661c4c910L,
        0xd67b085bbabc04d9L },
      { 0x9fb904595e93a96aL,0x7526c1eafbdc249aL,0x0d44d367ecdd0bb7L,
        0x953999179dc0d695L } },
    /* 5 << 49 */
    { { 0x61360ee99e240d18L,0x057cdcacb4b94466L,0xe7667cd12fe5325cL,
        0x1fa297b521974e3bL },
      { 0xfa4081e7db083d76L,0x31993be6f206bd15L,0x8949269b14c19f8cL,
        0x21468d72a9d92357L } },
    /* 6 << 49 */
    { { 0x2ccbc583a4c506ecL,0x957ed188d1acfe97L,0x8baed83312f1aea2L,
        0xef2a6cb48325362dL },
      { 0x130dde428e195c43L,0xc842025a0e6050c6L,0x2da972a708686a5dL,
        0xb52999a1e508b4a8L } },
    /* 7 << 49 */
    { { 0xd9f090b910a5a8bdL,0xca91d249096864daL,0x8e6a93be3f67dbc1L,
        0xacae6fbaf5f4764cL },
      { 0x1563c6e0d21411a0L,0x28fa787fda0a4ad8L,0xd524491c908c8030L,
        0x1257ba0e4c795f07L } },
    /* 8 << 49 */
    { { 0x83f49167ceca9754L,0x426d2cf64b7939a0L,0x2555e355723fd0bfL,
        0xa96e6d06c4f144e2L },
      { 0x4768a8dd87880e61L,0x15543815e508e4d5L,0x09d7e772b1b65e15L,
        0x63439dd6ac302fa0L } },
    /* 9 << 49 */
    { { 0xb93f802fc14e35c2L,0x71735b7c4341333cL,0x03a2510416d4f362L,
        0x3f4d069bbf433c8eL },
      { 0x0d83ae01f78f5a7cL,0x50a8ffbe7c4eed07L,0xc74f890676e10f83L,
        0x7d0809669ddaf8e1L } },
    /* 10 << 49 */
    { { 0xb11df8e1698e04ccL,0x877be203169005c8L,0x32749e8c4f3c6179L,
        0x2dbc9d0a7853fc05L },
      { 0x187d4f939454d937L,0xe682ce9db4800e1bL,0xa9129ad8165e68e8L,
        0x0fe29735be7f785bL } },
    /* 11 << 49 */
    { { 0x5303f40c5b9e02b7L,0xa37c969235ee04e8L,0x5f46cc2034d6632bL,
        0x55ef72b296ac545bL },
      { 0xabec5c1f7b91b062L,0x0a79e1c7bb33e821L,0xbb04b4283a9f4117L,
        0x0de1f28ffd2a475aL } },
    /* 12 << 49 */
    { { 0x31019ccf3a4434b4L,0xa34581111a7954dcL,0xa9dac80de34972a7L,
        0xb043d05474f6b8ddL },
      { 0x021c319e11137b1aL,0x00a754ceed5cc03fL,0x0aa2c794cbea5ad4L,
        0x093e67f470c015b6L } },
    /* 13 << 49 */
    { { 0x72cdfee9c97e3f6bL,0xc10bcab4b6da7461L,0x3b02d2fcb59806b9L,
        0x85185e89a1de6f47L },
      { 0x39e6931f0eb6c4d4L,0x4d4440bdd4fa5b04L,0x5418786e34be7eb8L,
        0x6380e5219d7259bcL } },
    /* 14 << 49 */
    { { 0x20ac0351d598d710L,0x272c4166cb3a4da4L,0xdb82fe1aca71de1fL,
        0x746e79f2d8f54b0fL },
      { 0x6e7fc7364b573e9bL,0x75d03f46fd4b5040L,0x5c1cc36d0b98d87bL,
        0x513ba3f11f472da1L } },
    /* 15 << 49 */
    { { 0x79d0af26abb177ddL,0xf82ab5687891d564L,0x2b6768a972232173L,
        0xefbb3bb08c1f6619L },
      { 0xb29c11dba6d18358L,0x519e2797b0916d3aL,0xd4dc18f09188e290L,
        0x648e86e398b0ca7fL } },
    /* 16 << 49 */
    { { 0x859d3145983c38b5L,0xb14f176c637abc8bL,0x2793fb9dcaff7be6L,
        0xebe5a55f35a66a5aL },
      { 0x7cec1dcd9f87dc59L,0x7c595cd3fbdbf560L,0x5b543b2226eb3257L,
        0x69080646c4c935fdL } },
    /* 17 << 49 */
    { { 0x7f2e440381e9ede3L,0x243c3894caf6df0aL,0x7c605bb11c073b11L,
        0xcd06a541ba6a4a62L },
      { 0x2916894949d4e2e5L,0x33649d074af66880L,0xbfc0c885e9a85035L,
        0xb4e52113fc410f4bL } },
    /* 18 << 49 */
    { { 0xdca3b70678a6513bL,0x92ea4a2a9edb1943L,0x02642216db6e2dd8L,
        0x9b45d0b49fd57894L },
      { 0x114e70dbc69d11aeL,0x1477dd194c57595fL,0xbc2208b4ec77c272L,
        0x95c5b4d7db68f59cL } },
    /* 19 << 49 */
    { { 0xb8c4fc6342e532b7L,0x386ba4229ae35290L,0xfb5dda42d201ecbcL,
        0x2353dc8ba0e38fd6L },
      { 0x9a0b85ea68f7e978L,0x96ec56822ad6d11fL,0x5e279d6ce5f6886dL,
        0xd3fe03cd3cb1914dL } },
    /* 20 << 49 */
    { { 0xfe541fa47ea67c77L,0x952bd2afe3ea810cL,0x791fef568d01d374L,
        0xa3a1c6210f11336eL },
      { 0x5ad0d5a9c7ec6d79L,0xff7038af3225c342L,0x003c6689bc69601bL,
        0x25059bc745e8747dL } },
    /* 21 << 49 */
    { { 0xfa4965b2f2086fbfL,0xf6840ea686916078L,0xd7ac762070081d6cL,
        0xe600da31b5328645L },
      { 0x01916f63529b8a80L,0xe80e48582d7d6f3eL,0x29eb0fe8d664ca7cL,
        0xf017637be7b43b0cL } },
    /* 22 << 49 */
    { { 0x9a75c80676cb2566L,0x8f76acb1b24892d9L,0x7ae7b9cc1f08fe45L,
        0x19ef73296a4907d8L },
      { 0x2db4ab715f228bf0L,0xf3cdea39817032d7L,0x0b1f482edcabe3c0L,
        0x3baf76b4bb86325cL } },
    /* 23 << 49 */
    { { 0xd49065e010089465L,0x3bab5d298e77c596L,0x7636c3a6193dbd95L,
        0xdef5d294b246e499L },
      { 0xb22c58b9286b2475L,0xa0b93939cd80862bL,0x3002c83af0992388L,
        0x6de01f9beacbe14cL } },
    /* 24 << 49 */
    { { 0x6aac688eadd70482L,0x708de92a7b4a4e8aL,0x75b6dd73758a6eefL,
        0xea4bf352725b3c43L },
      { 0x10041f2c87912868L,0xb1b1be95ef09297aL,0x19ae23c5a9f3860aL,
        0xc4f0f839515dcf4bL } },
    /* 25 << 49 */
    { { 0x3c7ecca397f6306aL,0x744c44ae68a3a4b0L,0x69cd13a0b3a1d8a2L,
        0x7cad0a1e5256b578L },
      { 0xea653fcd33791d9eL,0x9cc2a05d74b2e05fL,0x73b391dcfd7affa2L,
        0xddb7091eb6b05442L } },
    /* 26 << 49 */
    { { 0xc71e27bf8538a5c6L,0x195c63dd89abff17L,0xfd3152851b71e3daL,
        0x9cbdfda7fa680fa0L },
      { 0x9db876ca849d7eabL,0xebe2764b3c273271L,0x663357e3f208dceaL,
        0x8c5bd833565b1b70L } },
    /* 27 << 49 */
    { { 0xccc3b4f59837fc0dL,0x9b641ba8a79cf00fL,0x7428243ddfdf3990L,
        0x83a594c4020786b1L },
      { 0xb712451a526c4502L,0x9d39438e6adb3f93L,0xfdb261e3e9ff0ccdL,
        0x80344e3ce07af4c3L } },
    /* 28 << 49 */
    { { 0x75900d7c2fa4f126L,0x08a3b8655c99a232L,0x2478b6bfdb25e0c3L,
        0x482cc2c271db2edfL },
      { 0x37df7e645f321bb8L,0x8a93821b9a8005b4L,0x3fa2f10ccc8c1958L,
        0x0d3322182c269d0aL } },
    /* 29 << 49 */
    { { 0x20ab8119e246b0e6L,0xb39781e4d349fd17L,0xd293231eb31aa100L,
        0x4b779c97bb032168L },
      { 0x4b3f19e1c8470500L,0x45b7efe90c4c869dL,0xdb84f38aa1a6bbccL,
        0x3b59cb15b2fddbc1L } },
    /* 30 << 49 */
    { { 0xba5514df3fd165e8L,0x499fd6a9061f8811L,0x72cd1fe0bfef9f00L,
        0x120a4bb979ad7e8aL },
      { 0xf2ffd0955f4a5ac5L,0xcfd174f195a7a2f0L,0xd42301ba9d17baf1L,
        0xd2fa487a77f22089L } },
    /* 31 << 49 */
    { { 0x9cb09efeb1dc77e1L,0xe956693921c99682L,0x8c5469016c6067bbL,
        0xfd37857461c24456L },
      { 0x2b6a6cbe81796b33L,0x62d550f658e87f8bL,0x1b763e1c7f1b01b4L,
        0x4b93cfea1b1b5e12L } },
    /* 32 << 49 */
    { { 0xb93452381d531696L,0x57201c0088cdde69L,0xdde922519a86afc7L,
        0xe3043895bd35cea8L },
      { 0x7608c1e18555970dL,0x8267dfa92535935eL,0xd4c60a57322ea38bL,
        0xe0bf7977804ef8b5L } },
    /* 33 << 49 */
    { { 0x1a0dab28c06fece4L,0xd405991e94e7b49dL,0xc542b6d2706dab28L,
        0xcb228da3a91618fbL },
      { 0x224e4164107d1ceaL,0xeb9fdab3d0f5d8f1L,0xc02ba3860d6e41cdL,
        0x676a72c59b1f7146L } },
    /* 34 << 49 */
    { { 0xffd6dd984d6cb00bL,0xcef9c5cade2e8d7cL,0xa1bbf5d7641c7936L,
        0x1b95b230ee8f772eL },
      { 0xf765a92ee8ac25b1L,0xceb04cfc3a18b7c6L,0x27944cef0acc8966L,
        0xcbb3c957434c1004L } },
    /* 35 << 49 */
    { { 0x9c9971a1a43ff93cL,0x5bc2db17a1e358a9L,0x45b4862ea8d9bc82L,
        0x70ebfbfb2201e052L },
      { 0xafdf64c792871591L,0xea5bcae6b42d0219L,0xde536c552ad8f03cL,
        0xcd6c3f4da76aa33cL } },
    /* 36 << 49 */
    { { 0xbeb5f6230bca6de3L,0xdd20dd99b1e706fdL,0x90b3ff9dac9059d4L,
        0x2d7b29027ccccc4eL },
      { 0x8a090a59ce98840fL,0xa5d947e08410680aL,0x49ae346a923379a5L,
        0x7dbc84f9b28a3156L } },
    /* 37 << 49 */
    { { 0xfd40d91654a1aff2L,0xabf318ba3a78fb9bL,0x50152ed83029f95eL,
        0x9fc1dd77c58ad7faL },
      { 0x5fa5791513595c17L,0xb95046688f62b3a9L,0x907b5b24ff3055b0L,
        0x2e995e359a84f125L } },
    /* 38 << 49 */
    { { 0x87dacf697e9bbcfbL,0x95d0c1d6e86d96e3L,0x65726e3c2d95a75cL,
        0x2c3c9001acd27f21L },
      { 0x1deab5616c973f57L,0x108b7e2ca5221643L,0x5fee9859c4ef79d4L,
        0xbd62b88a40d4b8c6L } },
    /* 39 << 49 */
    { { 0xb4dd29c4197c75d6L,0x266a6df2b7076febL,0x9512d0ea4bf2df11L,
        0x1320c24f6b0cc9ecL },
      { 0x6bb1e0e101a59596L,0x8317c5bbeff9aaacL,0x65bb405e385aa6c9L,
        0x613439c18f07988fL } },
    /* 40 << 49 */
    { { 0xd730049f16a66e91L,0xe97f2820fa1b0e0dL,0x4131e003304c28eaL,
        0x820ab732526bac62L },
      { 0xb2ac9ef928714423L,0x54ecfffaadb10cb2L,0x8781476ef886a4ccL,
        0x4b2c87b5db2f8d49L } },
    /* 41 << 49 */
    { { 0xe857cd200a44295dL,0x707d7d2158c6b044L,0xae8521f9f596757cL,
        0x87448f0367b2b714L },
      { 0x13a9bc455ebcd58dL,0x79bcced99122d3c1L,0x3c6442479e076642L,
        0x0cf227782df4767dL } },
    /* 42 << 49 */
    { { 0x5e61aee471d444b6L,0x211236bfc5084a1dL,0x7e15bc9a4fd3eaf6L,
        0x68df2c34ab622bf5L },
      { 0x9e674f0f59bf4f36L,0xf883669bd7f34d73L,0xc48ac1b831497b1dL,
        0x323b925d5106703bL } },
    /* 43 << 49 */
    { { 0x22156f4274082008L,0xeffc521ac8482bcbL,0x5c6831bf12173479L,
        0xcaa2528fc4739490L },
      { 0x84d2102a8f1b3c4dL,0xcf64dfc12d9bec0dL,0x433febad78a546efL,
        0x1f621ec37b73cef1L } },
    /* 44 << 49 */
    { { 0x6aecd62737338615L,0x162082ab01d8edf6L,0x833a811919e86b66L,
        0x6023a251d299b5dbL },
      { 0xf5bb0c3abbf04b89L,0x6735eb69ae749a44L,0xd0e058c54713de3bL,
        0xfdf2593e2c3d4ccdL } },
    /* 45 << 49 */
    { { 0x1b8f414efdd23667L,0xdd52aacafa2015eeL,0x3e31b517bd9625ffL,
        0x5ec9322d8db5918cL },
      { 0xbc73ac85a96f5294L,0x82aa5bf361a0666aL,0x49755810bf08ac42L,
        0xd21cdfd5891cedfcL } },
    /* 46 << 49 */
    { { 0x918cb57b67f8be10L,0x365d1a7c56ffa726L,0x2435c5046532de93L,
        0xc0fc5e102674cd02L },
      { 0x6e51fcf89cbbb142L,0x1d436e5aafc50692L,0x766bffff3fbcae22L,
        0x3148c2fdfd55d3b8L } },
    /* 47 << 49 */
    { { 0x52c7fdc9233222faL,0x89ff1092e419fb6bL,0x3cd6db9925254977L,
        0x2e85a1611cf12ca7L },
      { 0xadd2547cdc810bc9L,0xea3f458f9d257c22L,0x642c1fbe27d6b19bL,
        0xed07e6b5140481a6L } },
    /* 48 << 49 */
    { { 0x6ada1d4286d2e0f8L,0xe59201220e8a9fd5L,0x02c936af708c1b49L,
        0x60f30fee2b4bfaffL },
      { 0x6637ad06858e6a61L,0xce4c77673fd374d0L,0x39d54b2d7188defbL,
        0xa8c9d250f56a6b66L } },
    /* 49 << 49 */
    { { 0x58fc0f5eb24fe1dcL,0x9eaf9dee6b73f24cL,0xa90d588b33650705L,
        0xde5b62c5af2ec729L },
      { 0x5c72cfaed3c2b36eL,0x868c19d5034435daL,0x88605f93e17ee145L,
        0xaa60c4ee77a5d5b1L } },
    /* 50 << 49 */
    { { 0xbcf5bfd23b60c472L,0xaf4ef13ceb1d3049L,0x373f44fce13895c9L,
        0xf29b382f0cbc9822L },
      { 0x1bfcb85373efaef6L,0xcf56ac9ca8c96f40L,0xd7adf1097a191e24L,
        0x98035f44bf8a8dc2L } },
    /* 51 << 49 */
    { { 0xf40a71b91e750c84L,0xc57f7b0c5dc6c469L,0x49a0e79c6fbc19c1L,
        0x6b0f5889a48ebdb8L },
      { 0x5d3fd084a07c4e9fL,0xc3830111ab27de14L,0x0e4929fe33e08dccL,
        0xf4a5ad2440bb73a3L } },
    /* 52 << 49 */
    { { 0xde86c2bf490f97caL,0x288f09c667a1ce18L,0x364bb8861844478dL,
        0x7840fa42ceedb040L },
      { 0x1269fdd25a631b37L,0x94761f1ea47c8b7dL,0xfc0c2e17481c6266L,
        0x85e16ea23daa5fa7L } },
    /* 53 << 49 */
    { { 0xccd8603392491048L,0x0c2f6963f4d402d7L,0x6336f7dfdf6a865cL,
        0x0a2a463cb5c02a87L },
      { 0xb0e29be7bf2f12eeL,0xf0a2200266bad988L,0x27f87e039123c1d7L,
        0x21669c55328a8c98L } },
    /* 54 << 49 */
    { { 0x186b980392f14529L,0xd3d056cc63954df3L,0x2f03fd58175a46f6L,
        0x63e34ebe11558558L },
      { 0xe13fedee5b80cfa5L,0xe872a120d401dbd1L,0x52657616e8a9d667L,
        0xbc8da4b6e08d6693L } },
    /* 55 << 49 */
    { { 0x370fb9bb1b703e75L,0x6773b186d4338363L,0x18dad378ecef7bffL,
        0xaac787ed995677daL },
      { 0x4801ea8b0437164bL,0xf430ad2073fe795eL,0xb164154d8ee5eb73L,
        0x0884ecd8108f7c0eL } },
    /* 56 << 49 */
    { { 0x0e6ec0965f520698L,0x640631fe44f7b8d9L,0x92fd34fca35a68b9L,
        0x9c5a4b664d40cf4eL },
      { 0x949454bf80b6783dL,0x80e701fe3a320a10L,0x8d1a564a1a0a39b2L,
        0x1436d53d320587dbL } },
    /* 57 << 49 */
    { { 0xf5096e6d6556c362L,0xbc23a3c0e2455d7eL,0x3a7aee54807230f9L,
        0x9ba1cfa622ae82fdL },
      { 0x833a057a99c5d706L,0x8be85f4b842315c9L,0xd083179a66a72f12L,
        0x2fc77d5dcdcc73cdL } },
    /* 58 << 49 */
    { { 0x22b88a805616ee30L,0xfb09548fe7ab1083L,0x8ad6ab0d511270cdL,
        0x61f6c57a6924d9abL },
      { 0xa0f7bf7290aecb08L,0x849f87c90df784a4L,0x27c79c15cfaf1d03L,
        0xbbf9f675c463faceL } },
    /* 59 << 49 */
    { { 0x91502c65765ba543L,0x18ce3cac42ea60ddL,0xe5cee6ac6e43ecb3L,
        0x63e4e91068f2aeebL },
      { 0x26234fa3c85932eeL,0x96883e8b4c90c44dL,0x29b9e738a18a50f6L,
        0xbfc62b2a3f0420dfL } },
    /* 60 << 49 */
    { { 0xd22a7d906d3e1fa9L,0x17115618fe05b8a3L,0x2a0c9926bb2b9c01L,
        0xc739fcc6e07e76a2L },
      { 0x540e9157165e439aL,0x06353a626a9063d8L,0x84d9559461e927a3L,
        0x013b9b26e2e0be7fL } },
    /* 61 << 49 */
    { { 0x4feaec3b973497f1L,0x15c0f94e093ebc2dL,0x6af5f22733af0583L,
        0x0c2af206c61f3340L },
      { 0xd25dbdf14457397cL,0x2e8ed017cabcbae0L,0xe3010938c2815306L,
        0xbaa99337e8c6cd68L } },
    /* 62 << 49 */
    { { 0x085131823b0ec7deL,0x1e1b822b58df05dfL,0x5c14842fa5c3b683L,
        0x98fe977e3eba34ceL },
      { 0xfd2316c20d5e8873L,0xe48d839abd0d427dL,0x495b2218623fc961L,
        0x24ee56e7b46fba5eL } },
    /* 63 << 49 */
    { { 0x9184a55b91e4de58L,0xa7488ca5dfdea288L,0xa723862ea8dcc943L,
        0x92d762b2849dc0fcL },
      { 0x3c444a12091ff4a9L,0x581113fa0cada274L,0xb9de0a4530d8eae2L,
        0x5e0fcd85df6b41eaL } },
    /* 64 << 49 */
    { { 0x6233ea68c094dbb5L,0xb77d062ed968d410L,0x3e719bbc58b3002dL,
        0x68e7dd3d3dc49d58L },
      { 0x8d825740013a5e58L,0x213117473c9e3c1bL,0x0cb0a2a77c99b6abL,
        0x5c48a3b3c2f888f2L } },
    /* 0 << 56 */
    { { 0x00, 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x00, 0x00 } },
    /* 1 << 56 */
    { { 0xc7913e91991724f3L,0x5eda799c39cbd686L,0xddb595c763d4fc1eL,
        0x6b63b80bac4fed54L },
      { 0x6ea0fc697e5fb516L,0x737708bad0f1c964L,0x9628745f11a92ca5L,
        0x61f379589a86967aL } },
    /* 2 << 56 */
    { { 0x9af39b2caa665072L,0x78322fa4efd324efL,0x3d153394c327bd31L,
        0x81d5f2713129dab0L },
      { 0xc72e0c42f48027f5L,0xaa40cdbc8536e717L,0xf45a657a2d369d0fL,
        0xb03bbfc4ea7f74e6L } },
    /* 3 << 56 */
    { { 0x46a8c4180d738dedL,0x6f1a5bb0e0de5729L,0xf10230b98ba81675L,
        0x32c6f30c112b33d4L },
      { 0x7559129dd8fffb62L,0x6a281b47b459bf05L,0x77c1bd3afa3b6776L,
        0x0709b3807829973aL } },
    /* 4 << 56 */
    { { 0x8c26b232a3326505L,0x38d69272ee1d41bfL,0x0459453effe32afaL,
        0xce8143ad7cb3ea87L },
      { 0x932ec1fa7e6ab666L,0x6cd2d23022286264L,0x459a46fe6736f8edL,
        0x50bf0d009eca85bbL } },
    /* 5 << 56 */
    { { 0x0b825852877a21ecL,0x300414a70f537a94L,0x3f1cba4021a9a6a2L,
        0x50824eee76943c00L },
      { 0xa0dbfcecf83cba5dL,0xf953814893b4f3c0L,0x6174416248f24dd7L,
        0x5322d64de4fb09ddL } },
    /* 6 << 56 */
    { { 0x574473843d9325f3L,0xa9bef2d0f371cb84L,0x77d2188ba61e36c5L,
        0xbbd6a7d7c602df72L },
      { 0xba3aa9028f61bc0bL,0xf49085ed6ed0b6a1L,0x8bc625d6ae6e8298L,
        0x832b0b1da2e9c01dL } },
    /* 7 << 56 */
    { { 0xa337c447f1f0ced1L,0x800cc7939492dd2bL,0x4b93151dbea08efaL,
        0x820cf3f8de0a741eL },
      { 0xff1982dc1c0f7d13L,0xef92196084dde6caL,0x1ad7d97245f96ee3L,
        0x319c8dbe29dea0c7L } },
    /* 8 << 56 */
    { { 0xd3ea38717b82b99bL,0x75922d4d470eb624L,0x8f66ec543b95d466L,
        0x66e673ccbee1e346L },
      { 0x6afe67c4b5f2b89aL,0x3de9c1e6290e5cd3L,0x8c278bb6310a2adaL,
        0x420fa3840bdb323bL } },
    /* 9 << 56 */
    { { 0x0ae1d63b0eb919b0L,0xd74ee51da74b9620L,0x395458d0a674290cL,
        0x324c930f4620a510L },
      { 0x2d1f4d19fbac27d4L,0x4086e8ca9bedeeacL,0x0cdd211b9b679ab8L,
        0x5970167d7090fec4L } },
    /* 10 << 56 */
    { { 0x3420f2c9faf1fc63L,0x616d333a328c8bb4L,0x7d65364c57f1fe4aL,
        0x9343e87755e5c73aL },
      { 0x5795176be970e78cL,0xa36ccebf60533627L,0xfc7c738009cdfc1bL,
        0xb39a2afeb3fec326L } },
    /* 11 << 56 */
    { { 0xb7ff1ba16224408aL,0xcc856e92247cfc5eL,0x01f102e7c18bc493L,
        0x4613ab742091c727L },
      { 0xaa25e89cc420bf2bL,0x00a5317690337ec2L,0xd2be9f437d025fc7L,
        0x3316fb856e6fe3dcL } },
    /* 12 << 56 */
    { { 0x27520af59ac50814L,0xfdf95e789a8e4223L,0xb7e7df2a56bec5a0L,
        0xf7022f7ddf159e5dL },
      { 0x93eeeab1cac1fe8fL,0x8040188c37451168L,0x7ee8aa8ad967dce6L,
        0xfa0e79e73abc9299L } },
    /* 13 << 56 */
    { { 0x67332cfc2064cfd1L,0x339c31deb0651934L,0x719b28d52a3bcbeaL,
        0xee74c82b9d6ae5c6L },
      { 0x0927d05ebaf28ee6L,0x82cecf2c9d719028L,0x0b0d353eddb30289L,
        0xfe4bb977fddb2e29L } },
    /* 14 << 56 */
    { { 0xbb5bb990640bfd9eL,0xd226e27782f62108L,0x4bf0098502ffdd56L,
        0x7756758a2ca1b1b5L },
      { 0xc32b62a35285fe91L,0xedbc546a8c9cd140L,0x1e47a013af5cb008L,
        0xbca7e720073ce8f2L } },
    /* 15 << 56 */
    { { 0xe10b2ab817a91caeL,0xb89aab6508e27f63L,0x7b3074a7dba3ddf9L,
        0x1c20ce09330c2972L },
      { 0x6b9917b45fcf7e33L,0xe6793743945ceb42L,0x18fc22155c633d19L,
        0xad1adb3cc7485474L } },
    /* 16 << 56 */
    { { 0x646f96796424c49bL,0xf888dfe867c241c9L,0xe12d4b9324f68b49L,
        0x9a6b62d8a571df20L },
      { 0x81b4b26d179483cbL,0x666f96329511fae2L,0xd281b3e4d53aa51fL,
        0x7f96a7657f3dbd16L } },
    /* 17 << 56 */
    { { 0xa7f8b5bf074a30ceL,0xd7f52107005a32e6L,0x6f9e090750237ed4L,
        0x2f21da478096fa2bL },
      { 0xf3e19cb4eec863a0L,0xd18f77fd9527620aL,0x9505c81c407c1cf8L,
        0x9998db4e1b6ec284L } },
    /* 18 << 56 */
    { { 0x7e3389e5c247d44dL,0x125071413f4f3d80L,0xd4ba01104a78a6c7L,
        0x312874a0767720beL },
      { 0xded059a675944370L,0xd6123d903b2c0bddL,0xa56b717b51c108e3L,
        0x9bb7940e070623e9L } },
    /* 19 << 56 */
    { { 0x794e2d5984ac066cL,0xf5954a92e68c69a0L,0x28c524584fd99dccL,
        0x60e639fcb1012517L },
      { 0xc2e601257de79248L,0xe9ef6404f12fc6d7L,0x4c4f28082a3b5d32L,
        0x865ad32ec768eb8aL } },
    /* 20 << 56 */
    { { 0xac02331b13fb70b6L,0x037b44c195599b27L,0x1a860fc460bd082cL,
        0xa2e25745c980cd01L },
      { 0xee3387a81da0263eL,0x931bfb952d10f3d6L,0x5b687270a1f24a32L,
        0xf140e65dca494b86L } },
    /* 21 << 56 */
    { { 0x4f4ddf91b2f1ac7aL,0xf99eaabb760fee27L,0x57f4008a49c228e5L,
        0x090be4401cf713bbL },
      { 0xac91fbe45004f022L,0xd838c2c2569e1af6L,0xd6c7d20b0f1daaa5L,
        0xaa063ac11bbb02c0L } },
    /* 22 << 56 */
    { { 0x0938a42259558a78L,0x5343c6698435da2fL,0x96f67b18034410dcL,
        0x7cc1e42484510804L },
      { 0x86a1543f16dfbb7dL,0x921fa9425b5bd592L,0x9dcccb6eb33dd03cL,
        0x8581ddd9b843f51eL } },
    /* 23 << 56 */
    { { 0x54935fcb81d73c9eL,0x6d07e9790a5e97abL,0x4dc7b30acf3a6babL,
        0x147ab1f3170bee11L },
      { 0x0aaf8e3d9fafdee4L,0xfab3dbcb538a8b95L,0x405df4b36ef13871L,
        0xf1f4e9cb088d5a49L } },
    /* 24 << 56 */
    { { 0x9bcd24d366b33f1dL,0x3b97b8205ce445c0L,0xe2926549ba93ff61L,
        0xd9c341ce4dafe616L },
      { 0xfb30a76e16efb6f3L,0xdf24b8ca605b953cL,0x8bd52afec2fffb9fL,
        0xbbac5ff7e19d0b96L } },
    /* 25 << 56 */
    { { 0x43c01b87459afccdL,0x6bd45143b7432652L,0x8473453055b5d78eL,
        0x81088fdb1554ba7dL },
      { 0xada0a52c1e269375L,0xf9f037c42dc5ec10L,0xc066060794bfbc11L,
        0xc0a630bbc9c40d2fL } },
    /* 26 << 56 */
    { { 0x5efc797eab64c31eL,0xffdb1dab74507144L,0xf61242871ca6790cL,
        0xe9609d81e69bf1bfL },
      { 0xdb89859500d24fc9L,0x9c750333e51fb417L,0x51830a91fef7bbdeL,
        0x0ce67dc8945f585cL } },
    /* 27 << 56 */
    { { 0x9a730ed44763eb50L,0x24a0e221c1ab0d66L,0x643b6393648748f3L,
        0x1982daa16d3c6291L },
      { 0x6f00a9f78bbc5549L,0x7a1783e17f36384eL,0xe8346323de977f50L,
        0x91ab688db245502aL } },
    /* 28 << 56 */
    { { 0x331ab6b56d0bdd66L,0x0a6ef32e64b71229L,0x1028150efe7c352fL,
        0x27e04350ce7b39d3L },
      { 0x2a3c8acdc1070c82L,0xfb2034d380c9feefL,0x2d729621709f3729L,
        0x8df290bf62cb4549L } },
    /* 29 << 56 */
    { { 0x02f99f33fc2e4326L,0x3b30076d5eddf032L,0xbb21f8cf0c652fb5L,
        0x314fb49eed91cf7bL },
      { 0xa013eca52f700750L,0x2b9e3c23712a4575L,0xe5355557af30fbb0L,
        0x1ada35167c77e771L } },
    /* 30 << 56 */
    { { 0x45f6ecb27b135670L,0xe85d19df7cfc202eL,0x0f1b50c758d1be9fL,
        0x5ebf2c0aead2e344L },
      { 0x1531fe4eabc199c9L,0xc703259256bab0aeL,0x16ab2e486c1fec54L,
        0x0f87fda804280188L } },
    /* 31 << 56 */
    { { 0xdc9f46fc609e4a74L,0x2a44a143ba667f91L,0xbc3d8b95b4d83436L,
        0xa01e4bd0c7bd2958L },
      { 0x7b18293273483c90L,0xa79c6aa1a7c7b598L,0xbf3983c6eaaac07eL,
        0x8f18181e96e0d4e6L } },
    /* 32 << 56 */
    { { 0x8553d37c051af62bL,0xe9a998eb0bf94496L,0xe0844f9fb0d59aa1L,
        0x983fd558e6afb813L },
      { 0x9670c0ca65d69804L,0x732b22de6ea5ff2dL,0xd7640ba95fd8623bL,
        0x9f619163a6351782L } },
    /* 33 << 56 */
    { { 0x0bfc27eeacee5043L,0xae419e732eb10f02L,0x19c028d18943fb05L,
        0x71f01cf7ff13aa2aL },
      { 0x7790737e8887a132L,0x6751330966318410L,0x9819e8a37ddb795eL,
        0xfecb8ef5dad100b2L } },
    /* 34 << 56 */
    { { 0x59f74a223021926aL,0xb7c28a496f9b4c1cL,0xed1a733f912ad0abL,
        0x42a910af01a5659cL },
      { 0x3842c6e07bd68cabL,0x2b57fa3876d70ac8L,0x8a6707a83c53aaebL,
        0x62c1c51065b4db18L } },
    /* 35 << 56 */
    { { 0x8de2c1fbb2d09dc7L,0xc3dfed12266bd23bL,0x927d039bd5b27db6L,
        0x2fb2f0f1103243daL },
      { 0xf855a07b80be7399L,0xed9327ce1f9f27a8L,0xa0bd99c7729bdef7L,
        0x2b67125e28250d88L } },
    /* 36 << 56 */
    { { 0x784b26e88670ced7L,0xe3dfe41fc31bd3b4L,0x9e353a06bcc85cbcL,
        0x302e290960178a9dL },
      { 0x860abf11a6eac16eL,0x76447000aa2b3aacL,0x46ff9d19850afdabL,
        0x35bdd6a5fdb2d4c1L } },
    /* 37 << 56 */
    { { 0xe82594b07e5c9ce9L,0x0f379e5320af346eL,0x608b31e3bc65ad4aL,
        0x710c6b12267c4826L },
      { 0x51c966f971954cf1L,0xb1cec7930d0aa215L,0x1f15598986bd23a8L,
        0xae2ff99cf9452e86L } },
    /* 38 << 56 */
    { { 0xd8dd953c340ceaa2L,0x263552752e2e9333L,0x15d4e5f98586f06dL,
        0xd6bf94a8f7cab546L },
      { 0x33c59a0ab76a9af0L,0x52740ab3ba095af7L,0xc444de8a24389ca0L,
        0xcc6f9863706da0cbL } },
    /* 39 << 56 */
    { { 0xb5a741a76b2515cfL,0x71c416019585c749L,0x78350d4fe683de97L,
        0x31d6152463d0b5f5L },
      { 0x7a0cc5e1fbce090bL,0xaac927edfbcb2a5bL,0xe920de4920d84c35L,
        0x8c06a0b622b4de26L } },
    /* 40 << 56 */
    { { 0xd34dd58bafe7ddf3L,0x55851fedc1e6e55bL,0xd1395616960696e7L,
        0x940304b25f22705fL },
      { 0x6f43f861b0a2a860L,0xcf1212820e7cc981L,0x121862120ab64a96L,
        0x09215b9ab789383cL } },
    /* 41 << 56 */
    { { 0x311eb30537387c09L,0xc5832fcef03ee760L,0x30358f5832f7ea19L,
        0xe01d3c3491d53551L },
      { 0x1ca5ee41da48ea80L,0x34e71e8ecf4fa4c1L,0x312abd257af1e1c7L,
        0xe3afcdeb2153f4a5L } },
    /* 42 << 56 */
    { { 0x9d5c84d700235e9aL,0x0308d3f48c4c836fL,0xc0a66b0489332de5L,
        0x610dd39989e566efL },
      { 0xf8eea460d1ac1635L,0x84cbb3fb20a2c0dfL,0x40afb488e74a48c5L,
        0x29738198d326b150L } },
    /* 43 << 56 */
    { { 0x2a17747fa6d74081L,0x60ea4c0555a26214L,0x53514bb41f88c5feL,
        0xedd645677e83426cL },
      { 0xd5d6cbec96460b25L,0xa12fd0ce68dc115eL,0xc5bc3ed2697840eaL,
        0x969876a8a6331e31L } },
    /* 44 << 56 */
    { { 0x60c36217472ff580L,0xf42297054ad41393L,0x4bd99ef0a03b8b92L,
        0x501c7317c144f4f6L },
      { 0x159009b318464945L,0x6d5e594c74c5c6beL,0x2d587011321a3660L,
        0xd1e184b13898d022L } },
    /* 45 << 56 */
    { { 0x5ba047524c6a7e04L,0x47fa1e2b45550b65L,0x9419daf048c0a9a5L,
        0x663629537c243236L },
      { 0xcd0744b15cb12a88L,0x561b6f9a2b646188L,0x599415a566c2c0c0L,
        0xbe3f08590f83f09aL } },
    /* 46 << 56 */
    { { 0x9141c5beb92041b8L,0x01ae38c726477d0dL,0xca8b71f3d12c7a94L,
        0xfab5b31f765c70dbL },
      { 0x76ae7492487443e9L,0x8595a310990d1349L,0xf8dbeda87d460a37L,
        0x7f7ad0821e45a38fL } },
    /* 47 << 56 */
    { { 0xed1d4db61059705aL,0xa3dd492ae6b9c697L,0x4b92ee3a6eb38bd5L,
        0xbab2609d67cc0bb7L },
      { 0x7fc4fe896e70ee82L,0xeff2c56e13e6b7e3L,0x9b18959e34d26fcaL,
        0x2517ab66889d6b45L } },
    /* 48 << 56 */
    { { 0xf167b4e0bdefdd4fL,0x69958465f366e401L,0x5aa368aba73bbec0L,
        0x121487097b240c21L },
      { 0x378c323318969006L,0xcb4d73cee1fe53d1L,0x5f50a80e130c4361L,
        0xd67f59517ef5212bL } },
    /* 49 << 56 */
    { { 0xf145e21e9e70c72eL,0xb2e52e295566d2fbL,0x44eaba4a032397f5L,
        0x5e56937b7e31a7deL },
      { 0x68dcf517456c61e1L,0xbc2e954aa8b0a388L,0xe3552fa760a8b755L,
        0x03442dae73ad0cdeL } },
    /* 50 << 56 */
    { { 0x37ffe747ceb26210L,0x983545e8787baef9L,0x8b8c853586a3de31L,
        0xc621dbcbfacd46dbL },
      { 0x82e442e959266fbbL,0xa3514c37339d471cL,0x3a11b77162cdad96L,
        0xf0cb3b3cecf9bdf0L } },
    /* 51 << 56 */
    { { 0x3fcbdbce478e2135L,0x7547b5cfbda35342L,0xa97e81f18a677af6L,
        0xc8c2bf8328817987L },
      { 0xdf07eaaf45580985L,0xc68d1f05c93b45cbL,0x106aa2fec77b4cacL,
        0x4c1d8afc04a7ae86L } },
    /* 52 << 56 */
    { { 0xdb41c3fd9eb45ab2L,0x5b234b5bd4b22e74L,0xda253decf215958aL,
        0x67e0606ea04edfa0L },
      { 0xabbbf070ef751b11L,0xf352f175f6f06dceL,0xdfc4b6af6839f6b4L,
        0x53ddf9a89959848eL } },
    /* 53 << 56 */
    { { 0xda49c379c21520b0L,0x90864ff0dbd5d1b6L,0x2f055d235f49c7f7L,
        0xe51e4e6aa796b2d8L },
      { 0xc361a67f5c9dc340L,0x5ad53c37bca7c620L,0xda1d658832c756d0L,
        0xad60d9118bb67e13L } },
    /* 54 << 56 */
    { { 0xd6c47bdf0eeec8c6L,0x4a27fec1078a1821L,0x081f7415c3099524L,
        0x8effdf0b82cd8060L },
      { 0xdb70ec1c65842df8L,0x8821b358d319a901L,0x72ee56eede42b529L,
        0x5bb39592236e4286L } },
    /* 55 << 56 */
    { { 0xd1183316fd6f7140L,0xf9fadb5bbd8e81f7L,0x701d5e0c5a02d962L,
        0xfdee4dbf1b601324L },
      { 0xbed1740735d7620eL,0x04e3c2c3f48c0012L,0x9ee29da73455449aL,
        0x562cdef491a836c4L } },
    /* 56 << 56 */
    { { 0x8f682a5f47701097L,0x617125d8ff88d0c2L,0x948fda2457bb86ddL,
        0x348abb8f289f7286L },
      { 0xeb10eab599d94bbdL,0xd51ba28e4684d160L,0xabe0e51c30c8f41aL,
        0x66588b4513254f4aL } },
    /* 57 << 56 */
    { { 0x147ebf01fad097a5L,0x49883ea8610e815dL,0xe44d60ba8a11de56L,
        0xa970de6e827a7a6dL },
      { 0x2be414245e17fc19L,0xd833c65701214057L,0x1375813b363e723fL,
        0x6820bb88e6a52e9bL } },
    /* 58 << 56 */
    { { 0x7e7f6970d875d56aL,0xd6a0a9ac51fbf6bfL,0x54ba8790a3083c12L,
        0xebaeb23d6ae7eb64L },
      { 0xa8685c3ab99a907aL,0xf1e74550026bf40bL,0x7b73a027c802cd9eL,
        0x9a8a927c4fef4635L } },
    /* 59 << 56 */
    { { 0xe1b6f60c08191224L,0xc4126ebbde4ec091L,0xe1dff4dc4ae38d84L,
        0xde3f57db4f2ef985L },
      { 0x34964337d446a1ddL,0x7bf217a0859e77f6L,0x8ff105278e1d13f5L,
        0xa304ef0374eeae27L } },
    /* 60 << 56 */
    { { 0xfc6f5e47d19dfa5aL,0xdb007de37fad982bL,0x28205ad1613715f5L,
        0x251e67297889529eL },
      { 0x727051841ae98e78L,0xf818537d271cac32L,0xc8a15b7eb7f410f5L,
        0xc474356f81f62393L } },
    /* 61 << 56 */
    { { 0x92dbdc5ac242316bL,0xabe060acdbf4aff5L,0x6e8c38fe909a8ec6L,
        0x43e514e56116cb94L },
      { 0x2078fa3807d784f9L,0x1161a880f4b5b357L,0x5283ce7913adea3dL,
        0x0756c3e6cc6a910bL } },
    /* 62 << 56 */
    { { 0x60bcfe01aaa79697L,0x04a73b2956391db1L,0xdd8dad47189b45a0L,
        0xbfac0dd048d5b8d9L },
      { 0x34ab3af57d3d2ec2L,0x6fa2fc2d207bd3afL,0x9ff4009266550dedL,
        0x719b3e871fd5b913L } },
    /* 63 << 56 */
    { { 0xa573a4966d17fbc7L,0x0cd1a70a73d2b24eL,0x34e2c5cab2676937L,
        0xe7050b06bf669f21L },
      { 0xfbe948b61ede9046L,0xa053005197662659L,0x58cbd4edf10124c5L,
        0xde2646e4dd6c06c8L } },
    /* 64 << 56 */
    { { 0x332f81088cad38c0L,0x471b7e906bd68ae2L,0x56ac3fb20d8e27a3L,
        0xb54660db136b4b0dL },
      { 0x123a1e11a6fd8de4L,0x44dbffeaa37799efL,0x4540b977ce6ac17cL,
        0x495173a8af60acefL } },
    /* 0 << 63 */
    { { 0x00, 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x00, 0x00 } },
    /* 1 << 63 */
    { { 0x9ebb284d391c2a82L,0xbcdd4863158308e8L,0x006f16ec83f1edcaL,
        0xa13e2c37695dc6c8L },
      { 0x2ab756f04a057a87L,0xa8765500a6b48f98L,0x4252face68651c44L,
        0xa52b540be1765e02L } },
    /* 2 << 63 */
    { { 0x4f922fc516a0d2bbL,0x0d5cc16c1a623499L,0x9241cf3a57c62c8bL,
        0x2f5e6961fd1b667fL },
      { 0x5c15c70bf5a01797L,0x3d20b44d60956192L,0x04911b37071fdb52L,
        0xf648f9168d6f0f7bL } },
    /* 3 << 63 */
    { { 0x6dc1acafe60b7cf7L,0x25860a5084a9d869L,0x56fc6f09e7ba8ac4L,
        0x828c5bd06148d29eL },
      { 0xac6b435edc55ae5fL,0xa527f56cc0117411L,0x94d5045efd24342cL,
        0x2c4c0a3570b67c0dL } },
    /* 4 << 63 */
    { { 0x027cc8b8fac61d9aL,0x7d25e062e3c6fe8aL,0xe08805bfe5bff503L,
        0x13271e6c6ff632f7L },
      { 0x55dca6c0232f76a5L,0x8957c32d701ef426L,0xee728bcba10a5178L,
        0x5ea60411b62c5173L } },
    /* 5 << 63 */
    { { 0xfc4e964ed0b8892bL,0x9ea176839301bb74L,0x6265c5aefcc48626L,
        0xe60cf82ebb3e9102L },
      { 0x57adf797d4df5531L,0x235b59a18deeefe2L,0x60adcf583f306eb1L,
        0x105c27533d09492dL } },
    /* 6 << 63 */
    { { 0x4090914bb5def996L,0x1cb69c83233dd1e7L,0xc1e9c1d39b3d5e76L,
        0x1f3338edfccf6012L },
      { 0xb1e95d0d2f5378a8L,0xacf4c2c72f00cd21L,0x6e984240eb5fe290L,
        0xd66c038d248088aeL } },
    /* 7 << 63 */
    { { 0x804d264af94d70cfL,0xbdb802ef7314bf7eL,0x8fb54de24333ed02L,
        0x740461e0285635d9L },
      { 0x4113b2c8365e9383L,0xea762c833fdef652L,0x4eec6e2e47b956c1L,
        0xa3d814be65620fa4L } },
    /* 8 << 63 */
    { { 0x9ad5462bb4d8bc50L,0x181c0b16a9195770L,0xebd4fe1c78412a68L,
        0xae0341bcc0dff48cL },
      { 0xb6bc45cf7003e866L,0xf11a6dea8a24a41bL,0x5407151ad04c24c2L,
        0x62c9d27dda5b7b68L } },
    /* 9 << 63 */
    { { 0x2e96423588cceff6L,0x8594c54f8b07ed69L,0x1578e73cc84d0d0dL,
        0x7b4e1055ff532868L },
      { 0xa348c0d5b5ec995aL,0xbf4b9d5514289a54L,0x9ba155a658fbd777L,
        0x186ed7a81a84491dL } },
    /* 10 << 63 */
    { { 0xd4992b30614c0900L,0xda98d121bd00c24bL,0x7f534dc87ec4bfa1L,
        0x4a5ff67437dc34bcL },
      { 0x68c196b81d7ea1d7L,0x38cf289380a6d208L,0xfd56cd09e3cbbd6eL,
        0xec72e27e4205a5b6L } },
    /* 11 << 63 */
    { { 0x15ea68f5a44f77f7L,0x7aa5f9fdb43c52bcL,0x86ff676f94f0e609L,
        0xa4cde9632e2d432bL },
      { 0x8cafa0c0eee470afL,0x84137d0e8a3f5ec8L,0xebb40411faa31231L,
        0xa239c13f6f7f7ccfL } },
    /* 12 << 63 */
    { { 0x32865719a8afd30bL,0x867983288a826dceL,0xdf04e891c4a8fbe0L,
        0xbb6b6e1bebf56ad3L },
      { 0x0a695b11471f1ff0L,0xd76c3389be15baf0L,0x018edb95be96c43eL,
        0xf2beaaf490794158L } },
    /* 13 << 63 */
    { { 0x152db09ec3076a27L,0x5e82908ee416545dL,0xa2c41272356d6f2eL,
        0xdc9c964231fd74e1L },
      { 0x66ceb88d519bf615L,0xe29ecd7605a2274eL,0x3a0473c4bf5e2fa0L,
        0x6b6eb67164284e67L } },
    /* 14 << 63 */
    { { 0xe8b97932b88756ddL,0xed4e8652f17e3e61L,0xc2dd14993ee1c4a4L,
        0xc0aaee17597f8c0eL },
      { 0x15c4edb96c168af3L,0x6563c7bfb39ae875L,0xadfadb6f20adb436L,
        0xad55e8c99a042ac0L } },
    /* 15 << 63 */
    { { 0x975a1ed8b76da1f5L,0x10dfa466a58acb94L,0x8dd7f7e3ac060282L,
        0x6813e66a572a051eL },
      { 0xb4ccae1e350cb901L,0xb653d65650cb7822L,0x42484710dfab3b87L,
        0xcd7ee5379b670fd0L } },
    /* 16 << 63 */
    { { 0x0a50b12e523b8bf6L,0x8009eb5b8f910c1bL,0xf535af824a167588L,
        0x0f835f9cfb2a2abdL },
      { 0xf59b29312afceb62L,0xc797df2a169d383fL,0xeb3f5fb066ac02b0L,
        0x029d4c6fdaa2d0caL } },
    /* 17 << 63 */
    { { 0xd4059bc1afab4bc5L,0x833f5c6f56783247L,0xb53466308d2d3605L,
        0x83387891d34d8433L },
      { 0xd973b30fadd9419aL,0xbcca1099afe3fce8L,0x081783150809aac6L,
        0x01b7f21a540f0f11L } },
    /* 18 << 63 */
    { { 0x65c29219909523c8L,0xa62f648fa3a1c741L,0x88598d4f60c9e55aL,
        0xbce9141b0e4f347aL },
      { 0x9af97d8435f9b988L,0x0210da62320475b6L,0x3c076e229191476cL,
        0x7520dbd944fc7834L } },
    /* 19 << 63 */
    { { 0x6a6b2cfec1ab1bbdL,0xef8a65bedc650938L,0x72855540805d7bc4L,
        0xda389396ed11fdfdL },
      { 0xa9d5bd3674660876L,0x11d67c54b45dff35L,0x6af7d148a4f5da94L,
        0xbb8d4c3fc0bbeb31L } },
    /* 20 << 63 */
    { { 0x87a7ebd1e0a1b12aL,0x1e4ef88d770ba95fL,0x8c33345cdc2ae9cbL,
        0xcecf127601cc8403L },
      { 0x687c012e1b39b80fL,0xfd90d0ad35c33ba4L,0xa3ef5a675c9661c2L,
        0x368fc88ee017429eL } },
    /* 21 << 63 */
    { { 0xd30c6761196a2fa2L,0x931b9817bd5b312eL,0xba01000c72f54a31L,
        0xa203d2c866eaa541L },
      { 0xf2abdee098939db3L,0xe37d6c2c3e606c02L,0xf2921574521ff643L,
        0x2781b3c4d7e2fca3L } },
    /* 22 << 63 */
    { { 0x664300b07850ec06L,0xac5a38b97d3a10cfL,0x9233188de34ab39dL,
        0xe77057e45072cbb9L },
      { 0xbcf0c042b59e78dfL,0x4cfc91e81d97de52L,0x4661a26c3ee0ca4aL,
        0x5620a4c1fb8507bcL } },
    /* 23 << 63 */
    { { 0x4b44d4aa049f842cL,0xceabc5d51540e82bL,0x306710fd15c6f156L,
        0xbe5ae52b63db1d72L },
      { 0x06f1e7e6334957f1L,0x57e388f031144a70L,0xfb69bb2fdf96447bL,
        0x0f78ebd373e38a12L } },
    /* 24 << 63 */
    { { 0xb82226052b7ce542L,0xe6d4ce997472bde1L,0x53e16ebe09d2f4daL,
        0x180ff42e53b92b2eL },
      { 0xc59bcc022c34a1c6L,0x3803d6f9422c46c2L,0x18aff74f5c14a8a2L,
        0x55aebf8010a08b28L } },
    /* 25 << 63 */
    { { 0x66097d587135593fL,0x32e6eff72be570cdL,0x584e6a102a8c860dL,
        0xcd185890a2eb4163L },
      { 0x7ceae99d6d97e134L,0xd42c6b70dd8447ceL,0x59ddbb4ab8c50273L,
        0x03c612df3cf34e1eL } },
    /* 26 << 63 */
    { { 0x84b9ca1504b6c5a0L,0x35216f3918f0e3a3L,0x3ec2d2bcbd986c00L,
        0x8bf546d9d19228feL },
      { 0xd1c655a44cd623c3L,0x366ce718502b8e5aL,0x2cfc84b4eea0bfe7L,
        0xe01d5ceecf443e8eL } },
    /* 27 << 63 */
    { { 0x8ec045d9036520f8L,0xdfb3c3d192d40e98L,0x0bac4ccecc559a04L,
        0x35eccae5240ea6b1L },
      { 0x180b32dbf8a5a0acL,0x547972a5eb699700L,0xa3765801ca26bca0L,
        0x57e09d0ea647f25aL } },
    /* 28 << 63 */
    { { 0xb956970e2fdd23ccL,0xb80288bc5682e971L,0xe6e6d91e9ae86ebcL,
        0x0564c83f8c9f1939L },
      { 0x551932a239560368L,0xe893752b049c28e2L,0x0b03cee5a6a158c3L,
        0xe12d656b04964263L } },
    /* 29 << 63 */
    { { 0x4b47554e63e3bc1dL,0xc719b6a245044ff7L,0x4f24d30ae48daa07L,
        0xa3f37556c8c1edc3L },
      { 0x9a47bf760700d360L,0xbb1a1824822ae4e2L,0x22e275a389f1fb4cL,
        0x72b1aa239968c5f5L } },
    /* 30 << 63 */
    { { 0xa75feacabe063f64L,0x9b392f43bce47a09L,0xd42415091ad07acaL,
        0x4b0c591b8d26cd0fL },
      { 0x2d42ddfd92f1169aL,0x63aeb1ac4cbf2392L,0x1de9e8770691a2afL,
        0xebe79af7d98021daL } },
    /* 31 << 63 */
    { { 0xcfdf2a4e40e50acfL,0xf0a98ad7af01d665L,0xefb640bf1831be1fL,
        0x6fe8bd2f80e9ada0L },
      { 0x94c103a16cafbc91L,0x170f87598308e08cL,0x5de2d2ab9780ff4fL,
        0x666466bc45b201f2L } },
    /* 32 << 63 */
    { { 0x58af2010f5b343bcL,0x0f2e400af2f142feL,0x3483bfdea85f4bdfL,
        0xf0b1d09303bfeaa9L },
      { 0x2ea01b95c7081603L,0xe943e4c93dba1097L,0x47be92adb438f3a6L,
        0x00bb7742e5bf6636L } },
    /* 33 << 63 */
    { { 0x136b7083824297b4L,0x9d0e55805584455fL,0xab48cedcf1c7d69eL,
        0x53a9e4812a256e76L },
      { 0x0402b0e065eb2413L,0xdadbbb848fc407a7L,0xa65cd5a48d7f5492L,
        0x21d4429374bae294L } },
    /* 34 << 63 */
    { { 0x66917ce63b5f1cc4L,0x37ae52eace872e62L,0xbb087b722905f244L,
        0x120770861e6af74fL },
      { 0x4b644e491058edeaL,0x827510e3b638ca1dL,0x8cf2b7046038591cL,
        0xffc8b47afe635063L } },
    /* 35 << 63 */
    { { 0x3ae220e61b4d5e63L,0xbd8647429d961b4bL,0x610c107e9bd16bedL,
        0x4270352a1127147bL },
      { 0x7d17ffe664cfc50eL,0x50dee01a1e36cb42L,0x068a762235dc5f9aL,
        0x9a08d536df53f62cL } },
    /* 36 << 63 */
    { { 0x4ed714576be5f7deL,0xd93006f8c2263c9eL,0xe073694ccacacb36L,
        0x2ff7a5b43ae118abL },
      { 0x3cce53f1cd871236L,0xf156a39dc2aa6d52L,0x9cc5f271b198d76dL,
        0xbc615b6f81383d39L } },
    /* 37 << 63 */
    { { 0xa54538e8de3eee6bL,0x58c77538ab910d91L,0x31e5bdbc58d278bdL,
        0x3cde4adfb963acaeL },
      { 0xb1881fd25302169cL,0x8ca60fa0a989ed8bL,0xa1999458ff96a0eeL,
        0xc1141f03ac6c283dL } },
    /* 38 << 63 */
    { { 0x7677408d6dfafed3L,0x33a0165339661588L,0x3c9c15ec0b726fa0L,
        0x090cfd936c9b56daL },
      { 0xe34f4baea3c40af5L,0x3469eadbd21129f1L,0xcc51674a1e207ce8L,
        0x1e293b24c83b1ef9L } },
    /* 39 << 63 */
    { { 0x17173d131e6c0bb4L,0x1900469590776d35L,0xe7980e346de6f922L,
        0x873554cbf4dd9a22L },
      { 0x0316c627cbf18a51L,0x4d93651b3032c081L,0x207f27713946834dL,
        0x2c08d7b430cdbf80L } },
    /* 40 << 63 */
    { { 0x137a4fb486df2a61L,0xa1ed9c07ecf7b4a2L,0xb2e460e27bd042ffL,
        0xb7f5e2fa5f62f5ecL },
      { 0x7aa6ec6bcc2423b7L,0x75ce0a7fba63eea7L,0x67a45fb1f250a6e1L,
        0x93bc919ce53cdc9fL } },
    /* 41 << 63 */
    { { 0x9271f56f871942dfL,0x2372ff6f7859ad66L,0x5f4c2b9633cb1a78L,
        0xe3e291015838aa83L },
      { 0xa7ed1611e4e8110cL,0x2a2d70d5330198ceL,0xbdf132e86720efe0L,
        0xe61a896266a471bfL } },
    /* 42 << 63 */
    { { 0x796d3a85825808bdL,0x51dc3cb73fd6e902L,0x643c768a916219d1L,
        0x36cd7685a2ad7d32L },
      { 0xe3db9d05b22922a4L,0x6494c87edba29660L,0xf0ac91dfbcd2ebc7L,
        0x4deb57a045107f8dL } },
    /* 43 << 63 */
    { { 0x42271f59c3d12a73L,0x5f71687ca5c2c51dL,0xcb1f50c605797bcbL,
        0x29ed0ed9d6d34eb0L },
      { 0xe5fe5b474683c2ebL,0x4956eeb597447c46L,0x5b163a4371207167L,
        0x93fa2fed0248c5efL } },
    /* 44 << 63 */
    { { 0x67930af231f63950L,0xa77797c114caa2c9L,0x526e80ee27ac7e62L,
        0xe1e6e62658b28aecL },
      { 0x636178b0b3c9fef0L,0xaf7752e06d5f90beL,0x94ecaf18eece51cfL,
        0x2864d0edca806e1fL } },
    /* 45 << 63 */
    { { 0x6de2e38397c69134L,0x5a42c316eb291293L,0xc77792196a60bae0L,
        0xa24de3466b7599d1L },
      { 0x49d374aab75d4941L,0x989005862d501ff0L,0x9f16d40eeb7974cfL,
        0x1033860bcdd8c115L } },
    /* 46 << 63 */
    { { 0xb6c69ac82094cec3L,0x9976fb88403b770cL,0x1dea026c4859590dL,
        0xb6acbb468562d1fdL },
      { 0x7cd6c46144569d85L,0xc3190a3697f0891dL,0xc6f5319548d5a17dL,
        0x7d919966d749abc8L } },
    /* 47 << 63 */
    { { 0x65104837dd1c8a20L,0x7e5410c82f683419L,0x958c3ca8be94022eL,
        0x605c31976145dac2L },
      { 0x3fc0750101683d54L,0x1d7127c5595b1234L,0x10b8f87c9481277fL,
        0x677db2a8e65a1adbL } },
    /* 48 << 63 */
    { { 0xec2fccaaddce3345L,0x2a6811b7012a4350L,0x96760ff1ac598bdcL,
        0x054d652ad1bf4128L },
      { 0x0a1151d492a21005L,0xad7f397133110fdfL,0x8c95928c1960100fL,
        0x6c91c8257bf03362L } },
    /* 49 << 63 */
    { { 0xc8c8b2a2ce309f06L,0xfdb27b59ca27204bL,0xd223eaa50848e32eL,
        0xb93e4b2ee7bfaf1eL },
      { 0xc5308ae644aa3dedL,0x317a666ac015d573L,0xc888ce231a979707L,
        0xf141c1e60d5c4958L } },
    /* 50 << 63 */
    { { 0xb53b7de561906373L,0x858dbadeeb999595L,0x8cbb47b2a59e5c36L,
        0x660318b3dcf4e842L },
      { 0xbd161ccd12ba4b7aL,0xf399daabf8c8282aL,0x1587633aeeb2130dL,
        0xa465311ada38dd7dL } },
    /* 51 << 63 */
    { { 0x5f75eec864d3779bL,0x3c5d0476ad64c171L,0x874103712a914428L,
        0x8096a89190e2fc29L },
      { 0xd3d2ae9d23b3ebc2L,0x90bdd6dba580cfd6L,0x52dbb7f3c5b01f6cL,
        0xe68eded4e102a2dcL } },
    /* 52 << 63 */
    { { 0x17785b7799eb6df0L,0x26c3cc517386b779L,0x345ed9886417a48eL,
        0xe990b4e407d6ef31L },
      { 0x0f456b7e2586abbaL,0x239ca6a559c96e9aL,0xe327459ce2eb4206L,
        0x3a4c3313a002b90aL } },
    /* 53 << 63 */
    { { 0x2a114806f6a3f6fbL,0xad5cad2f85c251ddL,0x92c1f613f5a784d3L,
        0xec7bfacf349766d5L },
      { 0x04b3cd333e23cb3bL,0x3979fe84c5a64b2dL,0x192e27207e589106L,
        0xa60c43d1a15b527fL } },
    /* 54 << 63 */
    { { 0x2dae9082be7cf3a6L,0xcc86ba92bc967274L,0xf28a2ce8aea0a8a9L,
        0x404ca6d96ee988b3L },
      { 0xfd7e9c5d005921b8L,0xf56297f144e79bf9L,0xa163b4600d75ddc2L,
        0x30b23616a1f2be87L } },
    /* 55 << 63 */
    { { 0x4b070d21bfe50e2bL,0x7ef8cfd0e1bfede1L,0xadba00112aac4ae0L,
        0x2a3e7d01b9ebd033L },
      { 0x995277ece38d9d1cL,0xb500249e9c5d2de3L,0x8912b820f13ca8c9L,
        0xc8798114877793afL } },
    /* 56 << 63 */
    { { 0x19e6125dec3f1decL,0x07b1f040911178daL,0xd93ededa904a6738L,
        0x55187a5a0bebedcdL },
      { 0xf7d04722eb329d41L,0xf449099ef170b391L,0xfd317a69ca99f828L,
        0x50c3db2b34a4976dL } },
    /* 57 << 63 */
    { { 0xe9ba77843757b392L,0x326caefdaa3ca05aL,0x78e5293bf1e593d4L,
        0x7842a9370d98fd13L },
      { 0xe694bf965f96b10dL,0x373a9df606a8cd05L,0x997d1e51e8f0c7fcL,
        0x1d01979063fd972eL } },
    /* 58 << 63 */
    { { 0x0064d8585499fb32L,0x7b67bad977a8aeb7L,0x1d3eb9772d08eec5L,
        0x5fc047a6cbabae1dL },
      { 0x0577d159e54a64bbL,0x8862201bc43497e4L,0xad6b4e282ce0608dL,
        0x8b687b7d0b167aacL } },
    /* 59 << 63 */
    { { 0x6ed4d3678b2ecfa9L,0x24dfe62da90c3c38L,0xa1862e103fe5c42bL,
        0x1ca73dcad5732a9fL },
      { 0x35f038b776bb87adL,0x674976abf242b81fL,0x4f2bde7eb0fd90cdL,
        0x6efc172ea7fdf092L } },
    /* 60 << 63 */
    { { 0x3806b69b92222f1fL,0x5a2459ca6cf7ae70L,0x6789f69ca85217eeL,
        0x5f232b5ee3dc85acL },
      { 0x660e3ec548e9e516L,0x124b4e473197eb31L,0x10a0cb13aafcca23L,
        0x7bd63ba48213224fL } },
    /* 61 << 63 */
    { { 0xaffad7cc290a7f4fL,0x6b409c9e0286b461L,0x58ab809fffa407afL,
        0xc3122eedc68ac073L },
      { 0x17bf9e504ef24d7eL,0x5d9297943e2a5811L,0x519bc86702902e01L,
        0x76bba5da39c8a851L } },
    /* 62 << 63 */
    { { 0xe9f9669cda94951eL,0x4b6af58d66b8d418L,0xfa32107417d426a4L,
        0xc78e66a99dde6027L },
      { 0x0516c0834a53b964L,0xfc659d38ff602330L,0x0ab55e5c58c5c897L,
        0x985099b2838bc5dfL } },
    /* 63 << 63 */
    { { 0x061d9efcc52fc238L,0x712b27286ac1da3fL,0xfb6581499283fe08L,
        0x4954ac94b8aaa2f7L },
      { 0x85c0ada47fb2e74fL,0xee8ba98eb89926b0L,0xe4f9d37d23d1af5bL,
        0x14ccdbf9ba9b015eL } },
    /* 64 << 63 */
    { { 0xb674481b7bfe7178L,0x4e1debae65405868L,0x061b2821c48c867dL,
        0x69c15b35513b30eaL },
      { 0x3b4a166636871088L,0xe5e29f5d1220b1ffL,0x4b82bb35233d9f4dL,
        0x4e07633318cdc675L } },
    /* 0 << 70 */
    { { 0x00, 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x00, 0x00 } },
    /* 1 << 70 */
    { { 0x0d53f5c7a3e6fcedL,0xe8cbbdd5f45fbdebL,0xf85c01df13339a70L,
        0x0ff71880142ceb81L },
      { 0x4c4e8774bd70437aL,0x5fb32891ba0bda6aL,0x1cdbebd2f18bd26eL,
        0x2f9526f103a9d522L } },
    /* 2 << 70 */
    { { 0x40ce305192c4d684L,0x8b04d7257612efcdL,0xb9dcda366f9cae20L,
        0x0edc4d24f058856cL },
      { 0x64f2e6bf85427900L,0x3de81295dc09dfeaL,0xd41b4487379bf26cL,
        0x50b62c6d6df135a9L } },
    /* 3 << 70 */
    { { 0xd4f8e3b4c72dfe67L,0xc416b0f690e19fdfL,0x18b9098d4c13bd35L,
        0xac11118a15b8cb9eL },
      { 0xf598a318f0062841L,0xbfe0602f89f356f4L,0x7ae3637e30177a0cL,
        0x3409774761136537L } },
    /* 4 << 70 */
    { { 0x0db2fb5ed005832aL,0x5f5efd3b91042e4fL,0x8c4ffdc6ed70f8caL,
        0xe4645d0bb52da9ccL },
      { 0x9596f58bc9001d1fL,0x52c8f0bc4e117205L,0xfd4aa0d2e398a084L,
        0x815bfe3a104f49deL } },
    /* 5 << 70 */
    { { 0x97e5443f23885e5fL,0xf72f8f99e8433aabL,0xbd00b154e4d4e604L,
        0xd0b35e6ae5e173ffL },
      { 0x57b2a0489164722dL,0x3e3c665b88761ec8L,0x6bdd13973da83832L,
        0x3c8b1a1e73dafe3bL } },
    /* 6 << 70 */
    { { 0x4497ace654317cacL,0xbe600ab9521771b3L,0xb42e409eb0dfe8b8L,
        0x386a67d73942310fL },
      { 0x25548d8d4431cc28L,0xa7cff142985dc524L,0x4d60f5a193c4be32L,
        0x83ebd5c8d071c6e1L } },
    /* 7 << 70 */
    { { 0xba3a80a7b1fd2b0bL,0x9b3ad3965bec33e8L,0xb3868d6179743fb3L,
        0xcfd169fcfdb462faL },
      { 0xd3b499d79ce0a6afL,0x55dc1cf1e42d3ff8L,0x04fb9e6cc6c3e1b2L,
        0x47e6961d6f69a474L } },
    /* 8 << 70 */
    { { 0x54eb3acce548b37bL,0xb38e754284d40549L,0x8c3daa517b341b4fL,
        0x2f6928ec690bf7faL },
      { 0x0496b32386ce6c41L,0x01be1c5510adadcdL,0xc04e67e74bb5faf9L,
        0x3cbaf678e15c9985L } },
    /* 9 << 70 */
    { { 0x8cd1214550ca4247L,0xba1aa47ae7dd30aaL,0x2f81ddf1e58fee24L,
        0x03452936eec9b0e8L },
      { 0x8bdc3b81243aea96L,0x9a2919af15c3d0e5L,0x9ea640ec10948361L,
        0x5ac86d5b6e0bcccfL } },
    /* 10 << 70 */
    { { 0xf892d918c36cf440L,0xaed3e837c939719cL,0xb07b08d2c0218b64L,
        0x6f1bcbbace9790ddL },
      { 0x4a84d6ed60919b8eL,0xd89007918ac1f9ebL,0xf84941aa0dd5daefL,
        0xb22fe40a67fd62c5L } },
    /* 11 << 70 */
    { { 0x97e15ba2157f2db3L,0xbda2fc8f8e28ca9cL,0x5d050da437b9f454L,
        0x3d57eb572379d72eL },
      { 0xe9b5eba2fb5ee997L,0x01648ca2e11538caL,0x32bb76f6f6327974L,
        0x338f14b8ff3f4bb7L } },
    /* 12 << 70 */
    { { 0x524d226ad7ab9a2dL,0x9c00090d7dfae958L,0x0ba5f5398751d8c2L,
        0x8afcbcdd3ab8262dL },
      { 0x57392729e99d043bL,0xef51263baebc943aL,0x9feace9320862935L,
        0x639efc03b06c817bL } },
    /* 13 << 70 */
    { { 0x1fe054b366b4be7aL,0x3f25a9de84a37a1eL,0xf39ef1ad78d75cd9L,
        0xd7b58f495062c1b5L },
      { 0x6f74f9a9ff563436L,0xf718ff29e8af51e7L,0x5234d31315e97fecL,
        0xb6a8e2b1292f1c0aL } },
    /* 14 << 70 */
    { { 0xa7f53aa8327720c1L,0x956ca322ba092cc8L,0x8f03d64a28746c4dL,
        0x51fe178266d0d392L },
      { 0xd19b34db3c832c80L,0x60dccc5c6da2e3b4L,0x245dd62e0a104cccL,
        0xa7ab1de1620b21fdL } },
    /* 15 << 70 */
    { { 0xb293ae0b3893d123L,0xf7b75783b15ee71cL,0x5aa3c61442a9468bL,
        0xd686123cdb15d744L },
      { 0x8c616891a7ab4116L,0x6fcd72c8a4e6a459L,0xac21911077e5fad7L,
        0xfb6a20e7704fa46bL } },
    /* 16 << 70 */
    { { 0xe839be7d341d81dcL,0xcddb688932148379L,0xda6211a1f7026eadL,
        0xf3b2575ff4d1cc5eL },
      { 0x40cfc8f6a7a73ae6L,0x83879a5e61d5b483L,0xc5acb1ed41a50ebcL,
        0x59a60cc83c07d8faL } },
    /* 17 << 70 */
    { { 0x1b73bdceb1876262L,0x2b0d79f012af4ee9L,0x8bcf3b0bd46e1d07L,
        0x17d6af9de45d152fL },
      { 0x735204616d736451L,0x43cbbd9756b0bf5aL,0xb0833a5bd5999b9dL,
        0x702614f0eb72e398L } },
    /* 18 << 70 */
    { { 0x0aadf01a59c3e9f8L,0x40200e77ce6b3d16L,0xda22bdd3deddafadL,
        0x76dedaf4310d72e1L },
      { 0x49ef807c4bc2e88fL,0x6ba81291146dd5a5L,0xa1a4077a7d8d59e9L,
        0x87b6a2e7802db349L } },
    /* 19 << 70 */
    { { 0xd56799971b4e598eL,0xf499ef1f06fe4b1dL,0x3978d3aefcb267c5L,
        0xb582b557235786d0L },
      { 0x32b3b2ca1715cb07L,0x4c3de6a28480241dL,0x63b5ffedcb571ecdL,
        0xeaf53900ed2fe9a9L } },
    /* 20 << 70 */
    { { 0xdec98d4ac3b81990L,0x1cb837229e0cc8feL,0xfe0b0491d2b427b9L,
        0x0f2386ace983a66cL },
      { 0x930c4d1eb3291213L,0xa2f82b2e59a62ae4L,0x77233853f93e89e3L,
        0x7f8063ac11777c7fL } },
    /* 21 << 70 */
    { { 0xff0eb56759ad2877L,0x6f4546429865c754L,0xe6fe701a236e9a84L,
        0xc586ef1606e40fc3L },
      { 0x3f62b6e024bafad9L,0xc8b42bd264da906aL,0xc98e1eb4da3276a0L,
        0x30d0e5fc06cbf852L } },
    /* 22 << 70 */
    { { 0x1b6b2ae1e8b4dfd4L,0xd754d5c78301cbacL,0x66097629112a39acL,
        0xf86b599993ba4ab9L },
      { 0x26c9dea799f9d581L,0x0473b1a8c2fafeaaL,0x1469af553b2505a5L,
        0x227d16d7d6a43323L } },
    /* 23 << 70 */
    { { 0x3316f73cad3d97f9L,0x52bf3bb51f137455L,0x953eafeb09954e7cL,
        0xa721dfeddd732411L },
      { 0xb4929821141d4579L,0x3411321caa3bd435L,0xafb355aa17fa6015L,
        0xb4e7ef4a18e42f0eL } },
    /* 24 << 70 */
    { { 0x604ac97c59371000L,0xe1c48c707f759c18L,0x3f62ecc5a5db6b65L,
        0x0a78b17338a21495L },
      { 0x6be1819dbcc8ad94L,0x70dc04f6d89c3400L,0x462557b4a6b4840aL,
        0x544c6ade60bd21c0L } },
    /* 25 << 70 */
    { { 0x6a00f24e907a544bL,0xa7520dcb313da210L,0xfe939b7511e4994bL,
        0x918b6ba6bc275d70L },
      { 0xd3e5e0fc644be892L,0x707a9816fdaf6c42L,0x60145567f15c13feL,
        0x4818ebaae130a54aL } },
    /* 26 << 70 */
    { { 0x28aad3ad58d2f767L,0xdc5267fdd7e7c773L,0x4919cc88c3afcc98L,
        0xaa2e6ab02db8cd4bL },
      { 0xd46fec04d0c63eaaL,0xa1cb92c519ffa832L,0x678dd178e43a631fL,
        0xfb5ae1cd3dc788b3L } },
    /* 27 << 70 */
    { { 0x68b4fb906e77de04L,0x7992bcf0f06dbb97L,0x896e6a13c417c01dL,
        0x8d96332cb956be01L },
      { 0x902fc93a413aa2b9L,0x99a4d915fc98c8a5L,0x52c29407565f1137L,
        0x4072690f21e4f281L } },
    /* 28 << 70 */
    { { 0x36e607cf02ff6072L,0xa47d2ca98ad98cdcL,0xbf471d1ef5f56609L,
        0xbcf86623f264ada0L },
      { 0xb70c0687aa9e5cb6L,0xc98124f217401c6cL,0x8189635fd4a61435L,
        0xd28fb8afa9d98ea6L } },
    /* 29 << 70 */
    { { 0xb9a67c2a40c251f8L,0x88cd5d87a2da44beL,0x437deb96e09b5423L,
        0x150467db64287dc1L },
      { 0xe161debbcdabb839L,0xa79e9742f1839a3eL,0xbb8dd3c2652d202bL,
        0x7b3e67f7e9f97d96L } },
    /* 30 << 70 */
    { { 0x5aa5d78fb1cb6ac9L,0xffa13e8eca1d0d45L,0x369295dd2ba5bf95L,
        0xd68bd1f839aff05eL },
      { 0xaf0d86f926d783f2L,0x543a59b3fc3aafc1L,0x3fcf81d27b7da97cL,
        0xc990a056d25dee46L } },
    /* 31 << 70 */
    { { 0x3e6775b8519cce2cL,0xfc9af71fae13d863L,0x774a4a6f47c1605cL,
        0x46ba42452fd205e8L },
      { 0xa06feea4d3fd524dL,0x1e7246416de1acc2L,0xf53816f1334e2b42L,
        0x49e5918e922f0024L } },
    /* 32 << 70 */
    { { 0x439530b665c7322dL,0xcf12cc01b3c1b3fbL,0xc70b01860172f685L,
        0xb915ee221b58391dL },
      { 0x9afdf03ba317db24L,0x87dec65917b8ffc4L,0x7f46597be4d3d050L,
        0x80a1c1ed006500e7L } },
    /* 33 << 70 */
    { { 0x84902a9678bf030eL,0xfb5e9c9a50560148L,0x6dae0a9263362426L,
        0xdcaeecf4a9e30c40L },
      { 0xc0d887bb518d0c6bL,0x99181152cb985b9dL,0xad186898ef7bc381L,
        0x18168ffb9ee46201L } },
    /* 34 << 70 */
    { { 0x9a04cdaa2502753cL,0xbb279e2651407c41L,0xeacb03aaf23564e5L,
        0x1833658271e61016L },
      { 0x8684b8c4eb809877L,0xb336e18dea0e672eL,0xefb601f034ee5867L,
        0x2733edbe1341cfd1L } },
    /* 35 << 70 */
    { { 0xb15e809a26025c3cL,0xe6e981a69350df88L,0x923762378502fd8eL,
        0x4791f2160c12be9bL },
      { 0xb725678925f02425L,0xec8631947a974443L,0x7c0ce882fb41cc52L,
        0xc266ff7ef25c07f2L } },
    /* 36 << 70 */
    { { 0x3d4da8c3017025f3L,0xefcf628cfb9579b4L,0x5c4d00161f3716ecL,
        0x9c27ebc46801116eL },
      { 0x5eba0ea11da1767eL,0xfe15145247004c57L,0x3ace6df68c2373b7L,
        0x75c3dffe5dbc37acL } },
    /* 37 << 70 */
    { { 0x3dc32a73ddc925fcL,0xb679c8412f65ee0bL,0x715a3295451cbfebL,
        0xd9889768f76e9a29L },
      { 0xec20ce7fb28ad247L,0xe99146c400894d79L,0x71457d7c9f5e3ea7L,
        0x097b266238030031L } },
    /* 38 << 70 */
    { { 0xdb7f6ae6cf9f82a8L,0x319decb9438f473aL,0xa63ab386283856c3L,
        0x13e3172fb06a361bL },
      { 0x2959f8dc7d5a006cL,0x2dbc27c675fba752L,0xc1227ab287c22c9eL,
        0x06f61f7571a268b2L } },
    /* 39 << 70 */
    { { 0x1b6bb97104779ce2L,0xaca838120aadcb1dL,0x297ae0bcaeaab2d5L,
        0xa5c14ee75bfb9f13L },
      { 0xaa00c583f17a62c7L,0x39eb962c173759f6L,0x1eeba1d486c9a88fL,
        0x0ab6c37adf016c5eL } },
    /* 40 << 70 */
    { { 0xa2a147dba28a0749L,0x246c20d6ee519165L,0x5068d1b1d3810715L,
        0xb1e7018c748160b9L },
      { 0x03f5b1faf380ff62L,0xef7fb1ddf3cb2c1eL,0xeab539a8fc91a7daL,
        0x83ddb707f3f9b561L } },
    /* 41 << 70 */
    { { 0xc550e211fe7df7a4L,0xa7cd07f2063f6f40L,0xb0de36352976879cL,
        0xb5f83f85e55741daL },
      { 0x4ea9d25ef3d8ac3dL,0x6fe2066f62819f02L,0x4ab2b9c2cef4a564L,
        0x1e155d965ffa2de3L } },
    /* 42 << 70 */
    { { 0x0eb0a19bc3a72d00L,0x4037665b8513c31bL,0x2fb2b6bf04c64637L,
        0x45c34d6e08cdc639L },
      { 0x56f1e10ff01fd796L,0x4dfb8101fe3667b8L,0xe0eda2539021d0c0L,
        0x7a94e9ff8a06c6abL } },
    /* 43 << 70 */
    { { 0x2d3bb0d9bb9aa882L,0xea20e4e5ec05fd10L,0xed7eeb5f1a1ca64eL,
        0x2fa6b43cc6327cbdL },
      { 0xb577e3cf3aa91121L,0x8c6bd5ea3a34079bL,0xd7e5ba3960e02fc0L,
        0xf16dd2c390141bf8L } },
    /* 44 << 70 */
    { { 0xb57276d980101b98L,0x760883fdb82f0f66L,0x89d7de754bc3eff3L,
        0x03b606435dc2ab40L },
      { 0xcd6e53dfe05beeacL,0xf2f1e862bc3325cdL,0xdd0f7921774f03c3L,
        0x97ca72214552cc1bL } },
    /* 45 << 70 */
    { { 0x5a0d6afe1cd19f72L,0xa20915dcf183fbebL,0x9fda4b40832c403cL,
        0x32738eddbe425442L },
      { 0x469a1df6b5eccf1aL,0x4b5aff4228bbe1f0L,0x31359d7f570dfc93L,
        0xa18be235f0088628L } },
    /* 46 << 70 */
    { { 0xa5b30fbab00ed3a9L,0x34c6137473cdf8beL,0x2c5c5f46abc56797L,
        0x5cecf93db82a8ae2L },
      { 0x7d3dbe41a968fbf0L,0xd23d45831a5c7f3dL,0xf28f69a0c087a9c7L,
        0xc2d75471474471caL } },
    /* 47 << 70 */
    { { 0x36ec9f4a4eb732ecL,0x6c943bbdb1ca6bedL,0xd64535e1f2457892L,
        0x8b84a8eaf7e2ac06L },
      { 0xe0936cd32499dd5fL,0x12053d7e0ed04e57L,0x4bdd0076e4305d9dL,
        0x34a527b91f67f0a2L } },
    /* 48 << 70 */
    { { 0xe79a4af09cec46eaL,0xb15347a1658b9bc7L,0x6bd2796f35af2f75L,
        0xac9579904051c435L },
      { 0x2669dda3c33a655dL,0x5d503c2e88514aa3L,0xdfa113373753dd41L,
        0x3f0546730b754f78L }