package Music::Lyrics::LRC;

# Force me to write this properly
use strict;
use warnings;
use utf8;

# Include required modules
use Carp;
use English '-no_match_vars';

# Target reasonably old Perl
use 5.006;

# Declare package version
our $VERSION = '0.04';

# Patterns to match elements of the LRC file; these are somewhat tolerant
our %RE = (

    # A blank line
    blank => qr{
        \A      # Start of string
        \s*     # Any whitespace
        \z      # End of string
    }msx,

    # A meta tag line
    tag => qr{
        \A          # Start of string
        \s*         # Any whitespace
        \[          # Opening left bracket
            ([^:\r\n]+)  # Tag name, capture
            :          # Colon
            (.*)       # Tag value, capture
        \]          # Closing right bracket
        \s*         # Any whitespace
        \z          # End of string
    }msx,

    # A lyric line
    lyric => qr{
        \A            # Start of string
        \s*           # Any whitespace
        \[            # Opening left bracket
            (\d+)         # Minutes, capture
            :             # Colon
            (\d{2})       # Seconds, capture
            [.]           # Period
            (\d{2})       # Hundredths of a second, capture
        \]            # Closing right bracket
        [\t ]*        # Any tabs or spaces
        (.*\S)        # Lyric line, capture
        \s*           # Any whitespace
        \z            # End of string
    }msx,
);

# Parser functions to consume and process captures from the above patterns
my %parsers = (

    # A meta tag line
    tag => sub {
        my ( $self, $tag, $value ) = @_;
        $self->set_tag( $tag, $value );
    },

    # A lyric line
    lyric => sub {
        my ( $self, %ts, $text );
        ( $self, @ts{qw(min sec csec)}, $text ) = @_;

        # Calculate the number of milliseconds
        my $msec = $self->ts_to_msec( \%ts );

        # Push a lyric hashref onto our list
        $self->add_lyric( $msec, $text );
    },
);

# Oldschool constructor
sub new {
    my ($class) = @_;

    # Start with empty tags and lyrics
    my %self;
    $self{tags}   = {};
    $self{lyrics} = [];

    # Perlician, bless thyself
    return bless \%self, $class;
}

# Read-only accessor for lyrics, sorted by time
sub lyrics {
    my $self = shift;
    my @lyrics = sort { $a->{time} <=> $b->{time} } @{ $self->{lyrics} };
    return \@lyrics;
}

# Read-only accessor for tags
sub tags {
    my $self = shift;
    my %tags = %{ $self->{tags} };
    return \%tags;
}

# Add a new lyric to the object
sub add_lyric {
    my ( $self, $time, $text ) = @_;

    # Check parameters
    int $time >= 0
      or croak 'Bad lyric time';
    $text !~ m/ [\r\n] /msx
      or croak 'Bad lyric line';

    # Push the lyric onto our list
    return push @{ $self->{lyrics} },
      {
        time => $time,
        text => $text,
      };
}

# Set the value of a tag
sub set_tag {
    my ( $self, $name, $value ) = @_;

    # Check parameters
    $name !~ m/ [:\r\n] /msx
      or croak 'Bad tag name';

    # Tag content cannot have vertical whitespace
    $value !~ m/ [\r\n] /msx
      or croak 'Bad tag value';

    # Set the tag's value on our hash
    return ( $self->{tags}{$name} = $value );
}

# Unset a tag
sub unset_tag {
    my ( $self, $name ) = @_;

    # Check parameters
    $name !~ m/ [:\r\n] /msx
      or croak 'Bad tag name';
    exists $self->{tags}{$name}
      or carp 'Tag not set';

    # Delete the tag's value
    return defined delete $self->{tags}{$name};
}

# Parse an LRC file from a given filehandle
sub load {
    my ( $self, $fh ) = @_;

    # Panic if this doesn't look like a filehandle
    ref $fh eq 'GLOB'
      or croak 'Not a filehandle';

    # Iterate through lines
  LINE: while ( my $line = <$fh> ) {

        # Iterate through line types until one matches
      TYPE: for my $type (qw(lyric tag blank)) {
            my @vals = $line =~ $RE{$type}
              or next TYPE;
            exists $parsers{$type}
              or next LINE;
            $parsers{$type}->( $self, @vals );
            next LINE;
        }

        # This line doesn't match anything we understand, complain but persist
        warn "Unknown format for line $NR\n";
    }

    # Check we got to the end of the file
    eof $fh or die "Failed file read: $ERRNO\n";

    # All done, return the number of lyrics we have now
    return scalar @{ $self->lyrics };
}

# Write an LRC file to a given filehandle
sub save {
    my ( $self, $fh ) = @_;

    # Panic if this doesn't look like a filehandle
    ref $fh eq 'GLOB'
      or croak 'Not a filehandle';

    # Start counting lines written
    my $lines = 0;

    # Iterate through tags
    for my $name ( sort keys %{ $self->{tags} } ) {
        my $value = $self->{tags}{$name};
        $lines += printf {$fh} "[%s:%s]\n", $name, $value
          or die "Failed tag write: $ERRNO\n";
    }

    # Iterate through lyrics (sorted by time)
    for my $lyric ( @{ $self->lyrics } ) {

        # Convert milliseconds to timestamp hash
        my $msec = $lyric->{time};
        my %ts   = %{ $self->msec_to_ts($msec) };

        # Write the line to the file, counting the lines
        $lines += printf {$fh} "[%02u:%02u:%02u]%s\n",
          @ts{qw(min sec csec)}, $lyric->{text}
          or die "Failed lyric write: $ERRNO\n";
    }

    # Return the number of lines written
    return $lines;
}

# Factors for timestamp<->millisecond conversions
our %MSF = (
    min  => 60_000,
    sec  => 1_000,
    csec => 10,
);

# Convert an LRC timestamp hashref to milliseconds
sub ts_to_msec {
    my ( $self, $ts ) = @_;
    my $msec = 0;
    for my $k ( keys %{$ts} ) {
        $msec += $ts->{$k} * $MSF{$k};
    }
    return $msec;
}

# Convert milliseconds to an LRC timestamp hashref
sub msec_to_ts {
    my ( $self, $msec ) = @_;
    my %ts;
    for my $k (qw(min sec csec)) {
        $ts{$k} = int $msec / $MSF{$k};
        $msec %= $MSF{$k};
    }
    return \%ts;
}

1;

__END__

=pod

=for stopwords
LRC tradename licensable MERCHANTABILITY arrayref hashrefs hashref millseconds
filehandle writeable whitespace

=head1 NAME

Music::Lyrics::LRC - Manipulate LRC karaoke timed lyrics files

=head1 VERSION

Version 0.04

=head1 DESCRIPTION

Read, write, and do simple manipulations of the LRC lyrics files used for some
karaoke devices.

For details on the LRC file format, please see Wikipedia:

L<https://en.wikipedia.org/wiki/LRC_(file_format)>

=head1 SYNOPSIS

    use Music::Lyrics::LRC;
    ...
    my $lrc = Music::Lyrics::LRC->new();
    open my $rfh, '<', 'mylyrics.lrc';
    $lrc->load($rfh);
    ...
    my $lyrics = $lrc->lyrics();  # arrayref of hashrefs: time (msec), text
    my $tags = $lrc->tags();  # hashref, name => value
    ...
    $lrc->add_lyric(5500, q(Now I'm singin' at exactly 5.5 seconds...));
    $lrc->add_tag('author', 'Justin A. Perlhacker');
    ...
    open my $wfh, '>', 'savelyrics.lrc';
    $lrc->save($wfh);

=head1 SUBROUTINES/METHODS

=head2 C<new()>

Constructor; no arguments.

=head2 C<lyrics()>

Retrieve an arrayref of hashrefs representing lyric lines, sorted by time
ascending. Each one has C<time> and C<text> keys. The time is in milliseconds.

    [
        {
            time => 5500,
            text => 'Now I\'m singin\' at exactly 5.5 seconds...',
        },
        {
            time => 6001,
            text => 'And now a moment after the sixth...',
        },
    ...
    ]

=head2 C<tags()>

Retrieve a hashref of tag names to tag values for this lyrics file.

    {
        ar => 'Justin A. Perlhacker',
        ti => 'Perl timekeeping blues',
        ...
    }

=head2 C<add_lyric($time, $text)>

Add a lyric at the given non-negative time in millseconds and with the given
text. The text must not include newlines or carriage returns.

=head2 C<set_tag($name, $value)>

Set a tag with the given name and value. The name must be at least one
character and cannot have colons. Neither the name nor the value can include
newlines or carriage returns.

=head2 C<unset_tag($name)>

Clear a tag with the given name. Raises a warning if the tag has not been set.

=head2 C<load($fh)>

Load lyrics from the given readable filehandle.

=head2 C<save($fh)>

Save lyrics to the given writeable filehandle.

=head2 C<ts_to_msec(\%ts)>

Convert the internal LRC timestamp hash structure to milliseconds. You are
probably not interested in this.

=head2 C<msec_to_ts($msec)>

Convert milliseconds to the internal LRC timestamp hash structure. You are
probably not interested in this, either.

=head1 DIAGNOSTICS

=over 4

=item C<Bad lyric time>

A lyric could not be added with the given time. It may have been negative.

=item C<Bad lyric line>

The line you tried to add had illegal characters in it, probably a carriage
return or a newline.

=item C<Bad tag name>

The tag you tried to set had an illegal name. It needs to be at least one
character, and can't include colons or whitespace.

=item C<Bad tag value>

You tried to set a tag to an illegal value. The value probably had a carriage
return or a newline in it.

=item C<Tag not set>

You tried to clear a tag that had not already been set.

=item C<Unknown format for line %s>

The parser ran across a line in the LRC file that it could not understand. It
tolerates blank lines, tags, and lyric lines, and doesn't know anything else.

=item C<Failed file read: %s>

The file read failed with the given system error.

=item C<Not a filehandle>

You passed C<load()> or C<save()> something that wasn't a filehandle.

=item C<Failed tag write: %s>

An attempt to write a tag to the output filehandle in C<save()> failed with the
given system error.

=item C<Failed lyric write: %s>

An attempt to write a lyric timestamp and line to the output filehandle in
C<save()> failed with the given system error.

=back

=head1 CONFIGURATION AND ENVIRONMENT

You'll need to make sure that you're passing in a filehandle with the
appropriate I/O layers you want, especially encoding.

=head1 DEPENDENCIES

=over 4

=item *

Perl 5.6 or newer

=item *

C<Carp>

=item *

C<English>

=back

=head1 INCOMPATIBILITIES

This module does not support any "extended" or "enhanced" LRC format; in
particular, at the time of writing it can't handle per-word times syntax. This
may change in future revisions.

=head1 BUGS AND LIMITATIONS

The format accepted here is very liberal, and needs to be tested with lots of
different LRC files from the wild.

The test suite is skeletal, and needs a lot of fleshing out.

=head1 AUTHOR

Tom Ryder C<< <tom@sanctum.geek.nz> >>

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2017 Tom Ryder

This program is free software; you can redistribute it and/or modify it under
the terms of the Artistic License (2.0). You may obtain a copy of the full
license at:

<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified Versions is
governed by this Artistic License. By using, modifying or distributing the
Package, you accept this license. Do not use, modify, or distribute the
Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made by
someone other than you, you are nevertheless required to ensure that your
Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service mark,
tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge patent
license to make, have made, use, offer to sell, sell, import and otherwise
transfer the Package with respect to any patent claims licensable by the
Copyright Holder that are necessarily infringed by the Package. If you
institute patent litigation (including a cross-claim or counterclaim) against
any party alleging that the Package constitutes direct or contributory patent
infringement, then this Artistic License to you shall terminate on the date
that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER AND
CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES. THE IMPLIED
WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY YOUR LOCAL LAW.
UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR CONTRIBUTOR WILL BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING IN ANY WAY
OUT OF THE USE OF THE PACKAGE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.

=cut
