/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/** Clownfish::CFC::Hierarchy - A class hierarchy.
 *
 * A Clownfish::CFC::Hierarchy consists of all the classes defined in files within
 * a source directory and its subdirectories.
 *
 * There may be more than one tree within the Hierarchy, since all "inert"
 * classes are root nodes, and since Clownfish does not officially define any
 * core classes itself from which all instantiable classes must descend.
 */

#ifndef H_CFCHIERARCHY
#define H_CFCHIERARCHY

#ifdef __cplusplus
extern "C" {
#endif

typedef struct CFCHierarchy CFCHierarchy;
struct CFCClass;
struct CFCFile;

/**
 * @param source The directory we begin reading files from.
 * @param dest The directory where the autogenerated files will be written.
 */
CFCHierarchy*
CFCHierarchy_new(const char *source, const char *dest);

CFCHierarchy*
CFCHierarchy_init(CFCHierarchy *self, const char *source, const char *dest);

void
CFCHierarchy_destroy(CFCHierarchy *self);

/** Parse every Clownfish header file which can be found under C<source>,
 * building up the object hierarchy.
 */
void
CFCHierarchy_build(CFCHierarchy *self);

/** Visit all File objects in the hierarchy.  If a parent node is modified, mark
 * all of its children as modified.
 *
 * If the supplied argument is true, mark all Files as modified.
 */
int
CFCHierarchy_propagate_modified(CFCHierarchy *self, int modified);

/** Return all Classes as an array with the property that every parent class
 * will precede all of its children.
 */
struct CFCClass**
CFCHierarchy_ordered_classes(CFCHierarchy *self);

struct CFCFile**
CFCHierarchy_files(CFCHierarchy *self);

const char*
CFCHierarchy_get_source(CFCHierarchy *self);

const char*
CFCHierarchy_get_dest(CFCHierarchy *self);

#ifdef __cplusplus
}
#endif

#endif /* H_CFCHIERARCHY */

