package Crypto::Exchange::Binance::Spot::API;
$Crypto::Exchange::Binance::Spot::API::VERSION = '0.05';
=head1 NAME

Crypto::Exchange::Binance::Spot::API - Binance Spot API

=head1 USAGE

 my $binance = Crypto::Exchange::Binance::Spot::API->new;

Get Prices

 $hash = $binance->prices(pair => { coin => 'XRP', base => 'GBP' });

 Got: $hash = { last_price => 1234, pair => {coin => 'XRP', base => 'GBP'} }

Get Wallet Balances

 $hash = $binance->balances();

 Got: $hash = {
    XRP => { available => 123, in_order => 0, staking => 0 },
    GBP => { available => 321, in_order => 0, staking => 0 },
 }

Get the opening orders

 $list = $binance->list_open_orders(pair => { coin => 'XRP', base => 'GBP' });

 Got: $list = [
    {
        pair => { coin => 'XRP', base => 'GBP' },
        order_id => 123,
        external_id => 'YOUR OWN DEFINED ID',
        buy_or_sell => 'buy|sell',
        order_qty   => 55,
        filled_qty  => 54,
        timestamp   => 131321321321,
        filled_qty  => 131231321321,
    },
    ...
 ];

=head2 Open Source

 git@github.com:mvu8912/perl5-crypto-exchange-binance-spot-api.git

=cut

use Moo;
use DateTime;

extends 'Crypto::Exchange::API';

has time_zone => (
    is      => 'ro',
    lazy    => 1,
    builder => 1,
);

sub _build_time_zone { $ENV{TIME_ZONE} // 'Europe/London' }

sub _build_key {}

sub _build_secret {}

sub _build_base_url {
    return URI->new('https://api.binance.com');
}

sub _build_pre_defined_headers {
    my ($self) = @_;
    if ($self->key) {
        return {'X-MBX-APIKEY' => $self->key};
    }
    else {
        return {};
    }
}

sub _build_pre_defined_data {
    return {
        timestamp => sub { time * 1000 },
        signature => sub {
            my ($self, %o) = @_;
            if ($self->secret) {
                my $params = $self->kvp2str(%o, skip_key => { signature => 1 });
                $self->do_hmac_sha256_hex($params, $self->secret);
            }
        },
    };
}

## https://www.binance.com/bapi/asset/v2/public/asset-service/product/get-products?includeEtf=true
my $DATA = do {local $/; <DATA>};

sub _build_base_currencies {
    my ($self) = @_;

    if (my $my_currency_file = $ENV{MY_BINANCE_CURRENCIES}) {
        open my $fh, '<', $my_currency_file
            or die "Unable to open my currency file: $my_currency_file";
        local $/; $DATA = <$fh>;
        $fh->close;
    }

    my $data = $self->json->decode($DATA)->{data};
    my %rows = ();

    foreach my $row(@$data) {
        my ($base, $coin) = @$row{qw(q b)};
        $rows{"$coin$base"} = { base => $base, coin => $coin };
    }

    return \%rows;
}

sub request_attr_pair {
    my ($self, $pair) = @_;
    return join '', map {uc} @$pair{qw( coin base )};
}

sub response_attr_pair {
    my ($self, $pair) = @_;
    my $bases = $self->base_currencies;

    return $bases->{$pair}
       || die "Pair [$pair] is not a support trading pair";
}

sub response_attr_timestamp {
    my ($self, $epoch) = @_;
    my $t = DateTime->from_epoch(epoch => $epoch / 1000);
    if (my $tz = $self->time_zone) {
        $t->set_time_zone($tz);
    }
    return sub {wantarray ? $t : "$t"};
}

sub response_attr_filled_time {
    return response_attr_timestamp(@_);
}

sub response_attr_last_updated {
    return response_attr_timestamp(@_);
}

sub set_prices {{
    request => {
        method => 'get',
        path   => '/api/v3/ticker/price',
        data   => {
            pair => {
                field_name => 'symbol',
                required   => 1,
            },
        },
        events => {
            keys => [ 'pair' ],
        },
    },
    response => {
        row => {
            pair       => 'symbol',
            last_price => 'price',
        },
    },
}}

sub set_balances {{
    request => {
        method => 'get',
        path   => '/api/v3/account',
        events => {
            keys => [ qw( timestamp signature ) ],
        },
    },
    response => {
        key  => 'balances',
        row  => {
            coin      => 'asset',
            available => 'free',
            in_order  => 'locked',
        },
        row_filter => sub {
            my ($self, $row) = @_;
            if ($row->{available} == 0 && $row->{in_order} == 0) {
                return 'next';
            }
        },
        array2hash => 'coin',
        post_row => sub {
            my ($self, $row, $rows) = @_;
            my $coin = $row->{coin};
            return if $coin =~ m/^LD/;
            my $earn = delete $rows->{"LD$coin"} || {available => 0};
            $rows->{$coin}{staking} = $earn->{available};
        },
    },
}}

sub set_list_open_orders {
    my ($self) = @_;

    my $spec = $self->set_list_all_orders;

    my $old_filter = $spec->{response}{row_filter} || sub {};

    $spec->{response}{row_filter} = sub {
        my ($self, $row) = @_;
        return $self->$old_filter($row)
            || $row->{status} eq 'FILLED' ? 'next' : '';
    };

    $spec->{response}{sort} = sub {
        my ($self, $a, $b) = @_;
        return ($a->{price} || 0) <=> ($b->{price} || 0);
    };

    return $spec;
}

sub set_list_filled_orders {
    my ($self) = @_;

    my $spec = $self->set_list_all_orders;

    my $old_filter = $spec->{response}{row_filter} || sub {};

    $spec->{response}{row}{filled_time} = delete $spec->{response}{row}{last_updated};

    $spec->{response}{row_filter} = sub {
        my ($self, $row) = @_;
        return $self->$old_filter($row)
            || $row->{status} ne 'FILLED' ? 'next' : '';
    };

    $spec->{response}{sort} = sub {
        my ($self, $a, $b) = @_;
        return $a->{filled_time}->().'' cmp $b->{filled_time}->().'';
    };

    return $spec;
}

sub set_list_all_orders {{
    request => {
        method => 'get',
        path   => '/api/v3/allOrders',
        events => {
            keys => [ qw( pair order_id start_time end_time page_size timestamp signature ) ],
        },
        data => {
            pair => {
                field_name => 'symbol',
                required   => 1,
            },
            order_id => 'orderId',
            start_time => 'startTime',
            end_time   => 'endTime',
            page_size  => 'limit',
        },
    },
    response => {
        row => {
            pair         => 'symbol',
            order_id     => 'orderId',
            external_id  => 'clientOrderId',
            buy_or_sell  => 'side',
            order_qty    => 'origQty',
            filled_qty   => 'executedQty',
            timestamp    => 'time',
            last_updated => 'updateTime',
            price        => 'price',
            status       => 'status',
            order_type   => 'type',
            _others      => [qw(
                orderListId
                cummulativeQuoteQty
                timeInForce
                stopPrice
                icebergQty
                isWorking
                origQuoteOrderQty
            )],
        },
        sort => sub {
            my ($self, $a, $b) = @_;
            return $a->{timestamp}->().'' cmp $b->{timestamp}->().'';
        },
    },
}}

sub set_test_buy {
    my $spec = set_buy();

    $spec->{request}{path} .= '/test';

    return $spec;
}

sub set_buy {{
    request => {
        method => 'post',
        path   => '/api/v3/order',
        events => {
            keys => [ qw( pair amount price external_id buy_or_sell type timestamp signature ) ],
        },
        data => {
            pair => {
                field_name => 'symbol',
                required   => 1,
            },
            amount => {
                field_name => 'quantity',
                required   => 1,
            },
            price => {
                field_name => 'price',
                required   => 1,
            },
            external_id => 'newClientOrderId',
            ## ----------------------------
            buy_or_sell => {
                field_name => 'side',
                default    => 'BUY',
            },
            type => {
                field_name => 'type',
                default    => 'LIMIT_MAKER',
            },
        },
    },
    response => {
        row => {
            pair        => 'symbol',
            order_id    => 'orderId',
            external_id => 'clientOrderId',
            buy_or_sell => 'side',
            order_qty   => 'origQty',
            timestamp   => 'time',
            status      => 'status',
            order_type  => 'type',
            price       => 'price',
            _others     => [qw(
                orderListId
                cummulativeQuoteQty
                timeInForce
                stopPrice
                updateTime
                icebergQty
                isWorking
                origQuoteOrderQty
                fills
            )],
        },
    },
}}

sub set_test_sell {
    my $spec = set_sell();

    $spec->{request}{path} .= '/test';

    return $spec;
}

sub set_sell {
    my $spec = set_buy();

    $spec->{request}{data}{buy_or_sell}{default} .= 'SELL';

    return $spec;
}

sub DESTROY {}

no Moo;

1;

__DATA__
{
   "code" : "000000",
   "data" : [
      {
         "an" : "BNB",
         "as" : 459499.02,
         "b" : "BNB",
         "ba" : "",
         "c" : "0.010438",
         "cs" : 152665937,
         "etf" : false,
         "h" : "0.010485",
         "i" : "0.01000000",
         "l" : "0.009720",
         "o" : "0.010008",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "4624.48405048",
         "s" : "BNBBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.000001",
         "v" : "459499.020000",
         "y" : 0
      },
      {
         "an" : "Nuls",
         "as" : 1922629,
         "b" : "NULS",
         "ba" : "",
         "c" : "0.00001996",
         "cs" : 99620739,
         "etf" : false,
         "h" : "0.00002054",
         "i" : "1.00000000",
         "l" : "0.00001868",
         "o" : "0.00001922",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "37.36752036",
         "s" : "NULSBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "1922629.00000000",
         "y" : 0
      },
      {
         "an" : "NEO",
         "as" : 207285.14,
         "b" : "NEO",
         "ba" : "",
         "c" : "0.001715",
         "cs" : 65000000,
         "etf" : false,
         "h" : "0.001758",
         "i" : "0.01000000",
         "l" : "0.001606",
         "o" : "0.001649",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "349.9414973",
         "s" : "NEOBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "207285.140000",
         "y" : 0
      },
      {
         "an" : "ChainLink",
         "as" : 1253767.4,
         "b" : "LINK",
         "ba" : "",
         "c" : "0.00066264",
         "cs" : 419009556,
         "etf" : false,
         "h" : "0.00067350",
         "i" : "0.10000000",
         "l" : "0.00063347",
         "o" : "0.00064849",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "818.69422786",
         "s" : "LINKBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "1253767.40000000",
         "y" : 0
      },
      {
         "an" : "MIOTA",
         "as" : 7077034,
         "b" : "IOTA",
         "ba" : "",
         "c" : "0.00003958",
         "cs" : 2779530283,
         "etf" : false,
         "h" : "0.00004045",
         "i" : "1.00000000",
         "l" : "0.00003550",
         "o" : "0.00003732",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "266.37939168",
         "s" : "IOTABTC",
         "st" : "TRADING",
         "tags" : [
            "pow"
         ],
         "ts" : "0.00000001",
         "v" : "7077034.00000000",
         "y" : 0
      },
      {
         "an" : "Ethereum Classic",
         "as" : 457009.64,
         "b" : "ETC",
         "ba" : "",
         "c" : "0.0006149",
         "cs" : 0,
         "etf" : false,
         "h" : "0.0006285",
         "i" : "0.01000000",
         "l" : "0.0005884",
         "o" : "0.0005966",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "277.11094467",
         "s" : "ETCBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "457009.6400000",
         "y" : 0
      },
      {
         "an" : "KyberNetwork",
         "as" : 2074182,
         "b" : "KNC",
         "ba" : "",
         "c" : "0.00005309",
         "cs" : 205045092,
         "etf" : false,
         "h" : "0.00005396",
         "i" : "1.00000000",
         "l" : "0.00004940",
         "o" : "0.00005020",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "108.33839596",
         "s" : "KNCBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "2074182.00000000",
         "y" : 0
      },
      {
         "an" : "Walton",
         "as" : 1824637.69,
         "b" : "WTC",
         "ba" : "",
         "c" : "0.00002929",
         "cs" : 72133493,
         "etf" : false,
         "h" : "0.00002995",
         "i" : "0.01000000",
         "l" : "0.00002654",
         "o" : "0.00002790",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "50.97943831",
         "s" : "WTCBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "1824637.69000000",
         "y" : 0
      },
      {
         "an" : "Breaker",
         "as" : 41554398,
         "b" : "SNGLS",
         "ba" : "",
         "c" : "0.00000050",
         "cs" : 600000000,
         "etf" : false,
         "h" : "0.00000051",
         "i" : "1.00000000",
         "l" : "0.00000045",
         "o" : "0.00000046",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "19.77301334",
         "s" : "SNGLSBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "41554398.00000000",
         "y" : 0
      },
      {
         "an" : "EOS",
         "as" : 1386264.69,
         "b" : "EOS",
         "ba" : "",
         "c" : "0.002349",
         "cs" : 952879989,
         "etf" : false,
         "h" : "0.002488",
         "i" : "0.01000000",
         "l" : "0.002223",
         "o" : "0.002262",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "3169.43094016",
         "s" : "EOSETH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "1386264.690000",
         "y" : 0
      },
      {
         "an" : "Status",
         "as" : 4371181,
         "b" : "SNT",
         "ba" : "",
         "c" : "0.00006440",
         "cs" : 0,
         "etf" : false,
         "h" : "0.00006508",
         "i" : "1.00000000",
         "l" : "0.00005770",
         "o" : "0.00005954",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "270.16257759",
         "s" : "SNTETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "4371181.00000000",
         "y" : 0
      },
      {
         "an" : "Bitcoin",
         "as" : 65634.507654,
         "b" : "BTC",
         "ba" : "à¸¿",
         "c" : "54926.22",
         "cs" : 18692200,
         "etf" : false,
         "h" : "55281.16",
         "i" : "0.00000100",
         "l" : "52600.36",
         "o" : "53769.31",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "3538809166.67917833",
         "s" : "BTCUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "65634.51",
         "y" : 0
      },
      {
         "an" : "OMG Network",
         "as" : 80710.73,
         "b" : "OMG",
         "ba" : "",
         "c" : "0.002920",
         "cs" : 102042552,
         "etf" : false,
         "h" : "0.002947",
         "i" : "0.01000000",
         "l" : "0.002736",
         "o" : "0.002769",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "228.51020058",
         "s" : "OMGETH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "80710.730000",
         "y" : 0
      },
      {
         "an" : "NeoGas",
         "as" : 567094.81,
         "b" : "GAS",
         "ba" : "",
         "c" : "0.0002820",
         "cs" : 10128375,
         "etf" : false,
         "h" : "0.0003100",
         "i" : "0.01000000",
         "l" : "0.0002581",
         "o" : "0.0002693",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "159.97859118",
         "s" : "GASBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "567094.8100000",
         "y" : 0
      },
      {
         "an" : "Voyager Token",
         "as" : 247363,
         "b" : "BQX",
         "ba" : "",
         "c" : "0.0016045",
         "cs" : null,
         "etf" : false,
         "h" : "0.0017187",
         "i" : "1.00000000",
         "l" : "0.0014449",
         "o" : "0.0014695",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "388.3461193",
         "s" : "BQXETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "247363.0000000",
         "y" : 0
      },
      {
         "an" : "Qtum",
         "as" : 185674.31,
         "b" : "QTUM",
         "ba" : "",
         "c" : "0.005788",
         "cs" : 98317668,
         "etf" : false,
         "h" : "0.006036",
         "i" : "0.01000000",
         "l" : "0.005261",
         "o" : "0.005339",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "1056.11797332",
         "s" : "QTUMETH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "185674.310000",
         "y" : 0
      },
      {
         "an" : "Bancor",
         "as" : 37466.03,
         "b" : "BNT",
         "ba" : "",
         "c" : "0.002563",
         "cs" : 178540878,
         "etf" : false,
         "h" : "0.002600",
         "i" : "0.01000000",
         "l" : "0.002504",
         "o" : "0.002517",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "95.78084509",
         "s" : "BNTETH",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.000001",
         "v" : "37466.030000",
         "y" : 0
      },
      {
         "an" : "SONM",
         "as" : 5874055,
         "b" : "SNM",
         "ba" : "",
         "c" : "0.00000155",
         "cs" : 0,
         "etf" : false,
         "h" : "0.00000155",
         "i" : "1.00000000",
         "l" : "0.00000142",
         "o" : "0.00000144",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "8.80475087",
         "s" : "SNMBTC",
         "st" : "BREAK",
         "tags" : [
            "storage-zone"
         ],
         "ts" : "0.00000001",
         "v" : "5874055.00000000",
         "y" : 0
      },
      {
         "an" : "Voyager Token",
         "as" : 2126790,
         "b" : "BQX",
         "ba" : "",
         "c" : "0.00007498",
         "cs" : null,
         "etf" : false,
         "h" : "0.00008050",
         "i" : "1.00000000",
         "l" : "0.00006700",
         "o" : "0.00006836",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "155.54529522",
         "s" : "BQXBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "2126790.00000000",
         "y" : 0
      },
      {
         "an" : "NEO",
         "as" : 38738.6,
         "b" : "NEO",
         "ba" : "",
         "c" : "0.036865",
         "cs" : 65000000,
         "etf" : false,
         "h" : "0.037500",
         "i" : "0.01000000",
         "l" : "0.034659",
         "o" : "0.035417",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "1405.52993187",
         "s" : "NEOETH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "38738.600000",
         "y" : 0
      },
      {
         "an" : "KyberNetwork",
         "as" : 78321,
         "b" : "KNC",
         "ba" : "",
         "c" : "0.0011375",
         "cs" : 205045092,
         "etf" : false,
         "h" : "0.0011603",
         "i" : "1.00000000",
         "l" : "0.0010623",
         "o" : "0.0010749",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "87.5753738",
         "s" : "KNCETH",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.0000001",
         "v" : "78321.0000000",
         "y" : 0
      },
      {
         "an" : "0x",
         "as" : 195637,
         "b" : "ZRX",
         "ba" : "",
         "c" : "0.00063475",
         "cs" : 781547659,
         "etf" : false,
         "h" : "0.00064010",
         "i" : "1.00000000",
         "l" : "0.00059428",
         "o" : "0.00060526",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "121.46156535",
         "s" : "ZRXETH",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "195637.00000000",
         "y" : 0
      },
      {
         "an" : "Qtum",
         "as" : 884297.09,
         "b" : "QTUM",
         "ba" : "",
         "c" : "0.0002704",
         "cs" : 98317668,
         "etf" : false,
         "h" : "0.0002829",
         "i" : "0.01000000",
         "l" : "0.0002441",
         "o" : "0.0002487",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "234.49117732",
         "s" : "QTUMBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "884297.0900000",
         "y" : 0
      },
      {
         "an" : "FunFair",
         "as" : 34653503,
         "b" : "FUN",
         "ba" : "",
         "c" : "0.00001313",
         "cs" : 10298879188,
         "etf" : false,
         "h" : "0.00001342",
         "i" : "1.00000000",
         "l" : "0.00001241",
         "o" : "0.00001266",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "450.45611328",
         "s" : "FUNETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "34653503.00000000",
         "y" : 0
      },
      {
         "an" : "Litecoin",
         "as" : 250026.97,
         "b" : "LTC",
         "ba" : "Å",
         "c" : "0.004668",
         "cs" : 66752414,
         "etf" : false,
         "h" : "0.004699",
         "i" : "0.01000000",
         "l" : "0.004498",
         "o" : "0.004524",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "1149.0826926",
         "s" : "LTCBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "250026.970000",
         "y" : 0
      },
      {
         "an" : "ChainLink",
         "as" : 233028.51,
         "b" : "LINK",
         "ba" : "",
         "c" : "0.014229",
         "cs" : 419009556,
         "etf" : false,
         "h" : "0.014408",
         "i" : "0.01000000",
         "l" : "0.013583",
         "o" : "0.013935",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "3256.60595309",
         "s" : "LINKETH",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.000001",
         "v" : "233028.510000",
         "y" : 0
      },
      {
         "an" : "Ethereum",
         "as" : 149058.628,
         "b" : "ETH",
         "ba" : "Î",
         "c" : "0.046578",
         "cs" : 115634925,
         "etf" : false,
         "h" : "0.047099",
         "i" : "0.00100000",
         "l" : "0.046000",
         "o" : "0.046563",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "6944.85559163",
         "s" : "ETHBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "149058.628000",
         "y" : 0
      },
      {
         "an" : "Verge",
         "as" : 106945720,
         "b" : "XVG",
         "ba" : "",
         "c" : "0.00002041",
         "cs" : 16447987349,
         "etf" : false,
         "h" : "0.00002100",
         "i" : "1.00000000",
         "l" : "0.00001836",
         "o" : "0.00001964",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "2093.69432174",
         "s" : "XVGETH",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "106945720.00000000",
         "y" : 0
      },
      {
         "an" : "0x",
         "as" : 2267697,
         "b" : "ZRX",
         "ba" : "",
         "c" : "0.00002961",
         "cs" : 781547659,
         "etf" : false,
         "h" : "0.00002987",
         "i" : "1.00000000",
         "l" : "0.00002767",
         "o" : "0.00002818",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "65.5159374",
         "s" : "ZRXBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "2267697.00000000",
         "y" : 0
      },
      {
         "an" : "MIOTA",
         "as" : 574682,
         "b" : "IOTA",
         "ba" : "",
         "c" : "0.00084756",
         "cs" : 2779530283,
         "etf" : false,
         "h" : "0.00086617",
         "i" : "1.00000000",
         "l" : "0.00076754",
         "o" : "0.00080064",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "467.17085574",
         "s" : "IOTAETH",
         "st" : "TRADING",
         "tags" : [
            "pow"
         ],
         "ts" : "0.00000001",
         "v" : "574682.00000000",
         "y" : 0
      },
      {
         "an" : "OMG Network",
         "as" : 786158.6,
         "b" : "OMG",
         "ba" : "",
         "c" : "0.0001360",
         "cs" : 102042552,
         "etf" : false,
         "h" : "0.0001380",
         "i" : "0.01000000",
         "l" : "0.0001278",
         "o" : "0.0001292",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "104.23673959",
         "s" : "OMGBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "786158.6000000",
         "y" : 0
      },
      {
         "an" : "1inch",
         "as" : 1041123.2,
         "b" : "1INCH",
         "ba" : "",
         "c" : "0.00008754",
         "cs" : null,
         "etf" : false,
         "h" : "0.00008950",
         "i" : "0.10000000",
         "l" : "0.00008264",
         "o" : "0.00008341",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "89.24776064",
         "s" : "1INCHBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "1041123.20000000",
         "y" : 0
      },
      {
         "an" : "1inch",
         "as" : 768665.52,
         "b" : "1INCH",
         "ba" : "",
         "c" : "4.8169",
         "cs" : null,
         "etf" : false,
         "h" : "4.9389",
         "i" : "0.01000000",
         "l" : "4.3600",
         "o" : "4.4860",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "3546414.624039",
         "s" : "1INCHBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "768665.5200",
         "y" : 0
      },
      {
         "an" : "1INCHDOWN",
         "as" : 112208.13,
         "b" : "1INCHDOWN",
         "ba" : "",
         "c" : "7.23",
         "cs" : null,
         "etf" : true,
         "h" : "9.36",
         "i" : "0.01000000",
         "l" : "6.70",
         "o" : "8.96",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "870511.6438",
         "s" : "1INCHDOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.01",
         "v" : "112208.13",
         "y" : 0
      },
      {
         "an" : "1INCHUP",
         "as" : 1638671.66,
         "b" : "1INCHUP",
         "ba" : "",
         "c" : "4.13",
         "cs" : null,
         "etf" : true,
         "h" : "4.33",
         "i" : "0.01000000",
         "l" : "3.34",
         "o" : "3.53",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "6414570.2328",
         "s" : "1INCHUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.01",
         "v" : "1638671.66",
         "y" : 0
      },
      {
         "an" : "1inch",
         "as" : 7112085.78,
         "b" : "1INCH",
         "ba" : "",
         "c" : "4.8088",
         "cs" : null,
         "etf" : false,
         "h" : "4.9360",
         "i" : "0.01000000",
         "l" : "4.3708",
         "o" : "4.4800",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "33000060.79307",
         "s" : "1INCHUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "7112085.7800",
         "y" : 0
      },
      {
         "an" : "Aave",
         "as" : 4409.13,
         "b" : "AAVE",
         "ba" : "",
         "c" : "0.7538",
         "cs" : null,
         "etf" : false,
         "h" : "0.7758",
         "i" : "0.01000000",
         "l" : "0.7289",
         "o" : "0.7414",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "3311.667706",
         "s" : "AAVEBNB",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "4409.1300",
         "y" : 0
      },
      {
         "an" : "Aave",
         "as" : 47730.35,
         "b" : "AAVE",
         "ba" : "",
         "c" : "0.007873",
         "cs" : null,
         "etf" : false,
         "h" : "0.007900",
         "i" : "0.01000000",
         "l" : "0.007299",
         "o" : "0.007430",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "360.23001939",
         "s" : "AAVEBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.000001",
         "v" : "47730.350000",
         "y" : 0
      },
      {
         "an" : "Aave",
         "as" : 11530.0202,
         "b" : "AAVE",
         "ba" : "",
         "c" : "433.17",
         "cs" : null,
         "etf" : false,
         "h" : "434.62",
         "i" : "0.00010000",
         "l" : "382.26",
         "o" : "399.91",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "4695711.1055856",
         "s" : "AAVEBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.01",
         "v" : "11530.03",
         "y" : 0
      },
      {
         "an" : "AAVEDOWN",
         "as" : 5060520.14,
         "b" : "AAVEDOWN",
         "ba" : "",
         "c" : "0.5436",
         "cs" : null,
         "etf" : true,
         "h" : "0.7300",
         "i" : "0.01000000",
         "l" : "0.5370",
         "o" : "0.6715",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "3289872.615319",
         "s" : "AAVEDOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "5060520.1400",
         "y" : 0
      },
      {
         "an" : "Aave",
         "as" : 6918.716,
         "b" : "AAVE",
         "ba" : "",
         "c" : "0.16909",
         "cs" : null,
         "etf" : false,
         "h" : "0.16998",
         "i" : "0.00100000",
         "l" : "0.15708",
         "o" : "0.15984",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "1117.20703561",
         "s" : "AAVEETH",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00001",
         "v" : "6918.71600",
         "y" : 0
      },
      {
         "an" : "AAVEUP",
         "as" : 466876.62,
         "b" : "AAVEUP",
         "ba" : "",
         "c" : "20.440",
         "cs" : null,
         "etf" : true,
         "h" : "20.621",
         "i" : "0.01000000",
         "l" : "16.050",
         "o" : "17.108",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "8199855.72837",
         "s" : "AAVEUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "466876.620",
         "y" : 0
      },
      {
         "an" : "Aave",
         "as" : 259953.5379,
         "b" : "AAVE",
         "ba" : "",
         "c" : "432.40",
         "cs" : null,
         "etf" : false,
         "h" : "434.78",
         "i" : "0.00010000",
         "l" : "384.57",
         "o" : "399.49",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "106000723.351585",
         "s" : "AAVEUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.01",
         "v" : "259953.54",
         "y" : 0
      },
      {
         "an" : "AC Milan Fan Token",
         "as" : 60209.45,
         "b" : "ACM",
         "ba" : "",
         "c" : "0.0001880",
         "cs" : null,
         "etf" : false,
         "h" : "0.0001928",
         "i" : "0.01000000",
         "l" : "0.0001834",
         "o" : "0.0001891",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "11.32468784",
         "s" : "ACMBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "60209.4500000",
         "y" : 0
      },
      {
         "an" : "AC Milan Fan Token",
         "as" : 13906.186,
         "b" : "ACM",
         "ba" : "",
         "c" : "10.333",
         "cs" : null,
         "etf" : false,
         "h" : "10.473",
         "i" : "0.00100000",
         "l" : "9.741",
         "o" : "10.243",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "141312.794228",
         "s" : "ACMBUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "13906.186",
         "y" : 0
      },
      {
         "an" : "AC Milan Fan Token",
         "as" : 208032.483,
         "b" : "ACM",
         "ba" : "",
         "c" : "10.339",
         "cs" : null,
         "etf" : false,
         "h" : "10.460",
         "i" : "0.00100000",
         "l" : "9.744",
         "o" : "10.189",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "2111053.657973",
         "s" : "ACMUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "208032.483",
         "y" : 0
      },
      {
         "an" : "Cardano",
         "as" : 2211130.756,
         "b" : "ADA",
         "ba" : "",
         "c" : "1.694",
         "cs" : 31948309440,
         "etf" : false,
         "h" : "1.735",
         "i" : "0.00100000",
         "l" : "1.537",
         "o" : "1.591",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "AUD",
         "qa" : "",
         "qn" : "Australian Dollar",
         "qv" : "3625899.573774",
         "s" : "ADAAUD",
         "st" : "TRADING",
         "tags" : [
            "mining-zone",
            "pos"
         ],
         "ts" : "0.001",
         "v" : "2211130.756",
         "y" : 0
      },
      {
         "an" : "Cardano",
         "as" : 11025854,
         "b" : "ADA",
         "ba" : "",
         "c" : "0.002285",
         "cs" : 31948309440,
         "etf" : false,
         "h" : "0.002380",
         "i" : "1.00000000",
         "l" : "0.002269",
         "o" : "0.002288",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "25586.484522",
         "s" : "ADABNB",
         "st" : "TRADING",
         "tags" : [
            "mining-zone",
            "pos"
         ],
         "ts" : "0.000001",
         "v" : "11025854.000000",
         "y" : 0
      },
      {
         "an" : "Cardano",
         "as" : 1101853.14,
         "b" : "ADA",
         "ba" : "",
         "c" : "7.158",
         "cs" : 31948309440,
         "etf" : false,
         "h" : "7.300",
         "i" : "0.01000000",
         "l" : "6.545",
         "o" : "6.810",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BRL",
         "qa" : "",
         "qn" : "Brazilian Real",
         "qv" : "7588142.95598",
         "s" : "ADABRL",
         "st" : "TRADING",
         "tags" : [
            "mining-zone",
            "pos"
         ],
         "ts" : "0.001",
         "v" : "1101853.140",
         "y" : 0
      },
      {
         "an" : "Cardano",
         "as" : 61290306,
         "b" : "ADA",
         "ba" : "",
         "c" : "0.00002384",
         "cs" : 31948309440,
         "etf" : false,
         "h" : "0.00002425",
         "i" : "1.00000000",
         "l" : "0.00002261",
         "o" : "0.00002291",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "1427.82431548",
         "s" : "ADABTC",
         "st" : "TRADING",
         "tags" : [
            "mining-zone",
            "pos"
         ],
         "ts" : "0.00000001",
         "v" : "61290306.00000000",
         "y" : 0
      },
      {
         "an" : "Cardano",
         "as" : 54048006.07,
         "b" : "ADA",
         "ba" : "",
         "c" : "1.3109",
         "cs" : 31948309440,
         "etf" : false,
         "h" : "1.3349",
         "i" : "0.01000000",
         "l" : "1.1945",
         "o" : "1.2341",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "67565255.327605",
         "s" : "ADABUSD",
         "st" : "TRADING",
         "tags" : [
            "mining-zone",
            "pos"
         ],
         "ts" : "0.0001",
         "v" : "54048006.0700",
         "y" : 0
      },
      {
         "an" : "ADADOWN",
         "as" : 1255082.93,
         "b" : "ADADOWN",
         "ba" : null,
         "c" : "1.6601",
         "cs" : null,
         "etf" : true,
         "h" : "2.0646",
         "i" : "0.01000000",
         "l" : "1.5815",
         "o" : "1.9263",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "2329876.45501",
         "s" : "ADADOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "1255082.9300",
         "y" : 0
      },
      {
         "an" : "Cardano",
         "as" : 5936430,
         "b" : "ADA",
         "ba" : "",
         "c" : "0.00051197",
         "cs" : 31948309440,
         "etf" : false,
         "h" : "0.00051812",
         "i" : "1.00000000",
         "l" : "0.00048285",
         "o" : "0.00049245",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "2964.55285448",
         "s" : "ADAETH",
         "st" : "TRADING",
         "tags" : [
            "mining-zone",
            "pos"
         ],
         "ts" : "0.00000001",
         "v" : "5936430.00000000",
         "y" : 0
      },
      {
         "an" : "Cardano",
         "as" : 20597743.14,
         "b" : "ADA",
         "ba" : "",
         "c" : "1.0863",
         "cs" : 31948309440,
         "etf" : false,
         "h" : "1.1051",
         "i" : "0.01000000",
         "l" : "0.9900",
         "o" : "1.0224",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "21472463.431942",
         "s" : "ADAEUR",
         "st" : "TRADING",
         "tags" : [
            "mining-zone",
            "pos"
         ],
         "ts" : "0.0001",
         "v" : "20597743.1400",
         "y" : 0
      },
      {
         "an" : "Cardano",
         "as" : 6517702.39,
         "b" : "ADA",
         "ba" : "",
         "c" : "0.9441",
         "cs" : 31948309440,
         "etf" : false,
         "h" : "0.9622",
         "i" : "0.01000000",
         "l" : "0.8608",
         "o" : "0.8910",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "GBP",
         "qa" : "",
         "qn" : "Pound Sterling",
         "qv" : "5938499.658406",
         "s" : "ADAGBP",
         "st" : "TRADING",
         "tags" : [
            "mining-zone",
            "pos"
         ],
         "ts" : "0.0001",
         "v" : "6517702.3900",
         "y" : 0
      },
      {
         "an" : "Cardano",
         "as" : 262481.64,
         "b" : "ADA",
         "ba" : "",
         "c" : "98.88",
         "cs" : 31948309440,
         "etf" : false,
         "h" : "100.55",
         "i" : "0.01000000",
         "l" : "90.32",
         "o" : "93.01",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "RUB",
         "qa" : "",
         "qn" : "Russian Ruble",
         "qv" : "24735492.1115",
         "s" : "ADARUB",
         "st" : "TRADING",
         "tags" : [
            "mining-zone",
            "pos"
         ],
         "ts" : "0.01",
         "v" : "262481.64",
         "y" : 0
      },
      {
         "an" : "Cardano",
         "as" : 3982609.34,
         "b" : "ADA",
         "ba" : "",
         "c" : "10.662",
         "cs" : 31948309440,
         "etf" : false,
         "h" : "10.865",
         "i" : "0.01000000",
         "l" : "9.721",
         "o" : "10.051",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "40637722.43673",
         "s" : "ADATRY",
         "st" : "TRADING",
         "tags" : [
            "mining-zone",
            "pos"
         ],
         "ts" : "0.001",
         "v" : "3982609.340",
         "y" : 0
      },
      {
         "an" : "Cardano",
         "as" : 552034.6,
         "b" : "ADA",
         "ba" : "",
         "c" : "1.2988",
         "cs" : 31948309440,
         "etf" : false,
         "h" : "1.3294",
         "i" : "0.01000000",
         "l" : "1.1929",
         "o" : "1.2364",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TUSD",
         "qa" : "",
         "qn" : "TrueUSD",
         "qv" : "691079.160473",
         "s" : "ADATUSD",
         "st" : "TRADING",
         "tags" : [
            "mining-zone",
            "pos"
         ],
         "ts" : "0.0001",
         "v" : "552034.6000",
         "y" : 0
      },
      {
         "an" : "ADAUP",
         "as" : 152295.21,
         "b" : "ADAUP",
         "ba" : null,
         "c" : "79.926",
         "cs" : null,
         "etf" : true,
         "h" : "82.956",
         "i" : "0.01000000",
         "l" : "65.537",
         "o" : "70.355",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "11053450.88898",
         "s" : "ADAUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "152295.210",
         "y" : 0
      },
      {
         "an" : "Cardano",
         "as" : 2449307.06,
         "b" : "ADA",
         "ba" : "",
         "c" : "1.3119",
         "cs" : 31948309440,
         "etf" : false,
         "h" : "1.3350",
         "i" : "0.01000000",
         "l" : "1.1940",
         "o" : "1.2328",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDC",
         "qa" : "",
         "qn" : "USD Coin",
         "qv" : "3076952.988693",
         "s" : "ADAUSDC",
         "st" : "TRADING",
         "tags" : [
            "mining-zone",
            "pos"
         ],
         "ts" : "0.0001",
         "v" : "2449307.0600",
         "y" : 0
      },
      {
         "an" : "Cardano",
         "as" : 395474691.14,
         "b" : "ADA",
         "ba" : "",
         "c" : "1.3093",
         "cs" : 31948309440,
         "etf" : false,
         "h" : "1.3333",
         "i" : "0.01000000",
         "l" : "1.1930",
         "o" : "1.2320",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "495190501.540168",
         "s" : "ADAUSDT",
         "st" : "TRADING",
         "tags" : [
            "mining-zone",
            "pos"
         ],
         "ts" : "0.0001",
         "v" : "395474691.1400",
         "y" : 0
      },
      {
         "an" : "AdEx",
         "as" : 2782598,
         "b" : "ADX",
         "ba" : "",
         "c" : "0.00002500",
         "cs" : 73430701,
         "etf" : false,
         "h" : "0.00002558",
         "i" : "1.00000000",
         "l" : "0.00002295",
         "o" : "0.00002326",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "67.97490999",
         "s" : "ADXBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "2782598.00000000",
         "y" : 0
      },
      {
         "an" : "AdEx",
         "as" : 214126,
         "b" : "ADX",
         "ba" : "",
         "c" : "0.0005367",
         "cs" : 73430701,
         "etf" : false,
         "h" : "0.0005468",
         "i" : "1.00000000",
         "l" : "0.0004950",
         "o" : "0.0004991",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "111.759519",
         "s" : "ADXETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "214126.0000000",
         "y" : 0
      },
      {
         "an" : "Aergo",
         "as" : 4233387,
         "b" : "AERGO",
         "ba" : "",
         "c" : "0.00000665",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000675",
         "i" : "1.00000000",
         "l" : "0.00000631",
         "o" : "0.00000642",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "27.91022016",
         "s" : "AERGOBTC",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00000001",
         "v" : "4233387.00000000",
         "y" : 0
      },
      {
         "an" : "Aergo",
         "as" : 3674320.8,
         "b" : "AERGO",
         "ba" : "",
         "c" : "0.36821",
         "cs" : null,
         "etf" : false,
         "h" : "0.37314",
         "i" : "0.10000000",
         "l" : "0.33726",
         "o" : "0.34863",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1319054.646766",
         "s" : "AERGOBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00001",
         "v" : "3674320.80000",
         "y" : 0
      },
      {
         "an" : "SingularityNET",
         "as" : 7468976,
         "b" : "AGI",
         "ba" : "",
         "c" : "0.00000740",
         "cs" : 867679580,
         "etf" : false,
         "h" : "0.00000780",
         "i" : "1.00000000",
         "l" : "0.00000689",
         "o" : "0.00000723",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "54.5941586",
         "s" : "AGIBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "7468976.00000000",
         "y" : 0
      },
      {
         "an" : "AION",
         "as" : 12166836,
         "b" : "AION",
         "ba" : "",
         "c" : "0.00000606",
         "cs" : 318443045,
         "etf" : false,
         "h" : "0.00000643",
         "i" : "1.00000000",
         "l" : "0.00000521",
         "o" : "0.00000536",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "69.69981353",
         "s" : "AIONBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "12166836.00000000",
         "y" : 0
      },
      {
         "an" : "AION",
         "as" : 2168117.13,
         "b" : "AION",
         "ba" : "",
         "c" : "0.000129",
         "cs" : 318443045,
         "etf" : false,
         "h" : "0.000137",
         "i" : "0.01000000",
         "l" : "0.000112",
         "o" : "0.000116",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "264.67568697",
         "s" : "AIONETH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "2168117.130000",
         "y" : 0
      },
      {
         "an" : "AION",
         "as" : 35536697.91,
         "b" : "AION",
         "ba" : "",
         "c" : "0.3320",
         "cs" : 318443045,
         "etf" : false,
         "h" : "0.3538",
         "i" : "0.01000000",
         "l" : "0.2752",
         "o" : "0.2882",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "10977338.653927",
         "s" : "AIONUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "35536697.9100",
         "y" : 0
      },
      {
         "an" : "Akropolis",
         "as" : 119191173,
         "b" : "AKRO",
         "ba" : "",
         "c" : "0.00000092",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000096",
         "i" : "1.00000000",
         "l" : "0.00000081",
         "o" : "0.00000081",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "105.5923562",
         "s" : "AKROBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "Polkadot"
         ],
         "ts" : "0.00000001",
         "v" : "119191173.00000000",
         "y" : 0
      },
      {
         "an" : "Akropolis",
         "as" : 340353270,
         "b" : "AKRO",
         "ba" : "",
         "c" : "0.050375",
         "cs" : null,
         "etf" : false,
         "h" : "0.052244",
         "i" : "1.00000000",
         "l" : "0.043260",
         "o" : "0.043914",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "16202782.769094",
         "s" : "AKROUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "Polkadot"
         ],
         "ts" : "0.000001",
         "v" : "340353270.000000",
         "y" : 0
      },
      {
         "an" : "Algorand",
         "as" : 578023.7,
         "b" : "ALGO",
         "ba" : "",
         "c" : "0.002213",
         "cs" : 2643333333,
         "etf" : false,
         "h" : "0.002396",
         "i" : "0.10000000",
         "l" : "0.002213",
         "o" : "0.002281",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1332.8959623",
         "s" : "ALGOBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "578023.700000",
         "y" : 0
      },
      {
         "an" : "Algorand",
         "as" : 6668951,
         "b" : "ALGO",
         "ba" : "",
         "c" : "0.00002306",
         "cs" : 2643333333,
         "etf" : false,
         "h" : "0.00002375",
         "i" : "1.00000000",
         "l" : "0.00002232",
         "o" : "0.00002287",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "153.84635832",
         "s" : "ALGOBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "6668951.00000000",
         "y" : 0
      },
      {
         "an" : "Algorand",
         "as" : 3254369.6,
         "b" : "ALGO",
         "ba" : "",
         "c" : "1.2680",
         "cs" : 2643333333,
         "etf" : false,
         "h" : "1.2900",
         "i" : "0.01000000",
         "l" : "1.1821",
         "o" : "1.2297",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "4043751.422066",
         "s" : "ALGOBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "3254369.6000",
         "y" : 0
      },
      {
         "an" : "Algorand",
         "as" : 139694.59,
         "b" : "ALGO",
         "ba" : "",
         "c" : "1.2699",
         "cs" : 2643333333,
         "etf" : false,
         "h" : "1.2889",
         "i" : "0.01000000",
         "l" : "1.1832",
         "o" : "1.2373",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TUSD",
         "qa" : "",
         "qn" : "TrueUSD",
         "qv" : "172196.408831",
         "s" : "ALGOTUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "139694.5900",
         "y" : 0
      },
      {
         "an" : "Algorand",
         "as" : 35602838.54,
         "b" : "ALGO",
         "ba" : "",
         "c" : "1.2678",
         "cs" : 2643333333,
         "etf" : false,
         "h" : "1.2890",
         "i" : "0.01000000",
         "l" : "1.1790",
         "o" : "1.2291",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "44194265.608681",
         "s" : "ALGOUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "35602838.5400",
         "y" : 0
      },
      {
         "an" : "My Neighbor Alice",
         "as" : 595876.7,
         "b" : "ALICE",
         "ba" : "",
         "c" : "0.00017971",
         "cs" : null,
         "etf" : false,
         "h" : "0.00019031",
         "i" : "0.10000000",
         "l" : "0.00016000",
         "o" : "0.00016995",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "102.10804367",
         "s" : "ALICEBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "NFT"
         ],
         "ts" : "0.00000001",
         "v" : "595876.70000000",
         "y" : 0
      },
      {
         "an" : "My Neighbor Alice",
         "as" : 480559.52,
         "b" : "ALICE",
         "ba" : "",
         "c" : "9.8819",
         "cs" : null,
         "etf" : false,
         "h" : "10.4520",
         "i" : "0.01000000",
         "l" : "8.4503",
         "o" : "9.1514",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "4463739.340276",
         "s" : "ALICEBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "NFT"
         ],
         "ts" : "0.0001",
         "v" : "480559.5200",
         "y" : 0
      },
      {
         "an" : "My Neighbor Alice",
         "as" : 4813097.53,
         "b" : "ALICE",
         "ba" : "",
         "c" : "9.8656",
         "cs" : null,
         "etf" : false,
         "h" : "10.4500",
         "i" : "0.01000000",
         "l" : "8.4431",
         "o" : "9.1273",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "44609614.139865",
         "s" : "ALICEUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "NFT"
         ],
         "ts" : "0.0001",
         "v" : "4813097.5300",
         "y" : 0
      },
      {
         "an" : "Alpha Finance Lab",
         "as" : 505176,
         "b" : "ALPHA",
         "ba" : "",
         "c" : "0.0025127",
         "cs" : null,
         "etf" : false,
         "h" : "0.0028301",
         "i" : "1.00000000",
         "l" : "0.0024384",
         "o" : "0.0025654",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1304.5997582",
         "s" : "ALPHABNB",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0000001",
         "v" : "505176.0000000",
         "y" : 0
      },
      {
         "an" : "Alpha Finance Lab",
         "as" : 2238664,
         "b" : "ALPHA",
         "ba" : "",
         "c" : "0.00002627",
         "cs" : null,
         "etf" : false,
         "h" : "0.00002715",
         "i" : "1.00000000",
         "l" : "0.00002414",
         "o" : "0.00002572",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "57.8702098",
         "s" : "ALPHABTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "2238664.00000000",
         "y" : 0
      },
      {
         "an" : "Alpha Finance Lab",
         "as" : 1202242.52,
         "b" : "ALPHA",
         "ba" : "",
         "c" : "1.4461",
         "cs" : null,
         "etf" : false,
         "h" : "1.4800",
         "i" : "0.01000000",
         "l" : "1.2730",
         "o" : "1.3841",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1669116.853372",
         "s" : "ALPHABUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "1202242.5200",
         "y" : 0
      },
      {
         "an" : "Alpha Finance Lab",
         "as" : 20276266.6,
         "b" : "ALPHA",
         "ba" : "",
         "c" : "1.4438",
         "cs" : null,
         "etf" : false,
         "h" : "1.4793",
         "i" : "0.01000000",
         "l" : "1.2724",
         "o" : "1.3824",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "28070480.50039",
         "s" : "ALPHAUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "20276266.6000",
         "y" : 0
      },
      {
         "an" : "Ambrosus",
         "as" : 25933346,
         "b" : "AMB",
         "ba" : "",
         "c" : "0.00000138",
         "cs" : 175649450,
         "etf" : false,
         "h" : "0.00000140",
         "i" : "1.00000000",
         "l" : "0.00000129",
         "o" : "0.00000136",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "34.90326501",
         "s" : "AMBBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "25933346.00000000",
         "y" : 0
      },
      {
         "an" : "Ankr",
         "as" : 5131244,
         "b" : "ANKR",
         "ba" : "",
         "c" : "0.0002663",
         "cs" : 2629035843,
         "etf" : false,
         "h" : "0.0002831",
         "i" : "1.00000000",
         "l" : "0.0002582",
         "o" : "0.0002627",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1397.4415829",
         "s" : "ANKRBNB",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone",
            "pos"
         ],
         "ts" : "0.0000001",
         "v" : "5131244.0000000",
         "y" : 0
      },
      {
         "an" : "Ankr",
         "as" : 29229500,
         "b" : "ANKR",
         "ba" : "",
         "c" : "0.00000277",
         "cs" : 2629035843,
         "etf" : false,
         "h" : "0.00000283",
         "i" : "1.00000000",
         "l" : "0.00000257",
         "o" : "0.00000262",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "79.09121307",
         "s" : "ANKRBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone",
            "pos"
         ],
         "ts" : "0.00000001",
         "v" : "29229500.00000000",
         "y" : 0
      },
      {
         "an" : "Ankr",
         "as" : 199635680.2,
         "b" : "ANKR",
         "ba" : "",
         "c" : "0.15187",
         "cs" : 2629035843,
         "etf" : false,
         "h" : "0.15489",
         "i" : "0.10000000",
         "l" : "0.13720",
         "o" : "0.14101",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "29020927.9963",
         "s" : "ANKRUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone",
            "pos"
         ],
         "ts" : "0.00001",
         "v" : "199635680.20000",
         "y" : 0
      },
      {
         "an" : "Aragon",
         "as" : 16907.52,
         "b" : "ANT",
         "ba" : "",
         "c" : "0.01569",
         "cs" : 39609522,
         "etf" : false,
         "h" : "0.01767",
         "i" : "0.01000000",
         "l" : "0.01531",
         "o" : "0.01565",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "276.1457065",
         "s" : "ANTBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "16907.52000",
         "y" : 0
      },
      {
         "an" : "Aragon",
         "as" : 89854.25,
         "b" : "ANT",
         "ba" : "",
         "c" : "8.9871",
         "cs" : 39609522,
         "etf" : false,
         "h" : "9.2299",
         "i" : "0.01000000",
         "l" : "8.1995",
         "o" : "8.3830",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "778537.290802",
         "s" : "ANTBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "89854.2500",
         "y" : 0
      },
      {
         "an" : "Aragon",
         "as" : 944248.3,
         "b" : "ANT",
         "ba" : "",
         "c" : "8.9825",
         "cs" : 39609522,
         "etf" : false,
         "h" : "9.2220",
         "i" : "0.01000000",
         "l" : "8.1742",
         "o" : "8.3824",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "8156491.666472",
         "s" : "ANTUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "944248.3000",
         "y" : 0
      },
      {
         "an" : "AppCoins",
         "as" : 8052759,
         "b" : "APPC",
         "ba" : "",
         "c" : "0.00000407",
         "cs" : 245321973,
         "etf" : false,
         "h" : "0.00000423",
         "i" : "1.00000000",
         "l" : "0.00000392",
         "o" : "0.00000415",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "32.77876819",
         "s" : "APPCBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "8052759.00000000",
         "y" : 0
      },
      {
         "an" : "Ardor",
         "as" : 6017948,
         "b" : "ARDR",
         "ba" : "",
         "c" : "0.00000680",
         "cs" : 998999495,
         "etf" : false,
         "h" : "0.00000704",
         "i" : "1.00000000",
         "l" : "0.00000625",
         "o" : "0.00000636",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "40.1430418",
         "s" : "ARDRBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "6017948.00000000",
         "y" : 0
      },
      {
         "an" : "Ardor",
         "as" : 9469128.9,
         "b" : "ARDR",
         "ba" : "",
         "c" : "0.37290",
         "cs" : 998999495,
         "etf" : false,
         "h" : "0.38241",
         "i" : "0.10000000",
         "l" : "0.33451",
         "o" : "0.34277",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "3433737.727693",
         "s" : "ARDRUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "9469128.90000",
         "y" : 0
      },
      {
         "an" : "Ark",
         "as" : 3614720,
         "b" : "ARK",
         "ba" : "",
         "c" : "0.00004013",
         "cs" : 149502282,
         "etf" : false,
         "h" : "0.00004267",
         "i" : "1.00000000",
         "l" : "0.00003205",
         "o" : "0.00003244",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "139.44840624",
         "s" : "ARKBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "3614720.00000000",
         "y" : 0
      },
      {
         "an" : "ARPA Chain",
         "as" : 5104520,
         "b" : "ARPA",
         "ba" : "",
         "c" : "0.0001655",
         "cs" : 684000000,
         "etf" : false,
         "h" : "0.0001870",
         "i" : "1.00000000",
         "l" : "0.0001628",
         "o" : "0.0001668",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "871.1166258",
         "s" : "ARPABNB",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "5104520.0000000",
         "y" : 0
      },
      {
         "an" : "ARPA Chain",
         "as" : 50460007,
         "b" : "ARPA",
         "ba" : "",
         "c" : "0.00000173",
         "cs" : 684000000,
         "etf" : false,
         "h" : "0.00000184",
         "i" : "1.00000000",
         "l" : "0.00000161",
         "o" : "0.00000166",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "86.33649033",
         "s" : "ARPABTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "50460007.00000000",
         "y" : 0
      },
      {
         "an" : "ARPA Chain",
         "as" : 126695111,
         "b" : "ARPA",
         "ba" : "",
         "c" : "0.09479",
         "cs" : 684000000,
         "etf" : false,
         "h" : "0.09889",
         "i" : "0.10000000",
         "l" : "0.08510",
         "o" : "0.08953",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "11642611.380675",
         "s" : "ARPAUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "126695111.00000",
         "y" : 0
      },
      {
         "an" : "AS Roma Fan Token",
         "as" : 52844.36,
         "b" : "ASR",
         "ba" : "",
         "c" : "0.0001716",
         "cs" : null,
         "etf" : false,
         "h" : "0.0001786",
         "i" : "0.01000000",
         "l" : "0.0001655",
         "o" : "0.0001668",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "9.12093886",
         "s" : "ASRBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.0000001",
         "v" : "52844.3600000",
         "y" : 0
      },
      {
         "an" : "AS Roma Fan Token",
         "as" : 242281.471,
         "b" : "ASR",
         "ba" : "",
         "c" : "9.400",
         "cs" : null,
         "etf" : false,
         "h" : "9.771",
         "i" : "0.00100000",
         "l" : "8.915",
         "o" : "8.925",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "2258843.034964",
         "s" : "ASRUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "242281.471",
         "y" : 0
      },
      {
         "an" : "AirSwap",
         "as" : 5371885,
         "b" : "AST",
         "ba" : "",
         "c" : "0.00000707",
         "cs" : 0,
         "etf" : false,
         "h" : "0.00000721",
         "i" : "1.00000000",
         "l" : "0.00000601",
         "o" : "0.00000671",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "35.94483142",
         "s" : "ASTBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "5371885.00000000",
         "y" : 0
      },
      {
         "an" : "AtlÃ©tico de Madrid Fan Token",
         "as" : 70336.12,
         "b" : "ATM",
         "ba" : "",
         "c" : "0.0001752",
         "cs" : null,
         "etf" : false,
         "h" : "0.0001823",
         "i" : "0.01000000",
         "l" : "0.0001738",
         "o" : "0.0001788",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "12.50091803",
         "s" : "ATMBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.0000001",
         "v" : "70336.1200000",
         "y" : 0
      },
      {
         "an" : "AtlÃ©tico de Madrid Fan Token",
         "as" : 286944.076,
         "b" : "ATM",
         "ba" : "",
         "c" : "9.658",
         "cs" : null,
         "etf" : false,
         "h" : "9.850",
         "i" : "0.00100000",
         "l" : "9.347",
         "o" : "9.611",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "2749332.268622",
         "s" : "ATMUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "286944.076",
         "y" : 0
      },
      {
         "an" : "Cosmos",
         "as" : 121211.62,
         "b" : "ATOM",
         "ba" : "",
         "c" : "0.03947",
         "cs" : 190238328,
         "etf" : false,
         "h" : "0.04294",
         "i" : "0.01000000",
         "l" : "0.03825",
         "o" : "0.03880",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "4982.2785867",
         "s" : "ATOMBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "121211.62000",
         "y" : 0
      },
      {
         "an" : "Cosmos",
         "as" : 917183.01,
         "b" : "ATOM",
         "ba" : "",
         "c" : "0.0004122",
         "cs" : 190238328,
         "etf" : false,
         "h" : "0.0004270",
         "i" : "0.01000000",
         "l" : "0.0003810",
         "o" : "0.0003893",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "376.52332994",
         "s" : "ATOMBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "917183.0100000",
         "y" : 0
      },
      {
         "an" : "Cosmos",
         "as" : 233618.32,
         "b" : "ATOM",
         "ba" : "",
         "c" : "22.630",
         "cs" : 190238328,
         "etf" : false,
         "h" : "23.147",
         "i" : "0.00100000",
         "l" : "20.339",
         "o" : "20.990",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "5153630.0263",
         "s" : "ATOMBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "233618.320",
         "y" : 0
      },
      {
         "an" : "Cosmos",
         "as" : 12687.71,
         "b" : "ATOM",
         "ba" : "",
         "c" : "22.728",
         "cs" : 190238328,
         "etf" : false,
         "h" : "23.149",
         "i" : "0.00100000",
         "l" : "20.348",
         "o" : "21.040",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDC",
         "qa" : "",
         "qn" : "USD Coin",
         "qv" : "281339.736143",
         "s" : "ATOMUSDC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "12687.710",
         "y" : 0
      },
      {
         "an" : "Cosmos",
         "as" : 4073021.156,
         "b" : "ATOM",
         "ba" : "",
         "c" : "22.623",
         "cs" : 190238328,
         "etf" : false,
         "h" : "23.500",
         "i" : "0.00100000",
         "l" : "20.303",
         "o" : "20.947",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "89997073.782845",
         "s" : "ATOMUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "4073021.156",
         "y" : 0
      },
      {
         "an" : "Auction",
         "as" : 23139.43,
         "b" : "AUCTION",
         "ba" : "",
         "c" : "0.0007848",
         "cs" : null,
         "etf" : false,
         "h" : "0.0007980",
         "i" : "0.01000000",
         "l" : "0.0007300",
         "o" : "0.0007823",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "17.54188628",
         "s" : "AUCTIONBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "BSC"
         ],
         "ts" : "0.0000001",
         "v" : "23139.4300000",
         "y" : 0
      },
      {
         "an" : "Auction",
         "as" : 41084.679,
         "b" : "AUCTION",
         "ba" : "",
         "c" : "43.000",
         "cs" : null,
         "etf" : false,
         "h" : "44.000",
         "i" : "0.00100000",
         "l" : "39.002",
         "o" : "42.070",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1683911.770234",
         "s" : "AUCTIONBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "41084.679",
         "y" : 0
      },
      {
         "an" : "Australian Dollar",
         "as" : 8091273,
         "b" : "AUD",
         "ba" : "",
         "c" : "0.77390",
         "cs" : null,
         "etf" : false,
         "h" : "0.77798",
         "i" : "0.10000000",
         "l" : "0.77208",
         "o" : "0.77491",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "6268800.787865",
         "s" : "AUDBUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "8091273.00000",
         "y" : 0
      },
      {
         "an" : "Audius",
         "as" : 3969449,
         "b" : "AUDIO",
         "ba" : "",
         "c" : "0.00004185",
         "cs" : null,
         "etf" : false,
         "h" : "0.00004655",
         "i" : "1.00000000",
         "l" : "0.00003969",
         "o" : "0.00004372",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "168.29488173",
         "s" : "AUDIOBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "3969449.00000000",
         "y" : 0
      },
      {
         "an" : "Audius",
         "as" : 2263168.17,
         "b" : "AUDIO",
         "ba" : "",
         "c" : "2.3000",
         "cs" : null,
         "etf" : false,
         "h" : "2.4985",
         "i" : "0.01000000",
         "l" : "2.1035",
         "o" : "2.3514",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "5135277.835698",
         "s" : "AUDIOBUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "2263168.1700",
         "y" : 0
      },
      {
         "an" : "Audius",
         "as" : 16656147.58,
         "b" : "AUDIO",
         "ba" : "",
         "c" : "2.2999",
         "cs" : null,
         "etf" : false,
         "h" : "2.5000",
         "i" : "0.01000000",
         "l" : "2.0962",
         "o" : "2.3549",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "38265712.174135",
         "s" : "AUDIOUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "16656147.5800",
         "y" : 0
      },
      {
         "an" : "Australian Dollar",
         "as" : 30461422.6,
         "b" : "AUD",
         "ba" : "",
         "c" : "0.77278",
         "cs" : null,
         "etf" : false,
         "h" : "0.77740",
         "i" : "0.10000000",
         "l" : "0.77123",
         "o" : "0.77375",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "23573264.651666",
         "s" : "AUDUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "30461422.60000",
         "y" : 0
      },
      {
         "an" : "Auto",
         "as" : 368.5831,
         "b" : "AUTO",
         "ba" : "",
         "c" : "0.04549",
         "cs" : null,
         "etf" : false,
         "h" : "0.04678",
         "i" : "0.00010000",
         "l" : "0.03933",
         "o" : "0.04274",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "15.77733224",
         "s" : "AUTOBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC",
            "defi"
         ],
         "ts" : "0.00001",
         "v" : "368.58310",
         "y" : 0
      },
      {
         "an" : "Auto",
         "as" : 159.843775,
         "b" : "AUTO",
         "ba" : "",
         "c" : "2504.70",
         "cs" : null,
         "etf" : false,
         "h" : "2578.65",
         "i" : "0.00000100",
         "l" : "2106.20",
         "o" : "2285.00",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "371978.00392437",
         "s" : "AUTOBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC",
            "defi"
         ],
         "ts" : "0.01",
         "v" : "159.85",
         "y" : 0
      },
      {
         "an" : "Auto",
         "as" : 1786.216798,
         "b" : "AUTO",
         "ba" : "",
         "c" : "2515.02",
         "cs" : null,
         "etf" : false,
         "h" : "2559.99",
         "i" : "0.00000100",
         "l" : "2111.05",
         "o" : "2278.44",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "4120088.26861222",
         "s" : "AUTOUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC",
            "defi"
         ],
         "ts" : "0.01",
         "v" : "1786.22",
         "y" : 0
      },
      {
         "an" : "Travala.com",
         "as" : 162537.3,
         "b" : "AVA",
         "ba" : "",
         "c" : "0.00902",
         "cs" : null,
         "etf" : false,
         "h" : "0.00975",
         "i" : "0.10000000",
         "l" : "0.00852",
         "o" : "0.00925",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1477.863779",
         "s" : "AVABNB",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "162537.30000",
         "y" : 0
      },
      {
         "an" : "Travala.com",
         "as" : 606480.35,
         "b" : "AVA",
         "ba" : "",
         "c" : "0.0000939",
         "cs" : null,
         "etf" : false,
         "h" : "0.0000975",
         "i" : "0.01000000",
         "l" : "0.0000867",
         "o" : "0.0000926",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "55.16182203",
         "s" : "AVABTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "606480.3500000",
         "y" : 0
      },
      {
         "an" : "Travala.com",
         "as" : 188184.08,
         "b" : "AVA",
         "ba" : "",
         "c" : "5.1581",
         "cs" : null,
         "etf" : false,
         "h" : "5.2563",
         "i" : "0.01000000",
         "l" : "4.5912",
         "o" : "4.9980",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "921411.196294",
         "s" : "AVABUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "188184.0800",
         "y" : 0
      },
      {
         "an" : "Travala.com",
         "as" : 1949449.8,
         "b" : "AVA",
         "ba" : "",
         "c" : "5.1517",
         "cs" : null,
         "etf" : false,
         "h" : "5.2779",
         "i" : "0.01000000",
         "l" : "4.6244",
         "o" : "4.9863",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "9566854.561845",
         "s" : "AVAUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "1949449.8000",
         "y" : 0
      },
      {
         "an" : "Avalanche",
         "as" : 50551.9,
         "b" : "AVAX",
         "ba" : "",
         "c" : "0.04790",
         "cs" : null,
         "etf" : false,
         "h" : "0.05149",
         "i" : "0.10000000",
         "l" : "0.04693",
         "o" : "0.04715",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "2503.510596",
         "s" : "AVAXBNB",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00001",
         "v" : "50551.90000",
         "y" : 0
      },
      {
         "an" : "Avalanche",
         "as" : 451186.5,
         "b" : "AVAX",
         "ba" : "",
         "c" : "0.00049953",
         "cs" : null,
         "etf" : false,
         "h" : "0.00050921",
         "i" : "0.10000000",
         "l" : "0.00046572",
         "o" : "0.00047239",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "220.27697355",
         "s" : "AVAXBTC",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00000001",
         "v" : "451186.50000000",
         "y" : 0
      },
      {
         "an" : "Avalanche",
         "as" : 162510.805,
         "b" : "AVAX",
         "ba" : "",
         "c" : "27.486",
         "cs" : null,
         "etf" : false,
         "h" : "27.870",
         "i" : "0.00100000",
         "l" : "24.561",
         "o" : "25.457",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "4309911.9181802",
         "s" : "AVAXBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.001",
         "v" : "162510.805",
         "y" : 0
      },
      {
         "an" : "Avalanche",
         "as" : 42991.891,
         "b" : "AVAX",
         "ba" : "",
         "c" : "22.780",
         "cs" : null,
         "etf" : false,
         "h" : "23.151",
         "i" : "0.00100000",
         "l" : "20.400",
         "o" : "21.083",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "946485.670293",
         "s" : "AVAXEUR",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.001",
         "v" : "42991.891",
         "y" : 0
      },
      {
         "an" : "Avalanche",
         "as" : 144270.81,
         "b" : "AVAX",
         "ba" : "",
         "c" : "223.3",
         "cs" : null,
         "etf" : false,
         "h" : "227.0",
         "i" : "0.01000000",
         "l" : "200.3",
         "o" : "207.4",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "31061988.167",
         "s" : "AVAXTRY",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.1",
         "v" : "144270.9",
         "y" : 0
      },
      {
         "an" : "Avalanche",
         "as" : 2854173.24,
         "b" : "AVAX",
         "ba" : "",
         "c" : "27.439",
         "cs" : null,
         "etf" : false,
         "h" : "27.850",
         "i" : "0.00100000",
         "l" : "24.558",
         "o" : "25.419",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "74940970.161714",
         "s" : "AVAXUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.001",
         "v" : "2854173.240",
         "y" : 0
      },
      {
         "an" : "Axie Infinity",
         "as" : 126973,
         "b" : "AXS",
         "ba" : "",
         "c" : "0.014763",
         "cs" : null,
         "etf" : false,
         "h" : "0.016750",
         "i" : "1.00000000",
         "l" : "0.013278",
         "o" : "0.013392",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1918.31435",
         "s" : "AXSBNB",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.000001",
         "v" : "126973.000000",
         "y" : 0
      },
      {
         "an" : "Axie Infinity",
         "as" : 727916,
         "b" : "AXS",
         "ba" : "",
         "c" : "0.00015389",
         "cs" : null,
         "etf" : false,
         "h" : "0.00016759",
         "i" : "1.00000000",
         "l" : "0.00013202",
         "o" : "0.00013420",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "109.88230333",
         "s" : "AXSBTC",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.00000001",
         "v" : "727916.00000000",
         "y" : 0
      },
      {
         "an" : "Axie Infinity",
         "as" : 352872.79,
         "b" : "AXS",
         "ba" : "",
         "c" : "8.4581",
         "cs" : null,
         "etf" : false,
         "h" : "9.1500",
         "i" : "0.01000000",
         "l" : "7.0269",
         "o" : "7.2204",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2808233.194036",
         "s" : "AXSBUSD",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.0001",
         "v" : "352872.7900",
         "y" : 0
      },
      {
         "an" : "Axie Infinity",
         "as" : 3985860.01,
         "b" : "AXS",
         "ba" : "",
         "c" : "8.4461",
         "cs" : null,
         "etf" : false,
         "h" : "9.1439",
         "i" : "0.01000000",
         "l" : "7.0000",
         "o" : "7.2019",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "32241156.072341",
         "s" : "AXSUSDT",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.0001",
         "v" : "3985860.0100",
         "y" : 0
      },
      {
         "an" : "Badger DAO",
         "as" : 103175.62,
         "b" : "BADGER",
         "ba" : "",
         "c" : "0.0005777",
         "cs" : null,
         "etf" : false,
         "h" : "0.0005861",
         "i" : "0.01000000",
         "l" : "0.0005173",
         "o" : "0.0005350",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "57.21391869",
         "s" : "BADGERBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0000001",
         "v" : "103175.6200000",
         "y" : 0
      },
      {
         "an" : "Badger DAO",
         "as" : 21527.179,
         "b" : "BADGER",
         "ba" : "",
         "c" : "31.582",
         "cs" : null,
         "etf" : false,
         "h" : "31.965",
         "i" : "0.00100000",
         "l" : "27.980",
         "o" : "28.768",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "644761.217522",
         "s" : "BADGERBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.001",
         "v" : "21527.179",
         "y" : 0
      },
      {
         "an" : "Badger DAO",
         "as" : 239652.106,
         "b" : "BADGER",
         "ba" : "",
         "c" : "31.721",
         "cs" : null,
         "etf" : false,
         "h" : "31.955",
         "i" : "0.00100000",
         "l" : "27.923",
         "o" : "28.729",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "7176476.121874",
         "s" : "BADGERUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.001",
         "v" : "239652.106",
         "y" : 0
      },
      {
         "an" : "BakeryToken",
         "as" : 2195189,
         "b" : "BAKE",
         "ba" : "",
         "c" : "0.002732",
         "cs" : null,
         "etf" : false,
         "h" : "0.002799",
         "i" : "1.00000000",
         "l" : "0.002417",
         "o" : "0.002642",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "5668.583283",
         "s" : "BAKEBNB",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "BSC"
         ],
         "ts" : "0.000001",
         "v" : "2195189.000000",
         "y" : 0
      },
      {
         "an" : "BakeryToken",
         "as" : 11823867.1,
         "b" : "BAKE",
         "ba" : "",
         "c" : "1.5688",
         "cs" : null,
         "etf" : false,
         "h" : "1.6000",
         "i" : "0.01000000",
         "l" : "1.3233",
         "o" : "1.4212",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "16767606.68656",
         "s" : "BAKEBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "BSC"
         ],
         "ts" : "0.0001",
         "v" : "11823867.1000",
         "y" : 0
      },
      {
         "an" : "Balancer",
         "as" : 5811.974,
         "b" : "BAL",
         "ba" : "",
         "c" : "55.156",
         "cs" : null,
         "etf" : false,
         "h" : "56.235",
         "i" : "0.00100000",
         "l" : "50.913",
         "o" : "52.497",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "310360.719227",
         "s" : "BALBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "5811.974",
         "y" : 0
      },
      {
         "an" : "Balancer",
         "as" : 129913.32,
         "b" : "BAL",
         "ba" : "",
         "c" : "55.035",
         "cs" : null,
         "etf" : false,
         "h" : "56.125",
         "i" : "0.00100000",
         "l" : "50.831",
         "o" : "52.527",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "6961734.31464",
         "s" : "BALUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "129913.320",
         "y" : 0
      },
      {
         "an" : "BAND",
         "as" : 62069.5,
         "b" : "BAND",
         "ba" : "",
         "c" : "0.02860",
         "cs" : 17083333,
         "etf" : false,
         "h" : "0.03132",
         "i" : "0.10000000",
         "l" : "0.02749",
         "o" : "0.02816",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1821.923842",
         "s" : "BANDBNB",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos"
         ],
         "ts" : "0.00001",
         "v" : "62069.50000",
         "y" : 0
      },
      {
         "an" : "BAND",
         "as" : 486694.3,
         "b" : "BAND",
         "ba" : "",
         "c" : "0.00029864",
         "cs" : 17083333,
         "etf" : false,
         "h" : "0.00031100",
         "i" : "0.10000000",
         "l" : "0.00027209",
         "o" : "0.00028256",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "142.56068237",
         "s" : "BANDBTC",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos"
         ],
         "ts" : "0.00000001",
         "v" : "486694.30000000",
         "y" : 0
      },
      {
         "an" : "BAND",
         "as" : 114712.517,
         "b" : "BAND",
         "ba" : "",
         "c" : "16.472",
         "cs" : 17083333,
         "etf" : false,
         "h" : "17.040",
         "i" : "0.00100000",
         "l" : "14.344",
         "o" : "15.255",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1811294.53488",
         "s" : "BANDBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos"
         ],
         "ts" : "0.001",
         "v" : "114712.517",
         "y" : 0
      },
      {
         "an" : "BAND",
         "as" : 2426472.862,
         "b" : "BAND",
         "ba" : "",
         "c" : "16.423",
         "cs" : 17083333,
         "etf" : false,
         "h" : "17.039",
         "i" : "0.00100000",
         "l" : "14.315",
         "o" : "15.196",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "38264549.178966",
         "s" : "BANDUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos"
         ],
         "ts" : "0.001",
         "v" : "2426472.862",
         "y" : 0
      },
      {
         "an" : "FC Barcelona Fan Token BAR",
         "as" : 31920.04,
         "b" : "BAR",
         "ba" : "",
         "c" : "0.0007659",
         "cs" : null,
         "etf" : false,
         "h" : "0.0008049",
         "i" : "0.01000000",
         "l" : "0.0007400",
         "o" : "0.0007675",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "24.70056896",
         "s" : "BARBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone"
         ],
         "ts" : "0.0000001",
         "v" : "31920.0400000",
         "y" : 0
      },
      {
         "an" : "FC Barcelona Fan Token BAR",
         "as" : 40882.305,
         "b" : "BAR",
         "ba" : "",
         "c" : "42.230",
         "cs" : null,
         "etf" : false,
         "h" : "44.500",
         "i" : "0.00100000",
         "l" : "39.331",
         "o" : "41.312",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1708022.417295",
         "s" : "BARBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone"
         ],
         "ts" : "0.001",
         "v" : "40882.305",
         "y" : 0
      },
      {
         "an" : "Basic Attention Token",
         "as" : 388368.7,
         "b" : "BAT",
         "ba" : "",
         "c" : "0.002150",
         "cs" : 1492816440,
         "etf" : false,
         "h" : "0.002253",
         "i" : "0.10000000",
         "l" : "0.002100",
         "o" : "0.002121",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "845.8423543",
         "s" : "BATBNB",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "388368.700000",
         "y" : 0
      },
      {
         "an" : "Basic Attention Token",
         "as" : 4902164,
         "b" : "BAT",
         "ba" : "",
         "c" : "0.00002240",
         "cs" : 1492816440,
         "etf" : false,
         "h" : "0.00002281",
         "i" : "1.00000000",
         "l" : "0.00002099",
         "o" : "0.00002124",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "106.834854",
         "s" : "BATBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "4902164.00000000",
         "y" : 0
      },
      {
         "an" : "Basic Attention Token",
         "as" : 1658780.05,
         "b" : "BAT",
         "ba" : "",
         "c" : "1.2340",
         "cs" : 1492816440,
         "etf" : false,
         "h" : "1.2526",
         "i" : "0.01000000",
         "l" : "1.1081",
         "o" : "1.1443",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1937819.436858",
         "s" : "BATBUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "1658780.0500",
         "y" : 0
      },
      {
         "an" : "Basic Attention Token",
         "as" : 608219,
         "b" : "BAT",
         "ba" : "",
         "c" : "0.00048316",
         "cs" : 1492816440,
         "etf" : false,
         "h" : "0.00049583",
         "i" : "1.00000000",
         "l" : "0.00044901",
         "o" : "0.00045613",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "286.04721314",
         "s" : "BATETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "608219.00000000",
         "y" : 0
      },
      {
         "an" : "Basic Attention Token",
         "as" : 56991.05,
         "b" : "BAT",
         "ba" : "",
         "c" : "1.2385",
         "cs" : 1492816440,
         "etf" : false,
         "h" : "1.2536",
         "i" : "0.01000000",
         "l" : "1.1102",
         "o" : "1.1429",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDC",
         "qa" : "",
         "qn" : "USD Coin",
         "qv" : "66718.609071",
         "s" : "BATUSDC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "56991.0500",
         "y" : 0
      },
      {
         "an" : "Basic Attention Token",
         "as" : 19568906.78,
         "b" : "BAT",
         "ba" : "",
         "c" : "1.2317",
         "cs" : 1492816440,
         "etf" : false,
         "h" : "1.2516",
         "i" : "0.01000000",
         "l" : "1.1080",
         "o" : "1.1446",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "22977681.351914",
         "s" : "BATUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "19568906.7800",
         "y" : 0
      },
      {
         "an" : "Bitcoin Diamond",
         "as" : 1599234,
         "b" : "BCD",
         "ba" : "",
         "c" : "0.00004221",
         "cs" : 0,
         "etf" : false,
         "h" : "0.00004370",
         "i" : "1.00000000",
         "l" : "0.00003918",
         "o" : "0.00004003",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "66.24455966",
         "s" : "BCDBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "1599234.00000000",
         "y" : 0
      },
      {
         "an" : "Bitcoin Cash ABC",
         "as" : 32224.923,
         "b" : "BCHA",
         "ba" : "",
         "c" : "30.999",
         "cs" : null,
         "etf" : false,
         "h" : "31.232",
         "i" : "0.00100000",
         "l" : "28.921",
         "o" : "30.498",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "973396.559547",
         "s" : "BCHABUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "pow"
         ],
         "ts" : "0.001",
         "v" : "32224.923",
         "y" : 0
      },
      {
         "an" : "Bitcoin Cash",
         "as" : 2036.667,
         "b" : "BCH",
         "ba" : "",
         "c" : "1.526",
         "cs" : 18719775,
         "etf" : false,
         "h" : "1.629",
         "i" : "0.00100000",
         "l" : "1.515",
         "o" : "1.546",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "3200.815337",
         "s" : "BCHBNB",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "2036.667",
         "y" : 0
      },
      {
         "an" : "Bitcoin Cash",
         "as" : 33428.803,
         "b" : "BCH",
         "ba" : "",
         "c" : "0.015911",
         "cs" : 18719775,
         "etf" : false,
         "h" : "0.016113",
         "i" : "0.00100000",
         "l" : "0.015395",
         "o" : "0.015512",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "525.69723219",
         "s" : "BCHBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "33428.803000",
         "y" : 0
      },
      {
         "an" : "Bitcoin Cash",
         "as" : 7424.39011,
         "b" : "BCH",
         "ba" : "",
         "c" : "875.14",
         "cs" : 18719775,
         "etf" : false,
         "h" : "883.12",
         "i" : "0.00001000",
         "l" : "816.19",
         "o" : "835.51",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "6286416.6177219",
         "s" : "BCHBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "7424.40",
         "y" : 0
      },
      {
         "an" : "BCHDOWN",
         "as" : 512288.35,
         "b" : "BCHDOWN",
         "ba" : "",
         "c" : "3.58",
         "cs" : null,
         "etf" : true,
         "h" : "4.39",
         "i" : "0.01000000",
         "l" : "3.46",
         "o" : "4.14",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "1970348.1124",
         "s" : "BCHDOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.01",
         "v" : "512288.35",
         "y" : 0
      },
      {
         "an" : "Bitcoin Cash",
         "as" : 2129.93607,
         "b" : "BCH",
         "ba" : "",
         "c" : "725.58",
         "cs" : 18719775,
         "etf" : false,
         "h" : "732.24",
         "i" : "0.00001000",
         "l" : "676.81",
         "o" : "691.38",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "1497442.6518203",
         "s" : "BCHEUR",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "2129.94",
         "y" : 0
      },
      {
         "an" : "Bitcoin Cash",
         "as" : 25.51713,
         "b" : "BCH",
         "ba" : "",
         "c" : "874.16",
         "cs" : 18719775,
         "etf" : false,
         "h" : "884.28",
         "i" : "0.00001000",
         "l" : "816.85",
         "o" : "830.96",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "PAX",
         "qa" : "",
         "qn" : "Paxos Standard",
         "qv" : "21469.1508683",
         "s" : "BCHPAX",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "25.52",
         "y" : 0
      },
      {
         "an" : "Bitcoin Cash",
         "as" : 301.21864,
         "b" : "BCH",
         "ba" : "",
         "c" : "874.51",
         "cs" : 18719775,
         "etf" : false,
         "h" : "881.94",
         "i" : "0.00001000",
         "l" : "817.08",
         "o" : "829.87",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TUSD",
         "qa" : "",
         "qn" : "TrueUSD",
         "qv" : "254472.5091509",
         "s" : "BCHTUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "301.22",
         "y" : 0
      },
      {
         "an" : "BCHUP",
         "as" : 537266.15,
         "b" : "BCHUP",
         "ba" : "",
         "c" : "7.83",
         "cs" : null,
         "etf" : true,
         "h" : "8.03",
         "i" : "0.01000000",
         "l" : "6.72",
         "o" : "7.10",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "3927437.8136",
         "s" : "BCHUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.01",
         "v" : "537266.15",
         "y" : 0
      },
      {
         "an" : "Bitcoin Cash",
         "as" : 1460.78977,
         "b" : "BCH",
         "ba" : "",
         "c" : "875.35",
         "cs" : 18719775,
         "etf" : false,
         "h" : "883.72",
         "i" : "0.00001000",
         "l" : "816.18",
         "o" : "834.00",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDC",
         "qa" : "",
         "qn" : "USD Coin",
         "qv" : "1227954.8150265",
         "s" : "BCHUSDC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "1460.79",
         "y" : 0
      },
      {
         "an" : "Bitcoin Cash",
         "as" : 198044.0966,
         "b" : "BCH",
         "ba" : "",
         "c" : "874.04",
         "cs" : 18719775,
         "etf" : false,
         "h" : "883.00",
         "i" : "0.00001000",
         "l" : "814.80",
         "o" : "833.70",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "167407590.5172194",
         "s" : "BCHUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "198044.10",
         "y" : 0
      },
      {
         "an" : "Beam",
         "as" : 2195888.36,
         "b" : "BEAM",
         "ba" : "",
         "c" : "0.00002494",
         "cs" : null,
         "etf" : false,
         "h" : "0.00002551",
         "i" : "0.01000000",
         "l" : "0.00002375",
         "o" : "0.00002530",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "54.20954703",
         "s" : "BEAMBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "2195888.36000000",
         "y" : 0
      },
      {
         "an" : "Beam",
         "as" : 5911680.58,
         "b" : "BEAM",
         "ba" : "",
         "c" : "1.3690",
         "cs" : null,
         "etf" : false,
         "h" : "1.4050",
         "i" : "0.01000000",
         "l" : "1.2747",
         "o" : "1.3616",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "7921526.216476",
         "s" : "BEAMUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "5911680.5800",
         "y" : 0
      },
      {
         "an" : "Bella Protocol",
         "as" : 62451.8,
         "b" : "BEL",
         "ba" : "",
         "c" : "0.00676",
         "cs" : null,
         "etf" : false,
         "h" : "0.00719",
         "i" : "0.10000000",
         "l" : "0.00667",
         "o" : "0.00674",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "429.063708",
         "s" : "BELBNB",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.00001",
         "v" : "62451.80000",
         "y" : 0
      },
      {
         "an" : "Bella Protocol",
         "as" : 702056,
         "b" : "BEL",
         "ba" : "",
         "c" : "0.00007081",
         "cs" : null,
         "etf" : false,
         "h" : "0.00007259",
         "i" : "1.00000000",
         "l" : "0.00006245",
         "o" : "0.00006715",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "48.08930872",
         "s" : "BELBTC",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.00000001",
         "v" : "702056.00000000",
         "y" : 0
      },
      {
         "an" : "Bella Protocol",
         "as" : 125431.7,
         "b" : "BEL",
         "ba" : "",
         "c" : "3.8881",
         "cs" : null,
         "etf" : false,
         "h" : "3.9893",
         "i" : "0.01000000",
         "l" : "3.4705",
         "o" : "3.6119",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "463153.918895",
         "s" : "BELBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.0001",
         "v" : "125431.7000",
         "y" : 0
      },
      {
         "an" : "Bella Protocol",
         "as" : 3103572.4,
         "b" : "BEL",
         "ba" : "",
         "c" : "3.8842",
         "cs" : null,
         "etf" : false,
         "h" : "3.9888",
         "i" : "0.01000000",
         "l" : "3.4752",
         "o" : "3.6113",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "11515289.189539",
         "s" : "BELUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.0001",
         "v" : "3103572.4000",
         "y" : 0
      },
      {
         "an" : "BETH",
         "as" : 3360.7143,
         "b" : "BETH",
         "ba" : "",
         "c" : "0.9455",
         "cs" : null,
         "etf" : false,
         "h" : "0.9503",
         "i" : "0.00010000",
         "l" : "0.9315",
         "o" : "0.9497",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "3171.9316298",
         "s" : "BETHETH",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.0001",
         "v" : "3360.7143",
         "y" : 0
      },
      {
         "an" : "Beefy.Finance",
         "as" : 351.647,
         "b" : "BIFI",
         "ba" : "",
         "c" : "2.445",
         "cs" : null,
         "etf" : false,
         "h" : "2.675",
         "i" : "0.00100000",
         "l" : "2.418",
         "o" : "2.543",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "888.04227",
         "s" : "BIFIBNB",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "351.647",
         "y" : 0
      },
      {
         "an" : "Beefy.Finance",
         "as" : 1378.62147,
         "b" : "BIFI",
         "ba" : "",
         "c" : "1387.88",
         "cs" : null,
         "etf" : false,
         "h" : "1490.00",
         "i" : "0.00001000",
         "l" : "1250.00",
         "o" : "1340.26",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1884063.6195155",
         "s" : "BIFIBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "BSC"
         ],
         "ts" : "0.01",
         "v" : "1378.63",
         "y" : 0
      },
      {
         "an" : "Bluzelle",
         "as" : 793797,
         "b" : "BLZ",
         "ba" : "",
         "c" : "0.000694",
         "cs" : 194767618,
         "etf" : false,
         "h" : "0.000704",
         "i" : "1.00000000",
         "l" : "0.000632",
         "o" : "0.000638",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "536.401332",
         "s" : "BLZBNB",
         "st" : "TRADING",
         "tags" : [
            "storage-zone"
         ],
         "ts" : "0.000001",
         "v" : "793797.000000",
         "y" : 0
      },
      {
         "an" : "Bluzelle",
         "as" : 11515839,
         "b" : "BLZ",
         "ba" : "",
         "c" : "0.00000723",
         "cs" : 194767618,
         "etf" : false,
         "h" : "0.00000725",
         "i" : "1.00000000",
         "l" : "0.00000625",
         "o" : "0.00000640",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "78.12964274",
         "s" : "BLZBTC",
         "st" : "TRADING",
         "tags" : [
            "storage-zone"
         ],
         "ts" : "0.00000001",
         "v" : "11515839.00000000",
         "y" : 0
      },
      {
         "an" : "Bluzelle",
         "as" : 1123406,
         "b" : "BLZ",
         "ba" : "",
         "c" : "0.00015396",
         "cs" : 194767618,
         "etf" : false,
         "h" : "0.00015804",
         "i" : "1.00000000",
         "l" : "0.00013563",
         "o" : "0.00013762",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "164.98389888",
         "s" : "BLZETH",
         "st" : "TRADING",
         "tags" : [
            "storage-zone"
         ],
         "ts" : "0.00000001",
         "v" : "1123406.00000000",
         "y" : 0
      },
      {
         "an" : "Bluzelle",
         "as" : 43417653,
         "b" : "BLZ",
         "ba" : "",
         "c" : "0.39700",
         "cs" : 194767618,
         "etf" : false,
         "h" : "0.39825",
         "i" : "0.10000000",
         "l" : "0.32949",
         "o" : "0.34472",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "15822541.460347",
         "s" : "BLZUSDT",
         "st" : "TRADING",
         "tags" : [
            "storage-zone"
         ],
         "ts" : "0.00001",
         "v" : "43417653.00000",
         "y" : 0
      },
      {
         "an" : "BNB",
         "as" : 7756.1566,
         "b" : "BNB",
         "ba" : "",
         "c" : "741.69",
         "cs" : 152665937,
         "etf" : false,
         "h" : "745.99",
         "i" : "0.00010000",
         "l" : "663.69",
         "o" : "696.20",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "AUD",
         "qa" : "",
         "qn" : "Australian Dollar",
         "qv" : "5437566.212254",
         "s" : "BNBAUD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.01",
         "v" : "7756.16",
         "y" : 0
      },
      {
         "an" : "BNB",
         "as" : 7736.93,
         "b" : "BNB",
         "ba" : "",
         "c" : "8487458",
         "cs" : 152665937,
         "etf" : false,
         "h" : "8500000",
         "i" : "0.00100000",
         "l" : "7701112",
         "o" : "8023222",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BIDR",
         "qa" : "",
         "qn" : "BIDR",
         "qv" : "62518419888.82",
         "s" : "BNBBIDR",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "1",
         "v" : "7737",
         "y" : 0
      },
      {
         "an" : "BNB",
         "as" : 9525.61369,
         "b" : "BNB",
         "ba" : "",
         "c" : "3135.68",
         "cs" : 152665937,
         "etf" : false,
         "h" : "3150.00",
         "i" : "0.00001000",
         "l" : "2822.44",
         "o" : "2985.54",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BRL",
         "qa" : "",
         "qn" : "Brazilian Real",
         "qv" : "28316487.4679749",
         "s" : "BNBBRL",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.01",
         "v" : "9525.62",
         "y" : 0
      },
      {
         "an" : "BNB",
         "as" : 641313.0662,
         "b" : "BNB",
         "ba" : "",
         "c" : "573.98",
         "cs" : 152665937,
         "etf" : false,
         "h" : "576.38",
         "i" : "0.00010000",
         "l" : "515.50",
         "o" : "539.19",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "347688866.67549948",
         "s" : "BNBBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.01",
         "v" : "641313.07",
         "y" : 0
      },
      {
         "an" : "BNB",
         "as" : 3716.9173,
         "b" : "BNB",
         "ba" : "",
         "c" : "571.33",
         "cs" : 152665937,
         "etf" : false,
         "h" : "575.72",
         "i" : "0.00010000",
         "l" : "515.62",
         "o" : "537.98",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "DAI",
         "qa" : "",
         "qn" : "Dai",
         "qv" : "2016175.661519",
         "s" : "BNBDAI",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.01",
         "v" : "3716.92",
         "y" : 0
      },
      {
         "an" : "BNBDOWN",
         "as" : 554871.15,
         "b" : "BNBDOWN",
         "ba" : null,
         "c" : "11.62",
         "cs" : null,
         "etf" : true,
         "h" : "15.54",
         "i" : "0.01000000",
         "l" : "11.47",
         "o" : "14.05",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "7550698.708",
         "s" : "BNBDOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.01",
         "v" : "554871.15",
         "y" : 0
      },
      {
         "an" : "BNB",
         "as" : 116073.48,
         "b" : "BNB",
         "ba" : "",
         "c" : "0.22401",
         "cs" : 152665937,
         "etf" : false,
         "h" : "0.22452",
         "i" : "0.00100000",
         "l" : "0.20742",
         "o" : "0.21495",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "24956.19864347",
         "s" : "BNBETH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.00001",
         "v" : "116073.48000",
         "y" : 0
      },
      {
         "an" : "BNB",
         "as" : 96544.3408,
         "b" : "BNB",
         "ba" : "",
         "c" : "475.58",
         "cs" : 152665937,
         "etf" : false,
         "h" : "477.31",
         "i" : "0.00010000",
         "l" : "427.00",
         "o" : "447.05",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "43611781.26077121",
         "s" : "BNBEUR",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.01",
         "v" : "96544.35",
         "y" : 0
      },
      {
         "an" : "BNB",
         "as" : 39622.572,
         "b" : "BNB",
         "ba" : "",
         "c" : "413.62",
         "cs" : 152665937,
         "etf" : false,
         "h" : "415.78",
         "i" : "0.00010000",
         "l" : "371.87",
         "o" : "389.17",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "GBP",
         "qa" : "",
         "qn" : "Pound Sterling",
         "qv" : "15566057.210491",
         "s" : "BNBGBP",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.01",
         "v" : "39622.58",
         "y" : 0
      },
      {
         "an" : "BNB",
         "as" : 681.711,
         "b" : "BNB",
         "ba" : "",
         "c" : "8444444",
         "cs" : 152665937,
         "etf" : false,
         "h" : "8500092",
         "i" : "0.00100000",
         "l" : "7684197",
         "o" : "8000000",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "IDRT",
         "qa" : "",
         "qn" : "Rupiah Token",
         "qv" : "5464144550.28",
         "s" : "BNBIDRT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "1",
         "v" : "682",
         "y" : 0
      },
      {
         "an" : "BNB",
         "as" : 288.8101,
         "b" : "BNB",
         "ba" : "",
         "c" : "571.13",
         "cs" : 152665937,
         "etf" : false,
         "h" : "575.44",
         "i" : "0.00010000",
         "l" : "514.75",
         "o" : "537.35",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "PAX",
         "qa" : "",
         "qn" : "Paxos Standard",
         "qv" : "160363.9304241",
         "s" : "BNBPAX",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.01",
         "v" : "288.82",
         "y" : 0
      },
      {
         "an" : "BNB",
         "as" : 2147.421,
         "b" : "BNB",
         "ba" : "",
         "c" : "43277.50",
         "cs" : 152665937,
         "etf" : false,
         "h" : "43500.00",
         "i" : "0.00100000",
         "l" : "38923.81",
         "o" : "40731.29",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "RUB",
         "qa" : "",
         "qn" : "Russian Ruble",
         "qv" : "88016121.26538",
         "s" : "BNBRUB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.01",
         "v" : "2147.43",
         "y" : 0
      },
      {
         "an" : "BNB",
         "as" : 37809.971,
         "b" : "BNB",
         "ba" : "",
         "c" : "4669",
         "cs" : 152665937,
         "etf" : false,
         "h" : "4690",
         "i" : "0.00100000",
         "l" : "4200",
         "o" : "4399",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "167377804.208",
         "s" : "BNBTRY",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "1",
         "v" : "37810",
         "y" : 0
      },
      {
         "an" : "BNB",
         "as" : 1453.2974,
         "b" : "BNB",
         "ba" : "",
         "c" : "570.60",
         "cs" : 152665937,
         "etf" : false,
         "h" : "576.19",
         "i" : "0.00010000",
         "l" : "515.29",
         "o" : "537.86",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TUSD",
         "qa" : "",
         "qn" : "TrueUSD",
         "qv" : "782067.4694554",
         "s" : "BNBTUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.01",
         "v" : "1453.30",
         "y" : 0
      },
      {
         "an" : "BNB",
         "as" : 378.50562,
         "b" : "BNB",
         "ba" : "",
         "c" : "16130.1",
         "cs" : 152665937,
         "etf" : false,
         "h" : "16266.9",
         "i" : "0.00001000",
         "l" : "14609.9",
         "o" : "15270.0",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "UAH",
         "qa" : "",
         "qn" : "Ukraine Hryvnia",
         "qv" : "5835980.902758",
         "s" : "BNBUAH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.1",
         "v" : "378.6",
         "y" : 0
      },
      {
         "an" : "BNBUP",
         "as" : 31679.25,
         "b" : "BNBUP",
         "ba" : null,
         "c" : "1677.090",
         "cs" : null,
         "etf" : true,
         "h" : "1692.238",
         "i" : "0.01000000",
         "l" : "1317.456",
         "o" : "1480.998",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "46101968.81334",
         "s" : "BNBUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "31679.250",
         "y" : 0
      },
      {
         "an" : "BNB",
         "as" : 13954.4169,
         "b" : "BNB",
         "ba" : "",
         "c" : "573.82",
         "cs" : 152665937,
         "etf" : false,
         "h" : "576.62",
         "i" : "0.00010000",
         "l" : "514.01",
         "o" : "539.28",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDC",
         "qa" : "",
         "qn" : "USD Coin",
         "qv" : "7522820.7522686",
         "s" : "BNBUSDC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.01",
         "v" : "13954.42",
         "y" : 0
      },
      {
         "an" : "Bancor",
         "as" : 205408.7,
         "b" : "BNT",
         "ba" : "",
         "c" : "0.00011940",
         "cs" : 178540878,
         "etf" : false,
         "h" : "0.00012098",
         "i" : "0.10000000",
         "l" : "0.00011642",
         "o" : "0.00011716",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "24.42955634",
         "s" : "BNTBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "205408.70000000",
         "y" : 0
      },
      {
         "an" : "Bancor",
         "as" : 32985.05,
         "b" : "BNT",
         "ba" : "",
         "c" : "6.5475",
         "cs" : 178540878,
         "etf" : false,
         "h" : "6.6496",
         "i" : "0.01000000",
         "l" : "6.2003",
         "o" : "6.2979",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "211483.23689",
         "s" : "BNTBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "32985.0500",
         "y" : 0
      },
      {
         "an" : "Bancor",
         "as" : 516818.01,
         "b" : "BNT",
         "ba" : "",
         "c" : "6.5665",
         "cs" : 178540878,
         "etf" : false,
         "h" : "6.6447",
         "i" : "0.01000000",
         "l" : "6.1727",
         "o" : "6.3021",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "3320760.966086",
         "s" : "BNTUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "516818.0100",
         "y" : 0
      },
      {
         "an" : "Bread",
         "as" : 2723229,
         "b" : "BRD",
         "ba" : "",
         "c" : "0.00000611",
         "cs" : 88733977,
         "etf" : false,
         "h" : "0.00000626",
         "i" : "1.00000000",
         "l" : "0.00000561",
         "o" : "0.00000586",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "16.1499893",
         "s" : "BRDBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "2723229.00000000",
         "y" : 0
      },
      {
         "an" : "Bread",
         "as" : 526201,
         "b" : "BRD",
         "ba" : "",
         "c" : "0.0001310",
         "cs" : 88733977,
         "etf" : false,
         "h" : "0.0001334",
         "i" : "1.00000000",
         "l" : "0.0001218",
         "o" : "0.0001253",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "66.9999965",
         "s" : "BRDETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "526201.0000000",
         "y" : 0
      },
      {
         "an" : "Bitcoin",
         "as" : 287.028331,
         "b" : "BTC",
         "ba" : "à¸¿",
         "c" : "71040.05",
         "cs" : 18692200,
         "etf" : false,
         "h" : "71575.25",
         "i" : "0.00000100",
         "l" : "67754.80",
         "o" : "69465.39",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "AUD",
         "qa" : "",
         "qn" : "Australian Dollar",
         "qv" : "20019657.32877839",
         "s" : "BTCAUD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "287.03",
         "y" : 0
      },
      {
         "an" : "Bitcoin",
         "as" : 75.971112,
         "b" : "BTC",
         "ba" : "à¸¿",
         "c" : "813223699",
         "cs" : 18692200,
         "etf" : false,
         "h" : "816842105",
         "i" : "0.00000100",
         "l" : "668800000",
         "o" : "800000000",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BIDR",
         "qa" : "",
         "qn" : "BIDR",
         "qv" : "60841564471.8",
         "s" : "BTCBIDR",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "1",
         "v" : "76",
         "y" : 0
      },
      {
         "an" : "Bitcoin",
         "as" : 329.719083,
         "b" : "BTC",
         "ba" : "à¸¿",
         "c" : "300250",
         "cs" : 18692200,
         "etf" : false,
         "h" : "302640",
         "i" : "0.00000100",
         "l" : "288337",
         "o" : "297875",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BRL",
         "qa" : "",
         "qn" : "Brazilian Real",
         "qv" : "97538235.108965",
         "s" : "BTCBRL",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "1",
         "v" : "330",
         "y" : 0
      },
      {
         "an" : "Bitcoin",
         "as" : 14717.320674,
         "b" : "BTC",
         "ba" : "à¸¿",
         "c" : "54999.99",
         "cs" : 18692200,
         "etf" : false,
         "h" : "55350.00",
         "i" : "0.00000100",
         "l" : "52650.00",
         "o" : "53854.17",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "794339678.17663762",
         "s" : "BTCBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "14717.33",
         "y" : 0
      },
      {
         "an" : "Bitcoin",
         "as" : 155.227195,
         "b" : "BTC",
         "ba" : "à¸¿",
         "c" : "54956.70",
         "cs" : 18692200,
         "etf" : false,
         "h" : "55233.17",
         "i" : "0.00000100",
         "l" : "52617.08",
         "o" : "53773.26",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "DAI",
         "qa" : "",
         "qn" : "Dai",
         "qv" : "8341217.56996924",
         "s" : "BTCDAI",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "155.23",
         "y" : 0
      },
      {
         "an" : "BTCDOWN",
         "as" : 543260614.53,
         "b" : "BTCDOWN",
         "ba" : "",
         "c" : "0.04770",
         "cs" : null,
         "etf" : true,
         "h" : "0.05393",
         "i" : "0.01000000",
         "l" : "0.04661",
         "o" : "0.05081",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "27371423.3689956",
         "s" : "BTCDOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "543260614.53000",
         "y" : 0
      },
      {
         "an" : "Bitcoin",
         "as" : 2980.255896,
         "b" : "BTC",
         "ba" : "à¸¿",
         "c" : "45572.23",
         "cs" : 18692200,
         "etf" : false,
         "h" : "45832.99",
         "i" : "0.00000100",
         "l" : "43627.60",
         "o" : "44650.00",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "133574603.99427111",
         "s" : "BTCEUR",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "2980.26",
         "y" : 0
      },
      {
         "an" : "Bitcoin",
         "as" : 1113.834878,
         "b" : "BTC",
         "ba" : "à¸¿",
         "c" : "39662.57",
         "cs" : 18692200,
         "etf" : false,
         "h" : "39900.00",
         "i" : "0.00000100",
         "l" : "37990.10",
         "o" : "38861.53",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "GBP",
         "qa" : "",
         "qn" : "Pound Sterling",
         "qv" : "43386320.87626525",
         "s" : "BTCGBP",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "1113.84",
         "y" : 0
      },
      {
         "an" : "Bitcoin",
         "as" : 4.705051,
         "b" : "BTC",
         "ba" : "à¸¿",
         "c" : "808005638",
         "cs" : 18692200,
         "etf" : false,
         "h" : "815228974",
         "i" : "0.00000100",
         "l" : "785180833",
         "o" : "795623223",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "IDRT",
         "qa" : "",
         "qn" : "Rupiah Token",
         "qv" : "3763010771.65",
         "s" : "BTCIDRT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "1",
         "v" : "5",
         "y" : 0
      },
      {
         "an" : "Bitcoin",
         "as" : 44.580276,
         "b" : "BTC",
         "ba" : "à¸¿",
         "c" : "34217178",
         "cs" : 18692200,
         "etf" : false,
         "h" : "34700000",
         "i" : "0.00000100",
         "l" : "33900000",
         "o" : "34293550",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "NGN",
         "qa" : "",
         "qn" : "Nigerian Naira",
         "qv" : "1528657844.189709",
         "s" : "BTCNGN",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "1",
         "v" : "45",
         "y" : 0
      },
      {
         "an" : "Bitcoin",
         "as" : 287.573547,
         "b" : "BTC",
         "ba" : "à¸¿",
         "c" : "54993.39",
         "cs" : 18692200,
         "etf" : false,
         "h" : "55353.41",
         "i" : "0.00000100",
         "l" : "52679.08",
         "o" : "53787.55",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "PAX",
         "qa" : "",
         "qn" : "Paxos Standard",
         "qv" : "15563503.94847485",
         "s" : "BTCPAX",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "287.58",
         "y" : 0
      },
      {
         "an" : "Bitcoin",
         "as" : 148.095315,
         "b" : "BTC",
         "ba" : "à¸¿",
         "c" : "4148011",
         "cs" : 18692200,
         "etf" : false,
         "h" : "4175000",
         "i" : "0.00000100",
         "l" : "3980379",
         "o" : "4068011",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "RUB",
         "qa" : "",
         "qn" : "Russian Ruble",
         "qv" : "602104146.606433",
         "s" : "BTCRUB",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "1",
         "v" : "149",
         "y" : 0
      },
      {
         "an" : "Bitcoin Standard Hashrate Token",
         "as" : 31892.16,
         "b" : "BTCST",
         "ba" : "",
         "c" : "0.0013446",
         "cs" : null,
         "etf" : false,
         "h" : "0.0013820",
         "i" : "0.01000000",
         "l" : "0.0012128",
         "o" : "0.0012188",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "41.98289592",
         "s" : "BTCSTBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "BSC"
         ],
         "ts" : "0.0000001",
         "v" : "31892.1600000",
         "y" : 0
      },
      {
         "an" : "Bitcoin Standard Hashrate Token",
         "as" : 47992.736,
         "b" : "BTCST",
         "ba" : "",
         "c" : "73.929",
         "cs" : null,
         "etf" : false,
         "h" : "74.581",
         "i" : "0.00100000",
         "l" : "65.202",
         "o" : "65.971",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "3416039.321858",
         "s" : "BTCSTBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "47992.736",
         "y" : 0
      },
      {
         "an" : "Bitcoin Standard Hashrate Token",
         "as" : 253400.085,
         "b" : "BTCST",
         "ba" : "",
         "c" : "73.900",
         "cs" : null,
         "etf" : false,
         "h" : "74.500",
         "i" : "0.00100000",
         "l" : "65.000",
         "o" : "65.676",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "18030417.770139",
         "s" : "BTCSTUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "253400.085",
         "y" : 0
      },
      {
         "an" : "Bitcoin",
         "as" : 288.085676,
         "b" : "BTC",
         "ba" : "à¸¿",
         "c" : "447146",
         "cs" : 18692200,
         "etf" : false,
         "h" : "450635",
         "i" : "0.00000100",
         "l" : "429186",
         "o" : "438923",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "126893897.595783",
         "s" : "BTCTRY",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "1",
         "v" : "289",
         "y" : 0
      },
      {
         "an" : "Bitcoin",
         "as" : 153.698225,
         "b" : "BTC",
         "ba" : "à¸¿",
         "c" : "54985.43",
         "cs" : 18692200,
         "etf" : false,
         "h" : "55347.75",
         "i" : "0.00000100",
         "l" : "52642.56",
         "o" : "53778.80",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TUSD",
         "qa" : "",
         "qn" : "TrueUSD",
         "qv" : "8302129.9693835",
         "s" : "BTCTUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "153.70",
         "y" : 0
      },
      {
         "an" : "Bitcoin",
         "as" : 9.242686,
         "b" : "BTC",
         "ba" : "à¸¿",
         "c" : "1545008",
         "cs" : 18692200,
         "etf" : false,
         "h" : "1562768",
         "i" : "0.00000100",
         "l" : "1490809",
         "o" : "1522748",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "UAH",
         "qa" : "",
         "qn" : "Ukraine Hryvnia",
         "qv" : "14163240.678323",
         "s" : "BTCUAH",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "1",
         "v" : "10",
         "y" : 0
      },
      {
         "an" : "BTCUP",
         "as" : 185292.67,
         "b" : "BTCUP",
         "ba" : "",
         "c" : "140.085",
         "cs" : null,
         "etf" : true,
         "h" : "142.660",
         "i" : "0.01000000",
         "l" : "128.000",
         "o" : "134.573",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "25049314.32101",
         "s" : "BTCUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "185292.670",
         "y" : 0
      },
      {
         "an" : "Bitcoin",
         "as" : 2375.385197,
         "b" : "BTC",
         "ba" : "à¸¿",
         "c" : "54995.30",
         "cs" : 18692200,
         "etf" : false,
         "h" : "55488.00",
         "i" : "0.00000100",
         "l" : "52500.00",
         "o" : "53860.35",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDC",
         "qa" : "",
         "qn" : "USD Coin",
         "qv" : "128279008.45923599",
         "s" : "BTCUSDC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "2375.39",
         "y" : 0
      },
      {
         "an" : "Bitcoin",
         "as" : 2.711138,
         "b" : "BTC",
         "ba" : "à¸¿",
         "c" : "56959.19",
         "cs" : 18692200,
         "etf" : false,
         "h" : "57334.13",
         "i" : "0.00000100",
         "l" : "54272.97",
         "o" : "55554.99",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "VAI",
         "qa" : "",
         "qn" : "Vai",
         "qv" : "149860.48012994",
         "s" : "BTCVAI",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "2.72",
         "y" : 0
      },
      {
         "an" : "Bitcoin Gold",
         "as" : 98771.04,
         "b" : "BTG",
         "ba" : "",
         "c" : "0.001703",
         "cs" : 17333349,
         "etf" : false,
         "h" : "0.001722",
         "i" : "0.01000000",
         "l" : "0.001501",
         "o" : "0.001538",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "158.07169096",
         "s" : "BTGBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "98771.040000",
         "y" : 0
      },
      {
         "an" : "Bitcoin Gold",
         "as" : 7828.023,
         "b" : "BTG",
         "ba" : "",
         "c" : "93.464",
         "cs" : 17333349,
         "etf" : false,
         "h" : "94.500",
         "i" : "0.00100000",
         "l" : "79.502",
         "o" : "82.700",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "668070.592327",
         "s" : "BTGBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "7828.023",
         "y" : 0
      },
      {
         "an" : "BitShares",
         "as" : 26022639,
         "b" : "BTS",
         "ba" : "",
         "c" : "0.00000198",
         "cs" : 2994880000,
         "etf" : false,
         "h" : "0.00000209",
         "i" : "1.00000000",
         "l" : "0.00000179",
         "o" : "0.00000186",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "51.03383989",
         "s" : "BTSBTC",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "26022639.00000000",
         "y" : 0
      },
      {
         "an" : "BitTorrent",
         "as" : 1346504652,
         "b" : "BTT",
         "ba" : "",
         "c" : "0.00001317",
         "cs" : 89100000000,
         "etf" : false,
         "h" : "0.00001429",
         "i" : "1.00000000",
         "l" : "0.00001275",
         "o" : "0.00001291",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "18049.65811491",
         "s" : "BTTBNB",
         "st" : "TRADING",
         "tags" : [
            "storage-zone",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "1346504652.00000000",
         "y" : 0
      },
      {
         "an" : "BitTorrent",
         "as" : 65072295.6,
         "b" : "BTT",
         "ba" : "",
         "c" : "0.04115",
         "cs" : 89100000000,
         "etf" : false,
         "h" : "0.04280",
         "i" : "0.10000000",
         "l" : "0.03704",
         "o" : "0.03853",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BRL",
         "qa" : "",
         "qn" : "Brazilian Real",
         "qv" : "2561138.418309",
         "s" : "BTTBRL",
         "st" : "TRADING",
         "tags" : [
            "storage-zone",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "65072295.60000",
         "y" : 0
      },
      {
         "an" : "BitTorrent",
         "as" : 2844923281,
         "b" : "BTT",
         "ba" : "",
         "c" : "0.0075594",
         "cs" : 89100000000,
         "etf" : false,
         "h" : "0.0078607",
         "i" : "1.00000000",
         "l" : "0.0067389",
         "o" : "0.0069638",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "20644347.4878696",
         "s" : "BTTBUSD",
         "st" : "TRADING",
         "tags" : [
            "storage-zone",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "2844923281.0000000",
         "y" : 0
      },
      {
         "an" : "BitTorrent",
         "as" : 687387001,
         "b" : "BTT",
         "ba" : "",
         "c" : "0.006269",
         "cs" : 89100000000,
         "etf" : false,
         "h" : "0.006546",
         "i" : "1.00000000",
         "l" : "0.005588",
         "o" : "0.005780",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "4140870.57734",
         "s" : "BTTEUR",
         "st" : "TRADING",
         "tags" : [
            "storage-zone",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "687387001.000000",
         "y" : 0
      },
      {
         "an" : "BitTorrent",
         "as" : 328073158.4,
         "b" : "BTT",
         "ba" : "",
         "c" : "0.05970",
         "cs" : 89100000000,
         "etf" : false,
         "h" : "0.06185",
         "i" : "0.10000000",
         "l" : "0.05827",
         "o" : "0.06052",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "TRX",
         "qa" : "",
         "qn" : "TRON",
         "qv" : "19691948.756382",
         "s" : "BTTTRX",
         "st" : "TRADING",
         "tags" : [
            "storage-zone",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "328073158.40000",
         "y" : 0
      },
      {
         "an" : "BitTorrent",
         "as" : 10715080017,
         "b" : "BTT",
         "ba" : "",
         "c" : "0.06145",
         "cs" : 89100000000,
         "etf" : false,
         "h" : "0.06410",
         "i" : "1.00000000",
         "l" : "0.05501",
         "o" : "0.05678",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "633899323.39386",
         "s" : "BTTTRY",
         "st" : "TRADING",
         "tags" : [
            "storage-zone",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "10715080017.00000",
         "y" : 0
      },
      {
         "an" : "BitTorrent",
         "as" : 120167150,
         "b" : "BTT",
         "ba" : "",
         "c" : "0.0075753",
         "cs" : 89100000000,
         "etf" : false,
         "h" : "0.0078436",
         "i" : "1.00000000",
         "l" : "0.0067386",
         "o" : "0.0069663",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TUSD",
         "qa" : "",
         "qn" : "TrueUSD",
         "qv" : "864541.8814512",
         "s" : "BTTTUSD",
         "st" : "TRADING",
         "tags" : [
            "storage-zone",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "120167150.0000000",
         "y" : 0
      },
      {
         "an" : "BitTorrent",
         "as" : 88942619,
         "b" : "BTT",
         "ba" : "",
         "c" : "0.0075101",
         "cs" : 89100000000,
         "etf" : false,
         "h" : "0.0078632",
         "i" : "1.00000000",
         "l" : "0.0067366",
         "o" : "0.0069579",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDC",
         "qa" : "",
         "qn" : "USD Coin",
         "qv" : "638894.0816193",
         "s" : "BTTUSDC",
         "st" : "TRADING",
         "tags" : [
            "storage-zone",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "88942619.0000000",
         "y" : 0
      },
      {
         "an" : "BitTorrent",
         "as" : 61806806567,
         "b" : "BTT",
         "ba" : "",
         "c" : "0.0075475",
         "cs" : 89100000000,
         "etf" : false,
         "h" : "0.0078700",
         "i" : "1.00000000",
         "l" : "0.0067300",
         "o" : "0.0069518",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "449268581.8592691",
         "s" : "BTTUSDT",
         "st" : "TRADING",
         "tags" : [
            "storage-zone",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "61806806567.0000000",
         "y" : 0
      },
      {
         "an" : "Burger Swap",
         "as" : 485002.7,
         "b" : "BURGER",
         "ba" : "",
         "c" : "0.00810",
         "cs" : null,
         "etf" : false,
         "h" : "0.00893",
         "i" : "0.10000000",
         "l" : "0.00803",
         "o" : "0.00838",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "4084.247321",
         "s" : "BURGERBNB",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "BSC"
         ],
         "ts" : "0.00001",
         "v" : "485002.70000",
         "y" : 0
      },
      {
         "an" : "BUSD",
         "as" : 3039794.19,
         "b" : "BUSD",
         "ba" : "",
         "c" : "14792",
         "cs" : 16651765,
         "etf" : false,
         "h" : "15027",
         "i" : "0.01000000",
         "l" : "14701",
         "o" : "14865",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BIDR",
         "qa" : "",
         "qn" : "BIDR",
         "qv" : "45203420772.12",
         "s" : "BUSDBIDR",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "1",
         "v" : "3039795",
         "y" : 0
      },
      {
         "an" : "BUSD",
         "as" : 7689257.71,
         "b" : "BUSD",
         "ba" : "",
         "c" : "5.460",
         "cs" : 16651765,
         "etf" : false,
         "h" : "5.539",
         "i" : "0.01000000",
         "l" : "5.442",
         "o" : "5.532",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BRL",
         "qa" : "",
         "qn" : "Brazilian Real",
         "qv" : "42174351.62879",
         "s" : "BUSDBRL",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "7689257.710",
         "y" : 0
      },
      {
         "an" : "BUSD",
         "as" : 71926.27,
         "b" : "BUSD",
         "ba" : "",
         "c" : "24027",
         "cs" : 16651765,
         "etf" : false,
         "h" : "24500",
         "i" : "0.01000000",
         "l" : "23968",
         "o" : "24342",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BVND",
         "qa" : "",
         "qn" : "BVND",
         "qv" : "1731659514.94",
         "s" : "BUSDBVND",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "1",
         "v" : "71927",
         "y" : 0
      },
      {
         "an" : "BUSD",
         "as" : 5568132.84,
         "b" : "BUSD",
         "ba" : "",
         "c" : "0.9989",
         "cs" : 16651765,
         "etf" : false,
         "h" : "0.9995",
         "i" : "0.01000000",
         "l" : "0.9984",
         "o" : "0.9992",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "DAI",
         "qa" : "",
         "qn" : "Dai",
         "qv" : "5562324.960823",
         "s" : "BUSDDAI",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.0001",
         "v" : "5568132.8400",
         "y" : 0
      },
      {
         "an" : "BUSD",
         "as" : 2154877.4,
         "b" : "BUSD",
         "ba" : "",
         "c" : "75.423",
         "cs" : 16651765,
         "etf" : false,
         "h" : "75.774",
         "i" : "0.10000000",
         "l" : "75.240",
         "o" : "75.594",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "RUB",
         "qa" : "",
         "qn" : "Russian Ruble",
         "qv" : "162661473.1392",
         "s" : "BUSDRUB",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "2154877.400",
         "y" : 0
      },
      {
         "an" : "BUSD",
         "as" : 13247724.38,
         "b" : "BUSD",
         "ba" : "",
         "c" : "8.14",
         "cs" : 16651765,
         "etf" : false,
         "h" : "8.19",
         "i" : "0.01000000",
         "l" : "8.09",
         "o" : "8.15",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "107917969.9635",
         "s" : "BUSDTRY",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.01",
         "v" : "13247724.38",
         "y" : 0
      },
      {
         "an" : "BUSD",
         "as" : 988965163.1,
         "b" : "BUSD",
         "ba" : "",
         "c" : "0.9987",
         "cs" : 16651765,
         "etf" : false,
         "h" : "0.9994",
         "i" : "0.01000000",
         "l" : "0.9985",
         "o" : "0.9986",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "987872460.75786",
         "s" : "BUSDUSDT",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.0001",
         "v" : "988965163.1000",
         "y" : 0
      },
      {
         "an" : "BUSD",
         "as" : 1863982.68,
         "b" : "BUSD",
         "ba" : "",
         "c" : "1.0354",
         "cs" : 16651765,
         "etf" : false,
         "h" : "1.0492",
         "i" : "0.01000000",
         "l" : "1.0251",
         "o" : "1.0294",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "VAI",
         "qa" : "",
         "qn" : "Vai",
         "qv" : "1931022.429299",
         "s" : "BUSDVAI",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.0001",
         "v" : "1863982.6800",
         "y" : 0
      },
      {
         "an" : "bZx Protocol",
         "as" : 7411375,
         "b" : "BZRX",
         "ba" : "",
         "c" : "0.00001371",
         "cs" : null,
         "etf" : false,
         "h" : "0.00001520",
         "i" : "1.00000000",
         "l" : "0.00001330",
         "o" : "0.00001445",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "104.54678876",
         "s" : "BZRXBTC",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone"
         ],
         "ts" : "0.00000001",
         "v" : "7411375.00000000",
         "y" : 0
      },
      {
         "an" : "bZx Protocol",
         "as" : 2828169.46,
         "b" : "BZRX",
         "ba" : "",
         "c" : "0.7548",
         "cs" : null,
         "etf" : false,
         "h" : "0.8023",
         "i" : "0.01000000",
         "l" : "0.7040",
         "o" : "0.7802",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2148562.474964",
         "s" : "BZRXBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone"
         ],
         "ts" : "0.0001",
         "v" : "2828169.4600",
         "y" : 0
      },
      {
         "an" : "bZx Protocol",
         "as" : 37841838.55,
         "b" : "BZRX",
         "ba" : "",
         "c" : "0.7531",
         "cs" : null,
         "etf" : false,
         "h" : "0.8000",
         "i" : "0.01000000",
         "l" : "0.7020",
         "o" : "0.7765",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "28647339.424517",
         "s" : "BZRXUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone"
         ],
         "ts" : "0.0001",
         "v" : "37841838.5500",
         "y" : 0
      },
      {
         "an" : "PancakeSwap",
         "as" : 1370167.6,
         "b" : "CAKE",
         "ba" : "",
         "c" : "0.06238",
         "cs" : null,
         "etf" : false,
         "h" : "0.06378",
         "i" : "0.10000000",
         "l" : "0.06032",
         "o" : "0.06247",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "85145.457976",
         "s" : "CAKEBNB",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "BSC"
         ],
         "ts" : "0.00001",
         "v" : "1370167.60000",
         "y" : 0
      },
      {
         "an" : "PancakeSwap",
         "as" : 1409633.5,
         "b" : "CAKE",
         "ba" : "",
         "c" : "0.0006513",
         "cs" : null,
         "etf" : false,
         "h" : "0.0006699",
         "i" : "0.01000000",
         "l" : "0.0006001",
         "o" : "0.0006248",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "873.59280346",
         "s" : "CAKEBTC",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "BSC"
         ],
         "ts" : "0.0000001",
         "v" : "1409633.5000000",
         "y" : 0
      },
      {
         "an" : "PancakeSwap",
         "as" : 1628924.242,
         "b" : "CAKE",
         "ba" : "",
         "c" : "35.819",
         "cs" : null,
         "etf" : false,
         "h" : "36.642",
         "i" : "0.00100000",
         "l" : "32.000",
         "o" : "33.656",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "55018735.094938",
         "s" : "CAKEBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "1628924.242",
         "y" : 0
      },
      {
         "an" : "PancakeSwap",
         "as" : 73986.425,
         "b" : "CAKE",
         "ba" : "",
         "c" : "25.828",
         "cs" : null,
         "etf" : false,
         "h" : "26.337",
         "i" : "0.00100000",
         "l" : "23.000",
         "o" : "24.315",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "GBP",
         "qa" : "",
         "qn" : "Pound Sterling",
         "qv" : "1818382.729092",
         "s" : "CAKEGBP",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "73986.425",
         "y" : 0
      },
      {
         "an" : "PancakeSwap",
         "as" : 8360849.941,
         "b" : "CAKE",
         "ba" : "",
         "c" : "35.781",
         "cs" : null,
         "etf" : false,
         "h" : "36.600",
         "i" : "0.00100000",
         "l" : "31.902",
         "o" : "33.608",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "280745253.502542",
         "s" : "CAKEUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "8360849.941",
         "y" : 0
      },
      {
         "an" : "Blox",
         "as" : 50387940,
         "b" : "CDT",
         "ba" : "",
         "c" : "0.00000082",
         "cs" : 674579184,
         "etf" : false,
         "h" : "0.00000084",
         "i" : "1.00000000",
         "l" : "0.00000072",
         "o" : "0.00000072",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "39.85897359",
         "s" : "CDTBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "50387940.00000000",
         "y" : 0
      },
      {
         "an" : "Blox",
         "as" : 16440960,
         "b" : "CDT",
         "ba" : "",
         "c" : "0.00001771",
         "cs" : 674579184,
         "etf" : false,
         "h" : "0.00001857",
         "i" : "1.00000000",
         "l" : "0.00001566",
         "o" : "0.00001569",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "278.21783287",
         "s" : "CDTETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "16440960.00000000",
         "y" : 0
      },
      {
         "an" : "Celo",
         "as" : 562365.6,
         "b" : "CELO",
         "ba" : "",
         "c" : "0.00010331",
         "cs" : null,
         "etf" : false,
         "h" : "0.00011516",
         "i" : "0.10000000",
         "l" : "0.00010274",
         "o" : "0.00011266",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "60.23018664",
         "s" : "CELOBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "562365.60000000",
         "y" : 0
      },
      {
         "an" : "Celo",
         "as" : 3846932.01,
         "b" : "CELO",
         "ba" : "",
         "c" : "5.6800",
         "cs" : null,
         "etf" : false,
         "h" : "6.1500",
         "i" : "0.01000000",
         "l" : "5.6000",
         "o" : "6.0679",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "22266880.925753",
         "s" : "CELOUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "3846932.0100",
         "y" : 0
      },
      {
         "an" : "Celer Network",
         "as" : 22638933,
         "b" : "CELR",
         "ba" : "",
         "c" : "0.0001033",
         "cs" : 3325000000,
         "etf" : false,
         "h" : "0.0001189",
         "i" : "1.00000000",
         "l" : "0.0001002",
         "o" : "0.0001051",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "2455.6945856",
         "s" : "CELRBNB",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "22638933.0000000",
         "y" : 0
      },
      {
         "an" : "Celer Network",
         "as" : 239983887,
         "b" : "CELR",
         "ba" : "",
         "c" : "0.00000107",
         "cs" : 3325000000,
         "etf" : false,
         "h" : "0.00000118",
         "i" : "1.00000000",
         "l" : "0.00000100",
         "o" : "0.00000105",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "260.87455401",
         "s" : "CELRBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "239983887.00000000",
         "y" : 0
      },
      {
         "an" : "Celer Network",
         "as" : 670583743.9,
         "b" : "CELR",
         "ba" : "",
         "c" : "0.05905",
         "cs" : 3325000000,
         "etf" : false,
         "h" : "0.06319",
         "i" : "0.10000000",
         "l" : "0.05354",
         "o" : "0.05683",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "39398647.8714",
         "s" : "CELRUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "670583743.90000",
         "y" : 0
      },
      {
         "an" : "Conflux Network",
         "as" : 812333.8,
         "b" : "CFX",
         "ba" : "",
         "c" : "0.00001863",
         "cs" : null,
         "etf" : false,
         "h" : "0.00002050",
         "i" : "0.10000000",
         "l" : "0.00001684",
         "o" : "0.00001712",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "14.96424573",
         "s" : "CFXBTC",
         "st" : "TRADING",
         "tags" : [
            "pow"
         ],
         "ts" : "0.00000001",
         "v" : "812333.80000000",
         "y" : 0
      },
      {
         "an" : "Conflux Network",
         "as" : 352723.284,
         "b" : "CFX",
         "ba" : "",
         "c" : "1.026",
         "cs" : null,
         "etf" : false,
         "h" : "1.076",
         "i" : "0.00100000",
         "l" : "0.900",
         "o" : "0.923",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "343970.329473",
         "s" : "CFXBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow"
         ],
         "ts" : "0.001",
         "v" : "352723.284",
         "y" : 0
      },
      {
         "an" : "Conflux Network",
         "as" : 4872414.16,
         "b" : "CFX",
         "ba" : "",
         "c" : "1.024",
         "cs" : null,
         "etf" : false,
         "h" : "1.076",
         "i" : "0.00100000",
         "l" : "0.902",
         "o" : "0.922",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "4798974.906655",
         "s" : "CFXUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow"
         ],
         "ts" : "0.001",
         "v" : "4872414.160",
         "y" : 0
      },
      {
         "an" : "Chromia",
         "as" : 18499614,
         "b" : "CHR",
         "ba" : "",
         "c" : "0.0005292",
         "cs" : 268455372,
         "etf" : false,
         "h" : "0.0005834",
         "i" : "1.00000000",
         "l" : "0.0005056",
         "o" : "0.0005134",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "10029.4136857",
         "s" : "CHRBNB",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.0000001",
         "v" : "18499614.0000000",
         "y" : 0
      },
      {
         "an" : "Chromia",
         "as" : 16604316,
         "b" : "CHR",
         "ba" : "",
         "c" : "0.00000552",
         "cs" : 268455372,
         "etf" : false,
         "h" : "0.00000578",
         "i" : "1.00000000",
         "l" : "0.00000505",
         "o" : "0.00000514",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "90.16127585",
         "s" : "CHRBTC",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.00000001",
         "v" : "16604316.00000000",
         "y" : 0
      },
      {
         "an" : "Chromia",
         "as" : 155522503,
         "b" : "CHR",
         "ba" : "",
         "c" : "0.30332",
         "cs" : 268455372,
         "etf" : false,
         "h" : "0.31529",
         "i" : "0.10000000",
         "l" : "0.26889",
         "o" : "0.27655",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "45470695.273817",
         "s" : "CHRUSDT",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.00001",
         "v" : "155522503.00000",
         "y" : 0
      },
      {
         "an" : "Chiliz",
         "as" : 10132782,
         "b" : "CHZ",
         "ba" : "",
         "c" : "0.0009072",
         "cs" : 4782431150,
         "etf" : false,
         "h" : "0.0009594",
         "i" : "1.00000000",
         "l" : "0.0008248",
         "o" : "0.0008546",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "9062.9828987",
         "s" : "CHZBNB",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.0000001",
         "v" : "10132782.0000000",
         "y" : 0
      },
      {
         "an" : "Chiliz",
         "as" : 5173444.71,
         "b" : "CHZ",
         "ba" : "",
         "c" : "2.846",
         "cs" : 4782431150,
         "etf" : false,
         "h" : "2.943",
         "i" : "0.01000000",
         "l" : "2.371",
         "o" : "2.548",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BRL",
         "qa" : "",
         "qn" : "Brazilian Real",
         "qv" : "13677366.61218",
         "s" : "CHZBRL",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.001",
         "v" : "5173444.710",
         "y" : 0
      },
      {
         "an" : "Chiliz",
         "as" : 99164588,
         "b" : "CHZ",
         "ba" : "",
         "c" : "0.00000946",
         "cs" : 4782431150,
         "etf" : false,
         "h" : "0.00000976",
         "i" : "1.00000000",
         "l" : "0.00000820",
         "o" : "0.00000856",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "888.38935339",
         "s" : "CHZBTC",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.00000001",
         "v" : "99164588.00000000",
         "y" : 0
      },
      {
         "an" : "Chiliz",
         "as" : 40098780.41,
         "b" : "CHZ",
         "ba" : "",
         "c" : "0.5197",
         "cs" : 4782431150,
         "etf" : false,
         "h" : "0.5385",
         "i" : "0.01000000",
         "l" : "0.4321",
         "o" : "0.4612",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "19369465.87987",
         "s" : "CHZBUSD",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.0001",
         "v" : "40098780.4100",
         "y" : 0
      },
      {
         "an" : "Chiliz",
         "as" : 20299363.53,
         "b" : "CHZ",
         "ba" : "",
         "c" : "0.4313",
         "cs" : 4782431150,
         "etf" : false,
         "h" : "0.4459",
         "i" : "0.01000000",
         "l" : "0.3585",
         "o" : "0.3820",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "8190460.205894",
         "s" : "CHZEUR",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.0001",
         "v" : "20299363.5300",
         "y" : 0
      },
      {
         "an" : "Chiliz",
         "as" : 8979345.09,
         "b" : "CHZ",
         "ba" : "",
         "c" : "0.3744",
         "cs" : 4782431150,
         "etf" : false,
         "h" : "0.3878",
         "i" : "0.01000000",
         "l" : "0.3125",
         "o" : "0.3320",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "GBP",
         "qa" : "",
         "qn" : "Pound Sterling",
         "qv" : "3171237.922003",
         "s" : "CHZGBP",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.0001",
         "v" : "8979345.0900",
         "y" : 0
      },
      {
         "an" : "Chiliz",
         "as" : 135925260,
         "b" : "CHZ",
         "ba" : "",
         "c" : "4.2285",
         "cs" : 4782431150,
         "etf" : false,
         "h" : "4.3800",
         "i" : "1.00000000",
         "l" : "3.5278",
         "o" : "3.7556",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "534175294.4957",
         "s" : "CHZTRY",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.0001",
         "v" : "135925260.0000",
         "y" : 0
      },
      {
         "an" : "Chiliz",
         "as" : 840072183.3,
         "b" : "CHZ",
         "ba" : "",
         "c" : "0.51914",
         "cs" : 4782431150,
         "etf" : false,
         "h" : "0.53800",
         "i" : "0.10000000",
         "l" : "0.43200",
         "o" : "0.45937",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "403825780.176824",
         "s" : "CHZUSDT",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.00001",
         "v" : "840072183.30000",
         "y" : 0
      },
      {
         "an" : "CKB",
         "as" : 174572619,
         "b" : "CKB",
         "ba" : "",
         "c" : "0.00000044",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000045",
         "i" : "1.00000000",
         "l" : "0.00000040",
         "o" : "0.00000042",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "74.53811145",
         "s" : "CKBBTC",
         "st" : "TRADING",
         "tags" : [
            "pow"
         ],
         "ts" : "0.00000001",
         "v" : "174572619.00000000",
         "y" : 0
      },
      {
         "an" : "CKB",
         "as" : 39270147,
         "b" : "CKB",
         "ba" : "",
         "c" : "0.024053",
         "cs" : null,
         "etf" : false,
         "h" : "0.024681",
         "i" : "1.00000000",
         "l" : "0.021707",
         "o" : "0.022815",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "912681.291942",
         "s" : "CKBBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow"
         ],
         "ts" : "0.000001",
         "v" : "39270147.000000",
         "y" : 0
      },
      {
         "an" : "CKB",
         "as" : 517198657,
         "b" : "CKB",
         "ba" : "",
         "c" : "0.023953",
         "cs" : null,
         "etf" : false,
         "h" : "0.024640",
         "i" : "1.00000000",
         "l" : "0.021630",
         "o" : "0.022778",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "11903595.091508",
         "s" : "CKBUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow"
         ],
         "ts" : "0.000001",
         "v" : "517198657.000000",
         "y" : 0
      },
      {
         "an" : "Cindicator",
         "as" : 36758352,
         "b" : "CND",
         "ba" : "",
         "c" : "0.00000066",
         "cs" : 2000000004,
         "etf" : false,
         "h" : "0.00000067",
         "i" : "1.00000000",
         "l" : "0.00000060",
         "o" : "0.00000062",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "23.41334251",
         "s" : "CNDBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "36758352.00000000",
         "y" : 0
      },
      {
         "an" : "Cocos-BCX",
         "as" : 345118.5,
         "b" : "COCOS",
         "ba" : "",
         "c" : "0.001688",
         "cs" : 15715214795,
         "etf" : false,
         "h" : "0.001886",
         "i" : "0.10000000",
         "l" : "0.001645",
         "o" : "0.001664",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "597.5587974",
         "s" : "COCOSBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "NFT",
            "BSC"
         ],
         "ts" : "0.000001",
         "v" : "345118.500000",
         "y" : 0
      },
      {
         "an" : "Cocos-BCX",
         "as" : 3751101.88,
         "b" : "COCOS",
         "ba" : "",
         "c" : "0.9587",
         "cs" : 15715214795,
         "etf" : false,
         "h" : "0.9900",
         "i" : "0.01000000",
         "l" : "0.8635",
         "o" : "0.8990",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "3513913.262834",
         "s" : "COCOSUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "NFT",
            "BSC"
         ],
         "ts" : "0.0001",
         "v" : "3751101.8800",
         "y" : 0
      },
      {
         "an" : "Compound",
         "as" : 22365.888,
         "b" : "COMP",
         "ba" : "",
         "c" : "0.012226",
         "cs" : null,
         "etf" : false,
         "h" : "0.012827",
         "i" : "0.00100000",
         "l" : "0.011531",
         "o" : "0.011850",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "272.89682804",
         "s" : "COMPBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.000001",
         "v" : "22365.888000",
         "y" : 0
      },
      {
         "an" : "Compound",
         "as" : 7632.78739,
         "b" : "COMP",
         "ba" : "",
         "c" : "673.19",
         "cs" : null,
         "etf" : false,
         "h" : "689.64",
         "i" : "0.00001000",
         "l" : "612.70",
         "o" : "638.09",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "5005924.3833866",
         "s" : "COMPBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.01",
         "v" : "7632.79",
         "y" : 0
      },
      {
         "an" : "Compound",
         "as" : 91419.24855,
         "b" : "COMP",
         "ba" : "",
         "c" : "671.73",
         "cs" : null,
         "etf" : false,
         "h" : "689.24",
         "i" : "0.00001000",
         "l" : "611.75",
         "o" : "637.33",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "59931330.1663878",
         "s" : "COMPUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.01",
         "v" : "91419.25",
         "y" : 0
      },
      {
         "an" : "Contentos",
         "as" : 57337778,
         "b" : "COS",
         "ba" : "",
         "c" : "0.00005444",
         "cs" : 1310656724,
         "etf" : false,
         "h" : "0.00005654",
         "i" : "1.00000000",
         "l" : "0.00005261",
         "o" : "0.00005320",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "3120.65491806",
         "s" : "COSBNB",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "57337778.00000000",
         "y" : 0
      },
      {
         "an" : "Contentos",
         "as" : 53128264,
         "b" : "COS",
         "ba" : "",
         "c" : "0.00000056",
         "cs" : 1310656724,
         "etf" : false,
         "h" : "0.00000058",
         "i" : "1.00000000",
         "l" : "0.00000052",
         "o" : "0.00000053",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "28.99347447",
         "s" : "COSBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "53128264.00000000",
         "y" : 0
      },
      {
         "an" : "Contentos",
         "as" : 163348680.7,
         "b" : "COS",
         "ba" : "",
         "c" : "0.03119",
         "cs" : 1310656724,
         "etf" : false,
         "h" : "0.03150",
         "i" : "0.10000000",
         "l" : "0.02765",
         "o" : "0.02863",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "4813676.367844",
         "s" : "COSUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "163348680.70000",
         "y" : 0
      },
      {
         "an" : "COTI",
         "as" : 6234166,
         "b" : "COTI",
         "ba" : "",
         "c" : "0.0006515",
         "cs" : 312250374,
         "etf" : false,
         "h" : "0.0006840",
         "i" : "1.00000000",
         "l" : "0.0005613",
         "o" : "0.0005730",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "3896.9454545",
         "s" : "COTIBNB",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "6234166.0000000",
         "y" : 0
      },
      {
         "an" : "COTI",
         "as" : 48625316,
         "b" : "COTI",
         "ba" : "",
         "c" : "0.00000681",
         "cs" : 312250374,
         "etf" : false,
         "h" : "0.00000689",
         "i" : "1.00000000",
         "l" : "0.00000560",
         "o" : "0.00000575",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "302.52570175",
         "s" : "COTIBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "48625316.00000000",
         "y" : 0
      },
      {
         "an" : "COTI",
         "as" : 109112026.3,
         "b" : "COTI",
         "ba" : "",
         "c" : "0.37418",
         "cs" : 312250374,
         "etf" : false,
         "h" : "0.38000",
         "i" : "0.10000000",
         "l" : "0.29659",
         "o" : "0.30973",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "36829623.133961",
         "s" : "COTIUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "109112026.30000",
         "y" : 0
      },
      {
         "an" : "COVER Protocol",
         "as" : 871.02625,
         "b" : "COVER",
         "ba" : "",
         "c" : "489.43",
         "cs" : null,
         "etf" : false,
         "h" : "495.14",
         "i" : "0.00001000",
         "l" : "425.97",
         "o" : "446.96",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "403328.3031975",
         "s" : "COVERBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.01",
         "v" : "871.03",
         "y" : 0
      },
      {
         "an" : "COVER Protocol",
         "as" : 273.3233,
         "b" : "COVER",
         "ba" : "",
         "c" : "0.1913",
         "cs" : null,
         "etf" : false,
         "h" : "0.1944",
         "i" : "0.00010000",
         "l" : "0.1723",
         "o" : "0.1796",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "50.11850421",
         "s" : "COVERETH",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "273.3233",
         "y" : 0
      },
      {
         "an" : "Cream Finance",
         "as" : 4952.78,
         "b" : "CREAM",
         "ba" : "",
         "c" : "0.2084",
         "cs" : null,
         "etf" : false,
         "h" : "0.3043",
         "i" : "0.01000000",
         "l" : "0.1908",
         "o" : "0.1932",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1047.14243",
         "s" : "CREAMBNB",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone"
         ],
         "ts" : "0.0001",
         "v" : "4952.7800",
         "y" : 0
      },
      {
         "an" : "Cream Finance",
         "as" : 17125.1772,
         "b" : "CREAM",
         "ba" : "",
         "c" : "119.53",
         "cs" : null,
         "etf" : false,
         "h" : "128.12",
         "i" : "0.00010000",
         "l" : "101.27",
         "o" : "103.75",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1926549.742862",
         "s" : "CREAMBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone"
         ],
         "ts" : "0.01",
         "v" : "17125.18",
         "y" : 0
      },
      {
         "an" : "Curve",
         "as" : 130460.36,
         "b" : "CRV",
         "ba" : "",
         "c" : "0.005119",
         "cs" : null,
         "etf" : false,
         "h" : "0.005596",
         "i" : "0.01000000",
         "l" : "0.004983",
         "o" : "0.005190",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "679.98483366",
         "s" : "CRVBNB",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.000001",
         "v" : "130460.360000",
         "y" : 0
      },
      {
         "an" : "Curve",
         "as" : 4103822.61,
         "b" : "CRV",
         "ba" : "",
         "c" : "0.00005358",
         "cs" : null,
         "etf" : false,
         "h" : "0.00005650",
         "i" : "0.01000000",
         "l" : "0.00004960",
         "o" : "0.00005189",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "214.82344623",
         "s" : "CRVBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "4103822.61000000",
         "y" : 0
      },
      {
         "an" : "Curve",
         "as" : 1022818.13,
         "b" : "CRV",
         "ba" : "",
         "c" : "2.946",
         "cs" : null,
         "etf" : false,
         "h" : "3.039",
         "i" : "0.00100000",
         "l" : "2.622",
         "o" : "2.798",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2885549.935152",
         "s" : "CRVBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "1022818.130",
         "y" : 0
      },
      {
         "an" : "Curve",
         "as" : 23824766.703,
         "b" : "CRV",
         "ba" : "",
         "c" : "2.943",
         "cs" : null,
         "etf" : false,
         "h" : "3.038",
         "i" : "0.00100000",
         "l" : "2.619",
         "o" : "2.792",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "67073715.496523",
         "s" : "CRVUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "23824766.703",
         "y" : 0
      },
      {
         "an" : "CertiK",
         "as" : 321874,
         "b" : "CTK",
         "ba" : "",
         "c" : "0.004863",
         "cs" : null,
         "etf" : false,
         "h" : "0.006200",
         "i" : "0.10000000",
         "l" : "0.004429",
         "o" : "0.004636",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1564.2793222",
         "s" : "CTKBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "BSC"
         ],
         "ts" : "0.000001",
         "v" : "321874.000000",
         "y" : 0
      },
      {
         "an" : "CertiK",
         "as" : 810428,
         "b" : "CTK",
         "ba" : "",
         "c" : "0.00005070",
         "cs" : null,
         "etf" : false,
         "h" : "0.00005300",
         "i" : "0.10000000",
         "l" : "0.00004380",
         "o" : "0.00004611",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "39.7887151",
         "s" : "CTKBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "BSC"
         ],
         "ts" : "0.00000001",
         "v" : "810428.00000000",
         "y" : 0
      },
      {
         "an" : "CertiK",
         "as" : 476009.91,
         "b" : "CTK",
         "ba" : "",
         "c" : "2.7846",
         "cs" : null,
         "etf" : false,
         "h" : "2.9094",
         "i" : "0.01000000",
         "l" : "2.3366",
         "o" : "2.4853",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1235784.988485",
         "s" : "CTKBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "BSC"
         ],
         "ts" : "0.0001",
         "v" : "476009.9100",
         "y" : 0
      },
      {
         "an" : "CertiK",
         "as" : 7708775.08,
         "b" : "CTK",
         "ba" : "",
         "c" : "2.7870",
         "cs" : null,
         "etf" : false,
         "h" : "2.9184",
         "i" : "0.01000000",
         "l" : "2.3162",
         "o" : "2.4831",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "20166194.845759",
         "s" : "CTKUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "BSC"
         ],
         "ts" : "0.0001",
         "v" : "7708775.0800",
         "y" : 0
      },
      {
         "an" : "Cartesi",
         "as" : 3943691,
         "b" : "CTSI",
         "ba" : "",
         "c" : "0.001194",
         "cs" : 192733334,
         "etf" : false,
         "h" : "0.001370",
         "i" : "1.00000000",
         "l" : "0.000939",
         "o" : "0.000962",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "4810.181184",
         "s" : "CTSIBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "3943691.000000",
         "y" : 0
      },
      {
         "an" : "Cartesi",
         "as" : 19398262,
         "b" : "CTSI",
         "ba" : "",
         "c" : "0.00001244",
         "cs" : 192733334,
         "etf" : false,
         "h" : "0.00001358",
         "i" : "1.00000000",
         "l" : "0.00000937",
         "o" : "0.00000968",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "232.77870469",
         "s" : "CTSIBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "19398262.00000000",
         "y" : 0
      },
      {
         "an" : "Cartesi",
         "as" : 6388917.2,
         "b" : "CTSI",
         "ba" : "",
         "c" : "0.68140",
         "cs" : 192733334,
         "etf" : false,
         "h" : "0.74104",
         "i" : "0.10000000",
         "l" : "0.49940",
         "o" : "0.51932",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "4197059.538301",
         "s" : "CTSIBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "6388917.20000",
         "y" : 0
      },
      {
         "an" : "Cartesi",
         "as" : 96315746.1,
         "b" : "CTSI",
         "ba" : "",
         "c" : "0.68214",
         "cs" : 192733334,
         "etf" : false,
         "h" : "0.74000",
         "i" : "0.10000000",
         "l" : "0.49837",
         "o" : "0.52114",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "62630081.19062",
         "s" : "CTSIUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "96315746.10000",
         "y" : 0
      },
      {
         "an" : "Cortex",
         "as" : 3122029,
         "b" : "CTXC",
         "ba" : "",
         "c" : "0.00000637",
         "cs" : 249938725,
         "etf" : false,
         "h" : "0.00000680",
         "i" : "1.00000000",
         "l" : "0.00000594",
         "o" : "0.00000599",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "19.76582815",
         "s" : "CTXCBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "3122029.00000000",
         "y" : 0
      },
      {
         "an" : "Cortex",
         "as" : 11676759.34,
         "b" : "CTXC",
         "ba" : "",
         "c" : "0.3500",
         "cs" : 249938725,
         "etf" : false,
         "h" : "0.3718",
         "i" : "0.01000000",
         "l" : "0.3130",
         "o" : "0.3218",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "3989083.352583",
         "s" : "CTXCUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "11676759.3400",
         "y" : 0
      },
      {
         "an" : "Civic",
         "as" : 6049024,
         "b" : "CVC",
         "ba" : "",
         "c" : "0.00000963",
         "cs" : 342699966,
         "etf" : false,
         "h" : "0.00001033",
         "i" : "1.00000000",
         "l" : "0.00000900",
         "o" : "0.00000916",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "58.94251519",
         "s" : "CVCBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "6049024.00000000",
         "y" : 0
      },
      {
         "an" : "Civic",
         "as" : 857874,
         "b" : "CVC",
         "ba" : "",
         "c" : "0.00020659",
         "cs" : 342699966,
         "etf" : false,
         "h" : "0.00022148",
         "i" : "1.00000000",
         "l" : "0.00019364",
         "o" : "0.00019581",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "178.5821859",
         "s" : "CVCETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "857874.00000000",
         "y" : 0
      },
      {
         "an" : "Civic",
         "as" : 35293253,
         "b" : "CVC",
         "ba" : "",
         "c" : "0.52954",
         "cs" : 342699966,
         "etf" : false,
         "h" : "0.54999",
         "i" : "0.10000000",
         "l" : "0.48034",
         "o" : "0.49128",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "18458201.417305",
         "s" : "CVCUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "35293253.00000",
         "y" : 0
      },
      {
         "an" : "PowerPool",
         "as" : 89926.01,
         "b" : "CVP",
         "ba" : "",
         "c" : "2.7045",
         "cs" : null,
         "etf" : false,
         "h" : "2.7461",
         "i" : "0.01000000",
         "l" : "2.4982",
         "o" : "2.6236",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "237427.7278",
         "s" : "CVPBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "89926.0100",
         "y" : 0
      },
      {
         "an" : "PowerPool",
         "as" : 47893.4,
         "b" : "CVP",
         "ba" : "",
         "c" : "0.001055",
         "cs" : null,
         "etf" : false,
         "h" : "0.001080",
         "i" : "0.01000000",
         "l" : "0.001024",
         "o" : "0.001047",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "50.31015423",
         "s" : "CVPETH",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.000001",
         "v" : "47893.400000",
         "y" : 0
      },
      {
         "an" : "Dash",
         "as" : 1244.843,
         "b" : "DASH",
         "ba" : "",
         "c" : "0.5048",
         "cs" : 10097402,
         "etf" : false,
         "h" : "0.5419",
         "i" : "0.00100000",
         "l" : "0.5017",
         "o" : "0.5091",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "648.4911421",
         "s" : "DASHBNB",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "1244.8430",
         "y" : 0
      },
      {
         "an" : "Dash",
         "as" : 27763.509,
         "b" : "DASH",
         "ba" : "",
         "c" : "0.005247",
         "cs" : 10097402,
         "etf" : false,
         "h" : "0.005357",
         "i" : "0.00100000",
         "l" : "0.005033",
         "o" : "0.005116",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "144.33209732",
         "s" : "DASHBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "27763.509000",
         "y" : 0
      },
      {
         "an" : "Dash",
         "as" : 13436.56155,
         "b" : "DASH",
         "ba" : "",
         "c" : "288.75",
         "cs" : 10097402,
         "etf" : false,
         "h" : "291.10",
         "i" : "0.00001000",
         "l" : "266.67",
         "o" : "274.81",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "3770006.9087496",
         "s" : "DASHBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "13436.57",
         "y" : 0
      },
      {
         "an" : "Dash",
         "as" : 2782.123,
         "b" : "DASH",
         "ba" : "",
         "c" : "0.11253",
         "cs" : 10097402,
         "etf" : false,
         "h" : "0.11459",
         "i" : "0.00100000",
         "l" : "0.10834",
         "o" : "0.10947",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "311.9305322",
         "s" : "DASHETH",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "2782.12300",
         "y" : 0
      },
      {
         "an" : "Dash",
         "as" : 159385.33048,
         "b" : "DASH",
         "ba" : "",
         "c" : "288.23",
         "cs" : 10097402,
         "etf" : false,
         "h" : "290.88",
         "i" : "0.00001000",
         "l" : "266.32",
         "o" : "274.99",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "44561552.5080686",
         "s" : "DASHUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "159385.34",
         "y" : 0
      },
      {
         "an" : "Streamr DATAcoin",
         "as" : 14965619,
         "b" : "DATA",
         "ba" : "",
         "c" : "0.00000274",
         "cs" : 869250679,
         "etf" : false,
         "h" : "0.00000281",
         "i" : "1.00000000",
         "l" : "0.00000244",
         "o" : "0.00000250",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "39.4165683",
         "s" : "DATABTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "14965619.00000000",
         "y" : 0
      },
      {
         "an" : "Streamr DATAcoin",
         "as" : 1408306.6,
         "b" : "DATA",
         "ba" : "",
         "c" : "0.15000",
         "cs" : 869250679,
         "etf" : false,
         "h" : "0.15483",
         "i" : "0.10000000",
         "l" : "0.12940",
         "o" : "0.13498",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "200498.290604",
         "s" : "DATABUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "1408306.60000",
         "y" : 0
      },
      {
         "an" : "Streamr DATAcoin",
         "as" : 1170105,
         "b" : "DATA",
         "ba" : "",
         "c" : "0.00005948",
         "cs" : 869250679,
         "etf" : false,
         "h" : "0.00006013",
         "i" : "1.00000000",
         "l" : "0.00005253",
         "o" : "0.00005375",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "65.29323691",
         "s" : "DATAETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "1170105.00000000",
         "y" : 0
      },
      {
         "an" : "Streamr DATAcoin",
         "as" : 28134079.3,
         "b" : "DATA",
         "ba" : "",
         "c" : "0.15032",
         "cs" : 869250679,
         "etf" : false,
         "h" : "0.15331",
         "i" : "0.10000000",
         "l" : "0.12931",
         "o" : "0.13475",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "3970462.093193",
         "s" : "DATAUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "28134079.30000",
         "y" : 0
      },
      {
         "an" : "Decred",
         "as" : 13093.941,
         "b" : "DCR",
         "ba" : "",
         "c" : "0.003851",
         "cs" : 12864597,
         "etf" : false,
         "h" : "0.003890",
         "i" : "0.00100000",
         "l" : "0.003701",
         "o" : "0.003814",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "50.15960597",
         "s" : "DCRBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "13093.941000",
         "y" : 0
      },
      {
         "an" : "Decred",
         "as" : 8864.9195,
         "b" : "DCR",
         "ba" : "",
         "c" : "211.32",
         "cs" : 12864597,
         "etf" : false,
         "h" : "213.68",
         "i" : "0.00010000",
         "l" : "200.01",
         "o" : "205.00",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "1832092.8476342",
         "s" : "DCRUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "8864.92",
         "y" : 0
      },
      {
         "an" : "Dego Finance",
         "as" : 401862.17,
         "b" : "DEGO",
         "ba" : "",
         "c" : "0.0002186",
         "cs" : null,
         "etf" : false,
         "h" : "0.0002228",
         "i" : "0.01000000",
         "l" : "0.0001941",
         "o" : "0.0002093",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "82.63590403",
         "s" : "DEGOBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "NFT"
         ],
         "ts" : "0.0000001",
         "v" : "401862.1700000",
         "y" : 0
      },
      {
         "an" : "Dego Finance",
         "as" : 152303.104,
         "b" : "DEGO",
         "ba" : "",
         "c" : "12.058",
         "cs" : null,
         "etf" : false,
         "h" : "12.335",
         "i" : "0.00100000",
         "l" : "10.238",
         "o" : "11.241",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1705066.693294",
         "s" : "DEGOBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "NFT"
         ],
         "ts" : "0.001",
         "v" : "152303.104",
         "y" : 0
      },
      {
         "an" : "Dego Finance",
         "as" : 1271570.476,
         "b" : "DEGO",
         "ba" : "",
         "c" : "12.050",
         "cs" : null,
         "etf" : false,
         "h" : "12.271",
         "i" : "0.00100000",
         "l" : "10.200",
         "o" : "11.246",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "14184517.190948",
         "s" : "DEGOUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "NFT"
         ],
         "ts" : "0.001",
         "v" : "1271570.476",
         "y" : 0
      },
      {
         "an" : "DENT",
         "as" : 1815467442,
         "b" : "DENT",
         "ba" : "",
         "c" : "0.00000473",
         "cs" : 94033927360,
         "etf" : false,
         "h" : "0.00000519",
         "i" : "1.00000000",
         "l" : "0.00000368",
         "o" : "0.00000378",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "8046.30426929",
         "s" : "DENTETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "1815467442.00000000",
         "y" : 0
      },
      {
         "an" : "DENT",
         "as" : 38489201922,
         "b" : "DENT",
         "ba" : "",
         "c" : "0.0120852",
         "cs" : 94033927360,
         "etf" : false,
         "h" : "0.0128900",
         "i" : "1.00000000",
         "l" : "0.0090600",
         "o" : "0.0094432",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "422935694.8630972",
         "s" : "DENTUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "38489201922.0000000",
         "y" : 0
      },
      {
         "an" : "DeXe",
         "as" : 611256.833,
         "b" : "DEXE",
         "ba" : "",
         "c" : "19.995",
         "cs" : null,
         "etf" : false,
         "h" : "21.050",
         "i" : "0.00100000",
         "l" : "19.585",
         "o" : "19.793",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "12543034.343916",
         "s" : "DEXEBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.001",
         "v" : "611256.833",
         "y" : 0
      },
      {
         "an" : "DeXe",
         "as" : 505517.17,
         "b" : "DEXE",
         "ba" : "",
         "c" : "0.007834",
         "cs" : null,
         "etf" : false,
         "h" : "0.008461",
         "i" : "0.01000000",
         "l" : "0.007773",
         "o" : "0.007901",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "4126.26832778",
         "s" : "DEXEETH",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.000001",
         "v" : "505517.170000",
         "y" : 0
      },
      {
         "an" : "dForce",
         "as" : 1613337.05,
         "b" : "DF",
         "ba" : "",
         "c" : "0.3255",
         "cs" : null,
         "etf" : false,
         "h" : "0.3350",
         "i" : "0.01000000",
         "l" : "0.3037",
         "o" : "0.3181",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "515274.713716",
         "s" : "DFBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "1613337.0500",
         "y" : 0
      },
      {
         "an" : "dForce",
         "as" : 958485,
         "b" : "DF",
         "ba" : "",
         "c" : "0.0001277",
         "cs" : null,
         "etf" : false,
         "h" : "0.0001326",
         "i" : "1.00000000",
         "l" : "0.0001220",
         "o" : "0.0001268",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "121.7446381",
         "s" : "DFETH",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0000001",
         "v" : "958485.0000000",
         "y" : 0
      },
      {
         "an" : "DigiByte",
         "as" : 4902176,
         "b" : "DGB",
         "ba" : "",
         "c" : "0.0002342",
         "cs" : 10126279276,
         "etf" : false,
         "h" : "0.0002479",
         "i" : "1.00000000",
         "l" : "0.0002239",
         "o" : "0.0002268",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1157.4253602",
         "s" : "DGBBNB",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "4902176.0000000",
         "y" : 0
      },
      {
         "an" : "DigiByte",
         "as" : 7841641,
         "b" : "DGB",
         "ba" : "",
         "c" : "0.13456",
         "cs" : 10126279276,
         "etf" : false,
         "h" : "0.13940",
         "i" : "0.10000000",
         "l" : "0.11858",
         "o" : "0.12253",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1004793.720202",
         "s" : "DGBBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "7841641.00000",
         "y" : 0
      },
      {
         "an" : "DigiByte",
         "as" : 257719238.8,
         "b" : "DGB",
         "ba" : "",
         "c" : "0.13458",
         "cs" : 10126279276,
         "etf" : false,
         "h" : "0.14000",
         "i" : "0.10000000",
         "l" : "0.11836",
         "o" : "0.12223",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "32881992.883609",
         "s" : "DGBUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "257719238.80000",
         "y" : 0
      },
      {
         "an" : "DIA",
         "as" : 95142.7,
         "b" : "DIA",
         "ba" : "",
         "c" : "0.006714",
         "cs" : null,
         "etf" : false,
         "h" : "0.007317",
         "i" : "0.01000000",
         "l" : "0.006578",
         "o" : "0.006797",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "654.73468165",
         "s" : "DIABNB",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "95142.700000",
         "y" : 0
      },
      {
         "an" : "DIA",
         "as" : 1177656.5,
         "b" : "DIA",
         "ba" : "",
         "c" : "0.0000699",
         "cs" : null,
         "etf" : false,
         "h" : "0.0000720",
         "i" : "0.01000000",
         "l" : "0.0000652",
         "o" : "0.0000674",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "81.43968772",
         "s" : "DIABTC",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "1177656.5000000",
         "y" : 0
      },
      {
         "an" : "DIA",
         "as" : 317033.109,
         "b" : "DIA",
         "ba" : "",
         "c" : "3.847",
         "cs" : null,
         "etf" : false,
         "h" : "3.965",
         "i" : "0.00100000",
         "l" : "3.505",
         "o" : "3.640",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1188691.9846",
         "s" : "DIABUSD",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "317033.109",
         "y" : 0
      },
      {
         "an" : "DIA",
         "as" : 3095077.685,
         "b" : "DIA",
         "ba" : "",
         "c" : "3.837",
         "cs" : null,
         "etf" : false,
         "h" : "3.958",
         "i" : "0.00100000",
         "l" : "3.500",
         "o" : "3.628",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "11546262.206211",
         "s" : "DIAUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "3095077.685",
         "y" : 0
      },
      {
         "an" : "Agrello",
         "as" : 3603665,
         "b" : "DLT",
         "ba" : "",
         "c" : "0.00000355",
         "cs" : 101124580,
         "etf" : false,
         "h" : "0.00000364",
         "i" : "1.00000000",
         "l" : "0.00000338",
         "o" : "0.00000345",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "12.63682213",
         "s" : "DLTBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "3603665.00000000",
         "y" : 0
      },
      {
         "an" : "district0x",
         "as" : 7523177,
         "b" : "DNT",
         "ba" : "",
         "c" : "0.00000552",
         "cs" : 600000000,
         "etf" : false,
         "h" : "0.00000580",
         "i" : "1.00000000",
         "l" : "0.00000522",
         "o" : "0.00000523",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "41.11374747",
         "s" : "DNTBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "7523177.00000000",
         "y" : 0
      },
      {
         "an" : "district0x",
         "as" : 861074.7,
         "b" : "DNT",
         "ba" : "",
         "c" : "0.30370",
         "cs" : 600000000,
         "etf" : false,
         "h" : "0.31867",
         "i" : "0.10000000",
         "l" : "0.27908",
         "o" : "0.28245",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "252452.642017",
         "s" : "DNTBUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "861074.70000",
         "y" : 0
      },
      {
         "an" : "district0x",
         "as" : 18960607.4,
         "b" : "DNT",
         "ba" : "",
         "c" : "0.30304",
         "cs" : 600000000,
         "etf" : false,
         "h" : "0.31875",
         "i" : "0.10000000",
         "l" : "0.27828",
         "o" : "0.28293",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "5585447.509703",
         "s" : "DNTUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "18960607.40000",
         "y" : 0
      },
      {
         "an" : "DOCK",
         "as" : 106423469,
         "b" : "DOCK",
         "ba" : "",
         "c" : "0.00000199",
         "cs" : 661851088,
         "etf" : false,
         "h" : "0.00000239",
         "i" : "1.00000000",
         "l" : "0.00000185",
         "o" : "0.00000188",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "224.28039098",
         "s" : "DOCKBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "106423469.00000000",
         "y" : 0
      },
      {
         "an" : "DOCK",
         "as" : 323094239,
         "b" : "DOCK",
         "ba" : "",
         "c" : "0.109607",
         "cs" : 661851088,
         "etf" : false,
         "h" : "0.127647",
         "i" : "1.00000000",
         "l" : "0.098990",
         "o" : "0.101357",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "36535081.049336",
         "s" : "DOCKUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "323094239.000000",
         "y" : 0
      },
      {
         "an" : "DODO",
         "as" : 588324,
         "b" : "DODO",
         "ba" : "",
         "c" : "0.00006210",
         "cs" : null,
         "etf" : false,
         "h" : "0.00006769",
         "i" : "0.10000000",
         "l" : "0.00005636",
         "o" : "0.00005781",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "36.07602937",
         "s" : "DODOBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC",
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "588324.00000000",
         "y" : 0
      },
      {
         "an" : "DODO",
         "as" : 1213742.139,
         "b" : "DODO",
         "ba" : "",
         "c" : "3.415",
         "cs" : null,
         "etf" : false,
         "h" : "3.560",
         "i" : "0.00100000",
         "l" : "3.003",
         "o" : "3.116",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "3997832.926422",
         "s" : "DODOBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC",
            "defi"
         ],
         "ts" : "0.001",
         "v" : "1213742.139",
         "y" : 0
      },
      {
         "an" : "DODO",
         "as" : 9581447.86,
         "b" : "DODO",
         "ba" : "",
         "c" : "3.410",
         "cs" : null,
         "etf" : false,
         "h" : "3.556",
         "i" : "0.00100000",
         "l" : "3.000",
         "o" : "3.112",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "31302660.960735",
         "s" : "DODOUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC",
            "defi"
         ],
         "ts" : "0.001",
         "v" : "9581447.860",
         "y" : 0
      },
      {
         "an" : "Dogecoin",
         "as" : 13143797.3,
         "b" : "DOGE",
         "ba" : "",
         "c" : "0.35580",
         "cs" : 129348760639,
         "etf" : false,
         "h" : "0.36237",
         "i" : "0.10000000",
         "l" : "0.33392",
         "o" : "0.35660",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "AUD",
         "qa" : "",
         "qn" : "Australian Dollar",
         "qv" : "4608469.723099",
         "s" : "DOGEAUD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "13143797.30000",
         "y" : 0
      },
      {
         "an" : "Dogecoin",
         "as" : 15161735.74,
         "b" : "DOGE",
         "ba" : "",
         "c" : "1.5045",
         "cs" : 129348760639,
         "etf" : false,
         "h" : "1.5500",
         "i" : "0.01000000",
         "l" : "1.4201",
         "o" : "1.5211",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BRL",
         "qa" : "",
         "qn" : "Brazilian Real",
         "qv" : "22485193.655775",
         "s" : "DOGEBRL",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "15161735.7400",
         "y" : 0
      },
      {
         "an" : "Dogecoin",
         "as" : 417248378,
         "b" : "DOGE",
         "ba" : "",
         "c" : "0.00000502",
         "cs" : 129348760639,
         "etf" : false,
         "h" : "0.00000524",
         "i" : "1.00000000",
         "l" : "0.00000489",
         "o" : "0.00000512",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "2095.51035827",
         "s" : "DOGEBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "417248378.00000000",
         "y" : 0
      },
      {
         "an" : "Dogecoin",
         "as" : 392576954.8,
         "b" : "DOGE",
         "ba" : "",
         "c" : "0.27581",
         "cs" : 129348760639,
         "etf" : false,
         "h" : "0.28100",
         "i" : "0.10000000",
         "l" : "0.25930",
         "o" : "0.27565",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "106421342.507275",
         "s" : "DOGEBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "392576954.80000",
         "y" : 0
      },
      {
         "an" : "Dogecoin",
         "as" : 197533778.5,
         "b" : "DOGE",
         "ba" : "",
         "c" : "0.22849",
         "cs" : 129348760639,
         "etf" : false,
         "h" : "0.23296",
         "i" : "0.10000000",
         "l" : "0.21500",
         "o" : "0.22847",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "44509297.970576",
         "s" : "DOGEEUR",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "197533778.50000",
         "y" : 0
      },
      {
         "an" : "Dogecoin",
         "as" : 67983379.1,
         "b" : "DOGE",
         "ba" : "",
         "c" : "0.19887",
         "cs" : 129348760639,
         "etf" : false,
         "h" : "0.20299",
         "i" : "0.10000000",
         "l" : "0.18703",
         "o" : "0.19865",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "GBP",
         "qa" : "",
         "qn" : "Pound Sterling",
         "qv" : "13301595.150197",
         "s" : "DOGEGBP",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "67983379.10000",
         "y" : 0
      },
      {
         "an" : "Dogecoin",
         "as" : 774730.7,
         "b" : "DOGE",
         "ba" : "",
         "c" : "20.759",
         "cs" : 129348760639,
         "etf" : false,
         "h" : "21.296",
         "i" : "0.10000000",
         "l" : "19.630",
         "o" : "20.854",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "RUB",
         "qa" : "",
         "qn" : "Russian Ruble",
         "qv" : "15874438.9582",
         "s" : "DOGERUB",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "774730.700",
         "y" : 0
      },
      {
         "an" : "Dogecoin",
         "as" : 235365893,
         "b" : "DOGE",
         "ba" : "",
         "c" : "2.2428",
         "cs" : 129348760639,
         "etf" : false,
         "h" : "2.2900",
         "i" : "1.00000000",
         "l" : "2.1110",
         "o" : "2.2400",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "519622033.2879",
         "s" : "DOGETRY",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "235365893.0000",
         "y" : 0
      },
      {
         "an" : "Dogecoin",
         "as" : 4242593338.8,
         "b" : "DOGE",
         "ba" : "",
         "c" : "0.27532",
         "cs" : 129348760639,
         "etf" : false,
         "h" : "0.28077",
         "i" : "0.10000000",
         "l" : "0.25900",
         "o" : "0.27497",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "1148658136.2278746",
         "s" : "DOGEUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "4242593338.80000",
         "y" : 0
      },
      {
         "an" : "Polkadot",
         "as" : 10566.652,
         "b" : "DOT",
         "ba" : "",
         "c" : "508065",
         "cs" : null,
         "etf" : false,
         "h" : "515775",
         "i" : "0.00100000",
         "l" : "475752",
         "o" : "487500",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BIDR",
         "qa" : "",
         "qn" : "BIDR",
         "qv" : "5263614336.67",
         "s" : "DOTBIDR",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "Polkadot"
         ],
         "ts" : "1",
         "v" : "10567",
         "y" : 0
      },
      {
         "an" : "Polkadot",
         "as" : 537368.4,
         "b" : "DOT",
         "ba" : "",
         "c" : "0.05978",
         "cs" : null,
         "etf" : false,
         "h" : "0.06427",
         "i" : "0.10000000",
         "l" : "0.05978",
         "o" : "0.06100",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "33279.663539",
         "s" : "DOTBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "Polkadot"
         ],
         "ts" : "0.00001",
         "v" : "537368.40000",
         "y" : 0
      },
      {
         "an" : "Polkadot",
         "as" : 9346.4,
         "b" : "DOT",
         "ba" : "",
         "c" : "186.98",
         "cs" : null,
         "etf" : false,
         "h" : "191.00",
         "i" : "0.01000000",
         "l" : "175.01",
         "o" : "182.11",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BRL",
         "qa" : "",
         "qn" : "Brazilian Real",
         "qv" : "1711421.5159",
         "s" : "DOTBRL",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "Polkadot"
         ],
         "ts" : "0.01",
         "v" : "9346.40",
         "y" : 0
      },
      {
         "an" : "Polkadot",
         "as" : 1647579.5,
         "b" : "DOT",
         "ba" : "",
         "c" : "0.00062370",
         "cs" : null,
         "etf" : false,
         "h" : "0.00063558",
         "i" : "0.10000000",
         "l" : "0.00060200",
         "o" : "0.00061075",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "1020.4431676",
         "s" : "DOTBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "Polkadot"
         ],
         "ts" : "0.00000001",
         "v" : "1647579.50000000",
         "y" : 0
      },
      {
         "an" : "Polkadot",
         "as" : 1038678.026,
         "b" : "DOT",
         "ba" : "",
         "c" : "34.311",
         "cs" : null,
         "etf" : false,
         "h" : "35.000",
         "i" : "0.00100000",
         "l" : "31.929",
         "o" : "32.876",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "34805998.80097",
         "s" : "DOTBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "Polkadot"
         ],
         "ts" : "0.001",
         "v" : "1038678.026",
         "y" : 0
      },
      {
         "an" : "DOTDOWN",
         "as" : 1031479.13,
         "b" : "DOTDOWN",
         "ba" : "",
         "c" : "1.5197",
         "cs" : null,
         "etf" : true,
         "h" : "1.7894",
         "i" : "0.01000000",
         "l" : "1.4251",
         "o" : "1.6800",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "1650037.441391",
         "s" : "DOTDOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "1031479.1300",
         "y" : 0
      },
      {
         "an" : "Polkadot",
         "as" : 250186.282,
         "b" : "DOT",
         "ba" : "",
         "c" : "28.431",
         "cs" : null,
         "etf" : false,
         "h" : "28.994",
         "i" : "0.00100000",
         "l" : "26.474",
         "o" : "27.252",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "6940612.397051",
         "s" : "DOTEUR",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "Polkadot"
         ],
         "ts" : "0.001",
         "v" : "250186.282",
         "y" : 0
      },
      {
         "an" : "Polkadot",
         "as" : 62799.862,
         "b" : "DOT",
         "ba" : "",
         "c" : "24.761",
         "cs" : null,
         "etf" : false,
         "h" : "25.317",
         "i" : "0.00100000",
         "l" : "23.048",
         "o" : "23.760",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "GBP",
         "qa" : "",
         "qn" : "Pound Sterling",
         "qv" : "1514922.853113",
         "s" : "DOTGBP",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "Polkadot"
         ],
         "ts" : "0.001",
         "v" : "62799.862",
         "y" : 0
      },
      {
         "an" : "Polkadot",
         "as" : 39510.66,
         "b" : "DOT",
         "ba" : "",
         "c" : "279.2",
         "cs" : null,
         "etf" : false,
         "h" : "284.5",
         "i" : "0.01000000",
         "l" : "260.4",
         "o" : "268.0",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "10846441.955",
         "s" : "DOTTRY",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "Polkadot"
         ],
         "ts" : "0.1",
         "v" : "39510.7",
         "y" : 0
      },
      {
         "an" : "DOTUP",
         "as" : 113974.31,
         "b" : "DOTUP",
         "ba" : "",
         "c" : "91.865",
         "cs" : null,
         "etf" : true,
         "h" : "95.241",
         "i" : "0.01000000",
         "l" : "79.000",
         "o" : "83.900",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "9868994.14003",
         "s" : "DOTUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "113974.310",
         "y" : 0
      },
      {
         "an" : "Polkadot",
         "as" : 10171171.637,
         "b" : "DOT",
         "ba" : "",
         "c" : "34.265",
         "cs" : null,
         "etf" : false,
         "h" : "34.940",
         "i" : "0.00100000",
         "l" : "31.885",
         "o" : "32.838",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "340461478.02118",
         "s" : "DOTUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "Polkadot"
         ],
         "ts" : "0.001",
         "v" : "10171171.637",
         "y" : 0
      },
      {
         "an" : "DREP",
         "as" : 887769.8,
         "b" : "DREP",
         "ba" : "",
         "c" : "0.00002654",
         "cs" : 1786000000,
         "etf" : false,
         "h" : "0.00002832",
         "i" : "0.10000000",
         "l" : "0.00002412",
         "o" : "0.00002518",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "23.30324875",
         "s" : "DREPBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "887769.80000000",
         "y" : 0
      },
      {
         "an" : "DREP",
         "as" : 3184318.804,
         "b" : "DREP",
         "ba" : "",
         "c" : "1.460",
         "cs" : 1786000000,
         "etf" : false,
         "h" : "1.554",
         "i" : "0.00100000",
         "l" : "1.286",
         "o" : "1.354",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "4494163.0837",
         "s" : "DREPUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "3184318.804",
         "y" : 0
      },
      {
         "an" : "Dusk Network",
         "as" : 16033553,
         "b" : "DUSK",
         "ba" : "",
         "c" : "0.00000480",
         "cs" : 250000000,
         "etf" : false,
         "h" : "0.00000566",
         "i" : "1.00000000",
         "l" : "0.00000460",
         "o" : "0.00000476",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "81.01187889",
         "s" : "DUSKBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "16033553.00000000",
         "y" : 0
      },
      {
         "an" : "Dusk Network",
         "as" : 38554634.56,
         "b" : "DUSK",
         "ba" : "",
         "c" : "0.2631",
         "cs" : 250000000,
         "etf" : false,
         "h" : "0.3077",
         "i" : "0.01000000",
         "l" : "0.2458",
         "o" : "0.2559",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "10540579.79535",
         "s" : "DUSKUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "38554634.5600",
         "y" : 0
      },
      {
         "an" : "Elrond eGold",
         "as" : 11497.89,
         "b" : "EGLD",
         "ba" : "",
         "c" : "0.3194",
         "cs" : null,
         "etf" : false,
         "h" : "0.3300",
         "i" : "0.01000000",
         "l" : "0.3074",
         "o" : "0.3136",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "3700.676763",
         "s" : "EGLDBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "11497.8900",
         "y" : 0
      },
      {
         "an" : "Elrond eGold",
         "as" : 32235.98,
         "b" : "EGLD",
         "ba" : "",
         "c" : "0.003321",
         "cs" : null,
         "etf" : false,
         "h" : "0.003438",
         "i" : "0.00100000",
         "l" : "0.003041",
         "o" : "0.003133",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "103.25779152",
         "s" : "EGLDBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "32235.980000",
         "y" : 0
      },
      {
         "an" : "Elrond eGold",
         "as" : 37019.5242,
         "b" : "EGLD",
         "ba" : "",
         "c" : "182.65",
         "cs" : null,
         "etf" : false,
         "h" : "185.63",
         "i" : "0.00010000",
         "l" : "160.20",
         "o" : "168.89",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "6356338.069962",
         "s" : "EGLDBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "37019.53",
         "y" : 0
      },
      {
         "an" : "Elrond eGold",
         "as" : 14258.00584,
         "b" : "EGLD",
         "ba" : "",
         "c" : "151.53",
         "cs" : null,
         "etf" : false,
         "h" : "153.65",
         "i" : "0.00001000",
         "l" : "133.17",
         "o" : "139.82",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "2043543.5985906",
         "s" : "EGLDEUR",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "14258.01",
         "y" : 0
      },
      {
         "an" : "Elrond eGold",
         "as" : 215821.0254,
         "b" : "EGLD",
         "ba" : "",
         "c" : "182.55",
         "cs" : null,
         "etf" : false,
         "h" : "185.11",
         "i" : "0.00010000",
         "l" : "159.89",
         "o" : "168.61",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "37036160.0604475",
         "s" : "EGLDUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "215821.03",
         "y" : 0
      },
      {
         "an" : "aelf",
         "as" : 6595333,
         "b" : "ELF",
         "ba" : "",
         "c" : "0.00000758",
         "cs" : 544480199,
         "etf" : false,
         "h" : "0.00000776",
         "i" : "1.00000000",
         "l" : "0.00000704",
         "o" : "0.00000714",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "48.93026979",
         "s" : "ELFBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "6595333.00000000",
         "y" : 0
      },
      {
         "an" : "aelf",
         "as" : 1192375,
         "b" : "ELF",
         "ba" : "",
         "c" : "0.00016383",
         "cs" : 544480199,
         "etf" : false,
         "h" : "0.00016566",
         "i" : "1.00000000",
         "l" : "0.00015176",
         "o" : "0.00015271",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "189.7049674",
         "s" : "ELFETH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "1192375.00000000",
         "y" : 0
      },
      {
         "an" : "Enjin Coin",
         "as" : 862566,
         "b" : "ENJ",
         "ba" : "",
         "c" : "0.004544",
         "cs" : 756192535,
         "etf" : false,
         "h" : "0.005097",
         "i" : "1.00000000",
         "l" : "0.004354",
         "o" : "0.004448",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "4034.582648",
         "s" : "ENJBNB",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.000001",
         "v" : "862566.000000",
         "y" : 0
      },
      {
         "an" : "Enjin Coin",
         "as" : 114696.21,
         "b" : "ENJ",
         "ba" : "",
         "c" : "14.176",
         "cs" : 756192535,
         "etf" : false,
         "h" : "14.480",
         "i" : "0.01000000",
         "l" : "12.836",
         "o" : "13.268",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BRL",
         "qa" : "",
         "qn" : "Brazilian Real",
         "qv" : "1568545.43951",
         "s" : "ENJBRL",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.001",
         "v" : "114696.210",
         "y" : 0
      },
      {
         "an" : "Enjin Coin",
         "as" : 8696561,
         "b" : "ENJ",
         "ba" : "",
         "c" : "0.00004734",
         "cs" : 756192535,
         "etf" : false,
         "h" : "0.00004891",
         "i" : "1.00000000",
         "l" : "0.00004350",
         "o" : "0.00004445",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "405.94201977",
         "s" : "ENJBTC",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.00000001",
         "v" : "8696561.00000000",
         "y" : 0
      },
      {
         "an" : "Enjin Coin",
         "as" : 3282853,
         "b" : "ENJ",
         "ba" : "",
         "c" : "2.6011",
         "cs" : 756192535,
         "etf" : false,
         "h" : "2.6517",
         "i" : "0.01000000",
         "l" : "2.3198",
         "o" : "2.3931",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "8240181.6015358",
         "s" : "ENJBUSD",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.0001",
         "v" : "3282853.0000",
         "y" : 0
      },
      {
         "an" : "Enjin Coin",
         "as" : 1757430,
         "b" : "ENJ",
         "ba" : "",
         "c" : "0.00101560",
         "cs" : 756192535,
         "etf" : false,
         "h" : "0.00104696",
         "i" : "1.00000000",
         "l" : "0.00093547",
         "o" : "0.00095377",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "1763.89132517",
         "s" : "ENJETH",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.00000001",
         "v" : "1757430.00000000",
         "y" : 0
      },
      {
         "an" : "Enjin Coin",
         "as" : 1066722.869,
         "b" : "ENJ",
         "ba" : "",
         "c" : "2.155",
         "cs" : 756192535,
         "etf" : false,
         "h" : "2.200",
         "i" : "0.00100000",
         "l" : "1.923",
         "o" : "1.989",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "2242157.72969",
         "s" : "ENJEUR",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.001",
         "v" : "1066722.869",
         "y" : 0
      },
      {
         "an" : "Enjin Coin",
         "as" : 592935.23,
         "b" : "ENJ",
         "ba" : "",
         "c" : "1.8734",
         "cs" : 756192535,
         "etf" : false,
         "h" : "1.9123",
         "i" : "0.01000000",
         "l" : "1.6765",
         "o" : "1.7308",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "GBP",
         "qa" : "",
         "qn" : "Pound Sterling",
         "qv" : "1077909.923306",
         "s" : "ENJGBP",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.0001",
         "v" : "592935.2300",
         "y" : 0
      },
      {
         "an" : "Enjin Coin",
         "as" : 38134798.79,
         "b" : "ENJ",
         "ba" : "",
         "c" : "2.5990",
         "cs" : 756192535,
         "etf" : false,
         "h" : "2.6500",
         "i" : "0.01000000",
         "l" : "2.3151",
         "o" : "2.3893",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "95660871.25128",
         "s" : "ENJUSDT",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.0001",
         "v" : "38134798.7900",
         "y" : 0
      },
      {
         "an" : "EOS",
         "as" : 569136.59,
         "b" : "EOS",
         "ba" : "",
         "c" : "0.01049",
         "cs" : 952879989,
         "etf" : false,
         "h" : "0.01120",
         "i" : "0.01000000",
         "l" : "0.01034",
         "o" : "0.01054",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "6152.3681857",
         "s" : "EOSBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "569136.59000",
         "y" : 0
      },
      {
         "an" : "EOS",
         "as" : 4381009.02,
         "b" : "EOS",
         "ba" : "",
         "c" : "0.0001095",
         "cs" : 952879989,
         "etf" : false,
         "h" : "0.0001113",
         "i" : "0.01000000",
         "l" : "0.0001031",
         "o" : "0.0001056",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "471.89719612",
         "s" : "EOSBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "4381009.0200000",
         "y" : 0
      },
      {
         "an" : "EOS",
         "as" : 2287969.79,
         "b" : "EOS",
         "ba" : "",
         "c" : "6.0213",
         "cs" : 952879989,
         "etf" : false,
         "h" : "6.1128",
         "i" : "0.01000000",
         "l" : "5.5048",
         "o" : "5.6800",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "13247347.9219",
         "s" : "EOSBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "2287969.7900",
         "y" : 0
      },
      {
         "an" : "EOSDOWN",
         "as" : 108955982.89,
         "b" : "EOSDOWN",
         "ba" : "",
         "c" : "0.01708",
         "cs" : null,
         "etf" : true,
         "h" : "0.02164",
         "i" : "0.01000000",
         "l" : "0.01636",
         "o" : "0.02026",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "2071827.9544944",
         "s" : "EOSDOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "108955982.89000",
         "y" : 0
      },
      {
         "an" : "EOS",
         "as" : 279421.841,
         "b" : "EOS",
         "ba" : "",
         "c" : "4.990",
         "cs" : 952879989,
         "etf" : false,
         "h" : "5.064",
         "i" : "0.00100000",
         "l" : "4.572",
         "o" : "4.706",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "1335400.585274",
         "s" : "EOSEUR",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "279421.841",
         "y" : 0
      },
      {
         "an" : "EOS",
         "as" : 156291.37,
         "b" : "EOS",
         "ba" : "",
         "c" : "48.90",
         "cs" : 952879989,
         "etf" : false,
         "h" : "49.74",
         "i" : "0.01000000",
         "l" : "44.94",
         "o" : "46.31",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "7418922.5493",
         "s" : "EOSTRY",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "156291.37",
         "y" : 0
      },
      {
         "an" : "EOS",
         "as" : 74674.33,
         "b" : "EOS",
         "ba" : "",
         "c" : "5.9892",
         "cs" : 952879989,
         "etf" : false,
         "h" : "6.1116",
         "i" : "0.01000000",
         "l" : "5.5119",
         "o" : "5.6694",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TUSD",
         "qa" : "",
         "qn" : "TrueUSD",
         "qv" : "427495.622616",
         "s" : "EOSTUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "74674.3300",
         "y" : 0
      },
      {
         "an" : "EOSUP",
         "as" : 3564694.15,
         "b" : "EOSUP",
         "ba" : "",
         "c" : "2.774",
         "cs" : null,
         "etf" : true,
         "h" : "2.860",
         "i" : "0.01000000",
         "l" : "2.324",
         "o" : "2.478",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "9216786.81707",
         "s" : "EOSUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "3564694.150",
         "y" : 0
      },
      {
         "an" : "EOS",
         "as" : 368724.4,
         "b" : "EOS",
         "ba" : "",
         "c" : "6.0159",
         "cs" : 952879989,
         "etf" : false,
         "h" : "6.1068",
         "i" : "0.01000000",
         "l" : "5.5207",
         "o" : "5.6659",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDC",
         "qa" : "",
         "qn" : "USD Coin",
         "qv" : "2116893.748439",
         "s" : "EOSUSDC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "368724.4000",
         "y" : 0
      },
      {
         "an" : "EOS",
         "as" : 43729309.53,
         "b" : "EOS",
         "ba" : "",
         "c" : "6.0091",
         "cs" : 952879989,
         "etf" : false,
         "h" : "6.1068",
         "i" : "0.01000000",
         "l" : "5.4979",
         "o" : "5.6737",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "252850132.582973",
         "s" : "EOSUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "43729309.5300",
         "y" : 0
      },
      {
         "an" : "Ellipsis",
         "as" : 1144761.1,
         "b" : "EPS",
         "ba" : "",
         "c" : "0.00003626",
         "cs" : null,
         "etf" : false,
         "h" : "0.00003781",
         "i" : "0.10000000",
         "l" : "0.00003169",
         "o" : "0.00003189",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "39.22967573",
         "s" : "EPSBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC",
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "1144761.10000000",
         "y" : 0
      },
      {
         "an" : "Ellipsis",
         "as" : 754186.008,
         "b" : "EPS",
         "ba" : "",
         "c" : "1.995",
         "cs" : null,
         "etf" : false,
         "h" : "2.082",
         "i" : "0.00100000",
         "l" : "1.684",
         "o" : "1.715",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1394154.458168",
         "s" : "EPSBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC",
            "defi"
         ],
         "ts" : "0.001",
         "v" : "754186.008",
         "y" : 0
      },
      {
         "an" : "Ellipsis",
         "as" : 7535399.054,
         "b" : "EPS",
         "ba" : "",
         "c" : "1.995",
         "cs" : null,
         "etf" : false,
         "h" : "2.083",
         "i" : "0.00100000",
         "l" : "1.681",
         "o" : "1.714",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "13981459.628914",
         "s" : "EPSUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC",
            "defi"
         ],
         "ts" : "0.001",
         "v" : "7535399.054",
         "y" : 0
      },
      {
         "an" : "Ethereum Classic",
         "as" : 60010.8,
         "b" : "ETC",
         "ba" : "",
         "c" : "0.05926",
         "cs" : 0,
         "etf" : false,
         "h" : "0.06346",
         "i" : "0.01000000",
         "l" : "0.05836",
         "o" : "0.05965",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "3622.9228547",
         "s" : "ETCBNB",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "60010.80000",
         "y" : 0
      },
      {
         "an" : "Ethereum Classic",
         "as" : 111881.498,
         "b" : "ETC",
         "ba" : "",
         "c" : "33.807",
         "cs" : 0,
         "etf" : false,
         "h" : "34.708",
         "i" : "0.00100000",
         "l" : "31.336",
         "o" : "32.092",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "3678494.785294",
         "s" : "ETCBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "111881.498",
         "y" : 0
      },
      {
         "an" : "Ethereum Classic",
         "as" : 57339.89,
         "b" : "ETC",
         "ba" : "",
         "c" : "0.013226",
         "cs" : 0,
         "etf" : false,
         "h" : "0.013448",
         "i" : "0.01000000",
         "l" : "0.012673",
         "o" : "0.012780",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "743.25021808",
         "s" : "ETCETH",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "57339.890000",
         "y" : 0
      },
      {
         "an" : "Ethereum Classic",
         "as" : 3293109.061,
         "b" : "ETC",
         "ba" : "",
         "c" : "33.758",
         "cs" : 0,
         "etf" : false,
         "h" : "34.500",
         "i" : "0.00100000",
         "l" : "31.273",
         "o" : "32.064",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "107662145.976785",
         "s" : "ETCUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "3293109.061",
         "y" : 0
      },
      {
         "an" : "Ethereum",
         "as" : 1965.16654,
         "b" : "ETH",
         "ba" : "Î",
         "c" : "3312.83",
         "cs" : 115634925,
         "etf" : false,
         "h" : "3345.03",
         "i" : "0.00001000",
         "l" : "3129.42",
         "o" : "3239.84",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "AUD",
         "qa" : "",
         "qn" : "Australian Dollar",
         "qv" : "6388977.1900166",
         "s" : "ETHAUD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "1965.17",
         "y" : 0
      },
      {
         "an" : "Ethereum",
         "as" : 933.40872,
         "b" : "ETH",
         "ba" : "Î",
         "c" : "37888551",
         "cs" : 115634925,
         "etf" : false,
         "h" : "38105383",
         "i" : "0.00001000",
         "l" : "36177985",
         "o" : "37240510",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BIDR",
         "qa" : "",
         "qn" : "BIDR",
         "qv" : "34899122761.26",
         "s" : "ETHBIDR",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "1",
         "v" : "934",
         "y" : 0
      },
      {
         "an" : "Ethereum",
         "as" : 3570.74488,
         "b" : "ETH",
         "ba" : "Î",
         "c" : "13973.71",
         "cs" : 115634925,
         "etf" : false,
         "h" : "14150.00",
         "i" : "0.00001000",
         "l" : "13317.49",
         "o" : "13865.39",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BRL",
         "qa" : "",
         "qn" : "Brazilian Real",
         "qv" : "49141522.2262701",
         "s" : "ETHBRL",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "3570.75",
         "y" : 0
      },
      {
         "an" : "Ethereum",
         "as" : 267536.75385,
         "b" : "ETH",
         "ba" : "Î",
         "c" : "2561.84",
         "cs" : 115634925,
         "etf" : false,
         "h" : "2587.34",
         "i" : "0.00001000",
         "l" : "2432.05",
         "o" : "2507.01",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "672612400.2304885",
         "s" : "ETHBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "267536.76",
         "y" : 0
      },
      {
         "an" : "Ethereum",
         "as" : 8444.18597,
         "b" : "ETH",
         "ba" : "Î",
         "c" : "2558.19",
         "cs" : 115634925,
         "etf" : false,
         "h" : "2582.04",
         "i" : "0.00001000",
         "l" : "2430.00",
         "o" : "2504.17",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "DAI",
         "qa" : "",
         "qn" : "Dai",
         "qv" : "21175612.8411002",
         "s" : "ETHDAI",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "8444.19",
         "y" : 0
      },
      {
         "an" : "ETHDOWN",
         "as" : 7473656965.36,
         "b" : "ETHDOWN",
         "ba" : null,
         "c" : "0.001580",
         "cs" : null,
         "etf" : true,
         "h" : "0.001830",
         "i" : "0.01000000",
         "l" : "0.001536",
         "o" : "0.001682",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "12567176.43576534",
         "s" : "ETHDOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "7473656965.360000",
         "y" : 0
      },
      {
         "an" : "Ethereum",
         "as" : 42465.36299,
         "b" : "ETH",
         "ba" : "Î",
         "c" : "2122.41",
         "cs" : 115634925,
         "etf" : false,
         "h" : "2142.84",
         "i" : "0.00001000",
         "l" : "2016.73",
         "o" : "2078.08",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "88612035.8241155",
         "s" : "ETHEUR",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "42465.37",
         "y" : 0
      },
      {
         "an" : "Ethereum",
         "as" : 14225.61752,
         "b" : "ETH",
         "ba" : "Î",
         "c" : "1847.15",
         "cs" : 115634925,
         "etf" : false,
         "h" : "1864.70",
         "i" : "0.00001000",
         "l" : "1755.01",
         "o" : "1809.13",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "GBP",
         "qa" : "",
         "qn" : "Pound Sterling",
         "qv" : "25823104.0672896",
         "s" : "ETHGBP",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "14225.62",
         "y" : 0
      },
      {
         "an" : "Ethereum",
         "as" : 2307.75764,
         "b" : "ETH",
         "ba" : "Î",
         "c" : "2557.59",
         "cs" : 115634925,
         "etf" : false,
         "h" : "2587.55",
         "i" : "0.00001000",
         "l" : "2432.63",
         "o" : "2506.56",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "PAX",
         "qa" : "",
         "qn" : "Paxos Standard",
         "qv" : "5800040.6123166",
         "s" : "ETHPAX",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "2307.76",
         "y" : 0
      },
      {
         "an" : "Ethereum",
         "as" : 710.77612,
         "b" : "ETH",
         "ba" : "Î",
         "c" : "192865.5",
         "cs" : 115634925,
         "etf" : false,
         "h" : "195000.0",
         "i" : "0.00001000",
         "l" : "183656.3",
         "o" : "189789.5",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "RUB",
         "qa" : "",
         "qn" : "Russian Ruble",
         "qv" : "134826003.528447",
         "s" : "ETHRUB",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.1",
         "v" : "710.8",
         "y" : 0
      },
      {
         "an" : "Ethereum",
         "as" : 5344.969828,
         "b" : "ETH",
         "ba" : "Î",
         "c" : "20833",
         "cs" : 115634925,
         "etf" : false,
         "h" : "21061",
         "i" : "0.00000100",
         "l" : "19825",
         "o" : "20458",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "109580206.584476",
         "s" : "ETHTRY",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "1",
         "v" : "5345",
         "y" : 0
      },
      {
         "an" : "Ethereum",
         "as" : 3402.87769,
         "b" : "ETH",
         "ba" : "Î",
         "c" : "2561.85",
         "cs" : 115634925,
         "etf" : false,
         "h" : "2586.66",
         "i" : "0.00001000",
         "l" : "1320.00",
         "o" : "2505.97",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TUSD",
         "qa" : "",
         "qn" : "TrueUSD",
         "qv" : "8431227.7519455",
         "s" : "ETHTUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "3402.88",
         "y" : 0
      },
      {
         "an" : "ETHUP",
         "as" : 95456.08,
         "b" : "ETHUP",
         "ba" : null,
         "c" : "206.353",
         "cs" : null,
         "etf" : true,
         "h" : "210.469",
         "i" : "0.01000000",
         "l" : "187.939",
         "o" : "198.261",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "19132650.96852",
         "s" : "ETHUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "95456.080",
         "y" : 0
      },
      {
         "an" : "Ethereum",
         "as" : 14967.4994,
         "b" : "ETH",
         "ba" : "Î",
         "c" : "2561.07",
         "cs" : 115634925,
         "etf" : false,
         "h" : "2587.24",
         "i" : "0.00001000",
         "l" : "2421.52",
         "o" : "2505.43",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDC",
         "qa" : "",
         "qn" : "USD Coin",
         "qv" : "37653635.7296868",
         "s" : "ETHUSDC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "14967.50",
         "y" : 0
      },
      {
         "an" : "Euro",
         "as" : 18118132.81,
         "b" : "EUR",
         "ba" : "",
         "c" : "1.2071",
         "cs" : null,
         "etf" : false,
         "h" : "1.2077",
         "i" : "0.01000000",
         "l" : "1.2035",
         "o" : "1.2063",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "21846930.696055",
         "s" : "EURBUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "18118132.8100",
         "y" : 0
      },
      {
         "an" : "Euro",
         "as" : 116277394.26,
         "b" : "EUR",
         "ba" : "",
         "c" : "1.2054",
         "cs" : null,
         "etf" : false,
         "h" : "1.2066",
         "i" : "0.01000000",
         "l" : "1.2011",
         "o" : "1.2043",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "140036711.532221",
         "s" : "EURUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "116277394.2600",
         "y" : 0
      },
      {
         "an" : "Everex",
         "as" : 902842,
         "b" : "EVX",
         "ba" : "",
         "c" : "0.00002207",
         "cs" : 22700000,
         "etf" : false,
         "h" : "0.00002297",
         "i" : "1.00000000",
         "l" : "0.00002151",
         "o" : "0.00002178",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "20.02554207",
         "s" : "EVXBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "902842.00000000",
         "y" : 0
      },
      {
         "an" : "Fetch.AI",
         "as" : 1635886,
         "b" : "FET",
         "ba" : "",
         "c" : "0.000870",
         "cs" : 629361105,
         "etf" : false,
         "h" : "0.000962",
         "i" : "1.00000000",
         "l" : "0.000822",
         "o" : "0.000843",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1439.591675",
         "s" : "FETBNB",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.000001",
         "v" : "1635886.000000",
         "y" : 0
      },
      {
         "an" : "Fetch.AI",
         "as" : 46901550,
         "b" : "FET",
         "ba" : "",
         "c" : "0.00000907",
         "cs" : 629361105,
         "etf" : false,
         "h" : "0.00000957",
         "i" : "1.00000000",
         "l" : "0.00000821",
         "o" : "0.00000851",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "411.73822871",
         "s" : "FETBTC",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00000001",
         "v" : "46901550.00000000",
         "y" : 0
      },
      {
         "an" : "Fetch.AI",
         "as" : 96727116.2,
         "b" : "FET",
         "ba" : "",
         "c" : "0.49830",
         "cs" : 629361105,
         "etf" : false,
         "h" : "0.53765",
         "i" : "0.10000000",
         "l" : "0.43326",
         "o" : "0.45701",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "45492014.302357",
         "s" : "FETUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00001",
         "v" : "96727116.20000",
         "y" : 0
      },
      {
         "an" : "Filecoin",
         "as" : 13592.7,
         "b" : "FIL",
         "ba" : "",
         "c" : "0.26908",
         "cs" : null,
         "etf" : false,
         "h" : "0.29344",
         "i" : "0.10000000",
         "l" : "0.26442",
         "o" : "0.26903",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "3817.770766",
         "s" : "FILBNB",
         "st" : "TRADING",
         "tags" : [
            "storage-zone"
         ],
         "ts" : "0.00001",
         "v" : "13592.70000",
         "y" : 0
      },
      {
         "an" : "Filecoin",
         "as" : 101262.39,
         "b" : "FIL",
         "ba" : "",
         "c" : "0.0027926",
         "cs" : null,
         "etf" : false,
         "h" : "0.0028672",
         "i" : "0.01000000",
         "l" : "0.0026737",
         "o" : "0.0026966",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "283.5455868",
         "s" : "FILBTC",
         "st" : "TRADING",
         "tags" : [
            "storage-zone"
         ],
         "ts" : "0.0000001",
         "v" : "101262.3900000",
         "y" : 0
      },
      {
         "an" : "Filecoin",
         "as" : 37288.0643,
         "b" : "FIL",
         "ba" : "",
         "c" : "153.68",
         "cs" : null,
         "etf" : false,
         "h" : "155.38",
         "i" : "0.00010000",
         "l" : "143.06",
         "o" : "145.02",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "5634729.846087",
         "s" : "FILBUSD",
         "st" : "TRADING",
         "tags" : [
            "storage-zone"
         ],
         "ts" : "0.01",
         "v" : "37288.07",
         "y" : 0
      },
      {
         "an" : "FILDOWN",
         "as" : 233832597.63,
         "b" : "FILDOWN",
         "ba" : "",
         "c" : "0.003805",
         "cs" : null,
         "etf" : true,
         "h" : "0.004519",
         "i" : "0.01000000",
         "l" : "0.003674",
         "o" : "0.004405",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "922094.84758077",
         "s" : "FILDOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "233832597.630000",
         "y" : 0
      },
      {
         "an" : "FILUP",
         "as" : 220046.98,
         "b" : "FILUP",
         "ba" : "",
         "c" : "18.892",
         "cs" : null,
         "etf" : true,
         "h" : "19.400",
         "i" : "0.01000000",
         "l" : "16.810",
         "o" : "17.284",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "4065543.56203",
         "s" : "FILUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "220046.980",
         "y" : 0
      },
      {
         "an" : "Filecoin",
         "as" : 1095404.6067,
         "b" : "FIL",
         "ba" : "",
         "c" : "153.44",
         "cs" : null,
         "etf" : false,
         "h" : "155.24",
         "i" : "0.00010000",
         "l" : "142.89",
         "o" : "144.91",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "165275234.612862",
         "s" : "FILUSDT",
         "st" : "TRADING",
         "tags" : [
            "storage-zone"
         ],
         "ts" : "0.01",
         "v" : "1095404.61",
         "y" : 0
      },
      {
         "an" : "FIO Protocol",
         "as" : 4221425.3,
         "b" : "FIO",
         "ba" : "",
         "c" : "0.0005749",
         "cs" : null,
         "etf" : false,
         "h" : "0.0006496",
         "i" : "0.10000000",
         "l" : "0.0005669",
         "o" : "0.0005770",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "2566.46463725",
         "s" : "FIOBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "NFT"
         ],
         "ts" : "0.0000001",
         "v" : "4221425.3000000",
         "y" : 0
      },
      {
         "an" : "FIO Protocol",
         "as" : 8186922,
         "b" : "FIO",
         "ba" : "",
         "c" : "0.00000601",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000645",
         "i" : "1.00000000",
         "l" : "0.00000568",
         "o" : "0.00000578",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "49.71902156",
         "s" : "FIOBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "NFT"
         ],
         "ts" : "0.00000001",
         "v" : "8186922.00000000",
         "y" : 0
      },
      {
         "an" : "FIO Protocol",
         "as" : 5177285.9,
         "b" : "FIO",
         "ba" : "",
         "c" : "0.3310",
         "cs" : null,
         "etf" : false,
         "h" : "0.3523",
         "i" : "0.01000000",
         "l" : "0.3050",
         "o" : "0.3112",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1696384.292387",
         "s" : "FIOBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "NFT"
         ],
         "ts" : "0.0001",
         "v" : "5177285.9000",
         "y" : 0
      },
      {
         "an" : "FIO Protocol",
         "as" : 41688043.64,
         "b" : "FIO",
         "ba" : "",
         "c" : "0.3301",
         "cs" : null,
         "etf" : false,
         "h" : "0.3540",
         "i" : "0.01000000",
         "l" : "0.3040",
         "o" : "0.3108",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "13657600.758127",
         "s" : "FIOUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "NFT"
         ],
         "ts" : "0.0001",
         "v" : "41688043.6400",
         "y" : 0
      },
      {
         "an" : "Firo",
         "as" : 95910.25,
         "b" : "FIRO",
         "ba" : "",
         "c" : "0.0001984",
         "cs" : null,
         "etf" : false,
         "h" : "0.0002099",
         "i" : "0.01000000",
         "l" : "0.0001830",
         "o" : "0.0001870",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "18.85765073",
         "s" : "FIROBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "95910.2500000",
         "y" : 0
      },
      {
         "an" : "Firo",
         "as" : 15136.49,
         "b" : "FIRO",
         "ba" : "",
         "c" : "0.004260",
         "cs" : null,
         "etf" : false,
         "h" : "0.004481",
         "i" : "0.01000000",
         "l" : "0.003933",
         "o" : "0.004005",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "63.38104728",
         "s" : "FIROETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "15136.490000",
         "y" : 0
      },
      {
         "an" : "Firo",
         "as" : 263072.512,
         "b" : "FIRO",
         "ba" : "",
         "c" : "10.884",
         "cs" : null,
         "etf" : false,
         "h" : "11.500",
         "i" : "0.00100000",
         "l" : "9.796",
         "o" : "10.019",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "2777616.792731",
         "s" : "FIROUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "263072.512",
         "y" : 0
      },
      {
         "an" : "Stafi",
         "as" : 819078.2,
         "b" : "FIS",
         "ba" : "",
         "c" : "0.00004285",
         "cs" : null,
         "etf" : false,
         "h" : "0.00004787",
         "i" : "0.10000000",
         "l" : "0.00003574",
         "o" : "0.00003644",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "34.67867166",
         "s" : "FISBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "819078.20000000",
         "y" : 0
      },
      {
         "an" : "Stafi",
         "as" : 204388.728,
         "b" : "FIS",
         "ba" : "",
         "c" : "2.356",
         "cs" : null,
         "etf" : false,
         "h" : "2.634",
         "i" : "0.00100000",
         "l" : "1.908",
         "o" : "1.960",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "472314.331908",
         "s" : "FISBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.001",
         "v" : "204388.728",
         "y" : 0
      },
      {
         "an" : "Stafi",
         "as" : 2466943.922,
         "b" : "FIS",
         "ba" : "",
         "c" : "2.350",
         "cs" : null,
         "etf" : false,
         "h" : "2.589",
         "i" : "0.00100000",
         "l" : "1.903",
         "o" : "1.956",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "5668647.108151",
         "s" : "FISUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.001",
         "v" : "2466943.922",
         "y" : 0
      },
      {
         "an" : "Flamingo",
         "as" : 1846359,
         "b" : "FLM",
         "ba" : "",
         "c" : "0.00001435",
         "cs" : null,
         "etf" : false,
         "h" : "0.00001489",
         "i" : "1.00000000",
         "l" : "0.00001357",
         "o" : "0.00001383",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "26.13701778",
         "s" : "FLMBTC",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone"
         ],
         "ts" : "0.00000001",
         "v" : "1846359.00000000",
         "y" : 0
      },
      {
         "an" : "Flamingo",
         "as" : 730600.59,
         "b" : "FLM",
         "ba" : "",
         "c" : "0.7875",
         "cs" : null,
         "etf" : false,
         "h" : "0.8155",
         "i" : "0.01000000",
         "l" : "0.7148",
         "o" : "0.7432",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "566617.481317",
         "s" : "FLMBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone"
         ],
         "ts" : "0.0001",
         "v" : "730600.5900",
         "y" : 0
      },
      {
         "an" : "Flamingo",
         "as" : 25038070.54,
         "b" : "FLM",
         "ba" : "",
         "c" : "0.7895",
         "cs" : null,
         "etf" : false,
         "h" : "0.8148",
         "i" : "0.01000000",
         "l" : "0.7140",
         "o" : "0.7429",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "19133655.897253",
         "s" : "FLMUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone"
         ],
         "ts" : "0.0001",
         "v" : "25038070.5400",
         "y" : 0
      },
      {
         "an" : "ForTube",
         "as" : 461349286,
         "b" : "FOR",
         "ba" : "",
         "c" : "0.00000190",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000234",
         "i" : "1.00000000",
         "l" : "0.00000164",
         "o" : "0.00000164",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "910.94011607",
         "s" : "FORBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "BSC"
         ],
         "ts" : "0.00000001",
         "v" : "461349286.00000000",
         "y" : 0
      },
      {
         "an" : "ForTube",
         "as" : 233340270.8,
         "b" : "FOR",
         "ba" : "",
         "c" : "0.10406",
         "cs" : null,
         "etf" : false,
         "h" : "0.12500",
         "i" : "0.10000000",
         "l" : "0.08778",
         "o" : "0.08840",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "24169252.725954",
         "s" : "FORBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "BSC"
         ],
         "ts" : "0.00001",
         "v" : "233340270.80000",
         "y" : 0
      },
      {
         "an" : "Ampleforth Governance Token",
         "as" : 45727.88,
         "b" : "FORTH",
         "ba" : "",
         "c" : "0.0006580",
         "cs" : null,
         "etf" : false,
         "h" : "0.0007150",
         "i" : "0.01000000",
         "l" : "0.0006461",
         "o" : "0.0006691",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "30.77282026",
         "s" : "FORTHBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "45727.8800000",
         "y" : 0
      },
      {
         "an" : "Ampleforth Governance Token",
         "as" : 24948.08,
         "b" : "FORTH",
         "ba" : "",
         "c" : "36.205",
         "cs" : null,
         "etf" : false,
         "h" : "39.234",
         "i" : "0.00100000",
         "l" : "34.721",
         "o" : "36.297",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "913959.035489",
         "s" : "FORTHBUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "24948.080",
         "y" : 0
      },
      {
         "an" : "Ampleforth Governance Token",
         "as" : 414949.868,
         "b" : "FORTH",
         "ba" : "",
         "c" : "36.291",
         "cs" : null,
         "etf" : false,
         "h" : "39.000",
         "i" : "0.00100000",
         "l" : "34.124",
         "o" : "36.099",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "15220053.13491",
         "s" : "FORTHUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "414949.868",
         "y" : 0
      },
      {
         "an" : "Frontier",
         "as" : 669737.2,
         "b" : "FRONT",
         "ba" : "",
         "c" : "0.00003914",
         "cs" : null,
         "etf" : false,
         "h" : "0.00004104",
         "i" : "0.10000000",
         "l" : "0.00003571",
         "o" : "0.00003573",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "25.45650297",
         "s" : "FRONTBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "669737.20000000",
         "y" : 0
      },
      {
         "an" : "Frontier",
         "as" : 2059850.05,
         "b" : "FRONT",
         "ba" : "",
         "c" : "2.1487",
         "cs" : null,
         "etf" : false,
         "h" : "2.2723",
         "i" : "0.01000000",
         "l" : "1.9076",
         "o" : "1.9208",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "4198176.894437",
         "s" : "FRONTBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "2059850.0500",
         "y" : 0
      },
      {
         "an" : "Frontier",
         "as" : 122643,
         "b" : "FRONT",
         "ba" : "",
         "c" : "0.0008345",
         "cs" : null,
         "etf" : false,
         "h" : "0.0008794",
         "i" : "1.00000000",
         "l" : "0.0007646",
         "o" : "0.0007674",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "100.6946412",
         "s" : "FRONTETH",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0000001",
         "v" : "122643.0000000",
         "y" : 0
      },
      {
         "an" : "Fantom",
         "as" : 6445360,
         "b" : "FTM",
         "ba" : "",
         "c" : "0.000810",
         "cs" : 1813658595,
         "etf" : false,
         "h" : "0.000923",
         "i" : "1.00000000",
         "l" : "0.000715",
         "o" : "0.000747",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "5213.581917",
         "s" : "FTMBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "6445360.000000",
         "y" : 0
      },
      {
         "an" : "Fantom",
         "as" : 47586587,
         "b" : "FTM",
         "ba" : "",
         "c" : "0.00000845",
         "cs" : 1813658595,
         "etf" : false,
         "h" : "0.00000909",
         "i" : "1.00000000",
         "l" : "0.00000705",
         "o" : "0.00000748",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "383.24465259",
         "s" : "FTMBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "47586587.00000000",
         "y" : 0
      },
      {
         "an" : "Fantom",
         "as" : 239850842.7,
         "b" : "FTM",
         "ba" : "",
         "c" : "0.46400",
         "cs" : 1813658595,
         "etf" : false,
         "h" : "0.49635",
         "i" : "0.10000000",
         "l" : "0.37418",
         "o" : "0.40158",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "103605430.762038",
         "s" : "FTMUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "239850842.70000",
         "y" : 0
      },
      {
         "an" : "FTX Token",
         "as" : 14983.16,
         "b" : "FTT",
         "ba" : "",
         "c" : "0.0941",
         "cs" : 95161158,
         "etf" : false,
         "h" : "0.0974",
         "i" : "0.01000000",
         "l" : "0.0936",
         "o" : "0.0951",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1426.587824",
         "s" : "FTTBNB",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "14983.1600",
         "y" : 0
      },
      {
         "an" : "FTX Token",
         "as" : 158707.72,
         "b" : "FTT",
         "ba" : "",
         "c" : "0.0009793",
         "cs" : 95161158,
         "etf" : false,
         "h" : "0.0009825",
         "i" : "0.01000000",
         "l" : "0.0009325",
         "o" : "0.0009498",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "151.71381218",
         "s" : "FTTBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "158707.7200000",
         "y" : 0
      },
      {
         "an" : "FTX Token",
         "as" : 573885.803,
         "b" : "FTT",
         "ba" : "",
         "c" : "53.800",
         "cs" : 95161158,
         "etf" : false,
         "h" : "53.969",
         "i" : "0.00100000",
         "l" : "49.330",
         "o" : "51.119",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "29678605.337775",
         "s" : "FTTUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "573885.803",
         "y" : 0
      },
      {
         "an" : "FunFair",
         "as" : 70671399,
         "b" : "FUN",
         "ba" : "",
         "c" : "0.00000062",
         "cs" : 10298879188,
         "etf" : false,
         "h" : "0.00000062",
         "i" : "1.00000000",
         "l" : "0.00000057",
         "o" : "0.00000060",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "42.41426936",
         "s" : "FUNBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "70671399.00000000",
         "y" : 0
      },
      {
         "an" : "FunFair",
         "as" : 268495038,
         "b" : "FUN",
         "ba" : "",
         "c" : "0.033814",
         "cs" : 10298879188,
         "etf" : false,
         "h" : "0.034050",
         "i" : "1.00000000",
         "l" : "0.030841",
         "o" : "0.031821",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "8739743.911732",
         "s" : "FUNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "268495038.000000",
         "y" : 0
      },
      {
         "an" : "Frax Share",
         "as" : 192868.41,
         "b" : "FXS",
         "ba" : "",
         "c" : "0.0001009",
         "cs" : null,
         "etf" : false,
         "h" : "0.0001039",
         "i" : "0.01000000",
         "l" : "0.0000959",
         "o" : "0.0000989",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "19.35578411",
         "s" : "FXSBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0000001",
         "v" : "192868.4100000",
         "y" : 0
      },
      {
         "an" : "Frax Share",
         "as" : 422103.461,
         "b" : "FXS",
         "ba" : "",
         "c" : "5.540",
         "cs" : null,
         "etf" : false,
         "h" : "5.664",
         "i" : "0.00100000",
         "l" : "5.101",
         "o" : "5.331",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2286850.004653",
         "s" : "FXSBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.001",
         "v" : "422103.461",
         "y" : 0
      },
      {
         "an" : "Pound Sterling",
         "as" : 9558675.13,
         "b" : "GBP",
         "ba" : "",
         "c" : "1.3872",
         "cs" : null,
         "etf" : false,
         "h" : "1.3888",
         "i" : "0.01000000",
         "l" : "1.3828",
         "o" : "1.3861",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "13248600.094071",
         "s" : "GBPBUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "9558675.1300",
         "y" : 0
      },
      {
         "an" : "Pound Sterling",
         "as" : 46993965.77,
         "b" : "GBP",
         "ba" : "",
         "c" : "1.3851",
         "cs" : null,
         "etf" : false,
         "h" : "1.3882",
         "i" : "0.01000000",
         "l" : "1.3812",
         "o" : "1.3838",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "65062897.471989",
         "s" : "GBPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "46993965.7700",
         "y" : 0
      },
      {
         "an" : "Aavegotchi",
         "as" : 2732429.11,
         "b" : "GHST",
         "ba" : "",
         "c" : "1.1079",
         "cs" : null,
         "etf" : false,
         "h" : "1.2000",
         "i" : "0.01000000",
         "l" : "1.0499",
         "o" : "1.0603",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "3000066.849769",
         "s" : "GHSTBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "NFT"
         ],
         "ts" : "0.0001",
         "v" : "2732429.1100",
         "y" : 0
      },
      {
         "an" : "Aavegotchi",
         "as" : 564711,
         "b" : "GHST",
         "ba" : "",
         "c" : "0.0004326",
         "cs" : null,
         "etf" : false,
         "h" : "0.0004700",
         "i" : "1.00000000",
         "l" : "0.0004180",
         "o" : "0.0004253",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "245.4544451",
         "s" : "GHSTETH",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "NFT"
         ],
         "ts" : "0.0000001",
         "v" : "564711.0000000",
         "y" : 0
      },
      {
         "an" : "Golem",
         "as" : 4447553,
         "b" : "GLM",
         "ba" : "",
         "c" : "0.00000803",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000859",
         "i" : "1.00000000",
         "l" : "0.00000734",
         "o" : "0.00000741",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "35.21672451",
         "s" : "GLMBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "4447553.00000000",
         "y" : 0
      },
      {
         "an" : "Golem",
         "as" : 1859250,
         "b" : "GLM",
         "ba" : "",
         "c" : "0.0001719",
         "cs" : null,
         "etf" : false,
         "h" : "0.0001828",
         "i" : "1.00000000",
         "l" : "0.0001580",
         "o" : "0.0001604",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "311.1274672",
         "s" : "GLMETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "1859250.0000000",
         "y" : 0
      },
      {
         "an" : "GoChain",
         "as" : 28630284,
         "b" : "GO",
         "ba" : "",
         "c" : "0.00000088",
         "cs" : 1076391884,
         "etf" : false,
         "h" : "0.00000089",
         "i" : "1.00000000",
         "l" : "0.00000081",
         "o" : "0.00000083",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "24.29213624",
         "s" : "GOBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "28630284.00000000",
         "y" : 0
      },
      {
         "an" : "Groestlcoin",
         "as" : 1565954,
         "b" : "GRS",
         "ba" : "",
         "c" : "0.00002604",
         "cs" : 77425613,
         "etf" : false,
         "h" : "0.00002695",
         "i" : "1.00000000",
         "l" : "0.00002379",
         "o" : "0.00002546",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "39.97943672",
         "s" : "GRSBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "1565954.00000000",
         "y" : 0
      },
      {
         "an" : "The Graph",
         "as" : 2916168,
         "b" : "GRT",
         "ba" : "",
         "c" : "0.00002822",
         "cs" : null,
         "etf" : false,
         "h" : "0.00002861",
         "i" : "1.00000000",
         "l" : "0.00002636",
         "o" : "0.00002748",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "80.60083109",
         "s" : "GRTBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "2916168.00000000",
         "y" : 0
      },
      {
         "an" : "The Graph",
         "as" : 1613969.21,
         "b" : "GRT",
         "ba" : "",
         "c" : "1.5514",
         "cs" : null,
         "etf" : false,
         "h" : "1.5800",
         "i" : "0.01000000",
         "l" : "1.3920",
         "o" : "1.4780",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2403538.316588",
         "s" : "GRTBUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "1613969.2100",
         "y" : 0
      },
      {
         "an" : "The Graph",
         "as" : 343737,
         "b" : "GRT",
         "ba" : "",
         "c" : "0.00060310",
         "cs" : null,
         "etf" : false,
         "h" : "0.00061295",
         "i" : "1.00000000",
         "l" : "0.00056381",
         "o" : "0.00058809",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "203.38854277",
         "s" : "GRTETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "343737.00000000",
         "y" : 0
      },
      {
         "an" : "The Graph",
         "as" : 907490.01,
         "b" : "GRT",
         "ba" : "",
         "c" : "1.2860",
         "cs" : null,
         "etf" : false,
         "h" : "1.3080",
         "i" : "0.01000000",
         "l" : "1.1540",
         "o" : "1.2253",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "1119183.48122",
         "s" : "GRTEUR",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "907490.0100",
         "y" : 0
      },
      {
         "an" : "The Graph",
         "as" : 25756825.29,
         "b" : "GRT",
         "ba" : "",
         "c" : "1.5506",
         "cs" : null,
         "etf" : false,
         "h" : "1.5781",
         "i" : "0.01000000",
         "l" : "1.3901",
         "o" : "1.4768",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "38212099.339061",
         "s" : "GRTUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "25756825.2900",
         "y" : 0
      },
      {
         "an" : "Gifto",
         "as" : 66305219,
         "b" : "GTO",
         "ba" : "",
         "c" : "0.00000153",
         "cs" : 770236879,
         "etf" : false,
         "h" : "0.00000159",
         "i" : "1.00000000",
         "l" : "0.00000127",
         "o" : "0.00000133",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "92.93680317",
         "s" : "GTOBTC",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "66305219.00000000",
         "y" : 0
      },
      {
         "an" : "Gifto",
         "as" : 219609928.3,
         "b" : "GTO",
         "ba" : "",
         "c" : "0.08399",
         "cs" : 770236879,
         "etf" : false,
         "h" : "0.08752",
         "i" : "0.10000000",
         "l" : "0.06730",
         "o" : "0.07127",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "16613981.946816",
         "s" : "GTOUSDT",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "219609928.30000",
         "y" : 0
      },
      {
         "an" : "Genesis Vision",
         "as" : 229740.95,
         "b" : "GVT",
         "ba" : "",
         "c" : "0.0001487",
         "cs" : 3726781,
         "etf" : false,
         "h" : "0.0001552",
         "i" : "0.01000000",
         "l" : "0.0001407",
         "o" : "0.0001485",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "34.02924274",
         "s" : "GVTBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "229740.9500000",
         "y" : 0
      },
      {
         "an" : "GXChain",
         "as" : 2151084,
         "b" : "GXS",
         "ba" : "",
         "c" : "0.00001619",
         "cs" : 60000000,
         "etf" : false,
         "h" : "0.00001634",
         "i" : "1.00000000",
         "l" : "0.00001541",
         "o" : "0.00001588",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "33.92529552",
         "s" : "GXSBTC",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00000001",
         "v" : "2151084.00000000",
         "y" : 0
      },
      {
         "an" : "GXChain",
         "as" : 447123.21,
         "b" : "GXS",
         "ba" : "",
         "c" : "0.0003480",
         "cs" : 60000000,
         "etf" : false,
         "h" : "0.0003517",
         "i" : "0.01000000",
         "l" : "0.0003300",
         "o" : "0.0003400",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "151.17593938",
         "s" : "GXSETH",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.0000001",
         "v" : "447123.2100000",
         "y" : 0
      },
      {
         "an" : "GXChain",
         "as" : 4181741.23,
         "b" : "GXS",
         "ba" : "",
         "c" : "0.8904",
         "cs" : 60000000,
         "etf" : false,
         "h" : "0.8961",
         "i" : "0.01000000",
         "l" : "0.8146",
         "o" : "0.8520",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "3555975.273949",
         "s" : "GXSUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.0001",
         "v" : "4181741.2300",
         "y" : 0
      },
      {
         "an" : "HARD Protocol",
         "as" : 64519.7,
         "b" : "HARD",
         "ba" : "",
         "c" : "0.002737",
         "cs" : null,
         "etf" : false,
         "h" : "0.002886",
         "i" : "0.10000000",
         "l" : "0.002683",
         "o" : "0.002753",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "180.2199755",
         "s" : "HARDBNB",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.000001",
         "v" : "64519.700000",
         "y" : 0
      },
      {
         "an" : "HARD Protocol",
         "as" : 588410.9,
         "b" : "HARD",
         "ba" : "",
         "c" : "0.00002838",
         "cs" : null,
         "etf" : false,
         "h" : "0.00002981",
         "i" : "0.10000000",
         "l" : "0.00002663",
         "o" : "0.00002745",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "16.37220558",
         "s" : "HARDBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "588410.90000000",
         "y" : 0
      },
      {
         "an" : "HARD Protocol",
         "as" : 149379.13,
         "b" : "HARD",
         "ba" : "",
         "c" : "1.5561",
         "cs" : null,
         "etf" : false,
         "h" : "1.6292",
         "i" : "0.01000000",
         "l" : "1.4139",
         "o" : "1.4722",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "224575.848157",
         "s" : "HARDBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "149379.1300",
         "y" : 0
      },
      {
         "an" : "HARD Protocol",
         "as" : 5389452.55,
         "b" : "HARD",
         "ba" : "",
         "c" : "1.5577",
         "cs" : null,
         "etf" : false,
         "h" : "1.6257",
         "i" : "0.01000000",
         "l" : "1.4114",
         "o" : "1.4770",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "7975938.811092",
         "s" : "HARDUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "5389452.5500",
         "y" : 0
      },
      {
         "an" : "Hedera Hashgraph",
         "as" : 2303155,
         "b" : "HBAR",
         "ba" : "",
         "c" : "0.0005314",
         "cs" : 4077684788,
         "etf" : false,
         "h" : "0.0005793",
         "i" : "1.00000000",
         "l" : "0.0005185",
         "o" : "0.0005313",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1248.8968954",
         "s" : "HBARBNB",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.0000001",
         "v" : "2303155.0000000",
         "y" : 0
      },
      {
         "an" : "Hedera Hashgraph",
         "as" : 31838382,
         "b" : "HBAR",
         "ba" : "",
         "c" : "0.00000554",
         "cs" : 4077684788,
         "etf" : false,
         "h" : "0.00000574",
         "i" : "1.00000000",
         "l" : "0.00000516",
         "o" : "0.00000528",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "172.03027739",
         "s" : "HBARBTC",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00000001",
         "v" : "31838382.00000000",
         "y" : 0
      },
      {
         "an" : "Hedera Hashgraph",
         "as" : 12873932.8,
         "b" : "HBAR",
         "ba" : "",
         "c" : "0.30470",
         "cs" : 4077684788,
         "etf" : false,
         "h" : "0.31490",
         "i" : "0.10000000",
         "l" : "0.27502",
         "o" : "0.28471",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "3759533.359162",
         "s" : "HBARBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00001",
         "v" : "12873932.80000",
         "y" : 0
      },
      {
         "an" : "Hedera Hashgraph",
         "as" : 151731736.7,
         "b" : "HBAR",
         "ba" : "",
         "c" : "0.30415",
         "cs" : 4077684788,
         "etf" : false,
         "h" : "0.31481",
         "i" : "0.10000000",
         "l" : "0.27500",
         "o" : "0.28458",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "44174064.52432",
         "s" : "HBARUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00001",
         "v" : "151731736.70000",
         "y" : 0
      },
      {
         "an" : "Hegic",
         "as" : 2680070.77,
         "b" : "HEGIC",
         "ba" : "",
         "c" : "0.1640",
         "cs" : null,
         "etf" : false,
         "h" : "0.1700",
         "i" : "0.01000000",
         "l" : "0.1472",
         "o" : "0.1565",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "426240.800937",
         "s" : "HEGICBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "2680070.7700",
         "y" : 0
      },
      {
         "an" : "Hegic",
         "as" : 975436,
         "b" : "HEGIC",
         "ba" : "",
         "c" : "0.0000642",
         "cs" : null,
         "etf" : false,
         "h" : "0.0000661",
         "i" : "1.00000000",
         "l" : "0.0000602",
         "o" : "0.0000621",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "61.3888379",
         "s" : "HEGICETH",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0000001",
         "v" : "975436.0000000",
         "y" : 0
      },
      {
         "an" : "Hive",
         "as" : 22408345,
         "b" : "HIVE",
         "ba" : "",
         "c" : "0.00001262",
         "cs" : null,
         "etf" : false,
         "h" : "0.00001548",
         "i" : "1.00000000",
         "l" : "0.00000953",
         "o" : "0.00000990",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "295.37805554",
         "s" : "HIVEBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "22408345.00000000",
         "y" : 0
      },
      {
         "an" : "Hive",
         "as" : 77551493.66,
         "b" : "HIVE",
         "ba" : "",
         "c" : "0.6918",
         "cs" : null,
         "etf" : false,
         "h" : "0.8320",
         "i" : "0.01000000",
         "l" : "0.5047",
         "o" : "0.5350",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "55919190.440463",
         "s" : "HIVEUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "77551493.6600",
         "y" : 0
      },
      {
         "an" : "Helium",
         "as" : 136721.64,
         "b" : "HNT",
         "ba" : "",
         "c" : "0.0003015",
         "cs" : null,
         "etf" : false,
         "h" : "0.0003100",
         "i" : "0.01000000",
         "l" : "0.0002646",
         "o" : "0.0002676",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "39.74042328",
         "s" : "HNTBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "136721.6400000",
         "y" : 0
      },
      {
         "an" : "Helium",
         "as" : 885271.372,
         "b" : "HNT",
         "ba" : "",
         "c" : "16.581",
         "cs" : null,
         "etf" : false,
         "h" : "16.897",
         "i" : "0.00100000",
         "l" : "14.041",
         "o" : "14.399",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "13623413.880735",
         "s" : "HNTUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "885271.372",
         "y" : 0
      },
      {
         "an" : "Holo",
         "as" : 442687183,
         "b" : "HOT",
         "ba" : "",
         "c" : "0.00002965",
         "cs" : 133214575156,
         "etf" : false,
         "h" : "0.00003504",
         "i" : "1.00000000",
         "l" : "0.00002398",
         "o" : "0.00002454",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "12535.219442",
         "s" : "HOTBNB",
         "st" : "TRADING",
         "tags" : [
            "storage-zone"
         ],
         "ts" : "0.00000001",
         "v" : "442687183.00000000",
         "y" : 0
      },
      {
         "an" : "Holo",
         "as" : 49522030,
         "b" : "HOT",
         "ba" : "",
         "c" : "0.0927",
         "cs" : 133214575156,
         "etf" : false,
         "h" : "0.0964",
         "i" : "1.00000000",
         "l" : "0.0696",
         "o" : "0.0731",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BRL",
         "qa" : "",
         "qn" : "Brazilian Real",
         "qv" : "4029308.3784",
         "s" : "HOTBRL",
         "st" : "TRADING",
         "tags" : [
            "storage-zone"
         ],
         "ts" : "0.0001",
         "v" : "49522030.0000",
         "y" : 0
      },
      {
         "an" : "Holo",
         "as" : 379598998.3,
         "b" : "HOT",
         "ba" : "",
         "c" : "0.01702",
         "cs" : 133214575156,
         "etf" : false,
         "h" : "0.01768",
         "i" : "0.10000000",
         "l" : "0.01268",
         "o" : "0.01320",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "5808763.648863",
         "s" : "HOTBUSD",
         "st" : "TRADING",
         "tags" : [
            "storage-zone"
         ],
         "ts" : "0.00001",
         "v" : "379598998.30000",
         "y" : 0
      },
      {
         "an" : "Holo",
         "as" : 1267537932,
         "b" : "HOT",
         "ba" : "",
         "c" : "0.00000664",
         "cs" : 133214575156,
         "etf" : false,
         "h" : "0.00000693",
         "i" : "1.00000000",
         "l" : "0.00000514",
         "o" : "0.00000527",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "7535.99863067",
         "s" : "HOTETH",
         "st" : "TRADING",
         "tags" : [
            "storage-zone"
         ],
         "ts" : "0.00000001",
         "v" : "1267537932.00000000",
         "y" : 0
      },
      {
         "an" : "Holo",
         "as" : 185413565.7,
         "b" : "HOT",
         "ba" : "",
         "c" : "0.01407",
         "cs" : 133214575156,
         "etf" : false,
         "h" : "0.01467",
         "i" : "0.10000000",
         "l" : "0.01050",
         "o" : "0.01095",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "2353225.278305",
         "s" : "HOTEUR",
         "st" : "TRADING",
         "tags" : [
            "storage-zone"
         ],
         "ts" : "0.00001",
         "v" : "185413565.70000",
         "y" : 0
      },
      {
         "an" : "Holo",
         "as" : 6242497869,
         "b" : "HOT",
         "ba" : "",
         "c" : "0.13839",
         "cs" : 133214575156,
         "etf" : false,
         "h" : "0.14420",
         "i" : "1.00000000",
         "l" : "0.10330",
         "o" : "0.10780",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "766575352.93602",
         "s" : "HOTTRY",
         "st" : "TRADING",
         "tags" : [
            "storage-zone"
         ],
         "ts" : "0.00001",
         "v" : "6242497869.00000",
         "y" : 0
      },
      {
         "an" : "Holo",
         "as" : 23311537642,
         "b" : "HOT",
         "ba" : "",
         "c" : "0.017000",
         "cs" : 133214575156,
         "etf" : false,
         "h" : "0.017677",
         "i" : "1.00000000",
         "l" : "0.012630",
         "o" : "0.013202",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "350317393.0638433",
         "s" : "HOTUSDT",
         "st" : "TRADING",
         "tags" : [
            "storage-zone"
         ],
         "ts" : "0.000001",
         "v" : "23311537642.000000",
         "y" : 0
      },
      {
         "an" : "ICON",
         "as" : 201730.1,
         "b" : "ICX",
         "ba" : "",
         "c" : "0.003767",
         "cs" : 387431340,
         "etf" : false,
         "h" : "0.003942",
         "i" : "0.10000000",
         "l" : "0.003571",
         "o" : "0.003601",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "757.7204163",
         "s" : "ICXBNB",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.000001",
         "v" : "201730.100000",
         "y" : 0
      },
      {
         "an" : "ICON",
         "as" : 3341034,
         "b" : "ICX",
         "ba" : "",
         "c" : "0.00003938",
         "cs" : 387431340,
         "etf" : false,
         "h" : "0.00004040",
         "i" : "1.00000000",
         "l" : "0.00003550",
         "o" : "0.00003617",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "125.20353591",
         "s" : "ICXBTC",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00000001",
         "v" : "3341034.00000000",
         "y" : 0
      },
      {
         "an" : "ICON",
         "as" : 1725353.18,
         "b" : "ICX",
         "ba" : "",
         "c" : "2.1642",
         "cs" : 387431340,
         "etf" : false,
         "h" : "2.2310",
         "i" : "0.01000000",
         "l" : "1.8180",
         "o" : "1.9461",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "3478671.989508",
         "s" : "ICXBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.0001",
         "v" : "1725353.1800",
         "y" : 0
      },
      {
         "an" : "ICON",
         "as" : 653039.95,
         "b" : "ICX",
         "ba" : "",
         "c" : "0.000844",
         "cs" : 387431340,
         "etf" : false,
         "h" : "0.000866",
         "i" : "0.01000000",
         "l" : "0.000766",
         "o" : "0.000777",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "529.50188576",
         "s" : "ICXETH",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.000001",
         "v" : "653039.950000",
         "y" : 0
      },
      {
         "an" : "ICON",
         "as" : 14762475.18,
         "b" : "ICX",
         "ba" : "",
         "c" : "2.1636",
         "cs" : 387431340,
         "etf" : false,
         "h" : "2.2301",
         "i" : "0.01000000",
         "l" : "1.8686",
         "o" : "1.9439",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "29828018.51096",
         "s" : "ICXUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.0001",
         "v" : "14762475.1800",
         "y" : 0
      },
      {
         "an" : "IDEX",
         "as" : 25726596,
         "b" : "IDEX",
         "ba" : "",
         "c" : "0.00000233",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000242",
         "i" : "1.00000000",
         "l" : "0.00000230",
         "o" : "0.00000239",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "60.26271728",
         "s" : "IDEXBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "25726596.00000000",
         "y" : 0
      },
      {
         "an" : "IDEX",
         "as" : 17080598.2,
         "b" : "IDEX",
         "ba" : "",
         "c" : "0.12717",
         "cs" : null,
         "etf" : false,
         "h" : "0.13618",
         "i" : "0.10000000",
         "l" : "0.12114",
         "o" : "0.12900",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2160455.885661",
         "s" : "IDEXBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00001",
         "v" : "17080598.20000",
         "y" : 0
      },
      {
         "an" : "Injective Protocol",
         "as" : 648379.7,
         "b" : "INJ",
         "ba" : "",
         "c" : "0.03511",
         "cs" : null,
         "etf" : false,
         "h" : "0.03790",
         "i" : "0.10000000",
         "l" : "0.03289",
         "o" : "0.03309",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "22913.640918",
         "s" : "INJBNB",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos",
            "BSC"
         ],
         "ts" : "0.00001",
         "v" : "648379.70000",
         "y" : 0
      },
      {
         "an" : "Injective Protocol",
         "as" : 413627.1,
         "b" : "INJ",
         "ba" : "",
         "c" : "0.00036715",
         "cs" : null,
         "etf" : false,
         "h" : "0.00037569",
         "i" : "0.10000000",
         "l" : "0.00032900",
         "o" : "0.00033212",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "145.62238739",
         "s" : "INJBTC",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos",
            "BSC"
         ],
         "ts" : "0.00000001",
         "v" : "413627.10000000",
         "y" : 0
      },
      {
         "an" : "Injective Protocol",
         "as" : 200845.242,
         "b" : "INJ",
         "ba" : "",
         "c" : "20.175",
         "cs" : null,
         "etf" : false,
         "h" : "20.620",
         "i" : "0.00100000",
         "l" : "17.516",
         "o" : "17.876",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "3808540.966324",
         "s" : "INJBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "200845.242",
         "y" : 0
      },
      {
         "an" : "Injective Protocol",
         "as" : 2090852.034,
         "b" : "INJ",
         "ba" : "",
         "c" : "20.176",
         "cs" : null,
         "etf" : false,
         "h" : "20.595",
         "i" : "0.00100000",
         "l" : "17.521",
         "o" : "17.828",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "39787725.1620354",
         "s" : "INJUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "2090852.034",
         "y" : 0
      },
      {
         "an" : "IOST",
         "as" : 27969653,
         "b" : "IOST",
         "ba" : "",
         "c" : "0.0001090",
         "cs" : 12003807131,
         "etf" : false,
         "h" : "0.0001191",
         "i" : "1.00000000",
         "l" : "0.0000969",
         "o" : "0.0000990",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "3089.3813181",
         "s" : "IOSTBNB",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.0000001",
         "v" : "27969653.0000000",
         "y" : 0
      },
      {
         "an" : "IOST",
         "as" : 553964053,
         "b" : "IOST",
         "ba" : "",
         "c" : "0.00000114",
         "cs" : 12003807131,
         "etf" : false,
         "h" : "0.00000118",
         "i" : "1.00000000",
         "l" : "0.00000096",
         "o" : "0.00000100",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "606.0044834",
         "s" : "IOSTBTC",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00000001",
         "v" : "553964053.00000000",
         "y" : 0
      },
      {
         "an" : "IOST",
         "as" : 45843981.4,
         "b" : "IOST",
         "ba" : "",
         "c" : "0.06245",
         "cs" : 12003807131,
         "etf" : false,
         "h" : "0.06475",
         "i" : "0.10000000",
         "l" : "0.05117",
         "o" : "0.05345",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2736851.18956",
         "s" : "IOSTBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00001",
         "v" : "45843981.40000",
         "y" : 0
      },
      {
         "an" : "IOST",
         "as" : 43150017,
         "b" : "IOST",
         "ba" : "",
         "c" : "0.00002439",
         "cs" : 12003807131,
         "etf" : false,
         "h" : "0.00002557",
         "i" : "1.00000000",
         "l" : "0.00002079",
         "o" : "0.00002133",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "1003.5184565",
         "s" : "IOSTETH",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00000001",
         "v" : "43150017.00000000",
         "y" : 0
      },
      {
         "an" : "IOST",
         "as" : 1858998092,
         "b" : "IOST",
         "ba" : "",
         "c" : "0.062438",
         "cs" : 12003807131,
         "etf" : false,
         "h" : "0.064786",
         "i" : "1.00000000",
         "l" : "0.051126",
         "o" : "0.053497",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "109791415.879256",
         "s" : "IOSTUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.000001",
         "v" : "1858998092.000000",
         "y" : 0
      },
      {
         "an" : "MIOTA",
         "as" : 1108911.2,
         "b" : "IOTA",
         "ba" : "",
         "c" : "0.003800",
         "cs" : 2779530283,
         "etf" : false,
         "h" : "0.003976",
         "i" : "0.10000000",
         "l" : "0.003571",
         "o" : "0.003730",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "4172.9933648",
         "s" : "IOTABNB",
         "st" : "TRADING",
         "tags" : [
            "pow"
         ],
         "ts" : "0.000001",
         "v" : "1108911.200000",
         "y" : 0
      },
      {
         "an" : "MIOTA",
         "as" : 6343083.52,
         "b" : "IOTA",
         "ba" : "",
         "c" : "2.1766",
         "cs" : 2779530283,
         "etf" : false,
         "h" : "2.2263",
         "i" : "0.01000000",
         "l" : "1.8700",
         "o" : "2.0105",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "12961787.150138",
         "s" : "IOTABUSD",
         "st" : "TRADING",
         "tags" : [
            "pow"
         ],
         "ts" : "0.0001",
         "v" : "6343083.5200",
         "y" : 0
      },
      {
         "an" : "MIOTA",
         "as" : 42804049.27,
         "b" : "IOTA",
         "ba" : "",
         "c" : "2.1751",
         "cs" : 2779530283,
         "etf" : false,
         "h" : "2.2243",
         "i" : "0.01000000",
         "l" : "1.8700",
         "o" : "2.0071",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "86657149.482202",
         "s" : "IOTAUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow"
         ],
         "ts" : "0.0001",
         "v" : "42804049.2700",
         "y" : 0
      },
      {
         "an" : "IoTeX",
         "as" : 158189661,
         "b" : "IOTX",
         "ba" : "",
         "c" : "0.00000085",
         "cs" : 9644304442,
         "etf" : false,
         "h" : "0.00000087",
         "i" : "1.00000000",
         "l" : "0.00000074",
         "o" : "0.00000075",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "130.23042132",
         "s" : "IOTXBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "158189661.00000000",
         "y" : 0
      },
      {
         "an" : "IoTeX",
         "as" : 63160658,
         "b" : "IOTX",
         "ba" : "",
         "c" : "0.00001807",
         "cs" : 9644304442,
         "etf" : false,
         "h" : "0.00001862",
         "i" : "1.00000000",
         "l" : "0.00001593",
         "o" : "0.00001603",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "1108.08647725",
         "s" : "IOTXETH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "63160658.00000000",
         "y" : 0
      },
      {
         "an" : "IoTeX",
         "as" : 450623555,
         "b" : "IOTX",
         "ba" : "",
         "c" : "0.046291",
         "cs" : 9644304442,
         "etf" : false,
         "h" : "0.047368",
         "i" : "1.00000000",
         "l" : "0.039702",
         "o" : "0.040140",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "19917619.488606",
         "s" : "IOTXUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "450623555.000000",
         "y" : 0
      },
      {
         "an" : "Everipedia",
         "as" : 100502088,
         "b" : "IQ",
         "ba" : "",
         "c" : "0.00003921",
         "cs" : null,
         "etf" : false,
         "h" : "0.00004348",
         "i" : "1.00000000",
         "l" : "0.00003726",
         "o" : "0.00003765",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "4059.27856565",
         "s" : "IQBNB",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "100502088.00000000",
         "y" : 0
      },
      {
         "an" : "Everipedia",
         "as" : 122912639,
         "b" : "IQ",
         "ba" : "",
         "c" : "0.022488",
         "cs" : null,
         "etf" : false,
         "h" : "0.023222",
         "i" : "1.00000000",
         "l" : "0.019899",
         "o" : "0.020339",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2673712.003852",
         "s" : "IQBUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "122912639.000000",
         "y" : 0
      },
      {
         "an" : "IRISnet",
         "as" : 9835653,
         "b" : "IRIS",
         "ba" : "",
         "c" : "0.00000285",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000291",
         "i" : "1.00000000",
         "l" : "0.00000265",
         "o" : "0.00000276",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "27.30358018",
         "s" : "IRISBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "9835653.00000000",
         "y" : 0
      },
      {
         "an" : "IRISnet",
         "as" : 25033095.1,
         "b" : "IRIS",
         "ba" : "",
         "c" : "0.15702",
         "cs" : null,
         "etf" : false,
         "h" : "0.15990",
         "i" : "0.10000000",
         "l" : "0.13940",
         "o" : "0.14825",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "3758627.667783",
         "s" : "IRISUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "25033095.10000",
         "y" : 0
      },
      {
         "an" : "JUST",
         "as" : 4269060,
         "b" : "JST",
         "ba" : "",
         "c" : "0.0002227",
         "cs" : null,
         "etf" : false,
         "h" : "0.0002350",
         "i" : "1.00000000",
         "l" : "0.0002146",
         "o" : "0.0002190",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "953.921186",
         "s" : "JSTBNB",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.0000001",
         "v" : "4269060.0000000",
         "y" : 0
      },
      {
         "an" : "JUST",
         "as" : 6016427,
         "b" : "JST",
         "ba" : "",
         "c" : "0.00000232",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000236",
         "i" : "1.00000000",
         "l" : "0.00000214",
         "o" : "0.00000219",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "13.51208655",
         "s" : "JSTBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "6016427.00000000",
         "y" : 0
      },
      {
         "an" : "JUST",
         "as" : 63132074,
         "b" : "JST",
         "ba" : "",
         "c" : "0.12705",
         "cs" : null,
         "etf" : false,
         "h" : "0.12969",
         "i" : "0.10000000",
         "l" : "0.11354",
         "o" : "0.11778",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "7663212.651507",
         "s" : "JSTUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00001",
         "v" : "63132074.00000",
         "y" : 0
      },
      {
         "an" : "Juventus Fan Token",
         "as" : 117394.25,
         "b" : "JUV",
         "ba" : "",
         "c" : "0.0003073",
         "cs" : null,
         "etf" : false,
         "h" : "0.0003368",
         "i" : "0.01000000",
         "l" : "0.0003005",
         "o" : "0.0003038",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "36.51163074",
         "s" : "JUVBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.0000001",
         "v" : "117394.2500000",
         "y" : 0
      },
      {
         "an" : "Juventus Fan Token",
         "as" : 24739.96,
         "b" : "JUV",
         "ba" : "",
         "c" : "17.142",
         "cs" : null,
         "etf" : false,
         "h" : "18.068",
         "i" : "0.00100000",
         "l" : "16.100",
         "o" : "16.440",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "419190.083734",
         "s" : "JUVBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "24739.960",
         "y" : 0
      },
      {
         "an" : "Juventus Fan Token",
         "as" : 565812.065,
         "b" : "JUV",
         "ba" : "",
         "c" : "16.927",
         "cs" : null,
         "etf" : false,
         "h" : "18.030",
         "i" : "0.00100000",
         "l" : "16.026",
         "o" : "16.283",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "9530126.83967",
         "s" : "JUVUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "565812.065",
         "y" : 0
      },
      {
         "an" : "Kava",
         "as" : 72418.1,
         "b" : "KAVA",
         "ba" : "",
         "c" : "0.00915",
         "cs" : 27190672,
         "etf" : false,
         "h" : "0.01002",
         "i" : "0.10000000",
         "l" : "0.00876",
         "o" : "0.00893",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "677.546492",
         "s" : "KAVABNB",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "72418.10000",
         "y" : 0
      },
      {
         "an" : "Kava",
         "as" : 818999.5,
         "b" : "KAVA",
         "ba" : "",
         "c" : "0.00009514",
         "cs" : 27190672,
         "etf" : false,
         "h" : "0.00009898",
         "i" : "0.10000000",
         "l" : "0.00008705",
         "o" : "0.00008978",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "76.84906729",
         "s" : "KAVABTC",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "818999.50000000",
         "y" : 0
      },
      {
         "an" : "Kava",
         "as" : 4160828.05,
         "b" : "KAVA",
         "ba" : "",
         "c" : "5.2361",
         "cs" : 27190672,
         "etf" : false,
         "h" : "5.4000",
         "i" : "0.01000000",
         "l" : "4.5931",
         "o" : "4.8273",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "20775953.278702",
         "s" : "KAVAUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "4160828.0500",
         "y" : 0
      },
      {
         "an" : "Selfkey",
         "as" : 36814192,
         "b" : "KEY",
         "ba" : "",
         "c" : "0.00000663",
         "cs" : 4588195380,
         "etf" : false,
         "h" : "0.00000686",
         "i" : "1.00000000",
         "l" : "0.00000566",
         "o" : "0.00000583",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "227.68192065",
         "s" : "KEYETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "36814192.00000000",
         "y" : 0
      },
      {
         "an" : "Selfkey",
         "as" : 765984280,
         "b" : "KEY",
         "ba" : "",
         "c" : "0.016958",
         "cs" : 4588195380,
         "etf" : false,
         "h" : "0.017581",
         "i" : "1.00000000",
         "l" : "0.013868",
         "o" : "0.014667",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "11952586.460412",
         "s" : "KEYUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "765984280.000000",
         "y" : 0
      },
      {
         "an" : "Komodo",
         "as" : 3702570.93,
         "b" : "KMD",
         "ba" : "",
         "c" : "0.00005068",
         "cs" : 125505282,
         "etf" : false,
         "h" : "0.00005387",
         "i" : "0.01000000",
         "l" : "0.00004394",
         "o" : "0.00004515",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "181.93194542",
         "s" : "KMDBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "3702570.93000000",
         "y" : 0
      },
      {
         "an" : "Komodo",
         "as" : 469546.88,
         "b" : "KMD",
         "ba" : "",
         "c" : "0.001088",
         "cs" : 125505282,
         "etf" : false,
         "h" : "0.001152",
         "i" : "0.01000000",
         "l" : "0.000944",
         "o" : "0.000968",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "501.57533983",
         "s" : "KMDETH",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "469546.880000",
         "y" : 0
      },
      {
         "an" : "Komodo",
         "as" : 3100197.62,
         "b" : "KMD",
         "ba" : "",
         "c" : "2.7823",
         "cs" : 125505282,
         "etf" : false,
         "h" : "2.8900",
         "i" : "0.01000000",
         "l" : "2.3357",
         "o" : "2.4154",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "8232307.839382",
         "s" : "KMDUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "3100197.6200",
         "y" : 0
      },
      {
         "an" : "KyberNetwork",
         "as" : 289721.064,
         "b" : "KNC",
         "ba" : "",
         "c" : "2.913",
         "cs" : 205045092,
         "etf" : false,
         "h" : "2.956",
         "i" : "0.00100000",
         "l" : "2.628",
         "o" : "2.705",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "809706.943057",
         "s" : "KNCBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "289721.064",
         "y" : 0
      },
      {
         "an" : "KyberNetwork",
         "as" : 3954708.36,
         "b" : "KNC",
         "ba" : "",
         "c" : "2.915",
         "cs" : 205045092,
         "etf" : false,
         "h" : "2.960",
         "i" : "0.00100000",
         "l" : "2.624",
         "o" : "2.696",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "11082807.292884",
         "s" : "KNCUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "3954708.360",
         "y" : 0
      },
      {
         "an" : "Keep3rV1",
         "as" : 1223.096,
         "b" : "KP3R",
         "ba" : "",
         "c" : "0.5110",
         "cs" : null,
         "etf" : false,
         "h" : "0.5576",
         "i" : "0.00100000",
         "l" : "0.5066",
         "o" : "0.5478",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "648.3688184",
         "s" : "KP3RBNB",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "1223.0960",
         "y" : 0
      },
      {
         "an" : "Keep3rV1",
         "as" : 3741.46426,
         "b" : "KP3R",
         "ba" : "",
         "c" : "292.41",
         "cs" : null,
         "etf" : false,
         "h" : "305.00",
         "i" : "0.00001000",
         "l" : "272.00",
         "o" : "298.40",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1072235.7906847",
         "s" : "KP3RBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.01",
         "v" : "3741.47",
         "y" : 0
      },
      {
         "an" : "Kusama",
         "as" : 4846.53,
         "b" : "KSM",
         "ba" : "",
         "c" : "0.7036",
         "cs" : null,
         "etf" : false,
         "h" : "0.7733",
         "i" : "0.01000000",
         "l" : "0.6793",
         "o" : "0.6882",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "3509.880317",
         "s" : "KSMBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "Polkadot"
         ],
         "ts" : "0.0001",
         "v" : "4846.5300",
         "y" : 0
      },
      {
         "an" : "Kusama",
         "as" : 12812.8877,
         "b" : "KSM",
         "ba" : "",
         "c" : "403.85",
         "cs" : null,
         "etf" : false,
         "h" : "419.78",
         "i" : "0.00010000",
         "l" : "362.89",
         "o" : "370.63",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "5001659.740046",
         "s" : "KSMBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "Polkadot"
         ],
         "ts" : "0.01",
         "v" : "12812.89",
         "y" : 0
      },
      {
         "an" : "Kusama",
         "as" : 120564.3366,
         "b" : "KSM",
         "ba" : "",
         "c" : "403.76",
         "cs" : null,
         "etf" : false,
         "h" : "419.85",
         "i" : "0.00010000",
         "l" : "362.70",
         "o" : "370.54",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "47142034.475482",
         "s" : "KSMUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "Polkadot"
         ],
         "ts" : "0.01",
         "v" : "120564.34",
         "y" : 0
      },
      {
         "an" : "Linear",
         "as" : 40659985,
         "b" : "LINA",
         "ba" : "",
         "c" : "0.00000164",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000174",
         "i" : "1.00000000",
         "l" : "0.00000154",
         "o" : "0.00000165",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "66.66466293",
         "s" : "LINABTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC",
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "40659985.00000000",
         "y" : 0
      },
      {
         "an" : "Linear",
         "as" : 64469530.21,
         "b" : "LINA",
         "ba" : "",
         "c" : "0.0901",
         "cs" : null,
         "etf" : false,
         "h" : "0.0938",
         "i" : "0.01000000",
         "l" : "0.0813",
         "o" : "0.0888",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "5698311.637359",
         "s" : "LINABUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "64469530.2100",
         "y" : 0
      },
      {
         "an" : "Linear",
         "as" : 341011051.67,
         "b" : "LINA",
         "ba" : "",
         "c" : "0.0900",
         "cs" : null,
         "etf" : false,
         "h" : "0.0939",
         "i" : "0.01000000",
         "l" : "0.0811",
         "o" : "0.0886",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "30066202.526207",
         "s" : "LINAUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "341011051.6700",
         "y" : 0
      },
      {
         "an" : "ChainLink",
         "as" : 20720.128,
         "b" : "LINK",
         "ba" : "",
         "c" : "47.128",
         "cs" : 419009556,
         "etf" : false,
         "h" : "48.056",
         "i" : "0.00100000",
         "l" : "43.046",
         "o" : "45.066",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "AUD",
         "qa" : "",
         "qn" : "Australian Dollar",
         "qv" : "951516.197157",
         "s" : "LINKAUD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "20720.128",
         "y" : 0
      },
      {
         "an" : "ChainLink",
         "as" : 16374.56889,
         "b" : "LINK",
         "ba" : "",
         "c" : "199.20",
         "cs" : 419009556,
         "etf" : false,
         "h" : "203.46",
         "i" : "0.00001000",
         "l" : "183.27",
         "o" : "192.91",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BRL",
         "qa" : "",
         "qn" : "Brazilian Real",
         "qv" : "3148367.6775946",
         "s" : "LINKBRL",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.01",
         "v" : "16374.57",
         "y" : 0
      },
      {
         "an" : "ChainLink",
         "as" : 341963.893,
         "b" : "LINK",
         "ba" : "",
         "c" : "36.503",
         "cs" : 419009556,
         "etf" : false,
         "h" : "37.199",
         "i" : "0.00100000",
         "l" : "33.423",
         "o" : "34.924",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "12059449.980591",
         "s" : "LINKBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "341963.893",
         "y" : 0
      },
      {
         "an" : "LINKDOWN",
         "as" : 792247.24,
         "b" : "LINKDOWN",
         "ba" : null,
         "c" : "1.7341",
         "cs" : null,
         "etf" : true,
         "h" : "2.1740",
         "i" : "0.01000000",
         "l" : "1.6254",
         "o" : "1.9661",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "1490123.314859",
         "s" : "LINKDOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "792247.2400",
         "y" : 0
      },
      {
         "an" : "ChainLink",
         "as" : 86523.588,
         "b" : "LINK",
         "ba" : "",
         "c" : "30.260",
         "cs" : 419009556,
         "etf" : false,
         "h" : "30.817",
         "i" : "0.00100000",
         "l" : "27.729",
         "o" : "28.921",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "2527320.747863",
         "s" : "LINKEUR",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "86523.588",
         "y" : 0
      },
      {
         "an" : "ChainLink",
         "as" : 31826.992,
         "b" : "LINK",
         "ba" : "",
         "c" : "26.336",
         "cs" : 419009556,
         "etf" : false,
         "h" : "26.819",
         "i" : "0.00100000",
         "l" : "24.130",
         "o" : "25.157",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "GBP",
         "qa" : "",
         "qn" : "Pound Sterling",
         "qv" : "815937.454942",
         "s" : "LINKGBP",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "31826.992",
         "y" : 0
      },
      {
         "an" : "ChainLink",
         "as" : 30175.62,
         "b" : "LINK",
         "ba" : "",
         "c" : "297.2",
         "cs" : 419009556,
         "etf" : false,
         "h" : "302.9",
         "i" : "0.01000000",
         "l" : "272.0",
         "o" : "284.1",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "8677586.19",
         "s" : "LINKTRY",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.1",
         "v" : "30175.7",
         "y" : 0
      },
      {
         "an" : "ChainLink",
         "as" : 2778.436,
         "b" : "LINK",
         "ba" : "",
         "c" : "36.530",
         "cs" : 419009556,
         "etf" : false,
         "h" : "37.000",
         "i" : "0.00100000",
         "l" : "33.446",
         "o" : "34.831",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TUSD",
         "qa" : "",
         "qn" : "TrueUSD",
         "qv" : "97763.606684",
         "s" : "LINKTUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "2778.436",
         "y" : 0
      },
      {
         "an" : "LINKUP",
         "as" : 351350.6,
         "b" : "LINKUP",
         "ba" : null,
         "c" : "14.501",
         "cs" : null,
         "etf" : true,
         "h" : "15.200",
         "i" : "0.01000000",
         "l" : "11.950",
         "o" : "13.195",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "4760750.75606",
         "s" : "LINKUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "351350.600",
         "y" : 0
      },
      {
         "an" : "ChainLink",
         "as" : 60564.729,
         "b" : "LINK",
         "ba" : "",
         "c" : "36.449",
         "cs" : 419009556,
         "etf" : false,
         "h" : "37.150",
         "i" : "0.00100000",
         "l" : "33.200",
         "o" : "34.872",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDC",
         "qa" : "",
         "qn" : "USD Coin",
         "qv" : "2129716.860386",
         "s" : "LINKUSDC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "60564.729",
         "y" : 0
      },
      {
         "an" : "ChainLink",
         "as" : 5303727.379,
         "b" : "LINK",
         "ba" : "",
         "c" : "36.428",
         "cs" : 419009556,
         "etf" : false,
         "h" : "37.149",
         "i" : "0.00100000",
         "l" : "33.406",
         "o" : "34.855",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "186443969.929261",
         "s" : "LINKUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "5303727.379",
         "y" : 0
      },
      {
         "an" : "Litentry",
         "as" : 262005.4,
         "b" : "LIT",
         "ba" : "",
         "c" : "0.00016714",
         "cs" : null,
         "etf" : false,
         "h" : "0.00017130",
         "i" : "0.10000000",
         "l" : "0.00014626",
         "o" : "0.00015080",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "41.80575066",
         "s" : "LITBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "Polkadot"
         ],
         "ts" : "0.00000001",
         "v" : "262005.40000000",
         "y" : 0
      },
      {
         "an" : "Litentry",
         "as" : 342647.92,
         "b" : "LIT",
         "ba" : "",
         "c" : "9.1982",
         "cs" : null,
         "etf" : false,
         "h" : "9.3624",
         "i" : "0.01000000",
         "l" : "7.7474",
         "o" : "8.1255",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2933213.581958",
         "s" : "LITBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "Polkadot"
         ],
         "ts" : "0.0001",
         "v" : "342647.9200",
         "y" : 0
      },
      {
         "an" : "Litentry",
         "as" : 2751310.42,
         "b" : "LIT",
         "ba" : "",
         "c" : "9.2000",
         "cs" : null,
         "etf" : false,
         "h" : "9.3543",
         "i" : "0.01000000",
         "l" : "7.5060",
         "o" : "8.1101",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "23716755.891714",
         "s" : "LITUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "Polkadot"
         ],
         "ts" : "0.0001",
         "v" : "2751310.4200",
         "y" : 0
      },
      {
         "an" : "Loom Network",
         "as" : 17564269,
         "b" : "LOOM",
         "ba" : "",
         "c" : "0.00000267",
         "cs" : 1000000000,
         "etf" : false,
         "h" : "0.00000276",
         "i" : "1.00000000",
         "l" : "0.00000250",
         "o" : "0.00000256",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "46.14947866",
         "s" : "LOOMBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "17564269.00000000",
         "y" : 0
      },
      {
         "an" : "Loom Network",
         "as" : 3806847,
         "b" : "LOOM",
         "ba" : "",
         "c" : "0.00005755",
         "cs" : 1000000000,
         "etf" : false,
         "h" : "0.00005898",
         "i" : "1.00000000",
         "l" : "0.00005374",
         "o" : "0.00005480",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "214.01695908",
         "s" : "LOOMETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "3806847.00000000",
         "y" : 0
      },
      {
         "an" : "Loopring",
         "as" : 4007348,
         "b" : "LRC",
         "ba" : "",
         "c" : "0.00000953",
         "cs" : 1222767707,
         "etf" : false,
         "h" : "0.00000956",
         "i" : "1.00000000",
         "l" : "0.00000902",
         "o" : "0.00000912",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "37.16592532",
         "s" : "LRCBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "4007348.00000000",
         "y" : 0
      },
      {
         "an" : "Loopring",
         "as" : 650605.6,
         "b" : "LRC",
         "ba" : "",
         "c" : "0.52284",
         "cs" : 1222767707,
         "etf" : false,
         "h" : "0.52578",
         "i" : "0.10000000",
         "l" : "0.47733",
         "o" : "0.48882",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "324363.547533",
         "s" : "LRCBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00001",
         "v" : "650605.60000",
         "y" : 0
      },
      {
         "an" : "Loopring",
         "as" : 1688596,
         "b" : "LRC",
         "ba" : "",
         "c" : "0.00020431",
         "cs" : 1222767707,
         "etf" : false,
         "h" : "0.00020479",
         "i" : "1.00000000",
         "l" : "0.00019389",
         "o" : "0.00019553",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "336.120821",
         "s" : "LRCETH",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "1688596.00000000",
         "y" : 0
      },
      {
         "an" : "Loopring",
         "as" : 18272327.6,
         "b" : "LRC",
         "ba" : "",
         "c" : "0.52368",
         "cs" : 1222767707,
         "etf" : false,
         "h" : "0.52446",
         "i" : "0.10000000",
         "l" : "0.47622",
         "o" : "0.49043",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "9121335.395448",
         "s" : "LRCUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00001",
         "v" : "18272327.60000",
         "y" : 0
      },
      {
         "an" : "Lisk",
         "as" : 899519.83,
         "b" : "LSK",
         "ba" : "",
         "c" : "0.0000886",
         "cs" : 127891000,
         "etf" : false,
         "h" : "0.0000915",
         "i" : "0.01000000",
         "l" : "0.0000823",
         "o" : "0.0000838",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "78.47930641",
         "s" : "LSKBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "899519.8300000",
         "y" : 0
      },
      {
         "an" : "Lisk",
         "as" : 81712.3,
         "b" : "LSK",
         "ba" : "",
         "c" : "0.001912",
         "cs" : 127891000,
         "etf" : false,
         "h" : "0.001958",
         "i" : "0.01000000",
         "l" : "0.001770",
         "o" : "0.001818",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "150.96229743",
         "s" : "LSKETH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "81712.300000",
         "y" : 0
      },
      {
         "an" : "Lisk",
         "as" : 1446379.06,
         "b" : "LSK",
         "ba" : "",
         "c" : "4.8672",
         "cs" : 127891000,
         "etf" : false,
         "h" : "4.9369",
         "i" : "0.01000000",
         "l" : "4.3629",
         "o" : "4.4953",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "6744108.245172",
         "s" : "LSKUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "1446379.0600",
         "y" : 0
      },
      {
         "an" : "Litecoin",
         "as" : 16138.345,
         "b" : "LTC",
         "ba" : "Å",
         "c" : "0.4473",
         "cs" : 66752414,
         "etf" : false,
         "h" : "0.4728",
         "i" : "0.00100000",
         "l" : "0.4456",
         "o" : "0.4517",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "7403.6102155",
         "s" : "LTCBNB",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "16138.3450",
         "y" : 0
      },
      {
         "an" : "Litecoin",
         "as" : 2937.2046,
         "b" : "LTC",
         "ba" : "Å",
         "c" : "1402.20",
         "cs" : 66752414,
         "etf" : false,
         "h" : "1418.01",
         "i" : "0.00001000",
         "l" : "1300.44",
         "o" : "1347.16",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BRL",
         "qa" : "",
         "qn" : "Brazilian Real",
         "qv" : "3960204.9568627",
         "s" : "LTCBRL",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "2937.21",
         "y" : 0
      },
      {
         "an" : "Litecoin",
         "as" : 87722.78589,
         "b" : "LTC",
         "ba" : "Å",
         "c" : "256.86",
         "cs" : 66752414,
         "etf" : false,
         "h" : "259.29",
         "i" : "0.00001000",
         "l" : "237.35",
         "o" : "243.47",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "21764811.2790542",
         "s" : "LTCBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "87722.79",
         "y" : 0
      },
      {
         "an" : "LTCDOWN",
         "as" : 229285238.21,
         "b" : "LTCDOWN",
         "ba" : "",
         "c" : "0.002706",
         "cs" : null,
         "etf" : true,
         "h" : "0.003300",
         "i" : "0.01000000",
         "l" : "0.002621",
         "o" : "0.003077",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "680526.80468038",
         "s" : "LTCDOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "229285238.210000",
         "y" : 0
      },
      {
         "an" : "Litecoin",
         "as" : 25180.69,
         "b" : "LTC",
         "ba" : "Å",
         "c" : "0.10028",
         "cs" : 66752414,
         "etf" : false,
         "h" : "0.10108",
         "i" : "0.00100000",
         "l" : "0.09612",
         "o" : "0.09707",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "2478.52796478",
         "s" : "LTCETH",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "25180.69000",
         "y" : 0
      },
      {
         "an" : "Litecoin",
         "as" : 20263.589,
         "b" : "LTC",
         "ba" : "Å",
         "c" : "212.80",
         "cs" : 66752414,
         "etf" : false,
         "h" : "214.84",
         "i" : "0.00010000",
         "l" : "196.79",
         "o" : "202.21",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "4159196.747132",
         "s" : "LTCEUR",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "20263.59",
         "y" : 0
      },
      {
         "an" : "Litecoin",
         "as" : 3816.70725,
         "b" : "LTC",
         "ba" : "Å",
         "c" : "185.15",
         "cs" : 66752414,
         "etf" : false,
         "h" : "186.99",
         "i" : "0.00001000",
         "l" : "171.32",
         "o" : "175.94",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "GBP",
         "qa" : "",
         "qn" : "Pound Sterling",
         "qv" : "681437.0192639",
         "s" : "LTCGBP",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "3816.71",
         "y" : 0
      },
      {
         "an" : "Litecoin",
         "as" : 198.18485,
         "b" : "LTC",
         "ba" : "Å",
         "c" : "255.45",
         "cs" : 66752414,
         "etf" : false,
         "h" : "258.95",
         "i" : "0.00001000",
         "l" : "237.36",
         "o" : "241.59",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "PAX",
         "qa" : "",
         "qn" : "Paxos Standard",
         "qv" : "48729.6142759",
         "s" : "LTCPAX",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "198.19",
         "y" : 0
      },
      {
         "an" : "Litecoin",
         "as" : 935.54318,
         "b" : "LTC",
         "ba" : "Å",
         "c" : "19339.6",
         "cs" : 66752414,
         "etf" : false,
         "h" : "19550.0",
         "i" : "0.00001000",
         "l" : "17951.3",
         "o" : "18369.3",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "RUB",
         "qa" : "",
         "qn" : "Russian Ruble",
         "qv" : "17475964.229585",
         "s" : "LTCRUB",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.1",
         "v" : "935.6",
         "y" : 0
      },
      {
         "an" : "Litecoin",
         "as" : 1026.42277,
         "b" : "LTC",
         "ba" : "Å",
         "c" : "256.08",
         "cs" : 66752414,
         "etf" : false,
         "h" : "258.12",
         "i" : "0.00001000",
         "l" : "237.32",
         "o" : "244.55",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TUSD",
         "qa" : "",
         "qn" : "TrueUSD",
         "qv" : "253535.1354957",
         "s" : "LTCTUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "1026.43",
         "y" : 0
      },
      {
         "an" : "LTCUP",
         "as" : 292564.83,
         "b" : "LTCUP",
         "ba" : "",
         "c" : "26.051",
         "cs" : null,
         "etf" : true,
         "h" : "26.589",
         "i" : "0.01000000",
         "l" : "22.050",
         "o" : "23.220",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "7124957.12176",
         "s" : "LTCUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "292564.830",
         "y" : 0
      },
      {
         "an" : "Litecoin",
         "as" : 9968.42207,
         "b" : "LTC",
         "ba" : "Å",
         "c" : "256.94",
         "cs" : 66752414,
         "etf" : false,
         "h" : "260.00",
         "i" : "0.00001000",
         "l" : "237.50",
         "o" : "242.69",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDC",
         "qa" : "",
         "qn" : "USD Coin",
         "qv" : "2461464.5038777",
         "s" : "LTCUSDC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "9968.43",
         "y" : 0
      },
      {
         "an" : "Litecoin",
         "as" : 1082706.6419,
         "b" : "LTC",
         "ba" : "Å",
         "c" : "256.47",
         "cs" : 66752414,
         "etf" : false,
         "h" : "259.00",
         "i" : "0.00001000",
         "l" : "237.25",
         "o" : "243.30",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "268180593.5785823",
         "s" : "LTCUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "1082706.65",
         "y" : 0
      },
      {
         "an" : "LTO Network",
         "as" : 19322038,
         "b" : "LTO",
         "ba" : "",
         "c" : "0.00000987",
         "cs" : 229053494,
         "etf" : false,
         "h" : "0.00001000",
         "i" : "1.00000000",
         "l" : "0.00000948",
         "o" : "0.00000974",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "187.87059715",
         "s" : "LTOBTC",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00000001",
         "v" : "19322038.00000000",
         "y" : 0
      },
      {
         "an" : "LTO Network",
         "as" : 10600824.2,
         "b" : "LTO",
         "ba" : "",
         "c" : "0.54164",
         "cs" : 229053494,
         "etf" : false,
         "h" : "0.55127",
         "i" : "0.10000000",
         "l" : "0.49904",
         "o" : "0.52390",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "5572992.065988",
         "s" : "LTOUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00001",
         "v" : "10600824.20000",
         "y" : 0
      },
      {
         "an" : "Terra",
         "as" : 708809.7,
         "b" : "LUNA",
         "ba" : "",
         "c" : "0.03171",
         "cs" : null,
         "etf" : false,
         "h" : "0.03703",
         "i" : "0.10000000",
         "l" : "0.03148",
         "o" : "0.03208",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "24004.082189",
         "s" : "LUNABNB",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "708809.70000",
         "y" : 0
      },
      {
         "an" : "Terra",
         "as" : 2352953.9,
         "b" : "LUNA",
         "ba" : "",
         "c" : "0.00033113",
         "cs" : null,
         "etf" : false,
         "h" : "0.00036500",
         "i" : "0.10000000",
         "l" : "0.00031592",
         "o" : "0.00032112",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "795.4075945",
         "s" : "LUNABTC",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "2352953.90000000",
         "y" : 0
      },
      {
         "an" : "Terra",
         "as" : 900798.21,
         "b" : "LUNA",
         "ba" : "",
         "c" : "18.203",
         "cs" : null,
         "etf" : false,
         "h" : "19.615",
         "i" : "0.00100000",
         "l" : "16.902",
         "o" : "17.289",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "16534184.745861",
         "s" : "LUNABUSD",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "900798.210",
         "y" : 0
      },
      {
         "an" : "Terra",
         "as" : 215873.219,
         "b" : "LUNA",
         "ba" : "",
         "c" : "15.064",
         "cs" : null,
         "etf" : false,
         "h" : "16.270",
         "i" : "0.00100000",
         "l" : "14.037",
         "o" : "14.343",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "3293715.634579",
         "s" : "LUNAEUR",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "215873.219",
         "y" : 0
      },
      {
         "an" : "Terra",
         "as" : 11497766.455,
         "b" : "LUNA",
         "ba" : "",
         "c" : "18.176",
         "cs" : null,
         "etf" : false,
         "h" : "19.610",
         "i" : "0.00100000",
         "l" : "16.861",
         "o" : "17.268",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "209353521.015834",
         "s" : "LUNAUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "11497766.455",
         "y" : 0
      },
      {
         "an" : "Decentraland",
         "as" : 5215134,
         "b" : "MANA",
         "ba" : "",
         "c" : "0.00002370",
         "cs" : 1578089913,
         "etf" : false,
         "h" : "0.00002429",
         "i" : "1.00000000",
         "l" : "0.00002216",
         "o" : "0.00002250",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "121.19941402",
         "s" : "MANABTC",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.00000001",
         "v" : "5215134.00000000",
         "y" : 0
      },
      {
         "an" : "Decentraland",
         "as" : 1975680.39,
         "b" : "MANA",
         "ba" : "",
         "c" : "1.3024",
         "cs" : 1578089913,
         "etf" : false,
         "h" : "1.3285",
         "i" : "0.01000000",
         "l" : "1.1762",
         "o" : "1.2104",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2467591.307036",
         "s" : "MANABUSD",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.0001",
         "v" : "1975680.3900",
         "y" : 0
      },
      {
         "an" : "Decentraland",
         "as" : 351368,
         "b" : "MANA",
         "ba" : "",
         "c" : "0.00050832",
         "cs" : 1578089913,
         "etf" : false,
         "h" : "0.00051908",
         "i" : "1.00000000",
         "l" : "0.00047704",
         "o" : "0.00048335",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "175.43327187",
         "s" : "MANAETH",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.00000001",
         "v" : "351368.00000000",
         "y" : 0
      },
      {
         "an" : "Decentraland",
         "as" : 33928957.48,
         "b" : "MANA",
         "ba" : "",
         "c" : "1.3010",
         "cs" : 1578089913,
         "etf" : false,
         "h" : "1.3300",
         "i" : "0.01000000",
         "l" : "1.1735",
         "o" : "1.2094",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "42330391.555059",
         "s" : "MANAUSDT",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.0001",
         "v" : "33928957.4800",
         "y" : 0
      },
      {
         "an" : "Polygon",
         "as" : 15827732,
         "b" : "MATIC",
         "ba" : "",
         "c" : "0.0010225",
         "cs" : 3100000000,
         "etf" : false,
         "h" : "0.0011500",
         "i" : "1.00000000",
         "l" : "0.0008850",
         "o" : "0.0008850",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "16409.0463416",
         "s" : "MATICBNB",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.0000001",
         "v" : "15827732.0000000",
         "y" : 0
      },
      {
         "an" : "Polygon",
         "as" : 187409444,
         "b" : "MATIC",
         "ba" : "",
         "c" : "0.00001065",
         "cs" : 3100000000,
         "etf" : false,
         "h" : "0.00001142",
         "i" : "1.00000000",
         "l" : "0.00000886",
         "o" : "0.00000892",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "1910.99838115",
         "s" : "MATICBTC",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00000001",
         "v" : "187409444.00000000",
         "y" : 0
      },
      {
         "an" : "Polygon",
         "as" : 79853657.5,
         "b" : "MATIC",
         "ba" : "",
         "c" : "0.58430",
         "cs" : 3100000000,
         "etf" : false,
         "h" : "0.62354",
         "i" : "0.10000000",
         "l" : "0.47578",
         "o" : "0.47841",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "44065437.394155",
         "s" : "MATICBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00001",
         "v" : "79853657.50000",
         "y" : 0
      },
      {
         "an" : "Polygon",
         "as" : 19786336.16,
         "b" : "MATIC",
         "ba" : "",
         "c" : "0.4855",
         "cs" : 3100000000,
         "etf" : false,
         "h" : "0.5176",
         "i" : "0.01000000",
         "l" : "0.3911",
         "o" : "0.3978",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "8996517.979424",
         "s" : "MATICEUR",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.0001",
         "v" : "19786336.1600",
         "y" : 0
      },
      {
         "an" : "Polygon",
         "as" : 938764067.4,
         "b" : "MATIC",
         "ba" : "",
         "c" : "0.58448",
         "cs" : 3100000000,
         "etf" : false,
         "h" : "0.62599",
         "i" : "0.10000000",
         "l" : "0.47516",
         "o" : "0.47923",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "516646158.735986",
         "s" : "MATICUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00001",
         "v" : "938764067.40000",
         "y" : 0
      },
      {
         "an" : "MovieBloc",
         "as" : 48435711,
         "b" : "MBL",
         "ba" : "",
         "c" : "0.00003146",
         "cs" : 6386714356,
         "etf" : false,
         "h" : "0.00003620",
         "i" : "1.00000000",
         "l" : "0.00002929",
         "o" : "0.00002959",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1570.45183964",
         "s" : "MBLBNB",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "48435711.00000000",
         "y" : 0
      },
      {
         "an" : "MovieBloc",
         "as" : 756243740,
         "b" : "MBL",
         "ba" : "",
         "c" : "0.017999",
         "cs" : 6386714356,
         "etf" : false,
         "h" : "0.019463",
         "i" : "1.00000000",
         "l" : "0.015620",
         "o" : "0.015919",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "13164637.972473",
         "s" : "MBLUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "756243740.000000",
         "y" : 0
      },
      {
         "an" : "Moeda Loyalty Points",
         "as" : 1242424,
         "b" : "MDA",
         "ba" : "",
         "c" : "0.00002368",
         "cs" : 0,
         "etf" : false,
         "h" : "0.00002458",
         "i" : "1.00000000",
         "l" : "0.00002322",
         "o" : "0.00002370",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "29.38590077",
         "s" : "MDABTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "1242424.00000000",
         "y" : 0
      },
      {
         "an" : "Measurable Data Token",
         "as" : 28386416,
         "b" : "MDT",
         "ba" : "",
         "c" : "0.00000119",
         "cs" : 399848586,
         "etf" : false,
         "h" : "0.00000123",
         "i" : "1.00000000",
         "l" : "0.00000111",
         "o" : "0.00000113",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "33.30794855",
         "s" : "MDTBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "28386416.00000000",
         "y" : 0
      },
      {
         "an" : "Measurable Data Token",
         "as" : 73827043.4,
         "b" : "MDT",
         "ba" : "",
         "c" : "0.06543",
         "cs" : 399848586,
         "etf" : false,
         "h" : "0.06775",
         "i" : "0.10000000",
         "l" : "0.05843",
         "o" : "0.06078",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "4660790.379817",
         "s" : "MDTUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "73827043.40000",
         "y" : 0
      },
      {
         "an" : "Mainframe",
         "as" : 58142223,
         "b" : "MFT",
         "ba" : "",
         "c" : "0.00003121",
         "cs" : 10000000000,
         "etf" : false,
         "h" : "0.00003437",
         "i" : "1.00000000",
         "l" : "0.00002830",
         "o" : "0.00002891",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1844.44414706",
         "s" : "MFTBNB",
         "st" : "TRADING",
         "tags" : [
            "mining-zone",
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "58142223.00000000",
         "y" : 0
      },
      {
         "an" : "Mainframe",
         "as" : 43314348,
         "b" : "MFT",
         "ba" : "",
         "c" : "0.00000698",
         "cs" : 10000000000,
         "etf" : false,
         "h" : "0.00000731",
         "i" : "1.00000000",
         "l" : "0.00000606",
         "o" : "0.00000623",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "286.85257993",
         "s" : "MFTETH",
         "st" : "TRADING",
         "tags" : [
            "mining-zone",
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "43314348.00000000",
         "y" : 0
      },
      {
         "an" : "Mainframe",
         "as" : 601733704,
         "b" : "MFT",
         "ba" : "",
         "c" : "0.017884",
         "cs" : 10000000000,
         "etf" : false,
         "h" : "0.018449",
         "i" : "1.00000000",
         "l" : "0.014996",
         "o" : "0.015617",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "10146001.26882",
         "s" : "MFTUSDT",
         "st" : "TRADING",
         "tags" : [
            "mining-zone",
            "defi"
         ],
         "ts" : "0.000001",
         "v" : "601733704.000000",
         "y" : 0
      },
      {
         "an" : "Mirror Protocol",
         "as" : 457205.83,
         "b" : "MIR",
         "ba" : "",
         "c" : "0.0001902",
         "cs" : null,
         "etf" : false,
         "h" : "0.0001962",
         "i" : "0.01000000",
         "l" : "0.0001710",
         "o" : "0.0001758",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "84.47662534",
         "s" : "MIRBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone"
         ],
         "ts" : "0.0000001",
         "v" : "457205.8300000",
         "y" : 0
      },
      {
         "an" : "Mirror Protocol",
         "as" : 226232.811,
         "b" : "MIR",
         "ba" : "",
         "c" : "10.500",
         "cs" : null,
         "etf" : false,
         "h" : "10.908",
         "i" : "0.00100000",
         "l" : "9.126",
         "o" : "9.459",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2262869.829264",
         "s" : "MIRBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone"
         ],
         "ts" : "0.001",
         "v" : "226232.811",
         "y" : 0
      },
      {
         "an" : "Mithril",
         "as" : 15282858,
         "b" : "MITH",
         "ba" : "",
         "c" : "0.0001862",
         "cs" : 1000000000,
         "etf" : false,
         "h" : "0.0002070",
         "i" : "1.00000000",
         "l" : "0.0001801",
         "o" : "0.0001940",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "2921.2435115",
         "s" : "MITHBNB",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "15282858.0000000",
         "y" : 0
      },
      {
         "an" : "Mithril",
         "as" : 80385917,
         "b" : "MITH",
         "ba" : "",
         "c" : "0.00000194",
         "cs" : 1000000000,
         "etf" : false,
         "h" : "0.00000205",
         "i" : "1.00000000",
         "l" : "0.00000178",
         "o" : "0.00000195",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "153.21607975",
         "s" : "MITHBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "80385917.00000000",
         "y" : 0
      },
      {
         "an" : "Mithril",
         "as" : 376082899.5,
         "b" : "MITH",
         "ba" : "",
         "c" : "0.10679",
         "cs" : 1000000000,
         "etf" : false,
         "h" : "0.10983",
         "i" : "0.10000000",
         "l" : "0.09400",
         "o" : "0.10432",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "38529657.340897",
         "s" : "MITHUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "376082899.50000",
         "y" : 0
      },
      {
         "an" : "Maker",
         "as" : 178.629,
         "b" : "MKR",
         "ba" : "",
         "c" : "7.701",
         "cs" : 668228,
         "etf" : false,
         "h" : "8.340",
         "i" : "0.00100000",
         "l" : "7.327",
         "o" : "7.432",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1370.476525",
         "s" : "MKRBNB",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "178.629",
         "y" : 0
      },
      {
         "an" : "Maker",
         "as" : 1581.429,
         "b" : "MKR",
         "ba" : "",
         "c" : "0.080376",
         "cs" : 668228,
         "etf" : false,
         "h" : "0.083038",
         "i" : "0.00100000",
         "l" : "0.073000",
         "o" : "0.074340",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "123.04309724",
         "s" : "MKRBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.000001",
         "v" : "1581.429000",
         "y" : 0
      },
      {
         "an" : "Maker",
         "as" : 663.05678,
         "b" : "MKR",
         "ba" : "",
         "c" : "4416.85",
         "cs" : 668228,
         "etf" : false,
         "h" : "4536.32",
         "i" : "0.00001000",
         "l" : "3847.97",
         "o" : "3989.76",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2766090.5475522",
         "s" : "MKRBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.01",
         "v" : "663.06",
         "y" : 0
      },
      {
         "an" : "Maker",
         "as" : 8105.71553,
         "b" : "MKR",
         "ba" : "",
         "c" : "4413.21",
         "cs" : 668228,
         "etf" : false,
         "h" : "4533.11",
         "i" : "0.00001000",
         "l" : "3843.37",
         "o" : "3998.00",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "34156714.6492033",
         "s" : "MKRUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.01",
         "v" : "8105.72",
         "y" : 0
      },
      {
         "an" : "Monetha",
         "as" : 34063464,
         "b" : "MTH",
         "ba" : "",
         "c" : "0.00000087",
         "cs" : 402400000,
         "etf" : false,
         "h" : "0.00000094",
         "i" : "1.00000000",
         "l" : "0.00000077",
         "o" : "0.00000083",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "29.31647887",
         "s" : "MTHBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "34063464.00000000",
         "y" : 0
      },
      {
         "an" : "Metal",
         "as" : 1190984,
         "b" : "MTL",
         "ba" : "",
         "c" : "0.00007087",
         "cs" : 65588845,
         "etf" : false,
         "h" : "0.00007413",
         "i" : "1.00000000",
         "l" : "0.00006579",
         "o" : "0.00006717",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "83.10125195",
         "s" : "MTLBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "1190984.00000000",
         "y" : 0
      },
      {
         "an" : "Metal",
         "as" : 285963.64,
         "b" : "MTL",
         "ba" : "",
         "c" : "0.001517",
         "cs" : 65588845,
         "etf" : false,
         "h" : "0.001586",
         "i" : "0.01000000",
         "l" : "0.001416",
         "o" : "0.001442",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "426.96999475",
         "s" : "MTLETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "285963.640000",
         "y" : 0
      },
      {
         "an" : "Metal",
         "as" : 3677974.98,
         "b" : "MTL",
         "ba" : "",
         "c" : "3.8784",
         "cs" : 65588845,
         "etf" : false,
         "h" : "3.9846",
         "i" : "0.01000000",
         "l" : "3.4983",
         "o" : "3.6118",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "13764636.750032",
         "s" : "MTLUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "3677974.9800",
         "y" : 0
      },
      {
         "an" : "NANO",
         "as" : 810996.65,
         "b" : "NANO",
         "ba" : "",
         "c" : "0.0001565",
         "cs" : 0,
         "etf" : false,
         "h" : "0.0001622",
         "i" : "0.01000000",
         "l" : "0.0001434",
         "o" : "0.0001464",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "124.46253002",
         "s" : "NANOBTC",
         "st" : "TRADING",
         "tags" : [
            "pow"
         ],
         "ts" : "0.0000001",
         "v" : "810996.6500000",
         "y" : 0
      },
      {
         "an" : "NANO",
         "as" : 269096.32,
         "b" : "NANO",
         "ba" : "",
         "c" : "8.6153",
         "cs" : 0,
         "etf" : false,
         "h" : "8.9478",
         "i" : "0.01000000",
         "l" : "7.6746",
         "o" : "7.8819",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2218032.41409",
         "s" : "NANOBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow"
         ],
         "ts" : "0.0001",
         "v" : "269096.3200",
         "y" : 0
      },
      {
         "an" : "NANO",
         "as" : 77991.15,
         "b" : "NANO",
         "ba" : "",
         "c" : "0.003349",
         "cs" : 0,
         "etf" : false,
         "h" : "0.003479",
         "i" : "0.01000000",
         "l" : "0.003080",
         "o" : "0.003129",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "256.82202718",
         "s" : "NANOETH",
         "st" : "TRADING",
         "tags" : [
            "pow"
         ],
         "ts" : "0.000001",
         "v" : "77991.150000",
         "y" : 0
      },
      {
         "an" : "NANO",
         "as" : 3565643.98,
         "b" : "NANO",
         "ba" : "",
         "c" : "8.6122",
         "cs" : 0,
         "etf" : false,
         "h" : "8.9500",
         "i" : "0.01000000",
         "l" : "7.6665",
         "o" : "7.8823",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "29475003.696354",
         "s" : "NANOUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow"
         ],
         "ts" : "0.0001",
         "v" : "3565643.9800",
         "y" : 0
      },
      {
         "an" : "Nebulas",
         "as" : 1203905.12,
         "b" : "NAS",
         "ba" : "",
         "c" : "0.0000188",
         "cs" : 58696762,
         "etf" : false,
         "h" : "0.0000200",
         "i" : "0.01000000",
         "l" : "0.0000173",
         "o" : "0.0000174",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "22.18633078",
         "s" : "NASBTC",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.0000001",
         "v" : "1203905.1200000",
         "y" : 0
      },
      {
         "an" : "Nebulas",
         "as" : 851855.35,
         "b" : "NAS",
         "ba" : "",
         "c" : "0.0004040",
         "cs" : 58696762,
         "etf" : false,
         "h" : "0.0004269",
         "i" : "0.01000000",
         "l" : "0.0003709",
         "o" : "0.0003722",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "332.91346782",
         "s" : "NASETH",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.0000001",
         "v" : "851855.3500000",
         "y" : 0
      },
      {
         "an" : "Navcoin",
         "as" : 1176813,
         "b" : "NAV",
         "ba" : "",
         "c" : "0.00001211",
         "cs" : 70992368,
         "etf" : false,
         "h" : "0.00001221",
         "i" : "1.00000000",
         "l" : "0.00001144",
         "o" : "0.00001183",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "13.86194914",
         "s" : "NAVBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "1176813.00000000",
         "y" : 0
      },
      {
         "an" : "Navcoin",
         "as" : 320431,
         "b" : "NAV",
         "ba" : "",
         "c" : "0.0002615",
         "cs" : 70992368,
         "etf" : false,
         "h" : "0.0002676",
         "i" : "1.00000000",
         "l" : "0.0002474",
         "o" : "0.0002519",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "81.3102639",
         "s" : "NAVETH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "320431.0000000",
         "y" : 0
      },
      {
         "an" : "New BitShares",
         "as" : 82882616,
         "b" : "NBS",
         "ba" : "",
         "c" : "0.00000059",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000062",
         "i" : "1.00000000",
         "l" : "0.00000053",
         "o" : "0.00000057",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "47.01824384",
         "s" : "NBSBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "pos",
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "82882616.00000000",
         "y" : 0
      },
      {
         "an" : "New BitShares",
         "as" : 149095151.5,
         "b" : "NBS",
         "ba" : "",
         "c" : "0.03189",
         "cs" : null,
         "etf" : false,
         "h" : "0.03400",
         "i" : "0.10000000",
         "l" : "0.02774",
         "o" : "0.03026",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "4584926.802864",
         "s" : "NBSUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "pos",
            "defi"
         ],
         "ts" : "0.00001",
         "v" : "149095151.50000",
         "y" : 0
      },
      {
         "an" : "Nucleus Vision",
         "as" : 133431701,
         "b" : "NCASH",
         "ba" : "",
         "c" : "0.00000269",
         "cs" : 7124866221,
         "etf" : false,
         "h" : "0.00000280",
         "i" : "1.00000000",
         "l" : "0.00000256",
         "o" : "0.00000267",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "359.13422588",
         "s" : "NCASHETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "133431701.00000000",
         "y" : 0
      },
      {
         "an" : "NEAR Protocol",
         "as" : 71732,
         "b" : "NEAR",
         "ba" : "",
         "c" : "0.00900",
         "cs" : null,
         "etf" : false,
         "h" : "0.00959",
         "i" : "0.10000000",
         "l" : "0.00899",
         "o" : "0.00948",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "667.018963",
         "s" : "NEARBNB",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00001",
         "v" : "71732.00000",
         "y" : 0
      },
      {
         "an" : "NEAR Protocol",
         "as" : 724990.8,
         "b" : "NEAR",
         "ba" : "",
         "c" : "0.0000938",
         "cs" : null,
         "etf" : false,
         "h" : "0.0000960",
         "i" : "0.01000000",
         "l" : "0.0000896",
         "o" : "0.0000947",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "67.14098577",
         "s" : "NEARBTC",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.0000001",
         "v" : "724990.8000000",
         "y" : 0
      },
      {
         "an" : "NEAR Protocol",
         "as" : 399586.16,
         "b" : "NEAR",
         "ba" : "",
         "c" : "5.1679",
         "cs" : null,
         "etf" : false,
         "h" : "5.2522",
         "i" : "0.01000000",
         "l" : "4.8024",
         "o" : "5.0993",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2008174.103971",
         "s" : "NEARBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.0001",
         "v" : "399586.1600",
         "y" : 0
      },
      {
         "an" : "NEAR Protocol",
         "as" : 7451896.12,
         "b" : "NEAR",
         "ba" : "",
         "c" : "5.1600",
         "cs" : null,
         "etf" : false,
         "h" : "5.2500",
         "i" : "0.01000000",
         "l" : "4.7940",
         "o" : "5.0900",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "37282742.902805",
         "s" : "NEARUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.0001",
         "v" : "7451896.1200",
         "y" : 0
      },
      {
         "an" : "Neblio",
         "as" : 203810,
         "b" : "NEBL",
         "ba" : "",
         "c" : "0.00005356",
         "cs" : 17407073,
         "etf" : false,
         "h" : "0.00005473",
         "i" : "1.00000000",
         "l" : "0.00005066",
         "o" : "0.00005270",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "10.79111936",
         "s" : "NEBLBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "203810.00000000",
         "y" : 0
      },
      {
         "an" : "Neblio",
         "as" : 35071.16,
         "b" : "NEBL",
         "ba" : "",
         "c" : "0.001146",
         "cs" : 17407073,
         "etf" : false,
         "h" : "0.001178",
         "i" : "0.01000000",
         "l" : "0.001096",
         "o" : "0.001131",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "39.85726668",
         "s" : "NEBLETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "35071.160000",
         "y" : 0
      },
      {
         "an" : "NEO",
         "as" : 18969.1,
         "b" : "NEO",
         "ba" : "",
         "c" : "0.1644",
         "cs" : 65000000,
         "etf" : false,
         "h" : "0.1777",
         "i" : "0.01000000",
         "l" : "0.1611",
         "o" : "0.1646",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "3208.533521",
         "s" : "NEOBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "18969.1000",
         "y" : 0
      },
      {
         "an" : "NEO",
         "as" : 76514.726,
         "b" : "NEO",
         "ba" : "",
         "c" : "94.398",
         "cs" : 65000000,
         "etf" : false,
         "h" : "95.816",
         "i" : "0.00100000",
         "l" : "85.243",
         "o" : "88.752",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "6983116.33864",
         "s" : "NEOBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "76514.726",
         "y" : 0
      },
      {
         "an" : "NEO",
         "as" : 68513.31,
         "b" : "NEO",
         "ba" : "",
         "c" : "768.8",
         "cs" : 65000000,
         "etf" : false,
         "h" : "780.9",
         "i" : "0.01000000",
         "l" : "694.4",
         "o" : "723.7",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "50529712.383",
         "s" : "NEOTRY",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.1",
         "v" : "68513.4",
         "y" : 0
      },
      {
         "an" : "NEO",
         "as" : 3498.843,
         "b" : "NEO",
         "ba" : "",
         "c" : "94.223",
         "cs" : 65000000,
         "etf" : false,
         "h" : "95.838",
         "i" : "0.00100000",
         "l" : "85.440",
         "o" : "88.002",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDC",
         "qa" : "",
         "qn" : "USD Coin",
         "qv" : "316841.273929",
         "s" : "NEOUSDC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "3498.843",
         "y" : 0
      },
      {
         "an" : "NEO",
         "as" : 1582238.978,
         "b" : "NEO",
         "ba" : "",
         "c" : "94.209",
         "cs" : 65000000,
         "etf" : false,
         "h" : "95.714",
         "i" : "0.00100000",
         "l" : "85.028",
         "o" : "88.594",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "143483875.540831",
         "s" : "NEOUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "1582238.978",
         "y" : 0
      },
      {
         "an" : "NKN",
         "as" : 12708289,
         "b" : "NKN",
         "ba" : "",
         "c" : "0.00001263",
         "cs" : 583666666,
         "etf" : false,
         "h" : "0.00001318",
         "i" : "1.00000000",
         "l" : "0.00001155",
         "o" : "0.00001174",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "158.00356861",
         "s" : "NKNBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "12708289.00000000",
         "y" : 0
      },
      {
         "an" : "Numeraire",
         "as" : 1114.94,
         "b" : "NMR",
         "ba" : "",
         "c" : "0.1179",
         "cs" : 4911507,
         "etf" : false,
         "h" : "0.1257",
         "i" : "0.01000000",
         "l" : "0.1146",
         "o" : "0.1195",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "134.18605",
         "s" : "NMRBNB",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "1114.9400",
         "y" : 0
      },
      {
         "an" : "Numeraire",
         "as" : 15205.583,
         "b" : "NMR",
         "ba" : "",
         "c" : "0.001239",
         "cs" : 4911507,
         "etf" : false,
         "h" : "0.001253",
         "i" : "0.00100000",
         "l" : "0.001145",
         "o" : "0.001199",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "18.23600695",
         "s" : "NMRBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.000001",
         "v" : "15205.583000",
         "y" : 0
      },
      {
         "an" : "Numeraire",
         "as" : 5873.408,
         "b" : "NMR",
         "ba" : "",
         "c" : "68.288",
         "cs" : 4911507,
         "etf" : false,
         "h" : "68.854",
         "i" : "0.00100000",
         "l" : "61.000",
         "o" : "64.377",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "379500.970714",
         "s" : "NMRBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "5873.408",
         "y" : 0
      },
      {
         "an" : "Numeraire",
         "as" : 29874.61,
         "b" : "NMR",
         "ba" : "",
         "c" : "68.165",
         "cs" : 4911507,
         "etf" : false,
         "h" : "68.880",
         "i" : "0.00100000",
         "l" : "60.797",
         "o" : "64.500",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "1927612.764941",
         "s" : "NMRUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "29874.610",
         "y" : 0
      },
      {
         "an" : "Nuls",
         "as" : 5374814.26,
         "b" : "NULS",
         "ba" : "",
         "c" : "1.0976",
         "cs" : 99620739,
         "etf" : false,
         "h" : "1.1361",
         "i" : "0.01000000",
         "l" : "0.9888",
         "o" : "1.0322",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "5637635.068272",
         "s" : "NULSUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "5374814.2600",
         "y" : 0
      },
      {
         "an" : "Nexus",
         "as" : 536364,
         "b" : "NXS",
         "ba" : "",
         "c" : "0.00002515",
         "cs" : 70374743,
         "etf" : false,
         "h" : "0.00002596",
         "i" : "1.00000000",
         "l" : "0.00002404",
         "o" : "0.00002444",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "13.20270136",
         "s" : "NXSBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "536364.00000000",
         "y" : 0
      },
      {
         "an" : "openANX",
         "as" : 1759380,
         "b" : "OAX",
         "ba" : "",
         "c" : "0.00000638",
         "cs" : 76215393,
         "etf" : false,
         "h" : "0.00000650",
         "i" : "1.00000000",
         "l" : "0.00000600",
         "o" : "0.00000620",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "10.97047606",
         "s" : "OAXBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "1759380.00000000",
         "y" : 0
      },
      {
         "an" : "Ocean Protocol",
         "as" : 281039,
         "b" : "OCEAN",
         "ba" : "",
         "c" : "0.002331",
         "cs" : 337559409,
         "etf" : false,
         "h" : "0.002439",
         "i" : "0.10000000",
         "l" : "0.002243",
         "o" : "0.002255",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "659.761741",
         "s" : "OCEANBNB",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "281039.000000",
         "y" : 0
      },
      {
         "an" : "Ocean Protocol",
         "as" : 2849895,
         "b" : "OCEAN",
         "ba" : "",
         "c" : "0.00002424",
         "cs" : 337559409,
         "etf" : false,
         "h" : "0.00002456",
         "i" : "1.00000000",
         "l" : "0.00002243",
         "o" : "0.00002268",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "66.21954871",
         "s" : "OCEANBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "2849895.00000000",
         "y" : 0
      },
      {
         "an" : "Ocean Protocol",
         "as" : 1608489.75,
         "b" : "OCEAN",
         "ba" : "",
         "c" : "1.3320",
         "cs" : 337559409,
         "etf" : false,
         "h" : "1.3490",
         "i" : "0.01000000",
         "l" : "1.1892",
         "o" : "1.2185",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2019761.958723",
         "s" : "OCEANBUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "1608489.7500",
         "y" : 0
      },
      {
         "an" : "Ocean Protocol",
         "as" : 11256727.33,
         "b" : "OCEAN",
         "ba" : "",
         "c" : "1.3303",
         "cs" : 337559409,
         "etf" : false,
         "h" : "1.3479",
         "i" : "0.01000000",
         "l" : "1.1868",
         "o" : "1.2188",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "14130957.014279",
         "s" : "OCEANUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "11256727.3300",
         "y" : 0
      },
      {
         "an" : "OG Fan Token",
         "as" : 40770.17,
         "b" : "OG",
         "ba" : "",
         "c" : "0.0001734",
         "cs" : null,
         "etf" : false,
         "h" : "0.0001763",
         "i" : "0.01000000",
         "l" : "0.0001604",
         "o" : "0.0001671",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "6.8773726",
         "s" : "OGBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.0000001",
         "v" : "40770.1700000",
         "y" : 0
      },
      {
         "an" : "OriginToken",
         "as" : 487603,
         "b" : "OGN",
         "ba" : "",
         "c" : "0.003290",
         "cs" : 23862410,
         "etf" : false,
         "h" : "0.003589",
         "i" : "0.10000000",
         "l" : "0.003165",
         "o" : "0.003309",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1629.3744822",
         "s" : "OGNBNB",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.000001",
         "v" : "487603.000000",
         "y" : 0
      },
      {
         "an" : "OriginToken",
         "as" : 4029400,
         "b" : "OGN",
         "ba" : "",
         "c" : "0.00003434",
         "cs" : 23862410,
         "etf" : false,
         "h" : "0.00003568",
         "i" : "1.00000000",
         "l" : "0.00003119",
         "o" : "0.00003306",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "133.25015535",
         "s" : "OGNBTC",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.00000001",
         "v" : "4029400.00000000",
         "y" : 0
      },
      {
         "an" : "OriginToken",
         "as" : 19192676.91,
         "b" : "OGN",
         "ba" : "",
         "c" : "1.8890",
         "cs" : 23862410,
         "etf" : false,
         "h" : "1.9698",
         "i" : "0.01000000",
         "l" : "1.6550",
         "o" : "1.7788",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "34409846.761033",
         "s" : "OGNUSDT",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.0001",
         "v" : "19192676.9100",
         "y" : 0
      },
      {
         "an" : "OG Fan Token",
         "as" : 176160.349,
         "b" : "OG",
         "ba" : "",
         "c" : "9.551",
         "cs" : null,
         "etf" : false,
         "h" : "9.576",
         "i" : "0.00100000",
         "l" : "8.666",
         "o" : "8.985",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "1601922.389526",
         "s" : "OGUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "176160.349",
         "y" : 0
      },
      {
         "an" : "MANTRA DAO",
         "as" : 6052254,
         "b" : "OM",
         "ba" : "",
         "c" : "0.00000585",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000643",
         "i" : "1.00000000",
         "l" : "0.00000535",
         "o" : "0.00000545",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "35.93105224",
         "s" : "OMBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "6052254.00000000",
         "y" : 0
      },
      {
         "an" : "MANTRA DAO",
         "as" : 3177522.15,
         "b" : "OM",
         "ba" : "",
         "c" : "0.3222",
         "cs" : null,
         "etf" : false,
         "h" : "0.3485",
         "i" : "0.01000000",
         "l" : "0.2875",
         "o" : "0.2917",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1009834.307547",
         "s" : "OMBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "3177522.1500",
         "y" : 0
      },
      {
         "an" : "OMG Network",
         "as" : 83504.98,
         "b" : "OMG",
         "ba" : "",
         "c" : "7.4591",
         "cs" : 102042552,
         "etf" : false,
         "h" : "7.5322",
         "i" : "0.01000000",
         "l" : "6.7727",
         "o" : "6.9264",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "597988.296872",
         "s" : "OMGBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "83504.9800",
         "y" : 0
      },
      {
         "an" : "OMG Network",
         "as" : 4252534.24,
         "b" : "OMG",
         "ba" : "",
         "c" : "7.4710",
         "cs" : 102042552,
         "etf" : false,
         "h" : "7.5000",
         "i" : "0.01000000",
         "l" : "6.7597",
         "o" : "6.9486",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "30287680.397617",
         "s" : "OMGUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "4252534.2400",
         "y" : 0
      },
      {
         "an" : "MANTRA DAO",
         "as" : 24087237.87,
         "b" : "OM",
         "ba" : "",
         "c" : "0.3217",
         "cs" : null,
         "etf" : false,
         "h" : "0.3500",
         "i" : "0.01000000",
         "l" : "0.2869",
         "o" : "0.2914",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "7668124.619871",
         "s" : "OMUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "24087237.8700",
         "y" : 0
      },
      {
         "an" : "Harmony",
         "as" : 26625296,
         "b" : "ONE",
         "ba" : "",
         "c" : "0.0002423",
         "cs" : 1890000000,
         "etf" : false,
         "h" : "0.0002853",
         "i" : "1.00000000",
         "l" : "0.0002420",
         "o" : "0.0002574",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "6998.8674796",
         "s" : "ONEBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "26625296.0000000",
         "y" : 0
      },
      {
         "an" : "Harmony",
         "as" : 218910693,
         "b" : "ONE",
         "ba" : "",
         "c" : "0.00000254",
         "cs" : 1890000000,
         "etf" : false,
         "h" : "0.00000283",
         "i" : "1.00000000",
         "l" : "0.00000245",
         "o" : "0.00000258",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "573.86294145",
         "s" : "ONEBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "218910693.00000000",
         "y" : 0
      },
      {
         "an" : "Harmony",
         "as" : 113816502.2,
         "b" : "ONE",
         "ba" : "",
         "c" : "0.13933",
         "cs" : 1890000000,
         "etf" : false,
         "h" : "0.15220",
         "i" : "0.10000000",
         "l" : "0.12914",
         "o" : "0.13864",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "16092228.004916",
         "s" : "ONEBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "113816502.20000",
         "y" : 0
      },
      {
         "an" : "Harmony",
         "as" : 1189054437,
         "b" : "ONE",
         "ba" : "",
         "c" : "0.13915",
         "cs" : 1890000000,
         "etf" : false,
         "h" : "0.15200",
         "i" : "0.10000000",
         "l" : "0.12880",
         "o" : "0.13813",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "167619516.85182",
         "s" : "ONEUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "1189054437.00000",
         "y" : 0
      },
      {
         "an" : "Ontology Gas",
         "as" : 2078037,
         "b" : "ONG",
         "ba" : "",
         "c" : "0.00001695",
         "cs" : 60573114,
         "etf" : false,
         "h" : "0.00001796",
         "i" : "1.00000000",
         "l" : "0.00001568",
         "o" : "0.00001581",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "34.85175834",
         "s" : "ONGBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "2078037.00000000",
         "y" : 0
      },
      {
         "an" : "Ontology Gas",
         "as" : 5615449.72,
         "b" : "ONG",
         "ba" : "",
         "c" : "0.9285",
         "cs" : 60573114,
         "etf" : false,
         "h" : "0.9658",
         "i" : "0.01000000",
         "l" : "0.8365",
         "o" : "0.8477",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "5087134.832233",
         "s" : "ONGUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "5615449.7200",
         "y" : 0
      },
      {
         "an" : "Ontology",
         "as" : 662074.9,
         "b" : "ONT",
         "ba" : "",
         "c" : "0.002992",
         "cs" : 811891324,
         "etf" : false,
         "h" : "0.003222",
         "i" : "0.10000000",
         "l" : "0.002813",
         "o" : "0.002888",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1989.8121884",
         "s" : "ONTBNB",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.000001",
         "v" : "662074.900000",
         "y" : 0
      },
      {
         "an" : "Ontology",
         "as" : 15240202.53,
         "b" : "ONT",
         "ba" : "",
         "c" : "0.00003109",
         "cs" : 811891324,
         "etf" : false,
         "h" : "0.00003186",
         "i" : "0.01000000",
         "l" : "0.00002803",
         "o" : "0.00002893",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "458.72124535",
         "s" : "ONTBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.00000001",
         "v" : "15240202.53000000",
         "y" : 0
      },
      {
         "an" : "Ontology",
         "as" : 3993507.17,
         "b" : "ONT",
         "ba" : "",
         "c" : "1.7101",
         "cs" : 811891324,
         "etf" : false,
         "h" : "1.7449",
         "i" : "0.01000000",
         "l" : "1.4842",
         "o" : "1.5565",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "6481089.57095",
         "s" : "ONTBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.0001",
         "v" : "3993507.1700",
         "y" : 0
      },
      {
         "an" : "Ontology",
         "as" : 709621.39,
         "b" : "ONT",
         "ba" : "",
         "c" : "0.000663",
         "cs" : 811891324,
         "etf" : false,
         "h" : "0.000683",
         "i" : "0.01000000",
         "l" : "0.000604",
         "o" : "0.000621",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "457.05952422",
         "s" : "ONTETH",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.000001",
         "v" : "709621.390000",
         "y" : 0
      },
      {
         "an" : "Ontology",
         "as" : 3152712.66,
         "b" : "ONT",
         "ba" : "",
         "c" : "13.94",
         "cs" : 811891324,
         "etf" : false,
         "h" : "14.22",
         "i" : "0.01000000",
         "l" : "12.10",
         "o" : "12.73",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "41764042.5913",
         "s" : "ONTTRY",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.01",
         "v" : "3152712.66",
         "y" : 0
      },
      {
         "an" : "Ontology",
         "as" : 87895354.72,
         "b" : "ONT",
         "ba" : "",
         "c" : "1.7077",
         "cs" : 811891324,
         "etf" : false,
         "h" : "1.7433",
         "i" : "0.01000000",
         "l" : "1.4827",
         "o" : "1.5564",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "141756883.30327",
         "s" : "ONTUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.0001",
         "v" : "87895354.7200",
         "y" : 0
      },
      {
         "an" : "Orion Protocol",
         "as" : 165006.94,
         "b" : "ORN",
         "ba" : "",
         "c" : "0.0002481",
         "cs" : null,
         "etf" : false,
         "h" : "0.0002535",
         "i" : "0.01000000",
         "l" : "0.0002257",
         "o" : "0.0002410",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "39.29435074",
         "s" : "ORNBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.0000001",
         "v" : "165006.9400000",
         "y" : 0
      },
      {
         "an" : "Orion Protocol",
         "as" : 838126.361,
         "b" : "ORN",
         "ba" : "",
         "c" : "13.640",
         "cs" : null,
         "etf" : false,
         "h" : "13.821",
         "i" : "0.00100000",
         "l" : "11.883",
         "o" : "12.989",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "10734730.020816",
         "s" : "ORNUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "838126.361",
         "y" : 0
      },
      {
         "an" : "OST",
         "as" : 31043286,
         "b" : "OST",
         "ba" : "",
         "c" : "0.00000062",
         "cs" : 763665268,
         "etf" : false,
         "h" : "0.00000064",
         "i" : "1.00000000",
         "l" : "0.00000057",
         "o" : "0.00000058",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "18.64547607",
         "s" : "OSTBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "31043286.00000000",
         "y" : 0
      },
      {
         "an" : "OST",
         "as" : 3919942,
         "b" : "OST",
         "ba" : "",
         "c" : "0.00001322",
         "cs" : 763665268,
         "etf" : false,
         "h" : "0.00001366",
         "i" : "1.00000000",
         "l" : "0.00001225",
         "o" : "0.00001257",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "50.40583806",
         "s" : "OSTETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "3919942.00000000",
         "y" : 0
      },
      {
         "an" : "Orchid",
         "as" : 2273547,
         "b" : "OXT",
         "ba" : "",
         "c" : "0.00001150",
         "cs" : null,
         "etf" : false,
         "h" : "0.00001169",
         "i" : "1.00000000",
         "l" : "0.00001083",
         "o" : "0.00001106",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "25.57367215",
         "s" : "OXTBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "2273547.00000000",
         "y" : 0
      },
      {
         "an" : "Orchid",
         "as" : 8850871.29,
         "b" : "OXT",
         "ba" : "",
         "c" : "0.6322",
         "cs" : null,
         "etf" : false,
         "h" : "0.6400",
         "i" : "0.01000000",
         "l" : "0.5768",
         "o" : "0.5942",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "5369227.042685",
         "s" : "OXTUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "8850871.2900",
         "y" : 0
      },
      {
         "an" : "Paxos Standard",
         "as" : 829275.65,
         "b" : "PAX",
         "ba" : "",
         "c" : "0.9999",
         "cs" : 237000555,
         "etf" : false,
         "h" : "1.0002",
         "i" : "0.01000000",
         "l" : "0.9999",
         "o" : "1.0000",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "829265.301969",
         "s" : "PAXBUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "829275.6500",
         "y" : 0
      },
      {
         "an" : "PAX Gold",
         "as" : 97.814,
         "b" : "PAXG",
         "ba" : "",
         "c" : "3.143",
         "cs" : null,
         "etf" : false,
         "h" : "3.496",
         "i" : "0.00100000",
         "l" : "3.122",
         "o" : "3.310",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "325.426172",
         "s" : "PAXGBNB",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "97.814",
         "y" : 0
      },
      {
         "an" : "PAX Gold",
         "as" : 1159.7967,
         "b" : "PAXG",
         "ba" : "",
         "c" : "0.03265",
         "cs" : null,
         "etf" : false,
         "h" : "0.03418",
         "i" : "0.00010000",
         "l" : "0.03254",
         "o" : "0.03317",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "38.60534267",
         "s" : "PAXGBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "1159.79670",
         "y" : 0
      },
      {
         "an" : "PAX Gold",
         "as" : 3951.286364,
         "b" : "PAXG",
         "ba" : "",
         "c" : "1794.65",
         "cs" : null,
         "etf" : false,
         "h" : "1801.00",
         "i" : "0.00000100",
         "l" : "1781.00",
         "o" : "1783.14",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "7076692.72876183",
         "s" : "PAXGUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.01",
         "v" : "3951.29",
         "y" : 0
      },
      {
         "an" : "Paxos Standard",
         "as" : 6681519.34,
         "b" : "PAX",
         "ba" : "",
         "c" : "0.9988",
         "cs" : 237000555,
         "etf" : false,
         "h" : "0.9995",
         "i" : "0.01000000",
         "l" : "0.9983",
         "o" : "0.9986",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "6674106.273671",
         "s" : "PAXUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "6681519.3400",
         "y" : 0
      },
      {
         "an" : "Perlin",
         "as" : 2700951,
         "b" : "PERL",
         "ba" : "",
         "c" : "0.0002400",
         "cs" : 261737112,
         "etf" : false,
         "h" : "0.0002574",
         "i" : "1.00000000",
         "l" : "0.0002313",
         "o" : "0.0002360",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "662.9564216",
         "s" : "PERLBNB",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.0000001",
         "v" : "2700951.0000000",
         "y" : 0
      },
      {
         "an" : "Perlin",
         "as" : 14717064,
         "b" : "PERL",
         "ba" : "",
         "c" : "0.00000253",
         "cs" : 261737112,
         "etf" : false,
         "h" : "0.00000259",
         "i" : "1.00000000",
         "l" : "0.00000231",
         "o" : "0.00000236",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "35.89989128",
         "s" : "PERLBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "14717064.00000000",
         "y" : 0
      },
      {
         "an" : "Perlin",
         "as" : 38195451.5,
         "b" : "PERL",
         "ba" : "",
         "c" : "0.13871",
         "cs" : 261737112,
         "etf" : false,
         "h" : "0.14284",
         "i" : "0.10000000",
         "l" : "0.12176",
         "o" : "0.12702",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "5047496.356623",
         "s" : "PERLUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00001",
         "v" : "38195451.50000",
         "y" : 0
      },
      {
         "an" : "Perpetual Protocol",
         "as" : 111060.32,
         "b" : "PERP",
         "ba" : "",
         "c" : "0.0001201",
         "cs" : null,
         "etf" : false,
         "h" : "0.0001241",
         "i" : "0.01000000",
         "l" : "0.0001172",
         "o" : "0.0001212",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "13.30654115",
         "s" : "PERPBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0000001",
         "v" : "111060.3200000",
         "y" : 0
      },
      {
         "an" : "Perpetual Protocol",
         "as" : 97989.556,
         "b" : "PERP",
         "ba" : "",
         "c" : "6.591",
         "cs" : null,
         "etf" : false,
         "h" : "6.659",
         "i" : "0.00100000",
         "l" : "6.359",
         "o" : "6.526",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "634688.906861",
         "s" : "PERPBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.001",
         "v" : "97989.556",
         "y" : 0
      },
      {
         "an" : "Perpetual Protocol",
         "as" : 409997.261,
         "b" : "PERP",
         "ba" : "",
         "c" : "6.584",
         "cs" : null,
         "etf" : false,
         "h" : "6.653",
         "i" : "0.00100000",
         "l" : "6.349",
         "o" : "6.517",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "2654352.32187",
         "s" : "PERPUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.001",
         "v" : "409997.261",
         "y" : 0
      },
      {
         "an" : "Phala.Network",
         "as" : 3478534,
         "b" : "PHA",
         "ba" : "",
         "c" : "0.00001668",
         "cs" : null,
         "etf" : false,
         "h" : "0.00001752",
         "i" : "1.00000000",
         "l" : "0.00001356",
         "o" : "0.00001372",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "55.61741156",
         "s" : "PHABTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "Polkadot"
         ],
         "ts" : "0.00000001",
         "v" : "3478534.00000000",
         "y" : 0
      },
      {
         "an" : "Phala.Network",
         "as" : 24026806.07,
         "b" : "PHA",
         "ba" : "",
         "c" : "0.9188",
         "cs" : null,
         "etf" : false,
         "h" : "0.9431",
         "i" : "0.01000000",
         "l" : "0.7283",
         "o" : "0.7385",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "20643056.593623",
         "s" : "PHABUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "Polkadot"
         ],
         "ts" : "0.0001",
         "v" : "24026806.0700",
         "y" : 0
      },
      {
         "an" : "Phoenix Global",
         "as" : 121968702,
         "b" : "PHB",
         "ba" : "",
         "c" : "0.00000044",
         "cs" : 829588687,
         "etf" : false,
         "h" : "0.00000046",
         "i" : "1.00000000",
         "l" : "0.00000040",
         "o" : "0.00000041",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "52.13492552",
         "s" : "PHBBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "121968702.00000000",
         "y" : 0
      },
      {
         "an" : "Phoenix Global",
         "as" : 30227173,
         "b" : "PHB",
         "ba" : "",
         "c" : "0.02400",
         "cs" : 829588687,
         "etf" : false,
         "h" : "0.02499",
         "i" : "0.10000000",
         "l" : "0.02114",
         "o" : "0.02197",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TUSD",
         "qa" : "",
         "qn" : "TrueUSD",
         "qv" : "690085.070072",
         "s" : "PHBTUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "30227173.00000",
         "y" : 0
      },
      {
         "an" : "PIVX",
         "as" : 426674,
         "b" : "PIVX",
         "ba" : "",
         "c" : "0.00003005",
         "cs" : 65180591,
         "etf" : false,
         "h" : "0.00003073",
         "i" : "1.00000000",
         "l" : "0.00002933",
         "o" : "0.00003058",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "12.78279058",
         "s" : "PIVXBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "426674.00000000",
         "y" : 0
      },
      {
         "an" : "PIVX",
         "as" : 97777.66,
         "b" : "PIVX",
         "ba" : "",
         "c" : "0.0006464",
         "cs" : 65180591,
         "etf" : false,
         "h" : "0.0006568",
         "i" : "0.01000000",
         "l" : "0.0006262",
         "o" : "0.0006520",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "62.61444154",
         "s" : "PIVXETH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "97777.6600000",
         "y" : 0
      },
      {
         "an" : "pNetwork",
         "as" : 1493318,
         "b" : "PNT",
         "ba" : "",
         "c" : "0.00003289",
         "cs" : 31039489,
         "etf" : false,
         "h" : "0.00003442",
         "i" : "1.00000000",
         "l" : "0.00003070",
         "o" : "0.00003216",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "48.24019338",
         "s" : "PNTBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "1493318.00000000",
         "y" : 0
      },
      {
         "an" : "pNetwork",
         "as" : 9021077.13,
         "b" : "PNT",
         "ba" : "",
         "c" : "1.8075",
         "cs" : 31039489,
         "etf" : false,
         "h" : "1.8647",
         "i" : "0.01000000",
         "l" : "1.6244",
         "o" : "1.7279",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "15723097.551652",
         "s" : "PNTUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "9021077.1300",
         "y" : 0
      },
      {
         "an" : "POA Network",
         "as" : 6718563,
         "b" : "POA",
         "ba" : "",
         "c" : "0.00000148",
         "cs" : 220167621,
         "etf" : false,
         "h" : "0.00000149",
         "i" : "1.00000000",
         "l" : "0.00000136",
         "o" : "0.00000139",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "9.53505563",
         "s" : "POABTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "6718563.00000000",
         "y" : 0
      },
      {
         "an" : "Polymath",
         "as" : 8837760,
         "b" : "POLY",
         "ba" : "",
         "c" : "0.00000720",
         "cs" : 610331537,
         "etf" : false,
         "h" : "0.00000765",
         "i" : "1.00000000",
         "l" : "0.00000654",
         "o" : "0.00000672",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "63.12002017",
         "s" : "POLYBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "8837760.00000000",
         "y" : 0
      },
      {
         "an" : "Marlin",
         "as" : 15690897,
         "b" : "POND",
         "ba" : "",
         "c" : "0.00000276",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000290",
         "i" : "1.00000000",
         "l" : "0.00000255",
         "o" : "0.00000270",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "43.35758287",
         "s" : "PONDBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone"
         ],
         "ts" : "0.00000001",
         "v" : "15690897.00000000",
         "y" : 0
      },
      {
         "an" : "Marlin",
         "as" : 3045207.65,
         "b" : "POND",
         "ba" : "",
         "c" : "0.1516",
         "cs" : null,
         "etf" : false,
         "h" : "0.1596",
         "i" : "0.01000000",
         "l" : "0.1371",
         "o" : "0.1460",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "452035.926983",
         "s" : "PONDBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone"
         ],
         "ts" : "0.0001",
         "v" : "3045207.6500",
         "y" : 0
      },
      {
         "an" : "Marlin",
         "as" : 57196942.85,
         "b" : "POND",
         "ba" : "",
         "c" : "0.1516",
         "cs" : null,
         "etf" : false,
         "h" : "0.1578",
         "i" : "0.01000000",
         "l" : "0.1364",
         "o" : "0.1451",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "8485858.629644",
         "s" : "PONDUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone"
         ],
         "ts" : "0.0001",
         "v" : "57196942.8500",
         "y" : 0
      },
      {
         "an" : "PowerLedger",
         "as" : 5471183,
         "b" : "POWR",
         "ba" : "",
         "c" : "0.00000733",
         "cs" : 439897654,
         "etf" : false,
         "h" : "0.00000748",
         "i" : "1.00000000",
         "l" : "0.00000680",
         "o" : "0.00000703",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "39.15031104",
         "s" : "POWRBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "5471183.00000000",
         "y" : 0
      },
      {
         "an" : "PowerLedger",
         "as" : 774160,
         "b" : "POWR",
         "ba" : "",
         "c" : "0.00015744",
         "cs" : 439897654,
         "etf" : false,
         "h" : "0.00015943",
         "i" : "1.00000000",
         "l" : "0.00014625",
         "o" : "0.00015137",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "119.00625828",
         "s" : "POWRETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "774160.00000000",
         "y" : 0
      },
      {
         "an" : "Populous",
         "as" : 266384,
         "b" : "PPT",
         "ba" : "",
         "c" : "0.00008434",
         "cs" : 53252245,
         "etf" : false,
         "h" : "0.00008550",
         "i" : "1.00000000",
         "l" : "0.00007769",
         "o" : "0.00008378",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "21.73177743",
         "s" : "PPTBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "266384.00000000",
         "y" : 0
      },
      {
         "an" : "Prometeus",
         "as" : 8090.3,
         "b" : "PROM",
         "ba" : "",
         "c" : "0.01766",
         "cs" : null,
         "etf" : false,
         "h" : "0.01852",
         "i" : "0.10000000",
         "l" : "0.01664",
         "o" : "0.01717",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "141.985639",
         "s" : "PROMBNB",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "storage-zone"
         ],
         "ts" : "0.00001",
         "v" : "8090.30000",
         "y" : 0
      },
      {
         "an" : "Prometeus",
         "as" : 17476.89,
         "b" : "PROM",
         "ba" : "",
         "c" : "10.107",
         "cs" : null,
         "etf" : false,
         "h" : "10.257",
         "i" : "0.00100000",
         "l" : "8.850",
         "o" : "9.264",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "164891.433091",
         "s" : "PROMBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "storage-zone"
         ],
         "ts" : "0.001",
         "v" : "17476.890",
         "y" : 0
      },
      {
         "an" : "Prosper",
         "as" : 299762.84,
         "b" : "PROS",
         "ba" : "",
         "c" : "0.000954",
         "cs" : null,
         "etf" : false,
         "h" : "0.001017",
         "i" : "0.01000000",
         "l" : "0.000822",
         "o" : "0.000838",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "273.06359541",
         "s" : "PROSETH",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.000001",
         "v" : "299762.840000",
         "y" : 0
      },
      {
         "an" : "Paris Saint-Germain Fan Token",
         "as" : 208126.84,
         "b" : "PSG",
         "ba" : "",
         "c" : "0.0009809",
         "cs" : null,
         "etf" : false,
         "h" : "0.0011050",
         "i" : "0.01000000",
         "l" : "0.0009176",
         "o" : "0.0009445",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "210.46512117",
         "s" : "PSGBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.0000001",
         "v" : "208126.8400000",
         "y" : 0
      },
      {
         "an" : "Paris Saint-Germain Fan Token",
         "as" : 71128.309,
         "b" : "PSG",
         "ba" : "",
         "c" : "54.270",
         "cs" : null,
         "etf" : false,
         "h" : "58.999",
         "i" : "0.00100000",
         "l" : "49.980",
         "o" : "50.745",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "3896392.893218",
         "s" : "PSGBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "71128.309",
         "y" : 0
      },
      {
         "an" : "Paris Saint-Germain Fan Token",
         "as" : 1575082.839,
         "b" : "PSG",
         "ba" : "",
         "c" : "53.908",
         "cs" : null,
         "etf" : false,
         "h" : "59.000",
         "i" : "0.00100000",
         "l" : "49.923",
         "o" : "50.749",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "86077524.456789",
         "s" : "PSGUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "1575082.839",
         "y" : 0
      },
      {
         "an" : "Pundi X",
         "as" : 1460808.37,
         "b" : "PUNDIX",
         "ba" : "",
         "c" : "0.001374",
         "cs" : null,
         "etf" : false,
         "h" : "0.001429",
         "i" : "0.01000000",
         "l" : "0.001221",
         "o" : "0.001266",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "1957.38038006",
         "s" : "PUNDIXETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "1460808.370000",
         "y" : 0
      },
      {
         "an" : "Pundi X",
         "as" : 11838322.236,
         "b" : "PUNDIX",
         "ba" : "",
         "c" : "3.520",
         "cs" : null,
         "etf" : false,
         "h" : "3.600",
         "i" : "0.00100000",
         "l" : "3.030",
         "o" : "3.171",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "39594025.495079",
         "s" : "PUNDIXUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "11838322.236",
         "y" : 0
      },
      {
         "an" : "QuarkChain",
         "as" : 48411692,
         "b" : "QKC",
         "ba" : "",
         "c" : "0.00000056",
         "cs" : 6399906497,
         "etf" : false,
         "h" : "0.00000056",
         "i" : "1.00000000",
         "l" : "0.00000051",
         "o" : "0.00000052",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "25.93040709",
         "s" : "QKCBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "48411692.00000000",
         "y" : 0
      },
      {
         "an" : "QuarkChain",
         "as" : 11077056,
         "b" : "QKC",
         "ba" : "",
         "c" : "0.00001185",
         "cs" : 6399906497,
         "etf" : false,
         "h" : "0.00001204",
         "i" : "1.00000000",
         "l" : "0.00001089",
         "o" : "0.00001110",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "127.59086326",
         "s" : "QKCETH",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "11077056.00000000",
         "y" : 0
      },
      {
         "an" : "QLC Chain",
         "as" : 10691483,
         "b" : "QLC",
         "ba" : "",
         "c" : "0.00000142",
         "cs" : 240000000,
         "etf" : false,
         "h" : "0.00000144",
         "i" : "1.00000000",
         "l" : "0.00000132",
         "o" : "0.00000137",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "14.74791346",
         "s" : "QLCBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "10691483.00000000",
         "y" : 0
      },
      {
         "an" : "QLC Chain",
         "as" : 1329543,
         "b" : "QLC",
         "ba" : "",
         "c" : "0.00003058",
         "cs" : 240000000,
         "etf" : false,
         "h" : "0.00003096",
         "i" : "1.00000000",
         "l" : "0.00002861",
         "o" : "0.00002949",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "39.41819113",
         "s" : "QLCETH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "1329543.00000000",
         "y" : 0
      },
      {
         "an" : "Quantstamp",
         "as" : 14032549,
         "b" : "QSP",
         "ba" : "",
         "c" : "0.00000144",
         "cs" : 713801946,
         "etf" : false,
         "h" : "0.00000149",
         "i" : "1.00000000",
         "l" : "0.00000139",
         "o" : "0.00000147",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "20.27380209",
         "s" : "QSPBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "14032549.00000000",
         "y" : 0
      },
      {
         "an" : "Quantstamp",
         "as" : 2752424,
         "b" : "QSP",
         "ba" : "",
         "c" : "0.00003092",
         "cs" : 713801946,
         "etf" : false,
         "h" : "0.00003217",
         "i" : "1.00000000",
         "l" : "0.00002997",
         "o" : "0.00003146",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "85.49102376",
         "s" : "QSPETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "2752424.00000000",
         "y" : 0
      },
      {
         "an" : "Qtum",
         "as" : 214966.09,
         "b" : "QTUM",
         "ba" : "",
         "c" : "14.882",
         "cs" : 98317668,
         "etf" : false,
         "h" : "15.264",
         "i" : "0.00100000",
         "l" : "12.942",
         "o" : "13.400",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "3078280.189882",
         "s" : "QTUMBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "214966.090",
         "y" : 0
      },
      {
         "an" : "Qtum",
         "as" : 5336988.935,
         "b" : "QTUM",
         "ba" : "",
         "c" : "14.857",
         "cs" : 98317668,
         "etf" : false,
         "h" : "15.255",
         "i" : "0.00100000",
         "l" : "12.905",
         "o" : "13.391",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "75787468.999935",
         "s" : "QTUMUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "5336988.935",
         "y" : 0
      },
      {
         "an" : "RAMP",
         "as" : 3780333,
         "b" : "RAMP",
         "ba" : "",
         "c" : "0.00000968",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000997",
         "i" : "1.00000000",
         "l" : "0.00000909",
         "o" : "0.00000956",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "35.85214996",
         "s" : "RAMPBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "3780333.00000000",
         "y" : 0
      },
      {
         "an" : "RAMP",
         "as" : 3944191.4,
         "b" : "RAMP",
         "ba" : "",
         "c" : "0.5320",
         "cs" : null,
         "etf" : false,
         "h" : "0.5424",
         "i" : "0.01000000",
         "l" : "0.4794",
         "o" : "0.5145",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2034364.236406",
         "s" : "RAMPBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "3944191.4000",
         "y" : 0
      },
      {
         "an" : "RAMP",
         "as" : 42437751.21,
         "b" : "RAMP",
         "ba" : "",
         "c" : "0.5317",
         "cs" : null,
         "etf" : false,
         "h" : "0.5423",
         "i" : "0.01000000",
         "l" : "0.4770",
         "o" : "0.5148",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "21565401.993702",
         "s" : "RAMPUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "42437751.2100",
         "y" : 0
      },
      {
         "an" : "Ripio Credit Network",
         "as" : 5499067,
         "b" : "RCN",
         "ba" : "",
         "c" : "0.00000218",
         "cs" : 0,
         "etf" : false,
         "h" : "0.00000221",
         "i" : "1.00000000",
         "l" : "0.00000206",
         "o" : "0.00000208",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "11.73152914",
         "s" : "RCNBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "5499067.00000000",
         "y" : 0
      },
      {
         "an" : "Raiden Network Token",
         "as" : 1423103,
         "b" : "RDN",
         "ba" : "",
         "c" : "0.00001865",
         "cs" : 50593936,
         "etf" : false,
         "h" : "0.00001895",
         "i" : "1.00000000",
         "l" : "0.00001774",
         "o" : "0.00001854",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "26.07088041",
         "s" : "RDNBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "1423103.00000000",
         "y" : 0
      },
      {
         "an" : "Reef Finance",
         "as" : 268617356,
         "b" : "REEF",
         "ba" : "",
         "c" : "0.00000066",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000068",
         "i" : "1.00000000",
         "l" : "0.00000063",
         "o" : "0.00000065",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "176.62012716",
         "s" : "REEFBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "Polkadot"
         ],
         "ts" : "0.00000001",
         "v" : "268617356.00000000",
         "y" : 0
      },
      {
         "an" : "Reef Finance",
         "as" : 131952879.6,
         "b" : "REEF",
         "ba" : "",
         "c" : "0.03655",
         "cs" : null,
         "etf" : false,
         "h" : "0.03721",
         "i" : "0.10000000",
         "l" : "0.03359",
         "o" : "0.03481",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "4660685.556907",
         "s" : "REEFBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "Polkadot"
         ],
         "ts" : "0.00001",
         "v" : "131952879.60000",
         "y" : 0
      },
      {
         "an" : "Reef Finance",
         "as" : 1229907696,
         "b" : "REEF",
         "ba" : "",
         "c" : "0.036494",
         "cs" : null,
         "etf" : false,
         "h" : "0.037100",
         "i" : "1.00000000",
         "l" : "0.033542",
         "o" : "0.034886",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "43518485.380639",
         "s" : "REEFUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "Polkadot"
         ],
         "ts" : "0.000001",
         "v" : "1229907696.000000",
         "y" : 0
      },
      {
         "an" : "Ren",
         "as" : 4563399,
         "b" : "REN",
         "ba" : "",
         "c" : "0.00001591",
         "cs" : 997163051,
         "etf" : false,
         "h" : "0.00001636",
         "i" : "1.00000000",
         "l" : "0.00001538",
         "o" : "0.00001554",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "72.34318125",
         "s" : "RENBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "4563399.00000000",
         "y" : 0
      },
      {
         "an" : "renBTC",
         "as" : 5.8466,
         "b" : "RENBTC",
         "ba" : "",
         "c" : "0.99650",
         "cs" : null,
         "etf" : false,
         "h" : "0.99999",
         "i" : "0.00001000",
         "l" : "0.99555",
         "o" : "0.99776",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "5.83116966",
         "s" : "RENBTCBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00001",
         "v" : "5.84660",
         "y" : 0
      },
      {
         "an" : "renBTC",
         "as" : 0.71372,
         "b" : "RENBTC",
         "ba" : "",
         "c" : "21.275",
         "cs" : null,
         "etf" : false,
         "h" : "21.681",
         "i" : "0.00001000",
         "l" : "21.096",
         "o" : "21.482",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "15.28057478",
         "s" : "RENBTCETH",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "0.714",
         "y" : 0
      },
      {
         "an" : "Augur v2",
         "as" : 76555.012,
         "b" : "REP",
         "ba" : "",
         "c" : "0.0006829",
         "cs" : 11000000,
         "etf" : false,
         "h" : "0.0006950",
         "i" : "0.00100000",
         "l" : "0.0006404",
         "o" : "0.0006586",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "51.32922114",
         "s" : "REPBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.0000001",
         "v" : "76555.0120000",
         "y" : 0
      },
      {
         "an" : "Augur v2",
         "as" : 6089.427,
         "b" : "REP",
         "ba" : "",
         "c" : "0.01467",
         "cs" : 11000000,
         "etf" : false,
         "h" : "0.01492",
         "i" : "0.00100000",
         "l" : "0.01379",
         "o" : "0.01415",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "87.63697792",
         "s" : "REPETH",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00001",
         "v" : "6089.42700",
         "y" : 0
      },
      {
         "an" : "Augur v2",
         "as" : 105185.636,
         "b" : "REP",
         "ba" : "",
         "c" : "37.562",
         "cs" : 11000000,
         "etf" : false,
         "h" : "38.368",
         "i" : "0.00100000",
         "l" : "33.938",
         "o" : "35.452",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "3797910.934826",
         "s" : "REPUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "105185.636",
         "y" : 0
      },
      {
         "an" : "Request Network",
         "as" : 5701721,
         "b" : "REQ",
         "ba" : "",
         "c" : "0.00000246",
         "cs" : 729657214,
         "etf" : false,
         "h" : "0.00000246",
         "i" : "1.00000000",
         "l" : "0.00000230",
         "o" : "0.00000234",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "13.54597727",
         "s" : "REQBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "5701721.00000000",
         "y" : 0
      },
      {
         "an" : "RSK Infrastructure Framework",
         "as" : 3619420,
         "b" : "RIF",
         "ba" : "",
         "c" : "0.00000567",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000580",
         "i" : "1.00000000",
         "l" : "0.00000551",
         "o" : "0.00000553",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "20.45527508",
         "s" : "RIFBTC",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "storage-zone",
            "pow"
         ],
         "ts" : "0.00000001",
         "v" : "3619420.00000000",
         "y" : 0
      },
      {
         "an" : "RSK Infrastructure Framework",
         "as" : 6877356.51,
         "b" : "RIF",
         "ba" : "",
         "c" : "0.3110",
         "cs" : null,
         "etf" : false,
         "h" : "0.3319",
         "i" : "0.01000000",
         "l" : "0.2924",
         "o" : "0.2970",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "2097672.716353",
         "s" : "RIFUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "storage-zone",
            "pow"
         ],
         "ts" : "0.0001",
         "v" : "6877356.5100",
         "y" : 0
      },
      {
         "an" : "iExecRLC",
         "as" : 5620614,
         "b" : "RLC",
         "ba" : "",
         "c" : "0.00005739",
         "cs" : 80070793,
         "etf" : false,
         "h" : "0.00006026",
         "i" : "1.00000000",
         "l" : "0.00004568",
         "o" : "0.00004589",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "297.6817508",
         "s" : "RLCBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "5620614.00000000",
         "y" : 0
      },
      {
         "an" : "iExecRLC",
         "as" : 500257.76,
         "b" : "RLC",
         "ba" : "",
         "c" : "0.001229",
         "cs" : 80070793,
         "etf" : false,
         "h" : "0.001283",
         "i" : "0.01000000",
         "l" : "0.000985",
         "o" : "0.000986",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "562.8664423",
         "s" : "RLCETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "500257.760000",
         "y" : 0
      },
      {
         "an" : "iExecRLC",
         "as" : 11837541.03,
         "b" : "RLC",
         "ba" : "",
         "c" : "3.1500",
         "cs" : 80070793,
         "etf" : false,
         "h" : "3.2855",
         "i" : "0.01000000",
         "l" : "2.4556",
         "o" : "2.4672",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "34232378.737961",
         "s" : "RLCUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "11837541.0300",
         "y" : 0
      },
      {
         "an" : "Oasis Network",
         "as" : 18373426,
         "b" : "ROSE",
         "ba" : "",
         "c" : "0.00000286",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000289",
         "i" : "1.00000000",
         "l" : "0.00000265",
         "o" : "0.00000270",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "50.70657037",
         "s" : "ROSEBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "18373426.00000000",
         "y" : 0
      },
      {
         "an" : "Oasis Network",
         "as" : 13469906.8,
         "b" : "ROSE",
         "ba" : "",
         "c" : "0.15785",
         "cs" : null,
         "etf" : false,
         "h" : "0.15905",
         "i" : "0.10000000",
         "l" : "0.14054",
         "o" : "0.14623",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2001419.380057",
         "s" : "ROSEBUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "13469906.80000",
         "y" : 0
      },
      {
         "an" : "Oasis Network",
         "as" : 75669018.8,
         "b" : "ROSE",
         "ba" : "",
         "c" : "0.15749",
         "cs" : null,
         "etf" : false,
         "h" : "0.15898",
         "i" : "0.10000000",
         "l" : "0.14052",
         "o" : "0.14569",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "11230650.766642",
         "s" : "ROSEUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "75669018.80000",
         "y" : 0
      },
      {
         "an" : "Reserve Rights",
         "as" : 13809924,
         "b" : "RSR",
         "ba" : "",
         "c" : "0.0001504",
         "cs" : null,
         "etf" : false,
         "h" : "0.0001631",
         "i" : "1.00000000",
         "l" : "0.0001487",
         "o" : "0.0001517",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "2142.6332607",
         "s" : "RSRBNB",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.0000001",
         "v" : "13809924.0000000",
         "y" : 0
      },
      {
         "an" : "Reserve Rights",
         "as" : 133246778,
         "b" : "RSR",
         "ba" : "",
         "c" : "0.00000156",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000163",
         "i" : "1.00000000",
         "l" : "0.00000148",
         "o" : "0.00000152",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "206.42879271",
         "s" : "RSRBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "133246778.00000000",
         "y" : 0
      },
      {
         "an" : "Reserve Rights",
         "as" : 655974012.6,
         "b" : "RSR",
         "ba" : "",
         "c" : "0.08601",
         "cs" : null,
         "etf" : false,
         "h" : "0.08900",
         "i" : "0.10000000",
         "l" : "0.07800",
         "o" : "0.08176",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "54532615.466032",
         "s" : "RSRUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00001",
         "v" : "655974012.60000",
         "y" : 0
      },
      {
         "an" : "THORChain",
         "as" : 475092.1,
         "b" : "RUNE",
         "ba" : "",
         "c" : "0.02509",
         "cs" : null,
         "etf" : false,
         "h" : "0.02714",
         "i" : "0.10000000",
         "l" : "0.02479",
         "o" : "0.02493",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "12291.523958",
         "s" : "RUNEBNB",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00001",
         "v" : "475092.10000",
         "y" : 0
      },
      {
         "an" : "THORChain",
         "as" : 959014.4,
         "b" : "RUNE",
         "ba" : "",
         "c" : "0.00026109",
         "cs" : null,
         "etf" : false,
         "h" : "0.00027445",
         "i" : "0.10000000",
         "l" : "0.00024512",
         "o" : "0.00024994",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "248.19698789",
         "s" : "RUNEBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "959014.40000000",
         "y" : 0
      },
      {
         "an" : "THORChain",
         "as" : 467843.175,
         "b" : "RUNE",
         "ba" : "",
         "c" : "14.369",
         "cs" : null,
         "etf" : false,
         "h" : "15.096",
         "i" : "0.00100000",
         "l" : "12.976",
         "o" : "13.459",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "6558049.627951",
         "s" : "RUNEBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "467843.175",
         "y" : 0
      },
      {
         "an" : "THORChain",
         "as" : 4249609.103,
         "b" : "RUNE",
         "ba" : "",
         "c" : "14.349",
         "cs" : null,
         "etf" : false,
         "h" : "15.142",
         "i" : "0.00100000",
         "l" : "12.952",
         "o" : "13.452",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "59189629.236484",
         "s" : "RUNEUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "4249609.103",
         "y" : 0
      },
      {
         "an" : "Ravencoin",
         "as" : 4034828,
         "b" : "RVN",
         "ba" : "",
         "c" : "0.0002984",
         "cs" : 8643635000,
         "etf" : false,
         "h" : "0.0003100",
         "i" : "1.00000000",
         "l" : "0.0002796",
         "o" : "0.0002924",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1196.9481553",
         "s" : "RVNBNB",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "4034828.0000000",
         "y" : 0
      },
      {
         "an" : "Ravencoin",
         "as" : 81928199,
         "b" : "RVN",
         "ba" : "",
         "c" : "0.00000311",
         "cs" : 8643635000,
         "etf" : false,
         "h" : "0.00000321",
         "i" : "1.00000000",
         "l" : "0.00000279",
         "o" : "0.00000293",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "243.44101211",
         "s" : "RVNBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "81928199.00000000",
         "y" : 0
      },
      {
         "an" : "Ravencoin",
         "as" : 10654860.2,
         "b" : "RVN",
         "ba" : "",
         "c" : "0.17150",
         "cs" : 8643635000,
         "etf" : false,
         "h" : "0.17651",
         "i" : "0.10000000",
         "l" : "0.14733",
         "o" : "0.15781",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1712712.580462",
         "s" : "RVNBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "10654860.20000",
         "y" : 0
      },
      {
         "an" : "Ravencoin",
         "as" : 36183766.17,
         "b" : "RVN",
         "ba" : "",
         "c" : "1.395",
         "cs" : 8643635000,
         "etf" : false,
         "h" : "1.438",
         "i" : "0.01000000",
         "l" : "1.198",
         "o" : "1.287",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "47250895.62152",
         "s" : "RVNTRY",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "36183766.170",
         "y" : 0
      },
      {
         "an" : "Ravencoin",
         "as" : 304826638.2,
         "b" : "RVN",
         "ba" : "",
         "c" : "0.17113",
         "cs" : 8643635000,
         "etf" : false,
         "h" : "0.17719",
         "i" : "0.10000000",
         "l" : "0.14520",
         "o" : "0.15774",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "48848929.646882",
         "s" : "RVNUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "304826638.20000",
         "y" : 0
      },
      {
         "an" : "The Sandbox",
         "as" : 491213,
         "b" : "SAND",
         "ba" : "",
         "c" : "0.0009125",
         "cs" : null,
         "etf" : false,
         "h" : "0.0009564",
         "i" : "1.00000000",
         "l" : "0.0008840",
         "o" : "0.0008950",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "452.1471302",
         "s" : "SANDBNB",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.0000001",
         "v" : "491213.0000000",
         "y" : 0
      },
      {
         "an" : "The Sandbox",
         "as" : 7428489,
         "b" : "SAND",
         "ba" : "",
         "c" : "0.00000953",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000966",
         "i" : "1.00000000",
         "l" : "0.00000878",
         "o" : "0.00000900",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "68.47036311",
         "s" : "SANDBTC",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.00000001",
         "v" : "7428489.00000000",
         "y" : 0
      },
      {
         "an" : "The Sandbox",
         "as" : 1441524.4,
         "b" : "SAND",
         "ba" : "",
         "c" : "0.52223",
         "cs" : null,
         "etf" : false,
         "h" : "0.53369",
         "i" : "0.10000000",
         "l" : "0.46415",
         "o" : "0.48608",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "715645.981488",
         "s" : "SANDBUSD",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.00001",
         "v" : "1441524.40000",
         "y" : 0
      },
      {
         "an" : "The Sandbox",
         "as" : 22337867.6,
         "b" : "SAND",
         "ba" : "",
         "c" : "0.52342",
         "cs" : null,
         "etf" : false,
         "h" : "0.53391",
         "i" : "0.10000000",
         "l" : "0.46310",
         "o" : "0.48355",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "11092510.241171",
         "s" : "SANDUSDT",
         "st" : "TRADING",
         "tags" : [
            "NFT"
         ],
         "ts" : "0.00001",
         "v" : "22337867.60000",
         "y" : 0
      },
      {
         "an" : "Siacoin",
         "as" : 39876951,
         "b" : "SC",
         "ba" : "",
         "c" : "0.0000710",
         "cs" : 47583712992,
         "etf" : false,
         "h" : "0.0000782",
         "i" : "1.00000000",
         "l" : "0.0000627",
         "o" : "0.0000650",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "2836.2147388",
         "s" : "SCBNB",
         "st" : "TRADING",
         "tags" : [
            "storage-zone",
            "pow",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "39876951.0000000",
         "y" : 0
      },
      {
         "an" : "Siacoin",
         "as" : 801296118,
         "b" : "SC",
         "ba" : "",
         "c" : "0.00000074",
         "cs" : 47583712992,
         "etf" : false,
         "h" : "0.00000077",
         "i" : "1.00000000",
         "l" : "0.00000062",
         "o" : "0.00000065",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "560.07711869",
         "s" : "SCBTC",
         "st" : "TRADING",
         "tags" : [
            "storage-zone",
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "801296118.00000000",
         "y" : 0
      },
      {
         "an" : "Siacoin",
         "as" : 107920783,
         "b" : "SC",
         "ba" : "",
         "c" : "0.00001595",
         "cs" : 47583712992,
         "etf" : false,
         "h" : "0.00001653",
         "i" : "1.00000000",
         "l" : "0.00001346",
         "o" : "0.00001390",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "1652.81910891",
         "s" : "SCETH",
         "st" : "TRADING",
         "tags" : [
            "storage-zone",
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "107920783.00000000",
         "y" : 0
      },
      {
         "an" : "Secret",
         "as" : 871894,
         "b" : "SCRT",
         "ba" : "",
         "c" : "0.00005598",
         "cs" : null,
         "etf" : false,
         "h" : "0.00006084",
         "i" : "1.00000000",
         "l" : "0.00005272",
         "o" : "0.00005403",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "49.79751213",
         "s" : "SCRTBTC",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.00000001",
         "v" : "871894.00000000",
         "y" : 0
      },
      {
         "an" : "Secret",
         "as" : 321861.47,
         "b" : "SCRT",
         "ba" : "",
         "c" : "0.001200",
         "cs" : null,
         "etf" : false,
         "h" : "0.001311",
         "i" : "0.01000000",
         "l" : "0.001131",
         "o" : "0.001162",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "396.0226742",
         "s" : "SCRTETH",
         "st" : "TRADING",
         "tags" : [
            "pos"
         ],
         "ts" : "0.000001",
         "v" : "321861.470000",
         "y" : 0
      },
      {
         "an" : "Siacoin",
         "as" : 2684629917,
         "b" : "SC",
         "ba" : "",
         "c" : "0.040743",
         "cs" : 47583712992,
         "etf" : false,
         "h" : "0.041900",
         "i" : "1.00000000",
         "l" : "0.033044",
         "o" : "0.034832",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "101640794.237505",
         "s" : "SCUSDT",
         "st" : "TRADING",
         "tags" : [
            "storage-zone",
            "pow",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "2684629917.000000",
         "y" : 0
      },
      {
         "an" : "SafePal",
         "as" : 694307,
         "b" : "SFP",
         "ba" : "",
         "c" : "0.00004522",
         "cs" : null,
         "etf" : false,
         "h" : "0.00004611",
         "i" : "1.00000000",
         "l" : "0.00004111",
         "o" : "0.00004177",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "29.66025718",
         "s" : "SFPBTC",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.00000001",
         "v" : "694307.00000000",
         "y" : 0
      },
      {
         "an" : "SafePal",
         "as" : 1141158.62,
         "b" : "SFP",
         "ba" : "",
         "c" : "2.4877",
         "cs" : null,
         "etf" : false,
         "h" : "2.5334",
         "i" : "0.01000000",
         "l" : "2.1857",
         "o" : "2.2481",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2630320.542832",
         "s" : "SFPBUSD",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.0001",
         "v" : "1141158.6200",
         "y" : 0
      },
      {
         "an" : "SafePal",
         "as" : 9752654.35,
         "b" : "SFP",
         "ba" : "",
         "c" : "2.4886",
         "cs" : null,
         "etf" : false,
         "h" : "2.5300",
         "i" : "0.01000000",
         "l" : "2.1800",
         "o" : "2.2477",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "22634392.300357",
         "s" : "SFPUSDT",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.0001",
         "v" : "9752654.3500",
         "y" : 0
      },
      {
         "an" : "SKALE Network",
         "as" : 5046730,
         "b" : "SKL",
         "ba" : "",
         "c" : "0.00001030",
         "cs" : null,
         "etf" : false,
         "h" : "0.00001097",
         "i" : "1.00000000",
         "l" : "0.00000932",
         "o" : "0.00001001",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "51.2791746",
         "s" : "SKLBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "5046730.00000000",
         "y" : 0
      },
      {
         "an" : "SKALE Network",
         "as" : 1475080.8,
         "b" : "SKL",
         "ba" : "",
         "c" : "0.56876",
         "cs" : null,
         "etf" : false,
         "h" : "0.58744",
         "i" : "0.10000000",
         "l" : "0.49155",
         "o" : "0.53811",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "802176.311323",
         "s" : "SKLBUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "1475080.80000",
         "y" : 0
      },
      {
         "an" : "SKALE Network",
         "as" : 33039290.9,
         "b" : "SKL",
         "ba" : "",
         "c" : "0.56829",
         "cs" : null,
         "etf" : false,
         "h" : "0.58743",
         "i" : "0.10000000",
         "l" : "0.49064",
         "o" : "0.53672",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "17999864.47456",
         "s" : "SKLUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "33039290.90000",
         "y" : 0
      },
      {
         "an" : "Skycoin",
         "as" : 673473,
         "b" : "SKY",
         "ba" : "",
         "c" : "0.00005880",
         "cs" : 18000000,
         "etf" : false,
         "h" : "0.00005975",
         "i" : "1.00000000",
         "l" : "0.00005590",
         "o" : "0.00005787",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "38.68398826",
         "s" : "SKYBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "673473.00000000",
         "y" : 0
      },
      {
         "an" : "Small Love Potion",
         "as" : 13253365,
         "b" : "SLP",
         "ba" : "",
         "c" : "0.00001595",
         "cs" : null,
         "etf" : false,
         "h" : "0.00001658",
         "i" : "1.00000000",
         "l" : "0.00001555",
         "o" : "0.00001580",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "212.22260525",
         "s" : "SLPETH",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "NFT"
         ],
         "ts" : "0.00000001",
         "v" : "13253365.00000000",
         "y" : 0
      },
      {
         "an" : "Status",
         "as" : 14508373,
         "b" : "SNT",
         "ba" : "",
         "c" : "0.00000301",
         "cs" : 0,
         "etf" : false,
         "h" : "0.00000303",
         "i" : "1.00000000",
         "l" : "0.00000268",
         "o" : "0.00000279",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "42.15029559",
         "s" : "SNTBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "14508373.00000000",
         "y" : 0
      },
      {
         "an" : "Synthetix Network Token",
         "as" : 9131.58,
         "b" : "SNX",
         "ba" : "",
         "c" : "0.02914",
         "cs" : null,
         "etf" : false,
         "h" : "0.03067",
         "i" : "0.01000000",
         "l" : "0.02883",
         "o" : "0.02916",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "272.8652721",
         "s" : "SNXBNB",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00001",
         "v" : "9131.58000",
         "y" : 0
      },
      {
         "an" : "Synthetix Network Token",
         "as" : 199690.7,
         "b" : "SNX",
         "ba" : "",
         "c" : "0.0003024",
         "cs" : null,
         "etf" : false,
         "h" : "0.0003066",
         "i" : "0.01000000",
         "l" : "0.0002898",
         "o" : "0.0002937",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "59.69013094",
         "s" : "SNXBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.0000001",
         "v" : "199690.7000000",
         "y" : 0
      },
      {
         "an" : "Synthetix Network Token",
         "as" : 51812.908,
         "b" : "SNX",
         "ba" : "",
         "c" : "16.625",
         "cs" : null,
         "etf" : false,
         "h" : "16.832",
         "i" : "0.00100000",
         "l" : "15.428",
         "o" : "15.801",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "836897.889005",
         "s" : "SNXBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "51812.908",
         "y" : 0
      },
      {
         "an" : "Synthetix Network Token",
         "as" : 815201.318,
         "b" : "SNX",
         "ba" : "",
         "c" : "16.608",
         "cs" : null,
         "etf" : false,
         "h" : "16.808",
         "i" : "0.00100000",
         "l" : "15.398",
         "o" : "15.797",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "13134732.618402",
         "s" : "SNXUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "815201.318",
         "y" : 0
      },
      {
         "an" : "Solana",
         "as" : 333183.2,
         "b" : "SOL",
         "ba" : "",
         "c" : "0.07734",
         "cs" : null,
         "etf" : false,
         "h" : "0.08913",
         "i" : "0.10000000",
         "l" : "0.07725",
         "o" : "0.08534",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "28009.072256",
         "s" : "SOLBNB",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "333183.20000",
         "y" : 0
      },
      {
         "an" : "Solana",
         "as" : 1590020.2,
         "b" : "SOL",
         "ba" : "",
         "c" : "0.00080597",
         "cs" : null,
         "etf" : false,
         "h" : "0.00088668",
         "i" : "0.10000000",
         "l" : "0.00080527",
         "o" : "0.00085353",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "1337.3560045",
         "s" : "SOLBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "1590020.20000000",
         "y" : 0
      },
      {
         "an" : "Solana",
         "as" : 893807.779,
         "b" : "SOL",
         "ba" : "",
         "c" : "44.371",
         "cs" : null,
         "etf" : false,
         "h" : "47.634",
         "i" : "0.00100000",
         "l" : "43.319",
         "o" : "45.919",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "40621838.644201",
         "s" : "SOLBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "893807.779",
         "y" : 0
      },
      {
         "an" : "Solana",
         "as" : 8154373.551,
         "b" : "SOL",
         "ba" : "",
         "c" : "44.272",
         "cs" : null,
         "etf" : false,
         "h" : "47.500",
         "i" : "0.00100000",
         "l" : "43.267",
         "o" : "45.902",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "369456452.222542",
         "s" : "SOLUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "8154373.551",
         "y" : 0
      },
      {
         "an" : "Spartan Protocol",
         "as" : 1518665,
         "b" : "SPARTA",
         "ba" : "",
         "c" : "0.001911",
         "cs" : null,
         "etf" : false,
         "h" : "0.002019",
         "i" : "1.00000000",
         "l" : "0.001820",
         "o" : "0.001821",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "2940.077156",
         "s" : "SPARTABNB",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "BSC"
         ],
         "ts" : "0.000001",
         "v" : "1518665.000000",
         "y" : 0
      },
      {
         "an" : "Serum",
         "as" : 687336,
         "b" : "SRM",
         "ba" : "",
         "c" : "0.017862",
         "cs" : null,
         "etf" : false,
         "h" : "0.021851",
         "i" : "1.00000000",
         "l" : "0.017059",
         "o" : "0.018485",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "13084.395724",
         "s" : "SRMBNB",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.000001",
         "v" : "687336.000000",
         "y" : 0
      },
      {
         "an" : "Serum",
         "as" : 3984892,
         "b" : "SRM",
         "ba" : "",
         "c" : "0.00018723",
         "cs" : null,
         "etf" : false,
         "h" : "0.00021000",
         "i" : "1.00000000",
         "l" : "0.00016816",
         "o" : "0.00018454",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "755.87630774",
         "s" : "SRMBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "3984892.00000000",
         "y" : 0
      },
      {
         "an" : "Serum",
         "as" : 1609421.77,
         "b" : "SRM",
         "ba" : "",
         "c" : "10.2904",
         "cs" : null,
         "etf" : false,
         "h" : "11.3747",
         "i" : "0.01000000",
         "l" : "8.8912",
         "o" : "9.9339",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "16447452.994263",
         "s" : "SRMBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "1609421.7700",
         "y" : 0
      },
      {
         "an" : "Serum",
         "as" : 28045839.98,
         "b" : "SRM",
         "ba" : "",
         "c" : "10.2835",
         "cs" : null,
         "etf" : false,
         "h" : "11.3680",
         "i" : "0.01000000",
         "l" : "8.8834",
         "o" : "9.9340",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "285382643.388421",
         "s" : "SRMUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "28045839.9800",
         "y" : 0
      },
      {
         "an" : "Steem",
         "as" : 2564159,
         "b" : "STEEM",
         "ba" : "",
         "c" : "0.00001646",
         "cs" : 374300680,
         "etf" : false,
         "h" : "0.00001762",
         "i" : "1.00000000",
         "l" : "0.00001535",
         "o" : "0.00001569",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "42.57352531",
         "s" : "STEEMBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "2564159.00000000",
         "y" : 0
      },
      {
         "an" : "Steem",
         "as" : 565389.88,
         "b" : "STEEM",
         "ba" : "",
         "c" : "0.0003539",
         "cs" : 374300680,
         "etf" : false,
         "h" : "0.0003770",
         "i" : "0.01000000",
         "l" : "0.0003298",
         "o" : "0.0003367",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "201.63107484",
         "s" : "STEEMETH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "565389.8800000",
         "y" : 0
      },
      {
         "an" : "StormX",
         "as" : 9555096,
         "b" : "STMX",
         "ba" : "",
         "c" : "0.0000920",
         "cs" : null,
         "etf" : false,
         "h" : "0.0000945",
         "i" : "1.00000000",
         "l" : "0.0000831",
         "o" : "0.0000850",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "854.7556348",
         "s" : "STMXBNB",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "9555096.0000000",
         "y" : 0
      },
      {
         "an" : "StormX",
         "as" : 86766891,
         "b" : "STMX",
         "ba" : "",
         "c" : "0.00000096",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000098",
         "i" : "1.00000000",
         "l" : "0.00000083",
         "o" : "0.00000086",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "77.58062024",
         "s" : "STMXBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "86766891.00000000",
         "y" : 0
      },
      {
         "an" : "StormX",
         "as" : 66622726,
         "b" : "STMX",
         "ba" : "",
         "c" : "0.00002056",
         "cs" : null,
         "etf" : false,
         "h" : "0.00002101",
         "i" : "1.00000000",
         "l" : "0.00001789",
         "o" : "0.00001824",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "1287.59026416",
         "s" : "STMXETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "66622726.00000000",
         "y" : 0
      },
      {
         "an" : "StormX",
         "as" : 477219622,
         "b" : "STMX",
         "ba" : "",
         "c" : "0.052661",
         "cs" : null,
         "etf" : false,
         "h" : "0.053831",
         "i" : "1.00000000",
         "l" : "0.043944",
         "o" : "0.045728",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "23029947.437596",
         "s" : "STMXUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "477219622.000000",
         "y" : 0
      },
      {
         "an" : "Storj",
         "as" : 2287421,
         "b" : "STORJ",
         "ba" : "",
         "c" : "0.00003716",
         "cs" : 287149025,
         "etf" : false,
         "h" : "0.00003871",
         "i" : "1.00000000",
         "l" : "0.00003359",
         "o" : "0.00003408",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "82.48675606",
         "s" : "STORJBTC",
         "st" : "TRADING",
         "tags" : [
            "storage-zone"
         ],
         "ts" : "0.00000001",
         "v" : "2287421.00000000",
         "y" : 0
      },
      {
         "an" : "Storj",
         "as" : 11726597.58,
         "b" : "STORJ",
         "ba" : "",
         "c" : "2.0435",
         "cs" : 287149025,
         "etf" : false,
         "h" : "2.1207",
         "i" : "0.01000000",
         "l" : "1.7769",
         "o" : "1.8364",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "22815775.685967",
         "s" : "STORJUSDT",
         "st" : "TRADING",
         "tags" : [
            "storage-zone"
         ],
         "ts" : "0.0001",
         "v" : "11726597.5800",
         "y" : 0
      },
      {
         "an" : "Standard Tokenization Protocol",
         "as" : 17283256,
         "b" : "STPT",
         "ba" : "",
         "c" : "0.00000155",
         "cs" : 687660639,
         "etf" : false,
         "h" : "0.00000162",
         "i" : "1.00000000",
         "l" : "0.00000143",
         "o" : "0.00000147",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "26.34293224",
         "s" : "STPTBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "17283256.00000000",
         "y" : 0
      },
      {
         "an" : "Standard Tokenization Protocol",
         "as" : 63681209.6,
         "b" : "STPT",
         "ba" : "",
         "c" : "0.08473",
         "cs" : 687660639,
         "etf" : false,
         "h" : "0.08792",
         "i" : "0.10000000",
         "l" : "0.07629",
         "o" : "0.07857",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "5223349.732836",
         "s" : "STPTUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "63681209.60000",
         "y" : 0
      },
      {
         "an" : "Stratis",
         "as" : 628209.1,
         "b" : "STRAX",
         "ba" : "",
         "c" : "0.00003717",
         "cs" : null,
         "etf" : false,
         "h" : "0.00003814",
         "i" : "0.10000000",
         "l" : "0.00003373",
         "o" : "0.00003496",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "22.40793935",
         "s" : "STRAXBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "628209.10000000",
         "y" : 0
      },
      {
         "an" : "Stratis",
         "as" : 86488.6,
         "b" : "STRAX",
         "ba" : "",
         "c" : "2.0365",
         "cs" : null,
         "etf" : false,
         "h" : "2.1137",
         "i" : "0.01000000",
         "l" : "1.7852",
         "o" : "1.8904",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "167000.377082",
         "s" : "STRAXBUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "86488.6000",
         "y" : 0
      },
      {
         "an" : "Stratis",
         "as" : 144784.8,
         "b" : "STRAX",
         "ba" : "",
         "c" : "0.000794",
         "cs" : null,
         "etf" : false,
         "h" : "0.000816",
         "i" : "0.01000000",
         "l" : "0.000725",
         "o" : "0.000756",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "110.57645262",
         "s" : "STRAXETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "144784.800000",
         "y" : 0
      },
      {
         "an" : "Stratis",
         "as" : 2706408.44,
         "b" : "STRAX",
         "ba" : "",
         "c" : "2.0457",
         "cs" : null,
         "etf" : false,
         "h" : "2.0990",
         "i" : "0.01000000",
         "l" : "1.7805",
         "o" : "1.8816",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "5208216.459225",
         "s" : "STRAXUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "2706408.4400",
         "y" : 0
      },
      {
         "an" : "Stacks",
         "as" : 321807.9,
         "b" : "STX",
         "ba" : "",
         "c" : "0.003873",
         "cs" : 385937152,
         "etf" : false,
         "h" : "0.004180",
         "i" : "0.10000000",
         "l" : "0.003614",
         "o" : "0.003680",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1248.6980081",
         "s" : "STXBNB",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone",
            "storage-zone"
         ],
         "ts" : "0.000001",
         "v" : "321807.900000",
         "y" : 0
      },
      {
         "an" : "Stacks",
         "as" : 2417659,
         "b" : "STX",
         "ba" : "",
         "c" : "0.00004041",
         "cs" : 385937152,
         "etf" : false,
         "h" : "0.00004179",
         "i" : "1.00000000",
         "l" : "0.00003558",
         "o" : "0.00003695",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "93.73090282",
         "s" : "STXBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone",
            "storage-zone"
         ],
         "ts" : "0.00000001",
         "v" : "2417659.00000000",
         "y" : 0
      },
      {
         "an" : "Stacks",
         "as" : 6266474.79,
         "b" : "STX",
         "ba" : "",
         "c" : "2.2183",
         "cs" : 385937152,
         "etf" : false,
         "h" : "2.2820",
         "i" : "0.01000000",
         "l" : "1.8999",
         "o" : "1.9854",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "13138025.341333",
         "s" : "STXUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone",
            "storage-zone"
         ],
         "ts" : "0.0001",
         "v" : "6266474.7900",
         "y" : 0
      },
      {
         "an" : "SUN",
         "as" : 27653.782,
         "b" : "SUN",
         "ba" : "",
         "c" : "0.000609",
         "cs" : null,
         "etf" : false,
         "h" : "0.000623",
         "i" : "0.00100000",
         "l" : "0.000556",
         "o" : "0.000572",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "16.41181447",
         "s" : "SUNBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.000001",
         "v" : "27653.782000",
         "y" : 0
      },
      {
         "an" : "SUN",
         "as" : 198251.093,
         "b" : "SUN",
         "ba" : "",
         "c" : "33.427",
         "cs" : null,
         "etf" : false,
         "h" : "35.900",
         "i" : "0.00100000",
         "l" : "29.536",
         "o" : "30.784",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "6274776.995807",
         "s" : "SUNUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.001",
         "v" : "198251.093",
         "y" : 0
      },
      {
         "an" : "SuperFarm",
         "as" : 694255.3,
         "b" : "SUPER",
         "ba" : "",
         "c" : "0.00003894",
         "cs" : null,
         "etf" : false,
         "h" : "0.00004028",
         "i" : "0.10000000",
         "l" : "0.00003270",
         "o" : "0.00003430",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "24.98966608",
         "s" : "SUPERBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "NFT"
         ],
         "ts" : "0.00000001",
         "v" : "694255.30000000",
         "y" : 0
      },
      {
         "an" : "SuperFarm",
         "as" : 402289.883,
         "b" : "SUPER",
         "ba" : "",
         "c" : "2.144",
         "cs" : null,
         "etf" : false,
         "h" : "2.222",
         "i" : "0.00100000",
         "l" : "1.756",
         "o" : "1.857",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "796136.207173",
         "s" : "SUPERBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "NFT"
         ],
         "ts" : "0.001",
         "v" : "402289.883",
         "y" : 0
      },
      {
         "an" : "SuperFarm",
         "as" : 3573692.352,
         "b" : "SUPER",
         "ba" : "",
         "c" : "2.140",
         "cs" : null,
         "etf" : false,
         "h" : "2.220",
         "i" : "0.00100000",
         "l" : "1.750",
         "o" : "1.849",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "6979585.717503",
         "s" : "SUPERUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi",
            "NFT"
         ],
         "ts" : "0.001",
         "v" : "3573692.352",
         "y" : 0
      },
      {
         "an" : "sUSD",
         "as" : 321656,
         "b" : "SUSD",
         "ba" : "",
         "c" : "0.00001858",
         "cs" : null,
         "etf" : false,
         "h" : "0.00001939",
         "i" : "1.00000000",
         "l" : "0.00001845",
         "o" : "0.00001900",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "6.09063865",
         "s" : "SUSDBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "321656.00000000",
         "y" : 0
      },
      {
         "an" : "sUSD",
         "as" : 649459.85,
         "b" : "SUSD",
         "ba" : "",
         "c" : "1.0179",
         "cs" : null,
         "etf" : false,
         "h" : "1.0245",
         "i" : "0.01000000",
         "l" : "1.0150",
         "o" : "1.0232",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "661908.777993",
         "s" : "SUSDUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "649459.8500",
         "y" : 0
      },
      {
         "an" : "Sushi",
         "as" : 121727.32,
         "b" : "SUSHI",
         "ba" : "",
         "c" : "0.02223",
         "cs" : null,
         "etf" : false,
         "h" : "0.02379",
         "i" : "0.01000000",
         "l" : "0.02206",
         "o" : "0.02267",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "2773.4704338",
         "s" : "SUSHIBNB",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone"
         ],
         "ts" : "0.00001",
         "v" : "121727.32000",
         "y" : 0
      },
      {
         "an" : "Sushi",
         "as" : 629148.81,
         "b" : "SUSHI",
         "ba" : "",
         "c" : "0.0002319",
         "cs" : null,
         "etf" : false,
         "h" : "0.0002362",
         "i" : "0.01000000",
         "l" : "0.0002181",
         "o" : "0.0002269",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "143.84587539",
         "s" : "SUSHIBTC",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone"
         ],
         "ts" : "0.0000001",
         "v" : "629148.8100000",
         "y" : 0
      },
      {
         "an" : "Sushi",
         "as" : 316248.117,
         "b" : "SUSHI",
         "ba" : "",
         "c" : "12.762",
         "cs" : null,
         "etf" : false,
         "h" : "12.927",
         "i" : "0.00100000",
         "l" : "11.512",
         "o" : "12.219",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "3890171.516502",
         "s" : "SUSHIBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone"
         ],
         "ts" : "0.001",
         "v" : "316248.117",
         "y" : 0
      },
      {
         "an" : "SUSHIDOWN",
         "as" : 4780896.45,
         "b" : "SUSHIDOWN",
         "ba" : "",
         "c" : "0.2613368",
         "cs" : null,
         "etf" : true,
         "h" : "0.3511417",
         "i" : "0.01000000",
         "l" : "0.2496851",
         "o" : "0.2999809",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "1408990.91526903",
         "s" : "SUSHIDOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "4780896.4500000",
         "y" : 0
      },
      {
         "an" : "SUSHIUP",
         "as" : 2936675.19,
         "b" : "SUSHIUP",
         "ba" : "",
         "c" : "2.454",
         "cs" : null,
         "etf" : true,
         "h" : "2.530",
         "i" : "0.01000000",
         "l" : "2.038",
         "o" : "2.286",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "6765224.28331",
         "s" : "SUSHIUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "2936675.190",
         "y" : 0
      },
      {
         "an" : "Sushi",
         "as" : 4102458.981,
         "b" : "SUSHI",
         "ba" : "",
         "c" : "12.742",
         "cs" : null,
         "etf" : false,
         "h" : "12.924",
         "i" : "0.00100000",
         "l" : "11.500",
         "o" : "12.204",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "50368274.879085",
         "s" : "SUSHIUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone"
         ],
         "ts" : "0.001",
         "v" : "4102458.981",
         "y" : 0
      },
      {
         "an" : "Swerve",
         "as" : 230346.6,
         "b" : "SWRV",
         "ba" : "",
         "c" : "0.002836",
         "cs" : null,
         "etf" : false,
         "h" : "0.002994",
         "i" : "0.01000000",
         "l" : "0.002770",
         "o" : "0.002850",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "662.16041514",
         "s" : "SWRVBNB",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.000001",
         "v" : "230346.600000",
         "y" : 0
      },
      {
         "an" : "Swerve",
         "as" : 578286.631,
         "b" : "SWRV",
         "ba" : "",
         "c" : "1.603",
         "cs" : null,
         "etf" : false,
         "h" : "1.668",
         "i" : "0.00100000",
         "l" : "1.451",
         "o" : "1.526",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "894876.454277",
         "s" : "SWRVBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.001",
         "v" : "578286.631",
         "y" : 0
      },
      {
         "an" : "Swipe",
         "as" : 128272.306,
         "b" : "SXP",
         "ba" : "",
         "c" : "5.047",
         "cs" : 74966934,
         "etf" : false,
         "h" : "5.133",
         "i" : "0.00100000",
         "l" : "4.369",
         "o" : "4.667",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "AUD",
         "qa" : "",
         "qn" : "Australian Dollar",
         "qv" : "610449.238458",
         "s" : "SXPAUD",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "128272.306",
         "y" : 0
      },
      {
         "an" : "Swipe",
         "as" : 141252.02,
         "b" : "SXP",
         "ba" : "",
         "c" : "57773",
         "cs" : 74966934,
         "etf" : false,
         "h" : "58760",
         "i" : "0.01000000",
         "l" : "50406",
         "o" : "53741",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BIDR",
         "qa" : "",
         "qn" : "BIDR",
         "qv" : "7716421911.4",
         "s" : "SXPBIDR",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "1",
         "v" : "141253",
         "y" : 0
      },
      {
         "an" : "Swipe",
         "as" : 692334.3,
         "b" : "SXP",
         "ba" : "",
         "c" : "0.00681",
         "cs" : 74966934,
         "etf" : false,
         "h" : "0.00703",
         "i" : "0.10000000",
         "l" : "0.00644",
         "o" : "0.00669",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "4708.947262",
         "s" : "SXPBNB",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.00001",
         "v" : "692334.30000",
         "y" : 0
      },
      {
         "an" : "Swipe",
         "as" : 6383945,
         "b" : "SXP",
         "ba" : "",
         "c" : "0.00007094",
         "cs" : 74966934,
         "etf" : false,
         "h" : "0.00007260",
         "i" : "1.00000000",
         "l" : "0.00006415",
         "o" : "0.00006724",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "436.05223144",
         "s" : "SXPBTC",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.00000001",
         "v" : "6383945.00000000",
         "y" : 0
      },
      {
         "an" : "Swipe",
         "as" : 4559007.72,
         "b" : "SXP",
         "ba" : "",
         "c" : "3.9074",
         "cs" : 74966934,
         "etf" : false,
         "h" : "3.9795",
         "i" : "0.01000000",
         "l" : "3.3807",
         "o" : "3.6249",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "16751771.542844",
         "s" : "SXPBUSD",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.0001",
         "v" : "4559007.7200",
         "y" : 0
      },
      {
         "an" : "SXPDOWN",
         "as" : 190408.61,
         "b" : "SXPDOWN",
         "ba" : "",
         "c" : "24.21",
         "cs" : null,
         "etf" : true,
         "h" : "34.30",
         "i" : "0.01000000",
         "l" : "22.72",
         "o" : "31.01",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "5574769.169",
         "s" : "SXPDOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.01",
         "v" : "190408.61",
         "y" : 0
      },
      {
         "an" : "Swipe",
         "as" : 1062190.459,
         "b" : "SXP",
         "ba" : "",
         "c" : "3.236",
         "cs" : 74966934,
         "etf" : false,
         "h" : "3.299",
         "i" : "0.00100000",
         "l" : "2.807",
         "o" : "2.997",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "3226119.864631",
         "s" : "SXPEUR",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "1062190.459",
         "y" : 0
      },
      {
         "an" : "Swipe",
         "as" : 283984.71,
         "b" : "SXP",
         "ba" : "",
         "c" : "2.8182",
         "cs" : 74966934,
         "etf" : false,
         "h" : "2.8650",
         "i" : "0.01000000",
         "l" : "2.4406",
         "o" : "2.5882",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "GBP",
         "qa" : "",
         "qn" : "Pound Sterling",
         "qv" : "749639.774615",
         "s" : "SXPGBP",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.0001",
         "v" : "283984.7100",
         "y" : 0
      },
      {
         "an" : "Swipe",
         "as" : 2589675.28,
         "b" : "SXP",
         "ba" : "",
         "c" : "31.71",
         "cs" : 74966934,
         "etf" : false,
         "h" : "32.35",
         "i" : "0.01000000",
         "l" : "27.51",
         "o" : "29.51",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "77545011.8127",
         "s" : "SXPTRY",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.01",
         "v" : "2589675.28",
         "y" : 0
      },
      {
         "an" : "SXPUP",
         "as" : 5664966.57,
         "b" : "SXPUP",
         "ba" : "",
         "c" : "1.8721",
         "cs" : null,
         "etf" : true,
         "h" : "2.0000",
         "i" : "0.01000000",
         "l" : "1.4367",
         "o" : "1.6482",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "9500153.117635",
         "s" : "SXPUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "5664966.5700",
         "y" : 0
      },
      {
         "an" : "Swipe",
         "as" : 53426757.579,
         "b" : "SXP",
         "ba" : "",
         "c" : "3.900",
         "cs" : 74966934,
         "etf" : false,
         "h" : "3.977",
         "i" : "0.00100000",
         "l" : "3.372",
         "o" : "3.616",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "195242917.802112",
         "s" : "SXPUSDT",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "53426757.579",
         "y" : 0
      },
      {
         "an" : "Syscoin",
         "as" : 4052149,
         "b" : "SYS",
         "ba" : "",
         "c" : "0.00000617",
         "cs" : 611037651,
         "etf" : false,
         "h" : "0.00000639",
         "i" : "1.00000000",
         "l" : "0.00000573",
         "o" : "0.00000608",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "24.57451227",
         "s" : "SYSBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "4052149.00000000",
         "y" : 0
      },
      {
         "an" : "Syscoin",
         "as" : 1641850.1,
         "b" : "SYS",
         "ba" : "",
         "c" : "0.33864",
         "cs" : 611037651,
         "etf" : false,
         "h" : "0.35400",
         "i" : "0.10000000",
         "l" : "0.31000",
         "o" : "0.32618",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "543194.307288",
         "s" : "SYSBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "1641850.10000",
         "y" : 0
      },
      {
         "an" : "TokenClub Token",
         "as" : 30240703,
         "b" : "TCT",
         "ba" : "",
         "c" : "0.00000083",
         "cs" : 733102279,
         "etf" : false,
         "h" : "0.00000086",
         "i" : "1.00000000",
         "l" : "0.00000073",
         "o" : "0.00000076",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "24.05219141",
         "s" : "TCTBTC",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.00000001",
         "v" : "30240703.00000000",
         "y" : 0
      },
      {
         "an" : "TokenClub Token",
         "as" : 60864194,
         "b" : "TCT",
         "ba" : "",
         "c" : "0.045285",
         "cs" : 733102279,
         "etf" : false,
         "h" : "0.047000",
         "i" : "1.00000000",
         "l" : "0.038431",
         "o" : "0.041052",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "2553853.666673",
         "s" : "TCTUSDT",
         "st" : "TRADING",
         "tags" : [
            "BSC"
         ],
         "ts" : "0.000001",
         "v" : "60864194.000000",
         "y" : 0
      },
      {
         "an" : "Theta Fuel",
         "as" : 42587810,
         "b" : "TFUEL",
         "ba" : "",
         "c" : "0.00000587",
         "cs" : 5000000000,
         "etf" : false,
         "h" : "0.00000594",
         "i" : "1.00000000",
         "l" : "0.00000550",
         "o" : "0.00000576",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "243.23286967",
         "s" : "TFUELBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "42587810.00000000",
         "y" : 0
      },
      {
         "an" : "Theta Fuel",
         "as" : 103943395.9,
         "b" : "TFUEL",
         "ba" : "",
         "c" : "0.32270",
         "cs" : 5000000000,
         "etf" : false,
         "h" : "0.32550",
         "i" : "0.10000000",
         "l" : "0.29053",
         "o" : "0.31091",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "32075541.480565",
         "s" : "TFUELUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "103943395.90000",
         "y" : 0
      },
      {
         "an" : "Theta Token",
         "as" : 246113,
         "b" : "THETA",
         "ba" : "",
         "c" : "0.019976",
         "cs" : 1000000000,
         "etf" : false,
         "h" : "0.021734",
         "i" : "1.00000000",
         "l" : "0.019550",
         "o" : "0.020644",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "5043.690357",
         "s" : "THETABNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "246113.000000",
         "y" : 0
      },
      {
         "an" : "Theta Token",
         "as" : 2709361.7,
         "b" : "THETA",
         "ba" : "",
         "c" : "0.00020847",
         "cs" : 1000000000,
         "etf" : false,
         "h" : "0.00021521",
         "i" : "0.10000000",
         "l" : "0.00019385",
         "o" : "0.00020622",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "551.61130227",
         "s" : "THETABTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "2709361.70000000",
         "y" : 0
      },
      {
         "an" : "Theta Token",
         "as" : 182715,
         "b" : "THETA",
         "ba" : "",
         "c" : "0.0044784",
         "cs" : 1000000000,
         "etf" : false,
         "h" : "0.0046021",
         "i" : "1.00000000",
         "l" : "0.0041835",
         "o" : "0.0044304",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "798.11176803",
         "s" : "THETAETH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "182715.0000000",
         "y" : 0
      },
      {
         "an" : "Theta Token",
         "as" : 313866.309,
         "b" : "THETA",
         "ba" : "",
         "c" : "9.489",
         "cs" : 1000000000,
         "etf" : false,
         "h" : "9.770",
         "i" : "0.00100000",
         "l" : "8.460",
         "o" : "9.207",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "2867852.12847",
         "s" : "THETAEUR",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "313866.309",
         "y" : 0
      },
      {
         "an" : "Theta Token",
         "as" : 15839096.409,
         "b" : "THETA",
         "ba" : "",
         "c" : "11.457",
         "cs" : 1000000000,
         "etf" : false,
         "h" : "11.775",
         "i" : "0.00100000",
         "l" : "10.171",
         "o" : "11.093",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "173408955.328719",
         "s" : "THETAUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "15839096.409",
         "y" : 0
      },
      {
         "an" : "Tokocrypto",
         "as" : 1585506,
         "b" : "TKO",
         "ba" : "",
         "c" : "39000.00",
         "cs" : null,
         "etf" : false,
         "h" : "40500.00",
         "i" : "1.00000000",
         "l" : "33000.00",
         "o" : "36346.00",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BIDR",
         "qa" : "",
         "qn" : "BIDR",
         "qv" : "58058544095.65",
         "s" : "TKOBIDR",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone"
         ],
         "ts" : "0.01",
         "v" : "1585506.00",
         "y" : 0
      },
      {
         "an" : "Tokocrypto",
         "as" : 2153364,
         "b" : "TKO",
         "ba" : "",
         "c" : "0.00004797",
         "cs" : null,
         "etf" : false,
         "h" : "0.00005036",
         "i" : "1.00000000",
         "l" : "0.00004325",
         "o" : "0.00004557",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "99.35250519",
         "s" : "TKOBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone"
         ],
         "ts" : "0.00000001",
         "v" : "2153364.00000000",
         "y" : 0
      },
      {
         "an" : "Tokocrypto",
         "as" : 2318877.34,
         "b" : "TKO",
         "ba" : "",
         "c" : "2.6292",
         "cs" : null,
         "etf" : false,
         "h" : "2.7380",
         "i" : "0.01000000",
         "l" : "2.3050",
         "o" : "2.4523",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "5850930.978954",
         "s" : "TKOBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone"
         ],
         "ts" : "0.0001",
         "v" : "2318877.3400",
         "y" : 0
      },
      {
         "an" : "Tokocrypto",
         "as" : 18612800.34,
         "b" : "TKO",
         "ba" : "",
         "c" : "2.6334",
         "cs" : null,
         "etf" : false,
         "h" : "2.7400",
         "i" : "0.01000000",
         "l" : "2.2935",
         "o" : "2.4514",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "46780982.123448",
         "s" : "TKOUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone"
         ],
         "ts" : "0.0001",
         "v" : "18612800.3400",
         "y" : 0
      },
      {
         "an" : "Alien Worlds",
         "as" : 11716078.7,
         "b" : "TLM",
         "ba" : "",
         "c" : "0.00000504",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000514",
         "i" : "0.10000000",
         "l" : "0.00000472",
         "o" : "0.00000491",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "58.05689787",
         "s" : "TLMBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone"
         ],
         "ts" : "0.00000001",
         "v" : "11716078.70000000",
         "y" : 0
      },
      {
         "an" : "Alien Worlds",
         "as" : 8981148.8,
         "b" : "TLM",
         "ba" : "",
         "c" : "0.2773",
         "cs" : null,
         "etf" : false,
         "h" : "0.2841",
         "i" : "0.01000000",
         "l" : "0.2493",
         "o" : "0.2648",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2404457.513456",
         "s" : "TLMBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone"
         ],
         "ts" : "0.0001",
         "v" : "8981148.8000",
         "y" : 0
      },
      {
         "an" : "Alien Worlds",
         "as" : 84886475.02,
         "b" : "TLM",
         "ba" : "",
         "c" : "0.2768",
         "cs" : null,
         "etf" : false,
         "h" : "0.2843",
         "i" : "0.01000000",
         "l" : "0.2490",
         "o" : "0.2641",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "22685023.224992",
         "s" : "TLMUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone"
         ],
         "ts" : "0.0001",
         "v" : "84886475.0200",
         "y" : 0
      },
      {
         "an" : "TomoChain",
         "as" : 1685084,
         "b" : "TOMO",
         "ba" : "",
         "c" : "0.00004161",
         "cs" : 81194350,
         "etf" : false,
         "h" : "0.00004240",
         "i" : "1.00000000",
         "l" : "0.00003906",
         "o" : "0.00004036",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "68.85973258",
         "s" : "TOMOBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "1685084.00000000",
         "y" : 0
      },
      {
         "an" : "TomoChain",
         "as" : 263748.16,
         "b" : "TOMO",
         "ba" : "",
         "c" : "2.2766",
         "cs" : 81194350,
         "etf" : false,
         "h" : "2.3433",
         "i" : "0.01000000",
         "l" : "2.0767",
         "o" : "2.1689",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "586916.227826",
         "s" : "TOMOBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "263748.1600",
         "y" : 0
      },
      {
         "an" : "TomoChain",
         "as" : 4394529.53,
         "b" : "TOMO",
         "ba" : "",
         "c" : "2.2849",
         "cs" : 81194350,
         "etf" : false,
         "h" : "2.3233",
         "i" : "0.01000000",
         "l" : "2.0564",
         "o" : "2.1671",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "9651551.515681",
         "s" : "TOMOUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "4394529.5300",
         "y" : 0
      },
      {
         "an" : "Tellor Tributes",
         "as" : 107866.58,
         "b" : "TRB",
         "ba" : "",
         "c" : "0.001719",
         "cs" : null,
         "etf" : false,
         "h" : "0.001846",
         "i" : "0.00100000",
         "l" : "0.001298",
         "o" : "0.001302",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "176.03395749",
         "s" : "TRBBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "defi",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "107866.580000",
         "y" : 0
      },
      {
         "an" : "Tellor Tributes",
         "as" : 21212.868,
         "b" : "TRB",
         "ba" : "",
         "c" : "94.305",
         "cs" : null,
         "etf" : false,
         "h" : "101.199",
         "i" : "0.00100000",
         "l" : "68.806",
         "o" : "70.000",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1877076.412782",
         "s" : "TRBBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "defi",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "21212.868",
         "y" : 0
      },
      {
         "an" : "Tellor Tributes",
         "as" : 554794.243,
         "b" : "TRB",
         "ba" : "",
         "c" : "94.481",
         "cs" : null,
         "etf" : false,
         "h" : "101.000",
         "i" : "0.00100000",
         "l" : "68.474",
         "o" : "70.041",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "49261081.182011",
         "s" : "TRBUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "defi",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "554794.243",
         "y" : 0
      },
      {
         "an" : "Troy",
         "as" : 22038497,
         "b" : "TROY",
         "ba" : "",
         "c" : "0.00003920",
         "cs" : 1290000000,
         "etf" : false,
         "h" : "0.00004239",
         "i" : "1.00000000",
         "l" : "0.00003705",
         "o" : "0.00003920",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "869.37561241",
         "s" : "TROYBNB",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "22038497.00000000",
         "y" : 0
      },
      {
         "an" : "Troy",
         "as" : 299712986,
         "b" : "TROY",
         "ba" : "",
         "c" : "0.00000041",
         "cs" : 1290000000,
         "etf" : false,
         "h" : "0.00000042",
         "i" : "1.00000000",
         "l" : "0.00000037",
         "o" : "0.00000040",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "118.31823587",
         "s" : "TROYBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "299712986.00000000",
         "y" : 0
      },
      {
         "an" : "Troy",
         "as" : 269694956,
         "b" : "TROY",
         "ba" : "",
         "c" : "0.022436",
         "cs" : 1290000000,
         "etf" : false,
         "h" : "0.023000",
         "i" : "1.00000000",
         "l" : "0.019477",
         "o" : "0.021163",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "5743617.9683139",
         "s" : "TROYUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "269694956.000000",
         "y" : 0
      },
      {
         "an" : "TrueFi",
         "as" : 2304092.8,
         "b" : "TRU",
         "ba" : "",
         "c" : "0.00000657",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000676",
         "i" : "0.10000000",
         "l" : "0.00000590",
         "o" : "0.00000606",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "14.52316716",
         "s" : "TRUBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "2304092.80000000",
         "y" : 0
      },
      {
         "an" : "TrueFi",
         "as" : 845797.44,
         "b" : "TRU",
         "ba" : "",
         "c" : "0.3608",
         "cs" : null,
         "etf" : false,
         "h" : "0.3717",
         "i" : "0.01000000",
         "l" : "0.3112",
         "o" : "0.3245",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "288309.099726",
         "s" : "TRUBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "845797.4400",
         "y" : 0
      },
      {
         "an" : "TrueFi",
         "as" : 9459071.09,
         "b" : "TRU",
         "ba" : "",
         "c" : "0.3603",
         "cs" : null,
         "etf" : false,
         "h" : "0.3763",
         "i" : "0.01000000",
         "l" : "0.3101",
         "o" : "0.3260",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "3232372.486279",
         "s" : "TRUUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "9459071.0900",
         "y" : 0
      },
      {
         "an" : "TRON",
         "as" : 53696115,
         "b" : "TRX",
         "ba" : "",
         "c" : "0.0002199",
         "cs" : 0,
         "etf" : false,
         "h" : "0.0002327",
         "i" : "1.00000000",
         "l" : "0.0002100",
         "o" : "0.0002136",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "11911.3847284",
         "s" : "TRXBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "53696115.0000000",
         "y" : 0
      },
      {
         "an" : "TRON",
         "as" : 246094045,
         "b" : "TRX",
         "ba" : "",
         "c" : "0.00000229",
         "cs" : 0,
         "etf" : false,
         "h" : "0.00000235",
         "i" : "1.00000000",
         "l" : "0.00000209",
         "o" : "0.00000214",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "543.31103079",
         "s" : "TRXBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "246094045.00000000",
         "y" : 0
      },
      {
         "an" : "TRON",
         "as" : 87207616.2,
         "b" : "TRX",
         "ba" : "",
         "c" : "0.12623",
         "cs" : 0,
         "etf" : false,
         "h" : "0.12894",
         "i" : "0.10000000",
         "l" : "0.11134",
         "o" : "0.11516",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "10372347.406507",
         "s" : "TRXBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "87207616.20000",
         "y" : 0
      },
      {
         "an" : "TRXDOWN",
         "as" : 854537592.01,
         "b" : "TRXDOWN",
         "ba" : "",
         "c" : "0.001922",
         "cs" : null,
         "etf" : true,
         "h" : "0.002781",
         "i" : "0.01000000",
         "l" : "0.001800",
         "o" : "0.002554",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "1972328.04156359",
         "s" : "TRXDOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "854537592.010000",
         "y" : 0
      },
      {
         "an" : "TRON",
         "as" : 61011836,
         "b" : "TRX",
         "ba" : "",
         "c" : "0.00004923",
         "cs" : 0,
         "etf" : false,
         "h" : "0.00005042",
         "i" : "1.00000000",
         "l" : "0.00004500",
         "o" : "0.00004590",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "2896.69590929",
         "s" : "TRXETH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "61011836.00000000",
         "y" : 0
      },
      {
         "an" : "TRON",
         "as" : 316951.2,
         "b" : "TRX",
         "ba" : "",
         "c" : "0.12531",
         "cs" : 0,
         "etf" : false,
         "h" : "0.12851",
         "i" : "0.10000000",
         "l" : "0.11171",
         "o" : "0.11507",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "PAX",
         "qa" : "",
         "qn" : "Paxos Standard",
         "qv" : "37655.769208",
         "s" : "TRXPAX",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "316951.20000",
         "y" : 0
      },
      {
         "an" : "TRON",
         "as" : 114988084,
         "b" : "TRX",
         "ba" : "",
         "c" : "1.0265",
         "cs" : 0,
         "etf" : false,
         "h" : "1.0480",
         "i" : "1.00000000",
         "l" : "0.9090",
         "o" : "0.9397",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "112210238.3835",
         "s" : "TRXTRY",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "114988084.0000",
         "y" : 0
      },
      {
         "an" : "TRON",
         "as" : 7762400.9,
         "b" : "TRX",
         "ba" : "",
         "c" : "0.12625",
         "cs" : 0,
         "etf" : false,
         "h" : "0.12886",
         "i" : "0.10000000",
         "l" : "0.11147",
         "o" : "0.11550",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TUSD",
         "qa" : "",
         "qn" : "TrueUSD",
         "qv" : "919733.944381",
         "s" : "TRXTUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "7762400.90000",
         "y" : 0
      },
      {
         "an" : "TRXUP",
         "as" : 402070.99,
         "b" : "TRXUP",
         "ba" : "",
         "c" : "13.517",
         "cs" : null,
         "etf" : true,
         "h" : "14.099",
         "i" : "0.01000000",
         "l" : "10.660",
         "o" : "11.359",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "4969151.11198",
         "s" : "TRXUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "402070.990",
         "y" : 0
      },
      {
         "an" : "TRON",
         "as" : 19239574.3,
         "b" : "TRX",
         "ba" : "",
         "c" : "0.12611",
         "cs" : 0,
         "etf" : false,
         "h" : "0.12895",
         "i" : "0.10000000",
         "l" : "0.11135",
         "o" : "0.11513",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDC",
         "qa" : "",
         "qn" : "USD Coin",
         "qv" : "2260897.244139",
         "s" : "TRXUSDC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "19239574.30000",
         "y" : 0
      },
      {
         "an" : "TRON",
         "as" : 2605343361.3,
         "b" : "TRX",
         "ba" : "",
         "c" : "0.12607",
         "cs" : 0,
         "etf" : false,
         "h" : "0.12884",
         "i" : "0.10000000",
         "l" : "0.11119",
         "o" : "0.11494",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "309726123.912994",
         "s" : "TRXUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "2605343361.30000",
         "y" : 0
      },
      {
         "an" : "TRON",
         "as" : 25206768.8,
         "b" : "TRX",
         "ba" : "",
         "c" : "0.08927",
         "cs" : 0,
         "etf" : false,
         "h" : "0.09218",
         "i" : "0.10000000",
         "l" : "0.08267",
         "o" : "0.09104",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "XRP",
         "qa" : "",
         "qn" : "Ripple",
         "qv" : "2208253.25331",
         "s" : "TRXXRP",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "25206768.80000",
         "y" : 0
      },
      {
         "an" : "TrueUSD",
         "as" : 2879852.5,
         "b" : "TUSD",
         "ba" : "",
         "c" : "1.0000",
         "cs" : 321016056,
         "etf" : false,
         "h" : "1.0009",
         "i" : "0.01000000",
         "l" : "0.9999",
         "o" : "0.9999",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2880152.788788",
         "s" : "TUSDBUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "2879852.5000",
         "y" : 0
      },
      {
         "an" : "TrueUSD",
         "as" : 14578274.17,
         "b" : "TUSD",
         "ba" : "",
         "c" : "0.9987",
         "cs" : 321016056,
         "etf" : false,
         "h" : "1.0007",
         "i" : "0.01000000",
         "l" : "0.9983",
         "o" : "0.9984",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "14562963.892952",
         "s" : "TUSDUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "14578274.1700",
         "y" : 0
      },
      {
         "an" : "Terra Virtua",
         "as" : 12026094,
         "b" : "TVK",
         "ba" : "",
         "c" : "0.00000831",
         "cs" : null,
         "etf" : false,
         "h" : "0.00000887",
         "i" : "1.00000000",
         "l" : "0.00000726",
         "o" : "0.00000770",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "97.58887935",
         "s" : "TVKBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "NFT"
         ],
         "ts" : "0.00000001",
         "v" : "12026094.00000000",
         "y" : 0
      },
      {
         "an" : "Terra Virtua",
         "as" : 19270816.08,
         "b" : "TVK",
         "ba" : "",
         "c" : "0.4593",
         "cs" : null,
         "etf" : false,
         "h" : "0.4846",
         "i" : "0.01000000",
         "l" : "0.3818",
         "o" : "0.4144",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "8345966.854408",
         "s" : "TVKBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "NFT"
         ],
         "ts" : "0.0001",
         "v" : "19270816.0800",
         "y" : 0
      },
      {
         "an" : "Trust Wallet Token",
         "as" : 2309927,
         "b" : "TWT",
         "ba" : "",
         "c" : "0.00001214",
         "cs" : null,
         "etf" : false,
         "h" : "0.00001307",
         "i" : "1.00000000",
         "l" : "0.00001115",
         "o" : "0.00001128",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "27.9154213",
         "s" : "TWTBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.00000001",
         "v" : "2309927.00000000",
         "y" : 0
      },
      {
         "an" : "Trust Wallet Token",
         "as" : 2119852.55,
         "b" : "TWT",
         "ba" : "",
         "c" : "0.6695",
         "cs" : null,
         "etf" : false,
         "h" : "0.7009",
         "i" : "0.01000000",
         "l" : "0.5962",
         "o" : "0.6074",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1376996.394737",
         "s" : "TWTBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.0001",
         "v" : "2119852.5500",
         "y" : 0
      },
      {
         "an" : "Trust Wallet Token",
         "as" : 17822048.24,
         "b" : "TWT",
         "ba" : "",
         "c" : "0.6688",
         "cs" : null,
         "etf" : false,
         "h" : "0.7000",
         "i" : "0.01000000",
         "l" : "0.5950",
         "o" : "0.6066",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "11628274.552539",
         "s" : "TWTUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.0001",
         "v" : "17822048.2400",
         "y" : 0
      },
      {
         "an" : "UniLend",
         "as" : 1187010.88,
         "b" : "UFT",
         "ba" : "",
         "c" : "1.3929",
         "cs" : null,
         "etf" : false,
         "h" : "1.4300",
         "i" : "0.01000000",
         "l" : "1.2021",
         "o" : "1.3435",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1598744.873292",
         "s" : "UFTBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "1187010.8800",
         "y" : 0
      },
      {
         "an" : "UniLend",
         "as" : 828136.55,
         "b" : "UFT",
         "ba" : "",
         "c" : "0.0005449",
         "cs" : null,
         "etf" : false,
         "h" : "0.0005606",
         "i" : "0.01000000",
         "l" : "0.0005071",
         "o" : "0.0005357",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "441.15232471",
         "s" : "UFTETH",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone",
            "defi"
         ],
         "ts" : "0.0000001",
         "v" : "828136.5500000",
         "y" : 0
      },
      {
         "an" : "UMA",
         "as" : 64447.419,
         "b" : "UMA",
         "ba" : "",
         "c" : "0.0004818",
         "cs" : null,
         "etf" : false,
         "h" : "0.0004948",
         "i" : "0.00100000",
         "l" : "0.0004536",
         "o" : "0.0004855",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "30.40341272",
         "s" : "UMABTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.0000001",
         "v" : "64447.4190000",
         "y" : 0
      },
      {
         "an" : "UMA",
         "as" : 273057.515,
         "b" : "UMA",
         "ba" : "",
         "c" : "26.480",
         "cs" : null,
         "etf" : false,
         "h" : "26.630",
         "i" : "0.00100000",
         "l" : "23.981",
         "o" : "26.111",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "6937936.243383",
         "s" : "UMAUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "273057.515",
         "y" : 0
      },
      {
         "an" : "Unifi Protocol DAO",
         "as" : 4793.3,
         "b" : "UNFI",
         "ba" : "",
         "c" : "0.04216",
         "cs" : null,
         "etf" : false,
         "h" : "0.04420",
         "i" : "0.10000000",
         "l" : "0.04032",
         "o" : "0.04054",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "203.136306",
         "s" : "UNFIBNB",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.00001",
         "v" : "4793.30000",
         "y" : 0
      },
      {
         "an" : "Unifi Protocol DAO",
         "as" : 63084.5,
         "b" : "UNFI",
         "ba" : "",
         "c" : "0.00043857",
         "cs" : null,
         "etf" : false,
         "h" : "0.00045663",
         "i" : "0.10000000",
         "l" : "0.00040001",
         "o" : "0.00040594",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "26.70297111",
         "s" : "UNFIBTC",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.00000001",
         "v" : "63084.50000000",
         "y" : 0
      },
      {
         "an" : "Unifi Protocol DAO",
         "as" : 139287.263,
         "b" : "UNFI",
         "ba" : "",
         "c" : "24.174",
         "cs" : null,
         "etf" : false,
         "h" : "25.116",
         "i" : "0.00100000",
         "l" : "21.327",
         "o" : "21.825",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "3253949.225352",
         "s" : "UNFIBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "139287.263",
         "y" : 0
      },
      {
         "an" : "Unifi Protocol DAO",
         "as" : 283815.794,
         "b" : "UNFI",
         "ba" : "",
         "c" : "24.144",
         "cs" : null,
         "etf" : false,
         "h" : "25.123",
         "i" : "0.00100000",
         "l" : "21.280",
         "o" : "21.795",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "6457174.712624",
         "s" : "UNFIUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "283815.794",
         "y" : 0
      },
      {
         "an" : "Uniswap",
         "as" : 222670,
         "b" : "UNI",
         "ba" : "",
         "c" : "0.06887",
         "cs" : null,
         "etf" : false,
         "h" : "0.07519",
         "i" : "0.10000000",
         "l" : "0.06595",
         "o" : "0.06649",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "15864.70158",
         "s" : "UNIBNB",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00001",
         "v" : "222670.00000",
         "y" : 0
      },
      {
         "an" : "Uniswap",
         "as" : 782609.1,
         "b" : "UNI",
         "ba" : "",
         "c" : "0.00071867",
         "cs" : null,
         "etf" : false,
         "h" : "0.00074964",
         "i" : "0.10000000",
         "l" : "0.00065701",
         "o" : "0.00066540",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "556.8061652",
         "s" : "UNIBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00000001",
         "v" : "782609.10000000",
         "y" : 0
      },
      {
         "an" : "Uniswap",
         "as" : 381910.124,
         "b" : "UNI",
         "ba" : "",
         "c" : "39.567",
         "cs" : null,
         "etf" : false,
         "h" : "40.690",
         "i" : "0.00100000",
         "l" : "34.901",
         "o" : "35.829",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "14616416.6886528",
         "s" : "UNIBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "381910.124",
         "y" : 0
      },
      {
         "an" : "UNIDOWN",
         "as" : 4815966071.65,
         "b" : "UNIDOWN",
         "ba" : "",
         "c" : "0.000880",
         "cs" : null,
         "etf" : true,
         "h" : "0.001123",
         "i" : "0.01000000",
         "l" : "0.000822",
         "o" : "0.001074",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "4521694.95817059",
         "s" : "UNIDOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "4815966071.650000",
         "y" : 0
      },
      {
         "an" : "Uniswap",
         "as" : 62704.445,
         "b" : "UNI",
         "ba" : "",
         "c" : "32.700",
         "cs" : null,
         "etf" : false,
         "h" : "33.707",
         "i" : "0.00100000",
         "l" : "28.968",
         "o" : "29.731",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "2014538.479287",
         "s" : "UNIEUR",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "62704.445",
         "y" : 0
      },
      {
         "an" : "UNIUP",
         "as" : 269949.43,
         "b" : "UNIUP",
         "ba" : "",
         "c" : "37.794",
         "cs" : null,
         "etf" : true,
         "h" : "42.000",
         "i" : "0.01000000",
         "l" : "29.577",
         "o" : "31.158",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "9748054.68433",
         "s" : "UNIUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "269949.430",
         "y" : 0
      },
      {
         "an" : "Uniswap",
         "as" : 5328898.634,
         "b" : "UNI",
         "ba" : "",
         "c" : "39.473",
         "cs" : null,
         "etf" : false,
         "h" : "40.629",
         "i" : "0.00100000",
         "l" : "34.560",
         "o" : "35.770",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "204386675.625702",
         "s" : "UNIUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "5328898.634",
         "y" : 0
      },
      {
         "an" : "USD Coin",
         "as" : 24592412.52,
         "b" : "USDC",
         "ba" : "",
         "c" : "1.0000",
         "cs" : 12748949,
         "etf" : false,
         "h" : "1.0002",
         "i" : "0.01000000",
         "l" : "0.9998",
         "o" : "1.0000",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "24592265.034111",
         "s" : "USDCBUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "24592412.5200",
         "y" : 0
      },
      {
         "an" : "USD Coin",
         "as" : 114141098.03,
         "b" : "USDC",
         "ba" : "",
         "c" : "0.9986",
         "cs" : 12748949,
         "etf" : false,
         "h" : "0.9994",
         "i" : "0.01000000",
         "l" : "0.9983",
         "o" : "0.9986",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "114016964.775516",
         "s" : "USDCUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "114141098.0300",
         "y" : 0
      },
      {
         "an" : "TetherUS",
         "as" : 9358433.31,
         "b" : "USDT",
         "ba" : "",
         "c" : "14805",
         "cs" : 50305857792,
         "etf" : false,
         "h" : "15046",
         "i" : "0.01000000",
         "l" : "14660",
         "o" : "14890",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BIDR",
         "qa" : "",
         "qn" : "BIDR",
         "qv" : "139203267712.78",
         "s" : "USDTBIDR",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "1",
         "v" : "9358434",
         "y" : 0
      },
      {
         "an" : "TetherUS",
         "as" : 12007795.06,
         "b" : "USDT",
         "ba" : "",
         "c" : "5.466",
         "cs" : 50305857792,
         "etf" : false,
         "h" : "5.547",
         "i" : "0.01000000",
         "l" : "5.450",
         "o" : "5.539",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BRL",
         "qa" : "",
         "qn" : "Brazilian Real",
         "qv" : "65911295.11912",
         "s" : "USDTBRL",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "12007795.060",
         "y" : 0
      },
      {
         "an" : "TetherUS",
         "as" : 1081079,
         "b" : "USDT",
         "ba" : "",
         "c" : "24059",
         "cs" : 50305857792,
         "etf" : false,
         "h" : "24482",
         "i" : "0.01000000",
         "l" : "24001",
         "o" : "24354",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BVND",
         "qa" : "",
         "qn" : "BVND",
         "qv" : "26188480476.62",
         "s" : "USDTBVND",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "1",
         "v" : "1081079",
         "y" : 0
      },
      {
         "an" : "TetherUS",
         "as" : 20019465.4,
         "b" : "USDT",
         "ba" : "",
         "c" : "1.0000",
         "cs" : 50305857792,
         "etf" : false,
         "h" : "1.0015",
         "i" : "0.01000000",
         "l" : "0.9989",
         "o" : "1.0004",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "DAI",
         "qa" : "",
         "qn" : "Dai",
         "qv" : "20019639.032134",
         "s" : "USDTDAI",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "20019465.4000",
         "y" : 0
      },
      {
         "an" : "TetherUS",
         "as" : 819143.18,
         "b" : "USDT",
         "ba" : "",
         "c" : "14784",
         "cs" : 50305857792,
         "etf" : false,
         "h" : "15100",
         "i" : "0.01000000",
         "l" : "14726",
         "o" : "14842",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "IDRT",
         "qa" : "",
         "qn" : "Rupiah Token",
         "qv" : "12179564895.35",
         "s" : "USDTIDRT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "1",
         "v" : "819144",
         "y" : 0
      },
      {
         "an" : "TetherUS",
         "as" : 423699.66,
         "b" : "USDT",
         "ba" : "",
         "c" : "622.01",
         "cs" : 50305857792,
         "etf" : false,
         "h" : "647.21",
         "i" : "0.01000000",
         "l" : "620.48",
         "o" : "639.98",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "NGN",
         "qa" : "",
         "qn" : "Nigerian Naira",
         "qv" : "268660274.0243",
         "s" : "USDTNGN",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.01",
         "v" : "423699.66",
         "y" : 0
      },
      {
         "an" : "TetherUS",
         "as" : 6027030.5,
         "b" : "USDT",
         "ba" : "",
         "c" : "75.511",
         "cs" : 50305857792,
         "etf" : false,
         "h" : "75.869",
         "i" : "0.10000000",
         "l" : "75.300",
         "o" : "75.710",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "RUB",
         "qa" : "",
         "qn" : "Russian Ruble",
         "qv" : "455186595.4198",
         "s" : "USDTRUB",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "6027030.500",
         "y" : 0
      },
      {
         "an" : "TetherUS",
         "as" : 106875008.17,
         "b" : "USDT",
         "ba" : "",
         "c" : "8.145",
         "cs" : 50305857792,
         "etf" : false,
         "h" : "8.200",
         "i" : "0.01000000",
         "l" : "8.101",
         "o" : "8.167",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "871707207.35033",
         "s" : "USDTTRY",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "106875008.170",
         "y" : 0
      },
      {
         "an" : "TetherUS",
         "as" : 1165618.3,
         "b" : "USDT",
         "ba" : "",
         "c" : "28.196",
         "cs" : 50305857792,
         "etf" : false,
         "h" : "28.377",
         "i" : "0.10000000",
         "l" : "28.172",
         "o" : "28.351",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "UAH",
         "qa" : "",
         "qn" : "Ukraine Hryvnia",
         "qv" : "33003853.0455",
         "s" : "USDTUAH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "1165618.300",
         "y" : 0
      },
      {
         "an" : "Utrust",
         "as" : 4566938,
         "b" : "UTK",
         "ba" : "",
         "c" : "0.00000922",
         "cs" : 450000000,
         "etf" : false,
         "h" : "0.00000960",
         "i" : "1.00000000",
         "l" : "0.00000804",
         "o" : "0.00000853",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "40.37929238",
         "s" : "UTKBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "4566938.00000000",
         "y" : 0
      },
      {
         "an" : "Utrust",
         "as" : 11243640.54,
         "b" : "UTK",
         "ba" : "",
         "c" : "0.5057",
         "cs" : 450000000,
         "etf" : false,
         "h" : "0.5246",
         "i" : "0.01000000",
         "l" : "0.4255",
         "o" : "0.4580",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "5310750.961669",
         "s" : "UTKUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "11243640.5400",
         "y" : 0
      },
      {
         "an" : "VeChain",
         "as" : 61091435,
         "b" : "VET",
         "ba" : "",
         "c" : "0.0003751",
         "cs" : 55454734800,
         "etf" : false,
         "h" : "0.0003898",
         "i" : "1.00000000",
         "l" : "0.0003406",
         "o" : "0.0003523",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "22261.8411133",
         "s" : "VETBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "61091435.0000000",
         "y" : 0
      },
      {
         "an" : "VeChain",
         "as" : 348123240,
         "b" : "VET",
         "ba" : "",
         "c" : "0.00000391",
         "cs" : 55454734800,
         "etf" : false,
         "h" : "0.00000406",
         "i" : "1.00000000",
         "l" : "0.00000338",
         "o" : "0.00000353",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "1263.72859119",
         "s" : "VETBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "348123240.00000000",
         "y" : 0
      },
      {
         "an" : "VeChain",
         "as" : 350848715.6,
         "b" : "VET",
         "ba" : "",
         "c" : "0.21514",
         "cs" : 55454734800,
         "etf" : false,
         "h" : "0.22399",
         "i" : "0.10000000",
         "l" : "0.17973",
         "o" : "0.18990",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "68957817.239003",
         "s" : "VETBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "350848715.60000",
         "y" : 0
      },
      {
         "an" : "VeChain",
         "as" : 114606351,
         "b" : "VET",
         "ba" : "",
         "c" : "0.00008385",
         "cs" : 55454734800,
         "etf" : false,
         "h" : "0.00008698",
         "i" : "1.00000000",
         "l" : "0.00007300",
         "o" : "0.00007562",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "9043.29412827",
         "s" : "VETETH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "114606351.00000000",
         "y" : 0
      },
      {
         "an" : "VeChain",
         "as" : 92183696.47,
         "b" : "VET",
         "ba" : "",
         "c" : "0.1784",
         "cs" : 55454734800,
         "etf" : false,
         "h" : "0.1852",
         "i" : "0.01000000",
         "l" : "0.1490",
         "o" : "0.1576",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "15216640.465527",
         "s" : "VETEUR",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "92183696.4700",
         "y" : 0
      },
      {
         "an" : "VeChain",
         "as" : 32420404.66,
         "b" : "VET",
         "ba" : "",
         "c" : "0.1549",
         "cs" : 55454734800,
         "etf" : false,
         "h" : "0.1613",
         "i" : "0.01000000",
         "l" : "0.1298",
         "o" : "0.1372",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "GBP",
         "qa" : "",
         "qn" : "Pound Sterling",
         "qv" : "4633769.17934",
         "s" : "VETGBP",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "32420404.6600",
         "y" : 0
      },
      {
         "an" : "VeChain",
         "as" : 2877963312.7,
         "b" : "VET",
         "ba" : "",
         "c" : "0.21486",
         "cs" : 55454734800,
         "etf" : false,
         "h" : "0.22390",
         "i" : "0.10000000",
         "l" : "0.17942",
         "o" : "0.18950",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "566608525.3787124",
         "s" : "VETUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "2877963312.70000",
         "y" : 0
      },
      {
         "an" : "Viacoin",
         "as" : 2306482,
         "b" : "VIA",
         "ba" : "",
         "c" : "0.00003378",
         "cs" : 23173985,
         "etf" : false,
         "h" : "0.00003553",
         "i" : "1.00000000",
         "l" : "0.00003053",
         "o" : "0.00003225",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "76.18490951",
         "s" : "VIABTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "2306482.00000000",
         "y" : 0
      },
      {
         "an" : "Viberate",
         "as" : 13960051,
         "b" : "VIB",
         "ba" : "",
         "c" : "0.00000210",
         "cs" : 196825000,
         "etf" : false,
         "h" : "0.00000212",
         "i" : "1.00000000",
         "l" : "0.00000188",
         "o" : "0.00000193",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "27.90666635",
         "s" : "VIBBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "13960051.00000000",
         "y" : 0
      },
      {
         "an" : "Viberate",
         "as" : 4453866,
         "b" : "VIB",
         "ba" : "",
         "c" : "0.00004516",
         "cs" : 196825000,
         "etf" : false,
         "h" : "0.00004586",
         "i" : "1.00000000",
         "l" : "0.00004046",
         "o" : "0.00004149",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "191.72971747",
         "s" : "VIBETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "4453866.00000000",
         "y" : 0
      },
      {
         "an" : "VIDT Datalink",
         "as" : 1767185,
         "b" : "VIDT",
         "ba" : "",
         "c" : "0.00001630",
         "cs" : null,
         "etf" : false,
         "h" : "0.00001716",
         "i" : "1.00000000",
         "l" : "0.00001467",
         "o" : "0.00001529",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "27.52002547",
         "s" : "VIDTBTC",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone"
         ],
         "ts" : "0.00000001",
         "v" : "1767185.00000000",
         "y" : 0
      },
      {
         "an" : "VIDT Datalink",
         "as" : 1926848.22,
         "b" : "VIDT",
         "ba" : "",
         "c" : "0.8937",
         "cs" : null,
         "etf" : false,
         "h" : "1.0000",
         "i" : "0.01000000",
         "l" : "0.7777",
         "o" : "0.8193",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1636224.370214",
         "s" : "VIDTBUSD",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone"
         ],
         "ts" : "0.0001",
         "v" : "1926848.2200",
         "y" : 0
      },
      {
         "an" : "VITE",
         "as" : 16471933,
         "b" : "VITE",
         "ba" : "",
         "c" : "0.00000307",
         "cs" : 480929099,
         "etf" : false,
         "h" : "0.00000316",
         "i" : "1.00000000",
         "l" : "0.00000277",
         "o" : "0.00000298",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "48.62559689",
         "s" : "VITEBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "16471933.00000000",
         "y" : 0
      },
      {
         "an" : "VITE",
         "as" : 78083340.4,
         "b" : "VITE",
         "ba" : "",
         "c" : "0.16799",
         "cs" : 480929099,
         "etf" : false,
         "h" : "0.17400",
         "i" : "0.10000000",
         "l" : "0.14672",
         "o" : "0.16068",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "12370015.87395",
         "s" : "VITEUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "78083340.40000",
         "y" : 0
      },
      {
         "an" : "VeThor Token",
         "as" : 124703574,
         "b" : "VTHO",
         "ba" : "",
         "c" : "0.00003583",
         "cs" : null,
         "etf" : false,
         "h" : "0.00003778",
         "i" : "1.00000000",
         "l" : "0.00003218",
         "o" : "0.00003266",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "4260.68833279",
         "s" : "VTHOBNB",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "124703574.00000000",
         "y" : 0
      },
      {
         "an" : "VeThor Token",
         "as" : 1621695968,
         "b" : "VTHO",
         "ba" : "",
         "c" : "0.020582",
         "cs" : null,
         "etf" : false,
         "h" : "0.021522",
         "i" : "1.00000000",
         "l" : "0.017027",
         "o" : "0.017652",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "30174507.792713",
         "s" : "VTHOUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "1621695968.000000",
         "y" : 0
      },
      {
         "an" : "TAEL",
         "as" : 3610725,
         "b" : "WABI",
         "ba" : "",
         "c" : "0.000990",
         "cs" : 99699999,
         "etf" : false,
         "h" : "0.001114",
         "i" : "1.00000000",
         "l" : "0.000864",
         "o" : "0.000898",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "3642.622747",
         "s" : "WABIBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "3610725.000000",
         "y" : 0
      },
      {
         "an" : "TAEL",
         "as" : 15997012,
         "b" : "WABI",
         "ba" : "",
         "c" : "0.00001038",
         "cs" : 99699999,
         "etf" : false,
         "h" : "0.00001120",
         "i" : "1.00000000",
         "l" : "0.00000864",
         "o" : "0.00000891",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "159.3614556",
         "s" : "WABIBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "15997012.00000000",
         "y" : 0
      },
      {
         "an" : "Wanchain",
         "as" : 376051.4,
         "b" : "WAN",
         "ba" : "",
         "c" : "0.002916",
         "cs" : 169028581,
         "etf" : false,
         "h" : "0.003162",
         "i" : "0.10000000",
         "l" : "0.002641",
         "o" : "0.002720",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1078.5063575",
         "s" : "WANBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "376051.400000",
         "y" : 0
      },
      {
         "an" : "Wanchain",
         "as" : 1692755,
         "b" : "WAN",
         "ba" : "",
         "c" : "0.00003025",
         "cs" : 169028581,
         "etf" : false,
         "h" : "0.00003102",
         "i" : "1.00000000",
         "l" : "0.00002634",
         "o" : "0.00002708",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "49.16427438",
         "s" : "WANBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "1692755.00000000",
         "y" : 0
      },
      {
         "an" : "Wanchain",
         "as" : 384558,
         "b" : "WAN",
         "ba" : "",
         "c" : "0.0006498",
         "cs" : 169028581,
         "etf" : false,
         "h" : "0.0006780",
         "i" : "0.01000000",
         "l" : "0.0005673",
         "o" : "0.0005830",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "242.30676821",
         "s" : "WANETH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "384558.0000000",
         "y" : 0
      },
      {
         "an" : "Wanchain",
         "as" : 5494378.73,
         "b" : "WAN",
         "ba" : "",
         "c" : "1.6649",
         "cs" : 169028581,
         "etf" : false,
         "h" : "1.7100",
         "i" : "0.01000000",
         "l" : "1.3960",
         "o" : "1.4602",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "8571176.214375",
         "s" : "WANUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "5494378.7300",
         "y" : 0
      },
      {
         "an" : "Waves",
         "as" : 195742.1,
         "b" : "WAVES",
         "ba" : "",
         "c" : "0.03529",
         "cs" : 104949316,
         "etf" : false,
         "h" : "0.04063",
         "i" : "0.10000000",
         "l" : "0.02901",
         "o" : "0.02901",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "7231.002759",
         "s" : "WAVESBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "195742.10000",
         "y" : 0
      },
      {
         "an" : "Waves",
         "as" : 3113870.06,
         "b" : "WAVES",
         "ba" : "",
         "c" : "0.0003677",
         "cs" : 104949316,
         "etf" : false,
         "h" : "0.0004032",
         "i" : "0.01000000",
         "l" : "0.0002893",
         "o" : "0.0002902",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "1110.63443545",
         "s" : "WAVESBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "3113870.0600000",
         "y" : 0
      },
      {
         "an" : "Waves",
         "as" : 723222.031,
         "b" : "WAVES",
         "ba" : "",
         "c" : "20.237",
         "cs" : 104949316,
         "etf" : false,
         "h" : "21.931",
         "i" : "0.00100000",
         "l" : "15.587",
         "o" : "15.603",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "13891820.978898",
         "s" : "WAVESBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "723222.031",
         "y" : 0
      },
      {
         "an" : "Waves",
         "as" : 181777.68,
         "b" : "WAVES",
         "ba" : "",
         "c" : "0.007905",
         "cs" : 104949316,
         "etf" : false,
         "h" : "0.008599",
         "i" : "0.01000000",
         "l" : "0.006241",
         "o" : "0.006241",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "1364.03933092",
         "s" : "WAVESETH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "181777.680000",
         "y" : 0
      },
      {
         "an" : "Waves",
         "as" : 13388728.565,
         "b" : "WAVES",
         "ba" : "",
         "c" : "20.189",
         "cs" : 104949316,
         "etf" : false,
         "h" : "21.639",
         "i" : "0.00100000",
         "l" : "15.558",
         "o" : "15.600",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "255810423.3022582",
         "s" : "WAVESUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "13388728.565",
         "y" : 0
      },
      {
         "an" : "Wrapped Bitcoin",
         "as" : 468.72224,
         "b" : "WBTC",
         "ba" : "",
         "c" : "1.00003",
         "cs" : null,
         "etf" : false,
         "h" : "1.00100",
         "i" : "0.00001000",
         "l" : "0.99974",
         "o" : "1.00013",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "468.8545196",
         "s" : "WBTCBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "468.72224",
         "y" : 0
      },
      {
         "an" : "Wrapped Bitcoin",
         "as" : 121.99905,
         "b" : "WBTC",
         "ba" : "",
         "c" : "21.489",
         "cs" : null,
         "etf" : false,
         "h" : "21.747",
         "i" : "0.00001000",
         "l" : "21.162",
         "o" : "21.433",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "2628.68576732",
         "s" : "WBTCETH",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "122.000",
         "y" : 0
      },
      {
         "an" : "WINK",
         "as" : 9564123657,
         "b" : "WIN",
         "ba" : "",
         "c" : "0.00000256",
         "cs" : 209734500000,
         "etf" : false,
         "h" : "0.00000272",
         "i" : "1.00000000",
         "l" : "0.00000218",
         "o" : "0.00000228",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "23552.33756656",
         "s" : "WINBNB",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "9564123657.00000000",
         "y" : 0
      },
      {
         "an" : "WINK",
         "as" : 746198673,
         "b" : "WIN",
         "ba" : "",
         "c" : "0.008021",
         "cs" : 209734500000,
         "etf" : false,
         "h" : "0.008440",
         "i" : "1.00000000",
         "l" : "0.006334",
         "o" : "0.006783",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BRL",
         "qa" : "",
         "qn" : "Brazilian Real",
         "qv" : "5384695.060179",
         "s" : "WINBRL",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "746198673.000000",
         "y" : 0
      },
      {
         "an" : "WINK",
         "as" : 3131045425,
         "b" : "WIN",
         "ba" : "",
         "c" : "0.001216",
         "cs" : 209734500000,
         "etf" : false,
         "h" : "0.001281",
         "i" : "1.00000000",
         "l" : "0.000953",
         "o" : "0.001019",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "3509645.718999",
         "s" : "WINEUR",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "3131045425.000000",
         "y" : 0
      },
      {
         "an" : "Wing Token",
         "as" : 6184.87,
         "b" : "WING",
         "ba" : "",
         "c" : "0.0794",
         "cs" : null,
         "etf" : false,
         "h" : "0.0849",
         "i" : "0.01000000",
         "l" : "0.0695",
         "o" : "0.0740",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "473.327886",
         "s" : "WINGBNB",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone"
         ],
         "ts" : "0.0001",
         "v" : "6184.8700",
         "y" : 0
      },
      {
         "an" : "Wing Token",
         "as" : 32131.703,
         "b" : "WING",
         "ba" : "",
         "c" : "0.000826",
         "cs" : null,
         "etf" : false,
         "h" : "0.000884",
         "i" : "0.00100000",
         "l" : "0.000691",
         "o" : "0.000736",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "25.14114265",
         "s" : "WINGBTC",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone"
         ],
         "ts" : "0.000001",
         "v" : "32131.703000",
         "y" : 0
      },
      {
         "an" : "Wing Token",
         "as" : 7501.047,
         "b" : "WING",
         "ba" : "",
         "c" : "45.500",
         "cs" : null,
         "etf" : false,
         "h" : "48.367",
         "i" : "0.00100000",
         "l" : "36.900",
         "o" : "39.577",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "315674.89485",
         "s" : "WINGBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone"
         ],
         "ts" : "0.001",
         "v" : "7501.047",
         "y" : 0
      },
      {
         "an" : "Wing Token",
         "as" : 126636.349,
         "b" : "WING",
         "ba" : "",
         "c" : "45.401",
         "cs" : null,
         "etf" : false,
         "h" : "48.325",
         "i" : "0.00100000",
         "l" : "36.752",
         "o" : "39.502",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "5308645.686353",
         "s" : "WINGUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone"
         ],
         "ts" : "0.001",
         "v" : "126636.349",
         "y" : 0
      },
      {
         "an" : "WINK",
         "as" : 1959222983.7,
         "b" : "WIN",
         "ba" : "",
         "c" : "0.01167",
         "cs" : 209734500000,
         "etf" : false,
         "h" : "0.01203",
         "i" : "0.10000000",
         "l" : "0.01019",
         "o" : "0.01067",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "TRX",
         "qa" : "",
         "qn" : "TRON",
         "qv" : "21575922.250922",
         "s" : "WINTRX",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "1959222983.70000",
         "y" : 0
      },
      {
         "an" : "WINK",
         "as" : 1314055358,
         "b" : "WIN",
         "ba" : "",
         "c" : "0.0014696",
         "cs" : 209734500000,
         "etf" : false,
         "h" : "0.0015447",
         "i" : "1.00000000",
         "l" : "0.0011538",
         "o" : "0.0012297",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDC",
         "qa" : "",
         "qn" : "USD Coin",
         "qv" : "1757478.4231441",
         "s" : "WINUSDC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "1314055358.0000000",
         "y" : 0
      },
      {
         "an" : "WINK",
         "as" : 389833663442,
         "b" : "WIN",
         "ba" : "",
         "c" : "0.0014667",
         "cs" : 209734500000,
         "etf" : false,
         "h" : "0.0015473",
         "i" : "1.00000000",
         "l" : "0.0011530",
         "o" : "0.0012268",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "522339816.3757335",
         "s" : "WINUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0000001",
         "v" : "389833663442.0000000",
         "y" : 0
      },
      {
         "an" : "Wrapped NXM",
         "as" : 9148.821,
         "b" : "WNXM",
         "ba" : "",
         "c" : "0.16059",
         "cs" : null,
         "etf" : false,
         "h" : "0.17847",
         "i" : "0.00100000",
         "l" : "0.15496",
         "o" : "0.16200",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1526.49114147",
         "s" : "WNXMBNB",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone"
         ],
         "ts" : "0.00001",
         "v" : "9148.82100",
         "y" : 0
      },
      {
         "an" : "Wrapped NXM",
         "as" : 41294.077,
         "b" : "WNXM",
         "ba" : "",
         "c" : "0.001682",
         "cs" : null,
         "etf" : false,
         "h" : "0.001766",
         "i" : "0.00100000",
         "l" : "0.001529",
         "o" : "0.001614",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "68.36954043",
         "s" : "WNXMBTC",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone"
         ],
         "ts" : "0.000001",
         "v" : "41294.077000",
         "y" : 0
      },
      {
         "an" : "Wrapped NXM",
         "as" : 242197.288,
         "b" : "WNXM",
         "ba" : "",
         "c" : "92.320",
         "cs" : null,
         "etf" : false,
         "h" : "96.593",
         "i" : "0.00100000",
         "l" : "80.469",
         "o" : "86.812",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "21635403.95403",
         "s" : "WNXMUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "innovation-zone"
         ],
         "ts" : "0.001",
         "v" : "242197.288",
         "y" : 0
      },
      {
         "an" : "WePower",
         "as" : 26772947,
         "b" : "WPR",
         "ba" : "",
         "c" : "0.00000090",
         "cs" : 688788615,
         "etf" : false,
         "h" : "0.00000095",
         "i" : "1.00000000",
         "l" : "0.00000085",
         "o" : "0.00000087",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "23.94540535",
         "s" : "WPRBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "26772947.00000000",
         "y" : 0
      },
      {
         "an" : "WazirX",
         "as" : 824910,
         "b" : "WRX",
         "ba" : "",
         "c" : "0.005289",
         "cs" : 186821429,
         "etf" : false,
         "h" : "0.005820",
         "i" : "1.00000000",
         "l" : "0.004786",
         "o" : "0.004987",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "4394.95879",
         "s" : "WRXBNB",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "824910.000000",
         "y" : 0
      },
      {
         "an" : "WazirX",
         "as" : 2961317,
         "b" : "WRX",
         "ba" : "",
         "c" : "0.00005529",
         "cs" : 186821429,
         "etf" : false,
         "h" : "0.00005860",
         "i" : "1.00000000",
         "l" : "0.00004770",
         "o" : "0.00005006",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "157.39749174",
         "s" : "WRXBTC",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00000001",
         "v" : "2961317.00000000",
         "y" : 0
      },
      {
         "an" : "WazirX",
         "as" : 1198893.47,
         "b" : "WRX",
         "ba" : "",
         "c" : "3.0367",
         "cs" : 186821429,
         "etf" : false,
         "h" : "3.2040",
         "i" : "0.01000000",
         "l" : "2.5335",
         "o" : "2.6980",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "3466941.185662",
         "s" : "WRXBUSD",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "1198893.4700",
         "y" : 0
      },
      {
         "an" : "WazirX",
         "as" : 321095.8,
         "b" : "WRX",
         "ba" : "",
         "c" : "2.521",
         "cs" : 186821429,
         "etf" : false,
         "h" : "2.663",
         "i" : "0.10000000",
         "l" : "2.100",
         "o" : "2.240",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "773824.9252",
         "s" : "WRXEUR",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "321095.800",
         "y" : 0
      },
      {
         "an" : "WazirX",
         "as" : 15291086.08,
         "b" : "WRX",
         "ba" : "",
         "c" : "3.0315",
         "cs" : 186821429,
         "etf" : false,
         "h" : "3.2000",
         "i" : "0.01000000",
         "l" : "2.5238",
         "o" : "2.6959",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "43753796.706477",
         "s" : "WRXUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.0001",
         "v" : "15291086.0800",
         "y" : 0
      },
      {
         "an" : "Walton",
         "as" : 75236.9,
         "b" : "WTC",
         "ba" : "",
         "c" : "0.002782",
         "cs" : 72133493,
         "etf" : false,
         "h" : "0.002947",
         "i" : "0.10000000",
         "l" : "0.002674",
         "o" : "0.002779",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "211.7492528",
         "s" : "WTCBNB",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "75236.900000",
         "y" : 0
      },
      {
         "an" : "Walton",
         "as" : 3490728.23,
         "b" : "WTC",
         "ba" : "",
         "c" : "1.6118",
         "cs" : 72133493,
         "etf" : false,
         "h" : "1.6509",
         "i" : "0.01000000",
         "l" : "1.3993",
         "o" : "1.4983",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "5267081.467384",
         "s" : "WTCUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "3490728.2300",
         "y" : 0
      },
      {
         "an" : "NEM",
         "as" : 35566837,
         "b" : "XEM",
         "ba" : "",
         "c" : "0.00000614",
         "cs" : 8999999999,
         "etf" : false,
         "h" : "0.00000642",
         "i" : "1.00000000",
         "l" : "0.00000556",
         "o" : "0.00000569",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "212.82173596",
         "s" : "XEMBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "35566837.00000000",
         "y" : 0
      },
      {
         "an" : "NEM",
         "as" : 2377589,
         "b" : "XEM",
         "ba" : "",
         "c" : "0.00013175",
         "cs" : 8999999999,
         "etf" : false,
         "h" : "0.00013399",
         "i" : "1.00000000",
         "l" : "0.00011983",
         "o" : "0.00012182",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "302.83393927",
         "s" : "XEMETH",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "2377589.00000000",
         "y" : 0
      },
      {
         "an" : "NEM",
         "as" : 134717499.6,
         "b" : "XEM",
         "ba" : "",
         "c" : "0.3371",
         "cs" : 8999999999,
         "etf" : false,
         "h" : "0.3450",
         "i" : "0.01000000",
         "l" : "0.2942",
         "o" : "0.3060",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "43176073.993865",
         "s" : "XEMUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "134717499.6000",
         "y" : 0
      },
      {
         "an" : "Stellar Lumens",
         "as" : 3994364,
         "b" : "XLM",
         "ba" : "",
         "c" : "0.000897",
         "cs" : 22917382426,
         "etf" : false,
         "h" : "0.000951",
         "i" : "1.00000000",
         "l" : "0.000854",
         "o" : "0.000875",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "3604.472311",
         "s" : "XLMBNB",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "3994364.000000",
         "y" : 0
      },
      {
         "an" : "Stellar Lumens",
         "as" : 47440040,
         "b" : "XLM",
         "ba" : "",
         "c" : "0.00000936",
         "cs" : 22917382426,
         "etf" : false,
         "h" : "0.00000953",
         "i" : "1.00000000",
         "l" : "0.00000855",
         "o" : "0.00000879",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "427.7409228",
         "s" : "XLMBTC",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "47440040.00000000",
         "y" : 0
      },
      {
         "an" : "Stellar Lumens",
         "as" : 15678508.9,
         "b" : "XLM",
         "ba" : "",
         "c" : "0.51464",
         "cs" : 22917382426,
         "etf" : false,
         "h" : "0.52320",
         "i" : "0.10000000",
         "l" : "0.45367",
         "o" : "0.47304",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "7644699.732438",
         "s" : "XLMBUSD",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "15678508.90000",
         "y" : 0
      },
      {
         "an" : "XLMDOWN",
         "as" : 203898066.38,
         "b" : "XLMDOWN",
         "ba" : "",
         "c" : "0.002584",
         "cs" : null,
         "etf" : true,
         "h" : "0.003658",
         "i" : "0.01000000",
         "l" : "0.002478",
         "o" : "0.003355",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "606385.84170712",
         "s" : "XLMDOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "203898066.380000",
         "y" : 0
      },
      {
         "an" : "Stellar Lumens",
         "as" : 5600356,
         "b" : "XLM",
         "ba" : "",
         "c" : "0.00020097",
         "cs" : 22917382426,
         "etf" : false,
         "h" : "0.00020471",
         "i" : "1.00000000",
         "l" : "0.00018330",
         "o" : "0.00018866",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "1086.87254456",
         "s" : "XLMETH",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "5600356.00000000",
         "y" : 0
      },
      {
         "an" : "Stellar Lumens",
         "as" : 9599923.8,
         "b" : "XLM",
         "ba" : "",
         "c" : "0.42615",
         "cs" : 22917382426,
         "etf" : false,
         "h" : "0.43349",
         "i" : "0.10000000",
         "l" : "0.37618",
         "o" : "0.39219",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "3909114.313789",
         "s" : "XLMEUR",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "9599923.80000",
         "y" : 0
      },
      {
         "an" : "Stellar Lumens",
         "as" : 6037854.48,
         "b" : "XLM",
         "ba" : "",
         "c" : "4.189",
         "cs" : 22917382426,
         "etf" : false,
         "h" : "4.254",
         "i" : "0.01000000",
         "l" : "3.692",
         "o" : "3.859",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "24091349.39925",
         "s" : "XLMTRY",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "6037854.480",
         "y" : 0
      },
      {
         "an" : "XLMUP",
         "as" : 695494.31,
         "b" : "XLMUP",
         "ba" : "",
         "c" : "8.709",
         "cs" : null,
         "etf" : true,
         "h" : "8.959",
         "i" : "0.01000000",
         "l" : "6.707",
         "o" : "7.289",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "5427856.36587",
         "s" : "XLMUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "695494.310",
         "y" : 0
      },
      {
         "an" : "Stellar Lumens",
         "as" : 268702745.3,
         "b" : "XLM",
         "ba" : "",
         "c" : "0.51461",
         "cs" : 22917382426,
         "etf" : false,
         "h" : "0.52246",
         "i" : "0.10000000",
         "l" : "0.45311",
         "o" : "0.47196",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "130594919.615451",
         "s" : "XLMUSDT",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "268702745.30000",
         "y" : 0
      },
      {
         "an" : "Monero",
         "as" : 3085.528,
         "b" : "XMR",
         "ba" : "",
         "c" : "0.695",
         "cs" : 17896151,
         "etf" : false,
         "h" : "0.772",
         "i" : "0.00100000",
         "l" : "0.693",
         "o" : "0.702",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "2266.358258",
         "s" : "XMRBNB",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "3085.528",
         "y" : 0
      },
      {
         "an" : "Monero",
         "as" : 89805.875,
         "b" : "XMR",
         "ba" : "",
         "c" : "0.007256",
         "cs" : 17896151,
         "etf" : false,
         "h" : "0.007614",
         "i" : "0.00100000",
         "l" : "0.006905",
         "o" : "0.007044",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "658.03576618",
         "s" : "XMRBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "89805.875000",
         "y" : 0
      },
      {
         "an" : "Monero",
         "as" : 6745.37298,
         "b" : "XMR",
         "ba" : "",
         "c" : "399.59",
         "cs" : 17896151,
         "etf" : false,
         "h" : "413.38",
         "i" : "0.00001000",
         "l" : "366.45",
         "o" : "379.50",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2665996.0883481",
         "s" : "XMRBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "6745.38",
         "y" : 0
      },
      {
         "an" : "Monero",
         "as" : 5141.962,
         "b" : "XMR",
         "ba" : "",
         "c" : "0.15584",
         "cs" : 17896151,
         "etf" : false,
         "h" : "0.16333",
         "i" : "0.00100000",
         "l" : "0.14858",
         "o" : "0.15111",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "807.58838298",
         "s" : "XMRETH",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "5141.96200",
         "y" : 0
      },
      {
         "an" : "Monero",
         "as" : 153299.10532,
         "b" : "XMR",
         "ba" : "",
         "c" : "398.70",
         "cs" : 17896151,
         "etf" : false,
         "h" : "413.00",
         "i" : "0.00001000",
         "l" : "365.79",
         "o" : "378.68",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "60657126.4335705",
         "s" : "XMRUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "153299.11",
         "y" : 0
      },
      {
         "an" : "Ripple",
         "as" : 6162737.93,
         "b" : "XRP",
         "ba" : "",
         "c" : "1.8299",
         "cs" : 45404028640,
         "etf" : false,
         "h" : "1.8910",
         "i" : "0.01000000",
         "l" : "1.5503",
         "o" : "1.6281",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "AUD",
         "qa" : "",
         "qn" : "Australian Dollar",
         "qv" : "10779380.688146",
         "s" : "XRPAUD",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "6162737.9300",
         "y" : 0
      },
      {
         "an" : "Ripple",
         "as" : 29324859.6,
         "b" : "XRP",
         "ba" : "",
         "c" : "0.002468",
         "cs" : 45404028640,
         "etf" : false,
         "h" : "0.002737",
         "i" : "0.10000000",
         "l" : "0.002296",
         "o" : "0.002340",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "73745.9308868",
         "s" : "XRPBNB",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "29324859.600000",
         "y" : 0
      },
      {
         "an" : "Ripple",
         "as" : 6840530.73,
         "b" : "XRP",
         "ba" : "",
         "c" : "7.726",
         "cs" : 45404028640,
         "etf" : false,
         "h" : "8.000",
         "i" : "0.01000000",
         "l" : "6.724",
         "o" : "6.965",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BRL",
         "qa" : "",
         "qn" : "Brazilian Real",
         "qv" : "50568844.9343",
         "s" : "XRPBRL",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "6840530.730",
         "y" : 0
      },
      {
         "an" : "Ripple",
         "as" : 371223891,
         "b" : "XRP",
         "ba" : "",
         "c" : "0.00002573",
         "cs" : 45404028640,
         "etf" : false,
         "h" : "0.00002716",
         "i" : "1.00000000",
         "l" : "0.00002285",
         "o" : "0.00002342",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "9288.93381715",
         "s" : "XRPBTC",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "371223891.00000000",
         "y" : 0
      },
      {
         "an" : "Ripple",
         "as" : 143201637.02,
         "b" : "XRP",
         "ba" : "",
         "c" : "1.4147",
         "cs" : 45404028640,
         "etf" : false,
         "h" : "1.4638",
         "i" : "0.01000000",
         "l" : "1.2200",
         "o" : "1.2617",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "192597602.584764",
         "s" : "XRPBUSD",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "143201637.0200",
         "y" : 0
      },
      {
         "an" : "XRPDOWN",
         "as" : 722707678.43,
         "b" : "XRPDOWN",
         "ba" : "",
         "c" : "0.058290",
         "cs" : null,
         "etf" : true,
         "h" : "0.083200",
         "i" : "0.01000000",
         "l" : "0.052740",
         "o" : "0.076408",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "47289148.30536037",
         "s" : "XRPDOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "722707678.430000",
         "y" : 0
      },
      {
         "an" : "Ripple",
         "as" : 27567638,
         "b" : "XRP",
         "ba" : "",
         "c" : "0.00055259",
         "cs" : 45404028640,
         "etf" : false,
         "h" : "0.00057875",
         "i" : "1.00000000",
         "l" : "0.00049288",
         "o" : "0.00050164",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "14689.17305214",
         "s" : "XRPETH",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "27567638.00000000",
         "y" : 0
      },
      {
         "an" : "Ripple",
         "as" : 57773043.1,
         "b" : "XRP",
         "ba" : "",
         "c" : "1.1726",
         "cs" : 45404028640,
         "etf" : false,
         "h" : "1.2500",
         "i" : "0.01000000",
         "l" : "1.0097",
         "o" : "1.0456",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "64321072.406919",
         "s" : "XRPEUR",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "57773043.1000",
         "y" : 0
      },
      {
         "an" : "Ripple",
         "as" : 30591866.1,
         "b" : "XRP",
         "ba" : "",
         "c" : "1.02000",
         "cs" : 45404028640,
         "etf" : false,
         "h" : "1.05446",
         "i" : "0.10000000",
         "l" : "0.88089",
         "o" : "0.90990",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "GBP",
         "qa" : "",
         "qn" : "Pound Sterling",
         "qv" : "29492334.077413",
         "s" : "XRPGBP",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "30591866.10000",
         "y" : 0
      },
      {
         "an" : "Ripple",
         "as" : 394314.55,
         "b" : "XRP",
         "ba" : "",
         "c" : "1.4166",
         "cs" : 45404028640,
         "etf" : false,
         "h" : "1.4623",
         "i" : "0.01000000",
         "l" : "1.2216",
         "o" : "1.2603",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "PAX",
         "qa" : "",
         "qn" : "Paxos Standard",
         "qv" : "533715.637392",
         "s" : "XRPPAX",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "394314.5500",
         "y" : 0
      },
      {
         "an" : "Ripple",
         "as" : 2002918.8,
         "b" : "XRP",
         "ba" : "",
         "c" : "106.978",
         "cs" : 45404028640,
         "etf" : false,
         "h" : "110.487",
         "i" : "0.10000000",
         "l" : "92.154",
         "o" : "95.107",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "RUB",
         "qa" : "",
         "qn" : "Russian Ruble",
         "qv" : "203545773.282",
         "s" : "XRPRUB",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "2002918.800",
         "y" : 0
      },
      {
         "an" : "Ripple",
         "as" : 73347040.84,
         "b" : "XRP",
         "ba" : "",
         "c" : "11.516",
         "cs" : 45404028640,
         "etf" : false,
         "h" : "11.897",
         "i" : "0.01000000",
         "l" : "9.950",
         "o" : "10.288",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TRY",
         "qa" : "",
         "qn" : "Turkish Lira",
         "qv" : "799243425.7941",
         "s" : "XRPTRY",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "73347040.840",
         "y" : 0
      },
      {
         "an" : "Ripple",
         "as" : 3283503.41,
         "b" : "XRP",
         "ba" : "",
         "c" : "1.4125",
         "cs" : 45404028640,
         "etf" : false,
         "h" : "1.4624",
         "i" : "0.01000000",
         "l" : "1.2121",
         "o" : "1.2595",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "TUSD",
         "qa" : "",
         "qn" : "TrueUSD",
         "qv" : "4395097.61263",
         "s" : "XRPTUSD",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "3283503.4100",
         "y" : 0
      },
      {
         "an" : "XRPUP",
         "as" : 4936009.01,
         "b" : "XRPUP",
         "ba" : "",
         "c" : "18.077",
         "cs" : null,
         "etf" : true,
         "h" : "19.182",
         "i" : "0.01000000",
         "l" : "14.366",
         "o" : "15.066",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "83901693.91162",
         "s" : "XRPUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "4936009.010",
         "y" : 0
      },
      {
         "an" : "Ripple",
         "as" : 15137711.15,
         "b" : "XRP",
         "ba" : "",
         "c" : "1.4175",
         "cs" : 45404028640,
         "etf" : false,
         "h" : "1.4700",
         "i" : "0.01000000",
         "l" : "1.2212",
         "o" : "1.2547",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDC",
         "qa" : "",
         "qn" : "USD Coin",
         "qv" : "20263971.686136",
         "s" : "XRPUSDC",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "15137711.1500",
         "y" : 0
      },
      {
         "an" : "Ripple",
         "as" : 1921649905.85,
         "b" : "XRP",
         "ba" : "",
         "c" : "1.4130",
         "cs" : 45404028640,
         "etf" : false,
         "h" : "1.4620",
         "i" : "0.01000000",
         "l" : "1.2176",
         "o" : "1.2598",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "2585980688.314744",
         "s" : "XRPUSDT",
         "st" : "TRADING",
         "tags" : [
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "1921649905.8500",
         "y" : 0
      },
      {
         "an" : "Tezos",
         "as" : 70725.5,
         "b" : "XTZ",
         "ba" : "",
         "c" : "0.00947",
         "cs" : 607489041,
         "etf" : false,
         "h" : "0.01011",
         "i" : "0.10000000",
         "l" : "0.00938",
         "o" : "0.00943",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "688.557217",
         "s" : "XTZBNB",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "70725.50000",
         "y" : 0
      },
      {
         "an" : "Tezos",
         "as" : 1781732.62,
         "b" : "XTZ",
         "ba" : "",
         "c" : "0.0000986",
         "cs" : 607489041,
         "etf" : false,
         "h" : "0.0001001",
         "i" : "0.01000000",
         "l" : "0.0000938",
         "o" : "0.0000946",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "173.20621309",
         "s" : "XTZBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "1781732.6200000",
         "y" : 0
      },
      {
         "an" : "Tezos",
         "as" : 444076.43,
         "b" : "XTZ",
         "ba" : "",
         "c" : "5.4250",
         "cs" : 607489041,
         "etf" : false,
         "h" : "5.4989",
         "i" : "0.01000000",
         "l" : "5.0032",
         "o" : "5.0925",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2329681.762918",
         "s" : "XTZBUSD",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "444076.4300",
         "y" : 0
      },
      {
         "an" : "XTZDOWN",
         "as" : 64706301.55,
         "b" : "XTZDOWN",
         "ba" : null,
         "c" : "0.01973",
         "cs" : null,
         "etf" : true,
         "h" : "0.02624",
         "i" : "0.01000000",
         "l" : "0.01912",
         "o" : "0.02466",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "1404796.1846468",
         "s" : "XTZDOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.00001",
         "v" : "64706301.55000",
         "y" : 0
      },
      {
         "an" : "XTZUP",
         "as" : 7299934.09,
         "b" : "XTZUP",
         "ba" : null,
         "c" : "0.663",
         "cs" : null,
         "etf" : true,
         "h" : "0.684",
         "i" : "0.01000000",
         "l" : "0.560",
         "o" : "0.582",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "4519852.3848",
         "s" : "XTZUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "7299934.090",
         "y" : 0
      },
      {
         "an" : "Tezos",
         "as" : 8075209.03,
         "b" : "XTZ",
         "ba" : "",
         "c" : "5.4163",
         "cs" : 607489041,
         "etf" : false,
         "h" : "5.4899",
         "i" : "0.01000000",
         "l" : "4.9696",
         "o" : "5.0909",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "42400109.566323",
         "s" : "XTZUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "8075209.0300",
         "y" : 0
      },
      {
         "an" : "Verge",
         "as" : 495483388,
         "b" : "XVG",
         "ba" : "",
         "c" : "0.00000096",
         "cs" : 16447987349,
         "etf" : false,
         "h" : "0.00000098",
         "i" : "1.00000000",
         "l" : "0.00000085",
         "o" : "0.00000091",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "452.89677761",
         "s" : "XVGBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "495483388.00000000",
         "y" : 0
      },
      {
         "an" : "Verge",
         "as" : 274916297,
         "b" : "XVG",
         "ba" : "",
         "c" : "0.05228",
         "cs" : 16447987349,
         "etf" : false,
         "h" : "0.05380",
         "i" : "0.10000000",
         "l" : "0.04444",
         "o" : "0.04922",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "13501879.69892",
         "s" : "XVGBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "274916297.00000",
         "y" : 0
      },
      {
         "an" : "Venus",
         "as" : 52154.6,
         "b" : "XVS",
         "ba" : "",
         "c" : "0.1284",
         "cs" : null,
         "etf" : false,
         "h" : "0.1387",
         "i" : "0.01000000",
         "l" : "0.1250",
         "o" : "0.1306",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "6838.445587",
         "s" : "XVSBNB",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "BSC"
         ],
         "ts" : "0.0001",
         "v" : "52154.6000",
         "y" : 0
      },
      {
         "an" : "Venus",
         "as" : 74363.31,
         "b" : "XVS",
         "ba" : "",
         "c" : "0.0013371",
         "cs" : null,
         "etf" : false,
         "h" : "0.0013808",
         "i" : "0.01000000",
         "l" : "0.0012523",
         "o" : "0.0013094",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "97.22250083",
         "s" : "XVSBTC",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "BSC"
         ],
         "ts" : "0.0000001",
         "v" : "74363.3100000",
         "y" : 0
      },
      {
         "an" : "Venus",
         "as" : 94735.168,
         "b" : "XVS",
         "ba" : "",
         "c" : "73.522",
         "cs" : null,
         "etf" : false,
         "h" : "75.839",
         "i" : "0.00100000",
         "l" : "68.083",
         "o" : "70.420",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "6719036.232242",
         "s" : "XVSBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "94735.168",
         "y" : 0
      },
      {
         "an" : "Venus",
         "as" : 709452.507,
         "b" : "XVS",
         "ba" : "",
         "c" : "73.500",
         "cs" : null,
         "etf" : false,
         "h" : "76.000",
         "i" : "0.00100000",
         "l" : "68.000",
         "o" : "70.443",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "50168355.737872",
         "s" : "XVSUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "BSC"
         ],
         "ts" : "0.001",
         "v" : "709452.507",
         "y" : 0
      },
      {
         "an" : "yearn.finance",
         "as" : 3.7453,
         "b" : "YFI",
         "ba" : "",
         "c" : "80.13",
         "cs" : null,
         "etf" : false,
         "h" : "85.65",
         "i" : "0.00010000",
         "l" : "79.96",
         "o" : "81.17",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "309.070149",
         "s" : "YFIBNB",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.01",
         "v" : "3.75",
         "y" : 0
      },
      {
         "an" : "yearn.finance",
         "as" : 126.8177,
         "b" : "YFI",
         "ba" : "",
         "c" : "0.83525",
         "cs" : null,
         "etf" : false,
         "h" : "0.84862",
         "i" : "0.00010000",
         "l" : "0.80266",
         "o" : "0.81228",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "105.25602958",
         "s" : "YFIBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00001",
         "v" : "126.81770",
         "y" : 0
      },
      {
         "an" : "yearn.finance",
         "as" : 54.77029,
         "b" : "YFI",
         "ba" : "",
         "c" : "45917.39",
         "cs" : null,
         "etf" : false,
         "h" : "46375.00",
         "i" : "0.00000100",
         "l" : "42576.92",
         "o" : "43723.82",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2430315.75193492",
         "s" : "YFIBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.01",
         "v" : "54.78",
         "y" : 0
      },
      {
         "an" : "YFIDOWN",
         "as" : 2334768835.64,
         "b" : "YFIDOWN",
         "ba" : "",
         "c" : "0.000967",
         "cs" : null,
         "etf" : true,
         "h" : "0.001215",
         "i" : "0.01000000",
         "l" : "0.000935",
         "o" : "0.001133",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "2479319.29060902",
         "s" : "YFIDOWNUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.000001",
         "v" : "2334768835.640000",
         "y" : 0
      },
      {
         "an" : "yearn.finance",
         "as" : 30.843414,
         "b" : "YFI",
         "ba" : "",
         "c" : "38053.04",
         "cs" : null,
         "etf" : false,
         "h" : "38401.89",
         "i" : "0.00000100",
         "l" : "35275.14",
         "o" : "36389.76",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "EUR",
         "qa" : "",
         "qn" : "Euro",
         "qv" : "1139709.84968089",
         "s" : "YFIEUR",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.01",
         "v" : "30.85",
         "y" : 0
      },
      {
         "an" : "DFI.Money",
         "as" : 24.7096,
         "b" : "YFII",
         "ba" : "",
         "c" : "4.340",
         "cs" : null,
         "etf" : false,
         "h" : "4.640",
         "i" : "0.00010000",
         "l" : "4.340",
         "o" : "4.476",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "111.8443051",
         "s" : "YFIIBNB",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.001",
         "v" : "24.710",
         "y" : 0
      },
      {
         "an" : "DFI.Money",
         "as" : 524.0563,
         "b" : "YFII",
         "ba" : "",
         "c" : "0.04530",
         "cs" : null,
         "etf" : false,
         "h" : "0.04640",
         "i" : "0.00010000",
         "l" : "0.04393",
         "o" : "0.04464",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "23.63401627",
         "s" : "YFIIBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00001",
         "v" : "524.05630",
         "y" : 0
      },
      {
         "an" : "DFI.Money",
         "as" : 137.912939,
         "b" : "YFII",
         "ba" : "",
         "c" : "2493.75",
         "cs" : null,
         "etf" : false,
         "h" : "2544.21",
         "i" : "0.00000100",
         "l" : "2317.82",
         "o" : "2395.32",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "335199.8367952",
         "s" : "YFIIBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.01",
         "v" : "137.92",
         "y" : 0
      },
      {
         "an" : "DFI.Money",
         "as" : 3115.8869,
         "b" : "YFII",
         "ba" : "",
         "c" : "2488.37",
         "cs" : null,
         "etf" : false,
         "h" : "2549.85",
         "i" : "0.00000100",
         "l" : "2300.00",
         "o" : "2399.49",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "7580405.1660209",
         "s" : "YFIIUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.01",
         "v" : "3115.89",
         "y" : 0
      },
      {
         "an" : "YFIUP",
         "as" : 444977.03,
         "b" : "YFIUP",
         "ba" : "",
         "c" : "5.818",
         "cs" : null,
         "etf" : true,
         "h" : "5.939",
         "i" : "0.01000000",
         "l" : "5.079",
         "o" : "5.340",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "2484826.1623",
         "s" : "YFIUPUSDT",
         "st" : "TRADING",
         "tags" : [],
         "ts" : "0.001",
         "v" : "444977.030",
         "y" : 0
      },
      {
         "an" : "yearn.finance",
         "as" : 678.517527,
         "b" : "YFI",
         "ba" : "",
         "c" : "45860.86",
         "cs" : null,
         "etf" : false,
         "h" : "46327.68",
         "i" : "0.00000100",
         "l" : "42510.65",
         "o" : "43718.04",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "30182279.33559996",
         "s" : "YFIUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.01",
         "v" : "678.52",
         "y" : 0
      },
      {
         "an" : "YOYOW",
         "as" : 18318548,
         "b" : "YOYO",
         "ba" : "",
         "c" : "0.00000066",
         "cs" : 495718220,
         "etf" : false,
         "h" : "0.00000067",
         "i" : "1.00000000",
         "l" : "0.00000060",
         "o" : "0.00000063",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "11.52354296",
         "s" : "YOYOBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "18318548.00000000",
         "y" : 0
      },
      {
         "an" : "Zcash",
         "as" : 1413.085,
         "b" : "ZEC",
         "ba" : "",
         "c" : "0.3988",
         "cs" : 5196519,
         "etf" : false,
         "h" : "0.4322",
         "i" : "0.00100000",
         "l" : "0.3988",
         "o" : "0.4111",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "588.5552982",
         "s" : "ZECBNB",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "1413.0850",
         "y" : 0
      },
      {
         "an" : "Zcash",
         "as" : 28152.161,
         "b" : "ZEC",
         "ba" : "",
         "c" : "0.004159",
         "cs" : 5196519,
         "etf" : false,
         "h" : "0.004281",
         "i" : "0.00100000",
         "l" : "0.004071",
         "o" : "0.004138",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "117.6507449",
         "s" : "ZECBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.000001",
         "v" : "28152.161000",
         "y" : 0
      },
      {
         "an" : "Zcash",
         "as" : 7554.57658,
         "b" : "ZEC",
         "ba" : "",
         "c" : "228.94",
         "cs" : 5196519,
         "etf" : false,
         "h" : "231.77",
         "i" : "0.00001000",
         "l" : "216.87",
         "o" : "222.90",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1700264.807709",
         "s" : "ZECBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "7554.58",
         "y" : 0
      },
      {
         "an" : "Zcash",
         "as" : 3240.337,
         "b" : "ZEC",
         "ba" : "",
         "c" : "0.08949",
         "cs" : 5196519,
         "etf" : false,
         "h" : "0.09139",
         "i" : "0.00100000",
         "l" : "0.08776",
         "o" : "0.08857",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "290.01830258",
         "s" : "ZECETH",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "3240.33700",
         "y" : 0
      },
      {
         "an" : "Zcash",
         "as" : 592.97082,
         "b" : "ZEC",
         "ba" : "",
         "c" : "228.50",
         "cs" : 5196519,
         "etf" : false,
         "h" : "231.92",
         "i" : "0.00001000",
         "l" : "216.93",
         "o" : "222.45",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDC",
         "qa" : "",
         "qn" : "USD Coin",
         "qv" : "132326.3277268",
         "s" : "ZECUSDC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "592.98",
         "y" : 0
      },
      {
         "an" : "Zcash",
         "as" : 224108.97052,
         "b" : "ZEC",
         "ba" : "",
         "c" : "228.61",
         "cs" : 5196519,
         "etf" : false,
         "h" : "231.58",
         "i" : "0.00001000",
         "l" : "216.65",
         "o" : "222.87",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "50473508.5819038",
         "s" : "ZECUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "224108.98",
         "y" : 0
      },
      {
         "an" : "Horizen",
         "as" : 4750.54,
         "b" : "ZEN",
         "ba" : "",
         "c" : "0.2252",
         "cs" : 11017706,
         "etf" : false,
         "h" : "0.2620",
         "i" : "0.01000000",
         "l" : "0.2240",
         "o" : "0.2351",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "1154.937978",
         "s" : "ZENBNB",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.0001",
         "v" : "4750.5400",
         "y" : 0
      },
      {
         "an" : "Horizen",
         "as" : 41647.369,
         "b" : "ZEN",
         "ba" : "",
         "c" : "0.0023465",
         "cs" : 11017706,
         "etf" : false,
         "h" : "0.0025800",
         "i" : "0.01000000",
         "l" : "0.0022800",
         "o" : "0.0023540",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "101.21098071",
         "s" : "ZENBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "41647.3690000",
         "y" : 0
      },
      {
         "an" : "Horizen",
         "as" : 6257.744,
         "b" : "ZEN",
         "ba" : "",
         "c" : "0.05043",
         "cs" : 11017706,
         "etf" : false,
         "h" : "0.05529",
         "i" : "0.00100000",
         "l" : "0.04924",
         "o" : "0.05051",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "326.40713766",
         "s" : "ZENETH",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "6257.74400",
         "y" : 0
      },
      {
         "an" : "Horizen",
         "as" : 279923.4099,
         "b" : "ZEN",
         "ba" : "",
         "c" : "128.91",
         "cs" : 11017706,
         "etf" : false,
         "h" : "138.98",
         "i" : "0.00010000",
         "l" : "120.94",
         "o" : "126.46",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "36678154.325054",
         "s" : "ZENUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "279923.41",
         "y" : 0
      },
      {
         "an" : "Zilliqa",
         "as" : 1659791,
         "b" : "ZIL",
         "ba" : "1",
         "c" : "2931.42",
         "cs" : 11216919823,
         "etf" : false,
         "h" : "2931.42",
         "i" : "1.00000000",
         "l" : "2353.37",
         "o" : "2406.71",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BIDR",
         "qa" : "",
         "qn" : "BIDR",
         "qv" : "4314426178.94",
         "s" : "ZILBIDR",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "1659791.00",
         "y" : 0
      },
      {
         "an" : "Zilliqa",
         "as" : 8165840,
         "b" : "ZIL",
         "ba" : "1",
         "c" : "0.0003462",
         "cs" : 11216919823,
         "etf" : false,
         "h" : "0.0003462",
         "i" : "1.00000000",
         "l" : "0.0002989",
         "o" : "0.0003011",
         "pm" : "BNB",
         "pn" : "BNB",
         "pom" : false,
         "pomt" : null,
         "q" : "BNB",
         "qa" : "",
         "qn" : "BNB",
         "qv" : "2624.4799725",
         "s" : "ZILBNB",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "8165840.0000000",
         "y" : 0
      },
      {
         "an" : "Zilliqa",
         "as" : 86645516,
         "b" : "ZIL",
         "ba" : "1",
         "c" : "0.00000360",
         "cs" : 11216919823,
         "etf" : false,
         "h" : "0.00000362",
         "i" : "1.00000000",
         "l" : "0.00000297",
         "o" : "0.00000302",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "281.27013078",
         "s" : "ZILBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "86645516.00000000",
         "y" : 0
      },
      {
         "an" : "Zilliqa",
         "as" : 20481798.6,
         "b" : "ZIL",
         "ba" : "1",
         "c" : "0.19829",
         "cs" : 11216919823,
         "etf" : false,
         "h" : "0.19876",
         "i" : "0.10000000",
         "l" : "0.15777",
         "o" : "0.16162",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "3573506.180776",
         "s" : "ZILBUSD",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "20481798.60000",
         "y" : 0
      },
      {
         "an" : "Zilliqa",
         "as" : 8589544,
         "b" : "ZIL",
         "ba" : "1",
         "c" : "0.00007734",
         "cs" : 11216919823,
         "etf" : false,
         "h" : "0.00007757",
         "i" : "1.00000000",
         "l" : "0.00006402",
         "o" : "0.00006448",
         "pm" : "ALTS",
         "pn" : "ALTS",
         "pom" : false,
         "pomt" : null,
         "q" : "ETH",
         "qa" : "Î",
         "qn" : "Ethereum",
         "qv" : "595.42729409",
         "s" : "ZILETH",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "8589544.00000000",
         "y" : 0
      },
      {
         "an" : "Zilliqa",
         "as" : 283989752.1,
         "b" : "ZIL",
         "ba" : "1",
         "c" : "0.19794",
         "cs" : 11216919823,
         "etf" : false,
         "h" : "0.19850",
         "i" : "0.10000000",
         "l" : "0.15747",
         "o" : "0.16193",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "49379057.32526",
         "s" : "ZILUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "283989752.10000",
         "y" : 0
      },
      {
         "an" : "0x",
         "as" : 701807.62,
         "b" : "ZRX",
         "ba" : "",
         "c" : "1.6289",
         "cs" : 781547659,
         "etf" : false,
         "h" : "1.6486",
         "i" : "0.01000000",
         "l" : "1.4604",
         "o" : "1.5137",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "1096825.753642",
         "s" : "ZRXBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "701807.6200",
         "y" : 0
      },
      {
         "an" : "0x",
         "as" : 7441026.66,
         "b" : "ZRX",
         "ba" : "",
         "c" : "1.6289",
         "cs" : 781547659,
         "etf" : false,
         "h" : "1.6480",
         "i" : "0.01000000",
         "l" : "1.4568",
         "o" : "1.5162",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "11588422.975567",
         "s" : "ZRXUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.0001",
         "v" : "7441026.6600",
         "y" : 0
      },
      {
         "an" : "BNB",
         "as" : 2650523.273,
         "b" : "BNB",
         "ba" : "",
         "c" : "573.51",
         "cs" : 152665937,
         "etf" : false,
         "h" : "576.00",
         "i" : "0.00010000",
         "l" : "515.00",
         "o" : "538.57",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "1436905887.0329919",
         "s" : "BNBUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "BSC"
         ],
         "ts" : "0.01",
         "v" : "2650523.28",
         "y" : 0
      },
      {
         "an" : "Ethereum",
         "as" : 732787.43442,
         "b" : "ETH",
         "ba" : "Î",
         "c" : "2558.30",
         "cs" : 115634925,
         "etf" : false,
         "h" : "2584.34",
         "i" : "0.00001000",
         "l" : "2430.00",
         "o" : "2503.02",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "1840651928.2473409",
         "s" : "ETHUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.01",
         "v" : "732787.44",
         "y" : 0
      },
      {
         "an" : "Ren",
         "as" : 17238891.5,
         "b" : "REN",
         "ba" : "",
         "c" : "0.87436",
         "cs" : 997163051,
         "etf" : false,
         "h" : "0.89083",
         "i" : "0.10000000",
         "l" : "0.81064",
         "o" : "0.83550",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "14740689.805296",
         "s" : "RENUSDT",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "defi"
         ],
         "ts" : "0.00001",
         "v" : "17238891.50000",
         "y" : 0
      },
      {
         "an" : "NKN",
         "as" : 69912059.8,
         "b" : "NKN",
         "ba" : "",
         "c" : "0.69400",
         "cs" : 583666666,
         "etf" : false,
         "h" : "0.72000",
         "i" : "0.10000000",
         "l" : "0.61737",
         "o" : "0.63105",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "46759734.470286",
         "s" : "NKNUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "69912059.80000",
         "y" : 0
      },
      {
         "an" : "BitShares",
         "as" : 126151431.9,
         "b" : "BTS",
         "ba" : "",
         "c" : "0.10911",
         "cs" : 2994880000,
         "etf" : false,
         "h" : "0.11450",
         "i" : "0.10000000",
         "l" : "0.09432",
         "o" : "0.09957",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "13278064.82981",
         "s" : "BTSUSDT",
         "st" : "TRADING",
         "tags" : [
            "defi",
            "pos",
            "mining-zone"
         ],
         "ts" : "0.00001",
         "v" : "126151431.90000",
         "y" : 0
      },
      {
         "an" : "DigiByte",
         "as" : 70839589,
         "b" : "DGB",
         "ba" : "",
         "c" : "0.00000245",
         "cs" : 10126279276,
         "etf" : false,
         "h" : "0.00000254",
         "i" : "1.00000000",
         "l" : "0.00000225",
         "o" : "0.00000228",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "167.90736111",
         "s" : "DGBBTC",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.00000001",
         "v" : "70839589.00000000",
         "y" : 0
      },
      {
         "an" : "Balancer",
         "as" : 29974.15,
         "b" : "BAL",
         "ba" : "",
         "c" : "0.0010024",
         "cs" : null,
         "etf" : false,
         "h" : "0.0010187",
         "i" : "0.01000000",
         "l" : "0.0009645",
         "o" : "0.0009774",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "29.67999427",
         "s" : "BALBTC",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.0000001",
         "v" : "29974.1500000",
         "y" : 0
      },
      {
         "an" : "JUST",
         "as" : 3304250.3,
         "b" : "JST",
         "ba" : "",
         "c" : "0.12696",
         "cs" : null,
         "etf" : false,
         "h" : "0.12974",
         "i" : "0.10000000",
         "l" : "0.11399",
         "o" : "0.11767",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "402584.724001",
         "s" : "JSTBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00001",
         "v" : "3304250.30000",
         "y" : 0
      },
      {
         "an" : "Aragon",
         "as" : 245158.83,
         "b" : "ANT",
         "ba" : "",
         "c" : "0.0001632",
         "cs" : 39609522,
         "etf" : false,
         "h" : "0.0001680",
         "i" : "0.01000000",
         "l" : "0.0001538",
         "o" : "0.0001561",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "39.2667753",
         "s" : "ANTBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone"
         ],
         "ts" : "0.0000001",
         "v" : "245158.8300000",
         "y" : 0
      },
      {
         "an" : "Reserve Rights",
         "as" : 34147162.8,
         "b" : "RSR",
         "ba" : "",
         "c" : "0.08627",
         "cs" : null,
         "etf" : false,
         "h" : "0.08903",
         "i" : "0.10000000",
         "l" : "0.07845",
         "o" : "0.08186",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "BUSD",
         "qa" : "",
         "qn" : "BUSD",
         "qv" : "2857847.125694",
         "s" : "RSRBUSD",
         "st" : "TRADING",
         "tags" : [
            "defi"
         ],
         "ts" : "0.00001",
         "v" : "34147162.80000",
         "y" : 0
      },
      {
         "an" : "Kusama",
         "as" : 28647.035,
         "b" : "KSM",
         "ba" : "",
         "c" : "0.007350",
         "cs" : null,
         "etf" : false,
         "h" : "0.007667",
         "i" : "0.00100000",
         "l" : "0.006779",
         "o" : "0.006898",
         "pm" : "BTC",
         "pn" : "BTC",
         "pom" : false,
         "pomt" : null,
         "q" : "BTC",
         "qa" : "à¸¿",
         "qn" : "Bitcoin",
         "qv" : "205.32404895",
         "s" : "KSMBTC",
         "st" : "TRADING",
         "tags" : [
            "pos",
            "mining-zone",
            "Polkadot"
         ],
         "ts" : "0.000001",
         "v" : "28647.035000",
         "y" : 0
      },
      {
         "an" : "Bitcoin Gold",
         "as" : 167287.793,
         "b" : "BTG",
         "ba" : "",
         "c" : "93.699",
         "cs" : 17333349,
         "etf" : false,
         "h" : "94.500",
         "i" : "0.00100000",
         "l" : "79.103",
         "o" : "82.676",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "14354843.344837",
         "s" : "BTGUSDT",
         "st" : "TRADING",
         "tags" : [
            "pow",
            "mining-zone"
         ],
         "ts" : "0.001",
         "v" : "167287.793",
         "y" : 0
      },
      {
         "an" : "Mirror Protocol",
         "as" : 1975756.521,
         "b" : "MIR",
         "ba" : "",
         "c" : "10.446",
         "cs" : null,
         "etf" : false,
         "h" : "10.629",
         "i" : "0.00100000",
         "l" : "9.128",
         "o" : "9.452",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "19632989.625603",
         "s" : "MIRUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone"
         ],
         "ts" : "0.001",
         "v" : "1975756.521",
         "y" : 0
      },
      {
         "an" : "FC Barcelona Fan Token BAR",
         "as" : 312043.773,
         "b" : "BAR",
         "ba" : "",
         "c" : "42.148",
         "cs" : null,
         "etf" : false,
         "h" : "43.494",
         "i" : "0.00100000",
         "l" : "39.880",
         "o" : "41.103",
         "pm" : "FIAT",
         "pn" : "FIAT",
         "pom" : false,
         "pomt" : null,
         "q" : "USDT",
         "qa" : "",
         "qn" : "TetherUS",
         "qv" : "13011738.54546",
         "s" : "BARUSDT",
         "st" : "TRADING",
         "tags" : [
            "innovation-zone"
         ],
         "ts" : "0.001",
         "v" : "312043.773",
         "y" : 0
      }
   ],
   "message" : null,
   "messageDetail" : null,
   "success" : true
}
