#!/usr/bin/env perl

use Test::Most;

use autodie;
use feature qw(say);

use List::AllUtils;
use Path::Class qw(file);

use Bio::FastParsers;


my $class = 'Bio::FastParsers::Blast::Table';

check_hsps(
    file('test', 'blastp.m9'), [
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|2105139|gb|AAC47544.1|', 100, 348, 0, 0, 1, 348, 1, 348, 0, 714, 1, 1, 1, 348, 1, 348 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|322229318|dbj|BAJ72725.1|', 97.68, 345, 8, 0, 1, 345, 1, 345, 0, 665, 1, 1, 1, 345, 1, 345 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|322229317|dbj|BAJ72724.1|', 95.07, 345, 17, 0, 1, 345, 1, 345, 0, 651, 1, 1, 1, 345, 1, 345 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|4589706|dbj|BAA76871.1|', 91.64, 347, 29, 0, 1, 347, 1, 347, 0, 636, 1, 1, 1, 347, 1, 347 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|52355664|gb|AAU44972.1|', 91.74, 339, 28, 0, 7, 345, 1, 339, 0, 622, 1, 1, 7, 345, 1, 339 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|321467151|gb|EFX78142.1|', 79.59, 343, 69, 1, 3, 345, 2, 343, 0, 560, 1, 1, 3, 345, 2, 343 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|321477129|gb|EFX88088.1|', 82.32, 345, 58, 2, 1, 345, 1, 342, 0, 560, 1, 1, 1, 345, 1, 342 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|322229315|dbj|BAJ72722.1|', 80.35, 346, 67, 1, 1, 345, 1, 346, 0, 556, 1, 1, 1, 345, 1, 346 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|321477127|gb|EFX88086.1|', 80.06, 346, 68, 1, 1, 345, 1, 346, 0, 553, 1, 1, 1, 345, 1, 346 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|321477134|gb|EFX88093.1|', 78.84, 345, 72, 1, 3, 347, 2, 345, 0, 552, 1, 1, 3, 347, 2, 345 ],
    ]
);

check_hits(
    file('test', 'blastp.m9'), [
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|2105139|gb|AAC47544.1|', 100 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|322229318|dbj|BAJ72725.1|', 97.68 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|322229317|dbj|BAJ72724.1|', 95.07 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|4589706|dbj|BAA76871.1|', 91.64 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|52355664|gb|AAU44972.1|', 91.74 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|321467151|gb|EFX78142.1|', 79.59 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|321477129|gb|EFX88088.1|', 82.32 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|322229315|dbj|BAJ72722.1|', 80.35 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|321477127|gb|EFX88086.1|', 80.06 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|321477134|gb|EFX88093.1|', 78.84 ],
    ]
);

check_queries(
    file('test', 'blastp.m9'), [
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|2105139|gb|AAC47544.1|', 100 ],
    ]
);


check_hsps(
    file('test', 'blastn.m9'), [
        [ 'gi|11467088|ref|NC_001638.1|', 'gi|11467088|ref|NC_001638.1|', 100, 15758, 0, 0, 1, 15758, 1, 15758, 0, 29100, 1, 1, 1, 15758, 1, 15758 ],
        [ 'gi|11467088|ref|NC_001638.1|', 'gi|11467088|ref|NC_001638.1|', 100, 532, 0, 0, 15227, 15758, 532, 1, 0, 983, 1, -1, 15227, 15758, 1, 532 ],
        [ 'gi|11467088|ref|NC_001638.1|', 'gi|11467088|ref|NC_001638.1|', 100, 532, 0, 0, 1, 532, 15758, 15227, 0, 983, 1, -1, 1, 532, 15227, 15758 ],
        [ 'gi|11467088|ref|NC_001638.1|', 'gi|11467088|ref|NC_001638.1|', 100, 86, 0, 0, 14999, 15084, 1, 86, 2e-40, 159, 1, 1, 14999, 15084, 1, 86 ],
        [ 'gi|11467088|ref|NC_001638.1|', 'gi|11467088|ref|NC_001638.1|', 100, 86, 0, 0, 86, 1, 14999, 15084, 2e-40, 159, -1, 1, 1, 86, 14999, 15084 ],
        [ 'gi|11467088|ref|NC_001638.1|', 'gi|11467088|ref|NC_001638.1|', 100, 86, 0, 0, 15673, 15758, 15084, 14999, 2e-40, 159, 1, -1, 15673, 15758, 14999, 15084 ],
        [ 'gi|11467088|ref|NC_001638.1|', 'gi|11467088|ref|NC_001638.1|', 100, 86, 0, 0, 14999, 15084, 15758, 15673, 2e-40, 159, 1, -1, 14999, 15084, 15673, 15758 ],
    ]
);

check_hits(
    file('test', 'blastn.m9'), [
        [ 'gi|11467088|ref|NC_001638.1|', 'gi|11467088|ref|NC_001638.1|', 100 ],
    ]
);

check_queries(
    file('test', 'blastn.m9'), [
        [ 'gi|11467088|ref|NC_001638.1|', 'gi|11467088|ref|NC_001638.1|', 100 ],
    ]
);


check_hsps(
    file('test', 'blastx.m9'), [
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|180332|gb|AAA35680.1|', 100, 1480, 0, 0, 133, 4572, 1, 1480, 0, 2932, 1, 1, 133, 4572, 1, 1480 ],
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|306538|gb|AAC13657.1|', 99.86, 1480, 2, 0, 133, 4572, 1, 1480, 0, 2928, 1, 1, 133, 4572, 1, 1480 ],
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|90421313|ref|NP_000483.3|', 99.8, 1480, 3, 0, 133, 4572, 1, 1480, 0, 2925, 1, 1, 133, 4572, 1, 1480 ],
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|397474456|ref|XP_003808695.1|', 99.73, 1480, 4, 0, 133, 4572, 1, 1480, 0, 2923, 1, 1, 133, 4572, 1, 1480 ],
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|89348175|gb|ABD72213.1|', 99.73, 1480, 4, 0, 133, 4572, 1, 1480, 0, 2921, 1, 1, 133, 4572, 1, 1480 ],
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|189303585|ref|NP_001073386.1|', 99.59, 1480, 6, 0, 133, 4572, 1, 1480, 0, 2919, 1, 1, 133, 4572, 1, 1480 ],
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|118572195|sp|Q2IBF6.1|CFTR_GORGO', 99.53, 1480, 7, 0, 133, 4572, 1, 1480, 0, 2919, 1, 1, 133, 4572, 1, 1480 ],
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|281182582|ref|NP_001162017.1|', 98.85, 1480, 17, 0, 133, 4572, 1, 1480, 0, 2901, 1, 1, 133, 4572, 1, 1480 ],
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|86169629|gb|ABC87055.1|', 99.86, 1462, 2, 0, 187, 4572, 1, 1462, 0, 2893, 1, 1, 187, 4572, 1, 1462 ],
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|332224265|ref|XP_003261288.1|', 98.92, 1480, 16, 0, 133, 4572, 1, 1480, 0, 2887, 1, 1, 133, 4572, 1, 1480 ],
    ]
);

check_hits(
    file('test', 'blastx.m9'), [
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|180332|gb|AAA35680.1|', 100 ],
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|306538|gb|AAC13657.1|', 99.86 ],
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|90421313|ref|NP_000483.3|', 99.8 ],
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|397474456|ref|XP_003808695.1|', 99.73 ],
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|89348175|gb|ABD72213.1|', 99.73 ],
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|189303585|ref|NP_001073386.1|', 99.59 ],
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|118572195|sp|Q2IBF6.1|CFTR_GORGO', 99.53 ],
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|281182582|ref|NP_001162017.1|', 98.85 ],
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|86169629|gb|ABC87055.1|', 99.86 ],
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|332224265|ref|XP_003261288.1|', 98.92 ],
    ]
);

check_queries(
    file('test', 'blastx.m9'), [
        [ 'gi|6995995|ref|NM_000492.2|', 'gi|180332|gb|AAA35680.1|', 100 ],
    ]
);


check_hsps(
    file('test', 'tblastn.m9'), [
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|2105138|gb|U67067.1|DMU67067', 100, 348, 0, 0, 1, 348, 75, 1118, 0, 681, 1, 1, 1, 348, 75, 1118 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|4589711|dbj|AB021137.1|', 91.64, 347, 29, 0, 1, 347, 34, 1074, 0, 630, 1, 1, 1, 347, 34, 1074 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|4589709|dbj|AB021136.1|', 80.23, 349, 66, 2, 1, 347, 39, 1082, 0, 544, 1, 1, 1, 347, 39, 1082 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|15080668|dbj|AB055114.1|', 67.25, 345, 110, 2, 4, 345, 44, 1078, 1e-155, 454, 1, 1, 4, 345, 44, 1078 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|15080666|dbj|AB055113.1|', 67.44, 347, 110, 2, 4, 347, 45, 1085, 6e-155, 452, 1, 1, 4, 347, 45, 1085 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|20513953|gb|AF441742.1|', 77.03, 148, 4, 1, 76, 193, 151, 594, 2e-127, 216, 1, 1, 76, 193, 151, 594 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|20513953|gb|AF441742.1|', 77.55, 98, 16, 1, 234, 331, 904, 1179, 2e-127, 160, 1, 1, 234, 331, 904, 1179 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|20513953|gb|AF441742.1|', 89.13, 46, 5, 0, 189, 234, 693, 830, 2e-127, 87.8, 1, 1, 189, 234, 693, 830 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|20513953|gb|AF441742.1|', 93.33, 30, 2, 0, 47, 76, 3, 92, 2e-127, 63.5, 1, 1, 47, 76, 3, 92 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|20513975|gb|AF441753.1|', 79.58, 142, 5, 1, 76, 193, 151, 576, 5e-127, 217, 1, 1, 76, 193, 151, 576 ],
    ]
);

check_hits(
    file('test', 'tblastn.m9'), [
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|2105138|gb|U67067.1|DMU67067', 100 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|4589711|dbj|AB021137.1|', 91.64 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|4589709|dbj|AB021136.1|', 80.23 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|15080668|dbj|AB055114.1|', 67.25 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|15080666|dbj|AB055113.1|', 67.44 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|20513953|gb|AF441742.1|', 77.03 ],
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|20513975|gb|AF441753.1|', 79.58 ],
    ]
);

check_queries(
    file('test', 'tblastn.m9'), [
        [ 'gi|2105139|gb|AAC47544.1|', 'gi|2105138|gb|U67067.1|DMU67067', 100 ],
    ]
);


check_hsps(
    file('test', 'tblastx.m9'), [
        [ 'gi|11465907|ref|NC_001872.1|', 'gi|11465907|ref|NC_001872.1|', 100, 1522, 0, 0, 22539, 17974, 22539, 17974, 0, 3346, -1, -1, 17974, 22539, 17974, 22539 ],
        [ 'gi|11465907|ref|NC_001872.1|', 'gi|11465907|ref|NC_001872.1|', 100, 1309, 0, 0, 16836, 12910, 16836, 12910, 0, 2966, -1, -1, 12910, 16836, 12910, 16836 ],
        [ 'gi|11465907|ref|NC_001872.1|', 'gi|11465907|ref|NC_001872.1|', 100, 799, 0, 0, 19148, 21544, 19148, 21544, 0, 1897, 1, 1, 19148, 21544, 19148, 21544 ],
        [ 'gi|11465907|ref|NC_001872.1|', 'gi|11465907|ref|NC_001872.1|', 100, 818, 0, 0, 5454, 3001, 5454, 3001, 0, 1835, -1, -1, 3001, 5454, 3001, 5454 ],
        [ 'gi|11465907|ref|NC_001872.1|', 'gi|11465907|ref|NC_001872.1|', 100, 806, 0, 0, 15511, 17928, 15511, 17928, 0, 1715, 1, 1, 15511, 17928, 15511, 17928 ],
        [ 'gi|11465907|ref|NC_001872.1|', 'gi|11465907|ref|NC_001872.1|', 100, 630, 0, 0, 7729, 9618, 7729, 9618, 0, 1425, 1, 1, 7729, 9618, 7729, 9618 ],
        [ 'gi|11465907|ref|NC_001872.1|', 'gi|11465907|ref|NC_001872.1|', 100, 656, 0, 0, 11545, 13512, 11545, 13512, 0, 1401, 1, 1, 11545, 13512, 11545, 13512 ],
        [ 'gi|11465907|ref|NC_001872.1|', 'gi|11465907|ref|NC_001872.1|', 100, 645, 0, 0, 6324, 8258, 6324, 8258, 0, 1389, 1, 1, 6324, 8258, 6324, 8258 ],
        [ 'gi|11465907|ref|NC_001872.1|', 'gi|11465907|ref|NC_001872.1|', 100, 564, 0, 0, 14135, 12444, 14135, 12444, 0, 1266, -1, -1, 12444, 14135, 12444, 14135 ],
        [ 'gi|11465907|ref|NC_001872.1|', 'gi|11465907|ref|NC_001872.1|', 100, 465, 0, 0, 2393, 999, 2393, 999, 0, 1037, -1, -1, 999, 2393, 999, 2393 ],
    ]
);

check_hits(
    file('test', 'tblastx.m9'), [
        [ 'gi|11465907|ref|NC_001872.1|', 'gi|11465907|ref|NC_001872.1|', 100 ],
    ]
);

check_queries(
    file('test', 'tblastx.m9'), [
        [ 'gi|11465907|ref|NC_001872.1|', 'gi|11465907|ref|NC_001872.1|', 100 ],
    ]
);


check_hsps(
    file('test', 'bug-next-hit-blastp.m9'), [
        [ 'seq1', 'gnl|684364|328773696', 62.21, 217, 81, 1, 1, 216, 51, 267, 6e-86, 257, 1, 1, 1, 216, 51, 267 ],
        [ 'seq2', 'gnl|684364|328773696', 67.53, 194, 63, 0, 22, 215, 74, 267, 3e-87, 261, 1, 1, 22, 215, 74, 267 ],
    ]
);

check_hits(
    file('test', 'bug-next-hit-blastp.m9'), [
        [ 'seq1', 'gnl|684364|328773696', 62.21 ],
        [ 'seq2', 'gnl|684364|328773696', 67.53 ],
    ]
);

check_queries(
    file('test', 'bug-next-hit-blastp.m9'), [
        [ 'seq1', 'gnl|684364|328773696', 62.21 ],
        [ 'seq2', 'gnl|684364|328773696', 67.53 ],
    ]
);


check_hsps(
    file('test', 'usearch-global.m8'), [
        [ '36329|PFD0780w', 'gnl|Gln_40|PF3D7_0416100', 100.0, 826, 0, 0, 1, 826, 1, 826, undef, undef, 1, 1, 1, 826, 1, 826 ],
        [ '5823|PBANKA_071810', 'gnl|Gln_40|PF3D7_0416100', 42.3, 833, 294, 11, 1, 656, 1, 826, undef, undef, 1, 1, 1, 656, 1, 826 ],
    ]
);


sub check_hsps {
    my $infile = shift;
    my $exp_hsps_ref = shift;

    my @methods = qw(
        query_id hit_id
        percent_identity hsp_length mismatches gaps
        query_from  query_to
          hit_from    hit_to
        evalue bit_score
        query_strand
          hit_strand
        query_start query_end
          hit_start   hit_end
    );

    ok my $report = $class->new( file => $infile ), 'Blast::Table constructor';
    isa_ok $report, $class, $infile;

    my $n = 0;
    my @queries;
    my @hits;
    while ( my $exp_hsp = shift @{ $exp_hsps_ref } ) {
        my $hsp = $report->next_hsp;
        ok $hsp, 'Blast::Table::Hsp constructor';
        isa_ok $hsp, $class . '::Hsp';
        cmp_deeply [ map { $hsp->$_ } @methods ], $exp_hsp,
            'got expected values for all methods for hsp-' . $n++;
        push @queries, [ $hsp->query_start, $hsp->query_end ];
        push @hits,    [ $hsp->hit_start,   $hsp->hit_end   ];
    }

    ok((List::AllUtils::all { $_->[0] < $_->[1] } @queries),
        'got expected coordinates for queries');
    ok((List::AllUtils::all { $_->[0] < $_->[1] } @hits),
        'got expected coordinates for hits');

    return;
}


sub check_hits {
    my $infile = shift;
    my $exp_hsps_ref = shift;

    my $report = $class->new( file => $infile );

    while ( my $exp_hsp = shift @{ $exp_hsps_ref } ) {
        my $hsp = $report->next_hit;
        cmp_deeply [ $hsp->query_id, $hsp->hit_id, $hsp->percent_identity + 0 ],
            $exp_hsp, 'got expected next hit';
    }

    return;
}


sub check_queries {
    my $infile = shift;
    my $exp_hsps_ref = shift;

    my $report = $class->new( file => $infile );

    while ( my $exp_hsp = shift @{ $exp_hsps_ref } ) {
        my $hsp = $report->next_query;
        cmp_deeply [ $hsp->query_id, $hsp->hit_id, $hsp->percent_identity + 0 ],
            $exp_hsp, 'got expected next query';
    }

    return;
}

# TODO: test multi-query reports

done_testing;
